package com.sapportals.wcm.service.scheduler;

import com.sapportals.wcm.WcmException;

/**
 * Specifies a sequence of scheduler entries.
 * This is an additional order relation to perfom scheduler tasks at a certain time 
 * only after a previous sequenced task was performed before. 
 * <p> 
 * Each scheduler entry has to be unique in this sequence. 
 *
 * Copyright (c)2003 SAP AG
 *
 * @author     jen.kaiser@sap.com
 * @version    $Id:$
 */
public interface ISequence {

  /**
   * Get the unique id of the sequence.
   *
   * @return    a <code>String</code> with the entry's id.
   */
  String getID();

  /**
   * Appends a scheduler entry to the end of the sequence.
   * The scheduler entry must not be added before, otherwise an exception will be raised.
   * 
   * @param  entry A scheduler entry
   * @return the added scheduler entry
   * @exception WcmException If adding fails.
   */
  ISchedulerEntry add(ISchedulerEntry entry) throws WcmException;

  /**
   * Inserts a scheduler entry to the sequence at position <code>index</code>.
   * The scheduler entry must not be added before, otherwise an exception will be raised.
   * 
   * @param index The position of the entry in the sequence.
   * @param  entry A scheduler entry
   * @return the added scheduler entry
   * @exception WcmException If adding fails. 
   */
  ISchedulerEntry add(int index, ISchedulerEntry entry) throws WcmException;

  /**
   * Removes an entry from the sequence. If this sequence does not contain the entry, 
   * it is unchanged.
   * 
   * @param  entry The entry to remove. 
   * @return <code>true</code> if this list contained the specified element.
   */
  boolean remove(ISchedulerEntry entry) throws WcmException;

  /**
   * Removes all of the elements from this sequence. 
   * This sequence will be empty after this call returns (unless it throws an exception).
   */
  void clear() throws WcmException;

  /**
   * Converts the entries of this sequence to an array in the proper order.
   * 
   * @return an array of scheduler entries
   */
  ISchedulerEntry[] toArray();

  /**
   * Returns the number of entries in this sequence. 
   * 
   * @return the number of entries in this sequence.
   */
  int size();

  /**
   * Returns <code>true</code> if this sequence contains no entries.
   * 
   * @return <code>true</code> if this sequence contains no entries.
   */
  boolean isEmpty();
  
  /**
   * Returns the entry at the specified position in this sequence.
   * 
   * @param  index of element to return.
   * @return the element at the specified position in this list.
   */
  ISchedulerEntry get(int index);

  /**
   * Returns <code>true</code> if this sequence contains the specified entry. 
   * More formally, returns true if and only if this sequence contains exactly one 
   * entry e such that (entry==null ? e==null : entry.equals(e)).
   * 
   * @param  entry  entry whose presence in this sequence is to be tested.
   * @return        <code>true</code> if this list contains the specified element.
   */
  boolean contains(ISchedulerEntry entry);
}
