package com.sapportals.wcm.service.scheduler;

import java.util.Date;

import com.sapportals.wcm.WcmException;


/**
 * Timetables for the scheduler.
 * <p>
 * An <code>ISchedulerTimeTable</code> represents a timetable which specifies
 * the times, when an <code>ISchedulerTask</code> has to be executed by the
 * <code>IScheduler</code>. Such a timetable can have a start- and/or an end-
 * time and at least one <code>ISchedulerTime</code> which defines the
 * times.<br>
 * Simply think of it like a school timetable, where the several entries (the
 * times) denote repeating lessings.
 * <p>
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 * @author paul.goetz@sapportals.com
 * @author jen.kaiser@sap.com
 * 
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_scheduler/java/api/com/sapportals/wcm/service/scheduler/ISchedulerTimeTable.java#2 $
 * @see ISchedulerService
 * @see ISchedulerTime
 */
public interface ISchedulerTimeTable {

  /**
   * Test, if a given date is in the timetable.
   * @param date the <code>Date</code> to check.
   * @return a <code>boolean</code> <code>true</code> if the date is in the
   *        timetable, <code>false</code> if it's not.
   */
  boolean isInTable(Date date);

  /**
   * Returns a time object that matches the specific date.
   * @param date
   * @return
   */
  ISchedulerTime getTime(Date date);
  
  /**
   * Get the starting date for the timetable.
   * @return a <code>Date</code> with the starting point of the timetable
   *         or <code>null</code> if not defined.
   */
  Date getStartTime();


  /**
   * Set the starting date for the timetable.
   * @param start a <code>Date</code> with the starting point for the
   *        timetable or <code>null</code> if not start time defined.
   */
  void setStartTime(Date start) throws WcmException;


  /**
   * Get the ending date for the timetable.
   * @return a <code>Date</code> with the ending point of the timetable.
   *         or <code>null</code> if not defined.
   */
  Date getStopTime();


  /**
   * Set the ending date for the timetable.
   * @param stop a <code>Date</code> with the ending point for the
   *        timetable or <code>null</code> if not stop time defined.
   */
  void setStopTime(Date stop) throws WcmException;


  /**
   * Get the times of the timetable.
   * @return a <code>ISchedulerTimeList</code> with the valid times of this
   *        table for the scheduler.
   * 
   * @deprecated since KM5.0SP6
   */
  ISchedulerTimeList getTable();


  /**
   * Set the times of the timetable.
   * @param timelist a <code>ISchedulerTimeList</code> with the valid times
   *        for the scheduler.
   * @deprecated since KM5.0SP6
   */
  void setTable(ISchedulerTimeList timelist) throws WcmException;


  /**
   * Set the times of the timetable to the given time.
   * @param time a <code>ISchedulerTime</code> with the one time
   *        for the scheduler.
   * 
   * @deprecated since KM5.0SP6
   */
  void setTable(ISchedulerTime time) throws WcmException;


  /**
   * Add an time to the times of the timetable.
   * @param time a <code>ISchedulerTime</code> with the one time
   *        for the scheduler to add.
   */
  void addToTable(ISchedulerTime time) throws WcmException;
  
  /**
   * @param time
   * @return
   */
  ISchedulerTime removeFromTable( ISchedulerTime time ) throws WcmException;


  /**
   * Converts the content of this time table to an array.
   * 
   * @return an array of scheudler times
   */
  ISchedulerTime[] toArray();
  
  /**
   * Returns the size of this table.
   * @return The number of scheudler times added to this table
   */
  int size();
  
  /**
   * Performs a deep copy of the time table.
   * @return a new time table instance.
   */
  ISchedulerTimeTable copy();
}
