package com.sapportals.wcm.service.scheduler;


import com.sapportals.wcm.WcmException;

import java.util.Calendar;
import java.util.TimeZone;


/**
 * The times for a scheduler.
 * <p>
 * A <code>ISchedulerTime</code> represents a time in an
 * <code>ISchedulerTimeTable</code> when an <code>ISchedulerTask</code> should
 * be executed by the <code>IScheduler</code>.<br>
 * Examples: <i>every Friday at 10am</i>: <code>set(Calendar.FRIDAY, 10, 0);</code> or
 * <i>every Day in September at 9pm</i>: <code>set(Calendar.SEPTEMBER, ISchedulerTime.WILDCARD, 21, 0);</code>
 * <p>
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author     paul.goetz@sapportals.com
 * @author     jen.kaiser@sap.com
 * 
 * @version    $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_scheduler/java/api/com/sapportals/wcm/service/scheduler/ISchedulerTime.java#3 $
 * @see        ISchedulerTimeTable
 */
public interface ISchedulerTime {

  /**
   * A 'dont care'-value for the several timetable fields (means 'any').
   */
  int WILDCARD = -1;


  /**
   * Test, if a given date matches this time.
   *
   * @param  date  the <code>Calendar</code> with the date to check.
   * @return       a <code>boolean</code> <code>true</code> if the date matches the
   *        time, <code>false</code> if it does not match.
   */
  boolean matches( Calendar date );


  /**
   * Set the time to the given values for month and monthday.
   *
   * @param  year              an <code>int</code> with the year (1900-2100).
   * @param  month             an <code>int</code> with the month (constants from
   *        <code>Calendar</code>)
   * @param  hour              an <code>int</code> with the hour (0-23).
   * @param  minute            an <code>int</code> with the minute (0-59)
   * @param  monthday          The new monthDay value
   * @return                   a <code>boolean</code> <code>true</code> if successful,
   *         <code>false</code> if not.
   * @throws WcmException      if the time can not be set.
   */
  boolean setMonthDay( int year, int month, int monthday, int hour, int minute ) throws WcmException;


  /**
   * Set the time to the given values for weekday.
   *
   * @param  year              an <code>int</code> with the year (1900-2100).
   * @param  month             an <code>int</code> with the month (constants from
   *        <code>Calendar</code>)
   * @param  weekday           an <code>int</code> with the weekday (use constants from
   *        <code>Calendar</code>)
   * @param  hour              an <code>int</code> with the hour (0-23)
   * @param  minute            an <code>int</code> with the minute (0-59)
   * @return                   a <code>boolean</code> <code>true</code> if successful,
   *         <code>false</code> if not.
   * @throws WcmException      if the time can not be set.
   */
  boolean setWeekDay( int year, int month, int weekday, int hour, int minute ) throws WcmException;


  /**
   * Set the time to the given values for the week of the month.
   *
   * @param  year              an <code>int</code> with the year (1900-2100).
   * @param  month             an <code>int</code> with the month (constants from
   *        <code>Calendar</code>)
   * @param  weekday           an <code>int</code> with the weekday (use constants from
   *        <code>Calendar</code>)
   * @param  weekofmonth       an <code>int</code> with the week of the month (1-5)
   * @param  hour              an <code>int</code> with the hour (0-23)
   * @param  minute            an <code>int</code> with the minute (0-59)
   * @return                   a <code>boolean</code> <code>true</code> if successful,
   *         <code>false</code> if not.
   * @throws WcmException      if the time can not be set.
   */
  boolean setWeekOfMonth( int year, int month, int weekday, int weekofmonth, int hour, int minute )
       throws WcmException;


  /**
   * Get time values, as in <code>Calendar</code>.
   *
   * @param  field             an <code>int</code> which defines the field to get (constants
   *        from <code>Calendar</code>)
   * @return                   an <code>int</code> with the value for that field.
   * @throws WcmException      if the field can not be retrieved.
   */
  int get( int field ) throws WcmException;


  /**
   * Set time values, as in <code>Calendar</code>.
   *
   * @param  field             an <code>int</code> which defines the field to set (constants
   *        from <code>Calendar</code>)
   * @param  value             an <code>int</code> with the value to set.
   * @return                   a <code>boolean</code> <code>true</code> if successful,
   *         <code>false</code> if not.
   * @throws WcmException       if the field can not be set.
   */
  boolean set( int field, int value ) throws WcmException;


  /**
   * Get the time in textual representation.
   *
   * @return          a <code>String</code> with the time in a textual
   *        representation.
   * @see             #setAsText(String)
   */
  String getAsText();


  /**
   * Set the time in a textual representation.
   *
   * @param  text              The new asText value
   * @return                   a <code>boolean</code> <code>true</code> if successful,
   *         <code>false</code> if not (the String was not valid).<br>
   *        A valid String with the time has to be in the following
   *        format:
   *          <i>y</i>,<i>m</i>,<i>d</i>,<i>w</i>,<i>o</i>,<i>h</i>,<i>m</i>
   *        where
   *          <i>y</i> specifies the year (1900 to 2100 or * for any),
   *          <i>m</i> specifies the month (1 to 12, or * for any),
   *          <i>d</i> specifies the day of month (1 to 31, or * for any)
   *          <i>w</i> specifies the day of week (1 to 7, or * for any)
   *          <i>o</i> specifies the week of the month (1 to 5 or * for any)
   *          <i>h</i> specifies the hour (0 to 23, or * for any)
   *          <i>m</i> specifies the minute (0 to 59, or * for any)
   *        and either <i>d</i> or <i>w</i> or both have to be set to *,
   *        it's invalid to set both of them to a non-* value.
   * @exception  WcmException  Description of the Exception
   * @throws WcmException if the time can not be set.
   */
  boolean setAsText( String text ) throws WcmException;

  
  /**
   * Sets the time zone of this scheduler time belongs to.
   * 
   * @param zone The time zone this time belongs to.
   * @throws WcmException If the time zone could not be set.
   */
  void setTimeZone( TimeZone zone ) throws WcmException;
  
  /**
   * Returns the time zone this scheduler time belongs to.
   * @return the time zone this scheduler time belongs to.
   * @throws WcmException If the time zone could not be retrieved.
   */
  TimeZone getTimeZone() throws WcmException;
}
