package com.sapportals.wcm.service.scheduler;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.service.landscape.ISystemList;

import java.util.Properties;

/**
 * The scheduler service for the WCM.
 * <p>
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author     paul.goetz@sapportals.com
 * @author     jen.kaiser@sap.com
 * @version    $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_scheduler/java/api/com/sapportals/wcm/service/scheduler/ISchedulerService.java#3 $
 */

public interface ISchedulerService extends IService {

  /** identifier for Scheduler-Service */
  String SCHEDULER_SERVICE = "SchedulerService";

  //---------------------------------------------------------------------------
  // Scheduler Entry Handlers
  //---------------------------------------------------------------------------

  /**
   * Register an <code>ISchedulerTask</code> to the scheduler as a new <code>ISchedulerEntry</code>.
   * The returned scheudler entry will only be performed on the local system by default.
   *
   * @param  entryID           a <code>String</code> with an alphanumerical  unique id for the scheduler entry.
   * @param  priority          an <code>int</code> with the priority of the task
   *        (<code>0</code> is highest priority, negative values are mapped to
   *        <code>0</code>, the higher the <code>int</code> value, the lower the
   *        priority)
   * @param  task              a <code>ISchedulerTask</code> with the task to execute.
   * @param  timetable         a <code>ISchedulerTimeTable</code> with the timetable for
   *        this task.
   * @param  properties        some <code>Properties</code> for this task.
   * @return                   a <code>ISchedulerEntry</code> with the newly created task entry.
   * @exception  WcmException  if the scheduler entry could not be created (already exists or other error).
   */
  ISchedulerEntry createSchedulerEntry(String entryID, int priority, ISchedulerTask task, ISchedulerTimeTable timetable, Properties properties)
    throws WcmException;

  /**
   * Register an <code>ISchedulerTask</code> to the scheduler as a new
   * <code>ISchedulerEntry</code>.
   *
   * @param  entryID           a <code>String</code> with the alphanumerical unique id for the scheduler entry.
   * @param  priority          an <code>int</code> with the priority of the task
   *        (<code>0</code> is highest priority, negative values are mapped to
   *        <code>0</code>, the higher the <code>int</code> value, the lower the
   *        priority)
   * @param  task              a <code>ISchedulerTask</code> with the task to execute.
   * @param  timetable         a <code>ISchedulerTimeTable</code> with the timetable for
   *        this task.
   * @param  properties        some <code>Properties</code> for this task.
   * @param  systemList        A list of systems to run on
   * @return                   a <code>ISchedulerEntry</code> with the newly created task entry.
   * @exception  WcmException  if the scheduler entry could not be created (already exists or other error).
   */
  ISchedulerEntry createSchedulerEntry(
    String entryID,
    int priority,
    ISchedulerTask task,
    ISchedulerTimeTable timetable,
    ISystemList systemList,
    Properties properties)
    throws WcmException;

  /**
   * Get an <code>ISchedulerEntry</code> by ID.
   *
   * @param  entryID           a <code>String</code> with the unique id of the entry to search for.
   * @return                   a <code>ISchedulerEntry</code> with the entry for the given ID
   *        or <code>null</code> if not found.
   * @exception  WcmException  if the scheduler entry could not be retrieved.
   */
  ISchedulerEntry getSchedulerEntry(String entryID) throws WcmException;

  /**
   * Deregister an <code>ISchedulerEntry</code> from the scheduler.
   *
   * @param  entry             a <code>ISchedulerEntry</code> with the entry to delete.
   * @return                   the deleted <code>ISchedulerEntry</code>,
   *         <code>null</code> if not found.
   * @exception  WcmException  if the entry could not be deleted.
   */
  ISchedulerEntry deleteSchedulerEntry(ISchedulerEntry entry) throws WcmException;

  //---------------------------------------------------------------------------
  // Scheduler Entry Lifetime Handlers
  //---------------------------------------------------------------------------

  /**
   * Suspend an <code>ISchedulerEntry</code> (inactivate it).
   *
   * @param  entry             the <code>ISchedulerEntry</code> to suspend.
   * @return                   the <code>ISchedulerEntry</code> with the suspended entry,
   *        <code>null</code>  if the entry's ID could not be found.
   * @exception  WcmException  if the scheduler entry could not be suspended.
   */
  ISchedulerEntry suspendSchedulerEntryTask(ISchedulerEntry entry) throws WcmException;

  /**
   * Resume an <code>ISchedulerEntry</code> (activate it from suspended).
   *
   * @param  entry             the <code>ISchedulerEntry</code> to resume.
   * @return                   the <code>ISchedulerEntry</code> with the resumed entry,
   *        <code>null</code> if the entry's ID could not be found.
   * @exception  WcmException  if the scheduler entry could not be resumed.
   */
  ISchedulerEntry resumeSchedulerEntryTask(ISchedulerEntry entry) throws WcmException;

  /**
   * Returns all scheduler entries to be performed on <code>system</code>.
   * If no local scheduler entries exist an empty array will returned.
   *
   * @return    The local scheduler entries.
   */
  ISchedulerEntry[] getLocalSchedulerEntries() throws WcmException;

  /**
   * Returns all scheduler entries of the cluster. If cluster mode is disabled, this
   *  method is equivalent to {@link #getLocalSchedulerEntries()}.
   * If no scheduler entries exist an empty array will returned.
   *
   * @return    The scheduler entries
   */
  ISchedulerEntry[] getSchedulerEntries() throws WcmException;

  //---------------------------------------------------------------------------
  // Scheduler Task Sequence Handlers
  //---------------------------------------------------------------------------

  /**
   * Creates and registers a new task sequence.
   *
   * @param  id                a <code>String</code> with the unique id of the sequence to create.
   * @return                   a <code>ISchedulerEntry</code> with the entry for the given ID
   * @exception  WcmException  if the sequence could not be created.
   */
  ISequence createSequence(String id) throws WcmException;

  /**
   * Returns a registered sequence or <code>null</code> if the sequence does not exist.
   *
   * @param  id                a <code>String</code> with the unique id of the sequence to search for.
   * @return                   a <code>ISchedulerEntry</code> with the entry for the given ID
   * @exception  WcmException  if the sequence could not be retrieved.
   */
  ISequence getSequence(String id) throws WcmException;

  /**
   * Returns all registered sequences.
   * 
   * @return an array of all sequences registered on the scheduler service
   */  
  ISequence[] getSequences() throws WcmException;

  /**
   * Removes a sequence from the scheduler
   *
   * @param  sequence          The sequence to remove
   * @return                   The removed sequence or <code>null</code> if the sequnence was not registered.
   * 
   * @exception  WcmException  if the sequence could not be deleted.
   */
  ISequence deleteSequence(ISequence sequence) throws WcmException;

  /**
   * Returns all sequences that contain the <code>entry</code> or an empty array if
   * the entry does not belong to a sequence or the entry is not registered on the the scheduler service.
   * 
   * @param entry The entry for which to look up the sequences 
   * @return  all sequences that contain the <code>entry</code> or an empty array if
   * the entry does not belong to a sequence or the entry is not registered on the the scheduler service.
   * @throws WcmException if the sequences could not be retrieved.
   */
  ISequence[] getSequences(ISchedulerEntry entry) throws WcmException;

  //---------------------------------------------------------------------------
  // Factories
  //---------------------------------------------------------------------------

  /**
   * Create an (empty) time table.
   *
   * @return                   an empty <code>ISchedulerTimeTable</code>.
   * @exception  WcmException  if the time table cannot be created.
   */
  ISchedulerTimeTable createTimeTable() throws WcmException;

  /**
   * Get a scheduler time for the current time.
   *
   * @return                   an <code>ISchedulerTime</code>.
   * @exception  WcmException  if the time cannot be created.
   */
  ISchedulerTime createTimeWildcard() throws WcmException;

  //---------------------------------------------------------------------------
  // Deprecated stuff
  //---------------------------------------------------------------------------

  /**
   * Returns a factory implementation to create new <code>ISchedulerEntry</code>
   * objects without registering them.
   *
   * @return        The schedulerEntryFactory value
   * @deprecated    since KM5.0SP6
   */
  ISchedulerEntryFactory getSchedulerEntryFactory();

  /**
   * Update an <code>ISchedulerEntry</code> in the scheduler's list.
   *
   * @param  entry             a <code>ISchedulerEntry</code> with new attributes.
   * @return                   the <code>ISchedulerEntry</code> with the update attributes or
   *        <code>null</code> if the entry's ID could not be found.
   * @exception  WcmException  if the scheduler entry could not be updated.
   * @deprecated               since EP6.0SP1
   */
  ISchedulerEntry updateSchedulerEntry(ISchedulerEntry entry) throws WcmException;

  /**
   * Adds a scheduler entry to the scheduler.
   *
   * @param  entry             The feature to be added to the SchedulerEntry attribute
   * @return                   Description of the Return Value
   * @exception  WcmException  Description of the Exception
   * @deprecated               since EP6.0SP1: new scheduler tasks are already added to the scheduler
   */
  ISchedulerEntry addSchedulerEntry(ISchedulerEntry entry) throws WcmException;

  /**
   * Get a list of all scheduler entries to be performed on this system.
   *
   * @return                   a <code>ISchedulerEntryList</code> with all known scheduler entries.
   * @exception  WcmException  if the list of scheduler entries could not be retrieved.
   * @deprecated               since KM 5.0 SP6. Returns an empty list.
   */
  ISchedulerEntryList getAllSchedulerEntries() throws WcmException;

}
