package com.sapportals.wcm.service.scheduler;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.service.landscape.ISystemList;

import java.util.Properties;

/**
 * Scheduler entries contain data for scheduler tasks.
 * <p>
 * An <code>ISchedulerEntry</code> represents an entry in the
 * <code>IScheduler</code>'s internal list.
 * Each entry has a unique id, specifies the task to execute and a timetable,
 * which specifies the scheduling timetable (when to execute the task).
 * In addition, there might be some (String-)parameters for the task, passed as
 * properties.
 * <p>
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author     paul.goetz@sapportals.com
 * @author     jen.kaiser@sap.com
 * @version    $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_scheduler/java/api/com/sapportals/wcm/service/scheduler/ISchedulerEntry.java#2 $
 * @see        ISchedulerService
 */
public interface ISchedulerEntry extends IComponent {

  /**
   * Get the unique id of the entry.
   *
   * @return    a <code>String</code> with the entry's id.
   */
  String getID();

  /**
   * Get the priority of the entry.
   *
   * @return    an <code>int</code> with the entry's priority (a higher value
   *        is a higher priority).
   */
  int getPriority();

  /**
   * Set the priority of the entry.
   *
   * @param  priority  an <code>int</code> with the entry's priority (a higher
   *        value is a higher priority).
   */
  void setPriority(int priority) throws WcmException;

  /**
   * Check if the entry is running(currently being executed).
   *
   * @return                   a <code>boolean</code> <code>true</code>, if the entry's task
   *        is currently being executed, <code>false</code> if not.
   * @exception WcmException if the state could not be determined.
   */
  boolean isRunning() throws WcmException;

  /**
   * Check if the entry is suspended (currently inactive and not waiting).
   *
   * @return                   a <code>boolean</code> <code>true</code>, if the entry's task
   *        is suspended, <code>false</code> if not.
   * @exception  WcmException  if the state could not be determined.
   */
  boolean isSuspended() throws WcmException;

  /**
   * Check if the entry is not suspended and not active.
   *
   * @return                   a <code>boolean</code> <code>true</code>, if the entry's task
   *        is currently neither being executed nor suspended,
   *        <code>false</code> if not.
   * @exception  WcmException  if the state could not be determined.
   */
  boolean isWaiting() throws WcmException;

  /**
   * Get the task for the entry.
   *
   * @return                   an <code>ISchedulerTask</code> with the task.
   * @exception  WcmException  if the task could not be retrieved.
   */
  ISchedulerTask getTask() throws WcmException;

  /**
   * Set the entry's task.
   *
   * @param  task              an <code>ISchedulerTask</code> with the task to execute.
   * @exception  WcmException  if the scheduler entry could not be updated.
   */
  void setTask(ISchedulerTask task) throws WcmException;

  /**
   * Get the timetable for the entry.
   *
   * @return                   an <code>ISchedulerTimeTable</code> with the timetable.
   * @exception  WcmException  if the timetable could not be retrieved.
   */
  ISchedulerTimeTable getTimeTable() throws WcmException;

  /**
   * Set the entry's timetable.
   *
   * @param  timetable         an <code>ISchedulerTimeTable</code> with the timetable
   *        to use.
   * @exception  WcmException  if the timetable could not be set
   */
  void setTimeTable(ISchedulerTimeTable timetable) throws WcmException;

  /**
   * Get the properties to pass to the task of the entry.
   *
   * @return                   the <code>Properties</code> for the task.
   * @exception  WcmException   if the properties could not be retrieved.
   */
  Properties getProperties() throws WcmException;

  /**
   * Set the entry's properties to pass to the task.
   *
   * @param  properties        the <code>Properties</code> for the task.
   * @exception  WcmException  if the properties could not be set.
   */
  void setProperties(Properties properties) throws WcmException;

  /**
   * Gets the ID of the CM system where to run the <code>ISchedulerTask</code>
   * physically.
   *
   * @return        the ID of the CM system where to run the <code>ISchedulerTask</code>
   * physically.
   */
  ISystemList getSystemList();

  /**
   * Sets the ID of the CM system where to run the <code>ISchedulerTask</code>
   * physically.
   *
   * @param  systemList  The new systemList value
   * @exception  WcmException  if the system list could not be set.
   */
  void setSystemList(ISystemList systemList) throws WcmException;
}
