/*
**
** Toc.c
**
** Copyright (C) 1995-1997 Johannes Plass
** 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
** 
** Author:   Johannes Plass (plass@thep.physik.uni-mainz.de)
**           Department of Physics
**           Johannes Gutenberg-University
**           Mainz, Germany
**
*/

#include <stdio.h>
/*
#define MESSAGES
*/
#include "message.h"

#include "paths.h"
#include INC_X11(IntrinsicP.h)
#include INC_X11(StringDefs.h)
#include INC_XMU(Misc.h)
#include INC_XAW(XawInit.h)
#include "TocP.h"
#include "Frame.h"
#include INC_XMU(Converters.h)

#include "d_memdebug.h"

#define TOC_MARK_LEFT_INDENT  4
#define TOC_MARK_LABEL_INTERNAL_WIDTH 11
#define TOC_MARK_LABEL_INDENT 4
#define TOC_MARK_VERTICAL_INDENT 3
#define TOC_MARK_WIDTH 5

/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

/* Private Data */

static char defaultTranslations[] = "";

#define offset(field) XtOffsetOf(TocRec, field)
static XtResource resources[] = { 
   {XtNselectedShadowWidth, XtCShadowWidth, XtRDimension, sizeof(Dimension),
	offset(toc.selected_shadow_width), XtRImmediate, (XtPointer) 1},
   {XtNmarkShadowWidth, XtCShadowWidth, XtRDimension, sizeof(Dimension),
	offset(toc.mark_shadow_width), XtRImmediate, (XtPointer) 1},
   {XtNhighlightedShadowWidth, XtCShadowWidth, XtRDimension, sizeof(Dimension),
	offset(toc.highlighted_shadow_width), XtRImmediate, (XtPointer) 1},
   {XtNtoc,  XtCToc, XtRString, sizeof(String),
      offset(toc.toc), XtRString, NULL},
   {XtNmarkBackground, XtCMarkBackground, XtRPixel, sizeof(Pixel),
	offset(toc.mark_background), XtRString, XtDefaultBackground},
   {XtNselectedBackground, XtCSelectedBackground, XtRPixel, sizeof(Pixel),
	offset(toc.selected_background), XtRString, XtDefaultBackground},
   {XtNhighlightedBackground, XtCHighlightedBackground, XtRPixel, sizeof(Pixel),
	offset(toc.highlighted_background), XtRString, XtDefaultBackground},
   {XtNshadowWidth, XtCShadowWidth, XtRDimension, sizeof(Dimension),
      offset(threeD.shadow_width), XtRImmediate, (XtPointer) 2},
   {XtNinternalWidth, XtCWidth, XtRDimension,  sizeof(Dimension),
	offset(label.internal_width), XtRImmediate, (XtPointer)TOC_MARK_LABEL_INTERNAL_WIDTH},
   {XtNborderWidth, XtCBorderWidth, XtRDimension, sizeof(Dimension),
      XtOffsetOf(RectObjRec,rectangle.border_width), XtRImmediate,
      (XtPointer)0}
};
#undef offset

static Boolean SetValues();
static void Initialize(), Redisplay();
static void Destroy(), PaintTocWidget();
static void ClassInitialize();
static void Realize(), Resize();
static XtGeometryResult QueryGeometry();
static void PaintSelectionMarkOfEntry();
static void PaintMarkOfEntry();
static void PaintMarkMarkOfEntry();


#define SuperClass ((LabelWidgetClass)&labelClassRec)

TocClassRec tocClassRec = {
  {
    (WidgetClass) SuperClass,		/* superclass		  */	
    "Toc",				/* class_name		  */
    sizeof(TocRec),			/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    TRUE,				/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    Resize,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    defaultTranslations,		/* tm_table		  */
    QueryGeometry,			/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    XtInheritChangeSensitive		/* change_sensitive	*/
  },  /* SimpleClass fields initialization */
  {
    XtInheritXaw3dShadowDraw,           /* shadowdraw           */
  },  /* ThreeD Class fields initialization */
  {
    0,                                     /* field not used    */
  },  /* LabelClass fields initialization */
  {
    0,                                     /* field not used    */
  },  /* TocClass fields initialization */
};

  /* for public consumption */
WidgetClass tocWidgetClass = (WidgetClass) &tocClassRec;

/*---------------------------------------------------*/
/* ClassInitialize */
/*---------------------------------------------------*/

static void ClassInitialize()
{
  BEGINMESSAGE(ClassInitialize)
  XawInitializeWidgetSet();
  ENDMESSAGE(ClassInitialize)
}

/*---------------------------------------------------*/
/* Initialize */
/*---------------------------------------------------*/

static void 
Initialize(request, new, args, num_args)
Widget request, new;
ArgList args;			/* unused */
Cardinal *num_args;		/* unused */
{
  TocWidget tw = (TocWidget) new;
  String s="";
  char *c;
  XGCValues values;

  BEGINMESSAGE(Initialize)
  if (tw->toc.toc) s = tw->toc.toc;
  tw->toc.toc = GV_XtNewString(s);
  c = tw->toc.toc;
  tw->toc.selected = -1;
  tw->toc.highlighted = -1;
  tw->toc.entries = (int)strlen(tw->toc.toc);

  /* mark background GC */
  values.foreground	= tw->toc.mark_background;
  tw->toc.mark_background_GC = XtGetGC((Widget)tw,(unsigned) GCForeground,&values);
  /* selected background GC */
  values.foreground = tw->toc.selected_background;
  tw->toc.selected_background_GC = XtGetGC((Widget)tw,(unsigned) GCForeground,&values);
  /* highlighted background GC */
  values.foreground = tw->toc.highlighted_background;
  tw->toc.highlighted_background_GC = XtGetGC((Widget)tw,(unsigned) GCForeground,&values);
  /* background GC */
  values.foreground	= tw->core.background_pixel;
  values.graphics_exposures = False;
  tw->toc.background_GC = XtGetGC((Widget)tw,(unsigned) GCForeground | GCGraphicsExposures,&values);

  ENDMESSAGE(Initialize)
}

/*---------------------------------------------------*/
/* Realize */
/*---------------------------------------------------*/

static void Realize(w, valueMask, attributes)
    Widget w;
    Mask *valueMask;
    XSetWindowAttributes *attributes;
{
  BEGINMESSAGE(Realize)
  (*tocWidgetClass->core_class.superclass->core_class.realize)
    (w, valueMask, attributes);
  ENDMESSAGE(Realize)
}

/*---------------------------------------------------*/
/* Redisplay */
/*---------------------------------------------------*/

static void 
Redisplay(w, event, region)
Widget w;
XEvent *event;
Region region;
{
  BEGINMESSAGE(Redisplay)
  PaintTocWidget(w, event, region);
  ENDMESSAGE(Redisplay)
}

/*---------------------------------------------------*/
/* QueryGeometry */
/*---------------------------------------------------*/

static XtGeometryResult QueryGeometry(w, intended, preferred)
  Widget w;
  XtWidgetGeometry *intended, *preferred;
{
  XtGeometryResult gr;

  BEGINMESSAGE(QueryGeometry)
  gr=(*SuperClass->core_class.query_geometry)(w, intended, preferred);
  IIMESSAGE(preferred->width,preferred->height)
  IIMESSAGE(intended->width,intended->height)
  ENDMESSAGE(QueryGeometry)
  return gr;
}

/*---------------------------------------------------*/
/* Resize */
/*---------------------------------------------------*/

static void Resize(w)
    Widget w;
{
  TocWidget tw = (TocWidget) w;
  BEGINMESSAGE(Resize)
  (*tocWidgetClass->core_class.superclass->core_class.resize)(w);
  tw->label.label_x = tw->label.label_x + TOC_MARK_LABEL_INDENT;
  ENDMESSAGE(Resize)
}

/*---------------------------------------------------*/
/* SetValues */
/*---------------------------------------------------*/

static Boolean 
SetValues (current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
{
  TocWidget ctw = (TocWidget) current;
  TocWidget ntw = (TocWidget) new;
  Boolean changed = False;

  BEGINMESSAGE(SetValues)
  if (ntw->toc.toc == NULL) ntw->toc.toc = "";
  if (ctw->toc.toc != ntw->toc.toc) {
    GV_XtFree((char *)ctw->toc.toc);
    ntw->toc.toc = GV_XtNewString(ntw->toc.toc);
    ntw->toc.entries = (int)strlen(ntw->toc.toc);
    if (!ntw->toc.entries) ntw->toc.entries =- 1;
    SMESSAGE(ntw->toc.toc)
    IMESSAGE(ntw->toc.entries)
    changed = True;
  }
  ENDMESSAGE(SetValues)
  return (changed);
}

/*---------------------------------------------------*/
/* Destroy */
/*---------------------------------------------------*/

static void 
Destroy(w)
Widget w;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(Destroy)
  GV_XtFree(tw->toc.toc);
  XtReleaseGC(w,tw->toc.background_GC);
  XtReleaseGC(w,tw->toc.mark_background_GC);
  XtReleaseGC(w,tw->toc.selected_background_GC);
  XtReleaseGC(w,tw->toc.highlighted_background_GC);
  ENDMESSAGE(Destroy)
}

/*---------------------------------------------------*/
/* PaintEntryString */
/*---------------------------------------------------*/

static void
PaintEntryString(w, entry)
  Widget w;
  int entry;
{
  TocWidget tw = (TocWidget)w;
  char * s;
  int i;

  BEGINMESSAGE1(PaintEntryString)
  s = tw->label.label;
  i = entry;
  if (s) while (i > 0 && (s = strchr(s,'\n'))) { s++; i--; }
  if (s) {
    char *nl = strchr(s,'\n');
    if (nl) *nl = '\0';
    XDrawString(XtDisplay(w), XtWindow(w), tw->label.normal_GC,
		tw->label.label_x, 
		tw->label.label_y+entry*(tw->label.font->max_bounds.ascent +
					 tw->label.font->max_bounds.descent) +
		                         tw->label.font->max_bounds.ascent,
		s, (int)strlen(s));
    if (nl) *nl = '\n';
  }
  ENDMESSAGE1(PaintEntryString)
}

/*---------------------------------------------------*/
/* PaintMarkOfEntry */
/*---------------------------------------------------*/

static void
PaintMarkOfEntry(w, region, entry, erase)
  Widget w;
  Region region;
  int entry;
  Boolean erase;
{
  TocWidget tw = (TocWidget)w;
  int ss;
  Position x,y;
  Dimension width,height;

  BEGINMESSAGE(PaintMarkOfEntry)
  if (entry < 0 || entry >= tw->toc.entries) {
    INFMESSAGE(invalid entry)
    ENDMESSAGE(PaintMarkOfEntry)
    return;
  }

  x = (Position) 2;
  y = (Position) (((int) tw->label.label_y) -1 + (entry*(tw->label.label_height))/tw->toc.entries);
  width = (Dimension)((int) tw->core.width - 4);
  height= (Dimension)(((int) tw->label.label_height)/tw->toc.entries + 2.5);
  if      (tw->toc.selected==entry-1) { y=y+1; height=height-1; }
  else if (tw->toc.selected==entry+1) height=height-1;
  ss = XawSUNKEN;
  if (region == NULL || XRectInRegion(region,x,y,width,height) != RectangleOut) {
    if (erase) {
      INFMESSAGE(clearing highlighted)
      XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),tw->toc.background_GC,x,y,width,height);
    } else {
      INFMESSAGE(drawing highlighted)
      XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),tw->toc.highlighted_background_GC,x,y,width,height);
      FrameDrawFrame(w,x,y,width,height,ss,tw->toc.highlighted_shadow_width,
		tw->threeD.top_shadow_GC,
		tw->threeD.bot_shadow_GC);
    }
    if (tw->toc.selected == entry) PaintSelectionMarkOfEntry(w,region,entry,False);
    else PaintEntryString(w,entry);
    PaintMarkMarkOfEntry(w,region,entry,False);
  }
  ENDMESSAGE(PaintMarkOfEntry)
}

/*---------------------------------------------------*/
/* PaintSelectionMarkOfEntry */
/*---------------------------------------------------*/

static void
PaintSelectionMarkOfEntry(w, region, entry, erase)
  Widget w;
  Region region;
  int entry;
  Boolean erase;
{
  TocWidget tw = (TocWidget)w;
  int ss;
  Position x,y;
  Dimension width,height;

  BEGINMESSAGE(PaintSelectionMarkOfEntry)
  if (entry < 0) {
    INFMESSAGE(invalid entry)
    ENDMESSAGE(PaintSelectionMarkOfEntry)
    return;
  }

  x = (Position) (((int) tw->label.label_x) - 3);
  y = (Position) (((int) tw->label.label_y) + (entry*(tw->label.label_height))/tw->toc.entries);
  width = (Dimension)((int) tw->label.label_width) + 6;;
  height= (Dimension)(((int) tw->label.label_height)/tw->toc.entries + 0.5);
  ss = XawSUNKEN;
  if (region == NULL || XRectInRegion(region,x,y,width,height) != RectangleOut) {
    if (erase) {
      INFMESSAGE(clearing selected)
      XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),
		     (entry == tw->toc.highlighted) ? tw->toc.highlighted_background_GC : tw->toc.background_GC,
		     x,y,width,height);

      XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),tw->toc.background_GC,x,y,width,height);
    } else {
      INFMESSAGE(drawing selected)
      XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),tw->toc.selected_background_GC,x,y,width,height);
      FrameDrawFrame(w,x,y,width,height,ss,tw->toc.selected_shadow_width,
		tw->threeD.top_shadow_GC,
		tw->threeD.bot_shadow_GC);
    }
    PaintEntryString(w,entry);
  }
  ENDMESSAGE(PaintSelectionMarkOfEntry)
}

/*---------------------------------------------------*/
/* PaintMarkMarkOfEntry */
/*---------------------------------------------------*/

static void 
PaintMarkMarkOfEntry(w, region, entry, erase)
  Widget w;
  Region region;
  int entry;
  Boolean erase;
{
  TocWidget tw = (TocWidget)w;
  int ss;
  Position x,y;
  Dimension width,height;
  char *toc = tw->toc.toc;
  Boolean paint=False;

  BEGINMESSAGE1(PaintMarkMarkOfEntry)
  if (toc[entry] == '*') paint = True;
  if (paint || erase) {
    x = (Position) (TOC_MARK_LEFT_INDENT);
    y = (Position) (((int) tw->label.label_y) +
		    TOC_MARK_VERTICAL_INDENT +
		    (entry*(tw->label.label_height))/tw->toc.entries);
    width = (Dimension) (TOC_MARK_WIDTH);
    height= (Dimension)(((int) tw->label.label_height)/tw->toc.entries + 0.5 - 2*TOC_MARK_VERTICAL_INDENT);
    ss = XawSUNKEN;
    if (region == NULL || XRectInRegion(region,x,y,width,height) != RectangleOut) {
      if (paint) {
	INFMESSAGE(drawing mark)
	XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),tw->toc.mark_background_GC,x,y,width,height);
	FrameDrawFrame(w,x,y,width,height,ss,tw->toc.mark_shadow_width,
		  tw->threeD.top_shadow_GC,
		  tw->threeD.bot_shadow_GC);
      } else {
	INFMESSAGE(clearing mark)
	XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),
		       (entry == tw->toc.highlighted) ? tw->toc.highlighted_background_GC : tw->toc.background_GC,
		       x,y,width,height);
#if 0
	XClearArea(XtDisplayOfObject(w), XtWindowOfObject(w),x,y,width,height,False);
#endif
      }
    }
  }
  ENDMESSAGE1(PaintMarkMarkOfEntry)
}

/*---------------------------------------------------*/
/* PaintMarksOfEntries */
/*---------------------------------------------------*/

static void 
PaintMarksOfEntries(w, event, region)
Widget w;
XEvent *event;
Region region;
{
  TocWidget tw = (TocWidget)w;
  int i;

  BEGINMESSAGE(PaintMarksOfEntries)
  PaintMarkOfEntry(w, region, tw->toc.highlighted, False);
  i = 0;
  while (i < tw->toc.entries) {
    if (i != tw->toc.highlighted)
      PaintMarkMarkOfEntry(w, region, i, False);
    i++;
  }
  if (tw->toc.selected != tw->toc.highlighted && tw->toc.entries >= 0)
    PaintSelectionMarkOfEntry(w, region, tw->toc.selected, False);
  ENDMESSAGE(PaintMarksOfEntries)
}

/*---------------------------------------------------*/
/* PaintTocWidget */
/*---------------------------------------------------*/

static void 
PaintTocWidget(w, event, region)
Widget w;
XEvent *event;
Region region;
{
  BEGINMESSAGE(PaintTocWidget)
  PaintMarksOfEntries(w, event, region);
  (*SuperClass->core_class.expose) (w, event, region);
  ENDMESSAGE(PaintTocWidget)
}

/*####################################################################*/
/*####################################################################*/
/* Public Routines */
/*####################################################################*/
/*####################################################################*/

/*###################################################*/
/* TocSelected */
/*###################################################*/

int
TocSelected(w)
  Widget w;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocSelected)
  ENDMESSAGE(TocSelected)
  return(tw->toc.selected);
}

/*###################################################*/
/* TocHighlighted */
/*###################################################*/

int
TocHighlighted(w)
  Widget w;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocHighlighted)
  ENDMESSAGE(TocHighlighted)
  return(tw->toc.highlighted);
}

/*###################################################*/
/* TocEntries */
/*###################################################*/

int
TocEntries(w)
  Widget w;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocEntries)
  ENDMESSAGE(TocEntries)
  return(tw->toc.entries);
}

/*###################################################*/
/* TocToc */
/*###################################################*/

char*
TocToc(w)
  Widget w;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocToc)
  ENDMESSAGE(TocToc)
  return(tw->toc.toc);
}

/*###################################################*/
/* TocEntryOfPosition */
/*###################################################*/

int
TocEntryOfPosition(w,y)
  Widget w;
  int y;
{
  TocWidget tw = (TocWidget)w;
  int entry = -1;

  BEGINMESSAGE(EntryOfPosition)
  y = y - (int) tw->label.label_y;
  if (tw->label.label_height > 0) {
    if (y < 0) entry = -1;
    else       entry = (tw->toc.entries*y)/(int)tw->label.label_height;
  }
  IMESSAGE(entry)
  ENDMESSAGE(EntryOfPosition)
  return(entry);
}

/*###################################################*/
/* TocChangeMark */
/*###################################################*/

void
TocChangeMark(w,entry,change)
  Widget w;
  int entry;
  int change;
{
  TocWidget tw = (TocWidget)w;
  char *toc=tw->toc.toc;
  Boolean paint;
  int lb,ub,st;

  BEGINMESSAGE(TocChangeMark)
  switch (entry) {
    case XawTocAll:
      lb=0;
      ub=tw->toc.entries-1;
      st=1;
      break;
    case XawTocEven:
      lb=1;
      ub=tw->toc.entries-1;
      st=2;
      break;
    case XawTocOdd:
      lb=0;
      ub=tw->toc.entries-1;
      st=2;
      break;
    case XawTocCurrent:
      lb=tw->toc.selected;
      ub=tw->toc.selected;
      st=1;
      break;
    default:
      lb=entry;
      ub=entry;
      st=1;
      break;
  }

  if (ub < 0 || ub >= tw->toc.entries) {
    INFMESSAGE(invalid setup)
    ENDMESSAGE(TocChangeMark)
    return;
  }

  entry=lb;
  while (entry <= ub) {
    paint=False;
    switch (change) {
      case XawTocToggle:
	switch (toc[entry]) {
          case '*':
	    toc[entry]=' ';
	    paint=True;
	    break;
          default:
	    toc[entry]='*';
	    paint=True;
	    break;
        }
	break;
      case XawTocSet:
	toc[entry]='*';
	paint=True;
	break;
      case XawTocUnset:
	toc[entry]=' ';
	paint=True;
	break;
      default:
	INFMESSAGE(invalid action)
	break;
    }
    if (paint) PaintMarkMarkOfEntry(w,NULL,entry,True);
    entry += st;
  }
  ENDMESSAGE(TocChangeMark)
}

/*###################################################*/
/* TocChangeSelected */
/*###################################################*/

void 
TocChangeSelected(w,entry,change)
  Widget w;
  int entry;
  int change;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocChangeSelected)
  IIMESSAGE(tw->toc.selected,entry)
  switch (change) {
    case XawTocSet:
      PaintSelectionMarkOfEntry(w,NULL,tw->toc.selected,True);
      PaintSelectionMarkOfEntry(w,NULL,entry,False);
      tw->toc.selected = entry;
      break;
    case XawTocUnset:
      PaintSelectionMarkOfEntry(w,NULL,tw->toc.selected,True);
      tw->toc.selected = -1;
      break;
    default:
      INFMESSAGE(invalid action)
      break;
  }
  ENDMESSAGE(TocChangeSelected)
}

/*###################################################*/
/* TocChangeHighlighted */
/*###################################################*/

void 
TocChangeHighlighted(w,entry,change)
  Widget w;
  int entry;
  int change;
{
  TocWidget tw = (TocWidget)w;

  BEGINMESSAGE(TocChangeHighlighted)
  IIMESSAGE(tw->toc.highlighted,entry)
  switch (change) {
    case XawTocSet:
      PaintMarkOfEntry(w,NULL,tw->toc.highlighted,True);
      tw->toc.highlighted = entry;
      PaintMarkOfEntry(w,NULL,entry,False);
      break;
    case XawTocUnset:
      PaintMarkOfEntry(w,NULL,tw->toc.highlighted,True);
      tw->toc.highlighted = -1;
      break;
    default:
      INFMESSAGE(invalid action)
      break;
  }
  ENDMESSAGE(TocChangeHighlighted)
}
