;=============================================================================
; cmdline.inc:	command line parsing procedures
;
; Copyright (c) 2006, 2007 Oleg O. Chukaev.
; This code is free software distributed under the terms of the GNU GPL.
; See the file COPYING or http://www.gnu.org/licenses/licenses.html.
;
;-----------------------------------------------------------------------------
; Procedures:
;	parse_cmdline			get_param
;	get_filename			get_filenames
;	get_cmdline_tail
;
; Macros:
;	optstr				define_get_filename
;	define_parse_cmdline		define_get_param
;	define_get_filenames		define_get_cmdline_tail
;	options_begin			options_end
;
; Used global variables:
;	options				filename
;	lastfname			cmdline_tail
;
; Used global constants:
;	---
;=============================================================================


;=============================================================================
; Macro used by command line parsing procedures.  First parameter of macro
; is a letter of command line option.  Second parameter is a pointer to
; procedure, called by parser when it founds corresponding option.
;
%macro		optstr	2
		db	%1
		dw	%2
%endmacro
;
;=============================================================================
; ---
;
%macro		options_begin	0
options:
%endmacro
;
;=============================================================================
; ---
;
%macro		options_end	0
%endmacro
;=============================================================================


;=============================================================================
; parse_cmdline
;-----------------------------------------------------------------------------
; Determines the command line arguments
; In:	options -- array of structures:
;		{byte -- option char; word -> procedure}
;		if byte == 0ffh, proc. called when found `filename',
;		_array ordered by first field_.
; Out:	NC if no errors
;	CY if one of called proc. reported error or if option that present
;		in command line not found in table `options'
;		AL -- unknown option char
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
%macro		define_parse_cmdline	0
parse_cmdline:
		mov	si,81h
		mov	bl,[si-1]	;Command line length psp +80h
		mov	bh,0
		mov	[si+bx],bh	;NUL terminate
@@next_arg:
		call	get_param
		jnc	@@quit		;No more options, exit
		mov	di,options-2
@@next_char:
		scasw			;Skip over address
		scasb
		jb	@@quit		;Char not found, quit
		jne	@@next_char
		push	si
		push	ax
		call	[di]
		pop	ax
		pop	bx
		jc	@@quit
		cmp	si,bx
		jne	@@next_arg
		cmp	byte [si],0
		je	@@next_arg
		cmp	byte [si],' '
		je	@@next_arg
		cmp	byte [si],'/'
		je	@@next_arg
		stc
		mov	ah,0ffh
@@quit:
		ret
%endmacro
;
;=============================================================================
; get_param
;-----------------------------------------------------------------------------
; Retrieves the next command line option and its value, if any.
; Options may be preceded by `/' or `-'.
; In:	SI -> command line (NUL terminated)
; Out:	CY if option found
;		AL == option's letter or 0ffh if `/' or `-' not found
;		SI -> option's parameter OR
;		SI -> `filename' OR
;		SI -> whitespace
;	NC if no more options
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
%macro		define_get_param	0
get_param:
@@next:
		lodsb
		or	al,al
		jz	@@quit
		cmp	al,' '
		jbe	@@next
		cmp	al,'/'
		je	@@load_param
		cmp	al,'-'
		je	@@load_param
;-----------------------------------------------------------------------------
; Handle `filename' in command line
;-----------------------------------------------------------------------------
		dec	si
		mov	ax,00ffh
		stc
		ret
;-----------------------------------------------------------------------------
@@load_param:
		lodsb
		or	al,al
		jz	@@quit		;No more options
		cmp	byte [si],':'	;Option may have parameter, separated
		jne	@@no_colon	;by colon
		inc	si
@@no_colon:
		stc
@@quit:
		mov	ah,0
		ret
%endmacro
;
;=============================================================================
; get_filename
;-----------------------------------------------------------------------------
; ---
; NUL-terminates filename in command line.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
%macro		define_get_filename	0
get_filename:
		mov	[filename],si
@@loop:
		lodsb
		cmp	al,' '
		ja	@@loop
		dec	si
		mov	byte [si],0
		clc
		ret
%endmacro
;
;=============================================================================
; get_filenames
;-----------------------------------------------------------------------------
; Stores offsets of all filenames, contained in command line.
; NUL-terminates filenames in command line.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
%macro		define_get_filenames	0
get_filenames:
		mov	bx,[lastfname]
		mov	[bx],si
		mov	word [bx+2],0	;Terminate list of ptrs to filenames
@@loop:
		lodsb
		cmp	al,' '
		ja	@@loop
		mov	byte [si-1],0
		add	word [lastfname],2
		ret
%endmacro
;
;=============================================================================
; get_cmdline_tail
;-----------------------------------------------------------------------------
; 
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
%macro		define_get_cmdline_tail	0
get_cmdline_tail:
		mov	[cmdline_tail],si
@@loop:
		lodsb
		or	al,al
		jnz	@@loop
		dec	si
		ret
%endmacro
;
;=============================================================================
; E0F
;=============================================================================


