/*
 *      fdav.c
 *
 *      Copyright 2009 Blair <blair@blair-laptop>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *      MA 02110-1301, USA.
 */

#include "fdav.h"

struct arg_list {
    char       *arg_name;
    uint32_t    arg_flag;
};

/* All supported options are here; see the help page for descriptions */
struct arg_list fdav_args[] = {
    { "A",      FDAV_ALLDRV }, { "C",      FDAV_REMOVE },
    { "F",      FDAV_NODISP }, { "IM",     FDAV_NMOUSE },
    { "L",      FDAV_NONETD }, { "LE",     FDAV_LMOUSE },
    { "LOG",    FDAV_LOGFIL }, { "N",      FDAV_DSPTXT | FDAV_CLINEI },
    { "NGM",    FDAV_DMOUSE }, { "NLOG",   FDAV_LOGFIL | FDAV_NEWLOG },
    { "P",      FDAV_CLINEI }, { "PS2",    FDAV_RMOUSE },
    { "R",      FDAV_REPORT }, { "Q",      FDAV_QUTINE },
    { "QC",     FDAV_QCLEAN }, { "S",      FDAV_NREMOV },
    { "V",      FDAV_GETVER }, { "VIDEO",  FDAV_VIDEOM },
    { "25",     VIDEO_25    }, { "28",     VIDEO_28    },
    { "43",     VIDEO_43    }, { "50",     VIDEO_50    },
    { "60",     VIDEO_60    }, { "IN",     VIDEO_COLOR },
    { "BW",     VIDEO_BW    }, { "MONO",   VIDEO_MONO  },
    { "LCD",    VIDEO_LCD   }, { "FF",     VIDEO_FF    },
    { "ANSI",   VIDEO_ANSI  }, { "BF",     VIDEO_BIOS  },
    { "NF",     VIDEO_NF    }, { "BT",     VIDEO_BT    },
    { "?"       -1          }, { NULL,     0           }
};

/* When not in Scan All Files mode, FDAV only scans files with these extensions */
char *scan_exts[] = {
    "386",    "APP",    "BIN",    "CMD",    "DOM",    "DLL",    "DRV",
    "EXE",    "FON",    "ICO",    "OV*",    "PGM",    "PIF",    "PRG",
    "SYS",    "ARJ",    "BZ2",    "ZIP",    "CAB",    "CHM",    "BINHEX",
    "HTML",   "MBOX",   "BASE64", "UU",     "TAR",    "RTF",    "SZDD",
    "MAIL",   "DOC",    "PDF",    "PPT",    "SIS",    "TNEF",   "RAR",
    "GZ",     "MSI",    "ELF",    "SO",     "LZMA",   "7Z",     "TBZ",
    "BZ",     "TBZ2",   "CPIO",   "DEB",    "DMG",    "HFS",    "LHA",
    "LZH",    "RPM",    "SPLIT",  "SWM",    "TAZ",    "TGZ",    "TPZ",
    "WIM",    "XAR",    "Z",      "COM",    "JLM",    "PART1",  "001",
    NULL
};

char *ini_strings[] = {
    "fast_detection", "auto_save", "detection_only", "check_all_files", "anti_stealth", "prompt_while_detect",
    "create_report", "create_backup", "disable_alarm", "new_floppies", "new_checksums", "verify_integrity", "db_notemp", NULL
};

/* These are variables that FDAV uses */
uint32_t            opts = 0;
char                default_drive[] = "A:\\";
int                 files = 0;
char              **filev = NULL;
long int            viruses_removed = 0L;
long int            files_checked = 0L;
int                 db_loaded = 0;

/* The following store the location of fdav, its ini file, its quarrantine directory, and its logfile */
char               *fdav_filename = NULL;
char               *fdav_inifile = NULL;
char               *fdav_backup = NULL;
char               *fdav_logfile = NULL;
char               *fdav_dirname = NULL;
char               *fdav_helpfile = NULL;

/* The following are used by libclamav */
struct cl_engine   *scan_engine = NULL;
int                 scan_sigs = 0;
int                 scan_opts = CL_SCAN_STDOPT;
long int            scanned = 0L;
long int            viruses = 0L;

/* Should we support '-' as a switch? Also we should support switchar api */
int isarg( int ch )
{
#ifdef __UNIX__
    return( ch == '-' );
#else
    return( ch == '/' );
#endif
}

/* Parse arguments MS-DOS style */
uint32_t parseopts( char *arg, struct arg_list alist[] )
{
    char *a;
    int i;
    uint32_t retval = 0;

    if( isarg( *arg ) ) {
        while( *arg ) {
            arg++;
            for( a = arg; *a && !isarg( *a ); a++ );
            for( i = 0; ; i++ ) {
                if( !alist[ i ].arg_name ) {
                    printf( "Illegal argument: %s\n", arg - 1 );
                    return( -1 );
                }
                if( !strncasecmp( alist[ i ].arg_name, arg, a - arg ) ) {
                    retval |= alist[ i ].arg_flag;
                    break;
                }
            }

            arg = a;
        }
    }

    return( retval );
}

void help( int type )
{
    static char helpmsg[] =
        "\n"
        "Scan your computer for viruses.\n"
        "\n"
        "FDAV [drive:[path[\\file]]] [...] [options]\n"
        "\n"
        "/A     Scan all but floppy drives\n"
        "/C     Remove all of the viruses found (be cautious)\n"
        "/F     Do not display files as they are scanned (use with /N or /P\n"
        "/L     Scan all but network and floppy drives\n"
        "/LOG   Create or append to a logfile all of FDAV's activities\n"
        "/NLOG  Overwrite an existing logfile instead of appending to it\n"
        "/N     Display the contents of FDAV.TXT in place of regular interface\n"
        "/P     Use the command-line interface in place of the text interface\n"
        "/R     Create a report file in the root directory of the drive scanned\n"
        "/Q     Quarrantine found viruses and remove them\n"
        "/QC    Clean the quarrantine directory\n"
        "/S     Scan for, but don't remove any viruses (default)\n"
        "/V     Display version information\n"
        "/VIDEO Display a list of video and mouse related options\n"
        "\n";

    static char videomsg[] =
        "\n"
        "Video-related options.\n"
        "\n"
        "/25    Use 25 lines (default)\n"
        "/28    Use 28 lines (VGA only)\n"
        "/43    Use 43 lines (EGA or VGA only)\n"
        "/50    Use 50 lines (VGA only)\n"
        "/60    Use 60 lines (Video 7 display adapters only)\n"
//        "/IN    Use a color scheme even if color adapter is not detected\n"
        "/BW    Use a black and white color scheme\n"
        "/MONO  Use a monochromatic color scheme\n"
        "/LCD   Use an LCD color scheme\n"
//        "/FF    Use fast screen updating on CGA display adapters\n"
        "/ANSI  Use ANSI.SYS to display video\n"
        "/BF    Use the BIOS to display video (slow)\n"
//        "/NF    Do not use alternate fonts\n"
//        "/BT    Use a graphical mouse in windows\n"
        "\n"
        "Mouse-related options.\n"
        "/IM    Do not use the mouse\n"
        "/LE    Exchange left and right mouse buttons\n"
//        "/NGM   Use the default mouse character instead of a graphical cursor\n"
//        "/PS2   Reset the mouse if the cursor locks up or disappears\n"
        "\n";
    printf( type == 1 ? videomsg : helpmsg );
}

/* Convert a unix-style filename to a dos-style filename */
void dos_filename( char *filename )
{
    char *s = filename;

    while( *s ) {
        if( s[ 1 ] == ':' ) *s = toupper( *s );
        if( *s == '/' ) *s = '\\';
        s++;
    }
}

/* Return an array of available drives to scan - There seems to be a bug where memory gets corrupted here? */
char **get_drive_array( int types, int *read )
{
    char **array = NULL;
    FILE *drvs = setmntent( MNT_MNTTAB, "r" );
    struct mntent *ment;
    int i = 0;

    while( ( ment = getmntent( drvs ) ) ) {
        if( !( types & DRIVE_FLOPPY ) && !strcasecmp( ment->mnt_type, "fd"  ) ) continue;
        if( !( types & DRIVE_SUBST  ) && ( !strcasecmp( ment->mnt_type, "subst" ) || !strcasecmp( ment->mnt_type, "join" ) ) ) continue;
        if( !( types & DRIVE_NET    ) && !strcasecmp( ment->mnt_type, "net" ) ) continue;
        if( !( array = realloc( array, ( i + 1 ) * sizeof( void * ) ) ) ) return( NULL );
        array[ i ] = strdup( ment->mnt_dir );
        dos_filename( array[ i++ ] );
    }

    endmntent( drvs );
    *read = i;

    return( array );
}

/* Free the array returned above */
void free_drive_array( char **array, int size )
{
    int i;

    for( i = 0; i < size; i++ ) free( array[ i ] );
    free( array );
}

/* find the database and load it (should be in %DOSDIR%\lib */
int load_database( void )
{
    int r = 0, lo = CL_DB_STDOPT;
    char db_path[ PATH_MAX ];
    char test_path[ PATH_MAX ];
    char *db_test_paths[] = {
        "C:\\dos\\lib",
        "C:\\fdos\\lib",
        "\\dos\\lib",
        "\\fdos\\lib",
        "%s\\..\\lib",
        "%s",
        ".",
        ".\\lib",
        NULL
    };

    if( db_loaded ) return( 0 );

    if( TUI_OPTION( TUI_QUICK    ) ) lo  = 0;
    if( TUI_OPTION( TUI_ASTEALTH ) ) lo |= CL_DB_PUA, scan_opts |= CL_SCAN_PHISHING_BLOCKSSL | CL_SCAN_PHISHING_BLOCKCLOAK;
    if( TUI_OPTION( TUI_NOTEMP   ) ) lo |= CL_DB_CVDNOTMP;

    sprintf( db_path, "%s\\lib", getenv( "DOSDIR" ) );
    sprintf( test_path, "%s\\main.cvd", db_path );
    while( access( test_path, F_OK ) && db_test_paths[ r ] ) {
        sprintf( db_path, db_test_paths[ r++ ], fdav_dirname );
        sprintf( test_path, "%s\\main.cvd", db_path );
    }
    if( ( r = cl_init( CL_INIT_DEFAULT ) ) ) return( r );
    if( ( scan_engine = cl_engine_new() ) == NULL ) return( -1 );
    if( ( r = cl_load( db_path, scan_engine, &scan_sigs, lo ) ) ) return( r );
    if( ( r = cl_engine_compile( scan_engine ) ) ) return( r );
    db_loaded = 1;

    return( 0 );
}

/* Write options selected in the options dialog to the ini file */
void write_tui_ops( void )
{
    FILE *inif;
    int i;

    if( ( inif = fopen( fdav_inifile, "w+" ) ) == NULL ) return;
    fprintf( inif, "\n[options]\n" );
    for( i = 0; i < TUI_MAX_OPTS; i++ ) fprintf( inif, "%s = %i\n", ini_strings[ i ], av_options[ i ].state );
    fclose( inif );
}

/* Read options selected in the options dialog from the ini file */
void read_tui_ops( void )
{
    FILE *inif;
    char testbuf[ 50 ];
    int i, o, error = 0;

    if( ( inif = fopen( fdav_inifile, "r" ) ) == NULL ) error = 1;
    if( !error && fscanf( inif, "\n%15s\n", testbuf ) != 1 ) error = 1;
    if( !error && strcasecmp( testbuf, "[options]" ) ) error = 1;

    if( error ) {
        if( inif ) fclose( inif );
        write_tui_ops();
        return;
    }

    for( i = 0; i < TUI_MAX_OPTS; i++ ) {
        if( feof( inif ) && i < 12 ) {
            fclose( inif );
            write_tui_ops();
            return;
        }
        else if( feof( inif ) && i >= 12 ) o = 0;
        else if( fscanf( inif, "%40s = %i\n", testbuf, &o ) != 2 ||
            strcasecmp( testbuf, ini_strings[ i ] ) || ( o != 0 && o != 1 ) ) {
            fclose( inif );
            write_tui_ops();
            return;
        }
        av_options[ i ].state = o;
    }
    fclose( inif );
}

/* Create a report file after scanning a drive/directory in the root of the drive */
int create_report( const char *drivepath )
{
    char report_file[ PATH_MAX ], fullpath[ PATH_MAX ];
    FILE *rf;
    time_t curtime = time( NULL );

    if( realpath( drivepath, fullpath ) == NULL ) return( -1 );
    sprintf( report_file, "%.2s\\FDAV.RPT", fullpath );
    if( ( rf = fopen( report_file, "a" ) ) == NULL ) return( -1 );
    strftime( report_file, PATH_MAX, "Virus search report for date: %D, Time %T.\n\n\n", localtime( &curtime ) );
    fprintf( rf, "FreeDOS Anti-Virus.\n%s", report_file );
    fprintf( rf, "Total boot sector viruses     FOUND:0\n" );
    fprintf( rf, "Total boot sector viruses   REMOVED:0\n\n" );
    fprintf( rf, "Total Files                 CHECKED:%li\n", files_checked );
    fprintf( rf, "Total File viruses            FOUND:%li\n", viruses );
    fprintf( rf, "Total File viruses          REMOVED:%li\n\n", viruses_removed );
    fprintf( rf, "END OF REPORT.\n\n" );

    fclose( rf );

    return( 0 );
}

/* Print a log entry with a timestamp */
void logprintf( const char *format, ... )
{
    va_list args;
    char timestamp[ 100 ];
    time_t curtime;

    if( !( opts & FDAV_LOGFIL ) ) return;
    curtime = time( NULL );
    va_start( args, format );
    strftime( timestamp, 100, "FDAV@[%D]:[%T]: ", localtime( &curtime ) );
    fprintf( stderr, timestamp );
    vfprintf( stderr, format, args );
}

/* Quarrantine a virus */
int backup_file( const char *path )
{
    char backup_file[ PATH_MAX ], readbuf[ 4096 ];
    char *s = basename( path );
    FILE *old, *backup;
    int i;
    size_t bread;

    if( s == NULL ) return( -1 );
    sprintf( backup_file, "%s/%s", fdav_backup, s );
    for( i = 0; !access( backup_file, F_OK ) && i < 1000; i++ ) sprintf( backup_file, "%s/%s.%03d", fdav_backup, s, i );
    if( i == 1000 ) return( -2 );
    if( ( old = fopen( path, "rb" ) ) == NULL || ( backup = fopen( backup_file, "wb" ) ) == NULL ) {
        if( old ) fclose( old );
        return( -3 );
    }

    while( ( bread = fread( readbuf, 1, 4096, old ) ) > 0 ) fwrite( readbuf, 1, bread, backup );
    fclose( old );
    fclose( backup );

    return( 0 );
}

/* FTW handler for cleaning the qurrantine directory */
static int qclean_file( const char *path, struct stat *stbuf, int flag )
{
    if( flag != FTW_F ) return( 0 );
    unlink( path );
    printf( "Removed %s\n", path );

    return( 0 );
}

/* Call the ftw handler above with our quarrantine directory */
int clean_qdir( void )
{
    ftw( fdav_backup, qclean_file, 2 );
    return( 0 );
}

/* Test for input on stdin - used in fdav_tui */
int input_ready( void )
{
    struct timeval tm;
    fd_set inp;

    tm.tv_sec = 0;
    tm.tv_usec = 0;
    FD_ZERO( &inp );
    FD_SET( STDIN_FILENO, &inp );

    return( select( STDIN_FILENO + 1, &inp, NULL, NULL, &tm ) );
}

/* When not in scan all files mode, test for file extensions to scan */
int has_ext( const char *path )
{
    char *s = strrchr( path, '.' );
    int i;

    if( s == NULL ) return( 1 ); /* No extention; treat it as a binary */
    s++;
    for( i = 0; scan_exts[ i ]; i++ ) {
        if( !strcasecmp( s, scan_exts[ i ] ) ) return( 1 );
    }

    return( 0 );
}

int main( int argc, char** argv )
{
    uint32_t    r;
    int         i;

    for( i = 0; i < argc; i++ ) {   /* Parse all of the options first */
        r = parseopts( argv[ i ], fdav_args );
        if( r == ( uint32_t )-1 ) {
            help( 0 );
            return( 2 );
        }
        opts |= r;
        if( r ) argv[ i ] = NULL;   /* This argument has been parsed */
    }

    if( !( opts & FDAV_GETVER ) ) printf( "FreeDOS Anti-Virus " FDAV_VERSION_STRING "\n" );

    if( opts & FDAV_VIDEOM ) {
        help( 1 );
        return( 100 );
    }

    for( i = 1; i < argc; i++ ) {   /* Now check for files/drives */
		if( !argv[ i ] ) continue;
		if( !( filev = realloc( filev, ( ++files ) * sizeof( void * ) ) ) ) {
			printf( "Out of Memory!\n" );
			return( 2 );
		}
        filev[ files - 1 ] = argv[ i ];
        dos_filename( filev[ files - 1 ] );
    }

    if( !files ) {                  /* Use the current drive */
        if( opts & ( FDAV_ALLDRV | FDAV_NONETD ) ) {
            filev = get_drive_array( ( opts & FDAV_NONETD ) ? 0 : DRIVE_NET, &files );
            if( filev == NULL ) {
                printf( "Out of Memory!\n" );
                return( 2 );
            }
            if( files == 0 ) {
                printf( "No matching drives to scan.\n" );
                return( 2 );
            }
        } else {
            if( !( filev = malloc( sizeof( void * ) ) ) ) {
                printf( "Out of Memory!\n" );
                return( 2 );
            }
            filev[ 0 ] = strdup( default_drive );
            filev[ 0 ][ 0 ] += getdisk();
            files++;
        }
    }

    strlwr( argv[ 0 ] );
    fdav_filename = argv[ 0 ];
    fdav_dirname = dirname( fdav_filename );
    fdav_inifile = strdup( fdav_filename );
    fdav_backup = strdup( fdav_filename );
    fdav_helpfile = strdup( fdav_filename );

    if( opts & FDAV_LOGFIL ) {
        fdav_logfile = strdup( fdav_filename );
        if( fdav_logfile ) strcpy( strstr( fdav_logfile, ".exe" ), ".log" );
    } else fdav_logfile = "NUL";

    if( !fdav_dirname ) fdav_dirname = ".";
    strcpy( strstr( fdav_inifile, ".exe" ), ".ini" );
    strcpy( strstr( fdav_backup, ".exe" ), ".000" );
    strcpy( strstr( fdav_helpfile, ".exe" ), ".hlp" );

    if( fdav_inifile == NULL || fdav_backup == NULL || fdav_logfile == NULL || fdav_helpfile == NULL ) {
        printf( "Out of Memory!\n" );
        return( 2 );
    }

    if( access( fdav_backup, D_OK ) ) {
        if( mkdir( fdav_backup, S_IWUSR ) ) {
            printf( "Coult not open or create quarantine directory: %s.\n", fdav_backup );
            return( 2 );
        }
        _chmod( fdav_backup, 1, _A_HIDDEN | _A_SUBDIR );
    }

    read_tui_ops();

    freopen( fdav_logfile, opts & FDAV_NEWLOG ? "w" : "a", stderr );
    setbuf( stderr, NULL );

    if( opts & FDAV_GETVER ) {
        i = 0;
        printf( FDAV_VERSION_STRING "-%s\n", cl_retver() );
    }
    else if( opts & FDAV_QCLEAN ) i = clean_qdir();
    else if( opts & FDAV_CLINEI ) i = fdav_cli();
    else i = fdav_tui();

    if( TUI_OPTION( TUI_ASAVE ) ) write_tui_ops();

    return( i );
}
