/*
 *      fdav_tui.c
 *
 *      Copyright 2009 Blair <blair@blair-laptop>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *      MA 02110-1301, USA.
 */

#include "fdav.h"

#define MAIN_WINDOW_COLS        ( ( COLS * 70 ) / 80 )
#define MAIN_WINDOW_LINES       ( ( LINES * 20 ) / 25 )
#define MAIN_WINDOW_X           ( dlg_box_x_ordinate( MAIN_WINDOW_COLS  ) )
#define MAIN_WINDOW_Y           ( dlg_box_y_ordinate( MAIN_WINDOW_LINES ) )
#define SIDEBAR_WINDOW_COLS     ( MAIN_WINDOW_COLS - 32 )
#define SIDEBAR_WINDOW_LINES    ( MAIN_WINDOW_LINES - 4 )
#define SIDEBAR_WINDOW_X        ( dlg_box_x_ordinate( MAIN_WINDOW_COLS  ) + 28 )
#define SIDEBAR_WINDOW_Y        ( dlg_box_y_ordinate( MAIN_WINDOW_LINES ) + 2 )
#define SIDEBAR_LIST_LINES      ( SIDEBAR_WINDOW_LINES - 4 )

struct tui_help_button_info {
    int     thbi_startx;
    int     thbi_blen;
};

char *ini_strings[] = {
    "fast_detection", "auto_save", "detection_only", "check_all_files", "anti_stealth", "prompt_while_detect",
    "create_report", "create_backup", "disable_alarm", "new_floppies", "new_checksums", "verify_integrity", "db_notemp", NULL
};

/* Todo - Create our own structure for these options as we only use the first and last fields */
DIALOG_LISTITEM av_options[] = {
    { "        Quick Scan       ", NULL, NULL, 0 },
    { "        Auto Save        ", NULL, NULL, 1 },
    { " Only Detect the Viruses ", NULL, NULL, 0 },
    { "     Check All Files     ", NULL, NULL, 1 },
    { "    Anti-Stealth Mode    ", NULL, NULL, 1 },
    { " Prompt When Virus Found ", NULL, NULL, 1 },
    { "   Create a Report File  ", NULL, NULL, 0 },
    { "    Quarantine Viruses   ", NULL, NULL, 0 },
    { "   Disable Alarm Sound   ", NULL, NULL, 0 },
    { " Make Checksums on Floppy", NULL, NULL, 0 },
    { "    Make New Checksums   ", NULL, NULL, 0 },
    { "     Verify Integrity    ", NULL, NULL, 0 },
    { "  Do Not Use Temp Files  ", NULL, NULL, 0 }
};

/* The various button arrays used */
const char *tui_buttons[] = {
    "      Detect      ",
    " dEtect and Clean ",
    " Select New Drive ",
    "     Options      ",
    "       Help       ",
    "       eXit       ",
    NULL
};

const char *ok_button[] = {
    "OK", NULL
};

const char *yesno_buttons[] = {
    "Yes", "No", NULL
};

const char *sel_buttons[] = {
    "Ingore", "Remove", "Backup", NULL
};

struct tui_button_info {
    char *title;
    char *msg;
};

/* Information for all of the main buttons */
struct tui_button_info tui_info[] = {
    { "Detect", "Scan the selected drive or directory for viruses without removing them." },
    { "Detect and Clean", "Scan the selected drive or directory for viruses, removing them and taking any other specified action." },
    { "Select New Drive", "Select a new drive or directory to scan." },
    { "Options", "Opens a dialog box from which you can choose various scanning options. The options can be auto-saved on exit." },
    { "Help", "Launch the help viewer." },
    { "Exit", "Exit FreeDOS Anti-Virus to DOS." }
};

/* These symbols are in conio.h, but it can't be included with ncurses.h */
extern void _set_screen_lines( int nlines );
static int av_rows;

static int tui_sidebar_retval = 0;

static fsblkcnt_t fs_used;
static fsblkcnt_t fs_read;
static unsigned long fs_bsize;
static WINDOW *ftw_window;
static WINDOW *ftw_progress_window;
static WINDOW *ftw_parent_window;
static char *ftw_error;
static int show_progress;

/* This needs the most work - have ftw first find all of the files to scan, THEN calculate progress bar parameters */
void progress_bar( WINDOW *win, int max, int cur, int maxwidth )
{
    int cur_y, cur_x, i, j, dval = max / maxwidth;
    chtype save = getattrs( win );

    if( !dval ) dval = 1;
    j = cur / dval;
    if( j > maxwidth ) j = maxwidth;

    getyx( win, cur_y, cur_x );
    wmove( win, 1, 2 );
    wattrset( win, item_selected_attr );

    for( i = 0; i < j; i++ ) waddch( win, ' ' );
    wattrset( win, save );
    for( ; i < maxwidth; i++ ) waddch( win, ' ' );

    wnoutrefresh( win );
    wmove( win, cur_y, cur_x );
    wrefresh( win );
}

void clear_section( WINDOW *win, int x, int y, int max_x, int max_y, chtype attr )
{
    int cur_x, cur_y = y, old_x, old_y;
    chtype save = getattrs( win );

    getyx( win, old_y, old_x );
    wattrset( win, attr );
    while( cur_y <= max_y ) {
        for( cur_x = x; cur_x <= max_x; cur_x++ ) {
            wmove( win, cur_y, cur_x );
            waddch( win, ' ' );
        }
        cur_y++;
        wmove( win, cur_y, x );
    }
    wmove( win, old_y, old_x );
    wattrset( win, save );
}

void inline clear_ftw_window( void )
{
    clear_section( ftw_window, 1, 1, SIDEBAR_WINDOW_COLS - 3, SIDEBAR_LIST_LINES - 2, getattrs( ftw_window ) );
}

void tui_print_text( WINDOW *win, const char *msg, int maxy, int maxx )
{
    chtype save = getattrs( win );

    wattrset( win, tag_attr );
    dlg_print_autowrap( win, msg, maxy, maxx );
    wattrset( win, save );
}

static int ftw_handler( const char *path, struct stat *stbuf, int flag )
{
    int r, key, fkey;
    const char *vname;
    static char errbuf[ 4096 ];

    if( flag != FTW_F ) return( 0 );
    dos_filename( ( char * )path );
    if( !TUI_OPTION( TUI_ALLF ) && !has_ext( path ) ) return( 0 );

    clear_ftw_window();

    while( input_ready() ) { /* Test if user wants to abort */
        key = dlg_mouse_wgetch( ftw_window, &fkey );
        if( key == ESCAPE_KEY ) {
            logprintf( "User pressed ESC key during scanning operation\n" );
            strcpy( errbuf, "Really stop scanning?" );
            tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, yesno_buttons );
            ftw_error = NULL;
            if( !tui_sidebar_retval ) {
                logprintf( "User aborted scanning operation\n" );
                return( -1 );
            }
        }
        clear_ftw_window();
        if( show_progress ) progress_bar( ftw_progress_window, fs_used, fs_read, SIDEBAR_WINDOW_COLS - 5 );
    }

    tui_print_text( ftw_window, path, SIDEBAR_LIST_LINES, SIDEBAR_WINDOW_COLS - 2 );
    dlg_draw_title( ftw_progress_window, "Progress" );
    wnoutrefresh( ftw_window );
    wrefresh( ftw_window );

    if( ( r = cl_scanfile( path, &vname, &scanned, scan_engine, &vlimits, CL_SCAN_STDOPT ) ) == CL_VIRUS ) {
        viruses++;
        logprintf( "The virus \"%s\" was found in \"%s\"\n", vname, path );
        if( !TUI_OPTION( TUI_NALARM ) ) beep();
        if( TUI_OPTION( TUI_PROMPT ) ) {
            sprintf( errbuf, "%s is INFECTED with %s\nWhich action should be taken?", path, vname );
            tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, sel_buttons );
        } else if( opts & FDAV_REMOVE ) tui_sidebar_retval = TUI_CHOICE_REMOVE;
        else {
            tui_sidebar_retval = 0;
            sprintf( errbuf, "%s is INFECTED with %s.", path, vname );
            tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, ok_button );
        }
        if( tui_sidebar_retval > 0 && ( tui_sidebar_retval == TUI_CHOICE_BACKUP || TUI_OPTION( TUI_BACKUP ) || opts & FDAV_QUTINE ) ) {
            logprintf( "Attempting to quarrantine \"%s\"\n", path );
            if( backup_file( path ) ) {
                logprintf( "Could not quarrantine the file\n" );
                sprintf( errbuf, "Could not quarrantine %s", path );
                tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, ok_button );
                tui_sidebar_retval = 0;
            }
        }
        if( tui_sidebar_retval > 0 ) {
            logprintf( "Attempting to remove \"%s\"\n", path );
            if( unlink( path ) ) {
                logprintf( "Could not remove the virus\n" );
                sprintf( errbuf, "Could not remove %s: %s", path, strerror( errno ) );
                tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, ok_button );
            } else viruses_removed++;
        }
    } else if( r != CL_CLEAN ) {
        logprintf( "cl_scanfile reported an error while scanning \"%s\": %s\n", path, cl_strerror( r ) );
        sprintf( errbuf, "Error scanning %s: %s\nContinue Scanning?", path, cl_strerror( r ) );
        tui_draw_sidebar( ftw_parent_window, TUI_BUTTON_DETECT, TUI_SIDEBAR_MSGBOX, errbuf, yesno_buttons );
        return( tui_sidebar_retval ? -1 : 0 );
    }

    logprintf( "Successfully scanned or acted on \"%s\"\n", path );
    fs_read += ( stbuf->st_size / fs_bsize );
    if( show_progress ) progress_bar( ftw_progress_window, fs_used, fs_read, SIDEBAR_WINDOW_COLS - 5 );
    files_checked++;

    return( 0 );
}

void tui_list_arrows( WINDOW *win, int x, int y, int scroll, int choice, int nitems, int height )
{
    dlg_draw_arrows( win, scroll, scroll + choice < nitems, 1, 0, SIDEBAR_LIST_LINES + 1 );
}

void tui_list_item( WINDOW *win, DIALOG_LISTITEM *item, const char *states, int choice, int selected, int flag )
{
    chtype save = getattrs(win);
    int i;

    /* Clear 'residue' of last item */
    wattrset( win, menubox_attr );
    wmove( win, choice + 1, 2 );
    for( i = 0; i < SIDEBAR_WINDOW_COLS - 4; i++ ) waddch( win, ' ' );
    wnoutrefresh( win );

    wmove( win, choice + 1, 2 );
    wattrset( win, selected ? check_selected_attr : check_attr );
    wprintw( win, flag ? "[%c]" : "(%c)", states[ item->state ] );
    wattrset( win, menubox_attr ) ;
    waddch( win, ' ' );

    wattrset( win, selected ? tag_selected_attr : tag_attr );
    waddnstr( win, item->name, SIDEBAR_WINDOW_COLS - 7 );
    if( selected ) waddch( win, '-' );

    wattrset( win, save );
    wnoutrefresh( win );
}

int tui_sidebar_checklist( WINDOW *sidebar, int y, int x, int nitems, DIALOG_LISTITEM *items, int flag )
{
    static DLG_KEYS_BINDING binding[] = {
        ENTERKEY_BINDINGS,
        DLG_KEYS_DATA( DLGK_FIELD_NEXT, KEY_RIGHT ),
        DLG_KEYS_DATA( DLGK_FIELD_NEXT, TAB ),
        DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_BTAB ),
        DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_LEFT ),
        DLG_KEYS_DATA( DLGK_ITEM_FIRST, KEY_HOME ),
        DLG_KEYS_DATA( DLGK_ITEM_LAST,	KEY_END ),
        DLG_KEYS_DATA( DLGK_ITEM_LAST,	KEY_LL ),
        DLG_KEYS_DATA( DLGK_ITEM_NEXT,	'+' ),
        DLG_KEYS_DATA( DLGK_ITEM_NEXT,	KEY_DOWN ),
        DLG_KEYS_DATA( DLGK_ITEM_NEXT,  CHR_NEXT ),
        DLG_KEYS_DATA( DLGK_ITEM_PREV,	'-' ),
        DLG_KEYS_DATA( DLGK_ITEM_PREV,	KEY_UP ),
        DLG_KEYS_DATA( DLGK_ITEM_PREV,  CHR_PREVIOUS ),
        DLG_KEYS_DATA( DLGK_PAGE_NEXT,	KEY_NPAGE ),
        DLG_KEYS_DATA( DLGK_PAGE_NEXT,	DLGK_MOUSE(KEY_NPAGE) ),
        DLG_KEYS_DATA( DLGK_PAGE_PREV,	KEY_PPAGE ),
        DLG_KEYS_DATA( DLGK_PAGE_PREV,	DLGK_MOUSE(KEY_PPAGE) ),
        END_KEYS_BINDING
    };
    char *check = " X";
    int button = 0, scroll = 0, choice = 0;
    const char **buttons = dlg_ok_labels();
    int max_disp, i, result = DLG_EXIT_UNKNOWN, key, fkey, cur_x, cur_y, found, mkey;

    dlg_register_window( sidebar, "sidebar_checklist", binding );
    dlg_register_buttons( sidebar, "sidebar_checklist", buttons );
    dlg_does_output();
    dlg_mouse_setbase( x, y );

    dlg_draw_bottom_box( sidebar );
    wattrset( sidebar, menubox_attr );

    if( !flag ) check[ 1 ] = '*';
    else check[ 1 ] = 'X';
    max_disp = MIN( nitems, SIDEBAR_LIST_LINES );

    for( i = 0; i < max_disp; i++ ) {
        tui_list_item( sidebar, &items[ i ], check, i, i == choice, flag );
    }
    wnoutrefresh( sidebar );

    dlg_mouse_mkbigregion( 1, 2, SIDEBAR_LIST_LINES, SIDEBAR_WINDOW_COLS - 2, KEY_MAX, 1, 1, 1 );

    tui_list_arrows( sidebar, x, y, scroll, max_disp, nitems, SIDEBAR_LIST_LINES );
    dlg_draw_buttons( sidebar, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );

    while( result == DLG_EXIT_UNKNOWN ) {
        key = dlg_mouse_wgetch( sidebar, &fkey );
        if( dlg_result_key( key, fkey, &result ) ) break;

        mkey = ( fkey && is_DLGK_MOUSE( key ) );
        if( mkey ) key -= M_EVENT;

        if( mkey && ( key >= KEY_MAX ) ) { /* A mouse clicked on an item */
            getyx( sidebar, cur_y, cur_x );
            i = ( key - KEY_MAX );
            if( i > max_disp ) continue;
            tui_list_item( sidebar, &items[ scroll + choice ], check, choice, FALSE, flag );
            choice = i;
            tui_list_item( sidebar, &items[ scroll + choice ], check, choice, TRUE, flag );
            wnoutrefresh( sidebar );
            wmove( sidebar, cur_y, cur_x );

            key = ' ';	/* force the selected item to toggle */
            fkey = FALSE;
        } else if( mkey && key >= KEY_MIN ) {
            key = dlg_lookup_key( sidebar, key, &fkey );
        }

        if( key == ' ' ) {
            int curchoice = scroll + choice;
            int toggle = items[ curchoice ].state ? 0 : 1;

            getyx( sidebar, cur_y, cur_x );
            if( flag ) { /* Checklist */
                items[ curchoice ].state = toggle;
                tui_list_item( sidebar, &items[ curchoice ], check, choice, TRUE, flag );
            } else { /* Radiolist */
                for( i = 0; i < nitems; i++ ) if( i != curchoice ) items[ i ].state = 0;
                if( !items[ curchoice ].state ) {
                    items[ curchoice ].state = toggle;
                    for( i = 0; i < max_disp; i++ ) tui_list_item( sidebar, &items[ scroll + i ], check, i, i == choice, flag );
                }
            }

            wnoutrefresh( sidebar );
            wmove( sidebar, cur_y, cur_x );
            continue;
        }

        found = 1;
        if( fkey ) {
            switch( key ) {
                case DLGK_ITEM_FIRST:
                    i = -scroll;
                    break;
                case DLGK_ITEM_LAST:
                    i = nitems - 1 - scroll;
                    break;
                case DLGK_PAGE_PREV:
                    if( choice ) i = 0;
                    else if( scroll != 0 ) i = -MIN( scroll, max_disp );
                    else continue;
                    break;
                case DLGK_PAGE_NEXT:
                    i = MIN( choice + max_disp, nitems - scroll - 1);
                    break;
                case DLGK_ITEM_PREV:
                    i = choice - 1;
                    if( choice == 0 && scroll == 0 ) continue;
                    break;
                case DLGK_ITEM_NEXT:
                    i = choice + 1;
                    if( scroll + choice >= nitems - 1 ) continue;
                    break;
                default:
                    found = 0;
            }

            if( found ) {
                if( i == choice ) continue;
                getyx( sidebar, cur_y, cur_x );
                if( i < 0 || i >= max_disp ) {
                    if( i < 0 ) {
                        scroll += i;
                        choice = 0;
                    } else {
                        choice = max_disp - 1;
                        scroll += ( i - max_disp + 1 );
                    }
                    logprintf( "Scrolling the option dialog; scroll #%i\n", scroll );
                    for( i = 0; i < max_disp; i++ ) tui_list_item( sidebar, &items[ scroll + i ], check, i, i == choice, flag );
                    logprintf( "Finished scrolling\n" );
                } else {
                    tui_list_item( sidebar, &items[ scroll + choice ], check, choice, FALSE, flag );
                    choice = i;
                    tui_list_item( sidebar, &items[ scroll + choice ], check, choice, TRUE, flag );
                }
                tui_list_arrows( sidebar, x, y, scroll, max_disp, nitems, SIDEBAR_LIST_LINES );
                wnoutrefresh( sidebar );
                wmove( sidebar, cur_y, cur_x );
                doupdate();

                continue;
            }
        }

        if( fkey ) switch( key ) {
            case DLGK_ENTER:
                result = dlg_ok_buttoncode( button );
                break;
            case DLGK_FIELD_PREV:
                button = dlg_prev_button( buttons, button );
                dlg_draw_buttons( sidebar, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );
                break;
            case DLGK_FIELD_NEXT:
                button = dlg_next_button( buttons, button );
                dlg_draw_buttons( sidebar, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );
                break;
            default:
                if( mkey ) {
                    if( ( result = dlg_ok_buttoncode( key ) ) >= 0 ) break;
                    else result = DLG_EXIT_UNKNOWN;
                }
        }
    }

    dlg_unregister_window( sidebar );
    dlg_mouse_free_regions();

    return( result );
}

/* The following is a snipped from dialog's ui_getc.c */

static void dlg_raise_window( WINDOW *win )
{
    touchwin( win );
    wmove( win, getcury( win ), getcurx( win ) );
    wnoutrefresh( win );
    doupdate();
}

void tui_drawback( void )
{
    dlg_clear();
    dialog_vars.backtitle = "FreeDOS Anti-Virus 0.1";
    dlg_put_backtitle();
    dlg_raise_window( stdscr );
}

void tui_draw_sidebar( WINDOW *parent, int button, int type, ... )
{
    static WINDOW *iwin = NULL, *fwin = NULL;
    int x = SIDEBAR_WINDOW_X, y = SIDEBAR_WINDOW_Y;

    if( !iwin ) {
        iwin = dlg_sub_window( parent, SIDEBAR_WINDOW_LINES, SIDEBAR_WINDOW_COLS, y, x );
        fwin = dlg_sub_window( iwin, 3, SIDEBAR_WINDOW_COLS, y + SIDEBAR_LIST_LINES + 1, x );
    } else {
        wclear( iwin );
    }
    dlg_draw_box( iwin, 0, 0, SIDEBAR_WINDOW_LINES, SIDEBAR_WINDOW_COLS, menubox_border_attr, menubox_attr );
    dlg_draw_title( iwin, tui_info[ button ].title );

    if( type == TUI_SIDEBAR_DETECT ) {
        struct statvfs fsinf;
        struct stat finf;
        int i, r;
        char *cur_file;
        char errbuf[ 4096 ];

        dlg_draw_bottom_box( iwin );

        if( !db_loaded ) {
            logprintf( "Loading the virus database...\n" );
            tui_print_text( iwin, "Loading virus database...", SIDEBAR_LIST_LINES, SIDEBAR_WINDOW_COLS - 2 );
            wnoutrefresh( iwin );
            wrefresh( iwin );
        }

        if( ( r = load_database() ) ) {
            logprintf( "Could not load the virus database\n" );
            sprintf( errbuf, "Virus database load error: %s", cl_strerror( r ) );
            tui_draw_sidebar( parent, button, TUI_SIDEBAR_MSGBOX, errbuf, ok_button );
            return;
        }

        ftw_window = iwin;
        ftw_progress_window = fwin;
        ftw_parent_window = parent;
        if( button == TUI_BUTTON_DCLEAN ) opts |= FDAV_REMOVE;
        else opts &= ~FDAV_REMOVE;

        for( i = 0; i < files; i++ ) {
            cur_file = filev[ i ];
            if( statvfs( cur_file, &fsinf ) ) show_progress = 0;
            else show_progress = 1;
            fs_used = fsinf.f_blocks - fsinf.f_bfree;
            fs_bsize = fsinf.f_bsize;
            fs_read = 0;

            logprintf( "Scanning file, directory, or drive: %s\n", cur_file );
            if( access( cur_file, D_OK ) == 0 ) r = ftw( cur_file, ftw_handler, 16 );
            else {
                show_progress = 0;
                stat( cur_file, &finf );
                r = ftw_handler( cur_file, &finf, FTW_F );
            }

            if( r ) {
                if( ftw_error ) tui_draw_sidebar( parent, button, TUI_SIDEBAR_MSGBOX, ftw_error, ok_button );
                return;
            }

            if( ( TUI_OPTION( TUI_REPORT ) || opts & FDAV_REPORT ) && create_report( cur_file ) < 0 ) {
                logprintf( "Could not create a logfile\n" );
                sprintf( errbuf, "Could not create report file for %s.", cur_file );
                tui_draw_sidebar( parent, button, TUI_SIDEBAR_MSGBOX, errbuf, ok_button );
            }
            viruses = viruses_removed = files_checked = 0L;
        }
    }
    if( type == TUI_SIDEBAR_DRIVES ) {
        DIALOG_LISTITEM *sel_drives;
        char **sdrives;
        int i, j, ndrives;

        logprintf( "Giving the user a choice of drives\n" );
        sdrives = get_drive_array( DRIVE_ALL, &ndrives );
        if( sdrives == NULL ) {
            return;
        }

        if( ( sel_drives = malloc( ndrives * sizeof( DIALOG_LISTITEM ) ) ) == NULL ) {
            return;
        }

        for( i = 0; i < ndrives; i++ ) {
            if( ( sel_drives[ i ].name = strdup( sdrives[ i ] ) ) == NULL ) {
                for( j = 0; j < i; j++ ) free( sel_drives[ j ].name );
                free( sel_drives );
                return;
            }
            if( !strcasecmp( sel_drives[ i ].name, filev[ 0 ] ) ) sel_drives[ i ].state = 1;
        }
        free_drive_array( sdrives, ndrives );

        if( !tui_sidebar_checklist( iwin, y, x, ndrives, sel_drives, 0 ) ) {
            free_drive_array( filev, files );
            filev = malloc( sizeof( char * ) );
            files = 1;
            for( i = 0; i < ndrives; i++ ) if( sel_drives[ i ].state ) filev[ 0 ] = sel_drives[ i ].name; else free( sel_drives[ i ].name );
        }
        free( sel_drives );
    }
    if( type == TUI_SIDEBAR_OPTIONS ) {
        int i;
        int saved[ TUI_MAX_OPTS ];

        logprintf( "Displaying the option box dialog\n" );
        for( i = 0; i < TUI_MAX_OPTS; i++ ) saved[ i ] = av_options[ i ].state;
        if( tui_sidebar_checklist( iwin, y, x, TUI_MAX_OPTS, av_options, 1 ) )
            for( i = 0; i < TUI_MAX_OPTS; i++ ) av_options[ i ].state = saved[ i ];
        else {
            if( saved[ TUI_ASAVE ] && !TUI_OPTION( TUI_ASAVE ) ) write_tui_ops();
            if( db_loaded )
                tui_draw_sidebar( parent, button, TUI_SIDEBAR_MSGBOX, "FDAV may need to be restarted for some options to take effect.", ok_button );
        }
    }
    if( type == TUI_SIDEBAR_MSGBOX ) {
        va_list margs;
        char *msg;
        static DLG_KEYS_BINDING binding[] = {
            ENTERKEY_BINDINGS,
            DLG_KEYS_DATA( DLGK_ENTER,	' ' ),
            DLG_KEYS_DATA( DLGK_FIELD_NEXT,	KEY_DOWN ),
            DLG_KEYS_DATA( DLGK_FIELD_NEXT, KEY_RIGHT ),
            DLG_KEYS_DATA( DLGK_FIELD_NEXT, TAB ),
            DLG_KEYS_DATA( DLGK_FIELD_PREV,	KEY_UP ),
            DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_BTAB ),
            DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_LEFT ),
            END_KEYS_BINDING
        };
        int key, fkey, button = 0;
        const char **buttons;

        va_start( margs, type );
        msg = va_arg( margs, char * );
        buttons = va_arg( margs, const char ** );
        logprintf( "Printing a msgbox of \"%s\"\n", msg );

        dlg_register_window( iwin, "sidebar", binding );
        dlg_register_buttons( iwin, "sidebar", buttons );
        dlg_draw_bottom_box( iwin );
        wattrset( iwin, menubox_attr );
        tui_print_text( iwin, msg, SIDEBAR_LIST_LINES, SIDEBAR_WINDOW_COLS );
        dlg_draw_buttons( iwin, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );

        for( ;; ) {
            key = dlg_mouse_wgetch( iwin, &fkey );
            if( dlg_char_to_button( key, buttons ) != DLG_EXIT_UNKNOWN ) break;
            if( fkey ) {
                if( fkey == DLGK_FIELD_NEXT ) {
                    button = dlg_next_button( buttons, button );
                    if( button < 0 ) button = 0;
                    dlg_draw_buttons( iwin, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );
                }
                if( fkey == DLGK_FIELD_PREV ) {
                    button = dlg_prev_button( buttons, button );
                    if( button < 0 ) button = 0;
                    dlg_draw_buttons( iwin, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );
                }
                if( fkey == DLGK_ENTER ) {
                    break;
                }
                if( !is_DLGK_MOUSE( key ) ) continue;
                key -= M_EVENT;
                if( key >= 0 ) {
                    button = key;
                    dlg_draw_buttons( iwin, SIDEBAR_WINDOW_LINES - 2, 0, buttons, button, FALSE, SIDEBAR_WINDOW_COLS + 1 );
                    break;
                }
            }
        }

        dlg_unregister_window( iwin );
        tui_sidebar_retval = button;
    }
    if( type == TUI_SIDEBAR_INFO ) {
        dlg_draw_bottom_box( iwin );
        dlg_draw_title( fwin, "Selected Drive/Directory" );
        wattrset( iwin, menubox_attr );
        tui_print_text( iwin, tui_info[ button ].msg, SIDEBAR_LIST_LINES, SIDEBAR_WINDOW_COLS );
        tui_print_text( fwin, filev[ 0 ], 3, SIDEBAR_WINDOW_COLS - 2 );
    }

    wnoutrefresh( iwin );
    wrefresh( iwin );
}

void inline tui_main_draw_buttons( WINDOW *win, int button )
{
    dlg_draw_buttons( win, ( ( MAIN_WINDOW_LINES - 2 ) / 2 ) - 6, 3, tui_buttons, button, TRUE, 20 );
}

int tui_main( int cur_button )
{
    int x = MAIN_WINDOW_X, y = MAIN_WINDOW_Y;
    static DLG_KEYS_BINDING binding[] = {
        ENTERKEY_BINDINGS,
        DLG_KEYS_DATA( DLGK_ENTER,	' ' ),
        DLG_KEYS_DATA( DLGK_FIELD_NEXT,	KEY_DOWN ),
        DLG_KEYS_DATA( DLGK_FIELD_NEXT, KEY_RIGHT ),
        DLG_KEYS_DATA( DLGK_FIELD_NEXT, TAB ),
        DLG_KEYS_DATA( DLGK_FIELD_PREV,	KEY_UP ),
        DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_BTAB ),
        DLG_KEYS_DATA( DLGK_FIELD_PREV, KEY_LEFT ),
        END_KEYS_BINDING
    };
    int button = cur_button, selected = DLG_EXIT_UNKNOWN, fkey, key;
    WINDOW *main_win;

    tui_drawback();
    main_win = dlg_new_window( MAIN_WINDOW_LINES, MAIN_WINDOW_COLS, y, x );
    dlg_register_window( main_win, "fdav", binding );
    dlg_register_buttons( main_win, "fdav", tui_buttons );
    dlg_draw_box( main_win, 0, 0, MAIN_WINDOW_LINES, MAIN_WINDOW_COLS, dialog_attr, border_attr );
    dlg_draw_title( main_win, "Main Menu" );
    wattrset( main_win, dialog_attr );

    while( selected == DLG_EXIT_UNKNOWN ) {
        tui_main_draw_buttons( main_win, button );
        tui_draw_sidebar( main_win, button, TUI_SIDEBAR_INFO );
        key = dlg_mouse_wgetch( main_win, &fkey );
        if( key == ESCAPE_KEY ) {
            selected = TUI_BUTTON_AVEXIT;
            break;
        }
        if( ( selected = dlg_char_to_button( key, tui_buttons ) ) != DLG_EXIT_UNKNOWN ) break;
        if( fkey ) switch( fkey ) {
            case DLGK_FIELD_NEXT:
                button = dlg_next_button( tui_buttons, button );
                if( button < 0 ) button = 0;
                tui_main_draw_buttons( main_win, button );
                break;
            case DLGK_FIELD_PREV:
                button = dlg_prev_button( tui_buttons, button );
                if( button < 0 ) button = 0;
                tui_main_draw_buttons( main_win, button );
                break;
            case DLGK_ENTER:
                selected = button;
                break;
            default:
                if( !is_DLGK_MOUSE( key ) ) break;
                selected = button = key - M_EVENT;
                tui_main_draw_buttons( main_win, button );
                if( selected < 0 ) {
                    selected = DLG_EXIT_UNKNOWN;
                    button = 0;
                    tui_main_draw_buttons( main_win, button );
                }
        }

        if( selected == TUI_BUTTON_DETECT || selected == TUI_BUTTON_DCLEAN ) {
            selected = DLG_EXIT_UNKNOWN;
            tui_draw_sidebar( main_win, button, TUI_SIDEBAR_DETECT );
        }
        if( selected == TUI_BUTTON_NDRIVE ) {
            selected = DLG_EXIT_UNKNOWN;
            tui_draw_sidebar( main_win, button, TUI_SIDEBAR_DRIVES );
        }
        if( selected == TUI_BUTTON_OPTION ) {
            selected = DLG_EXIT_UNKNOWN;
            tui_draw_sidebar( main_win, button, TUI_SIDEBAR_OPTIONS );
        }
    }

    dlg_del_window( main_win );
    return( selected );
}

void tui_cleanup( void )
{
    end_dialog();
    _set_screen_lines( av_rows );
}

void tui_clearscr( void )
{
    printf( "%c[K%c[2J%c[1A", ESCAPE_KEY, ESCAPE_KEY, ESCAPE_KEY );
}

extern void *__libc_write_termios_hook;

int fdav_tui( void )
{
    int        lines = 25;
    int        b = 0;

    av_rows = ScreenRows();
    /* Set TERM up because FreeDOS doesn't set it by default */
    if( opts & ( VIDEO_BW | VIDEO_MONO | VIDEO_LCD ) ) putenv( "TERM=djgpp-m" );
    else putenv( "TERM=djgpp" ); /* Seems to work the best for libdialog */
    ESCDELAY = 0; /* We shouldn't be recieving any escape sequences from stdin anyway; speed up ESC handling */

    if( opts & VIDEO_BIOS ) putenv( "TTY_SCREEN_INTFACE=BIOS" );
    if( opts & VIDEO_LINES ) {
        if( opts & VIDEO_28 ) lines = 28;
        if( opts & VIDEO_43 ) lines = 43;
        if( opts & VIDEO_50 ) lines = 50;
        if( opts & VIDEO_60 ) lines = 60;

        _set_screen_lines( lines );
    }

    if( opts & ( FDAV_NMOUSE ) ) putenv( "NCURSES_GPM_TERMS=none" ); /* Disables GPM */
    atexit( tui_cleanup );
    init_dialog( stdin, stderr );
    button_inactive_attr = tag_attr;
    button_label_inactive_attr = tag_attr;
    if( opts & FDAV_LMOUSE ) {
        mouse_close();
        mousemask( BUTTON3_CLICKED, ( mmask_t * )0 );
    }

    while( ( b = tui_main( b ) ) != TUI_BUTTON_AVEXIT );

    end_dialog();

    _set_screen_lines( av_rows );

    return( 0 );
}
