//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: visual plug-in DLL : text-desktop block/mosaic random shift

#include <display\visualpi.h>
#include <newfunc\newfunc.h>
#include <newfunc\dll_load.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>

//#define MOSAIK_DEBUG 1

#ifdef MOSAIK_DEBUG
 static void mosaik_debugf(const char *format, ...);
#else
 __inline void mosaik_debugf(const char *format, ...){}
#endif

static struct mpxplay_resource_s *mrs;

#define BLOCK_SHIFT_VERTICAL   0
#define BLOCK_SHIFT_HORIZONTAL 1

#define BLOCK_SHIFT_UP    -1
#define BLOCK_SHIFT_DOWN  +1
#define BLOCK_SHIFT_LEFT  -1
#define BLOCK_SHIFT_RIGHT +1

typedef struct local_data_s{
 unsigned long  textscreen_target_address; // the real textscreen address (0xb8000)
 unsigned char *textscreen_source_address; // redirected address
 unsigned int  textscreen_resolution_x,textscreen_resolution_y;
 unsigned long textscreen_size;
 unsigned long *transfer_table;
 unsigned long tftable_size;
 unsigned int  blocknum_x,blocknum_y,blocksize_x,blocksize_y;
 unsigned int  currblock_x,currblock_y;       // current empy block
 unsigned int  shift_vertical_or_horizontal;  //  0:vertical or 1:horizontal
 int           shift_direction;               // -1 or +1
 unsigned int  shift_pos;
 unsigned long blackhole;
 display_visual_data_s *vds;
}local_data_s;

static void vis_close(void *ldp);
static void mosaic_get_textscreen_resolution(struct local_data_s *lds);
static void mosaic_init_transfer_table(struct local_data_s *lds);
static void mosaic_shift_block(struct local_data_s *lds);
static void mosaic_apply_transfer_table(struct local_data_s *lds);
static unsigned long pds_rand(int maxnum);

static unsigned int mosaic_alloc_fields(struct local_data_s *lds)
{
 lds->textscreen_size=lds->textscreen_resolution_x*lds->textscreen_resolution_y*TEXTSCREEN_BYTES_PER_CHAR;
 lds->textscreen_source_address=(unsigned char *)malloc(lds->textscreen_size);
 if(!lds->textscreen_source_address)
  return 0;
 memset(lds->textscreen_source_address,0,lds->textscreen_size);

 lds->tftable_size   =lds->textscreen_resolution_x*lds->textscreen_resolution_y;
 lds->transfer_table =(unsigned long *)malloc(lds->tftable_size*sizeof(unsigned long));
 if(!lds->transfer_table)
  return 0;

 return 1;
}

static void mosaic_free_fields(struct local_data_s *lds)
{
 lds->textscreen_size=0;
 if(lds->textscreen_source_address){
  free(lds->textscreen_source_address);
  lds->textscreen_source_address=NULL;
 }
 lds->tftable_size=0;
 if(lds->transfer_table){
  free(lds->transfer_table);
  lds->transfer_table=NULL;
 }
}

static void *vis_init(display_visual_data_s *vds)
{
 struct local_data_s *lds;

 lds=(struct local_data_s *)calloc(1,sizeof(struct local_data_s));
 if(!lds)
  return NULL;

 lds->vds=vds;

 mosaic_get_textscreen_resolution(lds);
 if(!lds->textscreen_resolution_x || !lds->textscreen_resolution_y)
  goto err_out_init;

 if(!mosaic_alloc_fields(lds))
  goto err_out_init;

 lds->blackhole=0x07000700;

 return (void *)lds;

err_out_init:
 vis_close(lds);
 return NULL;
}

static void vis_close(void *ldp)
{
 struct local_data_s *lds=ldp;
 if(lds){
  if(lds->textscreen_source_address)
   free(lds->textscreen_source_address);
  if(lds->transfer_table)
   free(lds->transfer_table);
  free(lds);
 }
}

static int vis_start(void *ldp)
{
 struct local_data_s *lds=ldp;
 display_visual_data_s *vds=lds->vds;

 mosaic_get_textscreen_resolution(lds);
 if(!lds->textscreen_resolution_x || !lds->textscreen_resolution_y)
  return 0;

 if(lds->textscreen_size<(lds->textscreen_resolution_x*lds->textscreen_resolution_y*TEXTSCREEN_BYTES_PER_CHAR)){
  mosaic_free_fields(lds);
  if(!mosaic_alloc_fields(lds))
   return 0;
 }

 lds->textscreen_target_address=vds->textscreen_linear_address;

 lds->blocknum_x=8;
 lds->blocknum_y=5;
 lds->blocksize_x=lds->textscreen_resolution_x/lds->blocknum_x;
 lds->blocksize_y=lds->textscreen_resolution_y/lds->blocknum_y;

 vds->textscreen_linear_address=(unsigned long)lds->textscreen_source_address;

 // copy original screen data into the new redirect-buffer
 memcpy((void *)lds->textscreen_source_address,(void *)lds->textscreen_target_address,lds->textscreen_size);

 mosaic_init_transfer_table(lds);

 return 1;
}

static void vis_stop(void *ldp)
{
 //text desktop is restored by Mpxplay
}

static void vis_draw(void *ldp)
{
 struct local_data_s *lds=ldp;
 mosaic_shift_block(lds);
 mosaic_apply_transfer_table(lds);
}

static void mosaic_init_transfer_table(struct local_data_s *lds)
{
 unsigned long addr=(unsigned long)lds->textscreen_source_address;
 unsigned long *tft=lds->transfer_table;
 unsigned int i=lds->tftable_size,j;

 lds->currblock_x=pds_rand(lds->blocknum_x-1);
 lds->currblock_y=pds_rand(lds->blocknum_y-1);

 do{
  *tft++=addr;
  addr+=TEXTSCREEN_BYTES_PER_CHAR;
 }while(--i);

 i=(lds->currblock_y*lds->blocksize_y*lds->textscreen_resolution_x)+(lds->currblock_x*lds->blocksize_x);
 tft=lds->transfer_table+i;
 for(i=0;i<lds->blocksize_y;i++)
  for(j=0;j<lds->blocksize_x;j++){
   tft[i*lds->textscreen_resolution_x+j]=(unsigned long)&lds->blackhole;
  }
}

static void mosaic_shift_block(struct local_data_s *lds)
{
 unsigned long *tft=lds->transfer_table;
 unsigned long mclen=lds->blocksize_x*sizeof(*tft);
 unsigned long x,y,basex,basey,basepos;

 if(lds->shift_direction==0){ // create new shift
  if(pds_rand(100)<50)
   lds->shift_vertical_or_horizontal=BLOCK_SHIFT_HORIZONTAL;
  else
   lds->shift_vertical_or_horizontal=BLOCK_SHIFT_VERTICAL;
  if(pds_rand(100)<50)
   lds->shift_direction=-1;
  else
   lds->shift_direction=+1;
  lds->shift_pos=0;
 }

 switch(lds->shift_vertical_or_horizontal){
  case BLOCK_SHIFT_VERTICAL:
   basex=lds->currblock_x*lds->blocksize_x;
   switch(lds->shift_direction){
    case BLOCK_SHIFT_UP:
     if(lds->currblock_y==0){
      lds->shift_direction=0;
     }else{
      basey=lds->currblock_y*lds->blocksize_y+lds->shift_pos;
      basepos=basey*lds->textscreen_resolution_x+basex;
      //mosaik_debugf("up  : cby:%d cbx:%d bsy:%2d bsx:%2d basey:%3d basex:%3d basepos:%4d sp:%2d  ",lds->currblock_y,lds->currblock_x,lds->blocksize_y,lds->blocksize_x,basey,basex,basepos,lds->shift_pos);
      for(y=0;y<lds->blocksize_y;y++){
       unsigned long dest=basepos;
       basepos-=lds->textscreen_resolution_x;
       memcpy((void *)(&tft[dest]),(void *)(&tft[basepos]),mclen);
      }
      for(x=0;x<lds->blocksize_x;x++)
       tft[basepos+x]=(unsigned long)&lds->blackhole;
     }
     break;
    case BLOCK_SHIFT_DOWN:
     if(lds->currblock_y>=(lds->blocknum_y-1)){
      lds->shift_direction=0;
     }else{
      basey=(lds->currblock_y+1)*lds->blocksize_y-lds->shift_pos-1;
      basepos=basey*lds->textscreen_resolution_x+basex;
      //mosaik_debugf("down: cby:%d cbx:%d bsy:%2d bsx:%2d basey:%3d basex:%3d basepos:%4d sp:%2d  ",lds->currblock_y,lds->currblock_x,lds->blocksize_y,lds->blocksize_x,basey,basex,basepos,lds->shift_pos);
      for(y=0;y<lds->blocksize_y;y++){
       unsigned long dest=basepos;
       basepos+=lds->textscreen_resolution_x;
       memcpy((void *)&tft[dest],(void *)&tft[basepos],mclen);
      }
      for(x=0;x<lds->blocksize_x;x++)
       tft[basepos+x]=(unsigned long)&lds->blackhole;
     }
     break;
   }
   lds->shift_pos++;
   if(lds->shift_pos>=lds->blocksize_y){
    lds->currblock_y+=lds->shift_direction;
    lds->shift_direction=0;
   }
   break;
  case BLOCK_SHIFT_HORIZONTAL:
   basey=lds->currblock_y*lds->blocksize_y;
   switch(lds->shift_direction){
    case BLOCK_SHIFT_LEFT:
     if(lds->currblock_x==0){
      lds->shift_direction=0;
     }else{
      basex=(lds->currblock_x-1)*lds->blocksize_x+lds->shift_pos;
      basepos=basey*lds->textscreen_resolution_x+basex;
      //mosaik_debugf("left: cby:%d cbx:%d bsy:%2d bsx:%2d basey:%3d basex:%3d basepos:%4d sp:%2d  ",lds->currblock_y,lds->currblock_x,lds->blocksize_y,lds->blocksize_x,basey,basex,basepos,lds->shift_pos);
      for(y=0;y<lds->blocksize_y;y++){
       memmove((void *)&tft[basepos+1],(void *)&tft[basepos],mclen);
       tft[basepos]=(unsigned long)&lds->blackhole;
       basepos+=lds->textscreen_resolution_x;
      }
     }
     break;
    case BLOCK_SHIFT_RIGHT:
     if(lds->currblock_x>=(lds->blocknum_x-1)){
      lds->shift_direction=0;
     }else{
      basex=(lds->currblock_x+1)*lds->blocksize_x-lds->shift_pos-1;
      basepos=basey*lds->textscreen_resolution_x+basex;
      //mosaik_debugf("righ: cby:%d cbx:%d bsy:%2d bsx:%2d basey:%3d basex:%3d basepos:%4d sp:%2d  ",lds->currblock_y,lds->currblock_x,lds->blocksize_y,lds->blocksize_x,basey,basex,basepos,lds->shift_pos);
      for(y=0;y<lds->blocksize_y;y++){
       memcpy((void *)&tft[basepos],(void *)&tft[basepos+1],mclen);
       tft[basepos+lds->blocksize_x]=(unsigned long)&lds->blackhole;
       basepos+=lds->textscreen_resolution_x;
      }
     }
     break;
   }
   lds->shift_pos++;
   if(lds->shift_pos>=lds->blocksize_x){
    lds->currblock_x+=lds->shift_direction;
    lds->shift_direction=0;
   }
   break;
 }
}

// copy (buffered) screen data into the real textscreen memory,
// based on the transfer table
static void mosaic_apply_transfer_table(struct local_data_s *lds)
{
 unsigned long *tft=lds->transfer_table;
 unsigned int i=lds->tftable_size;
 unsigned short *dest=(unsigned short *)lds->textscreen_target_address;
 do{
/*  unsigned long addr=*tft;
  if((addr==(unsigned long)&lds->blackhole) || (addr>=(unsigned long)lds->textscreen_source_address && addr<=(((unsigned long)lds->textscreen_source_address)+lds->textscreen_size)))
   *dest=*((unsigned short *)addr);
  dest++;
  tft++;*/
  *dest++=*((unsigned short *)(*tft++));
 }while(--i);
}

static void mosaic_get_textscreen_resolution(struct local_data_s *lds)
{
 unsigned char *biosmem=(char *)0;
 lds->textscreen_resolution_x=biosmem[0x44a];
 lds->textscreen_resolution_y=biosmem[0x484]+1;
}

static unsigned long pds_rand(int maxnum)
{
#ifdef __DOS__
 unsigned int i;
 char *biosmem;
#endif
 unsigned int r;

 if(maxnum<1)
  return 0;
#ifdef __DOS__
 r=0;
 biosmem=(char *)0;
 for(i=(biosmem[0x46c]&127);i;i--)
  r+=(biosmem[0x046c]);
 r+=rand();
#else
 r=rand()*rand()*rand();
#endif
 return (r%maxnum);
}

#ifdef MOSAIK_DEBUG

#include <stdarg.h>

static void mosaik_debugf(const char *format, ...)
{
 va_list ap;
 char sout[500];

 va_start(ap,format);
 vsprintf(sout, format, ap );
 va_end(ap);

 mrs->pds_textdisplay_textxy(15,0,49,sout);
}
#endif

static display_visual_func_s vis_blank_callbacks={
 &vis_init,      //
 &vis_close,     //
 &vis_start,     // called at start of visual-effect
 &vis_stop,      // called at stop of visual-effect
 &vis_draw,      // called FPS times per sec
 20 // FPS: 1-110 (if it's 0, then a default built-in value is used (currently 18))
};

static mpxplay_module_entry_s vis_blank_module_entry={
 MPXPLAY_DLLMODULETYPE_DISPLAY_VISUALPI,// module major-type
 0,                                     // module minor-type
 "MOSAI1",                              // module minor-name (rename it !!!)
 MPXPLAY_DLLMODULEVER_DISPLAY_VISUALPI, // current module (structure) version
 (void *)&vis_blank_callbacks           // callback functions (in a structure)
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &vis_blank_module_entry,
  NULL
 }
};

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin); // this DLL doesn't need it
 return (long)(&mpxplay_dll_entry_structure);
}
