/*
 * CMCOPY.C
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <io.h>
#include <dir.h>
#include <arch.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <progress.h>
#include <dzmain.h>

#define USE_CRC32

extern int flag_overwrite;

int copy_file(const char *, wfblk *);
int copy_subdir(const char *);
int confirm_delete(const char *, int __flag);

static int _docopy(void);
static int _doselected(FBLK *);
static int zip_docopy(void);
static int zip_doselected(FBLK *);

static int fblk_file(FBLK *);
static int fblk_directory(FBLK *);

void ercopyhndl(void)
{
	if (errno == ENOSPC)
		ermsg(NULL, "%s\n%s\n\n%s", cp_ercopy, __outfile, cp_enospc);
	else
		ermsg(NULL, "%s\n%s", cp_ercopy, __outfile);
}

int __copyfile(long size, int date, int time, char attrib)
{
	int sh;
	int oh;
	int result;

	errno = 0;
	if ((oh = getouthandle(__outfile)) == -1)
		return -1;
	else if (oh == 0) {
		cpy_jump = 1;
		return 0;
	}
	if ((sh = openfile(__srcfile, M_RDONLY, A_OPEN)) == -1) {
		close(oh);
		remove(__outfile);
		return -1;
	}
	result = copyhndl(sh, oh, size);
	close(sh);
	if (result == 0) {
		close(oh);
		ercopyhndl();
		remove(__outfile);
		return -1;
	}
	result = progress_update(size);
	if (_ifsmgr) {
		close(oh);
		wsetwrdate(__outfile, date, time);
	} else {
		setfdate(oh, date, time);
		close(oh);
	}
	if (ISRDONLY(attrib) && (cflag & _C_CDCLRDONLY)
		&& (*cpanel->flag & _W_CDROOM))
		attrib &= ~_A_RDONLY;
	setfattr(__outfile, attrib & _A_FATTRIB);
	return result;
}

int fblk_file(FBLK *b)
{
	strfcat(__srcfile, __srcpath, b->name);
	strfcat(__outfile, __outpath, b->name);
	if (progress_set(__srcfile, __outfile, b->size))
		return _ER_USERABORT;
	if (!test_filter(b->name, b->size, b->flag, b->date))
		return 0;
	return __copyfile(b->size, b->date, b->time, b->flag);
}

int fblk_directory(FBLK *b)
{
	char fb[256];

	if (progress_set(b->name, __outpath, 0))
		return _ER_USERABORT;
	mkdir(__outpath);
	strfcat(fb, __srcpath, b->name);
	return scansub(fb, cp_stdmask, 1);
}

int _doselected(FBLK *b)
{
	do {
		if (ISARCHIVE(b->flag)) {
			if (wsdecomp(cpanel->ide, b, __outpath))
				return 0;
		} else if (ISSUBDIR(b->flag)) {
			if (fblk_directory(b))
				return 0;
		} else if (fblk_file(b)) {
			return 0;
		}
		cpanel_deselect(b);
	} while ((b = panel_ffselected(cpanel)) != NULL);
	return 1;
}

int _docopy(void)
{
	FBLK *b;

	b = cpanel_findfirst();
	fp_fileblock = copy_file;
	fp_directory = copy_subdir;
	if (ISSELECTED(b->flag))
		return _doselected(b);
	if (ISSUBDIR(b->flag)) {
		if (ISARCHIVE(b->flag))
			return wsdecomp(cpanel->ide, b, __outpath);
		return fblk_directory(b);
	}
	if (progress_set(__srcfile, __outfile, b->size))
		return _ER_USERABORT;
	if (!test_filter(b->name, b->size, b->flag, b->date))
		return 0;
	if (ISARCHIVE(b->flag))
		return wsdecomp(cpanel->ide, b, __outpath);
	return __copyfile(b->size, b->date, b->time, b->flag);
}

int copy_file(const char *p, wfblk *w)
{
	if (progress_set(p, w->name, w->size_low))
		return _ER_USERABORT;
	if (!test_filter(w->name, w->size_low, w->attrib, w->date_modified))
		return 0;
	strfcat(__srcfile, p, w->name);
	strfcat(__outfile, __outpath, w->name);
	return __copyfile(w->size_low,
		w->date_modified, w->time_modified, w->attrib);
}

int copy_subdir(const char *p)
{
	const char *q;
	char b[WMAXPATH];
	int result;

	q = p + strlen(__srcpath);
	if (*q == '\\')
		q++;
	_ASSERTE(*q == 0);
	strcpy(b, __outpath);
	strfcat(__outpath, NULL, q);
	if (mkdir(__outpath) != -1) {
		if (setfattr(__outpath, 0) == 0)
			setfattr(__outpath, getfattr(p) & ~_A_SUBDIR);
	}
	result = scan_files(p);
	strcpy(__outpath, b);
	return result;
}

/*** .ZIP File Copy ***/

int zip_copy_file(const char *p, wfblk *w)
{
	if (!test_filter(w->name, w->size_low, w->attrib, w->date_modified))
		return 0;
	strfcat(__srcfile, p, w->name);
	return wzipadd(w->size_low,
		w->date_modified, w->time_modified, w->attrib);
}

int zip_copy_subdir(const char *p)
{
	const char *q;
	char b[WMAXPATH];
	int result;

	q = p + strlen(__srcpath);
	if (*q == '\\')
		q++;
	_ASSERTE(*q == 0);
	strcpy(b, __outpath);
	if (__outpath[0])
		strcat(__outpath, "/");
	strcat(__outpath, q);
	dostounix(__outpath);
	__srcfile[0] = 0;
	wzipadd(0L, dwdosdate(), twdostime(), getfattr(p));
	result = scan_files(p);
	strcpy(__outpath, b);

	return result;
}

int zip_fblk_file(FBLK *b)
{
	if (!test_filter(b->name, b->size, b->flag, b->date))
		return 0;
	strfcat(__srcfile, __srcpath, b->name);
	return wzipadd(b->size, b->date, b->time, b->flag);
}

int zip_fblk_directory(FBLK *b)
{
	char fb[256];

	strfcat(fb, __srcpath, b->name);
	return scansub(fb, cp_stdmask, 1);
}

int zip_doselected(FBLK *b)
{
	do {
		if (ISSUBDIR(b->flag)) {
			if (zip_fblk_directory(b))
				return 0;
		} else if (zip_fblk_file(b)) {
			return 0;
		}
		cpanel_deselect(b);
	} while ((b = panel_ffselected(cpanel)) != NULL);
	return 1;
}

int zip_docopy(void)
{
	FBLK *b;

	b = cpanel_findfirst();
	fp_fileblock = zip_copy_file;
	fp_directory = zip_copy_subdir;
	dostounix(__outpath);
	if (ISSELECTED(b->flag))
		return zip_doselected(b);
	if (ISSUBDIR(b->flag))
		return zip_fblk_directory(b);
	if (!test_filter(b->name, b->size, b->flag, b->date))
		return 0;
	progress_set(__srcfile, __outfile, b->size);
	return wzipadd(b->size, b->date, b->time, b->flag);
}

int cmcopy(void)
{
	FBLK *b;

	if (!(b = cpanel_findfirst()))
		return 0;
	if (cpy_initpath(b, 1) == 0)
		return 0;
	progress_open(cp_copy, cp_copy);
	if (cpy_flag & CPY_OUTARCH)
		return return_update_fcb(zip_docopy());
	return return_update_fcb(_docopy());
}
