/*
 * WZIPCOPY.C - Copy files and directories from .ZIP archive
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <io.h>
#include <errno.h>
#include <string.h>
#include <progress.h>
#include <tdialog.h>
#include <arch.h>
#include <alloc.h>

extern const char cp_WARNING[];

#define WSIZE	0x8000

int wzipexplode(void);
int wzipinflate(void);
int wzipdecrypt(void);

static int wzipfree(void)
{
	if (zip_slide != NULL) {
		free(zip_slide);
		zip_slide = NULL;
	}
	if (zip_inbuf != NULL) {
		free(zip_inbuf);
		zip_inbuf = NULL;
	}
	return _ER_MEM;
}

static int wzipinit(int sh, int oh)
{
	if ((zip_inbuf = (BYTE *)malloc(_MAXIOBUF)) == NULL)
		return _ER_MEM;
	if ((zip_slide = (BYTE *)malloc(WSIZE)) == NULL)
		return wzipfree();
	arc_enospc  = 0;
	arc_srchnd  = sh;
	arc_outhnd  = oh;
	zip_wrcount = 0L;
	zip_bindex  = _MAXIOBUF;
	zip_crcval  = 0xFFFFFFFFL;
	if (zip_attrib & _A_ENCRYPTED) {
		zip_bindex = 0;
		zip_bsize = osread(arc_srchnd, zip_inbuf, _MAXIOBUF);
		if (wzipdecrypt() == 0) {
			wzipfree();
			return _ER_BADERR;
		}
	}
	return 0;
}

int wunziphandle(int sh, int oh)
{
	int result = wzipinit(sh, oh);
	if (result == 0) {
		switch (zip_local.method) {
		case ZM_STORED:
			wzipfree();
			result = 0;
			if (copyhndl(sh, oh, zip_local.fsize) == 0) {
				result = _ER_BADERR;
				if (errno == ENOSPC)
					result = _ER_DISKFULL;
				else if (errno == ENOMEM)
					result = _ER_MEM;
			}
			break;
		case ZM_DEFLATED:
			result = wzipinflate();
			break;
		case ZM_IMPLODED:
			if ((zip_local.flag & 0x0006) == 0) {
				result = wzipexplode();
				break;
			}
		case ZM_SHRUNK:
		case ZM_REDUCED1:
		case ZM_REDUCED2:
		case ZM_REDUCED3:
		case ZM_REDUCED4:
		case ZM_TOKENIZED:
			ermsg(cp_WARNING, "%s\n'%02X'",
				sys_errlist[ENOSYS], zip_local.method);
		default:
			result = 3;
			break;
		}
	}
	wzipfree();
	if (arc_enospc)
		return _ER_DISKFULL;
	if (result == 0 && zip_local.method != ZM_STORED) {
		if ((zip_crcval = ((~zip_crcval) & 0xFFFFFFFFL)) != zip_local.crc)
			return _ER_CRCERR;
	}
	return result;
}

#define DC_MAXOBJ	5000

extern const char cp_emaxfb[];
extern const char cp_userabort[];

static int wzipfindentry(FBLK *, int);
static int wzipcopypath(WSUB *, FBLK *, const char *);
static int wzipcopyfile(WSUB *, FBLK *, const char *);

int wzipfindentry(FBLK *p, int h)
{
	DWORD *	wp;
	char *	cp;

	cp = (char *)p + strlen(p->name) + sizeof(FBLK);
	wp = (DWORD *)cp;
	lseek(h, wp[0], SEEK_SET);
	if (osread(h, &zip_local, sizeof(LZIP)) != sizeof(LZIP))
		return 0;
	if (zip_local.signature != ZID_LOCAL)
		return 0;
	lseek(h, zip_local.fnsize + zip_local.extsize, SEEK_CUR);
	if (zip_local.flag & 8) {
		zip_local.crc = wp[1];
		zip_local.csize = wp[2];
	}
	return 1;
}

int wzipcopyfile(WSUB *q, FBLK *fp, const char *out_path)
{
	int result;
	int sh,oh;
	char *p;
	char b[WMAXPATH];

	if ((sh = wsopenarch(q)) == -1)
		return stderror(q->file, EMERRNO, _ER_NOZIP);
	if (wzipfindentry(fp, sh) == 0) {
		close(sh);
		return stderror(fp->name, EMNOENT, _ER_FIND);
	}
	strfcat(b, out_path, fp->name);
	if ((oh = getouthandle(b)) == -1) {
		close(sh);
		return stderror(fp->name, EMEROPEN, -1);
	} else if (oh == 0) {
		close(sh);
		return 0;
	}
	zip_attrib = fp->flag;
	result = wunziphandle(sh, oh);
	close(sh);
	if (result) {
		close(oh);
		remove(b);
		if (result == _ER_USERABORT)
			return result;
		if (result == _ER_MEM)
			return stderror(b, EMNOMEM, result);
		if (result == _ER_DISKFULL)
			return stderror(b, EMDISKFULL, result);
		return stderror(b, EMARCHIVE, result);
	}
	if (_ifsmgr) {
		close(oh);
		wsetwrdate(b, fp->date, fp->time);
	} else {
		setfdate(oh, fp->date, fp->time);
		close(oh);
	}
	setfattr(b, fp->flag & _A_FATTRIB);
	return 0;
}

int wzipcopypath(WSUB *w, FBLK *fp, const char *out_path)
{
static	PATH s;
	WSUB ws;
	char *a,*p;
	int x;
	int result;

	ws.path = w->path;
	ws.file = w->file;
	ws.mask = w->mask;
	ws.arch = s.arch;
	ws.flag = &s.flag;
	ws.count = 0;
	ws.maxfb = DC_MAXOBJ;

	if (*w->arch)
		strfcat(ws.arch, w->arch, fp->name);
	else
		strcpy(ws.arch, fp->name);
	strfcat(s.file, out_path, fp->name);
	if (mkdir(s.file) != -1) {
		if (setfattr(s.file, 0) == 0)
			setfattr(s.file, fp->flag & ~_A_SUBDIR);
	}
	a = ws.arch + strlen(ws.arch);
	p = s.file + strlen(s.file);
	if ((result = progress_set(fp->name, out_path, 0)) != 0)
		return result;
	if (wsopen(&ws) == 0)
		return -1;
	if (wzipread(&ws) < 2) {
		wsclose(&ws);
		return 0;
	}
	if (ws.count == ws.maxfb)
		stdmsg("", cp_emaxfb, ws.maxfb, ws.maxfb);
	for (x = ws.count - 1; x >= 1 && result == 0; x--) {
		if (ISSUBDIR(ws.fcb[x]->flag)) {
			result = wzipcopypath(&ws, ws.fcb[x], s.file);
			*a = 0;
			*p = 0;
		} else {
			if ((result = progress_set(ws.fcb[x]->name, s.file,
				ws.fcb[x]->size)) != 0)
				break;
			result = wzipcopyfile(&ws, ws.fcb[x], s.file);
		}
		free(ws.fcb[x]);
		ws.fcb[x] = NULL;
	}
	wsclose(&ws);
	return result;
}

int wzipcopy(WSUB *w, FBLK *b, const char *dest)
{
	if (ISSUBDIR(b->flag))
		return wzipcopypath(w, b, dest);
	return wzipcopyfile(w, b, dest);
}
