#include "kernel.h"
#include "pci.h"
#include "pci_list.h"
//
l_ulong	AppVersion	= ULONG_ID(0,0,0,1);
l_char AppName[]	= "PCI Library";
l_uid nUID			= "PCI";
l_uid NeededLibs[]	= { "" };
//
// Private variable - Read through PciGetLastError()
//
l_byte pciLastError = 0;
//
// PciGetLastError()
//
// Input: Nothing
//
// Return: Last error ( AH ) from a BIOS call
//
l_byte PciGetLastError()
{
   return pciLastError;
}
//
// IsPciBiosPresent
//
// Input: Nothing
//
// Return: true if PCI Bios is present
// false if PCI Bios not present
//
l_bool IsPciBiosPresent()
{
   union REGS regs;
   //
   regs.x.ax = PCI_BIOS_INSTALL;
   regs.d.edi = 0;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   if( ( regs.h.ah==PCI_ERR_SUCCESS ) )
	  return true;
   //
   return false;
}
//
// GetPciBiosVerLo()
//
// Input: Nothing
//
// Return: PCI Bios Minor Version
//
l_byte GetPciBiosVerLo()
{
   union REGS regs;
   //
   regs.x.ax = PCI_BIOS_INSTALL;
   regs.d.edi = 0;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   if( ( regs.h.ah==PCI_ERR_SUCCESS ) )
	  return regs.h.bl;
   //
   return 0;
}
//
// GetPciBiosVerHi()
//
// Input: Nothing
//
// Return: PCI Bios Major Version
//
l_byte GetPciBiosVerHi()
{
   union REGS regs;
   //
   regs.x.ax = PCI_BIOS_INSTALL;
   regs.d.edi = 0;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   if( ( regs.h.ah==PCI_ERR_SUCCESS ) )
	  return regs.h.bh;
   //
   return 0;
}
//
// GetPciBussesInSystem()
//
// Input: Nothing
//
// Return: PCI Bus count
//
l_byte GetPciBussesInSystem()
{
   union REGS regs;
   //
   regs.x.ax = PCI_BIOS_INSTALL;
   regs.d.edi = 0;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   if( ( regs.h.ah==PCI_ERR_SUCCESS ) )
	  return regs.h.cl;
   //
   return 0;
}
//
// GetPciDeviceCount()
//
// Input: Nothing
//
// Return: total PCI device count
//
l_ushort GetPciDeviceCount()
{
   union REGS regs;
   //
   l_ushort numDev = 0;
   l_byte bus;
   l_byte dev;
   l_byte func;
   for( bus=0; bus<=GetPciBussesInSystem(); bus++ )
   {
	  for( dev=0; dev<=PCI_DEV_MAX; dev++ )
	  {
		 // Check if device present
		 if( 0xFFFF!=PciReadConfigWord( bus, dev, 0, PCI_CFG_R16_VENID ) )
		 {
			l_byte ucHeaderType = PciReadConfigByte( bus, dev, 0, PCI_CFG_R8_HDRTYPE );
			//
            ucHeaderType &= 0x80;
            //
            for( func=0; func<=PCI_FUNC_MAX; func++ )
            {
               if( 0xFFFF!=PciReadConfigWord( bus, dev, func, 0 ) )
               {
                  if( ucHeaderType==0 && func==0 )
                     numDev++;
                  else if( ucHeaderType==0x80 )
                     numDev++;
               }
            }
         }
      }
   }
   //
   return numDev;
}
//
// PciReadConfigByte()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
//
// Return: configuration register data
//
l_byte PciReadConfigByte( l_byte bus, l_byte dev, l_byte func, l_byte reg )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_RD8;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return regs.h.cl;
}
//
// PciReadConfigWord()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
//
// Return: configuration register data
//
l_ushort PciReadConfigWord( l_byte bus, l_byte dev, l_byte func, l_byte reg )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_RD16;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return regs.x.cx;
}
//
// PciReadConfigDWord()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
//
// Return: configuration register data
//
l_dword PciReadConfigDWord( l_byte bus, l_byte dev, l_byte func, l_byte reg )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_RD32;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return regs.d.ecx;
}
//
// PciWriteConfigByte()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
// data - Data to write to register
//
// Return: pciLastError
//
l_byte PciWriteConfigByte( l_byte bus, l_byte dev, l_byte func, l_byte reg, l_byte data )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_WR8;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.h.cl = data;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return pciLastError;
}
//
// PciWriteConfigWord()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
// data - Data to write to register
//
// Return: pciLastError
//
l_byte PciWriteConfigWord( l_byte bus, l_byte dev, l_byte func, l_byte reg, l_ushort data )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_WR16;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.x.cx = data;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return pciLastError;
}
//
// PciWriteConfigDWord()
//
// Input: bus - PCI bus number
// dev - PCI device number
// func - PCI function number
// reg - PCI configuration register
// data - Data to write to register
//
// Return: pciLastError
//
l_byte PciWriteConfigDWord( l_byte bus, l_byte dev, l_byte func, l_byte reg, l_dword data )
{
   union REGS regs;
   l_byte devfunc = 0;
   //
   // Mask the input values
   dev &= PCI_DEV_MAX;
   func &= PCI_FUNC_MAX;
   // Make the conbined dev/func parameter
   devfunc = dev;
   devfunc <<= 3;
   devfunc |= func;
   // Call the BIOS
   regs.x.ax = PCI_BIOS_CFG_WR32;
   regs.h.bh = bus;
   regs.h.bl = devfunc;
   regs.d.ecx = data;
   regs.x.di = reg;
   int386( PCI_INT, &regs, &regs );
   pciLastError = regs.h.ah;
   //
   return pciLastError;
}
//
// PciSearchForDevice()
//
// Input: venid - PCI vendor ID
// devid - PCI device ID
// marker - Set to 0 on first call. will be incremented for every device found!
//
// Return: true if there are more devices to search for
// false if no more devices are present
//
// bus - bus of device found
// dev - dev of device found
// func - function of device found
// marker - marker+1
//
l_bool PciSearchForDevice( l_ushort venid, l_ushort devid, l_byte* bus, l_byte* dev, l_byte* func, l_ushort* marker )
{
   union REGS regs;
   // Check parameters
   if( 0==bus || 0==dev || 0==func || 0==marker )
   {
	  return false;
   }
   // Call the BIOS
   regs.x.ax = PCI_BIOS_FINDDEV;
   regs.x.dx = venid;
   regs.x.cx = devid;
   regs.x.si = *marker;
   int386( PCI_INT, &regs, &regs );
   // Device found?
   if( regs.h.ah==PCI_ERR_SUCCESS )
   {
	  // fill in the return values
	  *bus = regs.h.bh;
	  *dev = regs.h.bl>>3;
	  *func = regs.h.bl&0x7;
	  // search for next device
	  *marker += 1;
	  return true;
   }
   //
   return false;
}
//
// DpmiMapPhysMemory()
//
// Input: start - address of physical memory
// length - length of memory map
//
// Return: Address of linear mapped memory or NULL if not mapped
//
l_dword DpmiMapPhysMemory( l_dword start, l_dword length )
{
   union REGS regs;
   //
   regs.w.ax = 0x0800;
   regs.w.bx = start>>16;
   regs.w.cx = start&0xffff;
   regs.w.si = length>>16;
   regs.w.di = length&0xffff;
   int386( 0x31, &regs, &regs );
   // if( regs.w.cflag&INTR_CF )
   // return 0;
   return ( ( regs.w.bx<<16 )&0xffff0000 )|regs.w.cx;
}
//
// DpmiUnMapPhysMemory()
//
// Input: mapmem - address returned by DpmiMapPhysMemory
//
// Return: Nothing
//
void DpmiUnMapPhysMemory( l_dword mapmem )
{
   union REGS regs;
   //
   regs.w.ax = 0x0801;
   regs.w.bx = mapmem>>16;
   regs.w.cx = mapmem&0xFFFF;
   int386( 0x31, &regs, &regs );
}
//
static l_text Cap( l_text st )
{
   l_int l = strlen( st );
   l_int i;
   //
   for( i = 0; i < l; i++ )
   {
	  if ( st[ i ] == ' ' )
		 st[ i ] = '_';
	  else if ( st[ i ] == '\\' )
		 st[ i ] = '-';
   }
   //
   return ( st );
}
//
void ShowVendorInfo( l_int pci_n, l_ushort vi )
{
   l_ushort i;
   l_byte k[ 0xFF ];
   //
   for( i = 0; i < PCI_VENTABLE_LEN; i++ )
   {
	  if ( PciVenTable[ i ].VenId == vi )
	  {
		 sprintf( k, "/HARDWARE/PCI/%02X/Vendor '%s'", pci_n, PciVenTable[ i ].VenFull );
		 CreateKey( Cap( k ) );
		 DebugMessage( "Vendor   : %s", PciVenTable[ i ].VenFull );
		 break;
	  }
   }
}
//
void ShowDevInfo( l_int pci_n, l_ushort vi, l_ushort di, l_byte func )
{
   l_ushort i;
   l_byte k[ 0xFF ];
   //
   for( i = 0; i < PCI_DEVTABLE_LEN; i++ )
   {
	  if ( PciDevTable[ i ].VenId == vi && PciDevTable[ i ].DevId == di )
	  {
		 sprintf( k, "/HARDWARE/PCI/%02X/Device '%04X:%04X Function %d'", pci_n, vi, di, func );
		 CreateKey( Cap( k ) );
		 DebugMessage( "Device   : %04X:%04X Function %d", vi, di, func );
		 sprintf( k, "/HARDWARE/PCI/%02X/Chip '%s'", pci_n, PciDevTable[ i ].Chip );
		 CreateKey( Cap( k ) );
		 DebugMessage( "Chip     : %s", PciDevTable[ i ].Chip );
		 sprintf( k, "/HARDWARE/PCI/%02X/Description '%s'", pci_n, PciDevTable[ i ].ChipDesc );
		 CreateKey( Cap( k ) );
		 DebugMessage( "Desc     : %s", PciDevTable[ i ].ChipDesc );
		 break;
	  }
   }
}
//
void ShowDevCommand( l_int pci_n, l_ushort co )
{
   l_int i = 0;
   l_byte buf[ 1024 ];
   l_byte k[ 1024 ];
   //
   sprintf( buf, "Command  : %04Xh ( ", co );
   if ( ( co & 1 ) == 1 )
   {
	  strcat( buf, "I\\O Access" );
	  i++;
   }
   if ( ( co & 2 ) == 2 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Memory Access" );
	  i++;
   }
   if ( ( co & 3 ) == 0 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Bus Access Disabled!!" );
	  i++;
   }
   if ( ( co & 4 ) == 4 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "BusMaster" );
	  i++;
   }
   if ( ( co & 8 ) == 8 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Special Cicles" );
	  i++;
   }
   if ( ( co & 16 ) == 16 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "MemWrite+Invalidate" );
	  i++;
   }
   if ( ( co & 32 ) == 32 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "VGA Palette Snoop" );
	  i++;
   }
   if ( ( co & 64 ) == 64 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Parity Error Response" );
	  i++;
   }
   if ( ( co & 128 ) == 128 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Wait Cycles" );
	  i++;
   }
   if ( ( ( co >> 8 ) & 1 ) == 1 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "System Error" );
	  i++;
   }
   if ( ( ( co >> 8 ) & 2 ) == 2 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Back-To-Back Transactions" );
	  i++;
   }
   if ( ( ( co >> 8 ) & 4 ) == 4 )
   {
	  if ( i )
		 strcat( buf, ", " );
	  strcat( buf, "Interrupt Disable" );
   }
   strcat( buf, " )" );
   DebugMessage( "%s\n", buf );
   //
   sprintf( k, "/HARDWARE/PCI/%02X/%s", pci_n, buf );
   CreateKey( Cap( k ) );
}
//
l_bool LibMain ( int argc, l_text *argv )
{
   l_ushort venid;
   l_ushort devid;
   l_ushort i, j;
   l_ushort marker = 0;
   l_byte k[ 0xFF ];
   l_int pci_n = 1;
   //
   for( i = 1; i < 0xFF; i++ )
   {
	  sprintf( k, "/HARDWARE/PCI/%02X", i );
	  DeleteKey( k );
   }
   //
   DebugMessage("*****************************************************************************");
   DebugMessage( "PCI Library" );
//   DebugMessage( "CopyRight (c) 2008 By EL CHANGO v4.\n" );
   // Check if PCI bios present
   if( !IsPciBiosPresent() )
   {
	  MsgInitErr();
	  DebugMessage( "ERROR: PCI System required!" );
	  return false;
   }
   // Print some information
   DebugMessage( "PCI BIOS V%X.%X present", GetPciBiosVerHi(), GetPciBiosVerLo() );
   // GetPciBussesInSystem() return the max. PCI bus in system
   // The root pci bus is 0. Adding 1 to include bus #0
   DebugMessage( "PCI Bios reports %d PCI busses in system", GetPciBussesInSystem()+1 );
   DebugMessage( "PCI Device count: %d\n", GetPciDeviceCount() );
   {
	  l_ushort VenId, DevId, Cmd;
	  l_byte bus;
	  l_byte dev;
	  l_byte func;
	  //
	  for( bus=0; bus<=GetPciBussesInSystem(); bus++ )
	  {
		 for( dev=0; dev<=PCI_DEV_MAX; dev++ )
		 {
			// Check if device present
			if( ( VenId=PciReadConfigWord( bus, dev, 0, PCI_CFG_R16_VENID ) )!=0xFFFF )
			{
			   l_byte ucHeaderType = PciReadConfigByte( bus, dev, 0, PCI_CFG_R8_HDRTYPE );
			   //
			   ucHeaderType &= 0x80;
			   //
			   for( func=0; func<=PCI_FUNC_MAX; func++ )
			   {
				  if( 0xFFFF!=PciReadConfigWord( bus, dev, func, 0 ) )
				  {
					 if( ucHeaderType==0 && func==0 )
					 {
						DevId=PciReadConfigWord( bus, dev, func, PCI_CFG_R16_DEVID );
						ShowVendorInfo( pci_n, VenId );
						ShowDevInfo( pci_n, VenId, DevId, func );
						Cmd=PciReadConfigWord( bus, dev, func, PCI_CFG_R16_CMD );
						ShowDevCommand( pci_n++, Cmd );
					 }
					 else if( ucHeaderType==0x80 )
					 {
						DevId=PciReadConfigWord( bus, dev, func, PCI_CFG_R16_DEVID );
						ShowVendorInfo( pci_n, VenId );
						ShowDevInfo( pci_n, VenId, DevId, func );
						Cmd=PciReadConfigWord( bus, dev, func, PCI_CFG_R16_CMD );
						ShowDevCommand( pci_n++, Cmd );
					 }
				  }
			   }
			}
		 }
	  }
   }
   //
   RegistrySave();
   //
   DebugMessage("*****************************************************************************");
   APPEXPORT(PciGetLastError);
   APPEXPORT(IsPciBiosPresent);
   APPEXPORT(GetPciBiosVerLo);
   APPEXPORT(GetPciBiosVerHi);
   APPEXPORT(GetPciBussesInSystem);
   APPEXPORT(GetPciDeviceCount);
   APPEXPORT(PciSearchForDevice);
   APPEXPORT(PciReadConfigByte);
   APPEXPORT(PciReadConfigWord);
   APPEXPORT(PciReadConfigDWord);
   APPEXPORT(PciWriteConfigByte);
   APPEXPORT(PciWriteConfigWord);
   APPEXPORT(PciWriteConfigDWord);
   //
   return true;
}
//
void Close (void)
{
}
