#include "kernel.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

/** @defgroup debug Debug and logging system.
	If your software or the system crashes during development use the set of debug functions to find the error.
	@{
*/
/**	@file
	Debug core file
*/

static FILE* DebugFile = NULL;

/**
	Writes a message into the debug file using the label "Message:"
*/
void DebugMessage ( char* szFormat, ... )
{
	va_list argptr;
	//
	va_start(argptr, szFormat);
	if (DebugFile)
	{
		fprintf(DebugFile, "Message: ");
		vfprintf(DebugFile, szFormat, argptr);
		fprintf(DebugFile,"\n");
		fflush(DebugFile);
	}
	//
	fsync( fileno( DebugFile ) );
	_flush_disk_cache();
	//
	va_end(argptr);
}

/**
	Writes a message into the debug file using the label "Warning:"
*/
void DebugWarning ( char* szFormat, ... )
{
	va_list argptr;
	//
	va_start(argptr, szFormat);
	if (DebugFile)
	{
		fprintf(DebugFile, "Warning: ");
		vfprintf(DebugFile, szFormat, argptr);
		fprintf(DebugFile,"\n");
		fflush(DebugFile);
	}
	va_end(argptr);
}

/**
	Writes a message into the debug file using the label "Error:"
*/
void DebugError ( char* szFormat, ... )
{
	va_list argptr;
	//
	va_start(argptr, szFormat);
	if (DebugFile)
	{
		fprintf(DebugFile, "ERROR: ");
		vfprintf(DebugFile, szFormat, argptr);
		fprintf(DebugFile,"\n");
		fflush(DebugFile);
	}
	//
	va_end(argptr);
}

/**
	Writes a message into the debug file using the label "Fatal:" and halts the kernel.
*/
void DebugFatal ( char* szFormat, ... )
{
	va_list argptr;
	//
	va_start(argptr, szFormat);
	if (DebugFile)
	{
		fprintf(DebugFile, "FATAL ERROR: ");
		vfprintf(DebugFile, szFormat, argptr);
		fprintf(DebugFile,"\n");
		fprintf(DebugFile,"\n\nA fatal error has occurred. Aborting program execution.\n");
		fflush(DebugFile);
	}
	//
	set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );
	//
	fprintf(stderr, "FATAL ERROR: ");
	vfprintf(stderr, szFormat, argptr);
	fprintf(stderr,"\n");
	fprintf(stderr,"\n\nA fatal error has occurred. Aborting program execution.\n");
	//
	va_end(argptr);
	exit(-1);
}

/*
	Initializes the debug system
*/
void GSSystemDebugInstall ( l_text szFilename )
{
	MsgInit("GSSystemDebugInstall");
	time_t t = time(NULL);
	//
	DebugFile = NULL;
	//
	if (!szFilename)
		DebugFile = NULL;
	else
	{
		DebugFile = fopen(szFilename, "wt");

		if ( !DebugFile )
			printf("DEBUG::DEBUG - Can't open file '%s' for writing\n", szFilename);
	}

	/*
	*	Print kernel information
	*/
	DebugMessage("Starting %s on %s\t %s\n\t KERNEL: %s\n\t DJGPP Release\n\n", WinDosTxt, ctime(&t), KernelCopyright, KernelName);
	MsgInitOk();
}


l_bool DebugCheckPtr ( l_text inf, void *ptr ) {
	if ( !ptr )
	{
		DebugWarning("NULL pointer at %s", inf);

		return false;
	}

	return true;
}

/*
	Uninstalls the debug system
*/
void GSSystemDebugUninstall ( void )
{
	MsgInit("GSSystemDebugUnInstall");
	DebugMessage("Logging stopped, closing log file");

	if (DebugFile)
	{
		fclose(DebugFile);
		DebugFile = NULL;
	}
	MsgInitOk();
}


