;
;   F R E E - D O S   X M S - D R I V E R
;
; Written by Till Gerken for the Free-DOS project.
;
; If you would like to use parts of this driver in one of your
;    projects, please check up with me first.
;
; I can be reached at:
;    Till.Gerken@ngosub0.ngo.ol.ni.schule.de (Internet)
;    2:2426/2190.16 (FidoNet)
;
; For questions concerning Free-DOS, mail the coordinator
; Morgan "Hannibal" Toal <hannibal@iastate.edu>
;
; Comments and bug reports are always appreciated.
;
; Copyright (c) 1995, Till Gerken

; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; Code ported to NASM by Louis P. Santillan, 2000.01.08-????.??.??
;    lsantil@calstatela.edu
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

;
; -- IMPLEMENTATION NOTES --
;
; - The Protected Mode handling is very simple, it fits only for the
;      least needs
; - I didn't care about reentrancy. If this thing should be used in
;      Multitasking Environments, well, somebody got to take a look
;      at the code.
; - INT15h, Func. 87h (Move Block) has been re-implemented to
;      preserve the state of A20. (very slow!)
; - INT15h, Func. 88h (Extended Memory Size) has been re-implemented
;      to return 0.
; - Function 0Bh (Move Block) uses it's own Protected Mode handling.
;      It doesn't provide Interrupt windows.
; - The code is not very optimised, I just wrote it down for now.
;      Later, when everything is tested, I'm going to see what can
;      be done
; - Some ideas were taken from the original XMS driver written by
;      Mark E. Huss (meh@bis.adp.com), but everything has been
;      completely rewritten, so if there are bugs, they are mine,
;      not his. ;)
;

%include "fdxms.inc"              ; this file contains structures,
                                  ; constants and the like

;
; 16-bit resident code and data
;

[ORG 0x0000]

;
; device driver header

   DD -1                          ; last driver in list
   DW 0x8000                      ; driver flags
   DW strategy                    ; pointer to strategy routine
   DW interrupt                   ; pointer to interrupt handler
   DB 'XMSXXXX0'                  ; device driver name

;
; global data

request_ptr DD 0

initialized DB 0                  ; contains one if driver has
                                  ; been initialised

xms_size DW 0                     ; size of XMS in kbytes

gdt32 DW gdt_size, dummy, 0
dummy DD 0, 0

code16dsc DB 0xFF, 0xFF, 0, 0, 0, 0x9A, 0, 0
core32dsc DB 0xFF, 0xFF, 0, 0, 0, 0x92, 0xCF, 0
gdt_size EQU $ - ( dummy )

code16idx EQU 0x8
core32idx EQU 0x10

old_int15 DD 0                    ; old INT15h vector
old_int2f DD 0                    ; old INT2fh vector

hma_used DB 0                     ; set if HMA is used
hma_min DW 0                      ; minimal space in HMA that
                                  ; has to be requested
a20_locks DW 0                    ; internal A20 lock count

xms_num_handles DW 32             ; number of available handles

;
; strategy routine. is called by DOS to initialize the driver once.
; only thing to be done here is to store the address of the device
; driver request block.
; In: ES:BX - address of request header
; Out: nothing

strategy:
   MOV [ WORD CS : request_ptr + 2 ], ES    ; store segment addr
   MOV [ WORD CS : request_ptr ], BX        ; store offset addr
   
   RETF                                     ; far return here!

;
; interrupt routine. called by DOS right after the strategy routine
; to process the incoming job. also used to initialize the driver.

interrupt:
   PUSH DI
   PUSH ES

   MOV DI, CS
   MOV DS, DI

   LES DI, [ request_ptr ]        ; load address of request header

   CMP BYTE [ ES : DI + request_hdr.cmd ], CMD_INIT   ; do we have to
                                                      ; initialize?
   JNE done
   CMP BYTE [ initialized ], 0    ; do we have initialized
                                  ; already?
   JNE done
   CALL initialize                ; no, do it now!
   
done:
   LDS SI, [ request_ptr ]        ; return this to DOS

   POP ES
   POP DI
   
   RETF                           ; far return here!

;
; just delays a bit

delay:
   JMP SHORT ( $ + 2 )
   JMP SHORT ( $ + 2 )
   RET

;
; empties the keyboard processor's command queue

empty_8042:
   CALL delay                     ; delay a bit
   IN AL, 0x64
   TEST AL, 1                     ; is there something to be read?
   JZ no_output                   ; no, go on
   CALL delay                     ; yes, first delay a bit
   IN AL, 0x60                    ; then read the output buffer
   JMP SHORT empty_8042           ; and try again
no_output:
   TEST AL, 2                     ; has it finished processing?
   JNZ empty_8042                 ; no, try again
   RET                            ; yes, done

;
; enables the A20 address line

enable_a20:
   CALL empty_8042                ; wait 'til he's done
   MOV AL, 0xD1                   ; write cmd
   OUT 0x64, AL
   CALL empty_8042
   MOV AL, 0xDF                   ; switch on A20
   OUT 0x60, AL
   CALL empty_8042
   RET

;
; disables the A20 address line

disable_a20:
   CALL empty_8042                ; wait 'til he's done
   MOV AL, 0xD1                   ; write cmd
   OUT 0x64, AL
   CALL empty_8042
   MOV AL, 0xDD                   ; switch off A20
   OUT 0x60, AL
   CALL empty_8042
   RET

;
; tests if the A20 address line is enabled.
; compares 256 bytes at 0:0 with ffffh:10h
; Out: ZF=0 - A20 enabled
; ZF=1 - A20 disabled

test_a20:
   PUSH CX
   PUSH SI
   PUSH DI
   PUSH DS
   PUSH ES
   XOR SI, SI
   MOV DS, SI
   MOV DI, 0xFFFF
   MOV ES, DI
   MOV DI, 0x10
   MOV CX, 0x100 / 4
   REP CMPSD

   POP ES
   POP DS
   POP DI
   POP SI
   POP CX
   RET

;
; checks if VDISK is already installed
; note: HMA check is skipped because of speed and some other (weird) reasons.
; In: nothing
; Out: ZF=0 -> VDISK is installed
; ZF=1 -> VDISK not installed

vdisk_id DB VDISK_IDSTR

check_vdisk:
   PUSH AX
   PUSH BX
   PUSH CX
   PUSH SI
   PUSH DI
   PUSH ES

; *** ORIGINAL COMMENT ***
;	call	enable_a20		; enable A20 for this

   XOR AX, AX                     ; get interrupt vector 19h
   MOV ES, AX
   LES BX, [ ES : 0x19 * 4 ]

   MOV DI, VDISK_IDOFS
   MOV SI, vdisk_id
   MOV CX, VDISK_IDLEN
   REP CMPSB                      ; is VDISK here?

; *** ORIGINAL COMMENTS
;	jnz	no_vdisk
;
;	MOV	AX,0ffffh
;	MOV	ES,AX
;	MOV	DI,VDISK_IDOFS+1
;	MOV	SI,offset vdisk_id
;	MOV	CX,VDISK_IDLEN
;	rep	cmpsb
;
;no_vdisk:
;
;	PUSHf				; save flags (thanks Yury!)
;	call	disable_a20
;	POPf

   POP ES
   POP DI
   POP SI
   POP CX
   POP BX
   POP AX
   RET

;
; Interrupt handlers
;

;
; new INT15h handler

a20state db 0                     ; keeps A20 state across INT15h call

int15_handler:
   CMP AH, 0x87                   ; is it a block move request?
   JE do_move
   CMP AH, 0x88                   ; is it a ext. mem size req.?
   JE ext_mem_size
   JMP [ CS : old_int15 ]         ; jump to old handler
do_move:
   CALL test_a20                  ; check if A20 is on or off
   JZ a20disabled
   MOV BYTE [ CS : a20state ], 1  ; preserve state
   JMP call_old_mover
a20disabled:
   MOV BYTE [ CS: a20state ], 0
call_old_mover:
   PUSHF                          ; simulate INT call
   CALL [ CS : old_int15 ]
   PUSHF                          ; save flags for return
   PUSH AX
   CMP BYTE [ CS : a20state ], 0  ; see if A20 has to be switched
   JZ disable_it
   CALL enable_a20
   JMP move_done
disable_it:
   CALL disable_a20
move_done:
   POP AX
   POPF
   IRET
ext_mem_size:
   XOR AX, AX                     ; no memory available
   CLC                            ; no error
   IRET

;
; new INT2Fh handler. Catches Func. 4300h+4310h

int2f_handler:
   PUSHF
   CMP AX, 0x4300                 ; is it "Installation Check"?
   JNE driver_address
   MOV AL, 0x80                   ; yes, we are installed ;)
   POPF
   IRET
driver_address:
   CMP AX, 0x4310                 ; is it "Get Driver Address"?
   JNE call_old2f
   MOV BX, CS
   MOV ES, BX
   MOV BX, xms_dispatcher
   POPF
   IRET
call_old2f:
   POPF
   JMP [ CS : old_int2f ]         ; jump to old handler

;
; XMS functions
;

;
; returns XMS version number
; In: AH=0
; Out: AX=XMS version number
; BX=internal revision number
; DX=1 if HMA exists, 0 if not

xms_get_version:
   POP BX
   MOV AX, INTERFACE_VER
   MOV BX, AX
   MOV DX, 1                      ; HMA is always available
   POPF
   RET

;
; requests HMA
; In: AH=1
; DX=space needed in HMA (0ffffh if application tries to request HMA)
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=80h -> function not implemented (implemented here ;) )
;    BL=81h -> VDISK is detected
;    BL=90h -> HMA does not exist
;    BL=91h -> HMA already in use
;    BL=92h -> DX less than HMA_MIN

xms_request_hma:
   POP BX
   CMP BYTE [ CS : hma_used ], 0  ; is HMA already used?
   MOV BL, XMS_HMA_IN_USE
   JNZ xrh_err
   CMP DX, [ CS : hma_min ]        ; is request big enough?
   MOV BL, XMS_HMAREQ_TOO_SMALL
   JB xrh_err
   MOV BYTE [ CS : hma_used ], 1   ; assign HMA to caller
   MOV AX, 1
   XOR BL, BL
   POPF
   RET
xrh_err:
   XOR AX, AX
   POPF
   RET

;
; releases HMA
; In: AH=2
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=80h -> function not implemented
;    BL=81h -> VDISK is detected
;    BL=90h -> HMA doesn't exist
;    BL=93h -> HMA wasn't allocated

xms_release_hma:
   POP BX
   CMP BYTE [ CS : hma_used ], 0  ; is HMA used?
   MOV BL, XMS_HMA_NOT_USED
   JZ xrh_err
   MOV BYTE [ CS : hma_used ], 0  ; now release it
   MOV AX,1
   XOR BL, BL
   POPF
   RET

;
; global A20 address line enable
; In: AH=3
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=80h -> function is not implemented
;    BL=81h -> VDISK is detected
;    BL=82h -> A20 failure

xms_global_enable_a20:
   POP BX
   CALL enable_a20                ; enable A20
   CALL test_a20                  ; is it really enabled?
   JZ xge_a20_err
   MOV AX, 1
   XOR BL, BL
   POPF
   RET
xge_a20_err:
   XOR AX, AX
   MOV BL, XMS_A20_FAILURE
   POPF
   RET

;
; global A20 address line disable
; In: AH=4
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=80h -> function is not implemented
;    BL=81h -> VDISK is detected
;    BL=82h -> A20 failure
;    BL=84h -> A20 still enabled

xms_global_disable_a20:
   POP BX
   CALL disable_a20               ; disable A20
   CALL test_a20                  ; is it really disabled?
   JNZ xge_a20_err
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; enables A20 locally
; In: AH=5
; Out: AX=1 if A20 is enabled, 0 otherwise
; BL=80h -> function not implemented
; BL=81h -> VDISK is detected
; BL=82h -> A20 failure

xms_local_enable_a20:
   POP BX
   INC WORD [ CS : a20_locks ]    ; increase lock counter
   CALL enable_a20                ; enable it
   CALL test_a20                  ; test if it's really enabled
   JZ xge_a20_err
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; disables A20 locally
; In: AH=6
; Out: AX=1 if A20 is disabled, 0 otherwise
; BL=80h -> function not implemented
; BL=81h -> VDISK is detected
; BL=82h -> A20 failure

xms_local_disable_a20:
   POP BX
   DEC WORD [ CS : a20_locks ]    ; decrease lock counter
   JNZ xld_dont_disable           ; disable only if needed
   CALL disable_a20               ; disable it
   CALL test_a20                  ; test if it's really disabled
   JNZ xge_a20_err
xld_dont_disable:
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; returns the state of A20
; In: AH=7
; Out: AX=1 if A20 is physically enabled, AX=0 if not
; BL=00h -> function was successful
; BL=80h -> function is not implemented
; BL=81h -> VDISK is detected

xms_query_a20:
   POP BX
   XOR AX, AX                     ; suppose A20 is disabled
   CALL test_a20
   JZ xqa_a20dis
   MOV AX, 1
xqa_a20dis:
   XOR BL, BL
   POPF
   RET

;
; searches a/next free XMS memory block
; In: DS=CS
; BX - offset of start handle (if search is continued)
; CX - remaining handles (if search is continued)
; Out: CY=1 - no free block
;    BX - offset of end of handle table
; CY=0 - free block found
;    BX - offset of free handle
;    CX - number of remaining handles

xms_find_free_block:
   MOV BX, driver_end             ; start at the beginning of the table
   MOV CX, [ xms_num_handles ]    ; check all handles
find_free_block:
   CMP BYTE [ BX + xms_handle.used ], 0          ; is it used?
   JNZ xms_find_next_free_block                  ; yes, go on
   CMP WORD [ BX + xms_handle.xbase ],0          ; assigned memory
                                                 ;    block or just
                                                 ;    blank?
   JNZ found_block                ; assigned, return it
xms_find_next_free_block:
   ADD BX, sizeOf_xms_handle      ; skip to next handle
   LOOP find_free_block           ; check next handle
   STC                            ; no free block found, error
   RET
found_block:
   CLC                            ; no error, return
   RET

;
; searches a/next free XMS memory handle
; In: DS=CS
; BX - offset of start handle (if search is continued)
; CX - remaining handles (if search is continued)
; Out: CY=1 - no free handle
;    BX - offset of end of handle table
; CY=0 - free handle found
;    BX - offset of free handle
;    CX - number of remaining handles

xms_find_free_handle:
   MOV BX, driver_end             ; start at the beginning of the table
   MOV CX, [ xms_num_handles ]    ; check all handles
find_free_handle:
   CMP BYTE [ BX + xms_handle.used ], 0          ; is it used?
   JNZ xms_find_next_free_handle                 ; yes, go on
   CMP WORD [ BX + xms_handle.xbase ], 0         ; really blank handle?
   JZ found_handle                               ; found a blank handle
xms_find_next_free_handle:
   ADD BX, sizeOf_xms_handle                     ; skip to next handle
   LOOP find_free_handle                         ; check next handle
   STC                                           ; no free block found,
                                                 ;    error
   RET
found_handle:
   CLC                            ; no error, return
   RET

;
; returns free XMS
; In: AH=8
; Out: AX=size of largest free XMS block in kbytes
; DX=total amount of free XMS in kbytes
; BL=0 if ok
; BL=080h -> function not implemented
; BL=081h -> VDISK is detected
; BL=0a0h -> all XMS is allocated

xms_query_free_xms:
   PUSH CX
   PUSH SI
   PUSH DS

   MOV AX, CS
   MOV DS, AX

   XOR AX, AX                     ; contains largest free block
   XOR DX, DX                     ; contains total free XMS

   CALL xms_find_free_block       ; search free block
   JC no_free_xms

check_next:
   MOV SI, [ BX + xms_handle.xsize ]              ; get size
   ADD DX, SI                                     ; update total amount
   CMP SI, AX                                     ; check if larger than
                                                  ;    largest
   JBE not_larger
   MOV AX, SI                     ; larger, update
not_larger:
   CALL xms_find_next_free_block
   JNC check_next

   POP DS
   POP SI
   POP CX
   POP BX
   XOR BL, BL
   POPF
   RET

no_free_xms:
   POP DS
   POP SI
   POP CX
   POP BX
   MOV BL, XMS_ALREADY_ALLOCATED
   POPF
   RET

;
; allocates an XMS block
; In: AH=9
; DX=amount of XMS being requested in kbytes
; Out: AX=1 if successful
;    DX=handle
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a0h -> all XMS is allocated
;    BL=0a1h -> no free handles left

xms_alloc_xms:
   PUSH CX
   PUSH SI
   PUSH DS

   MOV AX, CS
   MOV DS, AX

   CALL xms_find_free_block       ; see if there's a free block
   JNC check_size                 ; if it is, go on
no_free_mem:
   POP DS
   POP SI
   POP CX
   POP BX

   XOR AX, AX
   MOV BL, XMS_ALREADY_ALLOCATED
   POPF
   RET

get_next_block:
   CALL xms_find_next_free_block
   JC no_free_mem
check_size:
   CMP DX, [ BX + xms_handle.xsize ]            ; check if it's large enough
   JG get_next_block                            ; no, get next block

   MOV SI, BX                                   ; save handle address
   MOV WORD [ BX + xms_handle.used ], 1         ; this block is used from now on

   CALL xms_find_free_handle                    ; see if there's a blank handle
   JC perfect_fit                               ; no, there isn't, alloc all mem left
   MOV AX, [ SI + xms_handle.xsize ]            ; get size of old block
   SUB AX, DX                                   ; calculate resting memory
   JZ perfect_fit                               ; if it fits perfectly, go on
   MOV [ BX + xms_handle.xsize ], AX            ; store sizes of new blocks
   MOV [ SI + xms_handle.xsize ], DX
   MOV AX, [ SI + xms_handle.xbase ]            ; get base address of old block
   ADD AX, DX                                   ; calculate new base address
   MOV [ BX + xms_handle.xbase ], AX            ; store it in new handle
   MOV WORD [ BX + xms_handle.locks ], 0        ; no locks on this block

perfect_fit:
   MOV WORD [ SI + xms_handle.locks ], 0        ; no locks on this block

   MOV DX, SI                                   ; return handle in DX

   POP DS
   POP SI
   POP CX
   POP BX

   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; frees an XMS block
; In: AH=0ah
; DX=handle to allocated block that should be freed
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a2h -> handle is invalid
;    BL=0abh -> handle is locked

xms_free_xms:
   PUSH CX
   PUSH DX
   PUSH SI
   PUSH DS

   MOV AX, CS
   MOV DS, AX

   MOV SI, DX                                    ; get handle into SI

   CMP WORD [ SI + xms_handle.locks ], 0         ; is the block locked?
   JZ not_locked                                 ; no, go on
   POP DS
   POP SI
   POP DX
   POP CX
   POP BX
   XOR AX, AX
   MOV BL, XMS_BLOCK_LOCKED
   POPF
   RET

not_locked:
   CMP WORD [ SI + xms_handle.xsize ], 0         ; is it a zero-length
                                                 ;    handle?
   JNZ normal_handle
   MOV WORD [ SI + xms_handle.xbase ], 0         ; blank handle
   JMP xms_free_done

normal_handle:
   MOV AX, [ SI + xms_handle.xbase ]             ; get base address
   ADD AX, [ SI + xms_handle.xsize ]             ; calculate end-address

   CALL xms_find_free_block                      ; check free blocks
   JC xms_free_done                              ; no, was last handle
try_concat:
   CMP AX, [ BX + xms_handle.xbase ]             ; is it adjacent to
                                                 ;    old block?
   JNE not_adjacent
   MOV DX, [ BX + xms_handle.xsize ]             ; concat
   ADD AX, DX
   ADD [ SI + xms_handle.xsize ], DX
   MOV WORD [ BX + xms_handle.xbase ], 0         ; blank handle
   MOV WORD [ BX + xms_handle.xsize ], 0
not_adjacent:
   CALL xms_find_next_free_block                 ; see if there are
                                                 ;    other blks
   JNC try_concat
xms_free_done:
   MOV WORD [ SI + xms_handle.used ], 0          ; handle isn't used
                                                 ;    anymore
   POP DS
   POP SI
   POP DX
   POP CX
   POP BX
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; calculates the move address
; In: BX - handle (0 if EDX should be interpreted as seg:ofs value)
; EDX - offset
; Out: EBX - absolute move address
; Modifies: ECX, EDX

xms_get_move_addr:
   OR BX, BX                      ; translate address in EDX?
   JNZ dont_translate
   MOVZX ECX, DX                  ; save offset
   XOR DX, DX                     ; clear lower word
   SHR EDX, 12                    ; convert segment to absolute address
   ADD EDX, ECX                   ; add offset
   XOR EBX, EBX                   ; we're using conventional memory
   JMP no_handle
dont_translate:
   MOVZX EBX, WORD [ CS : BX + xms_handle.xbase ]     ; get block base
                                                      ;    address
   SHL EBX, 10                                        ; convert from kb
                                                      ;    to absolute
no_handle:
   ADD EBX, EBX                                       ; add offset into
                                                      ;    block
   RET

;
; moves an XMS block
; In: AH=0bh
; DS:SI=pointer to XMS move structure
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=082h -> A20 failure
;    BL=0a3h -> source handle is invalid
;    BL=0a4h -> source offset is invalid
;    BL=0a5h -> destination handle is invalid
;    BL=0a6h -> destination offset is invalid
;    BL=0a7h -> length is invalid
;    BL=0a8h -> move has invalid overlap
;    BL=0a9h -> parity error

xms_move_xms:
   POP BX
   PUSHAD
   PUSH DS
   PUSH ES
   CLI                            ; no interrupts

   MOV EAX, CR0
   TEST AL, 1                     ; are we already in PM?
   JNZ a20_failure                ; yes, simulate A20 failure

   MOV BX, [ SI + xms_move_strc.dest_handle ]
   MOV EBX, [ SI + xms_move_strc.dest_offset ]
   CALL xms_get_move_addr         ; get move address
   MOV EDI, EBX                   ; store in destination index

   MOV BX, [ SI + xms_move_strc.src_handle ]
   MOV EDX, [ SI + xms_move_strc.src_offset ]
   CALL xms_get_move_addr         ; get move address

   MOV ECX, [ SI + xms_move_strc.len ]           ; get length
   TEST CL, 1                                    ; is it even?
   JNZ invalid_length

   MOV ESI, EBX                   ; store in source index

   CALL test_a20                  ; get A20 state
   PUSHF                          ; save it for later
   CALL enable_a20                ; now enable it!

   LGDT [ CS : gdt32 ]            ; load GDTR
   MOV EAX, CR0
   OR AL, 1                       ; set PE bit
   MOV CR0, EAX                   ; shazamm!
   DB 0xEA                        ; JMP FAR
   DW to_pm, code16idx            ; flush IPQ and load CS sel.
to_pm:

   MOV AX, core32idx
   MOV DS, AX
   MOV ES, AX

   SHR ECX, 2                     ; get number of DWORDS to move
   JNC dword_boundary             ; is length a DWORD multiple?

; *************************************************
; * Is it me Marty, or are these MOVS* backwards? *
; * Also, NASM doesn't like the extended versions *
; *************************************************

   MOVSW ;[ WORD ESI ], [ WORD ESI ]              ; no, move first word
                                                 ;    to adjust
;	movs	[word esi],[word edi]		; no, move first word to adjust
dword_boundary:
   REP MOVSD ;[ DWORD ESI ], [ DWORD EDI ]        ; now move the main
                                                 ;    block
;	rep	movs [dword esi],[dword edi]	; now move the main block

   MOV EAX, CR0
   AND AL, 0xFE                                  ; clear PE bit
   MOV CR0, EAX                                  ; shazomm!

   DB 0xEA                                       ; JMP FAR
   DW to_rm
code_seg DW 0                                    ; flush IPQ and load
                                                 ;    CS sel.

to_rm:
   POPF                           ; get A20 state
   JNZ a20_was_enabled            ; if it was enabled, don't disable
   CALL disable_a20               ; it was disabled, so restore state
a20_was_enabled:

   POP ES
   POP DS
   POPAD                          ; restore everything
   POPF
   MOV AX, 1                      ; success
   RET

a20_failure:
   POP ES
   POP DS
   POPAD
   POPF
   XOR AX, AX
   MOV BL, XMS_A20_FAILURE
   RET
invalid_length:
   POP ES
   POP DS
   POPAD
   POPF
   XOR AX, AX
   MOV BL, XMS_INVALID_LENGTH
   RET

;
; locks an XMS block
; In: AH=0ch
; DX=XMS handle to be locked
; Out: AX=1 if block is locked
;    DX:BX=32-bit linear address of block
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a2h -> handle is invalid
;    BL=0ach -> lock count overflow
;    BL=0adh -> lock fails

xms_lock_xms:
   POP BX
   MOV BX, DX
   INC BYTE [ CS : BX + xms_handle.locks ]       ; increase lock counter
   JNC locked_successful                         ; go on if no overflow
   XOR AX, AX
   MOV BL, XMS_LOCK_COUNT_OVERFLOW               ; overflow, return with
                                                 ;    error
   POPF
   RET
locked_successful:
   PUSH EAX                                           ; save EAX
   MOVZX EAX, WORD [ CS : BX + xms_handle.xbase ]     ; get block base
                                                      ;    address
   SHL EAX, 10                                        ; calculate linear
                                                      ;    address
   MOV BX, AX                                         ; store LSW
   SHR EAX, 16
   MOV DX, AX                                         ; store MSW
   POP EAX                                            ; restore EAX
   MOV AX, 1
   POPF
   RET

;
; unlocks an XMS block
; In: AH=0dh
; DX=XMS handle to unlock
; Out: AX=1 if block is unlocked
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a2h -> handle is invalid
;    BL=0aah -> block is not locked

xms_unlock_xms:
   MOV BX,DX
   CMP BYTE [ CS : BX+xms_handle.locks], 0       ; check if block is
                                                 ;    locked
   JNZ is_locked                                 ; go on if true
   POP BX
   XOR AX, AX
   MOV BL, XMS_BLOCK_NOT_LOCKED
   POPF
   RET
is_locked:
   DEC BYTE [ CS : BX + xms_handle.locks ]       ; decrease lock counter
   POP BX
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; returns XMS handle information
; In: AH=0eh
; DX=XMS block handle
; Out: AX=1 if successful
;    BH=block's lock count
;    BL=number of free XMS handles
;    DX=block's length in kbytes
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a2h -> handle is invalid

xms_get_handle_info:
   POP BX
   PUSH CX
   PUSH SI
   PUSH DS

   MOV AX, CS
   MOV DS, AX

   MOV SI, DX

   XOR DL, DL                     ; reset free handle counter
   CALL xms_find_free_handle      ; chk if there's a blank handle
   JC nothing_free
find_next_free:
   INC DL                         ; increase handle counter
   CALL xms_find_next_free_handle ; and check if there's another
   JNC find_next_free

nothing_free:
   MOV BL, DL                                    ; store number of
                                                 ;    free handles
   MOV BH, [ SI + xms_handle.locks ]             ; store lock count
   MOV DX, [ SI + xms_handle.xsize ]             ; store block size

   POP DS
   POP SI
   POP CX
   MOV AX, 1
   POPF
   RET

;
; reallocates an XMS block. only supports shrinking.
; In: AH=0fh
; BX=new size for the XMS block in kbytes
; DX=unlocked XMS handle
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=081h -> VDISK is detected
;    BL=0a0h -> all XMS is allocated
;    BL=0a1h -> all handles are in use
;    BL=0a2h -> invalid handle
;    BL=0abh -> block is locked

xms_realloc_xms:
   POP BX
   PUSH BX
   PUSH DX
   PUSH SI
   PUSH DS
   
   MOV AX, CS
   MOV DS, AX

   XCHG BX, DX
   CMP DX, [ BX + xms_handle.xsize ]
   JBE shrink_it

no_xms_handles_left:
   POP DS
   POP SI
   POP DX
   POP BX
   XOR AX, AX
   MOV BL, XMS_NO_HANDLE_LEFT     ; simulate a "no handle" error
   POPF
   RET

shrink_it:
   MOV SI, BX
   CALL xms_find_free_handle                ; get blank handle
   JC no_xms_handles_left                   ; return if there's an error
   MOV AX, [ SI + xms_handle.xsize ]        ; get old size
   MOV [ SI + xms_handle.xsize ], DX
   SUB AX, DX                               ; calculate what's left over
   JZ dont_need_handle                      ; skip if we don't need it
   ADD DX, [ SI + xms_handle.xbase ]        ; calculate new base address
   MOV [ BX + xms_handle.xbase ], DX        ; store it
   MOV [ BX + xms_handle.xsize ], AX        ; store size
   MOV BYTE [ BX + xms_handle.locks ], 0    ; block is not locked...
   MOV BYTE [ BX + xms_handle.used ], 0     ; ...and not used
dont_need_handle:
   POP DS
   POP SI
   POP DX
   POP BX
   
   MOV AX, 1
   XOR BL, BL
   POPF
   RET

;
; requests an UMB block
; In: AH=10h
; DX=size of requested memory block in paragraphs
; Out: AX=1 if successful
;    BX=segment number of UMB
;    DX=actual size of the allocated block in paragraphs
; AX=0 if not successful
;    DX=size of largest available UMB in paragraphs
;    BL=080h -> function not implemented
;    BL=0b0h -> only a smaller UMB are available
;    BL=0b1h -> no UMBs are available

xms_request_umb:
   POP BX
   XOR AX, AX                     ; function fails...
   MOV BL, XMS_NO_UMB_AVAILABLE   ; ...because there are no UMBs we...
   XOR DX, DX                     ; ...are managing
   POPF
   RET

;
; releases an UMB block
; In: AH=11h
; DX=segment of UMB
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=0b2h -> UMB segment number is invalid

xms_release_umb:
   POP BX
   XOR AX, AX
   MOV BL, XMS_NOT_IMPLEMENTED
   POPF
   RET

;
; reallocates an UMB
; In: AH=12h
; BX=new size for UMB in paragraphs
; DX=segment of UMB to reallocate
; Out: AX=1 if successful
; AX=0 if not successful
;    BL=080h -> function not implemented
;    BL=0b0h -> no UMB large enough to satisfy request
;       DX=size of largest UMB in paragraphs
;    BL=0b2h -> UMB segment is invalid

xms_realloc_umb:
   POP BX
   XOR AX, AX
   MOV BL, XMS_NOT_IMPLEMENTED
   POPF
   RET

;
; XMS dispatcher
;

;
; XMS dispatcher
; In: AH - function number
; Out: AX=0 -> function not supported
; else see appr. routine

xms_table DW xms_get_version, xms_request_hma, xms_release_hma
          DW xms_global_enable_a20, xms_global_disable_a20
          DW xms_local_enable_a20, xms_local_disable_a20
          DW xms_query_a20, xms_query_free_xms, xms_alloc_xms
          DW xms_free_xms, xms_move_xms,xms_lock_xms, xms_unlock_xms
          DW xms_get_handle_info, xms_realloc_xms, xms_request_umb
          DW xms_release_umb, xms_realloc_umb

xms_dispatcher:
   JMP SHORT dispatcher_entry
   NOP                            ;
   NOP                            ; guarantee hookability
   NOP                            ;
dispatcher_entry:
   PUSHF                          ; save flags
   CLD
   CMP AH, 0x12                   ; is it a supported function?
   JA not_supported
   CALL check_vdisk               ; is VDISK installed?
   JZ vdisk_installed
hook_patch:
   JMP SHORT hook_ints
hook_return:
   PUSH BX
   MOVZX BX, AH
   SHL BX, 1
   JMP [ CS : xms_table + BX ]
not_supported:
   XOR AX, AX                     ; everything else fails
   MOV BL, XMS_NOT_IMPLEMENTED
   POPF                           ; and Yury strikes again...
   RET
vdisk_installed:
   XOR AX, AX
   MOV BL, XMS_VDISK_DETECTED
   POPF                           ; flags should be forbidden
   RET

hook_ints:
   OR AH, AH                      ; non-version call?
   JZ hook_return                 ; no, don't patch

   PUSH AX
   PUSH BX
   PUSH CX
   PUSH DX
   PUSH DS
   PUSH ES
   ; save registers
   
   MOV AX, CS
   MOV DS, AX
   
   XOR AX, AX                     ; get INT15h vector
   MOV ES, AX
   LES BX, [ ES : 0x15 * 4 ]
   MOV [ WORD old_int15 + 2 ], ES
   MOV [ WORD old_int15 ], BX

   MOV AX, 0x2515                 ; install own INT15h
   MOV DX, int15_handler
   INT 0x21

   MOV CX, [ xms_num_handles ]    ; get number of handles
   MOV BX, driver_end             ; get start of handle table
clear_table:
   MOV WORD [ BX + xms_handle.xbase ], 0         ; blank handle
   MOV WORD [ BX + xms_handle.xsize ], 0         ; blk doesn't occupy
                                                 ;    any space
   MOV WORD [ BX + xms_handle.used ], 0          ; handle not used
   ADD BX, sizeOf_xms_handle
   LOOP clear_table

   MOV BX, driver_end
   MOV WORD [ BX + xms_handle.xbase ], XMS_START ; init first block and give
   MOV AX, [ xms_size ]                          ; it all available memory
   MOV [ BX + xms_handle.xsize ], AX

   POP ES
   POP DS
   POP DX
   POP CX
   POP BX
   POP AX
   ; restore registers

   MOV WORD [ WORD CS : hook_patch ], 0x9090     ; insert two NOPs
   JMP SHORT hook_return                         ; and finish it

;
; mark for the driver end. above has to be the resident part, below the
; transient.

driver_end:

;
; 16-bit transient code and data. only used once.
;

;
; checks if CPU is a 386
; In: nothing
; Out: CY=0 - processor is a 386 or higher
; CY=1 - processor lower than 386

check_cpu:
   PUSHF
   XOR AX, AX
   PUSH AX
   POPF
   PUSHF
   POP AX
   AND AH, 0x0F
   CMP AH, 0x0F
   JE not386
   MOV AH, 7
   PUSH AX
   POPF
   PUSHF
   POP AX
   AND AH, 7
   JE not386
   POPF
   CLC
   RET
not386:
   POPF
   STC
   RET

;
; checks if A20 can be enabled and disabled
; Out: CF=0 - A20 switching works
; CF=1 - A20 failure

check_a20:
   CALL enable_a20
   CALL test_a20                  ; TEST_A20 should return ZF=0
   JZ a20failed
   CALL disable_a20
   CALL test_a20                  ; TEST_A20 should return ZF=1
   JNZ a20failed
   CLC
   RET
a20failed:
   STC
   RET

;
; initializes the driver. called only once!
; may modify DI
; In: ES:DI - pointer to init structure
; DS - initialized with code segment

init_message DB 13, 10
             TIMES 80 DB '-'
             DB 'Free-DOS XMS-Driver', 13, 10
             DB 'Preliminary version. Please don', 0x27, 't '
             DB 'spread.', 13, 10
             DB 'Copyright (c) 1995, Till Gerken', 13, 10, 13, 10
             DB 'Version ported to NASM by Louis P. Santillan 2000.01.08'
             DB 13, 10, 13, 10
             DB 'Driver Version: ', DRIVER_VERSION, 9
             DB 'Interface Version: ', INTERFACE_VERSION, 13, 10
             DB 'Information: ', INFO_STR, 13, 10, '$'

old_dos DB 'XMS needs at least DOS version 3.00.$'
xms_twice DB 'XMS is already installed.$'
vdisk_detected DB 'VDISK has been detected.$'
no_386 DB 'At least a 80386 is required.$'
a20_error DB 'Unable to switch A20 address line.$'
xms_sizeerr DB 'Unable to determine size of extended memory.$'
xms_toosmall DB 'Extended memory is too small or not available.$'

error_msg DB ' Driver won', 0x27, 't be installed.', 7, 13, 10, '$'

init_finished:
   TIMES 80 DB '='
   DB 13, 10,'$'

initialize:
   PUSHF
   PUSH AX
   PUSH BX
   PUSH CX
   PUSH DX
   PUSH SI

   CLD

   MOV AH, 9                      ; first, welcome the user!
   MOV DX, init_message
   INT 0x21

   MOV AX, 0x3000                 ; get DOS version number
   INT 0x21
   XCHG AH, AL                    ; convert to BSD
   CMP AX, 0x300                  ; we need at least 3.00
   MOV DX, old_dos
   JB NEAR error_exit

   MOV AX, 0x4300                 ; check if XMS is already
   INT 0x2F                       ; installed
   CMP AL, 0x80
   MOV DX, xms_twice
   JE NEAR error_exit

   CALL check_cpu                 ; do we have at least a 386?
   MOV DX, no_386
   JC NEAR error_exit

   CALL check_a20                 ; check A20
   MOV DX, a20_error
   JC NEAR error_exit

   CALL check_vdisk               ; is VDISK installed?
   MOV DX, vdisk_detected
   JZ NEAR error_exit

   CLC
   MOV AH, 0x88                   ; extended memory size
   INT 0x15
   MOV DX, xms_sizeerr
   JC NEAR error_exit
   MOV DX, xms_toosmall
   SUB AX, 64                     ; save HIMEM area
   JC NEAR error_exit             ; if there aren't 64k,
                                  ; there's nothing to do

   MOV [ xms_size ], AX           ; save size

   PUSH EAX                       ; save EAX

   MOV AX, CS                     ; setup descriptors
   MOV [ code_seg ], AX           ; eliminate relocation entry
   MOVZX EAX, AX
   SHL EAX, 4
   OR [ DWORD code16dsc + 2 ], eax
   ADD [ DWORD gdt32 + 2 ], eax

   POP EAX                        ; restore EAX

   PUSH ES
   XOR AX, AX                     ; get INT2Fh vector
   MOV ES, AX
   LES BX, [ ES : 0x2F * 4 ]
   MOV [ WORD old_int2f + 2 ], ES
   MOV [ WORD old_int2f ], BX

   MOV AX, 0x252F                 ; install own INT2Fh
   MOV DX, int2f_handler
   INT 0x21
   POP ES

   MOV [ WORD ES : DI + 2 + init_strc.end_addr], CS   ; set end address
   XOR DX, DX
   MOV AX, sizeOf_xms_handle
   MUL WORD [ xms_num_handles ]
   ADD AX, driver_end
   MOV [ WORD ES : DI + init_strc.end_addr ], AX
   MOV WORD [ ES : DI + request_hdr.status ], STATUS_OK    ; we're alright

   JMP SHORT exit

error_exit:
   MOV [ WORD ES : DI + 2 + init_strc.end_addr ], CS       ; set end address
   MOV DWORD [ WORD ES : DI + init_strc.end_addr ], 0      ; now, we don't
                                                           ;   use any space
   MOV DWORD [ ES : DI + request_hdr.status ], STATUS_BAD  ; waaaah!
   MOV AH, 9                                               ; print msg
   INT 0x21
   MOV DX, error_msg
   INT 0x21

exit:
   MOV AH, 9
   MOV DX, init_finished
   INT 0x21
   POP SI
   POP DX
   POP CX
   POP BX
   POP AX
   POPF
   RET

;--------------------------------------------------------------------
