/*  Move

    Copyright (c) Joe Cosentino 1997-2002.
    All Rights Reserved.

    Some code written by Rene Ableidinger.

*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define DIR_SEPARATOR "\\"
#define R_OK 4                          // Readable.

// G L O B A L S ////////////////////////////////////////////////////////////

char opt_verify=0,opt_help=0,opt_prompt=0,opt_subdir=0,old_verify=0,switch_intodir=0,dest_drive;
int found=0;

// P R O T O T Y P E S //////////////////////////////////////////////////////

void classify_args(char, char *[], char *, char *[], char *, char *[]);
void error(const char *);
void exit_fn(void);
char *addsep(char *);
int dir_exists(const char *);
int makedir(char *);
void build_filename(char *, const char *, const char *);
void build_name(char *, const char *, const char *);
void move_files(const char *, const char *, const char *, const char *);
void prepare_move(const char *, const char *);
void do_copy(const char *,const char *);

// F U N C T I O N S ////////////////////////////////////////////////////////

void classify_args(char argc, char *argv[], char *fileargc, char *fileargv[], char *optargc, char *optargv[])
{
    char i, *argptr;

    *fileargc=0;
    *optargc=0;
    for (i=1;i<argc;i++)
        {
        argptr=argv[i];
        if (argptr[0] == '/')
            {
            optargv[*optargc]=argptr+1;
            *optargc=*optargc+1;
            } // end if.
        else
            {
            fileargv[*fileargc]=argptr;
            *fileargc=*fileargc+1;
           } // end else.

        } // end for.

} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

void error(const char *error)
{
    printf(" [%s]\n", error);

} // end error.

/////////////////////////////////////////////////////////////////////////////

void exit_fn(void)
{
    if (opt_verify)
        {
        setverify(old_verify);          // Restore value of the verify flag.
        } // end if.

} // end exit_fn.

/////////////////////////////////////////////////////////////////////////////

char *strmcpy(char *dest, const char *src, const unsigned long int maxlen)
{
    unsigned long int i,tmp_maxlen;

    tmp_maxlen=maxlen-1;
    i=0;
    while (src[i] != '\0' && i<tmp_maxlen)
        {
        dest[i]=src[i];
        i=i+1;
        } // end while.

    dest[i]='\0';
    return dest;

} // end strmcpy.

/////////////////////////////////////////////////////////////////////////////

char *strmcat(char *dest, const char *src, const unsigned long int maxlen)
{
    unsigned long int i,tmp_maxlen;
    char *src_ptr;

    tmp_maxlen=maxlen-1;
    src_ptr=(char *)src;
    i=strlen(dest);
    while (*src_ptr != '\0' && i<tmp_maxlen)
        {
        dest[i]=*src_ptr;
        src_ptr=src_ptr+1;
        i=i+1;
        } // end while.

    dest[i]='\0';
    return dest;

} // end strmcat.

/////////////////////////////////////////////////////////////////////////////

char *addsep(char *path)
{
    int path_length;

    path_length=strlen(path);
    if (path[path_length-1] != *DIR_SEPARATOR)
        {
        path[path_length]=*DIR_SEPARATOR;
        path[path_length+1]='\0';
        } // end if.

    return path;

} // end addsep.

/////////////////////////////////////////////////////////////////////////////

int dir_exists(const char *path)
{
    char tmp_path[MAXPATH],i;
    int attrib;

    strmcpy(tmp_path, path, sizeof(tmp_path));
    i=strlen(tmp_path);
    if (i<3)
        {
        strmcat(tmp_path, DIR_SEPARATOR, sizeof(tmp_path));
        } // end if.
    else if (i>3)
        {
        i=i-1;
        if (tmp_path[i] == *DIR_SEPARATOR)
            {
            tmp_path[i]='\0';
            } // end if.

        } // end else.

    attrib=_chmod(tmp_path, 0);
    if (attrib == -1 || (attrib & FA_DIREC) == 0)
        {
        return 0;
        } // end if.

    return -1;

} // end dir_exists.

/////////////////////////////////////////////////////////////////////////////

int makedir(char *path)
{
    char tmp_path1[MAXPATH],tmp_path2[MAXPATH],i,length,mkdir_error;

    if (path[0] == '\0')
        {
        return -1;
        } // end if.

    strmcpy(tmp_path1, path, sizeof(tmp_path1));
    addsep(tmp_path1);
    length=strlen(tmp_path1);
    strncpy(tmp_path2, tmp_path1, 3);
    i=3;
    while (i<length)
        {
        if (tmp_path1[i]=='\\')
            {
            tmp_path2[i]='\0';
            if (!dir_exists(tmp_path2))
                {
                mkdir_error=mkdir(tmp_path2);
                if (mkdir_error)
                    {
                    path[i]='\0';
                    return mkdir_error;
                    } // end if.

                } // end if.

            } // end if.

        tmp_path2[i]=tmp_path1[i];
        i++;
        } // end while.

    return 0;

} // end makedir.

/////////////////////////////////////////////////////////////////////////////

void build_filename(char *dest_filename,const char *src_filename,const char *filepattern)
{
    char drive[MAXDRIVE],dir[MAXDIR],filename_file[MAXFILE],filename_ext[MAXEXT],filepattern_file[MAXFILE],filepattern_ext[MAXEXT],tmp_filename[MAXFILE],tmp_ext[MAXEXT];

    _splitpath(src_filename, drive, dir, filename_file, filename_ext);
    _splitpath(filepattern, drive, dir, filepattern_file, filepattern_ext);
    build_name(tmp_filename, filename_file, filepattern_file);
    build_name(tmp_ext, filename_ext, filepattern_ext);
    strmcpy(dest_filename, drive, MAXPATH);
    strmcat(dest_filename, dir, MAXPATH);
    strmcat(dest_filename, tmp_filename, MAXPATH);
    strmcat(dest_filename, tmp_ext, MAXPATH);

} // end build_filename.

/////////////////////////////////////////////////////////////////////////////

void build_name(char *dest, const char *src, const char *pattern)
{
    int i,pattern_i,src_length,pattern_length;

    src_length=strlen(src);
    pattern_length=strlen(pattern);
    i=0;
    pattern_i=0;
    while ((i<src_length || (pattern[pattern_i] != '\0' && pattern[pattern_i] != '?' && pattern[pattern_i] != '*')) && pattern_i<pattern_length)
        {
        switch (pattern[pattern_i])
            {
            case '*':
                dest[i]=src[i];
                break;
            case '?':
                dest[i]=src[i];
                pattern_i++;
                break;
            default:
                dest[i]=pattern[pattern_i];
                pattern_i++;
                break;

            } // end switch.

        i++;
        } // end while.

    dest[i]='\0';

} // end build_name.

/////////////////////////////////////////////////////////////////////////////

void move_files(const char *src_pathname, const char *src_filename, const char *dest_pathname, const char *dest_filename)
{
    char filepattern[MAXPATH],new_src_pathname[MAXPATH],new_dest_pathname[MAXPATH],src_path_filename[MAXPATH],dest_path_filename[MAXPATH],tmp_filename[MAXFILE+MAXEXT],tmp_pathname[MAXPATH];
    struct ffblk fileblock;
    int fileattrib,done;

    if (opt_subdir)
        {
        // Copy files in subdirectories too.
        strmcpy(filepattern, src_pathname, sizeof(filepattern));
        strmcat(filepattern, "*.*", sizeof(filepattern));
        done=findfirst(filepattern, &fileblock, FA_DIREC);
        while (!done)
            {
            if (fileblock.ff_attrib == FA_DIREC && strcmp(fileblock.ff_name, ".") != 0 && strcmp(fileblock.ff_name, "..") != 0)
                {
                // Build source pathname.
                strmcpy(new_src_pathname, src_pathname, sizeof(new_src_pathname));
                strmcat(new_src_pathname, fileblock.ff_name, sizeof(new_src_pathname));
                strmcat(new_src_pathname, DIR_SEPARATOR, sizeof(new_src_pathname));

                // Build destination pathname.
                strmcpy(new_dest_pathname, dest_pathname, sizeof(new_dest_pathname));
                strmcat(new_dest_pathname, fileblock.ff_name, sizeof(new_dest_pathname));
                strmcat(new_dest_pathname, DIR_SEPARATOR, sizeof(new_dest_pathname));
                move_files(new_src_pathname, src_filename,new_dest_pathname, dest_filename);
                } // end if.

            done=findnext(&fileblock);
            } // end while.

        } // end if.

    fileattrib=FA_RDONLY+FA_ARCH;

    // Find first source file.
    strmcpy(filepattern, src_pathname, sizeof(filepattern));
    strmcat(filepattern, src_filename, sizeof(filepattern));
    done=findfirst(filepattern, &fileblock, fileattrib);
    if (!done)
        {
        found=-1;
        } // end if.

    // Check if destination directory has to be created.
    if ((!done) && !dir_exists(dest_pathname))
        {
        strmcpy(tmp_pathname, dest_pathname, sizeof(tmp_pathname));
        if (makedir(tmp_pathname) != 0)
            {
            error("Unable to create directory");
            exit(4);
            } // end if.

        } // end if.

    // Copy files.
    while (!done)
        {
        // Check if copied files should have archive attribute set.
        if (fileblock.ff_attrib & FA_ARCH)
            {
            // Build source filename including path.
            strmcpy(src_path_filename, src_pathname, sizeof(src_path_filename));
            strmcat(src_path_filename, fileblock.ff_name, sizeof(src_path_filename));

            // Build destination filename including path.
            strmcpy(dest_path_filename, dest_pathname, sizeof(dest_path_filename));
            build_filename(tmp_filename, fileblock.ff_name, dest_filename);
            strmcat(dest_path_filename, tmp_filename, sizeof(dest_path_filename));
            prepare_move(src_path_filename, dest_path_filename);
            } // end if.

        done=findnext(&fileblock);
        } // end while.

} // end move_files.

/////////////////////////////////////////////////////////////////////////////

void prepare_move(const char *src_filename, const char *dest_filename)
{
    int dest_file_exists,fileattrib;
    struct stat src_statbuf,dest_statbuf;
    struct dfree disktable;
    unsigned long free_diskspace;
    char ch;

    fileattrib=fileattrib;dest_file_exists=dest_file_exists;  // Too keep warnings from appearing while compiling.
    if (opt_prompt)
        {
        // Ask for confirmation to create file.
        printf("Overwrite file [Y/N]? ");
        scanf("%c", &ch);
        fflush(stdin);
        ch=toupper(ch);
        if (ch == 'N')                  // No-skip file.
            {
            return;
            } // end if.

        } // end if.

    // Check if source and destination file are equal.
    if (stricmp(src_filename, dest_filename) == 0)
        {
        error("File cannot be copied onto itself");
        exit(4);
        } // end if.

    // Check source file for read permission.
    if (access(src_filename, R_OK) != 0)
        {
        error("Access denied");
        exit(5);
        } // end if.

    // Get info of source and destination file.
    stat((char *)src_filename, &src_statbuf);
    dest_file_exists=!stat((char *)dest_filename, &dest_statbuf);

    // Get amount of free disk space in destination drive.
    getdfree(dest_drive, &disktable);
    free_diskspace=(unsigned long) disktable.df_avail * disktable.df_sclus * disktable.df_bsec;

    switch (opt_prompt)
        {
        case 1:                         // Skip file.
            return;
        case 2:                         // Ask for confirmation to overwrite.
            printf("Overwrite file [Y/N]? ");
            scanf("%c", &ch);
            fflush(stdin);
            ch=toupper(ch);
            switch (ch)
                {
                case 'N':               // Don't overwrite.
                    return;
                case 'O':               // Overwrite all.
                    opt_prompt=0;
                    break;
                case 'S':               // Skip all.
                    opt_prompt=1;
                    return;

                } // end switch.

            break;

        } // end switch.

    // Check free space on destination disk.
    if (src_statbuf.st_size>free_diskspace-dest_statbuf.st_size)
        {
        error("Insufficient disk space in destination path");
        exit(39);
        } // end if.

    // Get file attribute from destination file.
    fileattrib=_chmod(dest_filename, 0);

    // Check free space on destination disk.
    if (src_statbuf.st_size>free_diskspace)
        {
        error("Insufficient disk space");
        exit(39);
        } // end if.

    do_copy(src_filename, dest_filename);

} // end move_file.

/////////////////////////////////////////////////////////////////////////////

void do_copy(const char *src_filename,const char *dest_filename)
{
    FILE *src_file,*dest_file;
    char buffer[16384];
    unsigned int buffersize;
    int readsize,fileattrib;
    struct ftime filetime;

    printf("%s => %s", src_filename, dest_filename);

    // Open source file.
    src_file=fopen(src_filename, "rb");
    if (src_file == NULL)
        {
        error("Cannot open source file");
        exit(30);
        } // end if.

    // Open destination file.
    dest_file=fopen(dest_filename, "wb");
    if (dest_file == NULL)
        {
        error("Cannot create destination file");
        fclose(src_file);
        exit(29);
        } // end if.

    // Copy file data.
    buffersize=sizeof(buffer);
    readsize=fread(buffer, sizeof(char), buffersize, src_file);
    while (readsize>0)
        {
        if (fwrite(buffer, sizeof(char), readsize, dest_file) != readsize)
            {
            error("Write error on destination file");
            fclose(src_file);
            fclose(dest_file);
            exit(29);
            } // end if.

        readsize=fread(buffer, sizeof(char), buffersize, src_file);
        } // end while.

    // Copy file timestamp.
    getftime(fileno(src_file), &filetime);
    setftime(fileno(dest_file), &filetime);

    // Close files.
    fclose(src_file);
    fclose(dest_file);

    // Copy file attributes.
    fileattrib=_chmod(src_filename, 0);
    _chmod(dest_filename, 1, fileattrib);
    unlink(src_filename);               // Delete file.
    printf(" [ok]\n");

} // end do_copy.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char fileargc,*fileargv[255],optargc,*optargv[255],src_pathname[MAXPATH]="",dest_pathname[MAXPATH]="",*ptr,option[255]="",i,environ[255],src_filename[MAXFILE+MAXEXT]="",dest_filename[MAXFILE+MAXEXT]="",ch;
    int length;

    classify_args(argc, argv, &fileargc, fileargv, &optargc, optargv);
    atexit(exit_fn);

    // Read COPYCMD to set confirmation.
    strncpy(environ,getenv("COPYCMD"),sizeof(environ));
    if (environ[0] != '\0')
        {
        strupr(environ);
        if (strcmp(environ, "/Y") == 0)
            opt_prompt=0;               // Overwrite existing files.
        else if (strcmp(environ, "/N") == 0)
            opt_prompt=1;               // Skip existing files.
        else
            opt_prompt=2;               // Ask for confirmation.

        } // end if.

    for (i=0;i<optargc;i++)             // Get options.
        {
        strcpy(option, optargv[i]);
        if (stricmp(option, "v") == 0)
            {
            old_verify=getverify();
            opt_verify=-1;
            } // end if.
        else if (stricmp(option, "y") == 0)
            opt_prompt=0;
        else if (stricmp(option, "-y") == 0)
            opt_prompt=2;
        else if (stricmp(option, "?") == 0)
            opt_help=-1;
        else
            {
            printf("Invalid parameter-%s\n", optargv[i]);
            exit(4);
            } // end else.

        } // end for.

    if (opt_help)
        {
        printf("Moves a file to another location.\n\n"
               "Syntax: MOVE [/Y | /-Y] source destination [/?]\n"
               "  source      The name of the file you want to move\n"
               "  destination Where you want to move the file to\n"
               "  /Y          Supresses prompting to confirm you want to overwrite\n"
               "              an existing destination file.\n"
               "  /-Y         Causes prompting to confirm you want to overwrite an\n"
               "              existing destination file.\n"
               "  /V          Verifies each file as it is written to the destination file\n"
               "              to make sure that the destination files are identical to the\n"
               "              source files\n"
               "  /?          Displays this help message\n");
        exit(0);
        } // end if.

    if (argc<2 || fileargc<1)
        {
        printf("Required parameter missing\n");
        return 1;
        } // end if.

    length=strlen(fileargv[0]);
    if (length>(MAXPATH-1))
        {
        error("Source path too long\n");
        exit(4);
        } // end if.

    _fullpath(src_pathname, fileargv[0], MAXPATH);
    if (src_pathname[0] == '\0')
        {
        printf("Invalid source drive specification\n");
        exit(4);
        } // end if.

    // Check source path.
    if (!dir_exists(src_pathname))
        {
        // Source path contains a filename/-pattern -> separate it.
        ptr=strrchr(src_pathname, *DIR_SEPARATOR);
        ptr++;
        strncpy(src_filename, ptr, sizeof(src_filename));
        *ptr='\0';
        if (!dir_exists(src_pathname))
            {
            error("Source path not found");
            exit(4);
            } // end if.

        } // end if.
    else
        {
        // Source is a directory -> filepattern="*.*".
        strncpy(src_filename, "*.*", sizeof(src_filename));
        } // end else.

    addsep(src_pathname);
    length=strlen(src_pathname);
    if (length>(MAXDRIVE-1+MAXDIR-1))
        {
        error("Source path too long\n");
        exit(4);
        } // end if.

    // Get destination pathname (with trailing backspace) and filename/-pattern.
    if (fileargc<2)
        {
        // No destination path specified -> use current.
        getcwd(dest_pathname, MAXPATH);
        strncpy(dest_filename, "*.*", sizeof(dest_filename));
        } // end if.
    else
        {
        // Destination path specified.
        length=strlen(fileargv[1]);
        if (length>(MAXPATH-1))
            {
            error("Destination path too long\n");
            exit(4);
            } // end if.

        _fullpath(dest_pathname, fileargv[1], MAXPATH);
        if (dest_pathname[0] == '\0')
            {
            error("Invalid destination drive specification\n");
            exit(4);
            } // end if.

        // Check destination path.
        if (fileargv[1][length-1] != *DIR_SEPARATOR && !dir_exists(dest_pathname))
            {
            ptr=strrchr(dest_pathname, *DIR_SEPARATOR);
            ptr++;
            strncpy(dest_filename, ptr, sizeof(dest_filename));
            *ptr='\0';
            if (((ptr=strchr(dest_filename, '*')) == NULL && (ptr=strchr(dest_filename, '?')) == NULL) || !switch_intodir)
                {
                if (switch_intodir)
                    {
                    // Destination is a directory.
                    strncat(dest_pathname, dest_filename, sizeof(dest_pathname));
                    strncpy(dest_filename, "*.*", sizeof(dest_filename));
                    } // end if.

                } // end if.

            } // end if.
        else
            {
            // Destination is a directory.
            strncpy(dest_filename, "*.*", sizeof(dest_filename));
            } // end else.

        } // end else.

    addsep(dest_pathname);
    length=strlen(dest_pathname);
    if (length>(MAXDRIVE-1+MAXDIR-1))
        {
        error("Destination path too long\n");
        exit(4);
        } // end if.

    dest_drive=toupper(dest_pathname[0])-64;
    if (opt_verify)
        setverify(1);

    move_files(src_pathname, src_filename, dest_pathname, dest_filename);
    return 0;

} // end main.
