;
; XCDROM.ASM    Written 20-Jan-2006 by Jack R. Ellis.
;
; XCDROM is free software.  You can redistribute and/or modify it under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later versions at your option.  XCDROM is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!   See the GPL for details.   You should have received a copy
; of the GPL with your XCDROM files.  If not write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; http://www.gnu.org/licenses/
;
; This is a DOS driver for 1 to 3 CD-ROM drives on PC mainboards having
; a VIA VT8235 or similar chipset.   On loading, XCDROM checks both IDE
; channels for CD-ROM drives and runs all the drives it finds.   It has
; switch options (see below) to indicate a desired "driver name" and to
; override its "IDE order" search and check for specific CD-ROM drives.
; XCDROM accepts requests from a "CD-ROM redirector" (SHCDX33A, MSCDEX,
; etc.) for the CD-ROM drive.   If the XDMA disk driver (V3.1 or later)
; is also present and is using output overlap, XCDROM shall synchronize
; all I-O activity on its drive with XDMA I-O.    This lets XDMA output
; overlap be used even where an UltraDMA hard-disk and the CD-ROM drive
; are sharing the same IDE channel!    Also, if V3.1+ XDMA with overlap
; or any V3.0+ XDMA/XDMAJR driver is present, a CD-ROM drive capable of
; UltraDMA will be enabled for it.   XCDROM can be run with older XDMA/
; UDMA2/UDMA drivers or "stand-alone", in which case it will scan for a
; mainboard UltraDMA controller by itself.    If one is found, a CD-ROM
; drive capable of UltraDMA will also be enabled for it.   Other XCDROM
; features are the same as for any DOS CD-ROM driver.   XCDROM does not
; use interrupts and is only for "legacy" IDE channels at I-O addresses
; 01F0h (primary) and 0170h (secondary).
;
; XCDROM switch options are as follows:
;
;    /AX   Excludes ALL audio functions.   This makes the driver report
;	     on a Device-Status request that it reads DATA tracks only!
;	     /AX reduces the resident driver by 480 bytes.    UltraDMA,
;	     dual-drives, and other driver features are NOT affected!
;
;    /D:   Specifies the desired "device name" which SHCDX33A or MSCDEX
;	     will use during their initialization to address the CD-ROM
;	     drives.   Examples are:  /D:CDROM1  /D:MYCDROM  etc.   The
;	     device name must be from 1 to 8 bytes valid for use in DOS
;	     filenames.   If /D: is omitted, or the "device name" after
;	     a /D: is missing or invalid, "XCDROM" will be the default.
;
;    /L    Limits UltraDMA to "low memory" below 640K.   /L is REQUIRED
;	     to use UMBPCI or a similar driver whose upper-memory areas
;	     cannot do UltraDMA.   If /L is given, the driver must load
;	     in LOW memory so its DMA command-lists can fetch preperly,
;	     or driver loading will ABORT!   /L causes any I-O requests
;	     above 640K to use "PIO mode" input.   Note that /L will be
;	     IGNORED if /UX is also given.
;
;    /Mn   Specifies the MAXIMUM UltraDMA "mode" to be set for a CD-ROM
;	     drive, where  n  is a number between 0 and 2, as follows:
;		     0 = ATA-16, 16 MB/sec.
;		     1 = ATA-25, 25 MB/sec.
;		     2 = ATA-33. 33 MB/sec.
;	     A CD-ROM drive designed to use "modes" LESS than the given
;	     value will be limited to its own highest "mode".   /M will
;	     be IGNORED for CD-ROM drives which cannot do UltraDMA, and
;	     it will be ignored for ALL drives if /UX is also given.
;
;    /PM   Requests the driver to check the IDE primary-master unit for
;	     a CD-ROM drive during driver init.    If a CD-ROM drive is
;	     NOT found as primary-master, driver loading will ABORT!
;
;    /PS   Same as /PM but tests the primary-slave unit only.
;
;    /SM   Same as /PM but tests the secondary-master unit only.
;
;    /SS   Same as /PM but tests the secondary-slave unit only.
;
;	     --- NOTE ---
;	     Using multiple drives, multiple  /PM /PS /SM /SS  switches
;	     can be given.    The first-specified drive is addressed as
;	     "unit 0", the second as "unit 1", etc.   If fewer switches
;	     than drives are given, the unreferenced drives will NOT be
;	     used.    If NO such switches are given, the driver "scans"
;	     for CD-ROM drives, from primary-master to secondary-slave.
;	     The first drive found will be "unit 0", the second will be
;	     "unit 1", etc.
;
;    /UF   Enables "Fast UltraDMA".   Data input requests that cross an
;	     UltraDMA "64K boundary" are executed using a 2-element DMA
;	     command list, one for data up to the boundary, and one for
;	     data beyond it.   CD-ROM speed is increased significantly.
;	     "PIO mode" input is still needed for user buffers that are
;	     misaligned (not at an even 4-byte address).    /UF will be
;	     IGNORED for CD-ROM drives which cannot do UltraDMA.
;
;	     --- NOTE ---
;	     Despite any UltraDMA specs, NOT ALL chipsets or mainboards
;	     can run multi-element DMA commands properly!   Although it
;	     is valuable, /UF must be TESTED on every system, and "Fast
;	     UltraDMA" should be enabled with CARE!!
;
;    /UX   Disables ALL UltraDMA, even for CD-ROM drives capable of it.
;	     The driver then uses "PIO mode" for all data input.    /UX
;	     should be needed only for tests and diagnostic work.
;
; For each switch, a dash may replace the slash, and lower-case letters
; may be used.
;
;
; Revision History:
; ----------------
;  V1.9  20-Jan-06  JRE  Fixed errors in Multi-Session and I-O logic.
;  V1.8  17-Jan-06  JRE  Fixed Multi-Session "TOC input" to support ALL
;			   drives, added a drive "reset" on I-O errors.
;  V1.7  14-Jan-06  JRE  Updated XCDROM to read a "Multi Session" disk.
;  V1.6  10-Jan-06  JRE  XCDROM now has stand-alone UltraDMA capability
;			   and no longer requires XDMA/XDMAJR!   "Audio
;			   Busy" status is now updated on ALL requests.
;  V1.5   5-Jan-06  JRE  Fixed "Audio Status" & /AX device-status flags
;			   and added support for up to 3 CD-ROM drives.
;  V1.4   2-Jan-06  JRE  Initial release, added /AX and dual drives.
;  V1.3  30-Dec-05  JRE  4th "Beta" issue, uses V3.1+ XDMA "OCheck".
;  V1.2  23-Dec-05  JRE  3rd "Beta" issue, new /UF and /UX switches.
;  V1.1  15-Dec-05  JRE  2nd "Beta" issue, improved XDMA linkage.
;  V1.0  14-Dec-05  JRE  Original "Beta" XCDROM issue.

;
;
; General Program Equations.
;

?STACK	equ 0
?SETUMODE equ 0

VER 	equ <'V1.0, 5-13-2007'>	;Driver version number and date.
%if ?STACK
BSTACK	equ	330		;"Basic" driver local-stack size.
_STACK	equ	332		;Regular driver local-stack size.
%endif
;XDDMAAD	equ	00008h		;XDMA "DMAAd" offset,  CANNOT CHANGE!
;XDFLAGS	equ	00012h		;XDMA "Flags" offset,  CANNOT CHANGE!
XDCHECK	equ	00314h		;XDMA "OCheck" offset, CANNOT CHANGE!
PCHADDR	equ	001F0h		;"Legacy" IDE primary base address.
SCHADDR	equ	00170h		;"Legacy" IDE secondary base address.
MSELECT	equ	0A0h		;"Master" device-select bits.
SSELECT	equ	0B0h		;"Slave"  device-select bits.
COOKSL	equ	2048		;CD-ROM "cooked" sector length.
RAWSL	equ	2352		;CD-ROM "raw" sector length.
CMDTO	equ	00Ah		;500-msec minimum command timeout.
SEEKTO	equ	025h		;2-second minimum "seek"  timeout.
STARTTO	equ	049h		;4-second minimum startup timeout.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
IXM		equ	2048		;IOCTL transfer-length multiplier.
CR		equ	00Dh		;ASCII carriage-return.
LF		equ	00Ah		;ASCII line-feed.
TAB		equ	009h		;ASCII "tab".
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Drive-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY		equ	080h		;IDE controller is busy.
DRQ		equ	008h		;IDE data request.
ERR		equ	001h		;IDE general error flag.
DMI		equ	004h		;DMA interrupt occured.
DME		equ	002h		;DMA error occurred.
SETM	equ	003h		;Set Mode subcommand.
SETF	equ	0EFh		;Set Features command.
LBABITS equ	0E0h		;Fixed LBA command bits.

;
; DOS "Request Packet" Layout.
;
RP		struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Command code.
RPStat	dw	?	 	;Status field.
		db 8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPSize	dd	?		;Resident driver size.
RPCL	dd	?		;Command-line data pointer.
RP		ends

RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
RPBUSY	equ	00200h		;Packet "busy" flag.
;
; IOCTL "Request Packet" Layout.
;
IOC	struc
		db	13	dup (?)	;Request "header" (unused by us).
		db	?		;Media descriptor byte (Unused by us).
IOCAdr	dd	? 		;Data-transfer address.
IOCLen	dw	?		;Data-transfer length.
		dw	?		;Starting sector (unused by us).
		dd	?		;Volume I.D. pointer (unused by us).
IOC	ends
;
; Read Long "Request Packet" Layout.
;
RL struc
		db	13	dup (?)	;Request "header" (unused by us).
RLAM	db	?		;Addressing mode.
RLAddr	dd	?		;Data-transfer address.
RLSC	dw	?		;Data-transfer sector count.
RLSec	dd	?		;Starting sector number.
RLDM	db	?		;Data-transfer mode.
RLIntlv	db	?		;Interleave size.
RLISkip	db	?		;Interleave skip factor.
RL	ends

		.386
        .model flat

		include jlm.inc
    
		.code

startcode label byte

pRequest	dd 0	;linear address where request header is stored
dwRequest	dd 0	;linear address request header
dwBase		dd 0	;linear address driver base
dwCmdLine	dd 0
wBaseSeg	dw 0

;
; DOS CD-ROM Driver Device Header.
;
;@		dd	0FFFFFFFFh	;Link to next header block.
;   	dw	0C800h		;Driver "device attributes".
;   	dw	offset Strat		;"Strategy" routine offset.
;   	dw	offset DevIntJ		;"Device-Interrupt" routine offset.
;DvrName	db	'XCDROM  '	;DOS "device name" (XCDROM default).
;		dw	0		;+18 (Reserved).
;		db	0		;+20 First assigned drive letter.
;Units	db	0		;+21 Number of CD-ROM drives (1 or 2).
;
; Main I-O Variables (here to align the VDS and DMA variables below).
;
XFRLn	dw	0		;I-O data transfer length.
XFRAd	dd	0		;I-O data transfer address.
;
; VDS and DMA Variables.
;
PRDAd	dd	offset IOAdr - offset startcode	;PRD 32-bit command addr. (Init set).

;--- VDS DDS structure

VDSLn	dd	offset ResEnd - offset startcode ; VDS buffer length.

VDSOf	dd	0		;VDS 32-bit offset.
VDSSg	dw	0		;VDS 16-bit segment (hi-order zero).
		dw  0       ;VDS buffer ID (not used)
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	0		;1st DMA byte count.
IOAdr2	dd	0		;2nd DMA 32-bit address & byte count
IOLen2	dd	080000000h	;  for input "across" a 64K boundary!
;
; ATAPI "Packet" Area (always 12 bytes for a CD-ROM).
;
Packet	db	0		;Opcode.
		db	0		;Unused (LUN and reserved).
PktLBA	dd	0		;CD-ROM logical block address.
PktLH	db	0		;"Transfer length" (sector count).
PktLn	dw	0		;Middle- and low-order sector count.
PktRM	db	0		;Read mode ("Raw" Read Long only).
		dw	0		;Unused ATAPI "pad" bytes (required).
;
; Miscellaneous Driver Variables.
;
;XOCheck	dw	XDCHECK	;XDMA's "OCheck" subroutine pointer.
;XDSeg	dw	0		;XDMA's segment address (set by Init).
AudAP	dd	0		;Current audio-start address pointer.
DMAAd	dw	0FFFFh	;Current DMA cmd. addr. (set by Init).
IDEAd	dw	0		;Current IDE data-register address.
IDESl	db	0		;Current device-select command byte.
SyncF	db	0		;Current XDMA synchronization flag.
BusyF	db	0		;"Sync busy" flag (in sync with XDMA).
VLF		db	0		;VDS "lock" flag (001h = buffer lock).
DMAFl	db	0		;DMA input flag (001h if so).
Try		db	0		;I-O retry counter.
		db	0,0		;(Unused alignment "filler").
;
; Audio Function Buffer (16 bytes) for most CD-ROM "audio" requests.
;   The variables below are used only during driver initialization.
;
InBuf	equ	$
UTblP	dd	offset UnitTbl		;Initialization unit table pointer.
PrDMA	dw	0FFFFh				;Primary DMA address   (set by Init).
IEMsg	dd	0					;Init error-message pointer.
UFXSw	db	0F1h				;UltraDMA "F/X" switch (set by Init).
MaxUM	db	0FFh				;UltraDMA "mode" limit (set by Init).
UFlag	db	0					;UltraDMA "mode" flags (set by Init).
UMode	db	0					;UltraDMA "mode" value (set by Init).
		db	0					;(Unused alignment "filler").
SyncX	db	0FFh				;"No XDMA synchronization" flag.
ScanX	dd	offset ScanP		;Scan table index (0FFFFh = no scan).
;
; Unit Parameter Tables.   If you want a 4th drive, simply add 1 more
;   parameter table -- NO extra code and NO other changes are needed!
;
UnitTbl	dw	0FFFFh		;Unit 0 DMA address   (set by Init).
		dw	0FFFFh		;	IDE address   (set by Init).
		db	0FFh		;	Device-select (set by Init).
		db	0FFh		;	XDMA sync bit (set by Init).
		db	0			;	(Unused alignment "filler").
		db	0FFh		;	Media-change flag.
		dd	0FFFFFFFFh	;	Current audio-start address.
		dd	0FFFFFFFFh	;	Current audio-end   address.
		dd	0FFFFFFFFh	;	Last-session starting LBA.
		dd	0FFFFFFFFh	;Unit 1 Parameters  (same as above).
		dd	0FF00FFFFh
		dd	0FFFFFFFFh
		dd	0FFFFFFFFh
		dd	0FFFFFFFFh
		dd	0FFFFFFFFh	;Unit 2 Parameters  (same as above).
		dd	0FF00FFFFh
		dd	0FFFFFFFFh
		dd	0FFFFFFFFh
		dd	0FFFFFFFFh
UTblEnd	equ	$		;(End of all unit tables).
;
; Dispatch Table for DOS CD-ROM request codes 0 through 14.
;
DspTbl1	dd	DspLmt1		;Number of valid request codes.
		dd	Try2ndD		;Invalid-request handler address.
DspTblA	dd	UnSupp		;00 -- Initialization  (special).
		dd	UnSupp		;01 -- Media Check	(unused).
		dd	UnSupp		;02 -- Build BPB	(unused).
		dd	Try3rdD		;03 -- IOCTL Input.
		dd	UnSupp		;04 -- Input		(unused).
		dd	UnSupp		;05 -- Input no-wait	(unused).
		dd	UnSupp		;06 -- Input Status	(unused).
		dd	UnSupp		;07 -- Input flush	(unused).
		dd	UnSupp		;08 -- Output		(unused).
		dd	UnSupp		;09 -- Output & verify	(unused).
		dd	UnSupp		;10 -- Output status	(unused).
		dd	UnSupp		;11 -- Output flush	(unused).
		dd	Try4thD		;12 -- IOCTL Output.
		dd	Ignored		;13 -- Device Open     (ignored).
		dd	Ignored		;14 -- Device Close    (ignored).
DspLmt1	equ	($ - offset DspTblA)/4	;Request-code limit for this table.

		dd DspLmt1 dup (0)
;
; Dispatch Table for DOS CD-ROM request codes 128 through 136.
;
DspTbl2	dd	DspLmt2		;Number of valid request codes.
		dd	UnSupp		;Invalid-request handler address.
DspTblB	dd	ReqRL		;128 -- Read Long.
		dd	UnSupp		;129 -- Reserved	(unused).
@RqPref	dd	ReqSeek		;130 -- Read Long Prefetch.
@RqSeek	dd	ReqSeek		;131 -- Seek.
@RqPlay	dd	ReqPlay		;132 -- Play Audio.
@RqStop	dd	ReqStop		;133 -- Stop Audio.
		dd	UnSupp		;134 -- Write Long	(unused).
		dd	UnSupp		;135 -- Wr. Long Verify	(unused).
@RqRsum	dd	ReqRsum		;136 -- Resume Audio.
DspLmt2	equ	( $ - offset DspTblB)/4	;Request-code limit for this table.

		dd DspLmt2 dup (0)

;
; Dispatch table for IOCTL Input requests.
;
DspTbl3	dd	DspLmt3		;Number of valid request codes.
		dd	UnSupp		;Invalid-request handler address.
DspTblC	dd	ReqDHA		;00 -- Device-header address.
@RqCHL	dd	ReqCHL		;01 -- Current head location.
		dd	UnSupp		;02 -- Reserved		(unused).
		dd	UnSupp		;03 -- Error Statistics	(unused).
		dd	UnSupp		;04 -- Audio chan. info (unused).
		dd	UnSupp		;05 -- Read drive bytes	(unused).
		dd	ReqDS		;06 -- Device status.
		dd	ReqSS		;07 -- Sector size.
@RqVS	dd	ReqVS		;08 -- Volume size.
		dd	ReqMCS		;09 -- Media-change status.
@RqADI	dd	ReqADI		;10 -- Audio disk info.
@RqATI	dd	ReqATI		;11 -- Audio track info.
@RqAQI	dd	ReqAQI		;12 -- Audio Q-channel info.
		dd	UnSupp		;13 -- Subchannel info	(unused).
		dd	UnSupp		;14 -- Read UPC code	(unused).
@RqASI	dd	ReqASI		;15 -- Audio status info.
DspLmt3	equ	( $ - offset DspTblC)/4	;Request-code limit for this table.

       	dd	5*IXM
		dd	6*IXM
		dd	0
		dd	0
		dd	0
		dd	0
		dd	5*IXM
		dd	4*IXM
		dd	5*IXM
		dd	2*IXM
		dd	7*IXM
		dd	7*IXM
		dd	11*IXM
		dd	0
		dd	0
		dd	11*IXM

;
; Dispatch table for IOCTL Output requests.
;
DspTbl4	dd	DspLmt4		;Number of valid request codes.
		dd	UnSupp		;Invalid-request handler address.
DspTblD	dd	ReqEjct		;00 -- Eject Disk.
		dd	ReqDoor		;01 -- Lock/Unlock Door.
		dd	ReqRS   	;02 -- Reset drive.
		dd	UnSupp		;03 -- Audio control	(unused).
		dd	UnSupp		;04 -- Write ctl. bytes	(unused).
		dd	ReqTray 	;05 -- Close tray.
DspLmt4	equ	( $ - offset DspTblD)/4	;Request-code limit for this table.

		dd	1*IXM
		dd	2*IXM
		dd	1*IXM
		dd	0
		dd	0
		dd	1*IXM


;
; "Device-Interrupt" routine -- This routine processes DOS requests.
;
DevInt:
	xor	ebx,ebx			;Zero BX-reg. for relative commands.
	call ZPacket		;Clear our ATAPI packet area.
    mov esi, [pRequest]
    movzx eax,word ptr [esi+0]
    movzx esi,word ptr [esi+2]
    shl esi, 4
    add esi, eax      	 	;Point to DOS request packet.
    mov [dwRequest], esi	;linear address request header

	mov	word ptr [esi].RP.RPStat,RPDON  ;Init status to "done".
	mov	al,[esi].RP.RPSubU	   ;Get unit-table offset.
	mov	ah,20
	mul	ah
    movzx eax,ax
	mov	edi,offset UnitTbl+8	;Set unit's audio-start address ptr.
	add	edi,eax
	mov	[AudAP],edi
	mov	eax,[edi-8]			;Set drive DMA and IDE addresses.
	mov	dword ptr [DMAAd],eax
	mov	ax,[edi-4]			;Set device-select & XDMA "sync" flag.
	mov	word ptr [IDESl],ax
	mov	al,[esi].RP.RPOp	;Get packet request code.
	mov	edi,offset DspTbl1	;Point to 1st DOS dispatch table.
	call Dspatch			;Dispatch to desired request handler.
    
    VMMCall Simulate_Far_Ret
	ret
;
; Function-Code "Dispatch" Routines.
;
Try2ndD:
	sub	al,080h				;Not request code 0-15:  subtract 128.
	mov	edi,offset DspTbl2	;Point to 2nd DOS dispatch table.
	jmp	short Dspatch		;Go try request-dispatch again.
Try3rdD:
	mov	edi,offset DspTbl3	;Point to IOCTL Input dispatch table.
	jmp	short TryIOC
Try4thD:
	mov	edi,offset DspTbl4	;Point to IOCTL Output dispatch table.
TryIOC:
;	les	si,[esi].IOC.IOCAdr  ;Get actual IOCTL request code.
	movzx ecx,word ptr [esi].IOC.IOCAdr+2
    movzx esi,word ptr [esi].IOC.IOCAdr+0
    shl ecx,4
    add esi, ecx
	mov	al,[esi]
	mov	esi,[dwRequest]		;Reload DOS request-packet address.
    
Dspatch:
	xor edx,edx
	movzx ecx, byte ptr [edi]
	add	edi,4   			;(Skip past table-limit value).
	cmp	al,cl				;Is request code out-of-bounds?
	jae	Dsptch1				;Yes?  Dispatch to error handler!
	add	edi,4   			;Skip past error-handler address.
	movzx eax,al			;Point to request-handler address.
	lea	edi,[edi+eax*4]
    mov edx,[edi+ecx*4]
Dsptch1:
	mov	edi,[edi]			;Get handler address from table.
;	mov	edi,007FFh
;	and	edi,edx
;	xor	edx,edi				;IOCTL request (xfr length > 0)?
;	jz	Dsptch2
	and edx, edx
	jz	Dsptch2
	shr	edx,11					;Ensure correct IOCTL transfer
	mov	[esi].IOC.IOCLen,dx		;length is set in DOS packet.
;   les	si,es:[si].IOC.IOCAdr	;Get IOCTL data-transfer address.
	movzx ecx,word ptr [esi].IOC.IOCAdr+2
    movzx esi,word ptr [esi].IOC.IOCAdr+0
    shl ecx,4
    add esi, ecx
Dsptch2:
DspGo:
	jmp edi					;Dispatch to desired request handler.
    
GenFail:
	mov	al,12				;General failure!  Get error code.
	jmp	short ReqErr		;Go post packet error code & exit.
UnSupp:
	mov	al,3				;Unsupported request!  Get error code.
	jmp	short ReqErr		;Go post packet error code & exit.
SectNF:
	mov	al,8				;Sector not found!  Get error code.
ReqErr:
	mov	esi,[dwRequest]		;Reload DOS request-packet address.
	mov	ah,081h				;Post error flags & code in packet.
	mov	[esi].RP.RPStat,ax
Ignored:
	ret						;Exit ("ignored" request handler).
;
; IOCTL Input "Device Header Address" handler, placed here to AVOID
;   the need for XDMA I-O synchronization in our "dispatch" logic.
;
ReqDHA:
	push eax
    mov ax,[wBaseSeg]
    shl eax,16
    mov ax,bx
	mov	[esi+1], eax
    pop eax
	ret
;
; IOCTL Input "Sector Size" handler, placed here to AVOID the need
;   for XDMA I-O synchronization in our "dispatch" logic.
;
ReqSS:
	cmp	byte ptr [esi+1],1  ;Is read mode "cooked" or "raw"
	ja	GenFail		;No?  Post "general failure" & exit.
	mov	ax,RAWSL	;Get "raw" sector length.
	je	RqSS1		;If "raw" mode, set sector length.
	mov	ax,COOKSL	;Get "cooked" sector length.
RqSS1:
	mov	[esi+2],ax	;Post sector length in IOCTL packet.
RqSSX:
	ret
SyncReq	equ	$			;Handlers beyond here need I-O sync!
;
; DOS "Read Long" handler.
;
ReqRL:
	call ValSN						;Validate starting sector number.
	call MultiS						;Handle Multi-Session disk if needed.
	jc	ReqErr						;If error, post return code & exit.
	mov	cx,[esi].RL.RLSC			;Get request sector count.
	jcxz RqSSX						;If zero, simply exit.
	xchg cl,ch						;Save swapped sector count.
	mov	[PktLn],cx
	cmp	byte ptr [esi].RL.RLDM,1	;"Cooked" or "raw" read mode?
	ja	SectNF						;No?  Return "sector not found"!
	mov	dl,028h						;Get "cooked" input values.
	mov	ax,COOKSL
	jb	RqRL1						;If "cooked" input, set values.
	mov	dl,0BEh						;Get "raw" input values.
	mov	ax,RAWSL
	mov	byte ptr [PktRM],0F8h		;Set "raw" input flags.
RqRL1:
	mov	byte ptr [Packet],dl		;Set "packet" opcode.
	mul	word ptr [esi].RL.RLSC		;Get desired input byte count.
	test dx,dx						;More than 64K bytes desired?
	jnz	SectNF						;Yes?  Return sector not found!
    movzx eax,ax
	mov	[VDSLn],eax					;Set VDS and DMA byte counts.
	mov	[IOLen],eax
    
	movzx eax,word ptr [esi].RL.RLAddr+2
    shl eax, 4
    movzx ecx,word ptr [esi].RL.RLAddr+0
    add eax, ecx
	mov	[VDSOf],eax
	mov	[XFRAd],eax

	test byte ptr [DMAAd],007h		;Is drive using UltraDMA?
	jnz	RqRL5						;No, do "PIO mode" input.

	mov	dx,0000h
    push esi
	mov	esi,[VDSOf]
    mov ecx,[VDSLn]
    VxDCall VDMAD_Lock_DMA_Region
    pop esi
	jc	RqRL5						;Error -- use PIO input.
    mov [IOAdr],edx
	mov eax,edx						;Get phys address.
	test al,003h					;Is user buffer 32-bit aligned?
	jnz	RqRL4						;No, use PIO.
	cmp	word ptr [IOAdr+2],-1  		;Is DMA beyond our limit?
@DMALmt	equ	$-1						;(009h for a 640K limit).
	ja	RqRL4						;Yes, "unlock" & use PIO.
	mov	byte ptr [IOLen+3],080h		;Set DMA list "end" flag.
	mov	ecx,[IOLen]					;Get lower ending DMA address.
	dec	ecx							;(IOLen - 1 + IOAdr).
	add	ax,cx						;Would input cross a 64K boundary?
	jnc	RqRL3						;No, set DMA flag & do transfer.
	inc	ax							;Get bytes above 64K boundary.
	cmp	ax,64						;Is this at least 64 bytes?
@UFFlag:
	jb	RqRL4						;No, "unlock" buffer and use PIO.
	inc	cx							;Get bytes below 64K boundary.
	sub	cx,ax
	cmp	cx,64						;Is this at least 64 bytes?
	jb	RqRL4						;No, "unlock" buffer and use PIO.
	mov	[IOLen2],eax				;Set 2nd command-list byte count.
	movzx eax,cx					;Set 1st command-list byte count.
	mov	[IOLen],eax
	add	eax,[IOAdr]					;Set 2nd command-list address.
	mov	[IOAdr2],eax
RqRL3:
	inc	byte ptr [DMAFl]			;Set UltraDMA input flag.
	jmp	short RqRL5					;Go execute read request.
    
RqRL4:
RqRL5:
	call DoIO						;Execute desired read request.
	jnc	RqRL6						;If no errors, go exit below.
	call ReqErr						;Post desired error code.
RqRL6:
	mov	[DMAFl],bl					;Reset UltraDMA input flag.
RqRL7:	
	ret

RqRLX:
	ret
;
; DOS "Seek" handler.
;
DOSSeek:
	call ValSN						;Validate desired seek address.
	call MultiS						;Handle Multi-Session disk if needed.
	jc	DOSSkE						;If error, post return code & exit.
	mov	byte ptr [Packet],02Bh 		;Set "seek" command code.
DOSSk1:
	call DoIOCmd					;Issue desired command to drive.
DOSSkE:
	jc ReqErr	;If error, post return code & exit.
	ret
;
; IOCTL Input "Device Status" handler.
;
ReqDS:
	mov	dword ptr [Packet],0002A005Ah  ;Set up mode-sense.
	mov	al,16			;Set input byte count of 16.
	call DoBufIO		;Issue mode-sense for hardware data.
	jc	DOSSkE			;If error, post return code & exit.
	mov	eax,00214h		;Get our basic driver status flags.
@Status	equ	$-4			;(Set by Init to 00204h for /AX).
	cmp	byte ptr [edi+2],071h  ;"Unknown CD", i.e. door open?
	jne	ReqDS1			;No, check "locked" status.
	or	al,001h			;Post "door open" status flag.
ReqDS1:
	test byte ptr [edi+14],002h ;Drive pushbutton "locked out"?
	jnz	ReqDS2			;No, set flags in IOCTL.
	or	al,002h			;Set "door locked" status flag.
ReqDS2:
	mov	[esi+1],eax		;Set status flags in IOCTL buffer.
@RqDSX:
	jmp	ReadAST			;Go post "busy" status and exit.
;
; IOCTL Input "Media-Change Status" handler.
;
ReqMCS:
	call DoIOCmd		;Issue "Test Unit Ready" command.
	mov	edi,[AudAP]		;Get media-change flag from table.
	mov	al,[edi-1]
	mov	[esi+1],al		;Return media-change flag to user.
	ret
;
; IOCTL Output "Eject Disk" handler.
;
ReqEjct:
	mov	word ptr [Packet],0011Bh	;Set "eject" commands.
	mov	byte ptr [PktLBA+2],002h	;Set "eject" function.
	jmp	short DOSSk1					;Go do "eject" & exit.
;
; IOCTL Output "Lock/Unlock Door" handler.
;
ReqDoor:
	mov	al,[esi+1]			;Get "lock" or "unlock" function.
	cmp	al,001h				;Is function byte too big?
	ja	RqRS1				;Yes, post "General Failure" & exit.
	mov	cx,0001Eh			;Get "lock" & "unlock" commands.
RqDoor1:
	mov	word ptr [Packet],cx	;Set "packet" command bytes.
	mov	byte ptr [PktLBA+2],al 	;Set "packet" function byte.
	call DoIOCmd			;Issue desired command to drive.
	jc	DOSSkE				;If error, post return code & exit.
	jmp	short @RqDSX		;Go post "busy" status and exit.
;
; IOCTL Output "Reset Drive" handler.
;
ReqRS:
	call StopDMA			;Stop previous DMA & select drive.
	inc	dx					;Point to IDE command register.
	mov	al,008h				;Do an ATAPI "soft reset" command.
	out	dx,al
	call TestTO				;Await controller-ready.
RqRS1:
	jc GenFail				;Timeout!  Return "General Failure".
	ret
;
; IOCTL Output "Close Tray" handler.
;
ReqTray:
	mov	al,003h				;Get "close tray" function byte.
	mov	cx,0011Bh			;Get "eject" & "close" commands.
	jmp	short RqDoor1		;Go do "close tray" command above.
;
; Subroutine to handle a Multi-Session disk for DOS reads and seeks.
;   Multi-Session disks require (A) saving the last-session starting
;   LBA for a new disk after any media-change and (B) "offsetting" a
;   read of the VTOC or initial directory block, sector 16 or 17, to
;   access the VTOC/directory of the disk's last session.
;
MultiS:
	mov	edi,[AudAP]				;Point to drive variables.
	cmp	byte ptr [edi+11],0FFh	;Is last-session LBA valid?
	jne	MultiS1					;Yes, proceed with request.
	mov	byte ptr [Packet],043h	;Set "Read TOC" command.
	inc	byte ptr [PktLBA]		;Set "format 1" request.
	mov	al,12					;Set 12-byte allocation ct.
	call DoBufIO				;Read first & last session.
	jc	MultiSX					;If any error, exit below.
	mov	byte ptr [PktLBA],bl	;Reset "format 1" request.
	mov	al,[edi+3]				;Set last session in packet.
	mov	[PktLH],al
	call DoIO					;Read disk info for last session.
	jc	MultiSX					;If error, exit with carry set.
	mov	eax,[edi+8]				;"Swap" & save this disk's last-
	call Swap32					;  session starting LBA address.
	mov	edi,[AudAP]
	mov	[edi+8],eax
	call ZPacket				;Reset our ATAPI packet area.
MultiS1:
	mov	eax,[esi].RL.RLSec		;Get starting sector number.
	mov	edx,eax					;"Mask" sector to an even number.
	and	dl,0FEh
	cmp	edx,16					;Sector 16 (VTOC) or 17 (directory)?
	jne	MultiS2					;No, set sector in packet.
	add	eax,[edi+8]				;Offset sector to last-session start.
MultiS2:
	call Swap32					;"Swap" sector into packet as LBA.
	mov	[PktLBA],eax
	clc							;Clear carry flag (no errors).
MultiSX:
	ret
;
; Ye Olde I-O Subroutine.   ALL of our CD-ROM I-O is executed here!
;
DoBufIO:
	mov	byte ptr [PktLn+1],al	;Buffered -- set packet count.
DoBufIn:
	movzx eax,al			;Save data-transfer length.
	mov	[VDSLn],eax
	mov	[VDSOf],offset InBuf;Use our buffer for I-O.
    movzx eax,[wBaseSeg]
    shl eax, 4
	mov	[XFRAd],eax
	jmp	short DoIO			;Go start I-O below.
DoIOCmd:
	mov	[VDSLn],ebx			;Command only -- reset xfr length.
DoIO:
	push esi
	mov	byte ptr [Try],4	;Set request retry count of 4.
DoIO1:
	call StopDMA			;Stop previous DMA & select drive.
	call TestTO				;Await controller-ready.
	jc	DoIO4				;Timeout!  Handle as a "hard error".
	mov	eax,[VDSOf]			;Reset data-transfer buffer address.
	mov	[XFRAd],eax
	mov	ax,word ptr [VDSLn]	;Reset data-transfer byte count.
	mov	[XFRLn],ax
	cmp	[DMAFl],bl			;UltraDMA input request?
	je	DoIO3				;No, output our ATAPI "packet".
	mov	dx,[DMAAd]			;Point to DMA command register.
	mov	al,008h				;Reset DMA commands & set read mode.
	out	dx,al
	inc	edx					;Point to DMA status register.
	inc	edx
	in	al,dx				;Reset DMA status register.
	or	al,006h				;(Done this way so we do NOT alter
	out	dx,al				;  the "DMA capable" status flags!).
	inc	edx					;Set PRD pointer to our DMA address.
	inc	edx
	mov	esi,offset PRDAd
	outsd
DoIO3:
	mov	dx,[IDEAd]			;Point to IDE "features" register.
	inc	edx
	mov	al,[DMAFl]			;If UltraDMA input, set "DMA" flag.
	out	dx,al
	add	edx,3				;Point to byte count registers.
	mov	ax,[XFRLn]			;Output data-transfer length.
	out	dx,al
	inc	edx
	mov	al,ah
	out	dx,al
	inc	edx					;Point to command register.
	inc	edx
	mov	al,0A0h				;Issue "Packet" command.
	out	dx,al
	mov	cl,DRQ				;Await controller- and data-ready.
	call TestTO1
DoIO4:
	jc	DoIO7				;Timeout!  Handle as a "hard error".
	xchg eax,esi			;Save BIOS timer address.
	mov	dx,[IDEAd]			;Point to IDE data register.
	mov	ecx,6				;Output all 12 "Packet" bytes.
	mov	esi,offset Packet
	rep	outsw
	xchg eax,esi			;Reload BIOS timer address.
	mov	ah,STARTTO			;Allow 4 seconds for drive startup.
	cmp	[DMAFl],bl			;UltraDMA input request?
	je	DoIO9				;No, do "PIO mode" transfer below.
	mov	[XFRLn],bx			;Reset transfer length (DMA does it).
	add	ah,[esi]			;Set 4-second timeout in AH-reg.
    
	mov	ds:[48Eh],bl		;Reset BIOS disk-interrupt flag.
    
	mov	dx,[DMAAd]			;Point to DMA command register.
	in	al,dx				;Set DMA Start/Stop bit (starts DMA).
	inc	eax
	out	dx,al
    
DoIO5:
	inc	edx					;Point to DMA status register.
	inc	edx
	in	al,dx				;Read DMA controller status.
	dec	edx					;Point back to DMA command register.
	dec	edx
	and	al,DMI+DME			;DMA interrupt or DMA error?
	jnz	DoIO6				;Yes, halt DMA and check results.
	cmp	ah,[esi]			;Has our DMA transfer timed out?
	jz	DoIO6
    
    VMMCall Yield

	cmp	ds:[48Eh],bl		;Did BIOS get a disk interrupt?
	je	DoIO5 				;No, loop back & retest status.
	mov	al,DMI				;Set "simulated" interrupt flag.
    
DoIO6:
	xchg eax,esi				;Save ending DMA status.
	in	al,dx				;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	xchg eax,esi				;Reload ending DMA status.
	cmp	al,DMI				;Did DMA end with only an interrupt?
	jne	DoIO19				;No?  Handle as a "hard error"!
	inc	edx					;Reread DMA controller status.
	inc	edx
	in	al,dx
	test al,DME				;Any "late" DMA error after DMA end?
	jz	DoIO18				;No, go await controller-ready.
DoIO7:
	jmp	DoIO19				;Timeouts and DMA errors are "hard"!

DoIO9:
	xor	cl,cl				;"PIO mode" -- await controller-ready.
	call TestTO2
	jc	DoIO19				;Timeout!  Handle as a "hard error".
	test al,DRQ				;Did we also get a data-request?
	jz	DoIO18				;No, go await controller-ready.
	dec	edx					;Get number of buffer bytes.
	dec	edx
	in	al,dx
	mov	ah,al
	dec	edx
	in	al,dx
	inc	eax					;Make buffer byte count "even".
	and	al,0FEh
	mov	dx,[IDEAd]			;Point to IDE data register.
	movzx esi,[XFRLn]		;Get our data-transfer length.
	test esi,esi			;Any remaining bytes to transfer?
	jz	DoIO14				;No, "eat" input or "pad" output.
	mov	ecx,eax				;Set buffer count as block count.
	cmp	ecx,esi				;Buffer count > remaining bytes?
	jbe	DoIO10				;No, save current block count.
	mov	ecx,esi				;Set remaining bytes as block count.
DoIO10:
	push ecx				;Save current block count.
	mov	edi,[XFRAd]			;Get input data-transfer address.
	shr	ecx,1				;Get input word count.
	rep	insw				;Input all data words.
	jnc	DoIO13				;"Odd" byte of input?
	xchg eax,ecx			;Yes, save buffer byte count.
	in	ax,dx				;Input "odd" byte of data.
	stosb
	xchg eax,ecx			;Reload buffer byte count.
DoIO13:
	pop	ecx					;Reload current block count.
	add	[XFRAd],ecx			;Increment data-transfer address.
	sub	[XFRLn],cx			;Decrement data-transfer length.
	inc	ecx					;If block count is odd, "round" it.
	and	cl,0FEh
	sub	ax,cx				;Decrement buffer count.
DoIO14:
	xchg ax,cx				;Any "residual" bytes to handle?
	jcxz DoIO17				;No, see if more I-O is needed.
	shr	cx,1				;Get residual buffer word count.
DoIO16:
	in	ax,dx				;"Eat" residual input bytes.
	loop DoIO16
DoIO17:
	mov	ah,SEEKTO			;Allow 2 seconds if drive must "seek".
	jmp	DoIO9				;Go see if more I-O is needed.
DoIO18:
	call TestTO				;Await controller-ready.
	jc	DoIO19				;Timeout!  Handle as a "hard error".
	mov	esi,[AudAP]			;Get drive media-change flag pointer.
	dec	esi
	and	ax,00001h			;Did controller detect any errors?
	jz	DoIO21				;No, see if all data was transferred.
	sub	edx,6				;Get controller's sense key value.
	in	al,dx
	shr	al,4
	cmp	al,006h				;Is sense key "Unit Attention"?
	je	DoIO23				;Yes, check for prior media-change.
	mov	ah,0FFh				;Get 0FFh M.C. flag for "Not Ready".
	cmp	al,002h				;Is sense key "Drive Not Ready"?
	je	DoIO24				;Yes, go set our media-change flag.
DoIO19:
	mov	dx,[IDEAd]			;Hard error!  Point to command reg.
	add	edx,7
	mov	al,008h				;Issue ATAPI "soft reset" to drive.
	out	dx,al
	mov	al,11				;Get "hard error" return code.
DoIO20:
	dec	byte ptr [Try]		;Do we have more I-O retries left?
	jz	DoIO25				;No, set carry & return error code.
	jmp	DoIO1				;Try re-executing this I-O request.
DoIO21:
	movzx ecx,[XFRLn]		;Get remaining data-transfer length.
	jecxz DoIO22			;If zero, reset media-change & exit.
	cmp	byte ptr [Packet],028h	;"Cooked" data input req.?
	je	DoIO19				;Yes, see if we can retry.
	cmp	byte ptr [Packet],0BEh	;"Raw" data input request?
	je	DoIO19				;Yes, see if we can retry.
	mov	edi,[XFRAd]			;Load data-transfer address.
	rep	stosb				;"Pad" remaining bytes to 0.
DoIO22:
	mov	byte ptr [esi],001h	;Set "no media change" flag.
	clc						;Reset carry flag (no error).
	jmp	short DoIO26		;Go reload regs. and exit below.
DoIO23:
	mov	al,002h				;"Attention":  Get "Not Ready" code.
	cmp	[esi],bl				;Is media-change flag already set?
	jle	DoIO20				;Yes, retry & see if it goes away!
DoIO24:
	xchg ah,[esi]			;Load & set our media-change flag.
	mov	byte ptr [esi+12],0FFh  ;Make last-session LBA invalid.
	dec	ah					;Is media-change flag already set?
	jnz	DoIO25				;Yes, set carry flag and exit.
	mov	al,15				;Return "Invalid Media Change".
DoIO25:
	stc						;Set carry flag (error!).
DoIO26:
	pop	esi
	mov	edi,offset InBuf	;For audio, point to our buffer.
	ret
;
; Subroutine to convert "RedBook" MSF values to an LBA sector number.
;
ConvLBA:
	mov	ecx,eax		;Save "seconds" & "frames" in CX-reg.
	shr	eax,16		;Get "minute" value.
	cmp	ax,99		;Is "minute" value too large?
	ja	CnvLBAE		;Yes, return -1 error value.
	cmp	ch,60		;Is "second" value too large?
	ja	CnvLBAE		;Yes, return -1 error value.
	cmp	cl,75		;Is "frame" value too large?
	ja	CnvLBAE		;Yes, return -1 error value.
	xor	edx,edx		;Zero EDX-reg. for 32-bit math below.
	mov	dl,60		;Convert "minute" value to "seconds".
	mul	dl			;(Multiply by 60, obviously!).
	mov	dl,ch		;Add in "second" value.
	add	ax,dx
	mov	dl,75		;Convert "second" value to "frames".
	mul	edx			;(Multiply by 75 "frames"/second).
	mov	dl,150		;Subtract offset - "frame".
	sub	dl,cl		;("Adds" frame, "subtracts" offset).
	sub	eax,edx
	ret
CnvLBAE:
	or	eax,-1		;Too large!  Set -1 error value.
	ret
;
; Subroutine to clear our ATAPI "packet" area.
;
ZPacket:
	mov	dword ptr [Packet+0],ebx   ;Zero 1st 10 ATAPI packet bytes.
	mov	dword ptr [Packet+4],ebx   ;(Last 2 are unused "pad" bytes).
	mov	word ptr [Packet+8],bx
	ret
;
; Subroutine to validate the starting disk sector number.
;
ValSN:
	mov	eax,[esi].RL.RLSec  ;Get starting sector number.
ValSN1:
	mov	dl,[esi].RL.RLAM	;Get desired addressing mode.
	cmp	dl,001h				;HSG or RedBook addressing?
	ja	ValSNE				;Neither -- set carry and exit!
	jb	ValSN2				;HSG -- check sector limit.
	call ConvLBA			;Get RedBook starting sector.
	nop						;(Unused alignment "filler").
ValSN2:	
;	cmp	eax,00006DD3Ah
	cmp	eax,001000000h		;Is starting sector too big?
	jb	TestTOX				;No, all is well -- exit above.
ValSNE:
	pop	eax					;Error!  Discard our exit address.
	jmp	SectNF				;Post "sector not found" and exit.
;
; Subroutine to test for I-O timeouts.   At entry, the CL-reg. is
;   008h to test for a data-request, also.   At exit, the DX-reg.
;   points to the IDE primary-status register.   The AH-, SI- and
;   ES-regs. will be lost.
;
TestTO:
	xor	cl,cl			;Check for only controller-ready.
TestTO1:
	mov	ah,CMDTO		;Use 500-msec command timeout.
TestTO2:
	mov	esi,BIOSTMR
	add	ah,[esi]		;Set timeout limit in AH-reg.
TestTO3:
	cmp	ah,[esi]		;Has our I-O timed out?
	stc					;(If so, set carry flag).
	je	TestTOX			;Yes?  Exit with carry flag on.
	mov	dx,[IDEAd]		;Read IDE primary status.
	add	edx,7
	in	al,dx
	test al,BSY			;Is our controller still busy?
	jnz	TestTO3			;Yes, loop back and test again.
	or	cl,cl			;Are we also awaiting I-O data?
	jz	TestTOX			;No, just exit.
	test al,cl			;Is data-request (DRQ) also set?
	jz	TestTO3			;No, loop back and test again.
TestTOX:
	ret					;Exit -- carry indicates timeout.
;
; Subroutine to ensure UltraDMA is stopped and then select our CD-ROM
;   drive.   For some older chipsets, if UltraDMA is running, reading
;   an IDE register causes the chipset to "HANG"!!
;
StopDMA:
	mov	dx,[DMAAd]		;Get drive UltraDMA command address.
	test dl,006h		;Is any UltraDMA controller present?
	jnz	StopDM1			;No, select "master" or "slave" unit.
	and	dl,0FEh			;Mask out "DMA disabled" flag.
	in	al,dx			;Ensure any previous DMA is stopped!
	and	al,0FEh
	out	dx,al
StopDM1:
	mov	dx,[IDEAd]	;Point to IDE device-select register.
	add	edx,6
	mov	al,[IDESl]	;Select IDE "master" or "slave" unit.
	out	dx,al
	ret
;
; Subroutine to "swap" the 4 bytes of a a 32-bit value.
;
Swap32:
	xchg al,ah		;"Swap" original low-order bytes.
	rol	eax,16			;"Exchange" low- and high-order.
	xchg al,ah		;"Swap" ending low-order bytes.
Swap32X:
	ret

;
; DOS "Audio Seek" handler.   All DOS and IOCTL routines beyond this
;   point are DISMISSED by driver-init when the /AX switch is given.
;
ReqSeek:
	call ReadAST		;Read current "audio" status.
	call ZPacket		;Reset our ATAPI packet area.
	jc	RqSK1			;If status error, do DOS seek.
	mov	al,[edi+1]		;Get "audio" status flag.
	cmp	al,011h			;Is drive in "play audio" mode?
	je	RqSK2			;Yes, validate seek address.
	cmp	al,012h			;Is drive in "pause" mode?
	je	RqSK2			;Yes, validate seek address.
RqSK1:
	jmp	DOSSeek			;Use DOS seek routine above.
RqSK2:
	call ValSN			;Validate desired seek address.
	mov	edi,[AudAP]		;Point to audio-start address.
	cmp	eax,[edi+4]		;Is address past "play" area?
	ja	RqSK1			;Yes, do DOS seek above.
	mov	[edi],eax		;Update "audio" start address.
	call ConvMSF		;Set "packet" audio-start address.
	mov	dword ptr [PktLBA+1],eax
	mov	eax,[edi+4]		;Set "packet" audio-end address.
	call ConvMSF
	mov	dword ptr [PktLH],eax
	mov	byte ptr [Packet],047h	;Set "Play Audio" command.
	call DoIOCmd		;Start drive playing audio.
	jc	RqPLE			;If error, post code & exit.
	cmp	byte ptr [edi+1],011h	;Playing audio before?
	je	RqPLX			;Yes, post "busy" status and exit.
	call ZPacket		;Reset our ATAPI packet area.
	jmp	short ReqStop	;Go put drive back in "pause" mode.
;
; DOS "Play Audio" handler.
;
ReqPlay:
	cmp	dword ptr [esi].RL.RLSC,0 ;Is sector count zero?
	je	Swap32X					;Yes, just exit above.
	mov	edi,[AudAP]				;Point to audio-start addr.
	mov	byte ptr [Packet],047h	;Set "Play Audio" command.
	mov	eax,[esi].RL.RLAddr		;Validate starting address.
	call ValSN1
	mov	[edi],eax		;Save audio-start address.
	call ConvMSF		;Set MSF start address in packet.
	mov	dword ptr [PktLBA+1],eax
	mov	eax,[esi+18]	;Get play-audio "end" address.
	add	eax,[edi]
	mov	edx,00006DD39h	;Get maximum audio address.
	jc	ReqPL1			;If "end" WAY too big, use max.
	cmp	eax,edx			;Is "end" address past maximum?
	jbe	ReqPL2			;No, use "end" address as-is.
ReqPL1:
	mov	eax,edx			;Set "end" address to maximum.
ReqPL2:
	mov	[edi+4],eax		;Save audio-end address.
	call ConvMSF		;Set MSF end address in packet.
	mov	dword ptr [PktLH],eax
	call DoIOCmd		;Issue "Play Audio" command.
RqPLE:
	jc	ReqErr			;Error!  Post return code & exit.
RqPLX:
	jmp	RdAST3			;Go post "busy" status and exit.
;
; DOS "Stop Audio" handler.
;
ReqStop:
	mov	byte ptr [Packet],04Bh	;Set "Pause/Resume" cmd.
	jmp	DoIOCmd					;Go pause "audio", then exit.
;
; DOS "Resume Audio" handler.
;
ReqRsum:
	inc	byte ptr [PktLn+1] 		;Set "Resume" flag for above.
	call ReqStop				;Issue "Pause/Resume" command.
	jmp	short RqPLE				;Go exit through "ReqPlay" above.
;
; IOCTL Input "Current Head Location" handler.
;
ReqCHL:
	mov	dword ptr [Packet],001400042h   ;Set command bytes.
	mov	al,16			;Set input byte count of 16.
	call RdAST2			;Issue "Read Subchannel" request.
	jc	RqPLE			;If error, post return code & exit.
	mov	[esi+1],bl		;Return "HSG" addressing mode.
	mov	eax,[edi+8]		;Return "swapped" head location.
	call Swap32
	mov	[esi+2],eax
	jmp	short RqATIX	;Go post "busy" status and exit.
;
; IOCTL Input "Volume Size" handler.
;
ReqVS:
	mov	byte ptr [Packet],025h  ;Set "Read Capacity" code.
	mov	al,008h			;Get 8 byte data-transfer length.
	call DoBufIn		;Issue "Read Capacity" command.
	jc	RqPLE			;If error, post return code & exit.
	mov	eax,[edi]		;Set "swapped" size in IOCTL packet.
	call Swap32
	mov	[esi+1],eax
	jmp	short RqATIX	;Go post "busy" status and exit.
;
; IOCTL Input "Audio Disk Info" handler.
;
ReqADI:
	mov	al,0AAh			;Specify "lead-out" session number.
	call ReadTOC		;Read disk table-of-contents (TOC).
	jc	RqASIE			;If error, post return code & exit.
	mov	[esi+3],eax		;Set "lead out" LBA addr. in IOCTL.
	mov	ax,[edi+2]		;Set first & last tracks in IOCTL.
	mov	[esi+1],ax
	jmp	short RqATIX	;Go post "busy" status and exit.
;
; IOCTL Input "Audio Track Info" handler.
;
ReqATI:
	mov	al,[esi+1]		;Specify desired session (track) no.
	call ReadTOC		;Read disk table-of-contents (TOC).
	jc	RqASIE			;If error, post return code & exit.
	mov	[esi+2],eax		;Set track LBA address in IOCTL.
	mov	al,[edi+5]
	shl	al,4
	mov	[esi+6],al
RqATIX:
	jmp	ReadAST			;Go post "busy" status and exit.
;
; IOCTL Input "Audio Q-Channel Info" handler.
;
ReqAQI:
	mov	ax,04010h		;Set "data in", use 16-byte count.
	call RdAST1			;Read current "audio" status.
	jc	RqASIE			;If error, post return code & exit.
	mov	eax,[edi+5]		;Set ctrl/track/index in IOCTL.
	mov	[esi+1],eax
	mov	eax,[edi+13]	;Set time-on-track in IOCTL.
	mov	[esi+4],eax
	mov	edx,[edi+9]		;Get time-on-disk & clear high
	shl	edx,8			;  order time-on-track in IOCTL.
	jmp	short RqASI4	;Go set value in IOCTL and exit.
;
; IOCTL Input "Audio Status Info" handler.
;
ReqASI:
	mov	ax,04010h		;Set "data in", use 16-byte count.
	call RdAST1			;Read current "audio" status.
RqASIE:
	jc	ReqErr			;If error, post return code & exit.
	mov	[esi+1],bx		;Reset audio "paused" flag.
	xor	eax,eax			;Reset starting audio address.
	xor	edx,edx			;Reset ending audio address.
	cmp	byte ptr [edi+1],011h  ;Is drive now "playing" audio?
	jne	RqASI1		  	;No, check for audio "pause".
	mov	edi,[AudAP]		;Point to drive's audio data.
	mov	eax,[edi]		;Get current audio "start" addr.
	jmp	short RqASI2	;Go get current audio "end" addr.
RqASI1:
	cmp	byte ptr [edi+1],012h	;Is drive now in audio "pause"?
	jne	RqASI3					;No, return "null" addresses.
	inc	byte ptr [esi+1]		;Set audio "paused" flag.
	mov	eax,[edi+8]				;Convert time-on-disk to LBA addr.
	call Swap32
	call ConvLBA
	mov	edi,[AudAP]		;Point to drive's audio data.
RqASI2:
	mov	edx,[edi+4]		;Get current audio "end" address.
RqASI3:
	mov	[esi+3],eax	;Set audio "start" addr. in IOCTL.
RqASI4:
	mov	[esi+7],edx	;Set audio "end" address in IOCTL.
	ret
;
; Subroutine to read the current "audio" status and disk address.
;
ReadAST:
	call ZPacket		;Status only -- reset ATAPI packet.
	mov	ax,00004h		;Clear "data in", use 4-byte count.
RdAST1:
	mov	dword ptr [Packet],001000242h	;Set command bytes.
	mov	byte ptr [PktLBA],ah  			;Set "data in" flag (RdAST1 only).
RdAST2:
	call DoBufIO		;Issue "Read Subchannel" command.
	jc	RdASTX			;If error, exit immediately.
	cmp	byte ptr [edi+1],011h  ;Is a "play audio" in progress?
	jnz	RdTOC1			;No, clear carry flag and exit.
RdAST3:
	push esi			;Save SI- and ES-regs.
	mov	esi,[dwRequest]	;Reload DOS request-packet addr.
	or	word ptr [esi].RP.RPStat,RPBUSY  ;Set "busy" status bit.
	pop	esi
RdASTX:
	ret
;
; Subroutine to read disk "Table of Contents" (TOC) values.
;
ReadTOC:
	mov	word ptr [Packet],00243h;Set TOC and MSF bytes.
	mov	[PktLH],al				;Set desired "session" number.
	mov	al,12					;Get 12-byte "allocation" count.
	call DoBufIO				;Issue "Read Table of Contents" cmd.
	jc	RdTOCX					;If error, exit immediately.
	mov	eax,[edi+8]				;Return "swapped" starting address.
	call Swap32
RdTOC1:
	clc			;Clear carry flag (no error).
RdTOCX:
	ret
    align 4
;
; Subroutine to convert an LBA sector number to "RedBook" MSF format.
;
ConvMSF:
	add	eax,150		;Add in offset.
	push eax		;Get address in DX:AX-regs.
	pop	ax
	pop	dx
	mov	cx,75		;Divide by 75 "frames"/second.
	div	cx
	shl	eax,16		;Set "frames" remainder in upper EAX.
	mov	al,dl
	ror	eax,16
	mov	cl,60		;Divide quotient by 60 seconds/min.
	div	cl
	ret				;Exit -- EAX-reg. contains MSF value.

;
; Driver Initialization Routine.   Note that this routine runs on
;   the DOS stack.   All logic past this point becomes our local-
;   stack or is DISMISSED, after initialization is completed.
;
I_Init:
	push eax		;Save needed 16-bit CPU registers.
	push ebx
	push edx
	xor	ebx,ebx		;Zero BX-reg. for relative commands.
	cld				;Ensure FORWARD "string" commands!
	mov	esi,[dwRequest]	;Point to DOS request packet.
	cmp	byte ptr [esi].RP.RPOp,0 ;Is this an "Init" packet?
	je	I_CPU		;Yes, test for minimum 80386 CPU.
	jmp	I_BadP		;Go post errors and exit quick!
I_CPU:
	pushad			;80386+ -- save all 32-bit registers.
	mov	edx,offset XCMsg	;Display driver "title" message.
	call I_Dsply
;   mov	esi,[dwRequest]	;Reload DOS request-packet pointer.
;	les	si,es:[esi].RP.RPCL	;Point to command line that loaded us.
	mov	esi,[dwCmdLine]	;Point to command line that loaded us.
I_NxtC:
	mov	al,[esi]	;Get next command-line byte.
	inc	esi			;Bump pointer past this byte.
	cmp	al,0		;Is byte the command-line terminator?
	je	I_TermJ		;Yes, go test for UltraDMA controller.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je	I_TermJ		;Yes, go test for UltraDMA controller.
	cmp	al,CR		;Is byte an ASCII carriage-return?
I_TermJ:
	je	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,'-'		;Is byte a dash?
	je	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'/'		;Is byte a slash?
	jne	I_NxtC		;No, check next command-line byte.
I_NxtS:
	mov	ax,[esi]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st lower-case bit (020h).
	cmp	al,'U'		;Is switch byte a "U" or "u"?
	jne	I_ChkA		;No, go see if byte is "A" or "a".
	inc	esi			;Bump pointer past "UltraDMA" switch.
	and	ah,0DFh		;Mask out 2nd lower-case bit (020h).
	mov	cl,0F0h		;Get "UX" switch value.
	cmp	ah,'X'		;Is following byte an "X" or "x"?
	je	I_SetUX		;Yes, update "UFX" switch.
	mov	cl,0F2h		;Get "UF" switch value.
	cmp	ah,'F'		;Is following byte an "F" or "f"?
	jne	I_NxtC		;No, see if byte is a terminator.
I_SetUX:
	mov	[UFXSw],cl	;Update "UFX" switch for below.
	inc	esi			;Bump pointer past "F" or "X".
I_ChkA:
	cmp	al,'A'		;Is switch byte an "A" or "a"?
	jne	I_ChkL		;No, go see if byte is "L" or "l".
	inc	esi			;Bump pointer past "Audio" switch.
	and	ah,0DFh		;Mask out 2nd lower-case bit (020h).
	cmp	ah,'X'		;Is following byte an "X" or "x"?
	jne	I_NxtC		;No, see if byte is a terminator.

	mov	eax,offset UnSupp	;Disable all unwanted dispatches.
	mov	[@RqPlay],eax
	mov	[@RqStop],eax
	mov	[@RqRsum],eax
	mov	[@RqCHL],eax
	mov	[@RqADI],eax
	mov	[@RqATI],eax
	mov	[@RqAQI],eax
	mov	[@RqASI],eax
	mov	eax,offset DOSSeek	;Do only LBA-address DOS seeks.
	mov	[@RqPref],eax
	mov	[@RqSeek],eax
	mov	al,004h		;Have "Device Status" declare
	mov	byte ptr ds:[@Status],al	;  we handle DATA reads only,
	db	0B0h		;  and have it NOT update the
	ret				;  IOCTL "busy" flag & return
	mov	byte ptr ds:[@RqDSX],al	;  ["ReadAST" gets DISMISSED]!
	inc	esi			;Bump pointer past "X" or "x".
I_ChkL:
	cmp	al,'L'		;Is switch byte an "L" or "l"?
	jne	I_ChkM		;No, go see if byte is "M" or "m".
	mov	byte ptr [@DMALmt],009h  ;Set 640K "DMA limit" above.
	inc	esi			;Bump pointer past "limit" switch.
I_ChkM:
	cmp	al,'M'		;Is this byte an "M" or "m"?
	jne	I_ChkP		;No, go see if byte is "P" or "p".
	inc	esi			;Bump pointer past "mode" switch.
	cmp	ah,'2'		;Is following byte above a two?
	ja	I_NxtCJ		;Yes, see if byte is a terminator.
	sub	ah,'0'		;Is following byte below a zero?
	jb	I_NxtCJ		;Yes, see if byte is a terminator.
	mov	[MaxUM],ah	;Set maximum UltraDMA "mode" above.
	inc	esi			;Bump pointer past "mode" value.
I_ChkP:
	cmp	al,'P'		;Is switch byte a "P" or "p"?
	jne	I_ChkS		;No, go see if byte is "S" or "s".
	mov	edi,offset ScanP	;Point to primary-channel values.
	jmp	short I_ChkMS	;Go check for "M" or "S" next.
I_ChkS:
	cmp	al,'S'		;Is switch byte an "S" or "s"?
	jne	I_ChkD		;No, check for "D" or "d".
	mov	edi,offset ScanS	;Point to secondary-channel values.
I_ChkMS:
	inc	esi			;Bump pointer past "channel" switch.
	and	ah,0DFh		;Mask out 2nd lower-case bit (020h).
	cmp	ah,'M'		;Is following byte an "M" or "m"?
	je	I_SetHW		;Yes, set desired hardware values.
	cmp	ah,'S'		;Is following byte an "S" or "s"?
	jne	I_NxtCJ		;No, see if byte is a terminator.
	add	edi,4		;Point to channel "slave" values.
I_SetHW:
	inc	esi			;Bump pointer past master/slave byte.
	or	[ScanX],-1  ;Set "no scan" flag.
	xor	edx,edx		;Get this device's hardware values.
	xchg edx,[edi]
	or	edx,edx		;Have we already used these values?
	jz	I_NxtCJ		;Yes, IGNORE duplicate switches!
	mov	edi,[UTblP]	;Get current unit-table pointer.
	cmp	edi,offset UTblEnd	;Have we already set up all units?
	je	I_NxtCJ		;Yes, IGNORE any more switches!
	mov	[edi+2],edx	;Set parameters in unit table.
	add	[UTblP],20  ;Bump to next unit table.
I_NxtCJ:
	jmp	I_NxtC		;Go check next command byte.
I_ChkD:
	cmp	al,'D'		;Is switch byte a "D" or "d"?
	jne	I_NxtCJ		;No, see if byte is a terminator.
	inc	esi			;Bump pointer past "device" switch.
	cmp	ah,':'		;Is following byte a colon?
	jne	I_NxtCJ		;No, see if byte is a terminator.
	inc	esi			;Bump pointer past colon.
	mov	edi,[dwBase]	;Blank out device name.
    add edi,10
    lea edx,[edi+8]
	mov	eax,"    "
	mov	[edi+0],eax
	mov	[edi+4],eax
I_NameB:
	mov	al,[esi]	;Get next device-name byte.
	cmp	al,TAB		;Is byte a "tab"?
	je	I_NxtCJ		;Yes, handle above, "name" has ended!
	cmp	al,' '		;Is byte a space?
	je	I_NxtCJ		;Yes, handle above, "name" has ended!
	cmp	al,'/'		;Is byte a slash?
	je	I_NxtCJ		;Yes, handle above, "name" has ended!
	cmp	al,0		;Is byte the command-line terminator?
	je	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,'a'		;Ensure letters are upper-case.
	jc	I_Name2
	cmp	al,'z'
	ja	I_Name2
	and	al,0DFh
I_Name2:
	cmp	al,'!'		;Is this byte an exclamation point?
	jz	I_Name3		;Yes, store it in device name.
	cmp	al,'#'		;Is byte below a pound-sign?
	jb	I_Name4		;Yes, Invalid!  Blank first byte.
	cmp	al,')'		;Is byte a right-parenthesis or less?
	jbe	I_Name3		;Yes, store it in device name.
	cmp	al,'-'		;Is byte a dash?
	jz	I_Name3		;Yes, store it in device name.
	cmp	al,'0'		;Is byte below a zero?
	jb	I_Name4		;Yes, invalid!  Blank first byte.
	cmp	al,'9'		;Is byte a nine or less?
	jbe	I_Name3		;Yes, store it in device name.
	cmp	al,'@'		;Is byte below an "at sign"?
	jb	I_Name4		;Yes, invalid!  Blank first byte.
	cmp	al,'Z'		;Is byte a "Z" or less?
	jbe	I_Name3		;Yes, store it in device name.
	cmp	al,'^'		;Is byte below a carat?
	jb	I_Name4		;Yes, invalid!  Blank first byte.
	cmp	al,'~'		;Is byte above a tilde?
	ja	I_Name4		;Yes, invalid!  Blank first byte.
	cmp	al,'|'		;Is byte an "or" symbol?
	je	I_Name4		;Yes, invalid!  Blank first byte.
I_Name3:
	mov	[edi],al			;Store next byte in device name.
	inc	esi					;Bump command-line pointer.
	inc	edi					;Bump device-name pointer.
	cmp	edi,edx				;Have we stored 8 device-name bytes?
	jb	I_NameB				;No, go get next byte.
	jmp	short I_Name5		;Go get next byte & check terminator.
I_Name4:
	mov	al,' '				;Invalid!  Blank first "name" byte,
    mov ecx,[dwBase]
	mov	byte ptr [ecx+10],' '	;Invalid!  Blank first byte.
I_Name5:
	jmp	I_NxtC				;Go get next command byte.
I_Term:
	xor	edi,edi				;UltraDMA controller check:  Request
	mov	al,001h				;  PCI BIOS I.D. (should be "PCI ").
    mov [ebp].Client_Reg_Struc.Client_EDI, 0
	call I_In1A

    mov edi, [ebp].Client_Reg_Struc.Client_EDI
    mov edx, [ebp].Client_Reg_Struc.Client_EDX
	cmp	edx," ICP"			;Do we have a V2.0C or newer PCI BIOS?
	jne	I_ChkNm				;No, check for valid driver name.
	mov	esi,offset ClCodes	;Point to interface byte table.
I_FindC:
	cmp	esi,offset ClCEnd	;More interface bytes to check?
	jae	I_ChkNm				;No, check for valid driver name.
	mov	ecx,000010100h		;Find class 1 storage, subclass 1 IDE.
	lodsb					;Use next class-code "interface" byte.
	mov	cl,al
	push esi				;Save class-code table pointer.
	mov	al,003h				;Inquire about an UltraDMA controller.
    mov [ebp].Client_Reg_Struc.Client_ESI, 0
    mov [ebp].Client_Reg_Struc.Client_ECX, ecx
	call I_In1A
	pop	esi					;Reload class-code table pointer.
	jc	I_FindC				;Not found -- test for more I/F bytes.
    mov ebx,[ebp].Client_Reg_Struc.Client_EBX
	push ebx				;Save PCI bus/device/function.
	mov	di,4				;Get low-order PCI command byte.
    mov [ebp].Client_Reg_Struc.Client_EDI, edi
    mov [ebp].Client_Reg_Struc.Client_EBX, ebx
	call I_PCID
    mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	pop	ebx					;Reload PCI bus/device/function.
	not	cl					;Mask Bus-Master and I-O Space bits.
	and	cl,005h				;Is this how our controller is set up?
	jnz	I_ChkNm				;No, check for valid driver name.
	push ebx				;Save PCI bus/device/function.
	xor	edi,edi				;Get Vendor and Device I.D.
    mov [ebp].Client_Reg_Struc.Client_EDI, edi
    mov [ebp].Client_Reg_Struc.Client_EBX, ebx
	call I_PCID
    mov ecx,[ebp].Client_Reg_Struc.Client_ECX
	pop	ebx					;Reload PCI bus/device/function.
	push ecx				;Save Vendor and Device I.D.
	mov	di,32				;Get PCI base address (register 4).
    mov [ebp].Client_Reg_Struc.Client_EDI, edi
    mov [ebp].Client_Reg_Struc.Client_EBX, ebx
	call I_PCID

    mov eax,[ebp].Client_Reg_Struc.Client_ECX ;get our DMA controller address.
	and	al,0FCh
	mov	[PrDMA],ax
	mov	esi,offset CtlrAdr	;Set controller address in message.
	call I_Hex
	mov	esi,offset CtlrID	;Set Vendor & Device I.D. in message.
	pop	eax
	call I_Hex
    shr eax,16
	call I_Hex
I_ChkNm:
	xor	ebx,ebx					;Zero BX-reg. for relative commands.
    mov ecx,[dwBase]
	cmp	byte ptr [ecx+10],' '	;Is driver "name" valid?
	jne	I_SetNm					;Yes, display driver name.
	mov	dword ptr [ecx+10],"RDCX"  ;Set our default "name".
	mov	dword ptr [ecx+10+4],"$$MO"
I_SetNm:
	mov	eax,[ecx+10+0]
	mov	dword ptr [DvrMsg1+0],eax
	mov	eax,[ecx+10+4]
	mov	dword ptr [DvrMsg1+4],eax
I_ScanN:
	mov	edx,offset DvrMsg	;Display our driver "name".
	call I_Dsply
    
	mov	ax,[wBaseSeg]		;Set our code segment in VDS block.
	mov	[VDSSg],ax

    mov eax, offset startcode
	test byte ptr ds:[VDSFLAG],020h  ;Are "VDS services" active?
	jz	I_SetAd				;No, set 20-bit virtual addresses.
    mov esi, eax
    mov ecx, 1000h
    xor dx,dx
    VxDCall VDMAD_Lock_DMA_Region
    mov [IOAdr], edx
	mov	eax, edx		   	;Get 32-bit starting driver addr.
I_SetAd:
	add	[PRDAd],eax			;Set relocated 32-bit PRD address.
    
	cmp	byte ptr [UFXSw],0F0h	;Did user disable UltraDMA?
	je	I_LinkX					;Yes, go try "linking" with XDMA.
	cmp	byte ptr [@DMALmt],-1	;Is UltraDMA limited to < 640K?
	je	I_LinkX					;No, go try "linking" with XDMA.
	mov	edx,offset LEMsg		;Point to "/L Invalid" message.
	cmp	word ptr [IOAdr+2],009h ;Are we loaded high?
	ja	I_InitE					;Yes?  Display message and exit!
I_LinkX:
	xor eax, eax
	mov	edi, eax
I_OurUC:
	mov	edx,offset CRMsg		;Display ending CR/LF message.
	call I_Dsply
	test byte ptr [PrDMA],007h  ;UltraDMA controller found?
	jnz	I_Spcfy					;No, see if drives were specified.
	mov	edx,offset CtlrMsg		;Display UltraDMA controller data.
	call I_Dsply
I_Spcfy:
	mov	eax,offset UnitTbl		;Reset our unit-table pointer.
	mov	[UTblP],eax
I_ScanU:
	mov	ax,[PrDMA]				;Set current UltraDMA command addr.
	mov	[DMAAd],ax
	mov	esi,[UTblP]			;Get current unit-table pointer.
	mov	edi,[ScanX]			;Get current parameters index.
	cmp	edi,-1				;Are we "scanning" for drives?
	je	I_GetPV				;No, get unit-table parameters.
	cmp	edi,offset ScanE	;Any more IDE units to check?
	je	I_ChkCD				;No, check for any drives to use.
	lea	esi,[edi-2]			;Point to IDE unit parameters.
	add	edi,4				;Update parameter-table index.
	mov	[ScanX],edi
I_GetPV:
	mov	eax,[esi+2]			;Get unit's IDE address, etc.
	cmp	ax,-1				;Not scanning & unit table "empty"?
I_ChkCD:
	je	I_AnyCD				;Yes, check for any drives to use.
	mov	dword ptr [IDEAd],eax;Set this unit's parameters.
	call I_ValDV			;Validate device as an ATAPI CD-ROM.
	jnc	I_AnySy				;If no error, we can USE this drive!
	cmp	[ScanX],-1			;"Scanning" for drives?
	je	I_AnySy				;No, must show unit & error.
	cmp	dword ptr [IEMsg],offset SEMsg	;UltraDMA "Set-Mode" error?
	jne	I_ScanU				;No, ignore & try next unit.
I_AnySy:
	cmp	[SyncX],bl			;Synchronizing with XDMA?
	je	I_NoDMA				;Yes, see if user disabled all DMA.
	mov	[SyncF],bl			;Disable run-time "sync" flags.
I_NoDMA:
	cmp	byte ptr [UFXSw],0F0h	;Was the /UX switch given?
	jne	I_DspDr				;No, display all drive data.
	or	byte ptr [DMAAd],001h	;Post drive "DMA disabled".
I_DspDr:
	mov	edx,offset UnitMsg	;Display "Unit n:" message.
	call I_Dsply
	mov	edx,offset PriMsg	;Point to "Primary" message.
	cmp	word ptr [IDEAd],PCHADDR  ;Primary-channel drive?
	je	I_PSMsg				;Yes, display "Primary" message.
	mov	edx,offset SecMsg	;Point to "Secondary" message.
	or	byte ptr [DMAAd],008h  ;Use secondary DMA channel.
I_PSMsg:
	call I_Dsply			;Display our CD-ROM's IDE channel.
	mov	edx,offset MstMsg	;Point to "Master" message.
	cmp	byte ptr [IDESl],SSELECT  ;Is our drive a "slave"?
	jnz	I_MSMsg				;No, display "Master".
	mov	edx,offset SlvMsg	;Point to "Slave" message.
I_MSMsg:
	call I_Dsply			;Display "Master" or "Slave".
	cmp	[IEMsg],ebx			;Did any validation ERROR occur?
	jz	I_ScnVN				;No, scan "vendor name" for data.
;	call I_EndSy			;End XDMA "synchronization" if needed.
	mov	edx,[IEMsg]			;Get init error-message pointer.
	jmp	I_InitE				;Go display error message and exit.
I_ScnVN:
	mov	edi,offset XCMsg+40	;Point to CD-ROM "vendor name" end.
I_ScnV1:
	mov	byte ptr [edi],'$'	;Set message terminator after name.
	dec	edi					;Point to previous name byte.
	cmp	byte ptr [edi],' '	;Is this byte a space?
	je	I_ScnV1				;Yes, keep scanning for a non-space.
	cmp	byte ptr [XCMsg],'$'  ;Is CD-ROM "name" all spaces?
	je	I_ModeM				;Yes, no need to display it!
	mov	edx,offset ComMsg	;Display comma/space before name.
	call I_Dsply
	mov	edx,offset XCMsg	;Display manufacturer CD-ROM "name".
	call I_Dsply
I_ModeM:
	mov	edx,offset PIOMsg	;Point to "PIO mode" message.
	test byte ptr [DMAAd],007h  ;Will drive be using UltraDMA?
	jnz	I_MsEnd				;No, display "PIO mode" message.
	mov	edx,offset UDMsg	;Point to "ATA-xx" message.
I_MsEnd:
	call I_Dsply			;Display drive's operating "mode".
	mov	edx,offset CRMsg	;Display terminating CR/LF/$.
	call I_Dsply
	mov	esi,[UTblP]			;Update all unit-table parameters.
	mov	eax,dword ptr [DMAAd];(If "scanning", table parameters
	mov	[esi],eax			;  are NOT set from our switches!).
	mov	ax,word ptr [IDESl]
	mov	[esi+4],ax
	add	esi,20				;Update unit-table pointer.
	mov	[UTblP],esi

;   inc	byte ptr [Units]	;Bump number of active units.
    mov eax, [dwBase]
	inc byte ptr [eax+21]	;units

	inc	byte ptr [UMsgNo]	;Bump display unit number.
	cmp	esi,offset UTblEnd	;Can we install another drive?
	jb	I_ScanU				;Yes, loop back & check for more.
I_AnyCD:
;	cmp	[Units],bl			;Do we have any CD-ROM drives to use?
	mov edx, [dwBase]
	cmp	[edx+21],bl			;Do we have any CD-ROM drives to use?
    
	ja	I_ClrSt				;Yes, success -- go zero local-stack.
	mov	edx,offset NDMsg	;NOT GOOD!  Point to "No CD-ROM" msg.
I_InitE:
I_DsplE:
	call I_Dsply			;Display desired error message.
	popad					;Reload all 32-bit CPU registers.
	mov	edx,offset Suffix	;Display error message suffix.
I_Quit:
	call I_Dsply
I_BadP:
	xor	eax,eax				;Get "null" length & error flags.
	mov	dx,RPDON+RPERR
	jmp	short I_Exit		;Go set "init" packet values & exit.
I_ClrSt:
	popad					;Reload all 32-bit CPU registers.
	xor	eax,eax				;Load & reset driver length.
	mov [VDSLn],eax
	mov	dx,RPDON			;Get initialization "success" code.
    mov ax,12h+4+sizermcode+5
I_Exit:
	mov	ebx,[dwRequest]		;Set result values in "init" packet.
	mov	word ptr [ebx].RP.RPSize,ax
	mov	[ebx].RP.RPStat,dx
	xor	eax,eax	;Reset returned "units found".
	mov	[ebx].RP.RPUnit,al
	pop	edx
	pop	ebx
	pop	eax
	ret
;
; Subroutine to "validate" an IDE unit as an ATAPI CD-ROM drive.
;
I_ValDV:
	mov	[IEMsg],ebx		;Reset our error-message pointer.
	call StopDMA		;Stop previous DMA & select drive.
	call TestTO			;Await controller-ready.
	mov	ecx,offset TOMsg;Get "select timeout" message ptr.
	jc	I_Val7			;If timeout, go post pointer & exit.
	mov	al,0A1h			;Issue "Identify Packet Device" cmd.
	out	dx,al
	call TestTO			;Await controller-ready.
	mov	ecx,offset IDMsg;Get "Identify" message pointer.
	jc	I_Val7			;If timeout, go post pointer & exit.
	test al,DRQ			;Did we also get a data-request?
	jz	I_Val6			;No, post "not ATAPI" ptr. & exit.
	sub	edx,7			;Point back to IDE data register.
	in	ax,dx			;Read I.D. word 0, main device flags.
	and	ax,0DF03h		;Mask off flags for an ATAPI CD-ROM.
	xchg eax,esi		;Save main device flags in SI-reg.
	mov	ecx,26			;Skip I.D. words 1-26 (unimportant).
I_Val1:
	in	ax,dx
	loop I_Val1
	mov	edi,offset XCMsg;Point to drive "name" input buffer.
	mov	cl,20			;Read & swap words 27-46 into buffer.
I_Val2:
	in ax,dx			;(Manufacturer "name" of this drive).
	xchg ah,al
	stosw
	loop I_Val2
	mov	cl,7			;Skip I.D. words 47-52 (unimportant)
I_Val3:
	in	ax,dx			;  and read I.D. word 53 into AX-reg.
	loop I_Val3
	mov	[UFlag],al		;Save UltraDMA "valid" flags.
	mov	cl,35			;Skip I.D. words 54-87 (unimportant)
I_Val4:
	in	ax,dx			;  and read I.D. word 88 into AX-reg.
	loop I_Val4
	and	ah,007h			;Mask & save UltraDMA "mode" value.
	mov	[UMode],ah
	mov	cl,167			;Skip all remaining I.D. data.
I_Val5:
	in	ax,dx
	loop I_Val5
	cmp	si,08500h		;Do device flags say "ATAPI CD-ROM"?
	je	I_Val9			;Yes, see about UltraDMA use.
I_Val6:
	mov	ecx,offset NCMsg;Get "not an ATAPI CD-ROM" msg. ptr.
I_Val7:
	mov	[IEMsg],ecx		;Post desired error-message pointer.
	stc					;Set carry flag on (error!).
I_Val8:
	ret
I_Val9:
	test byte ptr [DMAAd],007h	;Will we be using UltraDMA?
	jnz	I_Val8				;No, go exit above.
	test byte ptr [UFlag],004h	;Valid UltraDMA "mode" bits?
	jz	I_Val10				;No, reset UltraDMA address.
	mov	al,[UMode]			;Get UltraDMA "mode" value.
	or	al,al				;Can drive do mode 0 minimum?
	jnz	I_Val11				;Yes, set up UltraDMA mode 0.
I_Val10:
	or	byte ptr [DMAAd],001h  ;Post drive "DMA disabled".
	ret						;Exit -- must use "PIO mode"!
I_Val11:
	xor	cl,cl			;Get UltraDMA mode 0 values.
	mov	dx,"61"
	cmp	cl,[MaxUM]		;Are we limited to mode 0?
	je	I_Val12			;Yes, set mode 0 now.
	shr	al,1			;Will our drive do mode 1?
	jz	I_Val12			;No, set mode 0 now.
	inc	ecx				;Get UltraDMA mode 1 values.
	mov	dx,"52"
	cmp	cl,[MaxUM]		;Are we limited to mode 1?
	je	I_Val12			;Yes, set mode 1 now.
	shr	al,1			;Will our drive do mode 2?
	jz	I_Val12			;No, set mode 1 now.
	inc	ecx				;Get UltraDMA mode 2 values.
	mov	dx,"33"
I_Val12:
	mov	word ptr [UDMode],dx	;Set display UltraDMA mode value.
if ?SETUMODE
	mov	dx,[IDEAd]		;Point to IDE "features" register.
	inc	edx
	mov	al,SETM			;Set mode-select subcode.
	out	dx,al
	inc	edx				;Point to IDE sector-count register.
	xchg eax,ecx		;Set desired UltraDMA mode.
	or	al,040h
	out	dx,al
	add	edx,4			;Point to IDE drive-select register.
	mov	al,[IDESl]		;Select our desired CD-ROM drive.
	inc	edx				;Point to IDE command register.
	mov	al,SETF			;Issue "set features" command.
	out	dx,al
	call TestTO			;Await controller-ready.
	mov	ecx,offset SEMsg;Get "Set Mode error" message ptr.
	jc I_Val7			;If timeout, go post pointer & exit.
endif    
	ret
;
; Subroutines to do all initialization "external" calls.
;
I_PCID:
	mov	al,00Ah			;Set "PCI doubleword" request code.
I_In1A:                 ;<--- entry al = 1, 3
	mov	ah,0B1h			;PCI BIOS -- execute desired request.
    mov word ptr [ebp].Client_Reg_Struc.Client_EAX, ax
	VMMCall Begin_Nest_Exec
    mov eax, 1Ah
    VMMCall Exec_Int
	VMMCall End_Nest_Exec
    mov ah,byte ptr [ebp].Client_Reg_Struc.Client_EFlags
    sahf
    mov eax,[ebp].Client_Reg_Struc.Client_EAX
    ret

I_Dsply proc
	push esi
    mov esi, edx
	VMMCall Begin_Nest_Exec
@@nextitem:
    lodsb
    cmp al,'$'
    jz  I_Dsply_done
	mov byte ptr [ebp].Client_Reg_Struc.Client_EDX, al
    mov byte ptr [ebp].Client_Reg_Struc.Client_EAX+1, 2
    mov eax, 21h
    VMMCall Exec_Int
    jmp @@nextitem
I_Dsply_done:
	VMMCall End_Nest_Exec
    pop esi
	ret
I_Dsply endp
;
; Subroutine to convert a 4-digit hex number to ASCII for messages.
;   At entry, the number is in the AX-reg., and the message pointer
;   is in the ESI-reg.   At exit, the ESI-reg. is updated and the ECX-
;   reg. is zero.
;
I_Hex:
	mov	ecx,4		;Set 4-digit count.
I_HexA:
	rol	ax,4		;Get next hex digit in low-order (use AX here!).
	push eax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:
	add	al,030h		;Convert digit to ASCII.
	mov	[esi],al		;Store next digit in message.
	inc	esi			;Bump message pointer.
	pop	eax			;Reload remaining digits.
	loop I_HexA		;If more digits to go, loop back.
	ret
;
; Initialization PCI Class-Code "Interface" Bytes.   The 0BAh and 0B0h
; bytes handle ALi M5529 chips.   MANY THANKS to David Muller for this
; VALUABLE addition!
;
ClCodes	db	0FAh,0F0h,08Ah,080h,0BAh,0B0h
ClCEnd	equ	$
;
; Initialization IDE Parameter-Value Table.
;
ScanP	dw	CDATA		;Primary-master   drive parameters.
	db	0A0h,028h
	dw	CDATA			;Primary-slave    drive parameters.
	db	0B0h,028h
ScanS	dw	CDATA-080h	;Secondary-master drive parameters.
	db	0A0h,050h
	dw	CDATA-080h		;Secondary-slave  drive parameters.
	db	0B0h,050h
ScanE	equ	$			;(End of IDE parameter table).

ifdef language
	include XCDMSGS.TXT		;Include "user language" messages.
else
	include XCDMSGS.ENG		;Include default English messages.
endif

DllMain proc stdcall hModule:dword, dwReason:dword, dwRes:dword

	.if (dwReason == 1)

    	mov esi, dwRes
        test [esi].JLCOMM.wFlags, JLF_DRIVER
        jz error
        movzx ecx,[esi].JLCOMM.wLdrCS
        mov wBaseSeg,cx
        shl ecx, 4
        mov dwBase, ecx
        mov eax, 16h
        add eax, ecx
        mov pRequest, eax
        mov eax,[esi].JLCOMM.lpCmdLine
        mov dwCmdLine, eax
        mov eax,[esi].JLCOMM.lpRequest
        mov dwRequest, eax

        mov esi, offset DevInt
        xor edx, edx
        VMMCall Allocate_V86_Call_Back
        jc error
        mov dword ptr ds:[jmppm+1], eax

        mov edi, dwBase
        mov word ptr [edi+4],0C800h	;driver attributes
        mov word ptr [edi+6],16h+4
        mov word ptr [edi+8],16h+4+sizermcode
        mov word ptr [edi+20],0
        
        add edi, 16h+4
        mov esi, offset rmcode
        mov ecx, sizermcode+5
        rep movsb

        VMMCall Get_Cur_VM_Handle

;--- set EBP to the client pointer before calling I_Init

		push ebp
        mov ebp,[ebx].cb_s.CB_Client_Pointer
    	call I_Init
        pop ebp
        
        mov eax, 1
    .endif
    ret
error:
	xor eax, eax
    ret

DllMain endp

rmcode label byte
	db 2Eh, 89h, 1eh, 16h, 0	;mov cs:[12],bx
	db 2Eh,	8Ch, 06h, 18h, 0	;mov cs:[14],es
	db 0CBh			            ;retf
sizermcode equ $ - offset rmcode	
jmppm db 0EAh,0,0,0,0 			;jmp 0000:0000

ResEnd label byte

	end DllMain
  
