        
; Jemm's memory pool implementation
; originally written by Michael Devore
; this is Public Domain code

; to be assembled with MASM 6.1 or TASM 4.1/5.0

		.486P

		include jemm.inc		;common declarations
		include jemm32.inc		;declarations for Jemm32
		include debug.inc

ife ?MASM
		LOCALS
else
		option proc:private
  		option dotname
endif

if ?FLAT
		.model FLAT
endif

;--- equates

;--- assembly time constants

if ?INTEGRATED
?FREEXMS	equ 1		; std=1, 0 might work
?EXPANDFIRST equ 0		; std=0, 1 might work
else
?FREEXMS	equ 1		; std=1, 1=free all XMS on exit
?EXPANDFIRST equ 0		; std=0, 1 won't work
endif

;--- publics/externals

		include external.inc
        
;--- start

ife ?MASM
		assume CS:FLAT
endif      
		assume SS:FLAT,DS:FLAT,ES:FLAT

.text$01 SEGMENT

PoolAllocationTable 	DD	0
PoolAllocationEnd		DD	0

LastBlockAllocator		DD 0	; last pool block used for alloc
LastBlockFreed			DD 0	; last pool block used for free
XMSBlockSize			DD 0	; current size of XMS block to allocate for sharing, in K
XMSPoolBlockCount		DD 0	; count of XMS handles allocated for pool blocks (not counting initial UMB block)

.text$01 ends

.text$03 segment

; Pool memory management routines

; get free 4K page count in pool
; out: EAX=free pages
; destroy no other registers
; do not write segment registers here,
; the function is called from VCPI protected mode API

Pool_GetFree4KPages	PROC public
	push esi
	push ecx

	xor	eax, eax
    mov ecx, eax
	mov	esi, [PoolAllocationTable]

@@findblkloop:
;	cmp	[esi].POOL_SYSTEM_INFO.psi_addressK,0
;	je	@@nextblock		; unused/deallocated block

	mov cx,[esi].POOL_SYSTEM_INFO.psi_4kfree
	add	eax,ecx

;@@nextblock:

	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
if ?POOLDBG
	@DbgOutS <"Pool_GetFree4KPages: ">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	pop	ecx
	pop	esi
	ret
    
Pool_GetFree4KPages	ENDP


; get free 16K (EMS) page count in pool
; out: EAX = free EMS pages
; other registers preserved

Pool_GetFree16KPages PROC public
	push esi
	push ecx

	xor eax, eax
    mov ecx, eax
	mov	esi, [PoolAllocationTable]

@@findblkloop:
;	cmp	[esi].POOL_SYSTEM_INFO.psi_addressK,0
;	je	@@nextfind

	mov	cl,[esi].POOL_SYSTEM_INFO.psi_16kfree	; high words known zero
	add	eax,ecx

;@@nextfind:

	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
if ?POOLDBG
	@DbgOutS <"Pool_GetFree16KPages: ">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	pop	ecx
	pop	esi
	ret
    align 4

Pool_GetFree16KPages ENDP


; locate any adjacent free XMS block to current pool allocation block
; if found, try consuming 32K of it for pool allocation block
; the adjacent XMS block must be at the end of current pool block,
;  since the pool block base cannot be changed once set
; pool block base+block size == owner XMS handle base+handle size (end match end)
; ends must match since you can't span noncontiguous sub-blocks of an owner XMS
;  block with a single EMS/VCPI pool allocation block
; INP: EDX -> Pool block to expand (and is ensured to be expandable)
; OUT: NC if success, C if fail
; all registers preserved

Pool_ExpandBlock	PROC

	pushad

	@assumem edx,LPPOOL_SYSTEM_INFO

if ?POOLDBG
	@DbgOutS <"Pool_ExpandBlock: edx=">,1
	@DbgOutD edx,1
	@DbgOutS <", addrK=">
	@DbgOutD [edx].psi_addressK
	@DbgOutS <", pArray=">
	@DbgOutD [edx].psi_descptr
	@DbgOutS <10>,1
endif

;	cmp	[bNoPool],0				; dynamic allocation on?
;	jne	@@locfail
;	cmp	[edx].psi_16kmax,2 * POOLBLOCK_ALLOCATION_SPACE - 1	;32 k still free?
;	jae	@@locfail				; this block is full
;   test [edx].psi_flags,PBF_DONTEXPAND
;   jne	@@locfail				; can't expand this block

	mov	edi,[edx].psi_descptr
if ?EXPANDFIRST
	add edi,[dwRes]
endif
	
	@assumem edi, LPXMS_HANDLE
	
	mov	ebx,[edi].xh_baseK
	add	ebx,[edi].xh_sizeK	; ebx -> end of current pool block owner XMS

; see if owner XMS for EMS/VCPI allocation block end matches
;  end of pool allocation block

	movzx ecx,[edx].psi_startadj
	mov	eax,[edx].psi_addressK
	sub	eax,ecx					; true XMS start when pool allocation block created
	movzx ecx,[edx].psi_16kmax
	shl	ecx,4					; convert to K
	add	eax,ecx
	movzx ecx,[edx].psi_endadj
	add	eax,ecx					; true XMS end when block created
	cmp	eax,ebx
	jne	@@locfail				; owner XMS end no longer matches initial pool block owner XMS end

	movzx ecx, [XMS_Handle_Table.xht_numhandles]
	mov	esi, [XMS_Handle_Table.xht_pArray]

	@assumem esi, LPXMS_HANDLE

; esi -> test XMS block

	movzx eax, [XMS_Handle_Table.xht_sizeof]
@@hanloop:
	cmp	ebx,[esi].xh_baseK		; see if test block immediately follows current block	
	je	@@found
	add	esi,eax		; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
@@locfail:
	popad
	stc				; flag failure
	ret

@@found:
	test [esi].xh_flags,XMSF_FREE	; if block is not free, abort scan
	je	@@locfail
	movzx eax,[edx].psi_endadj
	add	eax,[esi].xh_sizeK
	cmp	eax,32		; free block plus unused end overlap must be >=32K
	jb	@@locfail

; transfer 32K of following block to current block - unused end K in current
	mov	eax,32
	movzx ecx,[edx].psi_endadj
	sub	eax,ecx					; adjust amount to change preceding block
	add	[esi].xh_baseK,eax		; move changed block address ahead
	sub	[esi].xh_sizeK,eax		; and adjust size
	mov	edi,[edx].psi_descptr
if ?EXPANDFIRST
	add edi,[dwRes]
endif
	add	[edi].xh_sizeK,eax		; increase EMS/VCPI associated XMS block size
	mov	[edx].psi_endadj,0		; no end overlap

	add	[edx].psi_16kmax,2		; adjust allocation tracking bytes
	add	[edx].psi_16kfree,2
	add	[edx].psi_4kfree,2*4

;--- zero tracking allocation byte (not really needed)
    
	movzx eax,[edx].psi_16kmax
	shr	eax,1					; byte offset in allocation space (32K/byte)
	mov	BYTE PTR [edx+eax+POOLBLOCK_SYSTEM_SPACE-1],0

; see if changed contiguous XMS block size went to <32K,
;  if so, transfer any remainder to pool block and zero XMS block

	mov	eax,[esi].xh_sizeK
	cmp	eax,31
	ja	@@loc2
	mov	[edx].psi_endadj,al

	xor	eax,eax
	mov	[esi].xh_baseK,eax			;required for FD Himem
	mov	[esi].xh_sizeK,eax
;	mov	[esi].xh_locks,al
	mov	[esi].xh_flags,XMSF_INPOOL	; flag: free handle!

@@loc2:
	mov	[LastBlockFreed],edx	; expanding block size is same as freeing space
	popad
	clc					; flag success
	ret
    align 4

	@assumem edi, nothing
	@assumem esi, nothing
	@assumem edx, nothing

Pool_ExpandBlock	ENDP


; expand any available allocation pool block by 32K, if possible
;  return NC + edx -> expanded allocation pool block
;  C on failure
; destroy no other registers
; do not modify segment registers here!

Pool_ExpandAnyBlock	PROC public

	push esi
	cmp	[bNoPool],0	; dynamic memory allocation on?
	jne	@@fail
	mov	esi, [PoolAllocationTable]
	
	@assumem esi, LPPOOL_SYSTEM_INFO

@@findblkloop:
	cmp	[esi].psi_addressK,0	; unused/deallocated block?
	je	@@nextblock
	cmp	[esi].psi_16kmax,2*POOLBLOCK_ALLOCATION_SPACE - 1	;32k still free?
	jae	@@nextblock				; current block is full
	test [esi].psi_flags,PBF_DONTEXPAND
	jne	@@nextblock 			; can't expand this block
	mov	edx,esi
	call Pool_ExpandBlock
	jnc @@done
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@fail:
	stc		; failure

@@done:
	pop	esi
	ret
    align 4

	@assumem esi, nothing

Pool_ExpandAnyBlock	ENDP


; find and allocate free 4K (VCPI) block in pool blocks
; return NC and EAX == physical address
;         C if none found
; modifies ECX, ESI, EDI
; do not write segment registers here! this function is
; called by VCPI protected-mode API

Pool_Allocate4KPage	PROC public

	@assumem esi, LPPOOL_SYSTEM_INFO

; first try last block allocated, to avoid searching full blocks if possible

	xor	eax,eax
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,eax
	je	@@nolastalloc
	cmp	[esi].psi_4kfree,ax
	jne	@@searchbytes

; try last freed chunk

@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,eax
	je	@@nolastfreed
	cmp	[esi].psi_4kfree,ax
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,eax	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_4kfree,ax
	jne	@@searchbytes
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
    stc
    ret

@@searchbytes:
	movzx ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block
    
    adc ecx,0
    
    lea edi,[esi+POOLBLOCK_SYSTEM_SPACE]
    mov al,-1
    repz scasb
    jz @@blockbad		; jump should never happen
    dec edi
    mov al,[edi]
    mov cl,al
    xor al,-1
    bsf eax,eax			; find the first '1' bit scanning from right to left
	bts	dword ptr [edi],eax	; flag page as 'used'
    cmp eax,4
    jc @@islownyb
    shr cl,4
@@islownyb:
	and cl,0Fh
    setz cl             ; cl will be 1 if low/high nibble became <> 0
	sub	[esi].psi_16kfree, cl
if ?POOLDBG    
	jnc	@@blockok
	@DbgOutS <"Pool_Allocate4kPage: 16k-free count underflow for block ">,1
    @DbgOutD esi,1
	@DbgOutS <10>,1
@@blockok:    
endif    
	dec	[esi].psi_4kfree
	mov	[LastBlockAllocator],esi	; update "last block allocated from"

	sub edi,esi
    sub edi,POOLBLOCK_SYSTEM_SPACE	; get "byte offset" into EDI
	shl	edi,15		; each byte covers 32K
    shl eax,12		; each bit covers 1000h
    add eax,edi		; compute base address of block addressed by byte
	mov	ecx,[esi].psi_addressK
	shl	ecx,10		; convert from K to bytes
	add	eax,ecx
if ?POOLDBG    
	@DbgOutS <"Pool_Allocate4kPage ok, page=">,1
    @DbgOutD eax,1
	@DbgOutS <", block=">,1
    @DbgOutD esi,1
	@DbgOutS <10>,1
endif    
	ret
@@blockbad:    
if ?POOLDBG    
	@DbgOutS <"Pool_Allocate4kPage: found inconsistent block ">,1
    @DbgOutD esi,1
	@DbgOutS <10>,1
endif    
	@CheckBlockIntegrity; nothing free, although block indicated there was
	jmp	@@nextblock		; continue search
    align 4

	@assumem esi, nothing

Pool_Allocate4KPage	ENDP

; find and allocate free 16K (EMS) block in pool blocks
; inp: ebx -> EMS page descriptor entry
; out: NC if ok,
;     eax == offset PoolAllocationTable,
;     edx == nibble offset in descriptor
;      C on errors
; destroy no other registers

Pool_Allocate16KPage PROC public
	push	ecx
	push	esi
	push	edi

	@assumem esi, LPPOOL_SYSTEM_INFO

	xor	edx,edx

; first try last block allocated, to avoid searching full blocks if possible
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,edx
	je	@@nolastalloc
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

; try last freed chunk
@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,edx
	je	@@nolastfreed
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,edx	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
    stc
	jmp	@@exit			;no 16k page free anymore

@@searchbytes:
	movzx ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block

	adc ecx,0    
    
	xor	edi,edi

@@findbyteloop:
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	xor	al,-1			; unallocated 4K areas show as bit set
	mov	ah,al
	and	al,0fh
	cmp	al,0fh
	je	@@lowfree		; low nybble unallocated, free 16K area

	and	ah,0f0h
	cmp	ah,0f0h
	je	@@highfree		; high nybble unallocated, free 16K area

; no free 16K area
	inc	edi
	loop @@findbyteloop
	@CheckBlockIntegrity
	jmp	@@nextblock

@@lowfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0fh
	mov	cl,0
	jmp	@@freeshared	; edx == 0

@@highfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0f0h
	mov	cl,1
	mov	edx,16		; nybble offset is four 4K pages, 16K

@@freeshared:
	mov	eax,edi
	shl	eax,15		; each byte covers 32K
	add	edx,[esi].psi_addressK	; add in base value
	shl	edx,10		; convert K to bytes
	add	edx,eax		; edx == 16k page memory address
	dec	[esi].psi_16kfree
	sub	[esi].psi_4kfree,4
	jnc	@@valid2
    
	@CheckBlockIntegrity	; force valid value

; update ebx pointer

@@valid2:
	mov	[LastBlockAllocator],esi	; update last block allocated from
	mov	eax,esi
	sub	eax, [PoolAllocationTable]
    mov edx,edi
    shl edx,1
    or dl,cl
    clc
@@exit:
	pop	edi
	pop	esi
	pop	ecx
	ret
    align 4

	@assumem esi, nothing

Pool_Allocate16KPage	ENDP

;--- find the pool block for page in EDI
;--- EDI is 1K address
;--- returns PD in ESI and EAX=block start
;--- destroy eax,ecx,esi

Pool_FindBlock proc

	mov	esi, [LastBlockFreed]
    
	@assumem esi, LPPOOL_SYSTEM_INFO
    
	or	esi,esi
	je	@@notlastfreed
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastfreed	; unused/deallocated block

	cmp	edi,eax
	jb	@@notlastfreed	; pool block starts after page
	movzx ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of block
	cmp	edi,ecx
	jb	@@rightblock	; block found?

@@notlastfreed:
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@notlastalloc
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastalloc	; unused/deallocated block

	cmp	edi,eax
	jb	@@notlastalloc	; pool block starts after page
	movzx ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of block
	cmp	edi,ecx
	jb	@@rightblock	; block found?

@@notlastalloc:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@nextblock		; unused/deallocated block

	cmp	edi,eax
	jb	@@nextblock		; pool block starts after page
	movzx ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of block
	cmp	edi,ecx
	jb	@@rightblock	; block found?

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
if ?POOLDBG    
	@DbgOutS <"Pool_FindBlock: page ">,1
    @DbgOutD edx,1
	@DbgOutS <" not in pool",10>,1
endif
	stc
    ret
@@rightblock:
	clc
    ret
    align 4
Pool_FindBlock endp

; upon entry edx = 4K page physical address to free
; return carry clear on success, set on fail
; destroy eax,ecx,esi,edi

Pool_Free4KPage	PROC public

	mov	edi,edx
	shr	edi,10			; convert bytes to K
	and	edi,NOT 3		; ensure 4K alignment

; edi == start of 4K page in K after alignment adjustment

	call Pool_FindBlock
    jc @@fail1			; error "page not in pool"

; the pool allocation block is in ESI

	sub	edi,eax			; 4K offset from block base in K
	mov	eax,edi
	shr	eax,2			; K to 4K page
	mov	cl,al			; keep bit offset
	shr	eax,3			; 4K page to 32K byte offset
	and	ecx,7

	btr	dword ptr [esi+eax+POOLBLOCK_SYSTEM_SPACE],ecx	; see if bit set (was allocated)
	jnc @@fail2			; no

	inc	[esi].psi_4kfree
	mov	[LastBlockFreed],esi

; check if this frees up a 16K chunk

	mov	al,[esi+eax+POOLBLOCK_SYSTEM_SPACE]
    cmp cl,4
	jc	@@islow
    shr al,4
@@islow:
	test al,0Fh			; see if all bits of nybble cleared
	jne	@@success		; no
	inc	[esi].psi_16kfree
	call Pool_TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
@@success:

if ?POOLDBG    
	@DbgOutS <"Pool_Free4kPage ok, page=">,1
    @DbgOutD edx,1
	@DbgOutS <" block=">,1
    @DbgOutD esi,1
	@DbgOutS <10>,1
endif    
	clc
@@fail1:
	ret
@@fail2:
if ?POOLDBG    
	@DbgOutS <"Pool_Free4kPage: page ">,1
    @DbgOutD edx,1
	@DbgOutS <" was not allocated (block=">,1
    @DbgOutD esi,1
	@DbgOutS <")",10>,1
endif    
	stc
    ret
	@assumem esi,nothing
    align 4

Pool_Free4KPage	ENDP


; upon entry edx -> EMS page descriptor
; upon return set EMSPD pool index to -1 (unused)
; destroys EDX

Pool_Free16KPage PROC public
	push esi
	push eax
	push ecx

	movzx esi,[edx].EMSPD.wPD		; [e]si == pool descriptor index
	cmp	si,-1
	je	@@fail1						; bad pointer
	shl	esi,6						; convert 64-byte count to byte offset
	add	esi, [PoolAllocationTable]	; esi -> pool allocation block

	sub edx, [EMSPageAllocationStart]
    mov ecx, [EMSPageAllocationEnd]
    shr edx, 2						; edx = EMS page no
	movzx ecx,[ecx+edx].EMSPD2.bNibOfs	; half-byte offset
    
    mov ah,0Fh
	shr	ecx,1			; byte offset
    jnc @@islow
    mov ah,0F0h
@@islow:
	mov	al,[esi+ecx+POOLBLOCK_SYSTEM_SPACE]
	and	al,ah 			; mask out bits which dont interest
    cmp al,ah
    jne @@fail2
    rol al,4
	and	[esi+ecx+POOLBLOCK_SYSTEM_SPACE],al	; reset all expected bits

	inc	[esi].POOL_SYSTEM_INFO.psi_16kfree
	add	[esi].POOL_SYSTEM_INFO.psi_4kfree,4
	mov	[LastBlockFreed],esi
	call Pool_TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
    mov eax,[EMSPageAllocationStart]
	mov	[eax+edx*4].EMSPD.wPD,-1
	clc
@@ret:
	pop	ecx
	pop	eax
	pop	esi
	ret

@@fail1:
if ?POOLDBG
	@DbgOutS <"Pool_Free16kPage failed edx=">,1
    @DbgOutD edx,1
	sub edx, [EMSPageAllocationStart]
    shr edx, 2
    @DbgOutS <" EMS page=">,1
    @DbgOutD edx,1
    @DbgOutS <10>,1
    jmp @@fail
endif    
@@fail2:
if ?POOLDBG
	@DbgOutS <"Pool_Free16kPage failed page=">,1
    @DbgOutD edx,1
	@DbgOutS <", masks=">,1
    @DbgOutD eax,1
    @DbgOutS <10>,1
endif    
@@fail:
	@CheckBlockIntegrity
	stc
	jmp	@@ret

Pool_Free16KPage	ENDP


; find an unused Pool block
; return NC and edx -> Pool block
; or C if none and no space available
; no other registers modified

Pool_GetUnusedBlock	PROC public
	mov	edx, [PoolAllocationTable]
@@findblkloop:
	cmp	[edx].POOL_SYSTEM_INFO.psi_addressK,0	; unused/deallocated block?
	je	@@found
	add	edx,POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@findblkloop
    stc
@@found:
	ret

Pool_GetUnusedBlock	ENDP

; prepare pool block for use
; upon entry:
;  edx -> pool allocation block
;  ecx == raw size in K before alignment (max 1536+3)
;  edi == raw address in K before alignment
;  esi == owner XMS handle psi_descptr value, do NOT use XMS handle values for
;	size and address since this call may be part of a multi-pool block span
; destroys no registers

Pool_PrepareBlock	PROC

	pushad
	
	@assumem edx, LPPOOL_SYSTEM_INFO
	
	mov	[edx].psi_descptr,esi

	mov	ebx,edi			; raw address - must be aligned to page boundary
    add ebx,3
    and ebx,not 3
	mov	[edx].psi_addressK, ebx
    sub ebx, edi		; 00->00, 01->03, 02->02, 03->01
	mov	[edx].psi_startadj,bl

; block size = (raw size - start adjustment) rounded down to 32K boundary
;  since each allocation byte covers 32K

	mov	eax,ecx			; raw size
	sub	eax,ebx
	and	al,NOT 31
	shr	eax,4			; 16K count, known 16-bit value going to 8-bit

	cmp	ax,POOLBLOCK_ALLOCATION_SPACE*2
	jbe	@@setmax
	mov	ax,POOLBLOCK_ALLOCATION_SPACE*2

@@setmax:
	mov	[edx].psi_16kmax,al
	mov	[edx].psi_16kfree,al
	shl	eax,2			; 8-bit value potentially going to 16-bit
	mov	[edx].psi_4kfree,ax

; compute end adjustment, raw size - (block size + start adjustment)
	movzx ebx,[edx].psi_16kmax
	shl	ebx,4			; convert to K
	movzx eax,[edx].psi_startadj
	add	eax,ebx
	sub	ecx,eax			; ecx == raw size
	mov	[edx].psi_endadj,cl

; zero allocation entries
	xor	eax,eax
	lea	edi,[edx+POOLBLOCK_SYSTEM_SPACE]
	mov	ecx,POOLBLOCK_ALLOCATION_SPACE/4
	rep stosd
	@BIG_NOP

	popad
	ret
	@assumem edx, nothing

Pool_PrepareBlock	ENDP

if ?FREEXMS

Pool_FreeAllBlocks proc public

	pushad
	mov	esi, [PoolAllocationTable]	; esi -> pool allocation block
@@nextitem:
	cmp [esi].POOL_SYSTEM_INFO.psi_addressK,0
    jz @@skipitem
	movzx eax, [esi].POOL_SYSTEM_INFO.psi_16kmax
	mov [esi].POOL_SYSTEM_INFO.psi_16kfree,al
    shl eax, 2
	mov [esi].POOL_SYSTEM_INFO.psi_4kfree,ax
	call Pool_TryFreeToXMS
@@skipitem:    
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
    jb @@nextitem
    popad
    ret
    
Pool_FreeAllBlocks endp

endif

; upon entry esi -> pool allocation block to check if freeable to XMS
; perform the free if possible
; destroys eax,ecx

Pool_TryFreeToXMS PROC

;	cmp	[bNoPool],0		; dynamic memory allocation on?
;	jne	@@exit			; no

	@assumem esi, LPPOOL_SYSTEM_INFO
	
	test [esi].psi_flags,PBF_DONTFREE
	jne	@@exit			; never free these blocks

	mov	al,[esi].psi_16kfree
	cmp	al,[esi].psi_16kmax
	ja	@@bad			; free more than max, try to fix
	jne	@@exit			; free is less than maximum, used

	movzx eax,[esi].psi_4kfree
	shr	eax,2
	or	ah,ah
	jne	@@bad
	cmp	al,[esi].psi_16kmax
	ja	@@bad
	jne	@@exit			; free less than max

; ok, this pool block is not used anymore.
; now scan all pool blocks and check all those linked to the same XMS handle
; if they are free too. If yes, then mark XMS handle as free and clear all
; blocks.

	push esi
	mov	esi,[esi].psi_descptr
	mov	eax,[PoolAllocationTable]

	@assumem esi, nothing
	@assumem eax, LPPOOL_SYSTEM_INFO

@@checkblkloop:
	cmp	[eax].psi_addressK,0
	je	@@checknext			; unused block

	cmp	esi,[eax].psi_descptr
	jne	@@checknext

	test [eax].psi_flags,PBF_DONTFREE
	jne	@@checkdone		; can't free this block

; see if block empty
	movzx ecx,[eax].psi_16kmax
	cmp	cl,[eax].psi_16kfree
	jne	@@checkdone

	shl	ecx,2				; convert to 4K max
	cmp	cx,[eax].psi_4kfree
	jne	@@checkdone

@@checknext:
	add	eax, POOLBLOCK_TOTAL_SPACE
	cmp	eax, [PoolAllocationEnd]
	jb	@@checkblkloop

	@assumem eax, nothing

; checked all blocks as empty, go through them again and mark unused

	push edi
	mov	edi, [PoolAllocationTable]

	@assumem edi, LPPOOL_SYSTEM_INFO

@@freeblkloop:
	cmp	[edi].psi_addressK,0
	je	@@freenext			; unused block

	cmp	esi,[edi].psi_descptr
	jne	@@freenext

; mark the block as free
	xor eax, eax
	mov [edi].psi_addressK, eax
	mov dword ptr [edi].psi_4kfree, eax

	@assumem edi, nothing

@@freenext:
	add	edi,POOLBLOCK_TOTAL_SPACE
	cmp	edi, [PoolAllocationEnd]
	jb	@@freeblkloop
	pop	edi

if ?EXPANDFIRST
	add esi, [dwRes]
endif
	call Pool_FreeEMB

@@checkdone:
	pop esi
@@exit:    
	ret

@@bad:
	@CheckBlockIntegrity
    stc
    ret

Pool_TryFreeToXMS ENDP

; populate empty pool blocks with XMS owner info
; inp:
;  esi -> XMS (pseudo-)handle
;  ecx == size in kB XMS block 
;  edi == physical address (in K, shifted 10 to right!)
; NOTE: ecx and edi may not match owner XMS size/address
; out: NC success
;       C if insufficient number of empty blocks to cover XMS handle range
; destroys eax,ebx,ecx,edx,edi

Pool_AllocBlocksForEMB	PROC

	mov	ebx,ecx
if ?EXPANDFIRST
	sub esi,[dwRes]
endif
@@allocloop:
	call Pool_GetUnusedBlock
	jc	@@exit		; no more blocks, remainder of XMS is effectively discarded

	mov	eax,edi		; compute size of candidate block/offset to new
    add eax,3
    and al,not 3
    sub eax,edi
	add	eax,1536	; 1.5M (in K) plus alignment adjustment size
	cmp	eax,ebx
	jbe	@@sizeok
	mov	eax,ebx
@@sizeok:
	mov	ecx,eax
	call Pool_PrepareBlock	; uses esi entry condition
	add	edi,eax			; update pool allocation block address
	sub	ebx,eax			; update size left to allocate
	cmp	ebx,32			; see if should remainder what's left
	jnb	@@allocloop
	mov	[edx].POOL_SYSTEM_INFO.psi_endadj,bl
	clc
@@exit:
	ret

Pool_AllocBlocksForEMB	ENDP

; walk XMS blocks, find largest XMS block which has size 1.5M >= x >= 32K
;  after 4K alignment and allocate it for new EMS/VCPI pool allocation block
; if all XMS blocks >1.5M, then pick smallest and try to put remainder
;  into a free handle.
; If no free handle, allocate sufficient new EMS/VCPI pool blocks to cover full range.
; If not enough free EMS/VCPI blocks available, then remaining allocation is lost
;  until handle is freed.  This could only happen under very bad XMS fragmentation,
;  if at all.
; return carry clear if success, set if fail
; destroy no other registers
; do not write segment registers here!

Pool_AllocateEMBForPool	PROC public

	pushad

	cmp	[bNoPool],0	; check if pool sharing
	jne	@@allocfail

	mov	ebp, 1024+512	;1536
    
	mov	esi, [XMS_Handle_Table.xht_pArray]
	movzx ecx, [XMS_Handle_Table.xht_numhandles]

if ?POOLDBG
	@DbgOutS <"Pool_AllocateEMBForPool: XMS array=">,1
	@DbgOutD esi,1
	@DbgOutS <", handles=">,1
	@DbgOutW cx,1
	@DbgOutS <10>,1
endif

	@assumem esi, LPXMS_HANDLE

	call Pool_GetUnusedBlock; test only, don't keep pointer
	jc	@@allocfail			; unable to make initial pool block allocation
	xor	edx,edx				; edx -> largest block <= 1.5M or smallest if none <=1.5M

@@hanloop:
	test [esi].xh_flags,XMSF_FREE
	je	@@next				; yes, don't check
	mov	ebx,[esi].xh_baseK
	mov	eax,[esi].xh_sizeK
ife ?INTEGRATED    
	and	ebx,ebx
	je	@@next				; FD Himem bug, can't check blank or zero-sized handle
	or	eax,eax
	je	@@next
endif    
	or	edx,edx
	je	@@newcandidate		; auto-match if first xms block available

; adjust for alignment loss (1-3K) in test

	and	bl,3
	mov	bh,4
	sub	bh,bl		;bh=4,3,2,1
	and	bh,3		;bh=3,2,1,0 bl=1,2,3,0
	movzx ebx,bh
	sub	eax,ebx		;eax=size of current block (kB)

	@assumem edx, LPXMS_HANDLE

; adjust for alignment lost in candidate

;	cmp	[edx].xh_sizeK,32	; ensure candidate size isn't so small that adjustment will underflow
;	jb	@@next			; doesn't even meet minimum requirements

	mov	bl,byte ptr [edx].xh_baseK
    
	and	bl,3
	mov	bh,4
	sub	bh,bl		;bh=4,3,2,1
	and	bh,3		;bh=3,2,1,0 if bl=1,2,3,0
	movzx ebx,bh
    
	neg	ebx
	add	ebx,[edx].xh_sizeK	;ebx = size of current candidate

; eax holds test value size, ebx holds current candidate, both alignment adjusted
	cmp	eax,ebx
	je	@@next
	ja	@@larger

; test XMS block smaller than candidate block
	cmp	ebx,ebp 		; in K
	jbe	@@next			; current candidate closer to match size
	cmp	eax,32
	jb	@@next			; test too small
	jmp	@@newcandidate

; test XMS block larger than candidate block
@@larger:
	cmp	ebx,ebp
	jae	@@next			; current candidate closer to match size
	cmp	eax,ebp
	ja	@@next			; test too large

@@newcandidate:
	cmp	[esi].xh_sizeK,32
	jb	@@next			; candidate doesn't even meet unadjusted requirements
	mov	edx,esi			; new best candidate

@@next:
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax			; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
	or	edx,edx
	je	@@allocfail

; candidate must be at least 32K, after 4K alignment adjustment
	mov	bl,BYTE PTR [edx].xh_baseK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx ebx,bh
	neg	ebx
	add	ebx,[edx].xh_sizeK
	cmp	ebx,32
	jnb	@@allocok				; candidate large enough?
@@allocfail:
	popad
	stc
	ret

@@allocok:
if ?POOLDBG
	@DbgOutS <"Pool_AllocateEMBForPool: used XMS block hdl=">,1
	@DbgOutD edx,1
	@DbgOutS <" addr=">,1
	@DbgOutD [edx].xh_baseK,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edx].xh_sizeK,1
	@DbgOutS <10>,1
endif

	mov	[edx].xh_flags,XMSF_USED	; flag candidate as used
	mov	[edx].xh_locks,1			; and locked

	mov	[XMSBlockSize],ebp 	; default allocation maximum size
	mov	eax, [XMSPoolBlockCount]
	cmp	eax,1
	jbe	@@trailadj			; use standard 1.5M size for first two blocks
	dec	eax					; should never overflow before we hit 4G total allocated

;@@noadj:
	and	al,0fh				; but ensure that overflow doesn't happen anyway
	mov	cl,al				; shift the block size higher by factor of two
	shl	[XMSBlockSize],cl

; if XMSBlockSize >= max free, then reduce XMSBlockSize

	mov	eax,[dwMaxMem4K]
	sub	eax,[dwUsedMem4K]
	jc	@@adjusted			; shouldn't happen, continue without adjustment
	shl	eax,2				; convert to 1K blocks

@@checksize:
	cmp	eax, [XMSBlockSize]
	jae	@@adjusted
	cmp	[XMSBlockSize],ebp	; see if XMSBlockSize is at minimum default
	jbe	@@adjusted			; yes, can't reduce it any further
	shr	[XMSBlockSize],1	; reduce block size by one shift and try again
	jmp	@@checksize
@@adjusted:

; allow up to 31K trailing bytes
@@trailadj:
	mov	eax, [XMSBlockSize]
	add	eax,31				; adjust for possible trail
	cmp	ebx,eax
	jbe	@@setblock			; no need to split XMS handle allocation

; search for a free XMS handle

	mov	  edi, [XMS_Handle_Table.xht_pArray]
	movzx ecx, [XMS_Handle_Table.xht_numhandles]
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	@assumem edi, LPXMS_HANDLE
@@freeloop:
	test [edi].xh_flags,XMSF_INPOOL
	jnz @@gotfree
ife ?INTEGRATED    
	cmp	[edi].xh_flags,XMSF_USED	; some Himems dont set XMSF_INPOOL, so
	je	@@nextfree					; check FREE items if address/size is NULL
	cmp	[edi].xh_baseK,0
	je	@@gotfree
	cmp	[edi].xh_sizeK,0
	je	@@gotfree
endif    
@@nextfree:
	add	edi,eax			; move to next handle descriptor
	loop @@freeloop

; no free handle found, try to allocate multiple blocks, discarding excess

	jmp @@setblock

@@gotfree:
	mov	cl,BYTE PTR [edx].xh_baseK	; compute size of candidate block/offset to new
	and	cl,3
	mov	ch,4
	sub	ch,cl
	and	ch,3
	movzx ecx,ch

;	add	ecx,1536			; 1.5M (in K) plus alignment adjustment size
	add	ecx, [XMSBlockSize]	; maximum size (exceeded) plus alignment adjustment size

; edx -> candidate block being allocated, edi -> new block receiving remainder
; update candidate XMS block size
	mov	eax,[edx].xh_sizeK		; keep original size for updating new block
	mov	[edx].xh_sizeK,ecx

; update new XMS block info
	sub	eax,ecx					; new block size == old block original size - old block new size
	mov	[edi].xh_sizeK,eax
	mov	[edi].xh_flags,XMSF_FREE	; explicitly flag free
	mov	[edi].xh_locks,0
	mov	eax,[edx].xh_baseK
	add	eax,ecx
	mov	[edi].xh_baseK,eax	; new block start == old block start + old block new size

if ?POOLDBG
	@DbgOutS <"Pool_AllocateEMBForPool: free XMS block hdl=">,1
	@DbgOutD edi,1
	@DbgOutS <" addr=">,1
	@DbgOutD [edi].xh_baseK,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edi].xh_sizeK,1
	@DbgOutS <10>,1
endif

; edx -> owner XMS handle for new pool allocation block(s)
; may be multiple blocks due to XMSBlockCount shifter

@@setblock:
	mov	esi,edx
	mov	ecx,[esi].xh_sizeK
	mov	edi,[esi].xh_baseK
	call Pool_AllocBlocksForEMB

	inc	[XMSPoolBlockCount]
	popad
	clc
	ret

	@assumem edx,nothing
	@assumem esi,nothing
	@assumem edi,nothing

Pool_AllocateEMBForPool	ENDP

;  count available XMS 4K-aligned 4K pages in 32K chunks
;  return count in edx
;  destroys no other registers
;  do NOT push/pop segment registers here! this function
;  is called by VCPI protected mode API.

Pool_GetFreeXMSPages	PROC public
	push esi
	push ecx
	xor	edx,edx
	cmp	[bNoPool], 0	; XMS memory pool?
	jne	@@countdone
	movzx ecx, [XMS_Handle_Table.xht_numhandles]
	mov	esi, [XMS_Handle_Table.xht_pArray]

	push eax
	push ebx

	@assumem esi, <LPXMS_HANDLE>

@@hanloop:
	test [esi].xh_flags,XMSF_FREE
	jz	@@next
ife ?INTEGRATED    
	xor	eax,eax
	cmp	eax,[esi].xh_baseK	; account for FD Himem bug
	je	@@next
	cmp	eax,[esi].xh_sizeK
	je	@@next
endif
	mov	eax,[esi].xh_baseK
	mov	ebx,eax
	add	ebx,3		; round up
	add	eax,[esi].xh_sizeK
	and	al,0fch		; align to 4K boundary
	and	bl,0fch
	sub	eax,ebx		; compute size of block after alignments
	jbe	@@next
	and	al,NOT 1fh	; mask to 32K
	shr	eax,2		; convert 1K to 4K
	add	edx,eax		; update total count

@@next:
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax	; move to next handle descriptor
	dec	ecx
	jne	@@hanloop

	pop	ebx
	pop	eax
@@countdone:
	pop	ecx
	pop	esi
	ret

	@assumem esi, nothing

Pool_GetFreeXMSPages	ENDP


; mark an XMS handle as free.
; scan through the XMS handle array
; and try to merge this block with other free blocks
; ESI = handle which just has become free
; preserves all registers

Pool_FreeEMB PROC

	pushad

	@assumem edi, LPXMS_HANDLE
	@assumem esi, LPXMS_HANDLE

	mov	[esi].xh_locks,0
	mov	[esi].xh_flags,XMSF_FREE
    
	dec	[XMSPoolBlockCount]

	mov	edi, [XMS_Handle_Table.xht_pArray]
	movzx ecx, [XMS_Handle_Table.xht_numhandles]

	mov edx, [esi].xh_baseK
    add edx, [esi].xh_sizeK
	
	movzx eax, [XMS_Handle_Table.xht_sizeof]
@@checkloop:
	cmp	[edi].xh_flags,XMSF_FREE	; see if free
	jne	@@checknext					; anything else is to ignore
ife ?INTEGRATED    
	cmp	[edi].xh_baseK,0			; FD Himem: free + base 0 is "INPOOL"
	je	@@checknext					; can't check blank handle
endif    
    cmp edi, esi
    je  @@checknext
	mov	ebx,[edi].xh_baseK			; which starts at EBX (end of test block)
	cmp edx,ebx
    jz  @@merge1
    add ebx,[edi].xh_sizeK
    cmp ebx,[esi].xh_baseK
    jz  @@merge2
@@checknext:
	add	edi,eax			; move to next handle descriptor
	loop @@checkloop
@@nodefrag:
	popad
	ret
    
@@merge2:
	push edi
	xchg esi, edi
    call merge
    pop edi
	jmp	@@checknext 	;there might come just another free block to merge
@@merge1:
	push offset @@checknext
merge:
	mov	ebx,[edi].xh_sizeK
    add [esi].xh_sizeK, ebx
	mov	[edi].xh_flags,XMSF_INPOOL	; flag handle as free
    xor ebx,ebx
	mov	[edi].xh_locks,bl
	mov	[edi].xh_baseK,ebx
	mov	[edi].xh_sizeK,ebx
    retn

	@assumem esi, nothing
	@assumem edi, nothing

Pool_FreeEMB ENDP

; hook left for debugging, no current actions taken
;; upon entry esi -> pool allocation block to perform integrity check upon
;;	update with valid information if allocation counts mismatch
;; destroy no registers

if ?POOLDBG
CheckBlockIntegrity	PROC
	ret
CheckBlockIntegrity	ENDP
endif

.text$03 ends

.text$04 segment

; initialize memory pool block descriptors
; each descriptor describes a memory block <= 1.5M (48*8 * 4K)
; and is 16 + 48 = 64 bytes in size.
; required are: ((dwMaxMem4K / 1.5M) + x) * 64 bytes bytes for these items
; x is max number of XMS handles
; ESI -> JEMMINIT
; EDI -> free memory

Pool_Init1 proc public

;---  is XMS pooling on? then direct access to XMS handle table required?

ife ?INTEGRATED
	cmp		[bNoPool],0
	jne		@@noxmsarray
endif    
	mov		ecx, [esi].JEMMINIT.jiXMSHandleTable
	movzx	eax, cx
	shr		ecx, 12
	and		cl, 0F0h
	add 	ecx, eax
; transfer XMS table info to fixed memory location, assume two dwords
	mov		eax,[ecx+0]
	mov		dword ptr [XMS_Handle_Table],eax
ife ?INTEGRATED
	test	eax,0FFFF0000h		;if size of array is null, disable pooling
    setz	[bNoPool]
    jz		@@noxmsarray
endif
	movzx edx,word ptr [ecx+4]	;this is the handle array
	movzx eax,word ptr [ecx+6]
	shl	eax,4
	add	eax,edx
ife ?INTEGRATED    
    and eax, eax		;if the array pointer is NULL, disable pooling
    setz [bNoPool]
    jz @@noxmsarray
endif    
	cmp eax, 100000h	;is handle array in HMA?
	jb @@nohmaadjust
	@GETPTEPTR edx, 2000h+?HMAPTE, 1	;make a copy of the HMA PTEs

if ?INITDBG    
	@DbgOutS <"copy of HMA PTEs at ">, 1
	@DbgOutD edx, 1
	@DbgOutS <10>, 1
endif

	mov  ecx,10h
	push eax
	mov eax, 100000h + 7
@@loophmacopy:
	mov  [edx], eax
	add  edx, 4
	add  eax, 1000h
	loop @@loophmacopy
	pop eax
	mov edx, ?HMALINEAR
	lea eax, [eax + edx - 100000h]
@@nohmaadjust:
	mov	[XMS_Handle_Table.xht_pArray],eax
@@noxmsarray:

if 0
	add	edi,64-1
	and	edi,NOT (64-1)	; 64-byte align tables for proper location by count
else
	add edi,15			; just do a paragraph align
    and edi,not 15
endif
	mov	[PoolAllocationTable],edi

	mov	eax,[dwMaxMem4K]	;= 30720 x 4k = 120 MB
	cdq
	mov	ecx,1536/4		; since dwMaxMem4k is in 4k units
	div	ecx				; default: 30720/384 = 80
	inc eax				; round up, not down
	cmp [bNoPool],0
	jnz @@isnopool
	movzx ecx,[XMS_Handle_Table.xht_numhandles]
	add eax, ecx
@@isnopool:    
	shl	eax,6-2			; 64 bytes/16 dwords each
	mov	ecx,eax

	xor	eax,eax
	rep	stosd

	mov	[PoolAllocationEnd],edi

if ?INITDBG
	@DbgOutS <"pool start/end=">,1
	@DbgOutD PoolAllocationTable,1
	@DbgOutS <"/">,1
	@DbgOutD PoolAllocationEnd,1
	@DbgOutS <10>,1
endif	 
	ret
    
Pool_Init1 endp    

; Pool Init phase 2
;
; ESI -> JEMMINIT
; EDI -> free memory (physical)
; EAX -> size still free in fix block

; first pool allocation block(s) are used to manage remainder of initial XMS
; allocated memory (fixed EMS/VCPI allocations).
; start and end adjustments leave as initialized zero value since the memory
; is originally 4K aligned, via edi alignment above

Pool_Init2 proc public

	push esi
    push ebp
	mov	ecx, eax			; count of available 16K blocks
	movzx ebp, [esi].JEMMINIT.jiXMSControlHandle

@@fixblkloop:
	call Pool_GetUnusedBlock	
    jc @@fixblkdone
    mov esi, edx
    
	@assumem esi, LPPOOL_SYSTEM_INFO

	mov	[esi].psi_descptr,ebp
	mov	eax,ecx
	cmp	eax,POOLBLOCK_ALLOCATION_SPACE * 2	; 92 == 1536 / 16
	jb	@@fix2
	mov	eax,POOLBLOCK_ALLOCATION_SPACE * 2
@@fix2:
	
	sub	ecx,eax

if 0;?INITDBG
	@DbgOutS <"pool desc init, block=">,1
	@DbgOutD esi,1
	@DbgOutS <" 16k pg=">,1
	@DbgOutD eax,1
	@DbgOutS <" remaining=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif	 
	mov bl,al
	mov	[esi].psi_16kmax,al
	mov	[esi].psi_16kfree,al
	shl	eax,2
	mov	[esi].psi_4kfree,ax
	test bl,1		;was it uneven?
	jz	@@iseven
;;	inc bl						;add one page since pool must have an even
;;	mov	[esi].psi_16kmax,bl		;number for max
	movzx ebx, bl
	shr ebx, 1
	mov byte ptr [esi+ebx+POOLBLOCK_SYSTEM_SPACE],0F0h	;last page is "used"
@@iseven:
	mov	ebx,edi
	shr	ebx,10			; convert byte address to K
	mov	[esi].psi_addressK,ebx
	shl	eax,12			; convert 4K count to bytes
	add	edi,eax			; okay to increment edi free memory ptr, it's not used further

; never expand these blocks, the descriptor holds a real XMS handle rather than
;  an XMS pseudo-handle/pointer taken from the XMS handle array

if ?EXPANDFIRST
	or	[esi].psi_flags,PBF_DONTFREE
else
	or	[esi].psi_flags,PBF_DONTEXPAND or PBF_DONTFREE
endif

	and ecx, ecx
	jnz	@@fixblkloop	; more memory to distribute to allocation blocks?

@@fixblkdone:

	@assumem esi, nothing
    pop ebp
    pop esi
	ret
	
Pool_Init2 ENDP

.text$04 ENDS

		END
