/*
// Program:  Format
// Version:  0.90c
// (0.90b/c - fixed compiler warnings, pointers - Eric Auer 2003)
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module Name:  createfs.c
// Module Description:  Functions to create the file systems.
*/

#define CREATEFS

#include <stdlib.h>
#include <string.h> /* strlen, memcpy -ea */
#include <dos.h>    /* FP_OFF, FP_SEG */
#include <stdio.h>  /* printf */

void E_memcpy(unsigned char far *, char *, size_t);
void E_memcpy(unsigned char far * dst, char * src, size_t cnt) {

  /*
  printf("E_memcpy(%4.4x:%4.4x, %4.4x:%4.4x, %d)\n",
    FP_SEG(dst), FP_OFF(dst), FP_SEG(src), FP_OFF(src), cnt);
  */

  while (cnt-- > 0) {
      dst[0] = src[0];
      dst++;
      src++;
  }
}

#include "bootcode.h"
#include "format.h"
#include "floppy.h"
#include "driveio.h"
#include "btstrct.h"
#include "createfs.h"

void Clear_Sector_Buffer(void); /* -ea */

#define CFSp(a) (void far *)(a)

void Clear_Reserved_Sectors(void);
void Create_Boot_Sect_Start(void);
void Create_Boot_Sect_Ext(void);
void Create_FS_Info_Sector(void);
void Get_FS_Info(void);
void Write_Boot_Sectors(void);
void Write_FAT_Tables(void);
void Write_Root_Directory(void);

unsigned long Create_Serial_Number(void);

FSI file_sys_info;

void Create_File_System()
{
  Get_FS_Info();

  /* Flush DOS disk buffers. */
  regs.h.ah = 0x0d;
  intdos(&regs,&regs);

  if(debug_prog==TRUE) printf("\n[DEBUG]  Creating File System\n");

  /* Clear reserved sectors. */
  Clear_Reserved_Sectors();

  /* Create boot sector structures. */
  Create_Boot_Sect_Start();
  Create_Boot_Sect_Ext();

  Clear_Sector_Buffer();

  /* *** Add Start of boot sector */
  E_memcpy(&sector_buffer[0], (char *)&bs_start, 11);

  /* *** Add Standard BPB */
  E_memcpy(&sector_buffer[11],(void *)&bpb_standard, 35-11+1);

  /* *** Add FAT32 BPB Extension, if applicable. */
  if(param.fat_type==FAT32)
    {
    E_memcpy(&sector_buffer[36], (void *)&bpb_fat32_ext, 63-36+1);
    }
  /* *** Add Extended boot sector fields. */
  if(param.fat_type!=FAT32)
    E_memcpy(&sector_buffer[36], (void *)&bs_ext, 61-36+1);
  else
    E_memcpy(&sector_buffer[64], (void *)&bs_ext, 89-64+1);

  /* Add Boot Code */
  if(param.fat_type!=FAT32)
    E_memcpy(&sector_buffer[62], (void *)&boot_code, 130);
  else
    E_memcpy(&sector_buffer[90], (void *)&boot_code, 130);

  /* Add Flag */
  sector_buffer[510]=0x55;
  sector_buffer[511]=0xaa;

  Write_Boot_Sectors();

  /* Flush DOS disk buffers so there is not any phantom information in */
  /* the directory.                                                    */
  regs.h.ah = 0x0d;
  intdos(&regs, &regs);
  /* this instance of int 21.0d got added by EA - is it useful? */

  Write_FAT_Tables();

  Write_Root_Directory();

  /* Flush DOS disk buffers so there is not any phantom information in */
  /* the directory.                                                    */
  regs.h.ah = 0x0d;
  intdos(&regs, &regs);
}

void Create_Boot_Sect_Start()
{
  /* Add Jump Instruction */
  if(param.fat_type==FAT12 || param.fat_type==FAT16)
    {
    /* Add instruction for FAT12 & FAT16 */
    bs_start.jmp[0]=0xeb;
    bs_start.jmp[1]=0x3c;
    bs_start.jmp[2]=0x90;
    }
  else
    {
    /* Add instruction for FAT32 */
    bs_start.jmp[0]=0xeb;
    bs_start.jmp[1]=0x58;
    bs_start.jmp[2]=0x90;
    }

  /* Add OEM ID */
  memcpy(&bs_start.oem_id,"FDOS_1.0", 8);
}

void Create_Boot_Sect_Ext()
{
  unsigned long serial_number;

  /* Add Physical Drive Number. */
  if(param.drive_type==FLOPPY) bs_ext.drive_number=0x00;
  if(param.drive_type==HARD)   bs_ext.drive_number=0x80;

  /* Zero reserved field. */
  bs_ext.reserved=0x00;

  /* Add Signature Byte */
  bs_ext.signature_byte=0x29;

  /* Add Serial Number. */
  serial_number=Create_Serial_Number();
  memcpy(&bs_ext.serial_number[0], &serial_number, 4);

  memcpy(&drive_statistics.serial_number_low,&bs_ext.serial_number[0],2);
  memcpy(&drive_statistics.serial_number_high,&bs_ext.serial_number[2],2);

  /* Add Volume Label */
  if(param.v==TRUE)
    {
    memset( &bs_ext.volume_label[0], ' ', 11);
    memcpy( &bs_ext.volume_label[0], param.volume_label, min(11, strlen(param.volume_label)));
    }
  else
    {
    memcpy( &bs_ext.volume_label[0], "NO NAME    ", 11);
    }

  /* Add System ID */
  memcpy(&bs_ext.file_system_type[0],"FAT1",4);

  if(param.fat_type==FAT12) bs_ext.file_system_type[4]='2';
  if(param.fat_type==FAT16) bs_ext.file_system_type[4]='6';
  if(param.fat_type==FAT32) memcpy(&bs_ext.file_system_type[3],"32",2);

  memcpy(&bs_ext.file_system_type[5],"   ",3);
}

void Create_FS_Info_Sector()
{
  static struct
    {
    unsigned long    lead_signature         ;
    char             reserved_1        [480];
    unsigned long    struc_signature        ;
    unsigned long    free_count             ;
    unsigned long    next_free              ;
    char             reserved_2         [12];
    unsigned long    trailing_signature     ;
    } fs_info_sector;

  unsigned i;
  char far * fsIS_p = (char far *)(&fs_info_sector);

  for (i=0; i<sizeof(fs_info_sector); i++) {
    fsIS_p[i] = 0;
  }    
  /* memset(CFSp(fsIS_p), 0, sizeof(fs_info_sector)); */ /* & changed -ea */

  fs_info_sector.lead_signature     =0x41615252L;
  fs_info_sector.struc_signature    =0x61417272L;
  fs_info_sector.free_count         =file_sys_info.total_clusters - 1;
                                     /* 1 cluster used by root directory */
  fs_info_sector.next_free          =0x2;
  fs_info_sector.trailing_signature =0xaa550000L;

  E_memcpy(&sector_buffer[0], (char *)(&fs_info_sector), 512);

  if(debug_prog) printf("\n[DEBUG]  Writing FS Info Sector->    1\n");
  Drive_IO(WRITE,1,1);

  if(debug_prog) printf("\n[DEBUG]  Writing FS Info Sector->    7\n");
  Drive_IO(WRITE,7,1);

}

static unsigned long cluster_count(void)
{
  unsigned long totalsect;
  
  totalsect = bpb_standard.total_sectors;
  if (totalsect == 0)
    totalsect = ((unsigned long)(bpb_standard.large_sector_count_high) << 16) |
      bpb_standard.large_sector_count_low;

  totalsect -= file_sys_info.start_root_dir_sect;

  if (param.fat_type != FAT32)
    totalsect -= file_sys_info.number_root_dir_sect;

  return totalsect / bpb_standard.sectors_per_cluster;
}

void Get_FS_Info()
{
  file_sys_info.start_fat_sector = bpb_standard.reserved_sectors;

  file_sys_info.number_fat_sectors = bpb_standard.sectors_per_fat;
  if (file_sys_info.number_fat_sectors == 0)
    file_sys_info.number_fat_sectors =
      ((unsigned long)(bpb_fat32_ext.fat_size_high) << 16) |
      bpb_fat32_ext.fat_size_low;

  file_sys_info.start_root_dir_sect =
    (file_sys_info.number_fat_sectors * bpb_standard.number_of_fats) +
    bpb_standard.reserved_sectors;

  if (param.fat_type == FAT32)
    file_sys_info.number_root_dir_sect = 1024u*32/512;
					 /* Pretend there are 1024 entries.
					    This is done to guarantee that
					    enough space is cleared.        */  
  else
    file_sys_info.number_root_dir_sect =
      (bpb_standard.root_directory_entries*32 + bpb_standard.bytes_per_sector - 1) /
      bpb_standard.bytes_per_sector;

  file_sys_info.total_clusters = cluster_count();
}

void Clear_Reserved_Sectors()
{
  int index = 0;

  Clear_Sector_Buffer();

  do
    {
    Drive_IO(WRITE,index,1);

    index++;
    } while(index< file_sys_info.start_fat_sector);
}

unsigned long Create_Serial_Number()
{
  unsigned long serial_number;
  unsigned long serial_number_high;
  unsigned long serial_number_low;
  unsigned long scratch;

  unsigned char month;
  unsigned char day;
  unsigned int year;

  unsigned char hour;
  unsigned char minute;
  unsigned char second;
  unsigned char hundredth;

#ifdef __TURBOC__
  /* Turbo C 2.01 cannot handle multi instruction asm{ ... } blocks -ea */
  /* ... and not to forget, Turbo C has getdate and gettime anyway! -ea */
  {
    struct time ti;
    struct date da;
    
    getdate( &da );
    gettime( &ti );

    year = da.da_year;
    day = da.da_day;
    month = da.da_mon;
    
    hour = ti.ti_hour;
    minute = ti.ti_min;
    second = ti.ti_sec;
    hundredth = ti.ti_hund;
    
  }
#else
  asm{
    mov ah,0x2a
    int 0x21

    mov WORD PTR year,cx
    mov BYTE PTR month,dh
    mov BYTE PTR day,dl

    mov ah,0x2c
    int 0x21

    mov BYTE PTR hour,ch
    mov BYTE PTR minute,cl
    mov BYTE PTR second,dh
    mov BYTE PTR hundredth,dl
    }
#endif

  serial_number_low = hour;
  serial_number_low = serial_number_low << 8;
  serial_number_low += minute;
  scratch = year;
  serial_number_low += scratch;

  serial_number_high = second;
  serial_number_high = serial_number_high << 8;
  serial_number_high += hundredth;
  scratch = month;
  scratch = scratch << 8;
  scratch += day;
  serial_number_high += scratch;

  serial_number = (serial_number_high << 16) + serial_number_low;

  return serial_number;
}

void Write_Boot_Sectors()
{
  /* Write boot sector to the first sector of the disk */
  if(debug_prog==TRUE) printf("\n[DEBUG]  Writing Boot Sector->    0\n");
  Drive_IO(WRITE,0,1);

  if(param.fat_type==FAT32)
    {
    if(debug_prog==TRUE)
      printf("\n[DEBUG]  Writing FAT32 Backup Boot Sector->    %d\n",
        bpb_fat32_ext.backup_boot_sector);
    Drive_IO(WRITE,bpb_fat32_ext.backup_boot_sector,1);

    Create_FS_Info_Sector();

    /* ** Create 3rd boot sector. */
    Clear_Sector_Buffer(); /* ??? */

    /* Add Flag */
    sector_buffer[510]=0x55;
    sector_buffer[511]=0xaa;
    /* Write sectors. */
    Drive_IO(WRITE,2,1);
    Drive_IO(WRITE,8,1);
    }
}

void Write_FAT_Tables()
{
  unsigned long sector;

  /* Configure FAT Tables */
  Clear_Sector_Buffer();
  sector = file_sys_info.start_fat_sector;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing FAT Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);

    sector++;
    }while(sector< (file_sys_info.start_fat_sector
	     +( 2 * file_sys_info.number_fat_sectors)));

  sector_buffer[0]=drive_specs[param.media_type].media_descriptor;
  sector_buffer[1]=0xff;
  sector_buffer[2]=0xff;

  if(param.fat_type==FAT16)
    sector_buffer[3]=0xff;

  if(param.fat_type==FAT32)
    {
    sector_buffer[3] =0x0f;
    sector_buffer[4] =0xff;
    sector_buffer[5] =0xff;
    sector_buffer[6] =0xff;
    sector_buffer[7] =0x0f;
    sector_buffer[8] =0xff;
    sector_buffer[9] =0xff;
    sector_buffer[10]=0xff;
    sector_buffer[11]=0x0f;
    }

  if(debug_prog) printf("[DEBUG]  Write FAT Headers To Sector->    %d\n",
                              file_sys_info.start_fat_sector);
  Drive_IO(WRITE,file_sys_info.start_fat_sector,1);

  if(debug_prog) printf("[DEBUG]  Write FAT Headers To Sector->    %d\n",
                        file_sys_info.start_fat_sector +
                        file_sys_info.number_fat_sectors);
  Drive_IO(WRITE,(file_sys_info.start_fat_sector
                  + file_sys_info.number_fat_sectors),1 );
}

void Write_Root_Directory()
{
  int index;
  int root_directory_first_sector;
  int root_directory_num_sectors;
  int space_fill;

  unsigned long sector;

  /* Clear Root Directory Area */
  Clear_Sector_Buffer();

  sector_buffer[0] = 0;  /* -ea */
  sector_buffer[32] = 0; /* -ea */


  sector=file_sys_info.start_root_dir_sect;
  root_directory_first_sector = (unsigned)sector;
  root_directory_num_sectors = (int)(file_sys_info.number_root_dir_sect);
  do
    {
    if (debug_prog==TRUE) printf("[DEBUG]  Clearing Root Directory Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);
    sector++;
    } while (sector<(root_directory_first_sector+root_directory_num_sectors));

  param.first_data_sector
   = root_directory_first_sector+root_directory_num_sectors+1;

  /* Add Volume Label to Root Directory */
  if(param.v==TRUE)
    {
    Clear_Sector_Buffer();
    sector=file_sys_info.start_root_dir_sect;
    index=0;
    space_fill=FALSE;
    do
      {
      if (param.volume_label[(index)]==0x00) space_fill=TRUE;

      if (space_fill==FALSE) sector_buffer[index]=param.volume_label[(index)];
      else sector_buffer[index]=' ';

      index++;
      } while (index<=10);

    sector_buffer[11]=0x08;

    if(debug_prog==TRUE) printf("[DEBUG]  Writing Volume Label To Root Directory Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);
    }
}
