/*
// Program:  Format
// Written By:  Brian E. Reifsnyder
// Version:  0.8
// Copyright:  1999 - 2001 under the terms of the GNU GPL, Version 2
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  SPECIAL
/////////////////////////////////////////////////////////////////////////////
*/

#define NAME "Format"
#define VERSION "0.8"

int debug_prog=0;            /* set to 1 for debug mode */

#define MAIN

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define UNUSED     99

#define READ       25
#define WRITE      26

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "bootcode.h"
#include "bpb.h"
#include "clstsize.h"
#include "format.h"

/* verify, the bootcode hasn't grown too much */
struct verify_boot_code_size {
    char x[1-2*( 62 + sizeof(boot_code) + 2 > 512 )];
    };

/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

/* Integers converted from long numbers */
int integer1;
int integer2;

union  REGS  regs;
struct SREGS sregs;

/* Buffers */
unsigned char sector_buffer[512];
unsigned char huge_sector_buffer[(512*32)];

/* The following lines were taken from partsize.c, written by Tom Ehlert. */
char secbuffer[512];

typedef unsigned long ULONG;
typedef unsigned long  __uint32;
typedef unsigned short __uint16;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

int Drive_IO(int command,unsigned long sector_number,int number_of_sectors);
int TE_AbsReadWrite(char DosDrive, int count, ULONG sector, void *buffer, unsigned ReadOrWrite);

void ASCII_CD_Number(unsigned long number);
void Ask_User_To_Insert_Disk(void);
void Clear_Sector_Buffer(void);
void Clear_Huge_Sector_Buffer(void);
void Clear_Sector_Buffer(void);
void Compute_Interleave_Factor(void);
void Compute_Sector_Skew(void);
void Confirm_Hard_Drive_Formatting(void);
void Convert_Long_To_Integer(long number);
void Create_File_System(void);
void Critical_Error_Handler(int error_code);
void Display_Drive_Statistics(void);
void Display_Invalid_Combination(void);
void Display_Help_Screen(void);
void Display_Percentage_Formatted(unsigned long percentage);
void Enable_Disk_Access();
void Format_Floppy_Cylinder(int cylinder,int head);
void Get_Device_Parameters(void);
void Get_DPB(void);
void Initialization(void);
void Save_File_System(void);
void Set_DPB_Access_Flag();
void Set_Floppy_Media_Type(void);
void Set_Hard_Drive_Media_Parameters(void);
void Setup_DDPT(void);
void Unconditional_Format(void);
void Unconditional_Floppy_Format(void);
void Unconditional_Hard_Disk_Format(void);
void Write_System_Files(void);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

void ASCII_CD_Number(unsigned long number)
{
  int comma_counter;
  int index;
  int right_index;
  int shift_counter;
  int shift_register;
  int shift_temp_register;
  int start_shift;


  index=0;

  do
    {
    ascii_cd_number[index]=0;

    index++;
    }while(index<14);

  ultoa(number,ascii_cd_number,10);

  /* Add Commas */
  index=13;
  right_index=13;
  start_shift=FALSE;

  do
    {
    if(ascii_cd_number[index]>0) start_shift=TRUE;

    if(start_shift==TRUE)
      {
      ascii_cd_number[right_index]=ascii_cd_number[index];
      ascii_cd_number[index]=0;
      right_index--;
      }

    index--;
    }while(index>=0);

  comma_counter=0;
  index=13;
  do
    {
    comma_counter++;

    if(ascii_cd_number[index]==0)
      {
      comma_counter=5;
      ascii_cd_number[index]=' ';
      }

    if(comma_counter==4)
      {
      shift_counter=index-1;
      shift_register=ascii_cd_number[index];
      ascii_cd_number[index]=',';
      do
	{
	shift_temp_register=ascii_cd_number[shift_counter];
	ascii_cd_number[shift_counter]=shift_register;
	shift_register=shift_temp_register;

	shift_counter--;
	}while(shift_counter>=0);

      comma_counter=0;
      }

    index--;
    }while(index>=0);

  ascii_cd_number[14]=0;
}

void Ask_User_To_Insert_Disk()
{
  printf(" Insert new diskette for drive %c:\n",param.drive_letter[0]);
  printf(" and press ENTER when ready...");

  /* Wait for a key */

  regs.h.ah = 0x08;
  intdos(&regs, &regs);

  printf("\n\n");
}

/* Clear Huge Sector Buffer */
void Clear_Huge_Sector_Buffer()
{
    memset(huge_sector_buffer, 0, sizeof(huge_sector_buffer));
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  memset(sector_buffer, 0, 512);
}


void Compute_Interleave_Factor()
{
  int index;
  int starting_sector;


  low_level.interleave_factor=drive_specs[param.media_type].interleave_factor;

  index=1;
  starting_sector=0;
  low_level.interleave_index=0;
  do
    {
    low_level.interleave_map[low_level.interleave_index]=index;
    low_level.interleave_index
     =low_level.interleave_index+low_level.interleave_factor;
    if(low_level.interleave_index>=drive_specs[param.media_type].sectors_per_cylinder)
      {
      starting_sector=starting_sector+1;
      low_level.interleave_index=starting_sector;
      }

    index++;
    }while(index<=drive_specs[param.media_type].sectors_per_cylinder);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Initial Interleave Map:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Compute_Sector_Skew()
{
  int carry;
  int index;
  int skew_counter;

  if(drive_specs[param.media_type].sector_skew==0) return;

  skew_counter=0;

  do
    {
    carry
     =low_level.interleave_map[(drive_specs[param.media_type].sectors_per_cylinder-1)];

    index=(drive_specs[param.media_type].sectors_per_cylinder-1);

    do
      {
      low_level.interleave_map[index]=low_level.interleave_map[(index-1)];

      index--;
      }while(index>0);

    low_level.interleave_map[0]=carry;

    skew_counter++;
    }while(skew_counter<drive_specs[param.media_type].sector_skew);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Interleave Map After Sector Skew:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Confirm_Hard_Drive_Formatting()
{

  printf("\n WARNING: ALL DATA ON NON-REMOVABLE DISK\n");
  printf(" DRIVE %c: WILL BE LOST!\n",param.drive_letter[0]);
  printf(" Proceed with Format (Y/N)?");

  /* Get keypress */
  regs.h.ah = 0x07;
  intdos(&regs, &regs);

  printf("\n");
  if( toupper(regs.h.al) != 'Y' ) exit(10);
  printf("\n");
}

/* Convert Huge number into 4 LMB integer values */
void Convert_Huge_To_Integers(unsigned long number)
{
  integer1=(unsigned)((number >> 0 ) & 0xff);
  integer2=(unsigned)((number >> 8 ) & 0xff);
  integer3=(unsigned)((number >> 16) & 0xff);
  integer4=(unsigned)((number >> 24) & 0xff);
}

/* Create File System */
void Create_File_System()
{
  int index;
  int root_directory_first_sector;
  int root_directory_num_sectors;
  int space_fill;

  unsigned char oem_id[8];

  unsigned long sector;


  /* Flush DOS disk buffers. */
  
  regs.h.ah = 0x0d;
  intdos(&regs,&regs);

  oem_id[0]='F';
  oem_id[1]='D';
  oem_id[2]='O';
  oem_id[3]='S';
  oem_id[4]='_';
  oem_id[5]='1';
  oem_id[6]='.';
  oem_id[7]='0';

  if(debug_prog==TRUE) printf("\n[DEBUG]  Creating File System\n");

  Clear_Sector_Buffer();

  /* Add Jump Instruction */
  sector_buffer[0]=0xeb;
  sector_buffer[1]=0x3c;
  sector_buffer[2]=0x90;

  /* Add OEM ID */
  memcpy(&sector_buffer[3], oem_id, 8);

  /* *** Add BPB */
  memcpy(&sector_buffer[11], &drive_specs[param.media_type].bytes_per_sector, 35-11+1);

  /* *** Add Extended BPB */
  if(param.media_type==HD) sector_buffer[36]=0x80;

  /* Add Signature Byte */
  sector_buffer[38]=0x29;

  /* Add Serial Number */
  randomize();
  sector_buffer[39]=rand();
  sector_buffer[40]=rand();
  sector_buffer[41]=rand();
  sector_buffer[42]=rand();

  drive_statistics.serial_number_low
   =(sector_buffer[39]) + (sector_buffer[40]<<8);

  drive_statistics.serial_number_high
   =(sector_buffer[41]) + (sector_buffer[42]<<8);

  /* Add Volume Label */
  if(param.v==TRUE)
    {
    memset( &sector_buffer[43], ' ', 11);   
        
    memcpy( &sector_buffer[43], param.volume_label, min(11, strlen(param.volume_label)));   
    }
  else
    {
    memcpy( &sector_buffer[43], "NO NAME    ", 11);
    }

  /* Add System ID */
  sector_buffer[54]='F';
  sector_buffer[55]='A';
  sector_buffer[56]='T';
  sector_buffer[57]='1';

  if(param.fat_type==FAT12) sector_buffer[58]='2';
  else sector_buffer[58]='6';

  sector_buffer[59]=' ';
  sector_buffer[60]=' ';
  sector_buffer[61]=' ';

  /* Add Boot Code */
  index=62;
  do
    {
    sector_buffer[index]=boot_code[(index-62)];

    index++;
    }while(index<=(130+62));

  /* Add Flag */
  sector_buffer[510]=0x55;
  sector_buffer[511]=0xaa;

  /* Write boot sector to the first sector of the disk */
  if(debug_prog==TRUE) printf("\n[DEBUG]  Writing Boot Sector->    0\n");
  Drive_IO(WRITE,0,1);

  /* Configure FAT Tables */
  Clear_Sector_Buffer();
  sector=1;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing FAT Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);

    sector++;
    }while(sector<(1+(2*drive_specs[param.media_type].sectors_per_fat)));

  sector_buffer[0]=drive_specs[param.media_type].media_descriptor;
  sector_buffer[1]=0xff;
  sector_buffer[2]=0xff;
  if(param.fat_type==FAT16) sector_buffer[3]=0xff;

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    1\n");
  Drive_IO(WRITE,1,1);

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    %3d\n",(1+drive_specs[param.media_type].sectors_per_fat));
  Drive_IO(WRITE,(1+drive_specs[param.media_type].sectors_per_fat),1 );

  /* Clear Root Directory Area */
  Clear_Sector_Buffer();
  sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
  root_directory_first_sector=(unsigned)sector;
  root_directory_num_sectors=drive_specs[param.media_type].root_directory_entries/(512/32);
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing Root Directory Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);
    sector++;
    }while(sector<(root_directory_first_sector+root_directory_num_sectors));

  /* Add Volume Label to Root Directory */
  if(param.v==TRUE)
    {
    Clear_Sector_Buffer();
    sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
    index=0;
    space_fill=FALSE;
    do
      {
      if(param.volume_label[(index)]==0x00) space_fill=TRUE;

      if(space_fill==FALSE) sector_buffer[index]=param.volume_label[(index)];
      else sector_buffer[index]=' ';

      index++;
      }while(index<=10);

    sector_buffer[11]=0x08;

    if(debug_prog==TRUE) printf("[DEBUG]  Writing Volume Label To Root Directory Sector->  %3d\n",sector);
    Drive_IO(WRITE,sector,1);
    }

  /* Flush DOS disk buffers so there is not any phantom information in */
  /* the directory.                                                    */
  regs.h.ah = 0x0d;
  intdos(&regs, &regs);
}

void Critical_Error_Handler(int error_code)
{
  if(error_code==2)                /* bad sector id mark              */
    {
    printf("\nBad sector ID\n");
    exit(1);
    }

  if(error_code==3)                /* write protect error             */
    {
    printf("\nWrite protect error\n");
    exit(1);
    }

  if(error_code==8)                /* DMA overrun                     */
    {
    printf("\nDMA overrun\n");
    exit(1);
    }

  if(error_code==9)                /* DMA error--64k boundary crossed */
    {
    printf("\nDMA error--64k boundary crossed\n");
    exit(1);
    }

  if(error_code==16)               /* CRC error                       */
    {
    printf("\nCRC error\n");
    exit(1);
    }

  if(error_code==32)               /* controller malfunction          */
    {
    printf("\nController malfunction\n");
    exit(1);
    }

  if(error_code==64)               /* seek failure                    */
    {
    printf("\nSeek failure\n");
    exit(1);
    }

  if(error_code==128)              /* time out                        */
    {
    printf("\nDevice timeout\n");
    exit(1);
    }

  if(error_code==255)
    {
    printf("\nInvalid drive specification\n");
    exit(1);
    }

  printf("\nUnknown error.  Error code:  %d\n",error_code);
  exit(1);

}

void Display_Drive_Statistics()
{
  drive_statistics.allocation_units_available_on_disk
   =drive_statistics.bytes_available_on_disk
   /drive_statistics.bytes_in_each_allocation_unit;


  /* The 2 lines below will eventually need to be re-located. */
  drive_statistics.bytes_total_disk_space
   =drive_statistics.allocation_units_available_on_disk
   *drive_statistics.bytes_in_each_allocation_unit;
  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  ASCII_CD_Number(drive_statistics.bytes_total_disk_space);
  printf("\n%13s bytes total disk space\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.bytes_available_on_disk);
  printf("%13s bytes available on disk\n",ascii_cd_number);

  printf("\n");

  ASCII_CD_Number(drive_statistics.bytes_in_each_allocation_unit);
  printf("%13s bytes in each allocation unit.\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.allocation_units_available_on_disk);
  printf("%13s allocation units available on disk.\n",ascii_cd_number);

  printf("\n");
  printf(" Volume Serial Number is %04X-%04X\n"
   ,drive_statistics.serial_number_high,drive_statistics.serial_number_low);
}

void Display_Invalid_Combination()
{
  printf("\n Invalid combination of options...please consult documentation.\n");
  printf(" Operation Terminated.\n");
  exit(4);
}

/* Help Routine */
void Display_Help_Screen()
{
  printf("\n%6s Version %s\n",NAME,VERSION);
  printf("Written By:  Brian E. Reifsnyder\n");
  printf("Copyright 1999 - 2001 under the terms of the GNU GPL, Version 2.\n\n");
  printf("Syntax:\n\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/F:size] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/T:tracks /N:sectors] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/1] [/4] [/B | /S]\n");
  printf("FORMAT drive: [/Q] [/U] [/1] [/4] [/8] [/B | /S]\n\n");
  printf(" /V:label   Specifies a volume label for the disk.\n");
  printf(" /Q         Quick formats the disk.  The disk can be UNFORMATed.\n");
  printf(" /U         Unconditionally formats the disk.  The disk cannot be UNFORMATted\n");
  printf(" /F:size    Specifies the size of the floppy disk to format.  Valid sizes are:\n");
  printf("              160, 180, 320, 360, 720, 1200, 1440, or 2880.\n");
  printf(" /B         Kept for compatibility, formally reserved space for the boot files.\n");
  printf(" /S         Copies the operating system files to make the disk bootable.\n");
  printf(" /T:tracks  Specifies the number of tracks on a floppy disk.\n");
  printf(" /N:sectors Specifies the number of sectors on a floppy disk.\n");
  printf(" /1         Formats a single side of a floppy disk.\n");
  printf(" /4         Formats a 360K floppy disk in a 1.2 MB floppy drive.\n");
  printf(" /8         Formats a 5.25\" disk with 8 sectors per track.\n");
}

void Display_Percentage_Formatted(unsigned long percentage)
{
  printf("%3d percent completed.\n",percentage);

  if(debug_prog!=TRUE)
    {
    /* Re-position cursor back to the beginning of the line */
//    printf("\r");       Didn't work...


    regs.h.ah = 0x0f;
    int86(0x10, &regs, &regs); /* Get current video display mode.           */

    regs.h.ah = 0x03;
    int86(0x10, &regs, &regs); /* Get cursor position.                      */

    regs.h.ah = 0x02;
    regs.h.dh = regs.h.dh--;
    int86(0x10, &regs, &regs); /* Set cursor position to beginning of line. */
    }
}

int Drive_IO(int command,unsigned long sector_number,int number_of_sectors)
{
  int return_code;

  return_code = 
       TE_AbsReadWrite(param.drive_number,number_of_sectors,sector_number,
           number_of_sectors==1 ?sector_buffer : huge_sector_buffer, 
           command);

  if(return_code!=0) Critical_Error_Handler(return_code);

  return (return_code);
}

void Enable_Disk_Access()
{
  unsigned long error_code=0;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Enable_Disk_Access() function\n");
    }

  /* Get the device parameters for the logical drive */

    regs.h.ah=0x44;                     /* IOCTL Block Device Request      */
    regs.h.al=0x0d;
    regs.h.bl=param.drive_number + 1;
    regs.h.ch=0x08;                     /* Always 0x08                     */
    regs.h.cl=0x67;                     /* Get Access Flags                */
    regs.x.dx=FP_OFF(&access_flags);
    sregs.ds =FP_SEG(&access_flags);

    intdosx(&regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag)
	{
	/* BO: if invalid function: try to format anyway maybe access
	       flags do not work this way in this DOS (e.g. DRDOS 7.03) */
        if (error_code == 0x1 || error_code == 0x16)
            return;

        /* Add error trapping here */
	printf("\nFatal error obtaining disk access flags...format terminated.\n", error_code);
	printf("Error Code:  %02x\n",error_code);
	exit(1);
	}

  if(access_flags.disk_access==0)
    {
    access_flags.disk_access++;

    regs.h.ah = 0x44;                     /* IOCTL Block Device Request          */
    regs.h.al = 0x0d;
    regs.h.bl = param.drive_number + 1;
    regs.h.ch = 0x08;                     /* Always 0x08                         */
    regs.h.cl = 0x47;                     /* Set device parameters               */
    regs.x.dx = FP_OFF(&access_flags);
    sregs.ds  = FP_SEG(&access_flags);
    intdosx( &regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag)
	{
      /* Add error trapping here */
	printf("\nFatal error writing setting disk access flags...format terminated.\n", error_code);
	printf("Error Code:  %02x\n",error_code);
	exit(1);
	}
    }

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Exit Exit_Enable_Disk_Access() function\n");
    }
}

void Format_Floppy_Cylinder(int cylinder,int head)
{
  int drive_number;

  unsigned int result;

  /* Set Up Track Address Fields */
  TAF track_address_fields[36];
  void *track_address_fields_p=track_address_fields;

  int index=0;
  do
    {
    track_address_fields[index].cylinder=cylinder;
    track_address_fields[index].head=head;
    track_address_fields[index].sector=low_level.interleave_map[index];
    track_address_fields[index].size_code=0x02;

    index++;
    }while(index<drive_specs[param.media_type].sectors_per_cylinder);

  drive_number=param.drive_number;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Formatting:  Cylinder->  %2d    Head->  %2d\n",cylinder,head);
    }

  /* Format the Track */
  regs.h.ah = 0x05;
  regs.h.ch = cylinder;
  regs.h.dh = head;
  regs.h.dl = drive_number;
  sregs.es  = FP_SEG(track_address_fields_p);
  regs.x.bx = FP_OFF(track_address_fields_p);
  int86x(0x13, &regs, &regs, &sregs); 
  result = regs.h.ah;

  if(result!=0) Critical_Error_Handler(result);

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Result From Interrupt 0x13, Service 0x05->  %X\n",result);
    }
}

void Get_Device_Parameters()
{
  unsigned long error_code=0;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Enter Get_Device_Parameters() function\n");
    }

  /* Get the device parameters for the logical drive */

    regs.h.ah=0x44;                     /* IOCTL Block Device Request          */
    regs.h.al=0x0d;
    regs.h.bl=param.drive_number + 1;
    regs.h.ch=0x08;                     /* Always 0x08                         */
    regs.h.cl=0x60;                     /* Get device parameters               */
    regs.x.dx=FP_OFF(&parameter_block);
    sregs.ds =FP_SEG(&parameter_block);

    intdosx(&regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag)
	{
      /* Add error trapping here */
	printf("\nCall to int 0x21, 0x440d 0x60, failed error %02x.\n", error_code);
	exit(1);
	}
#ifdef gdp_asss
  /* Remove this section because it is not needed. */

  if(parameter_block.all_sectors_same_size==0)
    {
    parameter_block.all_sectors_same_size++;

    regs.h.ah = 0x44;                     /* IOCTL Block Device Request          */
    regs.h.al = 0x0d;
    regs.h.bl = param.drive_number + 1;
    regs.h.ch = 0x08;                     /* Always 0x08                         */
    regs.h.cl = 0x40;                     /* Set device parameters               */
    regs.x.dx = FP_OFF(&parameter_block);
    sregs.ds  = FP_SEG(&parameter_block);
    intdosx( &regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag)
	{
      /* Add error trapping here */
	printf("\nCall to int 0x21, 0x440d 0x40, failed error %02x.\n", error_code);
	exit(1);
	}
    }
#endif
  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Exit Get_Device_Parameters() function\n");
    }
}

void Get_DPB()
{
    regs.h.ah = 0x32;
    regs.h.dl = param.drive_number + 1;
    intdosx(&regs, &regs, &sregs);

/*
    if (regs.x.cflag )
	{
	    printf("INT 0x21/0x32/drive=%c failed\n", param.drive_number+'A');
	    exit(1);
	}
*/
    dpb = MK_FP(sregs.ds,regs.x.bx);

    segread(&sregs);                    /* restore defaults */
}

void IllegalArg(char *argptr)
{
    printf("illegal argument <%s>\n", argptr);
    exit(1);
}

void Initialization()
{
  param.drive_letter[0]=NULL;
  param.volume_label[0]=NULL;

  param.drive_type=NULL;
  param.drive_number=NULL;
  param.fat_type=NULL;
  param.media_type=UNKNOWN;

  param.force_yes=FALSE;

  param.v=FALSE;
  param.q=FALSE;
  param.u=FALSE;
  param.f=FALSE;
  param.b=FALSE;
  param.s=FALSE;
  param.t=FALSE;
  param.n=FALSE;
  param.one=FALSE;
  param.four=FALSE;
  param.eight=FALSE;

  param.size=UNKNOWN;
  param.cylinders=0;
  param.sectors=0;

  segread(&sregs);

  Setup_DDPT();
}

void RequireColon(char *argptr)
{
    if (argptr[2] != ':')
	IllegalArg(argptr);
}

/* Save the old file system for possible recovery with unformat */
void Save_File_System()
{
  int bad_boot_sector=TRUE;
  int end_of_root_directory_flag;

  unsigned  loop=512;

  unsigned char mirror_map[5120];

  unsigned pointer=0;

  unsigned long offset_from_end=5;

  unsigned long destination_sector;
  unsigned long source_sector;

  unsigned long mirror_beginning;
  unsigned long mirror_map_beginning;
  unsigned long mirror_map_size;
  unsigned long mirror_size;
  unsigned long number_of_bytes_in_mirror_map;
  unsigned long number_of_logical_sectors_on_drive;
  unsigned long number_of_root_directory_entries;

  unsigned long beginning_of_fat;
  unsigned long beginning_of_root_directory;

  unsigned long sectors_per_fat;

  unsigned long end_of_root_directory;

  unsigned long temp_buffer_1=0;
  unsigned long temp_buffer_2=0;
  unsigned long temp_buffer_3=0;
  unsigned long temp_buffer_4=0;

  unsigned int sector_flag[] = {
  'A','M','S','E','S','L','I','F','V','A','S',
  'R','O','R','I','M','E','S','A','E','P'};

  unsigned int mirror_map_header[] = {
  ':','\\','M','I','R','R','O','R','.','F','I','L'};

  if(param.drive_type==HARD) offset_from_end=20;

  /* Get the boot sector, compute the FAT size, compute the root dir size,*/
  /* and get the end of the logical drive. */
  Drive_IO(READ,0,1);

  if(  (sector_buffer[510]==0x55) && (sector_buffer[511]==0xaa) )
    {
    temp_buffer_1=sector_buffer[17];
    temp_buffer_2=sector_buffer[18];
    number_of_root_directory_entries=temp_buffer_1|(temp_buffer_2<<8);

    temp_buffer_1=sector_buffer[19];
    temp_buffer_2=sector_buffer[20];
    number_of_logical_sectors_on_drive=temp_buffer_1|(temp_buffer_2<<8);

    if(number_of_logical_sectors_on_drive==0)
      {
      temp_buffer_1=sector_buffer[32];
      temp_buffer_2=sector_buffer[33];
      temp_buffer_3=sector_buffer[34];
      temp_buffer_4=sector_buffer[35];

      number_of_logical_sectors_on_drive=temp_buffer_1|(temp_buffer_2<<8)
       |(temp_buffer_3<<16)|(temp_buffer_4<<24);
      }

    temp_buffer_1=sector_buffer[22];
    temp_buffer_2=sector_buffer[23];
    sectors_per_fat=temp_buffer_1|(temp_buffer_2<<8);

    bad_boot_sector=FALSE;
    }

  /* If the boot sector is not any good, don't save the file system. */
  if(bad_boot_sector==FALSE)
    {
    printf(" Saving UNFORMAT information\n");

    /* Compute the beginning sector of the mirror map and the size of */
    /* the mirror image.     */
    mirror_size=1+sectors_per_fat+(number_of_root_directory_entries/16);

    mirror_map_size=(mirror_size/64)+1;

    mirror_beginning=(number_of_logical_sectors_on_drive-mirror_size)-offset_from_end;
    mirror_map_beginning=mirror_beginning-mirror_map_size;

    /* Compute the locations of the first FAT and the root directory */
    beginning_of_fat=sectors_per_fat+1;

    beginning_of_root_directory=(sectors_per_fat*2)+1;
    end_of_root_directory=beginning_of_root_directory+(number_of_root_directory_entries/16)-1;

    /* Write the mirror map pointer to the last sectors of the logical drive. */
    Clear_Sector_Buffer();

    Convert_Huge_To_Integers(mirror_map_beginning);

    sector_buffer[0]=integer1;
    sector_buffer[1]=integer2;
    sector_buffer[2]=integer3;
    sector_buffer[3]=integer4;

    pointer=4;

    do                                           /* Add pointer sector flag */
      {
      sector_buffer[pointer]=sector_flag[pointer-4];
      pointer++;
      }while(pointer<=24);

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  Writing mirror map pointer to sector->  %ld\n",
       (number_of_logical_sectors_on_drive-offset_from_end));
      printf("[DEBUG]  Writing mirror map beginning at sector->  %ld\n",mirror_map_beginning);
      }

    Drive_IO(WRITE,(number_of_logical_sectors_on_drive-offset_from_end),1);

    /* Create the mirror map and copy the file system to the mirror.  */
    Clear_Sector_Buffer();

    pointer=0;

    do                                           /* Clear mirror_map buffer */
      {
      mirror_map[pointer]=0;
      pointer++;
      }while(pointer<=5119);

    mirror_map[0]=param.drive_letter[0];

    pointer=1;

    do                                           /* Add mirror map header */
      {
      mirror_map[pointer]=mirror_map_header[pointer-1];
      pointer++;
      }while(pointer<=12);

					       /* Main mirror map creation */
					       /* and copying loop.        */
    pointer=84;
    source_sector=0;
    destination_sector=mirror_beginning;

    end_of_root_directory_flag=FALSE;
    number_of_bytes_in_mirror_map=0;

    do
      {
      if( (source_sector>0) && (source_sector<beginning_of_fat) )
       source_sector=beginning_of_fat;

      /* Copy mirror image one sector at a time */
      Drive_IO(READ,source_sector,1);

      Drive_IO(WRITE,destination_sector,1);

      /* Enter mapping information into mirror map buffer */

      Convert_Huge_To_Integers(source_sector);

      mirror_map[pointer+0]=integer1;
      mirror_map[pointer+1]=integer2;
      mirror_map[pointer+2]=integer3;
      mirror_map[pointer+3]=integer4;

      Convert_Huge_To_Integers(destination_sector);

      mirror_map[pointer+4]=integer1;
      mirror_map[pointer+5]=integer2;
      mirror_map[pointer+6]=integer3;
      mirror_map[pointer+7]=integer4;

      source_sector++;
      destination_sector++;
      pointer=pointer+8;
      number_of_bytes_in_mirror_map=pointer;

      if(source_sector>=end_of_root_directory) end_of_root_directory_flag=TRUE;

      }while(end_of_root_directory_flag==FALSE);

    /* Write trailer in mirror map */

    mirror_map[pointer+0]=0;
    mirror_map[pointer+1]=0;
    mirror_map[pointer+2]=0;
    mirror_map[pointer+3]=0;

    /* Write the mirror map onto the disk.   */

    pointer=0;
    destination_sector=mirror_map_beginning;

    do
      {
      loop=0;

      do                                         /* Load the sector buffer */
	{
	sector_buffer[loop]=mirror_map[pointer+loop];

	loop++;
	}while(loop<512);

      Drive_IO(WRITE,destination_sector,1);      /* Write the mirror map   */
						 /* sector.                */
      destination_sector++;
      pointer=pointer+512;
      }while(pointer < number_of_bytes_in_mirror_map);

    }
  else
    {
    printf(" Drive appears unformatted, UNFORMAT information not saved.\n");
    }
}

void Set_Floppy_Media_Type()
{
  int drive_number=param.drive_number;
  int number_of_cylinders;
  int sectors_per_cylinder;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Current Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }

  param.fat_type=FAT12;

  if(param.f==TRUE)
    {
    /* Standard Format Types */
    if(param.size==160) param.media_type=FD160;
    if(param.size==180) param.media_type=FD180;
    if(param.size==320) param.media_type=FD320;
    if(param.size==360) param.media_type=FD360;
    if(param.size==720) param.media_type=FD720;
    if(param.size==1200) param.media_type=FD1200;
    if(param.size==1440) param.media_type=FD1440;
    if(param.size==2880) param.media_type=FD2880;
    /* Non-Standard Format Types */
    if(param.size==400) param.media_type=FD400;
    if(param.size==800) param.media_type=FD800;
    if(param.size==1680) param.media_type=FD1680;
    if(param.size==3360) param.media_type=FD3360;
    if(param.size==1494) param.media_type=FD1494;
    if(param.size==1743) param.media_type=FD1743;
    if(param.size==3486) param.media_type=FD3486;
    if(param.size==1700) param.media_type=FD1700;
    }

  if(param.t==TRUE)
    {
    int index=0;
    do
      {
      if( (param.cylinders==drive_specs[index].cylinders)
       && (param.sectors==drive_specs[index].sectors_per_cylinder) )
	{
	param.media_type=index;
	index=20;
	}

      index++;
      }while(index<8);
    }

  drive_number=param.drive_number;

  if(param.media_type==UNKNOWN)
    {
    /* Attempt to automatically detect the media type */
    int drive_number=param.drive_number;
    int drive_type=0;

      regs.h.ah = 0x08;
      regs.h.dl = drive_number;
      int86(0x13, &regs, &regs);

      drive_type = regs.h.bl;

    if(drive_type==0x01)
      {
      param.size=360;
      param.media_type=FD360;
      }
    if(drive_type==0x02)
      {
      param.size=1200;
      param.media_type=FD1200;
      }
    if(drive_type==0x03)
      {
      param.size=720;
      param.media_type=FD720;
      }
    if(drive_type==0x04)
      {
      param.size=1440;
      param.media_type=FD1440;
      }
    if(drive_type==0x05)     /* Originally for floppy tape drives */
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    if(drive_type==0x06)
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    }

  number_of_cylinders=drive_specs[param.media_type].cylinders;
  sectors_per_cylinder=drive_specs[param.media_type].sectors_per_cylinder;

    regs.h.ah = 0x18;
    regs.h.ch = number_of_cylinders;
    regs.h.cl = sectors_per_cylinder;
    regs.h.dl = drive_number;

    int86(0x13, &regs, &regs);

    if (regs.x.cflag == 0)
        goto int_supported;


  ddpt->sectors_per_cylinder=sectors_per_cylinder;

  int_supported:

  if(param.media_type>HD)
    {
    /* Adjust for non-standard formats. */

    }

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector
   *(unsigned long)drive_specs[param.media_type].total_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);

  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;

  Compute_Interleave_Factor();

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Configured Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }
}

void Set_Hard_Drive_Media_Parameters()
{
  int index;
  int result;

  unsigned long number_of_sectors;

  param.media_type=HD;
  param.fat_type=FAT12;

  Get_Device_Parameters();
  Get_DPB();

  if(parameter_block.total_sectors!=0)
    {
    number_of_sectors = parameter_block.total_sectors;
    }
  else
    {
    number_of_sectors =  parameter_block.large_sector_count_high;
    number_of_sectors =  number_of_sectors<<16;
    number_of_sectors += parameter_block.large_sector_count_low;
    }

  if(number_of_sectors>32767) param.fat_type=FAT16;
  drive_specs[HD].total_logical_sectors=number_of_sectors;

  /* Copy the returned BPB information into drive_specs. */
  drive_specs[HD].sectors_per_cluster    =parameter_block.sectors_per_cluster;
  drive_specs[HD].sectors_per_fat        =parameter_block.sectors_per_fat;
  drive_specs[HD].root_directory_entries =parameter_block.root_directory_entries;
  drive_specs[HD].total_sectors          =parameter_block.total_sectors;
  drive_specs[HD].sectors_per_cylinder   =parameter_block.sectors_per_cylinder;
  drive_specs[HD].number_of_heads        =parameter_block.number_of_heads;
  drive_specs[HD].hidden_sectors_low     =parameter_block.hidden_sectors_low;
  drive_specs[HD].hidden_sectors_high    =0;
  drive_specs[HD].large_sector_count_low =parameter_block.large_sector_count_low;
  drive_specs[HD].large_sector_count_high=parameter_block.large_sector_count_high;

  param.size=number_of_sectors/2048;

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector*number_of_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);
  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;
}

void Setup_DDPT()
{
  /* Get the location of the DDPT */
    regs.h.ah =0x35;
    regs.h.al =0x1e;
    intdosx(&regs, &regs, &sregs);

    ddpt = MK_FP(sregs.es, regs.x.bx);
}

void Set_DPB_Access_Flag()
{
    dpb->dpb_access_flag=0xff;

    regs.h.ah = 0x32;
    regs.h.dl = param.drive_number + 1;
    intdosx(&regs, &regs, &sregs);

    segread(&sregs);                    /* restore defaults */
}

/* TE_AbsReadWrite() is written by Tom Ehlert. */
int TE_AbsReadWrite(char DosDrive, int count, ULONG sector, void *buffer, unsigned ReadOrWrite)
{
    struct {
        unsigned long  sectorNumber;
        unsigned short count;
        void far *address;
        } diskReadPacket;
    union REGS regs;


    diskReadPacket.sectorNumber = sector;
    diskReadPacket.count        = count;
    diskReadPacket.address      = buffer;

    regs.h.al = DosDrive;
    regs.x.bx = (short)&diskReadPacket;
    regs.x.cx = 0xffff;
    
    switch(ReadOrWrite)
        {
            case READ:  int86(0x25,&regs,&regs); break;
            case WRITE: int86(0x26,&regs,&regs); break;
	    default:
		printf("TE_AbsReadWrite wrong called %02x\n", ReadOrWrite);
                exit(1);
         }

    return regs.x.cflag ? regs.x.ax : 0;
}    

/* Unconditionally Format the Drive */
void Unconditional_Format()
{
  if(param.drive_type==FLOPPY) Unconditional_Floppy_Format();
  else Unconditional_Hard_Disk_Format();
}

void Unconditional_Floppy_Format()
{
  int drive_number=param.drive_number;
  int index=0;

  unsigned long percentage;

  /* Reset the floppy disk controller */
  
  regs.h.ah = 0x00;
  regs.h.dl = drive_number;
  int86( 0x13, &regs, &regs);

  do
    {
    if(drive_specs[param.media_type].number_of_heads==2)
     Format_Floppy_Cylinder(index,0);
    Format_Floppy_Cylinder(index,1);

    percentage
     =((100*index)/(drive_specs[param.media_type].cylinders));
    Display_Percentage_Formatted(percentage);

    Compute_Sector_Skew();

    index++;
    }while(index<=drive_specs[param.media_type].cylinders);
}

void Unconditional_Hard_Disk_Format()
{
  int error_code;
  int number_of_sectors=32;

  int percentage_refresh_rate=10;
  int percentage_refresh_counter=10;

  unsigned index=0;

  unsigned long sector_number=0;
  unsigned long max_logical_sector
   = drive_specs[param.media_type].total_logical_sectors;
  unsigned long percentage;

  /* Clear huge_sector_buffer */
  do
    {
    huge_sector_buffer[index]=0;

    index++;
    }while(index<(512*32));

  /* Clear sectors (maximum of 32 at a time) */
  do
    {
    error_code=Drive_IO(WRITE,sector_number, number_of_sectors);

    if(error_code>0)
      {
      if(error_code!=4) Critical_Error_Handler(error_code);
      else
	{
	/* Scan and Record Bad Sectors */

	}
      }

    if(percentage_refresh_counter==0)
      {
      percentage=(100*sector_number)/max_logical_sector;
      Display_Percentage_Formatted(percentage);
      }
    if(percentage_refresh_counter==percentage_refresh_rate)
     percentage_refresh_counter=-1;
    percentage_refresh_counter++;

    sector_number=sector_number+32;
    }while(sector_number<max_logical_sector);
}

/* Write System Files */
void Write_System_Files()
{
  char sys[9] = {'s','y','s',' ','x',':',13,0,0};

  sys[4]=param.drive_letter[0];

  system(sys); /**/
}


/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int loop=0;
  int index;

  Initialization();

  /* if FORMAT is typed without any options or with /? */
  if( (argc < 2) || (argv[1][1]=='?') )
    {
    Display_Help_Screen();
    exit(1);
    }

  if (!isalpha(argv[1] [0])       ||
               argv[1] [1] != ':' ||
               argv[1] [2] != 0)
    {
    printf("Required parameter missing -\n");
    exit(1);
    }

  param.drive_letter[0]=toupper(argv[1] [0]);
  param.drive_number = param.drive_letter[0] - 'A';
  param.drive_letter[1] = ':';


  /*if FORMAT is typed with a drive letter */


  if(debug_prog==TRUE) printf("\n[DEBUG]  Drive To Format->  %s \n\n",param.drive_letter);

    /* Set the type of disk */
  if(param.drive_number>1) param.drive_type=HARD;
  else param.drive_type=FLOPPY;

  for ( index = 2; index < argc; index++)
  {
    char *argptr = argv[index];

	/* Determine which switches were entered */

    if (*argptr != '/' && *argptr != '-')
    {
        IllegalArg(argptr);
    }

    switch(toupper(argptr[1]))
    {
	case 'V':
    	    param.v=TRUE;
    	    
    	    RequireColon(argptr);
    
            for (loop = 0; loop < 11; loop++)
    	        param.volume_label[loop]=toupper(argptr[loop+3]);
    
            break;
    
        case 'Q': param.q =TRUE;  	      break;
    
    	case 'U': param.u=TRUE;   	      break;
    	
    	case 'B': param.b=TRUE;    	      break;
    
        case 'S': param.s=TRUE;    	      break;
    
    	case 'Y': param.force_yes=TRUE;   break;
    
        case '1': param.one=TRUE;    	  break;
        case '4': param.four=TRUE;    	  break;
        case '8': param.eight=TRUE;    	  break;
    	    
    
        case 'F':           /* /F:size */
    	  param.f=TRUE;
    	  RequireColon(argptr);
	       if( ('1'==argptr[3]) && ('6'==argptr[4]) ) param.size=160;
	  else if( ('1'==argptr[3]) && ('8'==argptr[4]) ) param.size=180;
    	  else if( ('3'==argptr[3]) && ('2'==argptr[4]) ) param.size=320;
    	  else if( ('3'==argptr[3]) && ('6'==argptr[4]) ) param.size=360;
	  else if( ('7'==argptr[3]) ) param.size=720;
    	  else if( ('1'==argptr[3]) && ('2'==argptr[4]) ) param.size=1200;
	  else if( ('1'==argptr[3]) && ('2'==argptr[5]) ) param.size=1200;
	  else if( ('1'==argptr[3]) && ('4'==argptr[4]) ) param.size=1440;
    	  else if( ('1'==argptr[3]) && ('4'==argptr[5]) ) param.size=1440;
    	  else if( ('2'==argptr[3]) ) param.size=2880;
    	  else IllegalArg(argptr);
    	  
    	  break;
    	  
        case 'T':
    	  param.t=TRUE;
   	      RequireColon(argptr);
    	  if('4'==argptr[3]) param.cylinders=40;
    	  else if('8'==argptr[3]) param.cylinders=80;
    	  else IllegalArg(argptr);
    	  break;
    
        case 'N':
    	  param.n=TRUE;
    	  RequireColon(argptr);
    	  if('8'==argptr[3]) param.sectors=8;
    	  else if('9'==argptr[3]) param.sectors=9;
    	  else if( ('1'==argptr[3]) && ('5'==argptr[4]) ) param.sectors=15;
    	  else if( ('1'==argptr[3]) && ('8'==argptr[4]) ) param.sectors=18;
    	  else if('3'==argptr[3]) param.sectors=36;
    	  else IllegalArg(argptr);
          break;
    
        default:
    	  IllegalArg(argptr);
    	  break;

    }       /* switch (*argptr) */
  }         /* for all args     */


    /* Ensure that valid switch combinations were entered */
    if( (param.b==TRUE) && (param.s==TRUE) ) Display_Invalid_Combination();
    if( (param.v==TRUE) && (param.eight==TRUE) ) Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && ( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) ) ) Display_Invalid_Combination();
    if( ( (param.t==TRUE) && (param.n!=TRUE) ) || ( (param.n==TRUE) && (param.t!=TRUE) ) ) Display_Invalid_Combination();
    if( (param.f==TRUE) && ( (param.t==TRUE) || (param.n==TRUE) ) )Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && (param.eight==TRUE) )Display_Invalid_Combination();
    if( ( (param.four==TRUE) || (param.eight==TRUE) ) && (param.one==TRUE) )Display_Invalid_Combination();
    if( ( (param.eight==TRUE) || (param.one==TRUE) ) && (param.four==TRUE) )Display_Invalid_Combination();

    /* Make sure that a floppy disk is unconditionally formatted if /f /t */
    /* or /n are selected.                                                */
    if( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) )
      {
      param.u=TRUE;
      param.q=FALSE;
      }

    /* Set the media parameters */
    if(param.drive_type==FLOPPY)
      {
      if(param.force_yes==FALSE) Ask_User_To_Insert_Disk();
      Set_Floppy_Media_Type();
      }
    else
      {
      if(param.force_yes==FALSE) Confirm_Hard_Drive_Formatting();
      Set_Hard_Drive_Media_Parameters();
      Enable_Disk_Access();
      }

/* ********************************* */
/* THIS WILL ME REMOVED, EVENTUALLY. */
/* ********************************* */

/* Temporary code to force a hard drive format to be as /U /Q.  */
if(param.drive_type==HARD)
  {
  param.u=TRUE;
  param.q=TRUE;
  }

/* ********************************* */
/* ********************************* */

    /* Format Drive */
    if( (param.u==TRUE) && (param.q==FALSE) )
      {
      /* Unconditional Format */
      Unconditional_Format();
      Create_File_System();
      }

    if( (param.u==FALSE) && (param.q==TRUE) )
      {
      /* Quick Format */
      if(param.drive_type==HARD) Save_File_System();
      Create_File_System();
      }

    if( (param.u==TRUE) && (param.q==TRUE) )
      {
      /* Quick Unconditional format */
      printf(" QuickFormatting \n");
      Create_File_System();
      }

    if( (param.u==FALSE) && (param.q==FALSE) )
      {
      /* Safe Format */
      if(param.drive_type==HARD) Save_File_System();
      Create_File_System();
      }

    if(param.drive_type==HARD) Set_DPB_Access_Flag();

    printf(" Format complete.          \n");

    if(param.s==TRUE) Write_System_Files();

    Display_Drive_Statistics();

    exit(0);
}

