/*
// Program:  Format
// Version:  0.91e
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module Name:  hdisk.c
// Module Description:  Hard Drive Specific Functions
*/

#define HDISK


#include <stdlib.h>

#include "floppy.h"
#include "format.h"
#include "createfs.h"
#include "btstrct.h"
#include "hdisk.h"


void Get_Device_Parameters()
{
  unsigned long error_code = 0;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Enter Get_Device_Parameters() function\n");
    }

  parameter_block.bytes_per_sector = 0;		/* *** */
  parameter_block.sectors_per_cluster = 0;	/* *** */
  parameter_block.number_of_fats = 0;		/* *** */

  /* Get the device parameters for the logical drive */

  regs.h.ah=0x44;                     /* IOCTL Block Device Request          */
  regs.h.al=0x0d;
  regs.h.bl=param.drive_number + 1;
  regs.h.ch=0x08;                     /* Always 0x08...unless fs is FAT32                         */
  regs.h.cl=0x60;                     /* Get device parameters               */
  regs.x.dx=FP_OFF(&parameter_block);
  sregs.ds =FP_SEG(&parameter_block);
  parameter_block.use_current_bpb = 0;
  parameter_block.use_track_layout_fields = 0;
  parameter_block.all_sectors_same_size = 1;
  parameter_block.reserved = 0;
  parameter_block.sectors_per_fat = 0xffff; /* *** */
  intdosx(&regs, &regs, &sregs);

  error_code = regs.h.al;

  if ( (regs.x.cflag != 0) ||
       (parameter_block.sectors_per_fat == 0xffff) )
    {
    /* Add error trapping here */
    printf("\nCall to int 0x21, 0x440d 0x60, failed error %02x.\n", error_code);
    exit(1);
    }

  if (parameter_block.sectors_per_fat == 0)
    /* FAT32 stores 0 in this WORD and uses a DWORD elsewhere instead */
    {
    /* Ok, this is assumed to be a FAT32 partition and it will be       */
    /* formatted as such.                                               */

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  FAT32 detected.\n");
      }

    param.fat_type = FAT32;

    /* Get the device parameters for the logical drive */

    regs.h.ah=0x44;                     /* IOCTL Block Device Request          */
    regs.h.al=0x0d;
    regs.h.bl=param.drive_number + 1;
    regs.h.ch=0x48;                     /* Always 0x48 for FAT32                         */
    regs.h.cl=0x60;                     /* Get device parameters               */
    regs.x.dx=FP_OFF(&parameter_block);
    sregs.ds =FP_SEG(&parameter_block);
    parameter_block.use_current_bpb = 0;
    parameter_block.use_track_layout_fields = 0;
    parameter_block.all_sectors_same_size = 1;
    parameter_block.reserved = 0;
    parameter_block.sectors_per_fat = 1; /* *** */

    intdosx(&regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag || (parameter_block.sectors_per_fat == 1))
      {
      /* Add error trapping here */
      printf("\nCall to int 0x21, 0x440d 0x60, failed error %02x.\n", error_code);
      exit(1);
      }

    }

  if (error_code!=0)
    printf("Error code: %ul,\n", error_code);
  if (debug_prog==TRUE)
    printf("[DEBUG]  Exit Get_Device_Parameters() function\n");

}


void Get_DPB()
{
  if(param.fat_type!=FAT32)
    {
    regs.h.ah = 0x32;

    regs.h.dl = param.drive_number + 1;
    intdosx(&regs, &regs, &sregs);

    if (regs.h.al != 0)
	{
	    printf("INT 0x21/0x32/drive=%c failed\n", param.drive_number+'A');
	    printf("*** Drive not formatted??? Aborting! ***\n");
	    exit(1);
	    /* should not abort here, but rather create our own DPB */
	    /* *** TODO!!! *** */
	}

    dpb = MK_FP(sregs.ds,regs.x.bx);

    segread(&sregs);                    /* restore defaults */
    }
  else /* FAT32 case */
    {
    unsigned int drive_number = param.drive_number +1;

    unsigned int ext_param_block_seg = FP_SEG(&ext_parameter_block);
    unsigned int ext_param_block_off = FP_OFF(&ext_parameter_block);
    ext_parameter_block.buffer_length = 63; /* *** size for Win95 *** */
    ext_parameter_block.ext_dpb.bytes_per_sector = 1; /* impossible value */

    { /* limit variable scope of sregs */
      struct SREGS sregs;
      /* reactivated old Turbo C 2.01 style code -ea */
      /* also fixed that code. Turbo C 2.01 has no asm { ... } -ea */
      regs.x.ax = 0x7302; /* get FAT32 extended DPB */
      regs.x.cx = 0x003f; /* size of the buffer */

      sregs.es = ext_param_block_seg;
      regs.x.di = ext_param_block_off;

      regs.x.si = 0xf1a6; /* magic value to make dev-pointer valid */
                          /* and to make next-dpb pointer valid... */

      regs.h.dl = drive_number;
      intdosx(&regs, &regs, &sregs);
    }
   
    if (regs.x.cflag || /* non-FAT32 DOSes will not set carry here */
        (ext_parameter_block.ext_dpb.bytes_per_sector != 512))
	{
	    printf("INT 0x21/0x7302/drive=%c failed\n", param.drive_number+'A');
	    printf("Could not get FAT32 extended DPB. Aborting!\n");
	    exit(1);
	    /* *** TODO: calculate your own DPB instead! *** */
	}

    dpb = MK_FP(sregs.ds,regs.x.bx+2);	/* *** +2 ! *** */

    segread(&sregs);			/* restore defaults */
    }
}

void Set_DPB_Access_Flag()
{

  if(param.fat_type!=FAT32)
    {
    dpb->dpb_access_flag=0xff;

    regs.h.ah = 0x32;
    regs.h.dl = param.drive_number + 1;
    intdosx(&regs, &regs, &sregs);

    segread(&sregs);                    /* restore defaults */
    }
  else
    {
    unsigned char drive_no;

    unsigned dpb_format_struct_high;
    unsigned dpb_format_struct_low;

    struct
      {
      unsigned int  size;
      unsigned int  structure_version;
      unsigned long function_number;

      unsigned long field_1;
      unsigned long field_2;
      unsigned long field_3;
      unsigned long field_4;
      }dpb_for_format_structure;

    dpb_for_format_structure.size              = 0x18;
    dpb_for_format_structure.structure_version = 0x00;
    dpb_for_format_structure.function_number   = 0x02;
    dpb_for_format_structure.field_1           = 0x0;
    dpb_for_format_structure.field_2           = 0x0;
    dpb_for_format_structure.field_3           = 0x0;
    dpb_for_format_structure.field_4           = 0x0;

    drive_no=param.drive_number + 1;

    dpb_format_struct_high = FP_SEG(&dpb_for_format_structure);
    dpb_format_struct_low  = FP_OFF(&dpb_for_format_structure);

    /* Turbo C 2.01 has no asm { ... } -ea */
    { /* limit variable scope for r and s */
      union REGS r;
      struct SREGS s;
      r.x.ax = 0x7304;
      r.h.dl = drive_no;
      s.es = dpb_format_struct_high;
      r.x.di = dpb_format_struct_low;
      intdosx(&r, &r, &s);
    }

    }
}

/*
 * Do everything to find a correct BPB and stuff.
 * Added extra sanity checks in 0.91e...
 * FAT type detection improved, too.
 */
void Set_Hard_Drive_Media_Parameters()
{
  /* int index; */
  /* int result; */
  int i;

  unsigned long number_of_sectors;

  param.media_type = HD;
  param.fat_type = FAT12; /* assume FAT12 first */

  Get_Device_Parameters();
  Get_DPB();

  if (parameter_block.total_sectors!=0) /* 16bit value? */
    {
    number_of_sectors = parameter_block.total_sectors;
    }
  else /* else 32bit value */
    {
    number_of_sectors =  parameter_block.large_sector_count_high;
    number_of_sectors =  number_of_sectors<<16;
    number_of_sectors += parameter_block.large_sector_count_low;
    }

  drive_specs[HD].total_logical_sectors=number_of_sectors;

  /* Copy the returned BPB information into drive_specs. */
  memcpy(&drive_specs[HD].bytes_per_sector,&parameter_block.bytes_per_sector,25);

  /* Copy BPB into bpb_standard structure. */
  memcpy(&bpb_standard,&parameter_block.bytes_per_sector,25);

  /* If the file system is FAT32, copy the FAT32 BPB extension */
  if (param.fat_type==FAT32) /* Get_Device_Parameters or Get_DPB can set this */
    memcpy(&bpb_fat32_ext,&parameter_block.sectors_per_fat_low,27);

  if (parameter_block.bytes_per_sector != 512) /* SANITY CHECK */
    {
    printf("Not 512 bytes per sector - aborting.\n");
    exit(1);
    }

  if ((parameter_block.number_of_fats<1) ||
      (parameter_block.number_of_fats>2)) /* SANITY CHECK */
    {
    printf("Not 1 or 2 FATs - aborting.\n");
    exit(1);
    }

  i = parameter_block.sectors_per_cluster;
  while ((i < 64) && (i != 0)) 
    i += i;	/* shift left until 32k / cluster */
  if (i != 64) /* SANITY CHECK */
    {
    i = parameter_block.sectors_per_cluster;
    printf("FATAL: Cluster size not 0.5, 1, 2, 4, 8, 16 or 32k but %d.%dk!\n",
      i/2, (i & 1) ? 5 : 0);
    exit(1);
    }

  Get_FS_Info(); /* create FS info from BPB */
  /* calculates things like FAT and cluster size */
  /* e.g. total_clusters is created from 16 or 32bit drive size etc. */

  if (file_sys_info.total_clusters > 65525L)
    {
    if (param.fat_type!=FAT32)
      printf("Almost formatted FAT32 drive as FAT1x, phew...\n");
    param.fat_type = FAT32;
    }
  else
    {
    if (param.fat_type==FAT32)
      printf("Almost formatted FAT1x drive as FAT32, phew...\n");
    param.fat_type = (file_sys_info.total_clusters>=4085) ? FAT16 : FAT12;
    }

  param.size=number_of_sectors/2048;

  drive_statistics.bytes_total_disk_space
    = ((unsigned long)drive_specs[param.media_type].bytes_per_sector*number_of_sectors)
    - ((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
    + ((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
    * (unsigned long)drive_specs[param.media_type].bytes_per_sector);
  drive_statistics.bytes_available_on_disk
    = drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
    = (unsigned long)drive_specs[param.media_type].sectors_per_cluster
    * (unsigned long)drive_specs[param.media_type].bytes_per_sector;
   
  printf("Formatting as FAT%d.\n",
    (param.fat_type==FAT32) ? 32 : ( (param.fat_type==FAT16) ? 16 : 12 ) );
}
