/*
// Program:  Format
// Version:  0.90f
// (0.90b/c/d - DMA in track_address_fields fix - Eric Auer May 2003)
// (0.90e/f - more formats, better media type setting - Eric Auer)
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module:  floppy.c
// Description:  Floppy disk specific functions.
*/

#define FLOP

#include <stdlib.h>

#include "floppy.h"
#include "format.h"
#include "btstrct.h"
#include "userint.h" /* Critical_* */
#include "driveio.h" /* huge_sector_buffer* */

void Compute_Interleave_Factor(void);

void Compute_Interleave_Factor()
{
  int index;
  int starting_sector;


  low_level.interleave_factor = drive_specs[param.media_type].interleave_factor;

  index = 1;
  starting_sector = 0;
  low_level.interleave_index = 0;
  do
    {
    low_level.interleave_map[low_level.interleave_index]=index;
    low_level.interleave_index
     =low_level.interleave_index+low_level.interleave_factor;
    if ( low_level.interleave_index >=
         drive_specs[param.media_type].sectors_per_cylinder )
      {
      starting_sector = starting_sector + 1;
      low_level.interleave_index = starting_sector;
      }

    index++;
    } while (index <= drive_specs[param.media_type].sectors_per_cylinder);

  if ((debug_prog==TRUE)
#if 0
 && (low_level.interleave_factor != 1)
#endif
     )
    {
    printf("\n[DEBUG]  Initial Interleave Map:  \n          ");

    index = 0;
    do
      {
      printf("%2d ",low_level.interleave_map[index]);
      index++;
      } while (index <= (drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Compute_Sector_Skew()
{
  int carry;
  int index;
  int skew_counter;

  if (drive_specs[param.media_type].sector_skew==0)
    return;

  skew_counter = 0;

  do
    {
    carry =
      low_level.interleave_map[(drive_specs[param.media_type].sectors_per_cylinder-1)];

    index = (drive_specs[param.media_type].sectors_per_cylinder - 1);

    do
      {
      low_level.interleave_map[index] = low_level.interleave_map[(index-1)];

      index--;
      } while (index>0);

    low_level.interleave_map[0] = carry;

    skew_counter++;
    } while (skew_counter < drive_specs[param.media_type].sector_skew);

  if (debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Interleave Map After Sector Skew:  \n          ");

    index = 0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      } while (index <= (drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}



#define TAFinSEC 1 /* use 1 sector buffer to hold 36*4 bytes of TAF */

/* this saves stack space and avoids other hassles, but you may not */
/* use DriveIO of single sectors unless you accept buffer overwrite */


/* now returns the bad-sector-count -ea */
int Format_Floppy_Cylinder(int cylinder,int head) 
{
  int drive_number;
  int sector_index;
  int retry_count;
  int index;
  int secPerTrack=0;
  int badSecCount=0;

  unsigned int result;
  unsigned int result2;

  /* Set Up Track Address Fields */
#ifdef TAFinSEC
  TAF far *track_address_fields_p = (TAF far *) sector_buffer;
  /* use this buffer to save stack -ea */

#else
>  TAF track_address_fields[72]; /* 2 * 36 used, allow moving -ea */
>  int taf_which = 0;
>  TAF far *track_address_fields_p = &track_address_fields[taf_which];
>  unsigned long int where;
>
>  where = FP_SEG(track_address_fields_p);
>  where <<= 4;
>  where += FP_OFF(track_address_fields_p);
>
>  /* start DMA boundary avoidance -ea */
>  while ((where & 0xffff) > (0xffff - (4*36))) {
>    taf_which++;
>    track_address_fields_p = &track_address_fields[taf_which];
>    where = FP_SEG(track_address_fields_p);
>    where <<= 4;
>    where += FP_OFF(track_address_fields_p);
>    if(debug_prog==TRUE) {
>      printf(">f>");
>    }
>  }
>  if(debug_prog==TRUE) {
>    printf("[DEBUG]  track_address_fields_p = %4.4x:%4.4x\n",
>      FP_SEG(track_address_fields_p),FP_OFF(track_address_fields_p));
>  }
#endif

  /* DMA boundary avoided -ea */

  secPerTrack = drive_specs[param.media_type].sectors_per_cylinder;
  index = 0;
  do
    {
    (track_address_fields_p+index)->cylinder = cylinder;
    (track_address_fields_p+index)->head = head;
    (track_address_fields_p+index)->sector = low_level.interleave_map[index];
    (track_address_fields_p+index)->size_code = 0x02; /* 128<<2: 512by */

    index++;
    } while (index<secPerTrack);

  drive_number = param.drive_number;

  if (debug_prog==TRUE)
    {
    printf("[DEBUG]  Formatting: Cylinder: %2d  Head: %2d  Sectors: %2d\n",
      cylinder, head, secPerTrack);
    }

  /* Format the Track */
  result = 0;
  regs.h.ah = 0x05;
  regs.h.al = secPerTrack; /* feels better -ea */
  regs.h.ch = cylinder;
  regs.h.dh = head;
  regs.h.dl = drive_number;
  sregs.es  = FP_SEG(track_address_fields_p);
  regs.x.bx = FP_OFF(track_address_fields_p);
  int86x(0x13, &regs, &regs, &sregs);
  result = regs.h.ah;

  if (result!=0) {
    printf("Format_Floppy_Cylinder( head=%d cylinder=%d ) sectors=%d [int 13.5]\n",
      head, cylinder, secPerTrack );
    Critical_Error_Handler(BIOS,result);
  }


  result2 = result;

  if (param.verify==TRUE)
    {
    /* Verify the Track */
    /* According to RBIL, this uses cached CRC values, but it  */
    /* might use the ES:BX data buffer on 1985/before BIOSes!? */
    /* -> SHOULD we use READ to huge_sector_buffer instead???  */
    /* Warning: huge_sector_buffer is only 18 sectors for now. */
    result=0;
    regs.h.ah = 0x02; /* changed from VERIFY to READ */
    regs.h.al = secPerTrack;
    if ( regs.h.al > (sizeof(huge_sector_buffer_0)>>9) )
      {
        regs.h.al = sizeof(huge_sector_buffer_0) >> 9;
        printf("Only checking first %d sectors per track\n", regs.h.al);
        /* could read 2 half-tracks or use pinpointing below here     */
        /* instead... however, only 2.88MB disks have > 18 sect/track */
      }
    regs.h.ch = cylinder;
    regs.h.cl = 1;                     /* Start with the first sector. */
    regs.h.dh = head;
    regs.h.dl = drive_number;
    regs.x.bx = FP_OFF(huge_sector_buffer); /* if using READ */
    sregs.es  = FP_SEG(huge_sector_buffer); /* if using READ */
    int86x(0x13, &regs, &regs, &sregs);
    result = regs.h.ah;

    if ( (debug_prog==TRUE) && ((result!=0) || (result2!=0)) )
      {
      printf("[DEBUG]  Intr 0x13 results: FORMAT -> %X, VERIFY -> %X\n",
        result2, result);
      }

    if (result!=0)
      {
      if (debug_prog==TRUE)
	{
	printf("[DEBUG]  Errors found, pinpointing bad sectors now...\n");
	}

      retry_count  = 0;
      sector_index = 1;

      do
	{
retry:
	if (debug_prog==TRUE)
	  {
	  printf("[DEBUG]  Scanning sector number->  %d\n",sector_index);
	  }

	/* Verify the Track...sector by sector. */
	/* According to RBIL, this uses cached CRC values, but it  */
	/* might use the ES:BX data buffer on 1985/before BIOSes!? */
	/* -> SHOULD we use READ to sector_buffer instead???       */
	result = 0;
	regs.h.ah = 0x02; /* changed from 4 VERIFY to 2 READ */
	regs.h.al = 1;
	regs.h.ch = cylinder;
	regs.h.cl = sector_index;
	regs.h.dh = head;
	regs.h.dl = drive_number;
        regs.x.bx = FP_OFF(huge_sector_buffer); /* if using READ */
        sregs.es  = FP_SEG(huge_sector_buffer); /* if using READ */
	int86x(0x13, &regs, &regs, &sregs);
	result = regs.h.ah;

	if (result!=0)
	  {
	  if (debug_prog==TRUE)
	    {
	    printf("[DEBUG]  Possible bad sector found... testing.  retry_count->  %d\n",retry_count);
	    }

	  retry_count++;

	  if (retry_count>=3)
	    {
	    /* Record this sector as bad. */
	    bad_sector_map[bad_sector_map_pointer] =
	      ( (cylinder * (drive_specs[param.media_type].number_of_heads) )
	      * drive_specs[param.media_type].sectors_per_cylinder)
	      + (head * drive_specs[param.media_type].sectors_per_cylinder)
	      + sector_index;

	    bad_sector_map_pointer++;
	    drive_statistics.bytes_in_bad_sectors =
	      drive_statistics.bytes_in_bad_sectors +
	      drive_specs[param.media_type].bytes_per_sector;

	    if (debug_prog==TRUE)
	      {
	      printf("[DEBUG]  Bad Sector %4ld CHS=[%2d:%d:%2d] on drive %X\n",
	        bad_sector_map[(bad_sector_map_pointer-1)],
	        cylinder, head, sector_index, drive_number);
	      }
	    else
	      {
	      printf("Sector %4ld CHS=[%2d:%d:%2d] bad\n",
	        bad_sector_map[(bad_sector_map_pointer-1)],
	        cylinder, head, sector_index);
	      }

	    retry_count = 0;
            badSecCount++;
	    }
	  else
            {

            regs.x.ax = 0;
            regs.h.dl = drive_number;
            int86(0x13, &regs, &regs); /* reset drive */

            goto retry;   /* Yes, it's a goto.  :-)  Even goto has uses. */

            }
	  }

	sector_index++;
	} while (sector_index<=drive_specs[param.media_type].sectors_per_cylinder);

      }
    }

  return badSecCount;

}


/* void ddptPrinter(void); */


void ddptPrinter(void)
{
    /* note that the default step rate 15*16ms is pretty long */
    printf("[DEBUG]      Step Rate:                 %d msec\n",
      ddpt->step_rate * 16);               /* 1st byte, LO4 */

    /* this defaults to pretty long as well */
    printf("[DEBUG]      Head Unload Time:          %d msec\n",
      (16 - ddpt->head_unload_time) << 4); /* 1st byte, HI4 */

    /* default is 0 - always 0? */
    printf("[DEBUG]      DMA Flag:                  %d\n",
      ddpt->dma_flag);                     /* 2nd byte, LO1 */

    /* default unknown */
    printf("[DEBUG]      Head Load Time:            %d msec\n",
      ddpt->head_load_time << 1);          /* 2nd byte, HI7 */

    /* default is 2.035 sec */
    printf("[DEBUG]      Motor Post Rotating Time:  %d msec\n",
      ddpt->post_rt_of_disk_motor * 55); /* unit is timer ticks */

    /* default is 512 */
    printf("[DEBUG]      Sector Size:               %d bytes\n",
      128 << ddpt->sector_size);

    /* default depends on drive type */
    printf("[DEBUG]  *** Sectors Per Cylinder: ***  %d\n",
      ddpt->sectors_per_cylinder);

    /* better NOT MESS with this */
    printf("[DEBUG]   () Intersector Gap Length:    %d\n",
      ddpt->gap3_length_rw);

    /* better NOT MESS with this */
    printf("[DEBUG]   () Data Length:               %d\n",
      ddpt->dtl);

    /* better NOT MESS with this */
    printf("[DEBUG]   () Int.sect. Gap Len. Format: %d\n",
      ddpt->gap3_length_xmat);

    /* default 246 */
    printf("[DEBUG]      Fill Character:            0x0%x (%c)\n",
      ddpt->fill_char_xmat, ddpt->fill_char_xmat);

    /* default 25ms */
    printf("[DEBUG]      Head Settle Time:          %d msec\n",
      ddpt->head_settle_time);

    /* default BIOS 0.5 sec, DOS 0.25 sec */
    printf("[DEBUG]      Motor start-up time:       %d msec\n",
      ddpt->run_up_time * 125);
}


void Set_Floppy_Media_Type()
{
  int drive_number = param.drive_number;
  int number_of_cylinders;
  int sectors_per_cylinder;
  int drive_type = 0;


  param.media_type = UNKNOWN;

  if ((drive_number & 0x80) != 0)
    {
    printf("Drive number > 127: This is no floppy! Aborting.\n");
    exit(1);
    }

  regs.x.ax = 0;
  regs.h.dl = drive_number;
  int86(0x13, &regs, &regs); /* reset drive */

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Current Disk Drive Parameter Table Values:\n");
    ddptPrinter();
    }

  param.fat_type = FAT12;

  if (param.one==TRUE)
    {
    /* *** (this is only for 160k and 180k formats) *** */
    param.cylinders = 40;
    if ((param.sectors!=9) && (param.sectors!=8))
      {
      param.sectors = 9;
      }

    param.sides = 1;
    param.t = TRUE; /* trigger media type search */
    }
  else
    {
    param.sides = 2; /* the default (initialize value!) */
    }

  if (param.four==TRUE)
    {
    /* (also need special SETUP for 360k disk in 1200k drive) */
    param.sectors = 9;
    param.cylinders = 40;
    param.t = TRUE; /* trigger media type search */
    }

  if (param.eight==TRUE)
    {
    /* (this is only for 120k and 320k DOS 1.0 formats) */
    param.sectors = 8;
    param.cylinders = 40;
    param.t = TRUE; /* trigger media type search */
    }

  if (  (param.t==TRUE) /* user wanted a certain number of cylinders (tracks) */
     || (param.f==TRUE) /* user wanted a certain size */
     )
    {
    int index = 0; /* find matching media type from list */
    do
      {

      if ( (  (param.f==TRUE)
           && (param.size == (drive_specs[index].total_sectors >> 1))
           ) ||
           (  (param.t==TRUE)
           && (param.cylinders == drive_specs[index].cylinders)
           && (param.sectors   == drive_specs[index].sectors_per_cylinder)
           && (param.sides     == drive_specs[index].number_of_heads)
           )
         )
	{
	param.media_type = index;

        /* size is always found in the drive_specs, even if already given */
        param.size = drive_specs[index].total_sectors >> 1;

        if (/* *** param.f== *** */ TRUE) /* size given, geometry wanted */
          {
          param.cylinders = drive_specs[index].cylinders;
          param.sectors   = drive_specs[index].sectors_per_cylinder;
          param.sides     = drive_specs[index].number_of_heads;
          }

        printf("Formatting to %ldk (Cyl=%ld Head=%ld Sec=%2ld)\n",
          param.size, param.cylinders, param.sides, param.sectors);
	index = -10; /* break out of the loop */
        } /* end "if match" */
      else
        {
        index++;
        } /* search on if no match */

      } while ( (index>=0) && (drive_specs[index].bytes_per_sector==512) );

    if (index>0)
      {
      if (param.f==TRUE) /* only size given */
        {
        printf("No media type found for %ldk format\n", param.size);
        }
      else /* geometry given */
        {
        param.size = (param.cylinders * param.sides * param.sectors) >> 1;
        printf("No media type found for %ldk format (Cyl=%ld Head=%ld Sec=%2ld)\n",
        param.size, param.cylinders, param.sides, param.sectors);
        }
      exit(50); /* cannot continue anyway, media type needed later! */
      }
    }

  drive_number = param.drive_number;

  /* IF the user gave ANY size indication, we already know which */
  /* media_type the user wants, so we check drive capabilities now! */

  if (TRUE) /* just limiting variable scope */
    {
    int drive_number  = param.drive_number;
    int preferredsize = 0;

      regs.h.ah = 0x08;
      regs.h.dl = drive_number;
      int86(0x13, &regs, &regs);

      drive_type = regs.h.bl;
      preferredsize = 0; /* drive not installed */

    switch (drive_type)
      {
      case 1: preferredsize = 360;
        break;
      case 2: preferredsize = 1200;
        break;
      case 3: preferredsize = 720;
        break;
      case 4: preferredsize = 1440;
        break;
      case 5:
      case 6: preferredsize = 2880;
        break;
        /* Type 5 is originally for floppy tape drives */
      }

    if (param.media_type==UNKNOWN) /* do auto-detection of size! */
      {
      int index = 0; /* find matching media type from list */
      param.size = preferredsize;
      do
        {

        if (param.size == (drive_specs[index].total_sectors >> 1))
	  {
	  param.media_type = index;
          param.cylinders  = drive_specs[index].cylinders;
          param.sectors    = drive_specs[index].sectors_per_cylinder;
          param.sides      = drive_specs[index].number_of_heads;

          printf("Using drive default: %ldk (Cyl=%ld Head=%ld Sec=%2ld)\n",
            param.size, param.cylinders, param.sides, param.sectors);
	  index = -10;
	  }
        else
          {
          index++;
          }

        } while ( (index>=0) && (drive_specs[index].bytes_per_sector==512) );

      if (index >= 0)
        {
        printf("Internal error: default geometry for size %ldk unknown!\n",
          param.size);
        exit(51);
        }
      }
    else
      {
      if (param.size > preferredsize)
        {
        if (param.size > (preferredsize + (preferredsize >> 2)))
          {
          printf("You are trying to go format to more than 125 percent of the\n");
          printf("standard disk size for this drive. You must be kidding.\n");
          printf("Your size: %ldk Standard for this drive: %dk\n",
            param.size, preferredsize);
          exit(43);
          }
        printf("Formatting to more than 100 percent of the standard size\n");
        printf("for this drive: %ldk instead of only %dk. Good luck!\n",
          param.size, preferredsize);
        }
      }
    } /* variable scope end */

  number_of_cylinders  = drive_specs[param.media_type].cylinders;
  sectors_per_cylinder = drive_specs[param.media_type].sectors_per_cylinder;
  /* cannot configure drive for "number of heads" - only used in formatting */

    /* useful cases: 360k in 1200k, 1200k in 1200k, ... */
    /* formats >= 1440k are only set up by int 13.17, not here. */
    /* we already aborted if the format is ABOVE possible range */

    switch (drive_type)
      {
      case 1: /*  360k */
        regs.h.al = 1; /* mode 1: 160, 180, 320, 360, 400k */
        break;
      case 2: /* 1200k */
        regs.h.al = (param.size > 400) ? 3 : 2;
        /* mode 2: like 1, but in 1200k drive. mode 3: 1200k */
        /* mode 3 is probably only for 1220k, but we allow smaller */
        break;
      case 3: /*  720k */
        regs.h.al = 4; /* mode 3: 720, 800k */
        if (param.four || param.one || param.eight)
          {
          printf("This is a 720k drive which does not support /1, /4 or /8\n");
          exit(44);
          }
        if (param.size < 720)
          {
          printf("Minimum size for this drive type is 720k\n");
          exit(44);
          }
        break;
      case 4: /* 1440k */
        regs.h.al = 4; /* standard mode!? (int 13.18 sets details) */
        if (param.four || param.one || param.eight)
          {
          printf("This is an 1440k drive which does not support /1, /4 or /8\n");
          exit(45);
          }
        if ((param.size < 720) || (param.size == 1200))
          {
          printf("Minimum size for this drive type is 720k. 1200k not allowed.\n");
          exit(45);
          }
        break;
      case 5: /* 2880k */
      case 6: /* 2880k */
      default:
        regs.h.al = 0xff; /* no mode setting needed (?) */
        if (param.four || param.one || param.eight)
          {
          printf("This is a 2880k drive which does not support /1, /4 or /8\n");
          exit(46);
          }
        if ((param.size < 720) || (param.size == 1200))
          {
          printf("Minimum size for this drive type is 720k. 1200k not allowed.\n");
          exit(46);
          }
        break;
      }
    if (regs.h.al != 0xff)
      {
      int sizeclass = regs.h.al;

      do
        {
        regs.h.ah = 0x17; /* set disk mode, e.g "360k disk in 1200k drive */
        /* regs.x.cx = 0xffff; */ /* unused */
        regs.h.dl = drive_number;
        regs.x.cflag = 0;
        int86(0x13, &regs, &regs);
        } while ((regs.x.cflag != 0) && (regs.h.ah == 6));
      /* retry if error is only "disk changed" */

      if (regs.x.cflag != 0)
        {
        printf("Drive mode (size class %d) setting failed, error %2.2x (hex)\n",
          sizeclass, regs.h.ah);
        if (regs.h.ah == 0x80)
          {
          printf("No disk in drive!\n");
          exit(42);
          }
        else
          {
          printf("Continuing anyway.\n");
          }
        }
      }

#if 0
    /* why should we not set this now already???: */
    ddpt->sectors_per_cylinder = sectors_per_cylinder;
#endif

    /* only more modern drives / BIOSes support this at all, probably. */
    /* we may not give non-standard geometries NOW - no BIOS support.  */

    regs.h.ah = 0x18; /* set geometry, apart from head count of course */
    regs.h.ch = (number_of_cylinders > 43) ? 79 : 39;
    regs.h.cl = sectors_per_cylinder;

    if (regs.h.cl >= 36)
      {
      regs.h.cl = 36; /* 2.88M or oversized -> 2.88M */
      }
    else
      {
      if (regs.h.cl >= 18)
        {
        regs.h.cl = 18; /* 1.44M or oversized -> 1.44M */
        }
      else
        {
        if (regs.h.cl >= 15)
          {
          regs.h.cl = 15; /* 1.2M or oversized -> 1.2M */
          }
        else
          {
          regs.h.cl = (regs.h.cl > 9) ? 9 : (regs.h.cl);
          /* 720k or 360k or oversized -> 720k or 360k */
          }
        }
      }

    regs.h.dl = drive_number;

    if ((regs.h.cl > 15) || (regs.h.cl < 9) ||
        (ddpt->sectors_per_cylinder != regs.h.cl)
       )
      {
      int86x(0x13, &regs, &regs, &sregs);
      /* 40x[1,2]x8 (?), 80x2x[18,36]: 120, 320, 1440, 2880k */
      if (regs.x.cflag==0)
        {
        regs.h.ah = 0; /* ensure "ok" */
        }
      else
        {
        if (regs.h.ah == 0)
          regs.h.ah = 0x0c; /* ensure "invalid geometry" */
        }
      }
    else
      {
      if (debug_prog==TRUE)
        {
        printf("[DEBUG] Not setting geometry - disk mode should say it all already\n");
        /* 40x[1,2]x9, 80x2x[9,15]: 180, 360, 720, 1200k */
        }
      regs.h.ah = 1; /* "not supported by BIOS" (ignore errors) */
      }

    if (regs.h.ah != 0)
      {

      if (regs.h.ah==0x80)
        {
        printf("No disk in drive (timeout)!\n");
        exit(42);
        }

      if (regs.h.ah==1)
        {
        /* BIOS simply does not support 13.18, bad luck */
        }
      else
        {
        printf("Media type %ldk  (%d x %ld x %2d)  not supported by this drive!\n",
          param.size, number_of_cylinders, param.sides, sectors_per_cylinder);
        printf("Geometry set (int 13.18) error (%2.2x). Giving up.\n",
          regs.h.ah);
        exit(24);
        }

      }
    else /* if geometry setting did work */
      {
      /* ES:DI points to an array to which the int 1E vectors should point! */
      /* (added this! -ea) */
      regs.x.cx = sregs.es; /* backup ES */
      
      /* changed in 0.91b: memcpy but do not change pointer unless in BIOS */
      regs.h.ah = 0x35; /* get int vector */
      regs.h.al = 0x1e; /* ddpt vector */
      intdosx(&regs,&regs,&sregs);
      ddpt = MK_FP(sregs.es, regs.x.bx); /* CURRENT ddpt */

      if ((sregs.es < 0xa000) || ((sregs.es & 0x3ff) != 0))
        {
        /* found DDPT to reside in RAM - copying new data there */
        movedata(regs.x.cx /* srcseg */, regs.x.di /* srcoff */,
                 sregs.es, regs.x.bx, 11);

        ddpt->sectors_per_cylinder = sectors_per_cylinder;
      
        if (debug_prog==TRUE)
          {
          printf("[DEBUG] Updated INT 1E (DDPT) data at: %4.4x:%4.4x.\n",
            sregs.es, regs.x.bx);
          }

        } /* DDPT in RAM */
      else
        {

        /* found DDPT to reside in ROM - moving int 1e vector to new value */
        ddpt = MK_FP(sregs.es, regs.x.di); /* WE should know the update, too */
        regs.h.ah = 0x25; /* set int vector */
        regs.h.al = 0x1e; /* ddpt vector */
        regs.x.dx = regs.x.di; /* offset DX */
        sregs.ds  = regs.x.cx; /* segment DS (from backup) */
        intdosx(&regs,&regs,&sregs);

        printf("Using DDPT in ROM - only standard sizes possible.\n");

        if (debug_prog==TRUE)
          {
          printf("[DEBUG] New INT 1E (DDPT) vector: %4.4x:%4.4x.\n",
            regs.x.cx, regs.x.di);
          }

        } /* DDPT in ROM */

      } /* geometry setting did not work */


#if 0
  printf("Media type: %d\n",param.media_type);
#endif

#if 0
  if (drive_specs[param.media_type].interleave_factor != 1)
    /* could use OTHER marking for "non-standard format" here!      */
    /* e.g.: cylinders not 40 or 80, sectors not 8, 9, 15, 18 or 36 */
    {
    printf("*** DDPT tweaking for oversized formats not yet possible ***\n");
    exit(88);
    }
#endif

  drive_statistics.bytes_total_disk_space =
    ((unsigned long)drive_specs[param.media_type].bytes_per_sector *
    (unsigned long)drive_specs[param.media_type].total_sectors) -
    ((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat) +
    ((unsigned long)drive_specs[param.media_type].root_directory_entries/16)) *
    (unsigned long)drive_specs[param.media_type].bytes_per_sector);

  drive_statistics.bytes_available_on_disk =
    drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit =
    (unsigned long)drive_specs[param.media_type].sectors_per_cluster *
    (unsigned long)drive_specs[param.media_type].bytes_per_sector;

  /* Copy BPB into bpb_standard structure. */
  memcpy(&bpb_standard, &drive_specs[param.media_type].bytes_per_sector, 37);

  Compute_Interleave_Factor();

  if (debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Configured Disk Drive Parameter Table Values:\n");
    ddptPrinter();
    }

  if (debug_prog==TRUE)
    {
    char ch;
    printf("-- press ENTER to start formatting or ESCAPE to abort --\n");
    do
      {
      regs.h.ah = 7;
      intdos(&regs, &regs);
      ch = regs.h.al;
      }
      while ((ch != 13) && (ch != 27));
    /* (avoids important information scrolling away) */
    if (ch != 13)
      exit(1);
    }

}
