/*
// Program:  Format
// Version:  0.90d
// (0.90b/c/d - DMA in track_address_fields fix - Eric Auer May 2003)
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module:  floppy.c
// Description:  Floppy disk specific functions.
*/

#define FLOP

#include <stdlib.h>

#include "floppy.h"
#include "format.h"
#include "btstrct.h"
#include "userint.h" /* Critical_* */
#include "driveio.h" /* huge_sector_buffer* */

void Compute_Interleave_Factor(void);

void Compute_Interleave_Factor()
{
  int index;
  int starting_sector;


  low_level.interleave_factor=drive_specs[param.media_type].interleave_factor;

  index=1;
  starting_sector=0;
  low_level.interleave_index=0;
  do
    {
    low_level.interleave_map[low_level.interleave_index]=index;
    low_level.interleave_index
     =low_level.interleave_index+low_level.interleave_factor;
    if(low_level.interleave_index>=drive_specs[param.media_type].sectors_per_cylinder)
      {
      starting_sector=starting_sector+1;
      low_level.interleave_index=starting_sector;
      }

    index++;
    }while(index<=drive_specs[param.media_type].sectors_per_cylinder);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Initial Interleave Map:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Compute_Sector_Skew()
{
  int carry;
  int index;
  int skew_counter;

  if(drive_specs[param.media_type].sector_skew==0) return;

  skew_counter=0;

  do
    {
    carry
     =low_level.interleave_map[(drive_specs[param.media_type].sectors_per_cylinder-1)];

    index=(drive_specs[param.media_type].sectors_per_cylinder-1);

    do
      {
      low_level.interleave_map[index]=low_level.interleave_map[(index-1)];

      index--;
      }while(index>0);

    low_level.interleave_map[0]=carry;

    skew_counter++;
    }while(skew_counter<drive_specs[param.media_type].sector_skew);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Interleave Map After Sector Skew:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}



#define TAFinSEC 1 /* use 1 sector buffer to hold 36*4 bytes of TAF */

/* this saves stack space and avoids other hassles, but you may not */
/* use DriveIO of single sectors unless you accept buffer overwrite */


void Format_Floppy_Cylinder(int cylinder,int head) 
{
  int drive_number;
  int sector_index;
  int retry_count;
  int index;

  unsigned int result;
  unsigned int result2;

  /* Set Up Track Address Fields */
#ifdef TAFinSEC
  TAF far *track_address_fields_p = (TAF far *) sector_buffer;
  /* use this buffer to save stack -ea */
#else
  TAF track_address_fields[72]; /* 2 * 36 used, allow moving -ea */
  int taf_which = 0;
  TAF far *track_address_fields_p = &track_address_fields[taf_which];
  unsigned long int where;

  where = FP_SEG(track_address_fields_p);
  where <<= 4;
  where += FP_OFF(track_address_fields_p);

  /* start DMA boundary avoidance -ea */
  while ((where & 0xffff) > (0xffff - (4*36))) {
    taf_which++;
    track_address_fields_p = &track_address_fields[taf_which];
    where = FP_SEG(track_address_fields_p);
    where <<= 4;
    where += FP_OFF(track_address_fields_p);
    if(debug_prog==TRUE) {
      printf(">f>");
    }
  }
  if(debug_prog==TRUE) {
    printf("[DEBUG]  track_address_fields_p = %4.4x:%4.4x\n",
      FP_SEG(track_address_fields_p),FP_OFF(track_address_fields_p));
  }
#endif
  /* DMA boundary avoided -ea */

  index = 0;
  do
    {
    (track_address_fields_p+index)->cylinder=cylinder;
    (track_address_fields_p+index)->head=head;
    (track_address_fields_p+index)->sector=low_level.interleave_map[index];
    (track_address_fields_p+index)->size_code=0x02;

    index++;
    } while (index<drive_specs[param.media_type].sectors_per_cylinder);

  drive_number=param.drive_number;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Formatting:  Cylinder->  %2d    Head->  %2d\n",cylinder,head);
    }

  /* Format the Track */
  result=0;
  regs.h.ah = 0x05;
  regs.h.ch = cylinder;
  regs.h.dh = head;
  regs.h.dl = drive_number;
  sregs.es  = FP_SEG(track_address_fields_p);
  regs.x.bx = FP_OFF(track_address_fields_p);
  int86x(0x13, &regs, &regs, &sregs);
  result = regs.h.ah;

  if(result!=0) {
    printf("Format_Floppy_Cylinder( head=%d cylinder=%d ) [int 13.5]\n",
      head, cylinder );
    Critical_Error_Handler(BIOS,result);
  }


  result2 = result;

  if(param.verify==TRUE)
    {
    /* Verify the Track */
    /* According to RBIL, this uses cached CRC values, but it  */
    /* might use the ES:BX data buffer on 1985/before BIOSes!? */
    /* -> SHOULD we use READ to huge_sector_buffer instead???  */
    /* Warning: huge_sector_buffer is only 18 sectors for now. */
    result=0;
    regs.h.ah = 0x02; /* changed from VERIFY to READ */
    regs.h.al = drive_specs[param.media_type].sectors_per_cylinder;
    if ( regs.h.al > (sizeof(huge_sector_buffer_0)>>9) )
      {
        regs.h.al = sizeof(huge_sector_buffer_0) >> 9;
        printf("Only checking first %d sectors per track\n", regs.h.al);
        /* could read 2 half-tracks or use pinpointing below here     */
        /* instead... however, only 2.88MB disks have > 18 sect/track */
      }
    regs.h.ch = cylinder;
    regs.h.cl = 1;                     /* Start with the first sector. */
    regs.h.dh = head;
    regs.h.dl = drive_number;
    regs.x.bx = FP_OFF(huge_sector_buffer); /* if using READ */
    sregs.es  = FP_SEG(huge_sector_buffer); /* if using READ */
    int86x(0x13, &regs, &regs, &sregs);
    result = regs.h.ah;

    if( (debug_prog==TRUE) && ((result!=0) || (result2!=0)) )
      {
      printf("[DEBUG]  Intr 0x13 results: FORMAT -> %X, VERIFY -> %X\n",
        result2, result);
      }

    if(result!=0)
      {
      if(debug_prog==TRUE)
	{
	printf("[DEBUG]  Errors found, pinpointing bad sectors now...\n");
	}

      retry_count=0;
      sector_index=1;

      do
	{
retry:
	if(debug_prog==TRUE)
	  {
	  printf("[DEBUG]  Scanning sector number->  %d\n",sector_index);
	  }

	/* Verify the Track...sector by sector. */
	/* According to RBIL, this uses cached CRC values, but it  */
	/* might use the ES:BX data buffer on 1985/before BIOSes!? */
	/* -> SHOULD we use READ to sector_buffer instead???       */
	result=0;
	regs.h.ah = 0x02; /* changed from 4 VERIFY to 2 READ */
	regs.h.al = 1;
	regs.h.ch = cylinder;
	regs.h.cl = sector_index;
	regs.h.dh = head;
	regs.h.dl = drive_number;
        regs.x.bx = FP_OFF(huge_sector_buffer); /* if using READ */
        sregs.es  = FP_SEG(huge_sector_buffer); /* if using READ */
	int86x(0x13, &regs, &regs, &sregs);
	result = regs.h.ah;

	if(result!=0)
	  {
	  if(debug_prog==TRUE)
	    {
	    printf("[DEBUG]  Possible bad sector found... testing.  retry_count->  %d\n",retry_count);
	    }

	  retry_count++;

	  if(retry_count>=3)
	    {
	    /* Record this sector as bad. */
	    bad_sector_map[bad_sector_map_pointer] =
	      ( (cylinder * (drive_specs[param.media_type].number_of_heads) )
	      * drive_specs[param.media_type].sectors_per_cylinder)
	      + (head * drive_specs[param.media_type].sectors_per_cylinder)
	      + sector_index;

	    bad_sector_map_pointer++;
	    drive_statistics.bytes_in_bad_sectors
	     =drive_statistics.bytes_in_bad_sectors
	     +drive_specs[param.media_type].bytes_per_sector;

	    if(debug_prog==TRUE)
	      {
	      printf("[DEBUG]  Bad Sector %d CHS=[%d:%d:%d] on drive %X\n",
	        bad_sector_map[(bad_sector_map_pointer-1)],
	        cylinder, sector_index, head, drive_number);
	      }
	    else
	      {
	      printf("Sector %d CHS=[%d:%d:%d] bad\n",
	        bad_sector_map[(bad_sector_map_pointer-1)],
	        cylinder, sector_index, head);
	      }

	    retry_count=0;
	    }
	  else goto retry;   /* Yes, it's a goto.  :-)  Even goto has uses. */
	  }

	sector_index++;
	} while (sector_index<=drive_specs[param.media_type].sectors_per_cylinder);

      }
    }
}


void Set_Floppy_Media_Type()
{
  int drive_number=param.drive_number;
  int number_of_cylinders;
  int sectors_per_cylinder;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Current Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }

  param.fat_type=FAT12;

  if (param.one==TRUE)
    {
    /* *** TODO: this is only for 160k and 180k formats *** */
    param.cylinders = 40;
    if ((param.sectors!=9) && (param.sectors!=8))
      param.sectors = 9;
    param.sides = 1;
    param.t = TRUE; /* trigger media type search */
    }
  else
    {
    param.sides = 2; /* the default (initialize value!) */
    }

  if (param.four==TRUE)
    {
    /* *** TODO: special SETUP for 360k disk in 1200k drive *** */
    param.media_type=FD360;
    param.sectors = 9;
    param.cylinders = 40;
    param.t = TRUE; /* trigger media type search */
    }

  if (param.eight==TRUE)
    {
    /* *** TODO: this is only for 120k and 320k DOS 1.0 formats *** */
    param.sectors = 8;
    param.cylinders = 40;
    param.t = TRUE; /* trigger media type search */
    }

  if (param.f==TRUE) /* user wanted a certain size */
    {
    /* Standard Format Types */
    if(param.size==160) param.media_type=FD160;
    if(param.size==180) param.media_type=FD180;
    if(param.size==320) param.media_type=FD320;
    if(param.size==360) param.media_type=FD360;
    if(param.size==720) param.media_type=FD720;
    if(param.size==1200) param.media_type=FD1200;
    if(param.size==1440) param.media_type=FD1440;
    if(param.size==2880) param.media_type=FD2880;
    /* Non-Standard Format Types */
    if(param.size==400) param.media_type=FD400;
    if(param.size==800) param.media_type=FD800;
    if(param.size==1680) param.media_type=FD1680;
    if(param.size==3360) param.media_type=FD3360;
    if(param.size==1494) param.media_type=FD1494;
    if(param.size==1743) param.media_type=FD1743;
    if(param.size==3486) param.media_type=FD3486;
    if(param.size==1700) param.media_type=FD1700;
    }

  if (param.t==TRUE) /* user wanted a certain number of cylinders (tracks) */
    {
    int index=0; /* find matching media type from list */
    do
      {
      if (  (param.cylinders==drive_specs[index].cylinders)
         && (param.sectors==drive_specs[index].sectors_per_cylinder)
         && (param.sides==drive_specs[index].number_of_heads)
         )
	{
	param.media_type=index;
        printf("Formatting to %dk (Cyl=%ld Head=%ld Sec=%2ld)\n",
          drive_specs[index].total_sectors >> 1,
          param.cylinders, param.sides, param.sectors);
	index=-10;
	}

      index++;
      } while ( (index>=0) && (drive_specs[index].bytes_per_sector==512) );
      /* types 8..16 are special big formats */

    printf("No media type found for %ld x %ld x %2ld\n",
      param.cylinders, param.sides, param.sectors);
    }

  drive_number=param.drive_number;

  if (param.media_type==UNKNOWN)
    {
    /* Attempt to automatically detect the media type */
    int drive_number=param.drive_number;
    int drive_type=0;

      regs.h.ah = 0x08;
      regs.h.dl = drive_number;
      int86(0x13, &regs, &regs);

      drive_type = regs.h.bl;

    if (drive_type==0x01)
      {
      param.size=360;
      param.media_type=FD360;
      }
    if (drive_type==0x02)
      {
      param.size=1200;
      param.media_type=FD1200;
      }
    if (drive_type==0x03)
      {
      param.size=720;
      param.media_type=FD720;
      }
    if (drive_type==0x04)
      {
      param.size=1440;
      param.media_type=FD1440;
      }
    if (drive_type==0x05)     /* Originally for floppy tape drives */
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    if (drive_type==0x06)
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    }

  /* *** *** TODO: if media type was GIVEN,  *** *** */
  /* *** *** check if the drive supports it  *** *** */
  /* *** *** also configure the drive for it *** *** */


  number_of_cylinders=drive_specs[param.media_type].cylinders;
  sectors_per_cylinder=drive_specs[param.media_type].sectors_per_cylinder;

    regs.h.ah = 0x18;
    regs.h.ch = number_of_cylinders;
    regs.h.cl = sectors_per_cylinder;
    regs.h.dl = drive_number;

    int86(0x13, &regs, &regs);

    if (regs.x.cflag == 0)
	goto int_supported;


  ddpt->sectors_per_cylinder=sectors_per_cylinder;
  printf("*** int 13h, ah=18h returned an error!? ***\n");

  /* *** TODO: MORE ERROR HANDLING *** */
  printf("Media type %dk  (%d x %d x %2d)  not supported in this drive!\n",
    (number_of_cylinders * sectors_per_cylinder * param.sides) >> 1,
    number_of_cylinders, param.sides, sectors_per_cylinder);
  /* exit(42); */

  int_supported:

  if (param.media_type>HD)
    {
    /* Adjust for non-standard formats. */
    /* *** TODO *** */
    }

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector
   *(unsigned long)drive_specs[param.media_type].total_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);

  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;

  /* Copy BPB into bpb_standard structure. */
  memcpy(&bpb_standard,&drive_specs[param.media_type].bytes_per_sector,37);

  Compute_Interleave_Factor();

  if (debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Configured Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }
}
