/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Version:  0.71
// Copyright:  1998 under the terms of the GNU GPL
*/

/* Phil Brutsche - November 20, 1998
  Fixed a number of bugs:
  * In several places, I found the use of gets (bad in itself...) in
    conjunction with uninitialized pointers.  Specifically, this:

	  char *input_buffer;

	  long partition_size;

	     Clear_Screen();
	     printf("\n\n\nEnter Primary Partition Size: ");
	     gets(input_buffer);

    That's a no-no; you don't know where input_buffer points to - it could
    point to, say, 0x00000010, and royally screw up your interrupt table
    when you use gets.  Plus, `gets` doesn't check the length of the string
    you are reading - automatic buffer overflow.  You may get your input,
    but it'll royally screw up anything that it physically next to the
    buffer in memory - maybe even part of your program!
  * Created some preprocessor directives to simplify the help screens.
*/



#define NAME "Free FDISK"
#define VERSION "0.71"

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bootcode.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define READ 2

#define TRUE 1
#define FALSE 0

#define PRIMARY 1
#define EXTENDED 2
#define LOGICAL 3
#define SPECIAL 4

#define LAST 99

#define NULL 0

#define PERCENTAGE 1

#define STANDARD 0
#define TECHNICAL 1

#define INTERNAL 0
#define EXTERNAL 1

/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

/* Buffers */
unsigned char partition_lookup_table_buffer[256][80];
unsigned char sector_buffer[512];

unsigned long g_cylinder;
unsigned long g_sector;

/* Hard Drive Parameters */
unsigned long total_cylinders;
unsigned long total_heads;
unsigned long total_sectors;

/* Ending Mapping Variables */
unsigned long computed_ending_cylinder;
unsigned long computed_ending_head;
unsigned long computed_ending_sector;
unsigned long computed_partition_size;

/* Partition Table Information */
int extended_partition_exists=FALSE;

int active_flag[4];
char partition_type[24] [80];
int numeric_partition_type[24];

unsigned long starting_cylinder[24];
unsigned long starting_head[24];
unsigned long starting_sector[24];

unsigned long ending_cylinder[24];
unsigned long ending_head[24];
unsigned long ending_sector[24];

unsigned long true_partition_size[24];
long partition_size[24];
int number_of_partitions;
unsigned long hard_disk_space_used;
unsigned long whole_extended_partition_size;
unsigned long extended_partition_space_used;

unsigned long extended_partition_cylinder[24];
unsigned long extended_partition_head[24];
unsigned long extended_partition_sector[24];

/* Integers converted from long numbers */
int integer1;
int integer2;

/* Misc. Flags */
int partition_lookup_table=INTERNAL;
int sectors_have_changed=FALSE;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

char Input();

int External_Lookup_File_Exists();

unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector);

long Decimal_Number(long hex1, long hex2, long hex3);

void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size);
void Clear_Screen();
void Clear_Sector_Buffer();
void Clear_Partition_Table(int drive);
void Convert_Long_To_Integer(long number);
void Create_Alternate_MBR(int drive);
void Create_MBR(int drive);
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage,int special_flag,int special_partition_type);
void Delete_Partition(int drive, long partition_number);
void Display_Help_Screen();
void Display_Partition_Table(int drive,int tech_flag);
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2);
void Get_Hard_Drive_Parameters(int drive);
void Initialize_Variables();
void List_Available_Drives();
void Load_External_Lookup_Table();
void Menu_Routine();
void Modify_Partition_Type(int drive,int partition_number,int type_number);
void Pause();
void Read_Partition_Table(int drive,int view_type);
void Read_Sector(int drive, int head, long cyl, int sector);
void Remove_MBR(int drive);
void Save_MBR(int drive);
void Scan_External_Lookup_Table(int indicated_partition_type,int partition,int view_type);
void Toggle_Active_Partition(int drive, long partition_number, int toggle);
void Write_Sector(int drive, int head, long cyl, int sector);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Calculate the end of a Partition */
/* Replaces void Compute_Partition_End */
void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size)
{
  long cylinder_size=(total_heads+1)*(total_sectors);

  computed_partition_size=0;
  computed_ending_cylinder=start_cylinder;

  do
    {
    computed_ending_cylinder++;
    computed_partition_size=computed_partition_size+cylinder_size;
    }while(computed_partition_size<size);

  computed_ending_cylinder--;
  computed_partition_size=computed_partition_size-cylinder_size;
}

/* Clear Screen */
void Clear_Screen()
{
  asm{
    mov ah,0
    mov al,3
    int 0x10
    }
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  int index=0;

  do
    {
    sector_buffer[index]=0;
    index ++;
    } while(index<512);
}

/* Clear Partition Table */
void Clear_Partition_Table(int drive)
{
  Clear_Sector_Buffer();
  Write_Sector(drive, 0, 0, 1);
}

/* Combine Cylinder and Sector Values */
unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector)
{
  long value = 0;

  asm{
    mov ax,WORD PTR cylinder
    mov bx,WORD PTR sector

    mov dl,ah
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1

    mov dh,al

    add dx,bx

    mov WORD PTR value,dx
    }

  return(value);
}

/* Convert Long number to 2 integers */
void Convert_Long_To_Integer(long number)
{
  integer1=0;
  integer2=0;

  asm{
    mov ax,WORD PTR number

    mov BYTE PTR integer1, al
    mov BYTE PTR integer2, ah
    }
}

/* Create Alternate Master Boot Code */
void Create_Alternate_MBR(int drive)
{
  long index=0;

  FILE *file_pointer;

  Read_Sector(drive,0,0,1);

  /* Clear old MBR, if any */
  do
    {
    sector_buffer[index]=0x00;
    index++;
    }while(index<0x1be);
  index=0;

  file_pointer=fopen("boot.bin","rb");

  if(!file_pointer)
    {
    printf("\nThe\"BOOT.BIN\" file has not been found...Operation Terminated.\n");
    exit(1);
    }

  do
    {
    sector_buffer[index]=fgetc(file_pointer);
    index++;
    }while(index<0x1be);

  fclose(file_pointer);

  Write_Sector(drive,0,0,1);
}

/* Create Master Boot Code */
void Create_MBR(int drive)
{
  long number_of_bytes=445; /* ***** This is the number of bytes for the */
			    /* ***** boot code in the partition table.   */
			    /* ***** If the boot code in the partition   */
			    /* ***** table changes, this WILL need       */
			    /* ***** changed.  (Start the count at 0.)   */
  long loop=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[loop]=boot_code[loop];

    loop++;
    }while(loop<=number_of_bytes);

  Write_Sector(drive,0,0,1);
}

/* Create New Partition */
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage,int special_flag,int special_partition_type)
{
  int index;
  int next_available_partition;
  int partition_created=FALSE;

  unsigned long temp;

  unsigned long hard_disk_space_available;

  unsigned long extended_partition_size;
  unsigned long extended_partition_size_high;
  unsigned long extended_partition_size_low;
  unsigned long available_extended_partition_space;

  unsigned long relative_sectors;
  unsigned long relative_sectors_high;
  unsigned long relative_sectors_low;

  unsigned long total_hard_disk_size;

  long partition_table_pointer;

  long cp_starting_cylinder;
  long cp_starting_head;
  long cp_starting_sector;

  unsigned long partition_size_high;
  unsigned long partition_size_low;

  Read_Partition_Table(drive,TECHNICAL);
  Get_Hard_Drive_Parameters(drive);

  total_hard_disk_size = (total_cylinders+1) * (total_heads+1) * (total_sectors);
  hard_disk_space_available=total_hard_disk_size-hard_disk_space_used;

  /* Compute the size of the partition */
  available_extended_partition_space=whole_extended_partition_size-extended_partition_space_used;

  if(percentage==PERCENTAGE)
    {
    if(create_partition_type==PRIMARY)
      {
      if(total_hard_disk_size>4000000)
	{
	size=(4000000*size)/100;
	}
      else
	{
	size=(hard_disk_space_available*size)/100;
	}
      }
    if(create_partition_type==EXTENDED)
      {
      size=(hard_disk_space_available*size)/100;
      }
    if(create_partition_type==LOGICAL)
      {
      if(available_extended_partition_space>4000000)
	{
	size=(4000000*size)/100;
	}
      else
	{
	size=(available_extended_partition_space*size)/100;
	}
      }
    }
  else
    {
    size=size*2000;
    }
  if(size>total_hard_disk_size) size = total_hard_disk_size;
  if((create_partition_type==PRIMARY) && (size>4000000)) size=4000000;
  if((create_partition_type==EXTENDED) && (size>(hard_disk_space_available))) size=hard_disk_space_available;
  if((create_partition_type==LOGICAL) && (size>4000000)) size=4000000;
  if((create_partition_type==LOGICAL) && (size>extended_partition_size)) size=extended_partition_size;
  if((create_partition_type==LOGICAL) && (size>available_extended_partition_space)) size=available_extended_partition_space;
  if(size<4000)
    {
    printf("\nPartition size to be created is too small...Operation Terminated\n");
    exit(1);
    }

  if((create_partition_type==PRIMARY) || (create_partition_type==EXTENDED))
    {
    index=0;

    do
      {
      if( (extended_partition_exists==TRUE) && (create_partition_type==EXTENDED) )
	{
	printf("\nExtended partition already exists...operation terminated.\n");
	exit(1);
	}
      /* Find unused primary partition */
      if(0==numeric_partition_type[index])
	{
	/* Unused primary partition found! */


	/* If this is the first partition */
	if(0==index)
	  {
	  cp_starting_cylinder=0;
	  cp_starting_head=1;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(0!=index)
	  {
	  cp_starting_cylinder=(ending_cylinder[index-1]+1);
	  cp_starting_head=0;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(special_flag!=SPECIAL)
	  {
	  if(create_partition_type==PRIMARY)
	    {
	    if(size>0xffff) numeric_partition_type[index]=0x06;
	    if(size<=0xffff) numeric_partition_type[index]=0x04;
	    if(size<=4096) numeric_partition_type[index]=0x01;
	    }
	  else
	    {
	    numeric_partition_type[index]=0x05;
	    }
	  }
	else
	  {
	  numeric_partition_type[index]=special_partition_type;
	  }

	/* Make computations & write partition */
	Read_Sector(drive,0,0,1);

	partition_table_pointer=0x1be+(index*16);

	temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,cp_starting_sector);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x02]=integer1;
	sector_buffer[partition_table_pointer+0x03]=integer2;

	sector_buffer[partition_table_pointer+0x01]=cp_starting_head;

	sector_buffer[partition_table_pointer+0x04]=numeric_partition_type[index];

	temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_sectors);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x06]=integer1;
	sector_buffer[partition_table_pointer+0x07]=integer2;

	sector_buffer[partition_table_pointer+0x05]=total_heads;

	/* Adjust size to fit inside partition */
	if(size> ((total_heads+1)*total_sectors))
	  {
	  size=(computed_ending_cylinder-cp_starting_cylinder)*( (total_heads+1)*(total_sectors) );
	  }

	if(create_partition_type==PRIMARY) size=size-total_sectors;

	partition_size_high = size >> 16;
	temp=partition_size_high << 16;
	if(size > 0xffff)
	  {
	  partition_size_low = size - temp;
	  }
	else
	  {
	  partition_size_low = size;
	  }

	Convert_Long_To_Integer(partition_size_low);

	sector_buffer[partition_table_pointer+0x0c]=integer1;
	sector_buffer[partition_table_pointer+0x0d]=integer2;

	Convert_Long_To_Integer(partition_size_high);

	sector_buffer[partition_table_pointer+0x0e]=integer1;
	sector_buffer[partition_table_pointer+0x0f]=integer2;

	/* Set the relative sector field in the partition table. */
	if(create_partition_type==PRIMARY)
	  {
	  sector_buffer[partition_table_pointer+0x08]=total_sectors;
	  }
	else
	  {
	  int secondary_index=0;
	  int primary_partition_number=-1;

	  /* Which partition is the primary partition? */
	  do
	    {
	    if(0x01==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x04==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x06==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;

	    secondary_index++;
	    }while(primary_partition_number==-1);

	  relative_sectors=(cp_starting_cylinder-starting_cylinder[primary_partition_number])*( (total_heads+1) * (total_sectors) );

	  relative_sectors_high = relative_sectors >> 16;
	  temp=relative_sectors_high << 16;

	  if(relative_sectors > 0xffff)
	    {
	    relative_sectors_low = relative_sectors - temp;
	    }
	  else
	    {
	    relative_sectors_low = relative_sectors;
	    }

	  Convert_Long_To_Integer(relative_sectors_low);

	  sector_buffer[partition_table_pointer+0x08]=integer1;
	  sector_buffer[partition_table_pointer+0x09]=integer2;

	  Convert_Long_To_Integer(relative_sectors_high);

	  sector_buffer[partition_table_pointer+0x0a]=integer1;
	  sector_buffer[partition_table_pointer+0x0b]=integer2;
	  }

	sector_buffer[0x1fe]=0x55;
	sector_buffer[0x1ff]=0xaa;

	Write_Sector(drive,0,0,1);

	/* Clear boot sector of newly created partition */
	Clear_Sector_Buffer();
	Write_Sector(drive,cp_starting_head,cp_starting_cylinder,1);

	partition_created=TRUE;
	}
      index++;
      } while((index<4) && (partition_created==FALSE));
    }

  if((create_partition_type==LOGICAL) && (extended_partition_exists==TRUE))
    {

    /* If this is not the first logical drive,   */
    /* create the extended entry in the previous */
    /* logical drive table.                      */
    if(number_of_partitions>=5)
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);

      partition_table_pointer=0x1be+16;

      /* Set the 2nd partition entry in this table to extended */
      sector_buffer[partition_table_pointer+0x04]=0x05;

      Extract_Cylinder_and_Sector(sector_buffer[(0x1be+0x06)],sector_buffer[(0x1be+0x07)]);

      /* Save the location of the next partition */
      extended_partition_cylinder[number_of_partitions-3]=g_cylinder+1;
      extended_partition_sector[number_of_partitions-3]=1;
      extended_partition_head[number_of_partitions-3]=0;

      if(extended_partition_cylinder[number_of_partitions-3]>total_cylinders)
	{
	printf("\nPartition cannot be created past end of drive...Operation Terminated.\n");
	exit(1);
	}

      /* Add the logical drive pointer information to the sector buffer */
      sector_buffer[partition_table_pointer+0x01]=0;

      temp=Combine_Cylinder_and_Sector((extended_partition_cylinder[number_of_partitions-3]),1);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x02]=integer1;
      sector_buffer[partition_table_pointer+0x03]=integer2;

      /* Compute the end of the partition */
      Calculate_Partition_Ending_Cylinder((extended_partition_cylinder[number_of_partitions-3]),size);

      sector_buffer[partition_table_pointer+0x05]=total_heads;

      temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_heads);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x06]=integer1;
      sector_buffer[partition_table_pointer+0x07]=integer2;

      /* Compute the "relative sectors" */
      relative_sectors=(extended_partition_cylinder[number_of_partitions-3]-starting_cylinder[4])*((total_heads+1)*(total_sectors));

      relative_sectors_high = relative_sectors >> 16;
      temp=relative_sectors_high << 16;

      if(relative_sectors > 0xffff)
	{
	relative_sectors_low = relative_sectors - temp;
	}
      else
	{
	relative_sectors_low = relative_sectors;
	}

      Convert_Long_To_Integer(relative_sectors_low);

      sector_buffer[partition_table_pointer+0x08]=integer1;
      sector_buffer[partition_table_pointer+0x09]=integer2;

      Convert_Long_To_Integer(relative_sectors_high);

      sector_buffer[partition_table_pointer+0x0a]=integer1;
      sector_buffer[partition_table_pointer+0x0b]=integer2;

      /* Compute the partition size */
      extended_partition_size=(computed_ending_cylinder-extended_partition_cylinder[number_of_partitions-3])*( (total_heads+1)*(total_sectors)  );

      extended_partition_size_high = extended_partition_size >> 16;
      temp=extended_partition_size_high << 16;

      if(extended_partition_size > 0xffff)
	{
	extended_partition_size_low = extended_partition_size - temp;
	}
      else
	{
	extended_partition_size_low = extended_partition_size;
	}

      Convert_Long_To_Integer(extended_partition_size_low);

      sector_buffer[partition_table_pointer+0x0c]=integer1;
      sector_buffer[partition_table_pointer+0x0d]=integer2;

      Convert_Long_To_Integer(extended_partition_size_high);

      sector_buffer[partition_table_pointer+0x0e]=integer1;
      sector_buffer[partition_table_pointer+0x0f]=integer2;

      /* Write Sector */
      Write_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);
      }

    /* Get the next available partition to use */
    next_available_partition=number_of_partitions-3;

    /* Compute the beginning of the partition */
    cp_starting_cylinder=extended_partition_cylinder[next_available_partition];

    /* Compute the end of the partition */
    Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);

    Clear_Sector_Buffer();

    partition_table_pointer=0x1be;

    /* Create partition table */
    temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,1);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x02]=integer1;
    sector_buffer[partition_table_pointer+0x03]=integer2;

    sector_buffer[partition_table_pointer+0x01]=1;

    if(special_flag!=SPECIAL)
      {
      if(size>0xffff) sector_buffer[partition_table_pointer+0x04]=0x06;
      if(size<=0xffff) sector_buffer[partition_table_pointer+0x04]=0x04;
      if(size<=4096) sector_buffer[partition_table_pointer+0x04]=0x01;
      }
    else
      {
      sector_buffer[partition_table_pointer+0x04]=special_partition_type;
      }

    temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_sectors);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x06]=integer1;
    sector_buffer[partition_table_pointer+0x07]=integer2;

    sector_buffer[partition_table_pointer+0x05]=total_heads;

    /* Adjust size to fit inside partition */
    size=((computed_ending_cylinder-cp_starting_cylinder)*( (total_heads+1) * (total_sectors) )) - total_sectors;

    partition_size_high = size >> 16;
    temp=partition_size_high << 16;
    if(size > 0xffff)
      {
      partition_size_low = size - temp;
      }
    else
      {
      partition_size_low = size;
      }

    Convert_Long_To_Integer(partition_size_low);

    sector_buffer[partition_table_pointer+0x08]=total_sectors;

    sector_buffer[partition_table_pointer+0x0c]=integer1;
    sector_buffer[partition_table_pointer+0x0d]=integer2;

    Convert_Long_To_Integer(partition_size_high);

    sector_buffer[partition_table_pointer+0x0e]=integer1;
    sector_buffer[partition_table_pointer+0x0f]=integer2;

    sector_buffer[0x1fe]=0x55;
    sector_buffer[0x1ff]=0xaa;

    Write_Sector(drive,0,cp_starting_cylinder,1);

    /* Clear "boot" sector of new partition */
    Clear_Sector_Buffer();
    Write_Sector(drive,1,cp_starting_cylinder,1);
    }

  if((create_partition_type==LOGICAL) && (extended_partition_exists==FALSE))
    {
    printf("\nExtended partition non-existant...operation terminated.\n");
    exit(1);
    }
}

/* Convert Hexidecimal Number to a Decimal Number */
long Decimal_Number(long hex1, long hex2, long hex3)
{
  return((hex1) + (hex2*256) + (hex3*65536));
}

/* Delete Partition */
void Delete_Partition(int drive,long partition_number)
{
  int loop=0;

  long offset;

  if(partition_number==LAST)
    {
    Read_Partition_Table(drive,TECHNICAL);

    if(number_of_partitions==5)
      {
      Clear_Sector_Buffer();
      Write_Sector(drive,extended_partition_head[1],extended_partition_cylinder[1],extended_partition_sector[1]);
      }
    else
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);

      offset=(0x1be+16);

      do
	{
	sector_buffer[offset+loop]=0x00;
	loop++;
	}while(loop < 16);

      Write_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);
      }
    }
  else
    {
    Read_Sector(drive,0,0,1);

    offset=((partition_number*16)-16)+0x1be;

    do
      {
      sector_buffer[offset+loop]=0x00;
      loop++;
      }while(loop < 16);

    Write_Sector(drive,0,0,1);
    }
}

/* Display Help Screen */
void Display_Help_Screen()
{
  printf("\n\n%10s                                                         Version %s\n", NAME, VERSION);
  printf("\nWritten By:  Brian E. Reifsnyder\n\n");
  printf("                                                      *************************\n");
  printf("                                                      *        Warning!       *\n");
  printf("Syntax:                                               * Use this program with *\n");
  printf("                                                      *  extreme care!  The   *\n");
  printf("FDISK                                                 *   author assumes no   *\n");
  printf("FDISK [/N drive# partition_type size [/P] [/S type#] ]*   responsibility for  *\n");
  printf("FDISK [/D drive# [partition#] || [/L] ]               *  the results of this  *\n");
  printf("FDISK [/I [drive# [/tech] ]                           *  program's execution. *\n");
  printf("FDISK [/L]                                            *************************\n");
  printf("FDISK [/MBR [drive#] ]\n");
  printf("FDISK [/RMBR [drive#] ]\n");
  printf("FDISK [/SMBR [drive#] ]\n");
  printf("FDISK [/AMBR [drive#] ]\n");
  printf("FDISK [/C drive#]\n");
  printf("FDISK [/T drive# partition# active_toggle]\n");
  printf("FDISK [/M drive# primarypartition# newtype#]\n\n");
  Pause();
  printf("\n\n\n  /?      Displays these help screens.\n");
  printf("  /N      Creates a new partition based upon the drive# (1-4), type (P, E,\n");
  printf("            or L), and size (in megabytes).  If the /P switch is added,\n");
  printf("            the size is a percentage of the total drive space.  If the /S\n");
  printf("            switch is used, then a type# partition is created.\n");
  printf("  /D      Deletes a partition based upon the drive# (1-4) and the partition\n");
  printf("            number or the last logical partition.\n");
  printf("  /I      Displays the partition table for drive# (1-4).  If the \"/tech\"\n");
  printf("            switch is used, it also shows the starting and ending information\n");
  printf("            for each partition.\n");
  printf("  /L      Lists the available hard disks and their sizes.\n");
  printf("  /MBR    Writes the Master Boot Record to drive#.\n");
  printf("  /RMBR   Removes the Master Boot Record from drive#.\n");
  printf("  /SMBR   Saves the current Master Boot program to the \"BOOT.BIN\" file.\n");
  printf("  /AMBR   Writes the Master Boot program in \"BOOT.BIN\" to drive#.\n");
  printf("  /C      Clears the partition sector.  (Use with extreme caution!)\n");
  printf("  /T      Toggles the partition# as active or not active (1 or 0).\n");
  printf("  /M      Modifies the partition type of primarypartition# to\n");
  printf("            newtype#.\n");
  Pause();
  Clear_Screen();
  printf("\nCopyright 1998 under the terms of the GNU General Public License.\n");
  printf("\nThis program comes as-is and without warranty of any kind.  The author of this\n");
  printf("software assumes no responsibility pertaining to the use or mis-use of this\n");
  printf("software.  By using this software, the operator is understood to be agreeing to\n");
  printf("the terms of the above.\n");
}

/* Display Partition Table */
void Display_Partition_Table(int drive,int tech_flag)
{
  char table[9];
  char view[9];

  int partition_number=0;
  unsigned long total_hard_disk_size;

  /* Set the lookup table type for display */
  if(partition_lookup_table==EXTERNAL) strcpy(table,"External");
  else strcpy(table,"Internal");

  /* Set the view type variable */
  if(tech_flag==STANDARD)
    {
    strcpy(view,"Standard ");
    }
  else
    {
    strcpy(view,"Technical");
    }


  /* Get partition table */
  Read_Partition_Table(drive,tech_flag);

  /* Display partition table */
  printf("\nPartition:  Primary              | View:  %9s | Lookup Table: %9s |\n",view,table);
  printf("-------------------------------------------------------------------------------\n");

  if(tech_flag==TECHNICAL)
    {
    printf(" #  A   Type             Starting Location   Ending Location     Partition Size\n");
    printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");
    }
  else
    {
    printf(" #  A   Type                                                     Partition Size\n");
    printf("                                                                  in Megabytes\n\n");
    }
  do
    {
    /* Print partition number */
    printf(" %d",(partition_number+1));

    /* Is partition a boot partition? */
    if(active_flag[partition_number]==TRUE)
      {
      printf("  *   ");
      }
    else
      {
      printf("      ");
      }

    if(tech_flag==TECHNICAL)
      {
      /* Display Partition Type */
      printf(partition_type[partition_number]);

      /* Display Partition Mappings */
      if(0!=numeric_partition_type[partition_number])
	{
	printf("    %4d",starting_cylinder[partition_number]);
	printf("  %4d",starting_head[partition_number]);
	printf("  %4d",starting_sector[partition_number]);

	printf("    %4d",ending_cylinder[partition_number]);
	printf("  %4d",ending_head[partition_number]);
	printf("  %4d",ending_sector[partition_number]);

	printf("       %6d",partition_size[partition_number]);
	}
      }
    else
      {
      if(partition_lookup_table==INTERNAL)
	{
	printf(partition_type[partition_number]);
	printf("                                        ");
	}
      else
	{
	printf(partition_type[partition_number]);
	printf("   ");
	}

      /* If partition is used, print the size */
      if(0!=numeric_partition_type[partition_number])
	{
	printf("       %6d",partition_size[partition_number]);
	}
      }

    /* Carriage Return */
    printf("\n");

    partition_number++;
    } while(partition_number<4);

  printf("-------------------------------------------------------------------------------\n\n");

  if(number_of_partitions>4)
    {
    partition_number=4;

    printf("Partition:  Extended\n");
    printf("-------------------------------------------------------------------------------\n");

    if(tech_flag==TECHNICAL)
      {
      printf(" #      Type             Starting Location   Ending Location     Partition Size\n");
      printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");
      }
    else
      {
      printf(" #      Type                                                     Partition Size\n");
      printf("                                                                  in Megabytes\n\n");
      }

    do{

      /* Print partition number */
      printf(" %d",(partition_number+1));
      printf("      ");

      if(tech_flag==TECHNICAL)
	{
	/* Display Partition Type */
	printf(partition_type[partition_number]);

	/* Display Partition Mappings */
	if(0!=strcmp(partition_type[partition_number],"Unused       "))
	  {
	  printf("    %4d",starting_cylinder[partition_number]);
	  printf("  %4d",starting_head[partition_number]);
	  printf("  %4d",starting_sector[partition_number]);

	  printf("    %4d",ending_cylinder[partition_number]);
	  printf("  %4d",ending_head[partition_number]);
	  printf("  %4d",ending_sector[partition_number]);

	  printf("       %6d",partition_size[partition_number]);
	  }
	}
      else
	{
	if(partition_lookup_table==INTERNAL)
	  {
	  printf(partition_type[partition_number]);
	  printf("                                        ");
	  }
	else
	  {
	  printf(partition_type[partition_number]);
	  printf("   ");
	  }

	/* If the partition is used, print the size */
	if(0!=numeric_partition_type[partition_number])
	  {
	  printf("       %6d",partition_size[partition_number]);
	  }
	}

      /* Carriage Return */
      printf("\n");

      partition_number++;
      }while(partition_number<number_of_partitions);
    printf("-------------------------------------------------------------------------------\n");
    }

  Get_Hard_Drive_Parameters(drive);
  total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
  total_hard_disk_size=total_hard_disk_size/2000;
  printf("                                                  Total hard drive size:  %d\n",total_hard_disk_size);
}

/* Does "part.dat" exist? */
int External_Lookup_File_Exists()
{
  int result=NULL;

  FILE *file;

  file=fopen("part.dat","rt");

  if(!file)
    {
    result=FALSE;
    }
  else
    {
    result=TRUE;
    fclose(file);
    }

  return(result);
}

/* Extract Cylinder & Sector */
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );

  g_sector = cylinder_and_sector % 64;

  g_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );
}


/* Get Hard Drive Parameters */
void Get_Hard_Drive_Parameters(int drive)
{
  asm{
    mov ah, 0x08
    mov dl, BYTE PTR drive
    int 0x13

    mov bl,cl
    and bl,00111111B

    mov BYTE PTR total_sectors, bl

    mov bl,cl
    mov cl,ch
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1

    mov ch,bl

    mov WORD PTR total_cylinders, cx
    mov BYTE PTR total_heads, dh
    }
}

/* Initialize Variables */
void Initialize_Variables()
{
  int loop=0;

  do
    {
    active_flag[loop]=FALSE;

    loop++;
    }while(loop<=4);

  extended_partition_exists=FALSE;
}

/* Get input from keyboard */
char Input()
{
  char input;

  asm{
    mov ah,7
    int 0x21
    mov BYTE PTR input,al
    }

  return(input);
}

/* List Available Hard Drives */
void List_Available_Drives()
{
  char temp_sector_buffer[512];

  int loop=128;
  int result;
  unsigned long total_hard_disk_size;

  printf("\n  Drive              Size\n");
  printf("  Number          (Megabytes)\n\n");
  do
    {
    result=biosdisk(2, loop, 0, 0, 1, 1, temp_sector_buffer);

    if (result==0)
      {
      Get_Hard_Drive_Parameters(loop);
      total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
      total_hard_disk_size=total_hard_disk_size/2000;
      printf("    %d                %d\n",(loop-127),total_hard_disk_size);
      }

    loop++;
    }while(loop<=131);
  printf("\n\n");
}

/* Load External Partition Type Lookup Table */
void Load_External_Lookup_Table()
{
  int index=0;
  int end_of_file_marker_encountered=FALSE;

  FILE *file;

  char line_buffer[256];

  file=fopen("part.dat","rt");

  if(!file)
    {
    printf("\nThe \"part.dat\" file is not found...Operation Terminated.\n");
    exit(1);
    }

  while(fgets(line_buffer,255,file) !=NULL)
    {
    if( (0!=strncmp(line_buffer,";",1)) && (0!=strncmp(line_buffer,"999",3)) && (end_of_file_marker_encountered==FALSE) )
      {
      strncpy(partition_lookup_table_buffer[index],line_buffer,79);
      index++;
      }
    if(0==strncmp(line_buffer,"999",3)) end_of_file_marker_encountered=TRUE;
    }
  fclose(file);
}

/* Menu Routine */
void Menu_Routine()
{
  int exit=FALSE;
  int exit_sub_menu=FALSE;
  int drive=128;
  long partition_number;

  char input;

  do
    {
    Initialize_Variables();
    Clear_Screen();

    printf("                                  Version %s\n",VERSION);
    printf("                            Fixed Disk Setup Program\n");
    printf("                      Copyright 1998 by Brian E. Reifsnyder\n");
    printf("                         under the terms of the GNU GPL\n\n");
    printf("                                 FDISK Options\n\n");
    printf("Current fixed disk drive: %d\n\n",(drive-127));
    printf("Choose one of the following:\n\n");
    printf("1. Create DOS partition or Logical DOS Drive\n");
    printf("2. Set active partition\n");
    printf("3. Delete partition or Logical DOS Drive\n");
    printf("4. Display partition information\n");
    printf("5. Change current fixed disk drive\n");
    printf("6. Exit this utility\n\n");
    printf("Enter choice: ");

    /* Get input from keyboard. */
    input=Input();

    /* Create DOS partition or Logical DOS Drive */
    if(input==49)
      {
      exit_sub_menu=FALSE;

      do
	{
	Clear_Screen();
	printf("\n\n\n\n\n                   Create DOS Partition or Logical DOS Drive\n\n");
	printf("Current fixed disk drive: %d\n\n",(drive-127));
	printf("Choose on of the following:\n\n");
	printf("1. Create Primary DOS Partition\n");
	printf("2. Create Extended DOS Partition\n");
	printf("3. Create Logical DOS Drive(s) in the Extended DOS Partition\n");
	printf("4. Return to Main Menu\n\n\n");
	printf("Enter choice: ");
	input=Input();

	/* Create Primary DOS Partition */
	if(input==49)
	  {
	  int percentage_flag=NULL;
	  int true=1;

	  char input_buffer[10];

	  long partition_size;

	  Clear_Screen();
	  printf("\n\n\nEnter Primary Partition Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    }

	  if(partition_size>2048)
	    {
	    printf("\n\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\nIs the partition size entered a percentage (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y\n");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
	      /* trap bad input */
		{
		percentage_flag=NULL;
		}
	      }
	    }

	  Create_New_Partition(drive,PRIMARY,partition_size,percentage_flag,NULL,NULL);
	  }

	/* Create Extended DOS Partition */
	if(input==50)
	  {
	  int percentage_flag=NULL;
	  int true=1;

	  char input_buffer[10];

	  long partition_size;

	  Clear_Screen();
	  printf("\n\n\nEnter Extended Partition Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if(partition_size<=100)
	    {
	    printf("\nIs the partition size entered a percentage (Y/N)?");
	    fgets(input_buffer,10,stdin);

	    true=1;
	    true=stricmp(input_buffer,"Y\n");
	    if(0==true)
	      {
	      percentage_flag=PERCENTAGE;
	      }
	    else
	      {
	      percentage_flag=NULL;
	      strcpy(input_buffer,"null");
	      }
	    }

	  Create_New_Partition(drive,EXTENDED,partition_size,percentage_flag,NULL,NULL);
	  }

	/* Create Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  char input_buffer[10];

	  long partition_size;

	  int percentage_flag=NULL;
	  int true=1;

	  Clear_Screen();
	  printf("\n\n\nEnter Logical DOS Drive Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if(partition_size>2048)
	    {
	    printf("\n\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	    Pause();
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\nIs the partition size entered a percentage (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y\n");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
		{
		percentage_flag=NULL;
		strcpy(input_buffer,"null");
		}
	      }
	    Create_New_Partition(drive,LOGICAL,partition_size,percentage_flag,NULL,NULL);
	    }
	  }

	/* Return to Main Menu */
	if(input==52)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);

      input=99; /* change value to prevent any other options from being executed */
      }

    /* Set active partition */
    if(input==50)
      {
      Clear_Screen();
      printf("Fixed Disk Drive: %d",drive-127);
      Display_Partition_Table(drive,STANDARD);
      printf("\n\nEnter partition number to set active: ");

      /* Get input from keyboard. */
      input=Input();

      partition_number=input-48;

      if((partition_number>=1) && (partition_number <=4))
	{
	Toggle_Active_Partition(drive,partition_number,1);

	if(partition_number!=1) Toggle_Active_Partition(drive,1,0);
	if(partition_number!=2) Toggle_Active_Partition(drive,2,0);
	if(partition_number!=3) Toggle_Active_Partition(drive,3,0);
	if(partition_number!=4) Toggle_Active_Partition(drive,4,0);
	}
      else
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	Pause();
	}

      }

    /* Delete partition or Logical DOS Drive */
    if(input==51)
      {
      exit_sub_menu=FALSE;

      do
	{
	Clear_Screen();
	printf("\n\n\n\n\n                   Delete DOS Partition or Logical DOS Drive\n\n");
	printf("Current fixed disk drive: %d\n\n",(drive-127));
	printf("Choose one of the following:\n\n");
	printf("1. Delete Primary DOS Partition\n");
	printf("2. Delete Extended DOS Partition\n");
	printf("3. Delete Logical DOS Drive(s) in the Extended DOS Partition\n");
	printf("4. Delete Non-DOS Partition\n");
	printf("5. Return to Main Menu\n\n\n");
	input=Input();

	/* Delete Primary DOS Partition */
	if(input==49)
	  {
	  input=52;
	  }

	/* Delete Extended DOS Partition */
	if(input==50)
	  {
	  input=52;
	  }

	/* Delete Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  char input_buffer[10];

	  int true=1;

	  Clear_Screen();
	  printf("\n\n\nWarning:  you are about to DELETE the last logical drive in the extended\n");
	  printf("          partition.\n\n\n");
	  printf("Continue with operation (y/n)? ");
	  fgets(input_buffer,10,stdin);

	  true=1;
	  true=stricmp(input_buffer,"Y\n");
	  if(0==true)
	    {
	    Read_Partition_Table(drive,TECHNICAL);

	    if(number_of_partitions<5)
	      {
	      printf("\nNo logical partitions exist...Operation Terminated.\n");
	      Pause();
	      }
	    else
	      {
	      Delete_Partition(drive,LAST);
	      printf("\nPartition Deleted!\n");
	      Pause();
	      }
	    }
	  else
	    {
	    printf("\n\n\nOperation Aborted.\n");
	    Pause();
	    }
	  }

	/* Delete Non-DOS Partition */
	if(input==52)
	  {
	  char input_buffer[10];

	  int partition_number;

	  Clear_Screen();
	  printf("\n\n\nWarning:  You are about to DELETE a partition.  To go back, type \"0\"\n");
	  printf("          and press [ENTER].  Otherwise, enter the partition number to\n");
	  printf("          delete.\n\n\n");
	  printf("Partition number to delete: ");
	  fgets(input_buffer,10,stdin);

	  partition_number=atoi(input_buffer);

	  if((partition_number>=1) && (partition_number<=4))
	    {
	    Delete_Partition(drive,partition_number);
	    printf("\nPartition Deleted!\n");
	    Pause();
	    }
	  else
	    /* trap bad input */
	    {
	    printf("\n\n\nOperation Aborted.\n");
	    Pause();
	    partition_number=0;
	    strcpy(input_buffer,"null");
	    }
	  }

	/* Return to Main Menu */
	if(input==53)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);
      input=99; /* change value to prevent any other options from being executed */
      }

    /* Diplay partition information */
    if(input==52)
      {
      Clear_Screen();
      printf("Fixed Disk Drive: %d",drive-127);
      Display_Partition_Table(drive,STANDARD);
      Pause();
      }

    /* Change current fixed disk drive */
    if(input==53)
      {
      Clear_Screen();
      List_Available_Drives();
      printf("Enter Fixed Disk Drive Number: ");

      /* Get input from keyboard. */
      input=Input();

      input=input-47;

      /*check to make sure the drive is a legitimate number*/
      if( (input<2) || (input>6) )
	{
	printf("\nDrive number is out of range, no changes have been made.\n");
	Pause();
	}

      drive=input+126;
      }

    /* Exit */
    if(input==54) exit=TRUE;

    }while(exit==FALSE);

  Clear_Screen();
  if(sectors_have_changed==TRUE)
    {
    printf("\n\n\nReboot the computer to ensure that any changes take effect.\n\n");
    }
}

/* Modify Partition Type of Primary Partition */
void Modify_Partition_Type(int drive,int partition_number,int type_number)
{
  long offset;

  Read_Sector(drive,0,0,1);

  offset=((partition_number*16)-16)+(0x1be+4);

  sector_buffer[offset]=type_number;

  Write_Sector(drive,0,0,1);
}

/* Pause Routine */
void Pause()
{
  printf("\nPress any key to continue.\n");

  asm{
    mov ah,7
    int 0x21
    }
}

/* Read Partition Table */
void Read_Partition_Table(int drive,int view_type)
{
  long index=0x1be;

  int exiting_primary=TRUE;
  int extended=FALSE;
  int extended_pointer=1;
  int partition_designation=PRIMARY;
  int pointer=0;
  int record_extended_info_flag=FALSE;

  int done_flag=FALSE;

  unsigned long extended_cylinder;
  unsigned long extended_head;
  unsigned long extended_sector;

  whole_extended_partition_size=0;
  extended_partition_space_used=0;
  hard_disk_space_used=0;

  Read_Sector(drive,0,0,1);

  do{
    if(pointer==4) partition_designation=EXTENDED;

    if((pointer>=4) && (extended==TRUE))
      {
      Read_Sector(drive,extended_head,extended_cylinder,extended_sector);
      extended=FALSE;
      index=0x1be;

      if(exiting_primary==FALSE)
	{
	pointer--;
	}
      else
	{
	exiting_primary=FALSE;
	}
      }

    /* Flag boot partition */
    if((sector_buffer[index]==128) && (partition_designation==PRIMARY)) active_flag[pointer]=TRUE;

    /* Determine Partition Type */
    numeric_partition_type[pointer]=sector_buffer[index+4];

    if(sector_buffer[index+4]==0x00)
      {
      if(partition_lookup_table==INTERNAL) strcpy(partition_type[pointer],"Unused       ");

      if(partition_designation==EXTENDED)
	{
	number_of_partitions=pointer;
	done_flag=TRUE;
	}
      }

    if(sector_buffer[index+4]==0x05)
      {
      if(partition_lookup_table==INTERNAL) strcpy(partition_type[pointer],"Extended     ");
      extended=TRUE;
      extended_partition_exists=TRUE;
      record_extended_info_flag=TRUE;
      }

    if(partition_lookup_table==EXTERNAL)
      {
      Scan_External_Lookup_Table(sector_buffer[index+4],pointer,view_type);
      }
    else
      {
      if(sector_buffer[index+4]==0x01) strcpy(partition_type[pointer],"DOS-12       ");
      if(sector_buffer[index+4]==0x02) strcpy(partition_type[pointer],"XENIX root   ");
      if(sector_buffer[index+4]==0x03) strcpy(partition_type[pointer],"XENIX usr    ");
      if(sector_buffer[index+4]==0x04) strcpy(partition_type[pointer],"DOS-16 <32M  ");
      if(sector_buffer[index+4]==0x06) strcpy(partition_type[pointer],"DOS-16 >=32M ");
      if(sector_buffer[index+4]==0x07) strcpy(partition_type[pointer],"OS/2 HPFS    ");
      if(sector_buffer[index+4]==0x08) strcpy(partition_type[pointer],"AIX          ");
      if(sector_buffer[index+4]==0x09) strcpy(partition_type[pointer],"AIX bootable ");
      if(sector_buffer[index+4]==0x0a) strcpy(partition_type[pointer],"OS/2 boot mgr");
      if(sector_buffer[index+4]==0x0b) strcpy(partition_type[pointer],"Win 95 FAT32 ");
      if((sector_buffer[index+4]>=0x0c) && (sector_buffer[index+4]<0x64)) strcpy(partition_type[pointer],"Unknown      ");
      if(sector_buffer[index+4]==0x64) strcpy(partition_type[pointer],"Novell       ");
      if(sector_buffer[index+4]==0x65) strcpy(partition_type[pointer],"Novell       ");
      if((sector_buffer[index+4]>=0x66) && (sector_buffer[index+4]<0x80)) strcpy(partition_type[pointer],"Unknown      ");
      if(sector_buffer[index+4]==0x80) strcpy(partition_type[pointer],"Old MINIX    ");
      if(sector_buffer[index+4]==0x81) strcpy(partition_type[pointer],"Linux/MINIX  ");
      if(sector_buffer[index+4]==0x82) strcpy(partition_type[pointer],"Linux Swap   ");
      if(sector_buffer[index+4]==0x83) strcpy(partition_type[pointer],"Linux Native ");
      if(sector_buffer[index+4]>0x83) strcpy(partition_type[pointer],"Unknown      ");
      }

    starting_head[pointer] = sector_buffer[index+1];
    ending_head[pointer] = sector_buffer[index+5];

    true_partition_size[pointer]=Decimal_Number(sector_buffer[index+12],sector_buffer[index+13],sector_buffer[index+14]);
    partition_size[pointer]=true_partition_size[pointer]/2000;

    if(pointer<4)
      {
      hard_disk_space_used=hard_disk_space_used+true_partition_size[pointer];
      }

    if((sector_buffer[index+0x04]==0x05)&&(pointer<4))
      {
      whole_extended_partition_size=true_partition_size[pointer];
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+2],sector_buffer[index+3]);

    starting_cylinder[pointer]=g_cylinder;
    starting_sector[pointer]=g_sector;

    if((extended==TRUE) && (record_extended_info_flag==TRUE))
      {
      extended_cylinder=starting_cylinder[pointer];
      extended_head=starting_head[pointer];
      extended_sector=starting_sector[pointer];

      extended_partition_cylinder[extended_pointer]=extended_cylinder;
      extended_partition_head[extended_pointer]=extended_head;
      extended_partition_sector[extended_pointer]=extended_sector;

      extended_pointer++;

      record_extended_info_flag=FALSE;
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+6],sector_buffer[index+7]);

    ending_cylinder[pointer]=g_cylinder;
    ending_sector[pointer]=g_sector;

    number_of_partitions=pointer;
    pointer++;

    if((extended==FALSE) && (pointer==4))
      {
      number_of_partitions=4;
      done_flag=TRUE;
      }

    index=index + 16;
    } while(done_flag==FALSE);

  if(number_of_partitions>4)
    {
    index=5;
    do
      {
      extended_partition_space_used=extended_partition_space_used+true_partition_size[index-1];
      index++;
      }while(index<=number_of_partitions);
    }
}

/* Read_Sector */
void Read_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(2, drive, head, cyl, sector, 1, sector_buffer);

    if (result!=0)
      {
      printf("\nDisk read error...operation terminated.\n");
      exit(1);
      }
}

/* Remove MBR */
void Remove_MBR(int drive)
{
  long pointer=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[pointer]=0x00;
    pointer++;
    }while(pointer<0x1be);

  Write_Sector(drive,0,0,1);
}

/* Save MBR */
void Save_MBR(int drive)
{
  long index=0;

  FILE *file_pointer;

  Read_Sector(drive,0,0,1);

  file_pointer = fopen("boot.bin","wb");

  if(!file_pointer)
    {
    printf("\nError opening or creating \"BOOT.BIN\" for writing...Operation Terminated.\n");
    exit(1);
    }

  do
    {
    fputc(sector_buffer[index],file_pointer);
    index++;
    }while(index<0x1be);

  do{
    fputc(0,file_pointer);
    index++;
    }while(index<512);

  fclose(file_pointer);
}

/* Scan External Lookup Table */
void Scan_External_Lookup_Table(int indicated_partition_type,int partition,int view_type)
{
  int counter;
  int index=0;

  char possible_partition_type[3];

  do
    {
    strncpy(possible_partition_type,partition_lookup_table_buffer[index],3);
    if(indicated_partition_type==atoi(possible_partition_type))
      {
      if(view_type==STANDARD)
	{
	counter=0;
	do
	  {
	  partition_type[partition][counter]=partition_lookup_table_buffer[index][(counter+18)];
	  counter++;
	  }while(counter<50);
	index=256;
	}
      else
	{
	counter=0;
	do
	  {
	  partition_type[partition][counter]=partition_lookup_table_buffer[index][(counter+4)];
	  counter++;
	  }while(counter<13);
	index=256;
	}
      }
    index++;
    }while(index<=255);
}

/* Toggle Active Partition */
void Toggle_Active_Partition(int drive, long partition_number, int toggle)
{
  int value;

  long offset;

  Read_Sector(drive,0,0,1);

  if(1==toggle)
    {
    value=0x80;
    }
  else
    {
    value=0x00;
    }

  offset=((partition_number*16)-16)+0x1be;

  sector_buffer[offset]=value;

  Write_Sector(drive,0,0,1);
}

/* Write Sector */
void Write_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(3, drive, head, cyl, sector, 1, sector_buffer);

  if (result!=0)
    {
    printf("\nDisk write error...Operation Terminated.\n");
    exit(1);
    }

  sectors_have_changed=TRUE;
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int true;

  int drive;

  /* Is there an "part.dat" file? If so, use it. */
  if(External_Lookup_File_Exists()==TRUE)
    {
    partition_lookup_table=EXTERNAL;
    Load_External_Lookup_Table();
    }
  else partition_lookup_table=INTERNAL;

  /* if FDISK is typed without any options */
  if(argc==1)
    {
    Menu_Routine();
    exit(0);
    }

  /* if FDISK is typed with options */
  if(argc>=2)
    {

    /* if "FDISK /?" is typed */
    true=1;
    true=strcmp("/?",argv[1]);
    if(0==true)
      {
      Display_Help_Screen();
      exit(0);
      }

    /* if "FDISK /L" is typed */
    true=1;
    true=stricmp("/L",argv[1]);
    if(0==true)
      {
      List_Available_Drives();
      exit(0);
      }

    /*if there is a 2nd option, convert the option to a drive # */
    if(argc>=3)
      {
      drive=argv[2] [0];

      drive=drive-47;

      /*check to make sure the drive is a legitimate number */
      if( (drive<2) || (drive>6) )
	{
	printf("\nIncorrect drive designation...Operation Terminated.\n");
	exit(1);
	}

      drive=drive+126;
      }

    /*if no drive number is entered, set the drive to the first physical drive */
    if(argc<3)
      {
      drive=128;
      }

    /*if "FDISK /C" is typed */
    true=1;
    true=stricmp("/C",argv[1]);
    if(0==true)
      {
      Clear_Partition_Table(drive);
      exit(0);
      }

    /*if "FDISK /I [/drive# [/tech]]" is typed */
    true=1;
    true=stricmp("/I",argv[1]);

    if(0==true)
      {
      if(argc==4)
	{
	true=1;
	true=stricmp("/TECH",argv[3]);

	if(0==true)
	  {
	  Clear_Screen();
	  Display_Partition_Table(drive,TECHNICAL);
	  exit(0);
	  }
	else
	  {
	  printf("\nSyntax Error...Operation Terminated.\n");
	  exit(1);
	  }
	}
      else
	{
	Clear_Screen();
	Display_Partition_Table(drive,STANDARD);
	exit(0);
	}
      }

    /* if "FDISK /N" is typed*/

    /*   Syntax: FDISK /N drive# partition_type size [/P] [/S type]*/

    true=1;
    true=stricmp("/N",argv[1]);

    if(0==true)
      {
      int partition_type=NULL;
      int percentage_flag=NULL;
      int special_flag=NULL;
      int special_partition_type=NULL;


      unsigned long size;

      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Determine type */
      true=1;
      true=stricmp("P",argv[3]);
      if(0==true) partition_type = PRIMARY;

      true=1;
      true=stricmp("E",argv[3]);
      if(0==true) partition_type = EXTENDED;

      true=1;
      true=stricmp("L",argv[3]);
      if(0==true) partition_type = LOGICAL;

      if(partition_type==NULL)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Set partition size */

      size=atol(argv[4]);

      if(size>2048)
	{
	printf("\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	exit(1);
	}

      /* Set percentage_flag, if necessary. */
      if((argc==6) || (argc==8))
	{
	true=1;
	true=stricmp("/P",argv[5]);
	if(0==true) percentage_flag = PERCENTAGE;

	if(argc==8)
	  {
	  true=1;
	  true=stricmp("/P",argv[7]);
	  if(0==true) percentage_flag = PERCENTAGE;
	  }
	}

      /* Set special partition type, if necessary. */
      if((argc>=6) && (argc<=8))
	{
	true=1;
	true=stricmp("/S",argv[5]);
	if(0==true)
	  {
	  special_flag = SPECIAL;
	  special_partition_type=atoi(argv[6]);
	  }

	if(argc==8)
	  {
	  true=1;
	  true=stricmp("/S",argv[6]);
	  if(0==true)
	    {
	    special_flag = SPECIAL;
	    special_partition_type=atoi(argv[7]);
	    }
	  }
	}

      if( (special_flag==SPECIAL) && ( (special_partition_type<1) || (special_partition_type>255)  )  )
	{
	printf("\nSpecial partition type is out of range...Operation Terminated.\n");
	exit(1);
	}

      Create_New_Partition(drive,partition_type,size,percentage_flag,special_flag,special_partition_type);
      exit(0);
      }

    /* if "FDISK /T" is typed */

    /* Syntax:  FDISK [/T drive# partition# active_toggle] */
    true=1;
    true=stricmp("/T",argv[1]);
    if(0==true)
      {
      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}
      int partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	exit(1);
	}

      int toggle=atoi(argv[4]);

      if((toggle<0) || (toggle>1))
	{
	printf("\nInvalid toggle setting...Operation Terminated.\n");
	exit(1);
	}

      Toggle_Active_Partition(drive,partition_number,toggle);
      exit(0);
      }

    /* if "FDISK /D" is typed */

    /* Syntax:  FDISK [/D drive# partition#] */
    true=1;
    true=stricmp("/D",argv[1]);
    if(0==true)
      {
      int partition_number;

      if(argc<4)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      true=1;
      true=stricmp("/L",argv[3]);
      if(0==true)
	{
	Read_Partition_Table(drive,TECHNICAL);

	if(number_of_partitions<5)
	  {
	  printf("\nNo logical partitions exist...Operation Terminated.\n");
	  exit(1);
	  }
	partition_number=LAST;
	}
      else
	{
	partition_number=atoi(argv[3]);

	if((partition_number<1) || (partition_number>4))
	  {
	  printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	  exit(1);
	  }
	}
      Delete_Partition(drive,partition_number);
      exit(0);
      }

    /* if "FDISK /MBR" is typed */

    /* Syntax:  FDISK [/MBR drive#] */
    true=1;
    true=stricmp("/MBR",argv[1]);
    if(0==true)
      {
      Create_MBR(drive);
      exit(0);
      }

    /* if "FDISK /RMBR" is typed */

    /* Syntax:  FDISK [/RMBR drive#] */
    true=1;
    true=stricmp("/RMBR",argv[1]);
    if(0==true)
      {
      Remove_MBR(drive);
      exit(0);
      }

    /* if "FDISK /AMBR" is typed */

    /* Syntax: FDISK [/AMBR drive#] */
    true=1;
    true=stricmp("/AMBR",argv[1]);
    if(0==true)
      {
      Create_Alternate_MBR(drive);
      exit(0);
      }

    /* if "FDISK /SMBR" is typed */

    /* Syntax: FDISK [/SMBR drive#] */
    true=1;
    true=stricmp("/SMBR",argv[1]);
    if(0==true)
      {
      Save_MBR(drive);
      exit(0);
      }

    /* if "FDISK [/M drive# primarypartition# newtype#]" is typed */
    true=1;
    true=stricmp("/M",argv[1]);
    if(0==true)
      {
      int partition_number;
      int type_number;

      partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPrimary partition number is out of range...Operation Terminated.\n");
	exit(1);
	}

      type_number=atoi(argv[4]);

      if((type_number<=0) || (type_number>255))
	{
	printf("\nNew partition type is out of range...Operation Terminated.\n");
	exit(1);
	}

      Modify_Partition_Type(drive,partition_number,type_number);
      }
    }
}
