/*----------------------------------------------------------------------------+
|   Copyright (C) 2003  Hsu-Ping Feng                                         |
|                                                                             |
|   This program is free software; you can redistribute it and/or modify      |
|   it under the terms of the GNU General Public License as published by      |
|   the Free Software Foundation; either version 2 of the License, or         |
|   (at your option) any later version.                                       |
|                                                                             |
|   This program is distributed in the hope that it will be useful,           |
|   but WITHOUT ANY WARRANTY; without even the implied warranty of            |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             |
|   GNU General Public License for more details.                              |
|                                                                             |
|   You should have received a copy of the GNU General Public License         |
|   along with this program; if not, write to the Free Software               |
|                                                                             |
|   Foundation, Inc., 59 Temple Place, Suite 330,                             |
|   Boston, MA  02111-1307 USA                                                |
|                                                                             |
|                                                                             |
|   Author e-mail: spferng@ksts.seed.net.tw                                   |
+----------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "bootmgr.h"
#include "selemenu.h"
#include "common.h"
#include "global.h"
#include "keymap.h"

/*============================  o    m ============================*/
static sword Where_Is (sword type, dword *begin, PARTN *ptr)
{
    dword  fat_nSect, offset;
    byte   buff[512];
    sword  fat;
    DOSBPB *bpb;
    
    if ( chs_ReadWDK(ptr->side, ptr->stCyl, ptr->sector, 1, buff) )
        return(-1);

    fat = IsFAT(buff);
    if ( fat == 0 )
        return(-2);
    
    if ( type == BOOT )
    {
        *begin = GetStart(ptr);
        return(0);
    } /* end if */
    
    bpb = (DOSBPB *)buff;
    offset = bpb->comm.nBootSect;
    if ( type != FAT_1 )                                          /** FAT_2 **/
    {
        fat_nSect = (fat == 32) ? bpb->fat32.nFatSect : bpb->fat1x.nFatSect;
        offset += fat_nSect;
        if ( type != FAT_2 )                                       /** Root **/
            offset += fat_nSect;
    } /* end if */
    
    *begin = GetStart(ptr) + offset;                        /** Ǧ^ҩlϰ **/
    
    return(0);
} /* end Where_Is */


/*=========================   s     e =========================*/
static sword Comp_Sectors (DiskDT *dk, dword start1, dword start2, dword nSects)
{
    byte  buff_1[512], buff_2[512];

    for ( ; nSects > 0 ; nSects--, start1++, start2++ )
        if ( dk->drv->linearRead(dk, start1, 1, buff_1) ||
             dk->drv->linearRead(dk, start2, 1, buff_2) ||
             CompMEM(buff_1, buff_2, 512) )
        {
            return(-1);
        } /* end if */

    return(0);
} /* end Comp_Sectors */


/*============================   t    ============================*/
static void Show_System_Sector (sword type, PARTN *ptr)
{
    dword  begin, cyl, head, sector;
    sword  err;
    
    if ( (err = Where_Is(type, &begin, ptr)) == 0 )
    {
        Redraw_Fdisk_Picture(TRUE);
        Sect_to_CHS(begin, &cyl, &head, &sector, CUR_DK);
        if ( type != BOOT && type != ROOT )
            type = HEX_DUMP;
        ShowSector(cyl, head, sector, type);
    }
    else
        ShowError( ( err == -1 ) ? 23 : 21 );
} /* end Show_System_Sector */


/*=======================   FAT     BPB =========================*/
static void ShowBPB (sword fat, sword num, byte *buff)
{
    DOSBPB  *bpb = (DOSBPB *)buff;
    
    switch( num )
    {        
        case 0:                                                  /** OEM ID **/
            ShowRangeStr(33, 128, bpb->comm.oem, 8, NULL);
            break;
        
        case 1:                                        /** CӺϰ byte  **/
            c_printf("%-5u", bpb->comm.bytePerSect);
            break;
        
        case 3:                           /** ҰʺϰϨFATZAOdϰϼ **/
            c_printf("%-5u", bpb->comm.nBootSect);
            break;
        
        case 9:                                          /** Cyhֺϰ **/
            c_printf("%-5u", bpb->comm.sectPerTrack);
            break;
        
        case 10:                                         /** CϬWh֭ **/
            c_printf("%-5u", bpb->comm.nHead);
            break;
        
        case 2:                                        /** CӺLϰϼ **/
            c_printf("%-3u", bpb->comm.sectPerClust);
            break;
        
        case 4:                                                /** FAT  **/
            c_printf("%-3u", bpb->comm.nFAT);
            break;
        
        case 5:                                            /** ڥؿ` **/
            c_printf("%-5u", ((fat == 32) ? bpb->comm.sectPerClust :
                                            bpb->fat1x.nFdbRoot / 16) );
            break;
        
        case 6:                               /** κϰ`(p32Mϥ) **/
            if ( bpb->fat1x.tSector2 == 0 )
            {
                #if ( DISPLAY == CHINESE )
                    c_printf("%-13s", "ϥ");
                #else 
                    c_printf("%-13s", "Unused");
                #endif
            }
            else
                c_printf("%-10u (%04Xh)", bpb->fat1x.tSector2, bpb->fat1x.tSector2);
            break;
        
        case 7:                               /** CyzrAwЬ 0xf8 **/
            c_printf("%02Xh", bpb->comm.m_descriptor);
            break;
        
        case 8:                                       /** C FAT ϰϼ **/
            if ( fat == 32 )
                c_printf("%-10lu (%08lXh)",
                         bpb->fat32.nFatSect, bpb->fat32.nFatSect);
            else
                c_printf("%-10u (%04Xh)",
                         bpb->fat1x.nFatSect, bpb->fat1x.nFatSect);
            break;
        
        case 11:                     /** ΪҰʺϰϪZ(úϰϼ) **/
            c_printf("%-10lu (%08lXh)",
                  bpb->comm.nHiddSect, bpb->comm.nHiddSect);
            break;
        
        case 12:                              /** `ϰϼ(j32Mϥ) **/
            if ( bpb->comm.tSector4 == 0 )
            {
                #if ( DISPLAY == CHINESE )
                   c_printf("%-22s", "ϥ");
                #else 
                   c_printf("%-22s", "Unused");
                #endif
            }
            else
                c_printf("%-10lu (%08lXh)",
                         bpb->comm.tSector4, bpb->comm.tSector4);
            break;
        
        case 13:                                       /** Ĥ@wЪs **/
            c_printf("%02Xh", ((fat == 32) ? bpb->fat32.firstHD : bpb->fat1x.firstHD) );
            break;
        
        case 14:                                /** ҰʺϰϯSxX(29h) **/
            c_printf("%02Xh", ((fat == 32) ? bpb->fat32.signature : bpb->fat1x.signature) );
            break;
        
        case 15:                                               /** ϺЧǸ **/
            if ( fat == 32 )
                c_printf("%04X-%04X(hex)", bpb->fat32.serNoHigh, bpb->fat32.serNoLow);
            else
                c_printf("%04X-%04X(hex)", bpb->fat1x.serNoHigh, bpb->fat1x.serNoLow);
            break;
        
        case 16:                                               /** Ϻм **/
            ShowRangeStr(33, 128,
                         ((fat == 32) ? bpb->fat32.label : bpb->fat1x.label), 11, NULL);
            break;
        
        case 17:                                           /** ɮרtκ **/
            ShowRangeStr(33, 128,
                         ((fat == 32) ? bpb->fat32.fsType : bpb->fat1x.fsType), 8, NULL);
            break;
    } /* end switch */
} /* end ShowBPB */


/*============================   t    ============================*/
void ShowBootRecord (byte *buff)
{
    #if ( DISPLAY == CHINESE )
        #define X_OFFSET      4
        #define SEME_COMM     "G "
        static sbyte *title[] =
        {
            "OEM ID",
            "CӺϰϪ줸",
            "CӺL(Cluster)ϰϼ",
            "ҰʺϰϨFATZ(Odϰϼ)",
            "FAT ",
            "ڥؿTwϰϼ",
            "`ϰϼ(p32Mϥ)",
            "Cyzr",
            "C FAT ϰϼ",
            "Cyϰϼ",
            "CϬW(Sides)`",
            "Ұʺϰϫeϰϼ(úϰϼ)",
            "`ϰϼ(j32Mϥ)",
            "Ĥ@wЪs",
            "ҰʺϰϯSxX(29h)",
            "ϺЧǸ",
            "ϺмЦW",
            "ɮרtκ"
        };
    #else
        #define X_OFFSET      5   
        #define SEME_COMM     ": "
        static sbyte *title[] =
        {
            "OEM ID",
            "Bytes per sector",
            "Sectors per cluster",
            "Reserved sectors at beginning",
            "FAT Copies",
            "Root directory fix sector",
            "Total sectors of the partition",
            "Media descriptor byte",
            "Sectors per FAT",
            "Sectors per track",
            "Sides per cylinder",
            "Special hidden sectors",
            "Big total number of sectors",
            "Physical drive number",
            "Extended Boot Record Signature",
            "Volume Serial Number",
            "Volume Label",
            "The kind of File System"
        };   
    #endif /* DISPLAY == CHINESE */
    
    sword  fat, i;
    
    if ( (fat = IsFAT(buff)) == 0 )
        fat = 16;
    
    for ( i = 0 ; i < 18 ; i++ )
    {    
        c_printXY(X_OFFSET, 4 + i, "%34.34s" SEME_COMM, title[i]);
        c_textattr(MESSAGE_COLOR);
        ShowBPB(fat, i, buff);
        c_textattr(DEFAULT_COLOR);
    } /* end for */
} /* end ShowBootRecord */


/*============================     FDB ==============================*/
void ShowDIR (byte *buff)
{
    #if ( DISPLAY == CHINESE )
        #define TITLE_MSG    "DɦW   [W A       jp         ɶ    ҩlL  R H S V D A"
        #define UNUSED_MSG   "եϥΡ"
        #define LFNAME_FMT   "   ɦW  ǸG%2Xh   [`G%02Xh %23s"
        #define DELE_MSG     "R"
        #define DIR_MSG      "ؿ"
        #define VOL_MSG      ""
        #define FILE_MSG     "ɮ"
    #else
        #define TITLE_MSG    "Name      .Ext  Type       Size    Date     Time    ST_Clust  R H S V D A"
        #define UNUSED_MSG   "< Unused >"
        #define LFNAME_FMT   "   <LFN>   SerNO.%2Xh   CheckSum: %02Xh %23s"
        #define DELE_MSG     "Dele"
        #define DIR_MSG      "DIR"
        #define VOL_MSG      "VOL"
        #define FILE_MSG     "File"
    #endif  /* DISPLAY == CHINESE */
     
    sbyte  attrStr[12], date[11], time[9], *type;
    byte   longname[13];
    sword  mask, i, j, y;
    dword  firstClust;
    FDB    *fdb;
 
    c_printXY(4, 4, TITLE_MSG);
 
    c_gotoxy(3, 5);
    Repeat_Char('-', 76);
    for ( y = 6,  i = 0 ; i < 512 ; i += 32,  y++ )
    {
        if ( *(buff + i) == 0 )
        {
            c_textattr(NO_USE_COLOR);
            for ( ; y <= 21 ; y++ )
                c_printXY(4, y, "%-73.73s", UNUSED_MSG);
            break;
        } /* end if */
  
        fdb = (FDB *)(buff + i);
  
        mask = 1;
        strcpy(attrStr, "R H S V D A");
        for ( j = 0 ; j < 6 ; j++, mask <<= 1 )
            if ( (fdb->SFN.attr & mask) == 0 )
                attrStr[j * 2] = '-';
        
        c_gotoxy(4, y);
        if ( fdb->SFN.attr == 0x0f )                               /** ɦW **/
        {
            for ( j = 0 ; j < 5 ; j++ )
                longname[j] = fdb->LFN.name10[j * 2];
            for ( j = 0 ; j < 6 ; j++ )
                longname[j + 5] = fdb->LFN.name12[j * 2];
            for ( j = 0 ; j < 2 ; j++ )
                longname[j + 11] = fdb->LFN.name4[j * 2];
            longname[13] = 0;
   
            for ( j = 0 ; j < 13 && longname[j] ; j++ );
            for ( ; j < 13 ; j++ )
                longname[j] = ' ';
   
            c_textattr(LONGNAME_COLOR);
            ShowRangeStr(32, 254, longname, 13, NULL);
   
            c_printf(LFNAME_FMT, fdb->LFN.serNO, fdb->LFN.chkSum, attrStr);
        }
        else
        {
            firstClust = ((dword)fdb->SFN.clustHigh << 16) + fdb->SFN.clustLow;
            sprintf(date, "%02d/%02d/%04d", fdb->SFN.date.mon,
                                            fdb->SFN.date.day,
                                            fdb->SFN.date.year + 1980);
            sprintf(time, "%02d:%02d", fdb->SFN.time.hour, fdb->SFN.time.min);
   
            if ( fdb->SFN.name[0] == 0xe5 )
            {
                c_textattr(DELE_FILE_COLOR);
                type = DELE_MSG;
            }
            else if ( (fdb->SFN.attr & 0x10) != 0 )
            {
                c_textattr(DIRECTORY_COLOR);
                type = DIR_MSG;
            }
            else if ( (fdb->SFN.attr & 0x08) != 0 )
            {
                c_textattr(VOLUMN_COLOR);
                type = VOL_MSG;
            }
            else
            {
                c_textattr(FILE_COLOR);
                type = FILE_MSG;
            } /* end if */
   
            ShowRangeStr(32, 254, fdb->SFN.name, 8, NULL);
            c_printf("  ");
            ShowRangeStr(32, 254, fdb->SFN.ext, 3, NULL);
            c_printf("   %-4.4s %10lu  %10.10s %5.5s %10lu  %s",
                     type, fdb->SFN.size, date, time, firstClust, attrStr);
        } /* end if */
    } /* end for */
 
    c_textattr(DEFAULT_COLOR);
} /* end ShowDIR */


/*=========================  L      } =========================*/
sword IsDirSector (dword sector)
{
    sbyte  buff[512];
    
    if ( linear_ReadWDK(sector, 1, buff) )
        return( FALSE );
    return( strncmp(buff, ".          ", 11) == 0 &&
            strncmp(buff+32, "..         ", 11) == 0 );
} /* end IsDirSector */


/*=========================  L      } =========================*/
dword ClusterToSectorOffset (DOSBPB *bpb, dword clust)
{
    sword  type = IsFAT((byte *)bpb);
    dword  sect;

    if ( type == 0 )
    {
        sect = 0;
    }
    else if ( type == 32 )
    {
        sect = bpb->comm.nBootSect + bpb->fat32.nFatSect * 2;
        if ( clust > 2 )
            sect += ((clust - 2) * bpb->comm.sectPerClust);
    }
    else
    {
        sect = bpb->comm.nBootSect + bpb->fat1x.nFatSect * 2;
        if ( clust > 1 )
            sect += ((clust - 2) * bpb->comm.sectPerClust + 32);
    } /* end if */
    
    return( sect );
} /* end ClusterToSectorOffset */


/*======================  o U @  cluster s  =======================*/
dword GetFatValue (PARTN *ptr, DOSBPB *bpb, dword clust)
{
    sword  type = IsFAT((byte *)bpb);
    dword  sect, val;
    sword  off, i;
    byte   buff[512];

    if ( clust < 2 )
        return(0);

    switch( type )
    {
        case 32:
            val = clust * 4;
            break;
        case 16:
            val = clust * 2;
            break;
        case 12:
            val = clust * 3 / 2;
            break;
        default:
            return(0);
    } /* end switch */

    sect = val / 512 + bpb->comm.nBootSect;
    off  = (sword)(val % 512);

    for ( i = 0 ; i < 2 ; i++ )
    {
        if( linear_ReadWDK(GetStart(ptr) + sect, 1, buff) == 0 )
            break;
        sect += ((type == 32) ? bpb->fat32.nFatSect : bpb->fat1x.nFatSect);
    } /* end for */
    
    if ( i >= 2 )
        return(0);

    switch( type )
    {
        case 32:
            val = *(dword *)&buff[off];
            break;
        case 16:
            val = (dword)( *(word *)&buff[off] );
            break;
        case 12:
            val = (dword)( *(word *)&buff[off] );
            val = ((clust % 2) ? val : (val >> 4)) & 0x0fff;
            break;
    } /* end switch */

    return( val );
} /* end GetFatValue */


/*======================    w  cluster  e =======================*/
sword ReadDosBootSect (PARTN *ptr, byte *buff)
{
    sword  fatType;

    if ( linear_ReadWDK(GetStart(ptr), 1, buff) )          /** ŪҰʺϰ **/
        return(-1);

    fatType = IsFAT(buff);
    if ( fatType == 0 )
        return(-2);
    return( fatType );
} /* end ReadDosBootSect */


/*=========================  J  cluster s  ==========================*/
static sword InputCluster (DOSBPB *bpb, dword *clust, dword *maxclust)
{
    #if ( DISPLAY == CHINESE )
        #define INPUT_CLUSTER_NUM   "  J  L s (0-%lu)G "
    #else
        #define INPUT_CLUSTER_NUM   "Input cluster number(0-%lu): "
    #endif

    dword   maxClust;
    sword   fatType;
    sbyte   inbuf[11];

    fatType = IsFAT((byte *)bpb);
    
    /* get max cluster number */
    if ( fatType == 32 )
        maxClust = bpb->fat32.nFatSect * 512UL / 4;
    else
    {
        maxClust = bpb->fat1x.nFatSect * 512UL;
        maxClust = ( fatType == 16 ) ? maxClust / 2 : maxClust * 2 / 3;
    } /* end if */

    *inbuf = 0;
    do
    {
        ShowMSG(INPUT_CLUSTER_NUM, maxClust - 1);
        if ( GetStr(inbuf, 10, LIMIT_NUMERIC | ALLOW_ESC) == ESC_KEY || !*inbuf )
            return( FALSE );
        *clust = strtoul(inbuf, NULL, 10);
    } while ( *clust >= maxClust );
    
    if ( maxclust )
        *maxclust = maxClust;
    return( TRUE );
} /* end InputCluster */


/*=================== x s  w  cluster link   =====================*/
sword SaveClusterLink (PARTN *ptr)
{
    dword   maxClust, startSect, clust, val;
    byte    buff[512];
    sbyte   data[512];
    sword   fatType, nSect, ret, i;
    FILE    *fptr;
    DOSBPB  *bpb = (DOSBPB *)buff;

    fatType = ReadDosBootSect(ptr, buff);
    if ( fatType <= 0 )
        return( fatType );

    if ( !InputCluster(bpb, &clust, &maxClust) )
        return(0);
        
    if ( !Input_Filename(data, 42) || !Is_Overwrite(data) )
    {
        Cancel_Box();
        return(0);
    } /* end if */

    if ( (fptr = new_fopen(data, "wb")) == NULL )
        return(-2);

    ShowMSG(NULL);

    startSect = GetStart(ptr);
    ret = 0;
    
    if ( (fatType != 32) && (clust < 2) )                   /* fat12,16 root */
        nSect = bpb->fat1x.nFdbRoot / 16;
    else
        nSect = bpb->comm.sectPerClust;
    
    do
    {
        val = ClusterToSectorOffset(bpb, clust) + startSect;

        for ( i = 0 ; i < nSect ; i++ )
        {
            if ( linear_ReadWDK(val + i, 1, data) )
            {
                ret = -1;
                goto SAVE_FAILED;
            } /* end if */
          
            if ( new_fwrite(data, 512, 1, fptr) != 1 )
            {
                ret = -3;
                goto SAVE_FAILED;
            } /* end if */
        } /* end for */

        val = GetFatValue(ptr, bpb, clust);
        if ( val == 0 || val == clust || val >= maxClust )
            break;
        clust = val;
    } while ( 1 );

    SAVE_FAILED:

    new_fclose(fptr);
    return( ret );
} /* end SaveClusterLink */


/*======================    w  cluster  e =======================*/
sword ShowCluster (PARTN *ptr)
{
    #if ( DISPLAY == CHINESE )
        #define GOTO_NEXT_MSG       "~    U @   L   H"
        #define END_OF_LINK         "  w L  C"
    #else
        #define GOTO_NEXT_MSG       "Continue dump next cluster link ?"
        #define END_OF_LINK         "End of cluster link!"
    #endif

    dword   sector, head, cyl;
    dword   maxClust, clust, val;
    sword   showType, fatType;
    byte    buff[512];
    DOSBPB  *bpb = (DOSBPB *)buff;

    fatType = ReadDosBootSect(ptr, buff);
    if ( fatType < 0 )
        return( fatType );

    if ( !InputCluster(bpb, &clust, &maxClust) )
        return(0);

    ShowMSG(NULL);

    if ( fatType == 32 )
        showType = ( clust < 3 ) ? ROOT : HEX_DUMP;
    else
        showType = ( clust < 2 ) ? ROOT : HEX_DUMP;

    do
    {
        val = ClusterToSectorOffset(bpb, clust) + GetStart(ptr);
        Sect_to_CHS(val, &cyl, &head, &sector, CUR_DK);

        if ( (showType != ROOT) && IsDirSector(val) )
            showType = ROOT;

        ShowSector(cyl, head, sector, showType);   /** ܡBxsB^sϰ **/
        
        val = GetFatValue(ptr, bpb, clust);
        if ( val == 0 || val == clust || val >= maxClust )
        {
            Prompt_Msg_Box(END_OF_LINK, NULL);
            break;
        } /* end if */
        
        clust = val;
    } while ( YN_box(TALK_BOX, GOTO_NEXT_MSG) );

    return(0);
} /* end ShowCluster */


/*==============================   FAT  ===============================*/
static sword CopyFAT (sword fat_source, sword fat_target, PARTN *ptr)
{
    dword  fat_nSects, source, target;
    sword  error, fat;
    byte   buff[512];
    DOSBPB *bpb;
 
    error = 0;
    source = GetStart(ptr);                            /** oӷҦbϬW **/
    if ( linear_ReadWDK(source, 1, buff) )                 /** ŪҰʺϰ **/
        error = -1;
    else if ( (fat = IsFAT(buff)) == 0 )                     /** YO FAT **/
        error = -2;
    else
    {
        bpb = (DOSBPB *)buff;
        fat_nSects = ( fat == 32 ) ? bpb->fat32.nFatSect :  /**  FAT ϰϼ **/
                                     bpb->fat1x.nFatSect;
                                    
        source += bpb->comm.nBootSect;                  /** Ĥ@ FAT m **/
        target = source + fat_nSects * (fat_target - 1);   /** DXتϰ **/
        source += fat_nSects * (fat_source - 1);           /** DXӷϰ **/
  
        if ( CopySector(CUR_DK, source, target, fat_nSects) )        /*  */
            error = -1;
        ShowMSG(NULL);
    } /* end if */
 
    return( error );
} /* end CopyFAT */


/*========================      O _  T =======================*/
/*        Y     O  T  h  ^ ТA _ h  ^           */
/*===========================================================================*/
static sword IsBootSectorError (PARTN *partnArr, sword ndx)
{
    #if ( DISPLAY == CHINESE )
        #define COMPARE_TWO_FAT_EQU      "    FAT O _  DDD"
        #define SYS_ID_BE_MODIFIED       "t ID QץA|xs I"
    #else
        #define COMPARE_TWO_FAT_EQU      "It's compare FAT1 and FAT2....."
        #define SYS_ID_BE_MODIFIED       "System ID be modify, but no save !"
    #endif
     
    dword  fat_nSects, fat1_addr, fat2_addr, boot1_addr, boot2_addr;
    byte   bootrec[512];
    PARTN  *ptr = partnArr + ndx;
    sword  isFAT32, error, id;
    DOSBPB *bpb;
 
    bpb = (DOSBPB *)bootrec;
 
    error   = 0;
    isFAT32 = FALSE;
    boot1_addr = GetStart(ptr);
    if ( linear_ReadWDK(boot1_addr, 1, bootrec) )
        error = -1;
    else
    {
        id = IsFAT(bootrec);                                /** o FAT  **/
        if ( id == 32 )
        {
            isFAT32 = TRUE;
            fat_nSects = bpb->fat32.nFatSect;
        }
        else if ( id == 12 || id == 16 )
            fat_nSects = bpb->fat1x.nFatSect;
        else
            error = -1;
  
        if ( error == 0 )
        {
            fat1_addr = boot1_addr + bpb->comm.nBootSect;       /** FAT#1 m **/
            fat2_addr = fat1_addr + fat_nSects;                 /** FAT#2 m **/
   
            /*---------------------------------------------------+
            |      ˬdOO FAT B FAT eO_ۦP      |
            +---------------------------------------------------*/
            ShowMSG(COMPARE_TWO_FAT_EQU);

            #if 1
            {
                byte  buff[512];

                if ( linear_ReadWDK(fat1_addr, 1, buff) ||
                     (*(word *)buff != 0xfff8) || (buff[2] != 0xff) )
                {
                    ShowMSG(NULL);
                    error = -2;
                } /* end if */

                if ( linear_ReadWDK(fat2_addr, 1, buff) ||
                     (*(word *)buff != 0xfff8) || (buff[2] != 0xff) )
                {
                    ShowMSG(NULL);
                    error = -3;
                } /* end if */
            }
            #endif
   
            if ( Comp_Sectors(CUR_DK, fat1_addr, fat2_addr, fat_nSects) )
            {
                ShowMSG(NULL);
                error = -4;
            }
            else
            {  
                ShowMSG(NULL);
                id = ptr->old_id;
    
                if ( isFAT32 )
                {
                    /*--------------------------------------------+
                    |   d FAT32       O _  P  |
                    +--------------------------------------------*/
                    boot2_addr = boot1_addr + bpb->fat32.bakBootSect;
                    if ( Comp_Sectors(CUR_DK, boot1_addr, boot2_addr, 1) )
                        if ( linear_WriteWDK(boot2_addr, 1, bootrec) != 0 )
                            error = -3;
     
                    /*-------------------+
                    |   ץ System_id   |
                    +-------------------*/
                    if ( ptr->old_id != 0xb && ptr->old_id != 0xc )
                        ptr->id = ptr->old_id = ( ptr->endCyl > 1023 ) ? 0xc : 0xb;
                }
                else
                {
                    /*-------------------+
                    |   ץ System_id   |
                    +-------------------*/
                    if ( ptr->old_id != 1 && ptr->old_id != 4 &&
                         ptr->old_id != 6 && ptr->old_id != 0xe )
                    {
                        ptr->id = ptr->old_id = ( ptr->endCyl > 1023 ) ? 0xe : 6;
                    } /* end if */
                } /* end if */
    
                if ( ptr->old_id != id )                        /** Y ID **/
                {
                    if ( g_Info.fd.modified <= 0 )
                        g_Info.fd.modified = -1;
    
                    ShowMSG(SYS_ID_BE_MODIFIED);
                } /* end if */
            } /* end if */
        } /* end if */
    } /* end if */

    return( error );
} /* end IsBootSectorError */


/*============================       ============================*/
static sword Recovery_Bootrec (PARTN *partnArr, sword ndx)
{        
    byte    bootrec[512], fat[512], buff[512];
    dword   amount, count, begin, end, i;
    PARTN   *ptr = partnArr + ndx;
    sword   isFAT32, sw;
    DOSBPB  *bpb;
 
    if ( !IsAllowSavePartn() )
        return(0);
 
    bpb = (DOSBPB *)bootrec;
 
    sw      = FALSE;
    isFAT32 = FALSE;
    begin   = GetStart(ptr);
    end     = GetEnd(ptr);
    end     = begin + ((end - begin) / 2);
 
    i = begin + 1;
    ShowMSG("%lu ", end - i);
    for ( ; i <= end ; i++ )                             /** MĤ@ FAT **/
    {  
        if ( (i & 0x0fff) == 0 )
            c_printXY(13, 23, "%lu", end - i);
        
        if ( c_kbhit() && (WaitKey() == ESC_KEY) )
        {
            User_Interrupt_Box();
            return(0);
        } /* end if */
  
        if ( linear_ReadWDK(i, 1, buff) )
            continue;
           
        if ( (*(word *)buff == 0xfff8) && (buff[2] == 0xff) )  /** FAT }Y **/
        {
            if ( sw == FALSE )
            {
                sw = TRUE;
                CopyMem(fat, buff, 512);            /** ƥ FAT Ĥ@Ӻϰ **/
    
                /*---------------------------------------------------+
                |   Boot and FAT is not continuous and FAT is FAT32  |
                +---------------------------------------------------*/
                if ( i != (begin + 1) &&
                     (*(dword *)&buff[4] & 0x00ffffffL) == 0x00ffffffL )
                {
                    isFAT32 = TRUE;
                } /* end if */
    
                count  = i;                             /** Ĥ@ FAT m **/
                amount = (word)(i - begin);              /** oOdϰϼ **/ 
            }
            else                                        /** ĤG FAT m **/
                if ( CompMEM(fat, buff, 512) == 0 )   /** PĤ@ FAT ۦP **/
                {
                    count = i - count;              /** DX FAT κϰϼ **/
                    break;
                } /* end if */
        } /* end if */
    } /* end for */
 
 
    if ( i > end ) 
        return(-2);                                 /** 䤣ۦP FAT **/
    else
    {
        /*-----------------------------+
        |          t  ID        |
        +-----------------------------*/
        if ( (ptr->no < 5) && (ptr->endCyl > 1023) )
            sw = ( isFAT32 ) ? 0xc : 0xe;
        else
            sw = ( isFAT32 ) ? 0xb : 6;
  
        if ( (ptr->old_id != sw) || g_Info.fd.modified )
        {
            if ( ptr->old_id != sw )
            {
                ptr->id = ptr->old_id = sw;                  /** t ID **/
                if ( g_Info.fd.modified <= 0 )
                    g_Info.fd.modified = -1;
            } /* end if */
   
            if ( Write_Partn(partnArr, 0, 0) )               /** gJΪ **/
                return(-1);                   
        } /* end if */
  
        CreateBPB(partnArr, ndx, bpb);                         /**  BPB **/
        bpb->comm.nBootSect = (word)amount;
  
        if ( isFAT32 )                                  /** J FAT ϰϼ **/
        {
            Fill_Sector(CUR_DK, begin, (sword)amount, 0);
            bpb->fat32.nFatSect = count;
        }
        else
            bpb->fat1x.nFatSect = (word)count;
  
        if ( SaveDosBootSector(ptr, bpb, isFAT32) )        /** رҰʺϰ **/
            return(-1);
    } /* end if */
    
    return(0);
} /* end Recovery_Bootrec */


/*=======================  M         =======================*/
static sword Find_Boot_Secctor_Condition (byte *buff, void *para)
{
    (void)para;                          /* to avoid compile warning message */
    return( *(word *)&buff[510]  == 0xaa55        &&
            *(dword *)&buff[4]   != 0             &&
            *(dword *)&buff[8]   != 0             &&
            *(dword *)buff       != 0             &&
            *(dword *)buff       != 0x41615252UL  && 
            *(dword *)&buff[484] != 0x61417272UL );
} /* end Find_Boot_Secctor_Condition */


/*==========================  M    FAT =============================*/
static sword Find_FAT_Condition (byte *buff, void *para)
{
    (void)para;                          /* to avoid compile warning message */
    return( (*(word *)buff == 0xfff8) && (buff[2] == 0xff) );
} /* end Find_FAT_Condition */


/*====================  M         ========================*/
static sword Find_Directory_Condition (byte *buff, void *para)
{
    sbyte  *fname = (sbyte *)para;
    sword  i;
    
    for ( i = 0 ; i < 512 ; i += 32 )
        if ( CompMEM(buff + i, fname, 11) == 0 )                   /** ŦX **/
            return( TRUE );
    return( FALSE );
} /* Find_Directory_Condition */


/*===========================   j M  V =============================*/
sword SeleSearchUpDown (sword *sw)
{
    #if ( DISPLAY == CHINESE )
       #define SELECT_FIND_MSG  "[P]: W@ӡA [N]: U@"
       #define PROMPT_ACTION    "[  Q   j M ]"
    #else 
       #define SELECT_FIND_MSG  "[P]: Find Prev, [N]: Find Next,"
       #define PROMPT_ACTION    "[ Press Q to stop ]"
    #endif

    sword  key;

    do
    {
        key = toupper( Talk_Msg_Box(SELECT_FIND_MSG, PROMPT_ACTION) );
   
        if ( key == 'Q' )
            return( TRUE );
        else if ( key == 'N' )
            *sw = 1;
        else if ( key == 'P' )
            *sw = -1;
    } while ( (key != 'N') && (key != 'P') );

    return( FALSE );
} /* end SeleSearchUpDown */


/*==========================  M    FAT =============================*/
static void Finder (PARTN *ptr, sword (*cond)(byte *buff, void *para),
                    void *para, sword showMode)
{   
    dword  curr, begin, end;
    dword  cyl, head, sector;
    byte   buff[512];
    sword  sw = 1;
 
    begin = GetStart(ptr);
    end   = GetEnd(ptr);
    for ( curr = begin ; curr >= begin && curr <= end ; curr += sw )
    {
        if ( ((word)curr & 0xff) == 0 )
            c_printXY(13, 23, "%lu ", ((sw > 0) ? end - curr : curr - begin) );
  
        if ( linear_ReadWDK(curr, 1, buff) == 0 && cond(buff, para) )
        {
            Sect_to_CHS(curr, &cyl, &head, &sector, CUR_DK);
            ShowSector(cyl, head, sector, showMode);              /** ܺϰ **/
            ShowMSG(NULL);
            if ( SeleSearchUpDown(&sw) )
                return;
        } /* end if */
  
        if ( c_kbhit() && (WaitKey() == ESC_KEY) )
        {
            if ( SeleSearchUpDown(&sw) )
                return;
            ShowMSG(NULL);
            c_printXY(13, 23, "%lu ", ((sw > 0) ? end - curr : curr - begin) );
        } /* end if */
    } /* end for */
 
    ShowMSG(NULL);
 
    #if ( DISPLAY == CHINESE )
        Prompt_Msg_Box("j M   I", NULL);
    #else 
        Prompt_Msg_Box("Search done !", NULL);
    #endif
} /* end Finder */


/*=============================  M    ==============================*/
static void Find_Directory (PARTN *ptr)
{
    #if ( DISPLAY == CHINESE )
        #define INPUT_EXIST_FILENAME  "пJ@ӦsbؿUɦWG "
    #else
        #define INPUT_EXIST_FILENAME  "Please input exist filename that in the directory: "
    #endif
    
    sbyte  str[13], *tmp;
    sbyte  fname[12];                                           /* jMܼ */
    
    memset(fname, ' ', 11);
    fname[11] = 0;
    
    ShowMSG(INPUT_EXIST_FILENAME);
    
    if ( GetStr(str, 12, ALLOW_ESC) == ESC_KEY || (strlen(str) == 0) )
        return;
               
    if ( str[0] == '.' )                          /** YJO '.'  '..' **/
    {
        fname[0] = '.';
        if ( str[1] == '.' )
            fname[1] = '.';
    }
    else                                                  /** YJ䥦ɦW **/
    {
        tmp = strchr(str, '.');
        if ( tmp != NULL )
        {
            *tmp++ = '\0';
            CopyMem(fname + 8, tmp, strlen(tmp));                   /** [W **/
        } /* end if */
        
        CopyMem(fname, str, strlen(str));                            /** ɦW **/
        strupr( fname );                                       /** jgr **/
    } /* end if */
    
    ShowMSG(NULL);
    Finder(ptr, Find_Directory_Condition, fname, ROOT);
} /* end Find_Directory */


/*============================ DOS u    ==============================*/
void Dos_Tools (PARTN *partnArr, sword ndx)
{
    #if ( DISPLAY == CHINESE )
        #define BOX_X                    35
        #define FORMAT_DONE_MSG          "    I"
        #define RECOVERY_DONE_MSG        "   _  I"
        #define PROMPT_FIX_BOOT_SECT     "T w n    _    ϡH"
        #define PROMPT_FAT_COPY          "z T w n i  FAT  sH"
        #define COPY_DONE_MSG            "I"
        
        static sbyte correctMsg[] =      "o  DOS     O  T I";
        static sbyte *menuStr[]   =
        {                                                        
            "1.      _   ", "2.   t        ",
            "3.         ", "4.     @  FAT ",
            "5.     G  FAT ", "6.         ",
            "7.     L  e  ", "8.  x s  L    ",
            "9.         ", "A.         ",
            "B.  j M      ", "C.  j M        ",
            "D.  j M   FAT  ", "E.    FAT1  FAT2 ",
            "F.    FAT2  FAT1 "
        };
    #else
        #define BOX_X                    34
        #define FORMAT_DONE_MSG          "Format done !"
        #define RECOVERY_DONE_MSG        "Try to recovery done !"
        #define PROMPT_FIX_BOOT_SECT     "Are you sure to recovery boot sector ?"
        #define PROMPT_FAT_COPY          "Are you sure to copy FAT to FAT ?"
        #define COPY_DONE_MSG            "Copy done !"
        
        static sbyte correctMsg[] =      "The DOS boot sector is correction !";
        static sbyte *menuStr[]   =
        {                                                        
            "1. Reference from Boot ", "2. Quick Format Program",
            "3. Dump Boot Record    ", "4. Dump First FAT      ",
            "5. Dump Second FAT     ", "6. Dump Root Directory ",
            "7. Dump Cluster        ", "8. Save Cluster link   ",
            "9. Recovery Boot Sector", "A. Check Boot Record   ",
            "B. Find Boot Sector    ", "C. Find Directory      ",
            "D. Find FAT Sector     ", "E. Copy FAT_1 to FAT_2 ",
            "F. Copy FAT_2 to FAT_1 "
        };
    #endif  /* end CHINESE */

    extern SeleMenu g_Menu;                                 /** ο檫 **/
    static sword sele = 1;
    sword  sourceFat, targetFat, stop, key, i; 
    sword  line[] = { 6, 13, 0 };
    
    ndx--;
    stop = FALSE;
    
    while ( stop == FALSE )
    {
        SaveScreen();
        
        MenuBox(BOX_X, 3, BOX_X + 27, 21, line);           /** øsΤ **/
        g_Menu.menuColor(PAR_MENU_COLOR, LIGHT_BAR_COLOR);
        for ( i = 0 ; i < sizeof(menuStr) / sizeof(sbyte *) ; i++ )
        {
            g_Menu.prompt(BOX_X + 2, 4 + i + (i > 1) + (i > 7), menuStr[i],
                #if SUPPORT_QKFORMAT
                    1);
                #else
                    (i != 1));
                #endif
        } /* end for */
    
        g_Menu.enableCtrl( M_WRAP | M_LOCK );
        
        i    = sele;
        sele = g_Menu.select(sele);
        
        ShowMSG(NULL);
        stop = (g_Menu.lastKey() == ESC_KEY);
    
        switch( sele )
        {
            case 1:                                            /** Ѧ٭ **/
                if ( Reference_from_boot(partnArr, ndx) )
                {
                    stop = TRUE;
                    g_Info.fd.modified = 1;
                } /* end if */
                break;
        
            case 2:                                          /** ֳt榡 **/
                #if SUPPORT_QKFORMAT
                    i = Quick_Format(partnArr, ndx);
                    if ( i < 0 )
                        ShowError( (i == -1) ? 6 : 23 );
                    else if ( i == 1 )
                        Cancel_Box();                      /** ܨ **/
                    else if ( i == 0 )
                        Prompt_Msg_Box(FORMAT_DONE_MSG, NULL);
                #endif
                break;
            
            case 3:                                          /** ܨtΰ **/
            case 4:
            case 5:
            case 6:
                i = ( sele == 3 ) ? BOOT  :
                    ( sele == 4 ) ? FAT_1 :
                    ( sele == 5 ) ? FAT_2 : ROOT;
                Show_System_Sector(i, partnArr + ndx);
                break;

            case 7:                                        /** ܺLe **/
                i = ShowCluster(partnArr + ndx);
                if ( i )
                    ShowError( (i == -1) ? 7 : 21 );
                break;
        
            case 8:
                SaveClusterLink(partnArr + ndx);
                break;
            
            case 9:                                        /** ϱҰʺϰ **/
                if ( !IsUpdatePartn() )             /** Y partition  **/
                {            
                    if ( !YN_box(IMMEDIATE_BOX, PROMPT_FIX_BOOT_SECT) )
                        Cancel_Box();                      /** ܨ **/
                    else if ( IsBootSectorError(partnArr, ndx) == 0 )
                        Prompt_Msg_Box(correctMsg, NULL);
                    else
                    {
                        i = Recovery_Bootrec(partnArr, ndx);   /** ״_ **/
                        if ( i == -1 )
                            ShowError( 23 );
                        else if ( i == -2 )
                            ShowError( 26 );             /** tΰϸƿ **/
                        else
                            Prompt_Msg_Box(RECOVERY_DONE_MSG, NULL);
                    } /* end if */
                } /* end if */
                
                break;
           
            case 10:                                       /** Ұʺϰ **/
                i = IsBootSectorError(partnArr, ndx);
                if ( i == 0 )
                    Prompt_Msg_Box(correctMsg, NULL);
                else
                    ShowError( (i == -2 || i == -3) ? 30 :
                                          (i == -4) ? 31 : 26 );
                break;
           
            case 11:                                      /**  jMҰʺϰ **/
                Finder(partnArr + ndx, Find_Boot_Secctor_Condition, NULL, BOOT);
                break;
           
            case 12:                                       /** jMڥؿ **/
                Find_Directory(partnArr + ndx);
                break;
           
            case 13:                                        /** jM FAT  **/
                Finder(partnArr + ndx, Find_FAT_Condition, NULL, HEX_DUMP);
                break;
           
            case 14:                                /**  FAT_1  FAT_2 **/
            case 15:                                /**  FAT_2  FAT_1 **/
                if ( !IsUpdatePartn() )             /** Y partition  **/
                {
                    if ( sele == 14 )
                    {
                        sourceFat = 1;
                        targetFat = 2;
                    }
                    else
                    {
                        sourceFat = 2;
                        targetFat = 1;
                    } /* end if */
             
                    if ( !YN_box(IMMEDIATE_BOX, PROMPT_FAT_COPY) )
                        Cancel_Box();                /** ܨܮ **/
                    else
                    {
                        Disable_Ctrl_Break();
                        key = CopyFAT(sourceFat, targetFat, partnArr + ndx);
                        Enable_Ctrl_Break();
                  
                        if ( key == 0 )
                            Prompt_Msg_Box(COPY_DONE_MSG, NULL);
                        else
                            ShowError( (key == -1) ? 23 : 21 );
                    } /* end if */
                } /* end if */
            
                break;
           
            default:
                sele = i;
                GiveupScreenBlock();     /**  SaveScreen() xse **/
                return;
        
        } /* end switch */
       
        RestoreScreen();                                   /** ^sùe **/
    
    } /* end while */
} /* end Dos_Tools */
