/* dynstr.c -- dynamic strings & arrays of strings

  AUTHOR: Gregory Pietsch <GKP1@flash.net>

  DESCRIPTION:

  This file is a dynamic string library for edlin, an edlin-style line editor.

  COPYRIGHT NOTICE AND DISCLAIMER:

  Copyright (C) 2003 Gregory Pietsch

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

*/

/* includes */

#include "config.h"
#include <stdio.h>
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STDLIB_H)
#include <stdlib.h>
#endif
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include "dynstr.h"
#include "xmalloc.h"
#include "error.h"

/* defines */
#define DS_MINSIZE 31

/* DYNSTRING_T functions */

#if !defined(__STDC__) && !defined(STDC_HEADERS)

#ifndef HAVE_MEMCHR
/* no memchr(), so roll our own */
#ifndef OPTIMIZED_FOR_SIZE
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
 /* Nonzero if either X or Y is not aligned on an "unsigned long" boundary.  */
#ifdef ALIGN
#define UNALIGNED(X) ((unsigned long)X&(sizeof(unsigned long)-1))
#else
#define UNALIGNED(X) 0
#endif
 /* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))
 /* Threshhold for punting to the bytewise iterator.  */
#define TOO_SMALL(len) ((len)<LITTLEBLOCKSIZE)
 /* Null character detection.  */
#if ULONG_MAX == 0xFFFFFFFFUL
#define DETECTNULL(X) (((X)-0x01010101UL)&~(X)&0x80808080UL)
#elif ULONG_MAX == 0xFFFFFFFFFFFFFFFFUL
#define DETECTNULL(X) (((X)-0x0101010101010101UL)&~(X)&0x8080808080808080UL)
#else
#error unsigned long is not 32 or 64 bits wide
#endif
#if UCHAR_MAX != 0xFF
#error unsigned char is not 8 bits wide
#endif
#endif                          /* OPTIMIZED_FOR_SIZE */
static void *memchr(const void *s, int c, size_t n)
{
    const unsigned char *us;
    unsigned char uc = (unsigned char) c;

#ifndef OPTIMIZED_FOR_SIZE
    unsigned long *psrc;
    size_t i;
    unsigned long mask = 0, buffer = 0;
#endif

    us = s;
#ifndef OPTIMIZED_FOR_SIZE
    /* If the size is small, or s is unaligned, punt into the bytewise loop.
       This should be rare.
     */
    if (!TOO_SMALL && !UNALIGNED(s)) {
        psrc = (unsigned long *) s;

        /* The fast code reads the data one word at a time and only performs
           the bytewise search on word-sized segments if they contain the
           search character, which is detected by XORing the word-sized
           segment with a word-sized block of the search character and then
           detecting the presence of a null character in the result.
         */
        for (i = 0; i < LITTLEBLOCKSIZE; i++)
            mask =
                (mask << CHAR_BIT) +
                ((unsigned char) uc & ~(~0 << CHAR_BIT));

        /* Check a block at a time if possible. */
        while (n >= LITTLEBLOCKSIZE) {
            buffer = *psrc ^ mask;
            if (DETECTNULL(buffer))
                break;          /* We found the character, so go byte by byte
                                   from here.  */
            n -= LITTLEBLOCKSIZE;
            psrc++;
        }

        /* Pick up any residual with a bytewise iterator.  */
        us = (unsigned char *) psrc;
    }
#endif
    /* The normal bytewise loop.  */
    while (n--) {
        if (*us == uc)
            return (void *) us;
        us++;
    }
    return 0;
}

#ifndef OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef TOO_SMALL
#undef DETECTNULL
#endif
#endif
#ifndef HAVE_MEMCMP
#define memcmp bcmp
#endif
#ifndef HAVE_MEMCPY
#define memcpy(x,y,z) bcopy(y,x,z)
#endif
#ifndef HAVE_MEMMOVE
#define memmove(x,y,z) bcopy(y,x,z)
#endif
#ifndef HAVE_MEMSET
/* no memset(), so roll our own */
#ifndef OPTIMIZED_FOR_SIZE
 /* Nonzero if X is not aligned on an "unsigned long" boundary.  */
#ifdef ALIGN
#define UNALIGNED(X) ((unsigned long)X&(sizeof(unsigned long)-1))
#else
#define UNALIGNED(X) 0
#endif
 /* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))
 /* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(len) ((len)<LITTLEBLOCKSIZE)
#endif                          /* OPTIMIZED_FOR_SIZE */
static void *memset(void *s, int c, size_t n)
{
    unsigned char *us = s;
    unsigned char uc = (unsigned char) c;

#ifndef OPTIMIZED_FOR_SIZE
    unsigned long *ps;
    unsigned long mask = 0;
    size_t i;

    /* If the size is small, or s is unaligned, punt into the
       byte copy loop.  This should be rare.  */
    if (!TOO_SMALL(n) && !UNALIGNED(s)) {
        ps = (unsigned long *) s;
        /* Store uc into mask at each location.  */
        for (i = 0; i < LITTLEBLOCKSIZE; i++)
            mask = (mask << CHAR_BIT) + (uc & ~(~0 << CHAR_BIT));
        /* Copy a 4X block at a time if possible. */
        while (n >= LITTLEBLOCKSIZE * 4) {
            *ps++ = mask;
            *ps++ = mask;
            *ps++ = mask;
            *ps++ = mask;
            n -= LITTLEBLOCKSIZE * 4;
        }
        /* Copy a block at a time if possible. */
        while (n >= LITTLEBLOCKSIZE) {
            *ps++ = mask;
            n -= LITTLEBLOCKSIZE;
        }
        /* Pick up any residual with a byte copier.  */
        us = (unsigned char *) ps;
    }
#endif
    /* The normal byte-copy loop.  */
    while (n--)
        *us++ = uc;
    return s;
}

#ifndef OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef TOO_SMALL
#endif
#endif
#ifndef HAVE_BCMP
/* no bcmp(), so roll our own */
#ifndef OPTIMIZED_FOR_SIZE
 /* Nonzero if either X or Y is not aligned on an "unsigned long" boundary.  */
#ifdef ALIGN
#define UNALIGNED(X,Y) \
  (((unsigned long)X&(sizeof(unsigned long)-1))\
  |((unsigned long)Y&(sizeof(unsigned long)-1)))
#else
#define UNALIGNED(X,Y) 0
#endif
 /* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))
 /* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(len) ((len)<LITTLEBLOCKSIZE)
#endif                          /* OPTIMIZED_FOR_SIZE */
static int bcmp(const void *s1, const void *s2, size_t n)
{
    const unsigned char *us1 = s1;
    const unsigned char *us2 = s2;

#ifndef OPTIMIZED_FOR_SIZE
    unsigned long *ps1, *ps2;

    /* If the size is small, or either s1 or s2 is unaligned, punt into the
       byte compare loop.  This should be rare.  */
    if (!TOO_SMALL(n) && !UNALIGNED(s1, s2)) {
        ps1 = (unsigned long *) s1;
        ps2 = (unsigned long *) s2;
        /* Load and compare blocks of memory one word at a time. */
        while (n >= LITTLEBLOCKSIZE) {
            if (*ps1 != *ps2)
                break;
            ps1++;
            ps2++;
            n -= LITTLEBLOCKSIZE;
        }
        /* Pick up any residual with a byte comparer.  */
        us1 = (unsigned char *) ps1;
        us2 = (unsigned char *) ps2;
    }
#endif
    /* The normal byte-compare loop.  */
    while (n--) {
        if (*us1 != *us2)
            return ((*us1 < *us2) ? -1 : +1);
        us1++;
        us2++;
    }
    return 0;
}

#ifndef OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef TOO_SMALL
#endif
#endif
#ifndef HAVE_BCOPY
/* no bcopy(), so roll our own */
#ifndef OPTIMIZED_FOR_SIZE
 /* Nonzero if either X or Y is not aligned on an "unsigned long" boundary.  */
#ifdef ALIGN
#define UNALIGNED(X,Y) \
  (((unsigned long)X&(sizeof(unsigned long)-1))\
  |((unsigned long)Y&(sizeof(unsigned long)-1)))
#else
#define UNALIGNED(X,Y) 0
#endif
 /* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))
 /* How many bytes are copied each interation of the 4X unrolled loop.  */
#define BIGBLOCKSIZE (sizeof(unsigned long)<<2)
 /* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(len) ((len)<BIGBLOCKSIZE)
#endif                          /* OPTIMIZED_FOR_SIZE */
static void *bcopy(const void *s2, void *s1, size_t n)
{
    unsigned char *us1 = s1;
    const unsigned char *us2 = s2;
#ifndef OPTIMIZED_FOR_SIZE
    unsigned long *pdst, *psrc;
#endif

    if (us2 < us1 && us1 < us2 + n) {
        /* Have to copy backwards.  */
        us1 += n;
        us2 += n;
        while (n--)
            *--us1 = *--us2;
        return s1;
    }
#ifndef OPTIMIZED_FOR_SIZE
    /* If the size is small, or either s1 or s2 is unaligned, punt into the
       byte copy loop.  This should be rare.  */
    if (!TOO_SMALL(n) && !UNALIGNED(s2, s1)) {
        pdst = (unsigned long *) s1;
        psrc = (unsigned long *) s2;
        /* Copy a big block at a time if possible. */
        while (n >= BIGBLOCKSIZE) {
            *pdst++ = *psrc++;
            *pdst++ = *psrc++;
            *pdst++ = *psrc++;
            *pdst++ = *psrc++;
            n -= BIGBLOCKSIZE;
        }
        /* Copy a little block at a time if possible. */
        while (n >= LITTLEBLOCKSIZE) {
            *pdst++ = *psrc++;
            n -= LITTLEBLOCKSIZE;
        }
        /* Pick up any residual with a byte copier.  */
        us1 = (unsigned char *) pdst;
        us2 = (unsigned char *) psrc;
    }
#endif
    /* The normal byte-copy loop.  */
    while (n--)
        *us1++ = *us2++;
    return s1;
}

#ifndef OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef BIGBLOCKSIZE
#undef TOO_SMALL
#endif
#endif
#endif

/* DStidy - destroy any allocated string storage */
static void DStidy(DYNSTRING_T * this, int constructed)
{
    if (constructed && this->ptr)
        free(this->ptr);
    this->ptr = 0;
    this->len = this->res = 0;
}

/* DSgrow - make the DS grow */
static int DSgrow(DYNSTRING_T * this, size_t n, int trim)
{
    size_t osize = this->ptr == 0 ? 0 : this->res;
    size_t size;
    char *s;

    if (n == 0) {
        if (trim && DS_MINSIZE < osize)
            DStidy(this, 1);
        else if (this->ptr)
            this->ptr[this->len = 0] = '\0';
        return 0;
    } else if (n == osize || (n < osize && !trim))
        return 1;
    else {
        size = this->ptr == 0 && n < this->res ? this->res : n;
        if ((size |= DS_MINSIZE) == NPOS)
            --size;
        if ((s = realloc(this->ptr, size + 1)) == 0
            && (s = realloc(this->ptr, (size = n) + 1)) == 0) {
            error(ERR_OUT_OF_MEMORY);
            abort();
        }
        this->ptr = s;
        this->res = size;
        return 1;
    }
}

/* DScreate - create a new DYNSTRING_T */
DYNSTRING_T *DScreate(void)
{
    DYNSTRING_T *ds = XMALLOC(DYNSTRING_T, 1);

    DStidy(ds, 0);
    return ds;
}

/* DSdestroy - return a DYNSTRING_T to the void */
void DSdestroy(DYNSTRING_T * ds)
{
    if (ds) {
        DStidy(ds, 1);
        XFREE(ds);
    }
}

/* append char c to the end of ds nr times */
DYNSTRING_T *DSappendchar(DYNSTRING_T * ds, int c, size_t nr)
{
    size_t n;

    if (NPOS - ds->len <= nr)
        error(ERR_LENGTH);
    if (0 < nr && DSgrow(ds, n = ds->len + nr, 0)) {
        /* append to make nonempty string */
        memset(ds->ptr + ds->len, c, nr);
        ds->ptr[ds->len = n] = '\0';
    }
    return ds;
}

/* append substring to the end of this */
DYNSTRING_T *DSappendcstr(DYNSTRING_T * this, const char *s, size_t pos,
                          size_t ns)
{
    size_t n = strlen(s);

    if (n < pos)
        error(ERR_STRING_POSITION);     /* invalid string position error */
    n -= pos;
    if (n < ns)
        ns = n;
    if (NPOS - this->len <= ns)
        error(ERR_LENGTH);
    if (0 < ns && DSgrow(this, n = this->len + ns, 0)) {
        /* append to make non-empty string */
        memcpy(this->ptr + this->len, s + pos, ns);
        this->ptr[this->len = n] = '\0';
    }
    return this;
}

/* assign char c to the end of ds nr times */
DYNSTRING_T *DSassignchar(DYNSTRING_T * ds, int c, size_t nr)
{
    if (NPOS <= nr)
        error(ERR_LENGTH);
    if (0 < nr && DSgrow(ds, nr, 0)) {
        /* make nonempty string */
        memset(ds->ptr, c, nr);
        ds->ptr[ds->len = nr] = '\0';
    }
    return ds;
}

/* assign substring to the end of this */
DYNSTRING_T *DSassigncstr(DYNSTRING_T * this, const char *s, size_t pos,
                          size_t ns)
{
    size_t n = strlen(s);

    if (n < pos)
        error(ERR_STRING_POSITION);     /* invalid string position error */
    n -= pos;
    if (ns < n)
        n = ns;
    if (this->ptr && (this->ptr == s || strcmp(this->ptr, s) == 0)) {
        DSremove(this, pos + n, NPOS);
        DSremove(this, 0, pos);
    } else if (DSgrow(this, n, 1)) {
        /* assign to make non-empty string */
        memcpy(this->ptr, s + pos, n);
        this->ptr[this->len = n] = '\0';
    }
    return this;
}

/* remove a substring */
DYNSTRING_T *DSremove(DYNSTRING_T * ds, size_t p, size_t nr)
{
    size_t n;

    if (ds->len < p)
        error(ERR_STRING_POSITION);     /* invalid string position error */
    if (ds->len - p < nr)
        nr = ds->len - p;
    if (0 < nr) {
        /* remove the substring */
        memmove(ds->ptr + p, ds->ptr + p + nr, ds->len - p - nr);
        n = ds->len - nr;
        if (DSgrow(ds, n, 0))
            ds->ptr[ds->len = n] = '\0';
    }
    return ds;
}

/* resize a string */
void DSresize(DYNSTRING_T * ds, size_t n, int c)
{
    n <= ds->len ? DSremove(ds, n, NPOS) : DSappendchar(ds, c,
                                                        n - ds->len);
}

/* find a substring */
size_t DSfind(DYNSTRING_T * ds, const char *s, size_t p, size_t n)
{
    size_t nmax;
    const char *t, *u;

    if (n == 0 || (n == NPOS && (n = strlen(s)) == 0))
        return 0;
    if (p < ds->len && n <= (nmax = ds->len - p)) {
        /* find non-null substring in string */
        for (nmax -= n - 1, u = ds->ptr + p;
             (t = (const char *) memchr(u, *s, nmax)) != 0;
             nmax -= t - u + 1, u = t + 1)
            if (memcmp(t, s, n) == 0)
                return (size_t) (t - ds->ptr);
    }
    return NPOS;
}

/* replace a substring with a string */
DYNSTRING_T *DSreplace(DYNSTRING_T * ds, size_t p, size_t n, const char *s,
                       size_t ns)
{
    size_t nm, nn;

    if (ds->len < p)
        return ds;
    if (ns == NPOS)
        ns = strlen(s);
    if (NPOS - ns <= ds->len - n)
        error(ERR_STRING_POSITION);
    nm = ds->len - n - p;
    if (ns < n)
        memmove(ds->ptr + p + ns, ds->ptr + p + n, nm);
    if ((0 < ns || 0 < n) && DSgrow(ds, nn = ds->len + ns - n, 0)) {
        /* replace to make nonempty string */
        if (n < ns)
            memmove(ds->ptr + p + ns, ds->ptr + p + n, nm);
        memcpy(ds->ptr + p, s, ns);
        ds->ptr[ds->len = nn] = '\0';
    }
    return ds;
}

/* get a string's char pointer */
char *DScstr(DYNSTRING_T * ds)
{
    return ds->ptr ? ds->ptr : "";
}

/* get a string's length */
size_t DSlength(DYNSTRING_T * ds)
{
    return ds->ptr ? ds->len : 0;
}

/* DYNARRAYSTRING_T functions */

/* DAStidy - destroy any allocated array storage.
   Note that it DOES NOT destroy the strings in the array.
 */
static void DAStidy(DYNARRAYSTRING_T * this, int constructed)
{
    if (constructed && this->ptr)
        free(this->ptr);
    this->ptr = 0;
    this->len = this->res = 0;
}

/* DASgrow - make the DAS grow */
static void
DASgrow(DYNARRAYSTRING_T * this, size_t n, DYNSTRING_T * s, int trim)
{
    size_t osize = this->ptr == 0 ? 0 : this->res;
    size_t size, i, m;
    DYNSTRING_T **np;

    if (n == 0) {
        if (trim)
            DAStidy(this, 1);
    } else if (n == osize || (n < osize && !trim));
    else if (n == NPOS)
        error(ERR_LENGTH);
    else {
        size = this->ptr == 0 && n < this->res ? this->res : n;
        np = XMALLOC(DYNSTRING_T *, size);
        m = n < this->len ? n : this->len;
        for (i = 0; i < m; i++)
            np[i] = this->ptr[i];
        if (s)
            for (; i < this->res; i++)
                DSassigncstr(np[i], DScstr(s), 0, NPOS);
        DAStidy(this, 1);
        this->ptr = np;
        this->res = size;
    }
    this->len = n;
}

/* DAScreate - create a new DYNARRAYSTRING_T */
DYNARRAYSTRING_T *DAScreate(void)
{
    DYNARRAYSTRING_T *ds = XMALLOC(DYNARRAYSTRING_T, 1);

    DAStidy(ds, 0);
    return ds;
}

/* DASdestroy - return a DYNARRAYSTRING_T to the void */
void DASdestroy(DYNARRAYSTRING_T * ds)
{
    if (ds) {
        DAStidy(ds, 1);
        XFREE(ds);
    }
}

/* DASappend - append an array of DYNSTRING_Ts to a DYNARRAYSTRING_T */
DYNARRAYSTRING_T *DASappend(DYNARRAYSTRING_T * this, DYNSTRING_T * s,
                            size_t n, size_t d)
{
    size_t i;

    if (NPOS - this->len <= n)
        error(ERR_ARRAY_TOO_BIG);
    i = this->len;
    for (DASgrow(this, n += i, 0, 1); i < n; ++i, s += d)
        DSassigncstr(this->ptr[i], DScstr(s), 0, NPOS);
    return this;
}

/* DASassign - assign an array of DYNSTRING_Ts to a DYNARRAYSTRING_T */
DYNARRAYSTRING_T *DASassign(DYNARRAYSTRING_T * this, DYNSTRING_T * s,
                            size_t n, size_t d)
{
    size_t i;

    DASgrow(this, n, 0, 1);
    for (i = 0; i < n; ++i, s += d)
        DSassigncstr(this->ptr[i], DScstr(s), 0, NPOS);
    return this;
}

/* insert copies of strings into a DAS */
DYNARRAYSTRING_T *DASinsert(DYNARRAYSTRING_T * ds, size_t p,
                            DYNSTRING_T ** s, size_t n, size_t d)
{
    size_t i;

    if (ds->len < p)
        error(ERR_STRING_POSITION);
    if (NPOS - ds->len <= n)
        error(ERR_ARRAY_TOO_BIG);
    if (0 < n) {
        i = ds->len - p;
        for (DASgrow(ds, n + ds->len, 0, 0); 0 < i;) {
            --i;
            ds->ptr[p + n + i] = ds->ptr[p + i];
        }
        for (i = 0; i < n; i++, s += d) {
            ds->ptr[p + i] = DScreate();
            DSassigncstr(ds->ptr[p + i], DScstr(*s), 0, NPOS);
        }
    }
    return ds;
}

/* remove a bunch of lines */
DYNARRAYSTRING_T *DASremove(DYNARRAYSTRING_T * ds, size_t p, size_t n)
{
    size_t m, i;

    if (ds->len < p)
        error(ERR_INVALID_POSITION);
    if (ds->len - p < n)
        n = ds->len - p;
    if (0 < n) {
        m = ds->len - p - n;
        for (i = 0; i < m; i++)
            DSassigncstr(ds->ptr[p + i], DScstr(ds->ptr[p + i + n]), 0,
                         NPOS);
        DASgrow(ds, ds->len - n, 0, 0);
    }
    return ds;
}

/* return in x a subarray of this. x can be the same pointer as this,
   in which case this is altered. */
DYNARRAYSTRING_T *DASsubarray(DYNARRAYSTRING_T * this,
                              DYNARRAYSTRING_T * x, size_t p, size_t n)
{
    if (this->len < p)
        error(ERR_INVALID_POSITION);
    if (this->len - p < n)
        n = this->len - p;
    if (this == x) {
        DASremove(this, p + n, NPOS);
        DASremove(this, 0, p);
        return this;
    } else {
        DASassign(x, this->ptr[p], n, 1);
        return x;
    }
}

/* get the string at position i */
DYNSTRING_T *DASgetat(DYNARRAYSTRING_T * ds, size_t i)
{
    if (ds->len <= i)
        error(ERR_INVALID_POSITION);
    return ds->ptr[i];
}

/* put a string at position i */
void DASputat(DYNARRAYSTRING_T * ds, size_t i, DYNSTRING_T * x)
{
    if (ds->len <= i)
        error(ERR_INVALID_POSITION);
    DSassigncstr(ds->ptr[i], DScstr(x), 0, NPOS);
}

/* get base */
DYNSTRING_T **DASbase(DYNARRAYSTRING_T * this)
{
    return this->len ? this->ptr : 0;
}

/* get however many lines are in the DAS */
size_t DASlength(DYNARRAYSTRING_T * ds)
{
    return ds->len;
}

/* END OF FILE */
