/***
*fileio.c - file I/O
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   File read/write functions.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   1.5   02-Jul-1998    fixed signed bug in putfile() loop; fixes for
                        clusters array
   1.4   10-Jan-1998    disked_jmp; moved append() into DISKLIB.C
   1.3   29-Nov-1997    _FILEN; "%02x" and displaymask(); enhanced
                        put_sectors(); append/overwrite
   1.2   21-Feb-1997    better message handling (getfver()); added
                        dumpfile()
   1.1   13-Sep-1996    code from DISKED.C moved in here as functions
   1.0   10-Nov-1994    added calls to new error functions (see ERROR.C)
   0.1   07-Sep-1994    created

   Release Notes:

   This file handles the reading and writing of files. The sector
   and or file buffers can be written to a file; a file can be
   inserted into the file buffer; sectors can be written to a file.

   The functions here called by DISKED.C do all the pre-processing
   and then call the actual I/O functions.

   Programming Notes:

   I think I finally found a good use for setjmp() and longjump().
   All examples I have found is based on a simple floating point
   error handler.

   DISKED.C calls setjmp() at the start of it's command loop.
   For each of the file I/O functions here, there is a single
   error handler which calls longjmp(). The longjmp() saves
   quite a bit in brace-levels, interium processing flags, etc.
   However one must be careful to clean up properly.

   longjmp() can even be used for all returns. Why not? This way
   all of the file closings, etc., can be done in one place. hmmm...

   Error handling is still not very robust though. When copying
   sectors, sector read errors are ignored except when the drive
   door is open. All functions can be aborted by pressing a key.
   And I really should do free psace checking.

   O yeah. The interface I chose has turned out to be extremely
   cumbersome and rears it's ugly face to the fullest here...

   Doing things by sectors is slooooowww.....

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>         /* uses both stream and handle */
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <errno.h>
#include <setjmp.h>
#include <conio.h>         /* for kbhit() */

#include "disked.h"        /* append() */
#include "diskio.h"        /* drive parameters */
#include "console.h"       /* print(), output(), */
#include "error.h"         /* error handling */
#include "mylib.h"         /* input handling */
#include "files.h"         /* clusters */
#include "alloc.h"         /* alloc()/freep() */
#include "dosio.h"         /* doserror() */

extern jmp_buf disked_jmp;        /* address for long jump to jump to */

/* NO globals defined here */

/* internal data */

#define o_flag (_O_CREAT|_O_TRUNC|_O_BINARY|_O_RDWR)
#define p_mode (_S_IWRITE|_S_IREAD)

static int savedsec;
static FILE *fh;
static int fd;
static char *function;
static FILEMODE filemode;
static unsigned char *buffer;

/* internal functions */

static void putfile(char *file, int xlate, int mask);
static int getfile(char *file, int xlate, int mask);
static void dumpfile(char *file, unsigned char *buffer, unsigned bytecnt);
static void copyfile(const char *file, unsigned filenumber);
static void putsectors(char *, long, long);
static void fioerror(void);

struct Msg ps[]={          /* for inputing a two argument commnd */
   {" from "},
   {" number "},           /* 'len' and 'spec' set at runtime by getl() */
};

/***
*put_sectors   -  asks for what and where
*
****/

extern void put_sectors(void)
{
long from, number;
char tmpstr[MAXSTR+1];

   if (Display)
      print(" put");

   from = log_sector;                  /* default is current sector */
   number = 0;

   if (getl(ps,&from,&number) == 0)
      return;

   if (from < 0 || from > num_sectors)  /* must start within disk */
      return;

   if (number < 0) {                    /* make negative write to end of disk */
      number = (num_sectors-1) - from;
      if (Display) print(" (%lr)",number);
   }

   if (number == 0 || from+number >= num_sectors)  /* too many, not enuf */
      return;

   print((Display) ? " file: " : ":");

   if (getstr(tmpstr,MAXSTR,_FILEN) <= 0)
      return;

   if (Verify && !getver(" put file",MOV_YN))
      return;

   if (exist(tmpstr))
   {
      savecursor();
      if (!Verify)
         output(' ');
      if (!getver("exists, overwrite",MOV_YN))
         return;
      restcursor();
      clreol();
   }

   if (Display && !Verify)
      output(' ');
   if (Display && Verify)
      put(3,' '),put(3,8);

   putsectors(tmpstr,from,number);
   disk_moved = 0;
}

/***
*put_file - save file/sector buffer to disk
*
*           mode == 0  save file buffer in binary w/ translation
*           mode == 1  save file buffer in binary w/o translation
*           mode == 2  save file buffer as hex dump w/ ascii filter
*           mode == 3  save sector buffer as hex dump w/ ascii filter
****/

extern void put_file(PUTMODE mode)
{
char tmpstr[MAXSTR+1];
char s[40] = "put file ";                 /* (s is appended to) */
int e = 0;

   if (mode != SECBUF_H && byte_cnt == 0)
      return;

   if (mode == FILEBUF_H)
      print("(dump buffer)");
   else if (mode == SECBUF_H)
      print("(dump sector)");

   if (Display)
      print(" file: ");

   if (getstr(tmpstr,MAXSTR,_FILEN) <= 0)
      return;

   if (Verify)
   {
      output(' ');
      if (Display)
         if (mode == FILEBUF_X)
            strcat(s,"w/options");
         else if (mode == FILEBUF)
            strcat(s,"w/o options");
      if (!getver(s,CLR_ARG))
         return;
   }
   filemode = FILE_OVERWRITE;
   if ((e = exist(tmpstr)))
   {
      if (!Verify)
         output(' ');
      filemode = getfver("exists, overwrite/append",CLR_ARG);
      if (filemode == 0)
         return;
      clreol();
   }
   if (error.num != -1)
      fioerror();

   if (Display)
   {
      savecursor();              /* save and restore in case of error */
      if (!Verify && !e)
         output(' ');
      print( filemode == FILE_OVERWRITE ? "putting" : "appending");
      restcursor();
   }
   if (mode == FILEBUF_H)
      dumpfile(tmpstr,data_buf,byte_cnt);
   else if (mode == SECBUF_H)
      dumpfile(tmpstr,sec_buf,sec_size);
   else
      putfile(tmpstr,mode,Bufmask);
}

/***
*get_file   -  read a file into the file buffer
*
****/

extern void get_file(int xlate)
{
int i;
char tmpstr[MAXSTR+1];
char s[40] = "insert file ";              /* (s is appended to) */

   if (byte_cnt == max_bytes)
      return;

   if (Display)
      print(" file: ");

   if (getstr(tmpstr,MAXSTR,_FILEN) <= 0)
      return;

   if (!exist(tmpstr))
   {
      print(" not found");
      return;
   }

   if (Verify)
   {
      output(' ');
      if (Display)
         if (xlate)
            strcat(s,"w/options");
         else
            strcat(s,"w/o options");
      if (!getver(s,CLR_ARG))
         return;
   }

   if (Display)
   {
      savecursor();
      if (!Verify)
         output(' ');
      print("inserting");
      restcursor();
   }
   i = getfile(tmpstr,xlate,Bufmask);
   if (!Verify)
      output(' ');
   if (i == -2)
      print("buffer full ");
}

/***
*insert_file   -  insert a file into the file buffer
*
****/

extern void insert_file(int xlate)
{
int i;
int tempd;
char tmpstr[FILENAME_MAX];

   if (!Files)
      return;

   if (byte_cnt == max_bytes)    /* full? */
      return;

   tempd = sectortocluster(log_sector);
   i = clusters[tempd];
   if (i == 0 || (unsigned)i > n_files)
      return;

   if (files[i].dir)
   {
      print(" can not insert directories (yet)");
      return;
   }

   tmpstr[0] = (char)(cur_disk + '@');
   tmpstr[1] = ':';
   sprintf(tmpstr+2,gfile(i));

   if (Display)
      print(" insert file %s",tmpstr+2);

   if (Verify && !getver("",MOV_YN))
      return;

   if (Display)
   {
      savecursor();
      print(" inserting");
      restcursor();
   }

   i = getfile(tmpstr,xlate,Bufmask);
   if (!Verify)
      output(' ');
   if (i == -2)
      print("buffer full ");
}

/***
*copy_file  -  copy a file, sector by sector
*
****/

extern void copy_file(void)
{
unsigned int tempu;
char tmpstr[MAXSTR+1];

   if (!Files)
      return;

   tempu = sectortocluster(log_sector);
   if (clusters[tempu] == 0 || (unsigned)clusters[tempu] > n_files)
      return;

   if (files[clusters[tempu]].dir)
   {
      print(" can not copy directories");    /* actually we can... */
      return;                                /* but I've been busy */
   }

   sprintf(tmpstr,gfile(clusters[tempu]));

   print(" copy file %s to: ",tmpstr);
   if (getstr(tmpstr,MAXSTR,_FILEN) < 1)
      return;

   if (Display)
   {
      savecursor();
      print(" copying");
      restcursor();
   }
   if (exist(tmpstr))
   {
      savecursor();
      if (!Verify)
         output(' ');
      if (!getver("exists, overwrite",MOV_YN))
         return;
      restcursor();
      clreol();
   }
   copyfile(tmpstr,clusters[tempu]);
}

/* here are the real functions */

/***
*putsectors -  put sectors to a file
*
****/

static void putsectors(char *file, long start, long num)
{
int io;
long i;

   function = "putsectors";

   errno = 0;
   savedsec = 1;
   savesector();

   if ((fd = _open(file,o_flag,p_mode)) == ERROR)
      fioerror();

   if (start == 0)
      log_sector = num_sectors-1;
   else
      log_sector = start-1;

   for (i = 0; i < num; i++)
   {
      if (_kbhit())           /* abort if like errors or something */
         break;

      if (Display)
         put(print("%ld",i),'\b');

      if ((io = nextsector()) != DISK_OK)
      {
         output('\n');
         printerror(Debug);
         if (io == DOS_ENREADY || io == BIOS_ETIMEOUT || io == BIOS_ECHANGE)
            break;
      }

      if (_write(fd,sec_buf,sec_size) != (int)sec_size)
         fioerror();
   }
   _close(fd);
   fd = -1;
   restoresector();
   clreol();
}

/***
*putfile -  write data buffer to a file
*
****/

static void putfile(char *filename, int xlate, int mask)
{
unsigned int i;
unsigned int c;

   function = "putfile";
   errno = 0;

   if ((fh = fopen(filename, filemode == FILE_OVERWRITE ? "wb" : "ab")) == NULL)
      fioerror();

   for (i = 0; i < byte_cnt; i++)
   {
      if (_kbhit())
         break;

      c = data_buf[i];
      if (xlate && !isprint(c) && !isspace(c))
      {
         if (mask == B_STRIP)                /* do we strip? */
            continue;
         if (mask == B_MASK)                 /* do we mask ? */
         {
            c &= 0x7F;
            /* fall through */
         }
         else if (mask == B_CONV)            /* do we convert? */
         {
            if (fprintf(fh,"<%02x>",c) != 4)
               fioerror();
            continue;
         }
      }
      if (putc(c,fh) != (int)c)
         fioerror();
   }
   fclose(fh);
   fh = NULL;
}

/***
*dumpfile   -  hex/ascii dump of buffer to a file
*
****/

static void dumpfile(char *file, unsigned char *buffer, unsigned bytecnt)
{
unsigned int n;
int b;

   function = "dumpfile";
   errno = 0;

   if ((fh = fopen(file, filemode == FILE_OVERWRITE ? "wb" : "ab")) == NULL)
      fioerror();

   for (n = 0; n < bytecnt; n+=16)
   {
      fprintf(fh,"%04x: ",n);
      for (b = 0; b < 16; b++)
         fprintf(fh,"%02x ",buffer[n+b]);
      for (b = 0; b < 16; b++)
         fputc(displaymask(buffer[n+b]),fh);
      fprintf(fh,"\r\n");
   }
   fclose(fh);
   fh = NULL;
}

/***
*getfile -  read a file into the data buffer
*
*  returns -2 if buffer full
****/

static int getfile(char *filename, int xlate, int mask)
{
int i;
unsigned char buffer[512]; /* file reads of 512 bytes, perhaps a bit slow? */

   function = "getfile";
   errno = 0;

   if ((fd = _open(filename,_O_RDONLY|_O_BINARY)) == -1)
      fioerror();

   while ((i = _read(fd,buffer,512)) > 0)
   {
      if (_kbhit())
         break;
      if (xlate)
         append(mask,buffer,i);
      else
         append(0,buffer,i);
      if (byte_cnt == max_bytes)
         return -2;
   }
   if (i == -1)
      fioerror();
   _close(fd);
   fd = -1;
   return 1;
}

/***
*copyfile   -  copy a file sector by sector
*
****/

static void copyfile(const char *file, unsigned filenumber)
{
int io;
unsigned int i,j;
unsigned long sector;

   if ((buffer = alloc(sec_size,sizeof(char))) == NULL)
      return;

   errno = 0;
   if ((fd = _open(file,o_flag,p_mode)) == ERROR)
      fioerror();

   for (i = 2; i < num_clusters; i++)
   {
      if (clusters[i] == filenumber)
      {
         sector = clustertosector(i);
         for (j = 0; j < secs_cluster; j++, sector++)
         {
            if (_kbhit())
               goto exit;
            if ((io = diskio(DISK_READ,sector,buffer)) != DISK_OK)
            {
               output('\n');
               printerror(Debug);
               if (io == DOS_ENREADY || io == BIOS_ETIMEOUT || io == BIOS_ECHANGE)
                  goto exit;
            }
            if (_write(fd,buffer,sec_size) != (int)sec_size)
               fioerror();
         }
      }
   }
exit:
   _close(fd);
   fd = -1;
   freep(buffer);
   buffer = NULL;
}

/***
*fio_error  -  error handler
*
****/

static void fioerror(void)
{
const char *msg;

   if (savedsec)
   {
      restoresector();
      savedsec = 0;
   }

   if (buffer != NULL)
   {
      freep(buffer);
      buffer = NULL;
   }

   if (fh)                             /* close file stream if open */
      fclose(fh);
   if (fd > 0)                         /* close file handle if open */
      _close(fd);

   if (!errno)                         /* if errno NOT set, set it now */
      errno = exterror();

   if (error.num == -1)                /* may be set by INT 24 */
   {
      msg = doserror(errno);
      set_err_arg("%02Xh",errno);
      set_error(msg,"fileio",errno,function);
   }
   else
      set_err_info("fileio");

   longjmp(disked_jmp,-1);             /* return to main() */
}
