/***
*CONSOLE.C  -  Generic Console Library Functions
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Keyboard and Screen I/O
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
******************************************************************************/

/*
   Versions:

   1.8   02-Jul-1998    __GNUC__
   1.7   10-Jan-1998    __STDC__ macro bug fix; renamed some static functions
   1.6   07-Mar-1997    added xscankey() & xreadchar()
   1.5   21-Feb-1997    readchar() now returns extended keys directly
                        rather than the (stupid) DOS way of 00 and then
                        buffering the high 8 bits
   1.4   12-Sep-1996    added "%r" handler; replicate() is now put();
                        BUG FIX: non-Watcom && __STDC__ character
                        output functions were WRONG!
   1.3   13-Nov-1994    removed some unused functions, made many
                        static
   1.2   15-Sep-1994    oops, getcursor/type() had wrong INT no.
   1.1   01-Sep-1994    major cleanup; consolodation of all string
                        print functions.

   Release Notes:

   There are three global console I/O routines that are function
   pointers:

      input          charcater input
      output         character output
      print          formated string output

   There are three groups of static console I/O routines to which
   the global pointers are assigned at runtime:

      biosread/bioswrite      BIOS
      bdosread/bdoswrite      BDOS
      dosread/doswrite        DOS (STDIN/STDOUT)

   input and output are assigned to the above repsectively depending
   on the INI file i/o setting.  print uses output.

   Programming Notes:

   The basis of using pointers was testing the various kinds of
   outputs, speed testing for example.  It will also help in
   futher developments such as command macros and input/output
   redirection.

   This module should be broken up; i.e. the BIOS stuff should be
   seperate...

*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include <dos.h>              /* _bdos() */
#include <conio.h>

#include "general.h"
#include "disked.h"           /* for BDOS/BIOS */
#include "init.h"
#include "console.h"

/* NO globals referenced here */

/* external data defined here */

int (*input)(void);             /* pointer to keyboard input function */
int (*output)(int);             /* pointer to console output function */
int (*print)(const char *,...); /* formated printing */

/* static data */

static int cur_pos;

/* static functions */

static void reformat(char *s);
static int bioswrite(int);
static int biosread(void);
static int biosreadx(void);
static int bdoswrite(int);
static int bdosread(void);
static int doswrite(int);
static int dosread(void);
static int xprint(const char *fmt,...);
#ifndef __WATCOMC__
static void outchar(int c);
static int getcursor(void);
static void setcursor(int c);
#endif

#ifdef __WATCOMC__

void outchar(int c);                /* WCL doesn't like the `static' */
int getcursor(void);
void setcursor(int c);

#pragma aux outchar =   \
   "mov ah,0x0E"        \
   "xor bh,bh"          \
   "int 0x10"           \
   parm [ax]            \
   modify [bx];

#pragma aux setcursor = \
   "mov ah,2"           \
   "xor bh,bh"          \
   "int 0x10"           \
   parm [dx]            \
   modify [ax bx];

#pragma aux getcursor = \
   "mov ax,0300h"       \
   "xor bx,bx"          \
   "int 10h"            \
   "mov ax,dx"          \
   modify [ax bx cx dx];

#endif


extern void initvideo(int iomode)
{
   if (iomode == O_BIOSX)
   {
      input = biosreadx;
      output = bioswrite;
   }
   else if (iomode == O_BIOS)
   {
      input = biosread;
      output = bioswrite;
   }
   else if (iomode == O_BDOS)
   {
      input = bdosread;
      output = bdoswrite;
   }
   else
   {
      input = dosread;
      output = doswrite;
   }
   print = xprint;
}


static int xprint(const char *fmt,...)
{
int i;
va_list args;
char buf[BUFSIZ],b[BUFSIZ],*p;

   strcpy(b,fmt);
   reformat(b);
   va_start(args,fmt);
   i = vsprintf(buf,b,args);
   va_end(args);
   p = buf;
   while (*p)
      output(*p++);
   return i;
}

/***
*reformat   -  handles my printf() extension "%r"
*           -  %r = display in decimal or hex dpending on Radix
*           -  handles all formats, e.g. %r, % r, %*r, %lr etc.
*           -  %r is signed, %R unsigned
****/

static void reformat(char *s)
{
   while (*s)
   {
      if (*s == '%')
      {
         s++;
         if (*s != '%')
         {
            while (isspace(*s))  s++;
            while (isdigit(*s))  s++;

            if (*s == '*')       s++;
            if (*s == 'l')       s++;

            if (*s == 'r')
               *s = (char)((Radix == 10) ? 'd' : 'x');
            if (*s == 'R')
               *s = (char)((Radix == 10) ? 'u' : 'x');
         }
      }
      s++;
   }
}

static int bioswrite(int c)
{
#define tabs 8

   if (c == '\n')
      outchar('\r');

   if (c == '\t')
   {
      int t = tabs - ((getcursor() & 0xFF) % tabs);
      while (t--)
         outchar(' ');
   }
   else
      outchar(c);

   return c;
}

static int bdoswrite(int c)
{
   _bdos(2,c,0);
   if (c == '\n')
      _bdos(2,'\r',0);
   return c;
}

static int doswrite(int c)
{
   return (putchar)(c);
}

static int biosread(void)
{
int key;

   key = scankey();
   if (! ((key & 0xff) == 0 || key > 0x3920 || key < 0x011B))
      key &= 0xff;
   return key;
}

static int biosreadx(void)
{
int key;

   key = xscankey();
   if (! ((key & 0xff) == 0 || key > 0x3920 || key < 0x011B))
      key &= 0xff;
   if ((key & 0xff) == 0xe0)
      key &= 0xff00;
   return key;
}

static int bdosread(void)
{
   return _bdos(7,0,0) & 0xFF;          /* AH = 7, AL = char */
}

static int dosread(void)
{
   return _bdos(8,0,0) & 0xFF;          /* AH = 8, AL = char */
}

/* String output functions */

/***
*printc  -  display centered string
*
****/

extern void printc(const char *s)
{
int i;
char buf[80];

   i = (80 - strlen(s)) / 2;
   if (i < 79 && i > 0)
      memset(buf,' ',i);
   else
      i = 0;
   buf[i] = '\0';
   print("%s%s\n",buf,s);
}

/***
*bprint  -   display string with no cursor move
*
****/

extern void bprint(const char *s)
{
   savecursor();
   print("%s",s);
   restcursor();
}

/***
*disptext   -  displays an array of strings, each with a '\n'
*
****/

extern void disptext(char **text)
{
   while (*text)
      print("%s\n",*text++);
}

extern void put(int i, int c)
{
char buf[80];

   if (i > 79) i = 79;
   memset(buf,c,i);
   buf[i] = '\0';
   print(buf);
}


/* Cursor Positioning */

extern void get_cursor(unsigned int *row, unsigned int *col)
{
int t = getcursor();

   *row = t >> 8;
   *col = t & 0xFF;
}

extern void set_cursor(unsigned int row, unsigned int col)
{
   setcursor((row << 8) + col);
}

extern void savecursor(void)
{
   cur_pos = getcursor();
}

extern void restcursor(void)
{
   setcursor(cur_pos);
}

extern void curright(void)
{
   setcursor(getcursor()+1);
}

extern void curleft(void)
{
   setcursor(getcursor()-1);
}

#ifndef __WATCOMC__

#ifndef __STDC__

#if defined(_MSC_VER) && (_MSC_VER > 600)
#pragma optimize("gle",off)
#pragma warning(disable:4035)
#endif

#ifdef __BORLANDC__
#pragma warn -rvl
#endif

extern int scankey(void)
{
   __asm  xor   ax,ax
   __asm  int   0x16
}

extern int xscankey(void)
{
   __asm  mov   ah,0x10
   __asm  int   0x16
}

static void setcursor(int c)
{
   __asm mov ah,2
   __asm xor bh,bh
   __asm mov dx,c
   __asm int 0x10
}

static int getcursor(void)
{
   __asm mov ah,3
   __asm xor bh,bh
   __asm int 0x10
   __asm mov ax,dx
}

extern void clreol(void)
{
   __asm mov ax,0x0A20
   __asm xor bh,bh
   __asm mov cx,79
   __asm int 0x10
}

static void outchar(int c)
{
   __asm mov ax,0x0E00
   __asm add ax,c
   __asm xor bh,bh
   __asm int 0x10
}

extern void charout(int c)
{
   __asm mov ax,0x0A00
   __asm add ax,c
   __asm xor bh,bh
   __asm mov cx,1
   __asm int 0x10
}

extern void charouta(int c, int a)
{
   __asm mov ax,0x0900
   __asm add ax,c
   __asm mov bx,a
   __asm xor bh,bh
   __asm mov cx,1
   __asm int 0x10
}

#else    /* !__STDC__ */

extern int scankey(void)
{
union _REGS r;
   r.x.ax = 0;
   return _int86(0x16,&r,&r);
}

extern int xscankey(void)
{
union _REGS r;
   r.x.ax = 0x1000;
   return _int86(0x16,&r,&r);
}

static void setcursor(int c)
{
union _REGS r;
   r.h.ah = 2;
   r.h.bh = 0;
   r.x.dx = c;
   _int86(0x10,&r,&r);
}

static int getcursor(void)
{
union _REGS r;
   r.h.ah = 3;
   r.h.bh = 0;
   _int86(0x10,&r,&r);
   return r.x.dx;
}

#ifdef __GNUC__
extern void my_clreol(void)
#else
extern void clreol(void)
#endif
{
union _REGS r;
   r.x.ax = 0x0A20;
   r.h.bh = 0;
   r.x.cx = 79;
   _int86(0x10,&r,&r);
}

static void outchar(int c)
{
union _REGS r;
   r.x.ax = 0x0E00 + c;
   r.h.bh = 0;
   _int86(0x10,&r,&r);
}

extern void charout(int c)
{
union _REGS r;
   r.x.ax = 0x0A00 + c;
   r.h.bh = 0;
   r.x.cx = 1;
   _int86(0x10,&r,&r);
}

extern void charouta(int c, int a)
{
union _REGS r;
   r.x.ax = 0x0900 + c;
   r.x.bx = a;
   r.h.bh = 0;
   r.x.cx = 1;
   _int86(0x10,&r,&r);
}

#endif      /* __STDC__ */
#endif      /* __WATCOMC__ */


#ifdef TEST

main()
{
   charout('a');
   printf("scankey: %04x\n",scankey());
   clreol();
   charouta('a',7);
   printf("scankey: %04x\n",scankey());
}

#endif
