/*
 *  sa.c        FAT12/16; save allocated clusters
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings.
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 *  29-Dec-1998 prozac  added SIZE_T
 *  30-Nov-1998 greggj  arraymax() fixed to handle arrays > 64K
 *  06-Oct-1998 prozac  ds.d_units + 1
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#ifdef _WIN32
#include <direct.h>     /* _getdrive() */
#else
#include <dos.h>        /* _dos_getdrive() */
#endif
#include <conio.h>

#define NONFLOPPY

/*
    The sectors that make up the FAT of a typical GB drive can
    exceed the DOS/80x86 64K segment limit. Therefore, the __huge
    data C extension must be used with 16-bit DOS compilers.
*/

#include "d_disk.h"
#include "d_lib.h"      /* clustertosector() */

#ifdef SMALL_DATA
#define lmalloc(n) halloc(n,1)
#define lfree(p)   hfree(p)
#define SIZE_T long
#else
#define lmalloc(n) malloc(n)
#define lfree(p)   free(p)
#define SIZE_T unsigned int
#endif

/*
    NOTE: `SIZE_T' needs to be renamed because it might be confused
    with `size_t'. However, I think that I shall re-think much of
    this before I change it.
*/

void usage(void);
int saveclusters(int dh, struct d_stat *ds, char *file, size_t maxcluster);
size_t LARGE *readfat(int dh, struct d_stat *ds);
int readfat12(int dh, struct d_stat *ds, size_t LARGE *clusters);
int readfat16(int dh, struct d_stat *ds, size_t LARGE *clusters);
long arraymax(size_t LARGE *array, long max);
int percent(long m, long n);


int main(int argc, char **argv)
{
int i,dh;
struct d_stat ds;
size_t maxcluster;
size_t LARGE *clusters;
unsigned cdisk,sdisk,ddisk;             /* current, source, destination */


    setvbuf(stdout,NULL,_IONBF,0);      /* flush stdout every printf */

    printf("\n");

    if (argc < 3 || argv[1][0] == '-' || argv[1][0] == '/')
        usage();

    if (argv[1][1] != ':')
        usage();

#ifdef _WIN32
    cdisk = _getdrive();
#else
    _dos_getdrive(&cdisk);
#endif

    sdisk = toupper(argv[1][0])-'@';
    ddisk = (argv[2][1] == ':') ? toupper(argv[2][0])-'@' : cdisk;

#ifndef NONFLOPPY
    if (sdisk > 2) {
        printf("This version will only write to drives A: and B:\n\n");
        usage();
        /* NOT REACHED */
    }
#endif

    if (ddisk == sdisk) {
        printf("Writing the file to the same disk defeats the purpose"
               " of this program.\n\n");
        usage();
        /* NOT REACHED */
    }

    /* open drive */

    printf("opening drive...\r");

    if ((dh = d_open(argv[1],D_RDONLY)) == -1) {
        d_perror("d_open");
        printf("\n");
        return 1;
    }

    /* get drive geometry */

    printf("checking drive...\r");

    if (d_hstat(dh,&ds) == -1) {
        d_perror("d_stat");
        printf("\n");
        return 1;
    }

    /* read FAT */

    printf("reading FAT...   \r");

    if ((clusters = readfat(dh,&ds)) == NULL) {
        printf("  bummer dude...\n");
        return 1;
    }

    /* find greatest allocated cluster number */

    maxcluster = arraymax(clusters,ds.d_units+1);

    lfree(clusters);

    printf("Drive %c:, ",sdisk+'@');
    printf("%lu bytes/sector, %lu sectors/cluster.\n",
           ds.d_bsize,ds.d_usize);

    printf("Of %ld clusters, %u is the last allocated.\n",
           ds.d_units+1,maxcluster);

    printf("%ld bytes will be written to '%s'.\n",
           (long)(maxcluster-2)*ds.d_usize*ds.d_bsize + (ds.d_dsec * ds.d_bsize),
           argv[2]);

    printf("\nDo you want to continue? [y/n] ");

    i = getchar();

    if (i == 'y') {
        printf("\n");
        saveclusters(dh,&ds,argv[2],maxcluster);
    }

    printf("\n");

    return 0;
}

/*
 * saveclusters - writes reserved sectors and up to `maxcluster' clusters
 *
 * NOTE: disk reads are in number of sectors, not number of bytes.
 *
 */

int saveclusters(int dh, struct d_stat *ds, char *file, size_t maxcluster)
{
FILE *fh;
long t;
size_t i,j;
size_t bufsiz;
unsigned char *buf;

    if ((fh = fopen(file,"wb")) == NULL) {
        perror(file);
        return 0;
    }

    /* system area first (up to first data sector) */

    bufsiz = (size_t)(ds->d_dsec * ds->d_bsize);

    if ((buf = malloc(bufsiz)) == NULL) {
        fclose(fh);
        return 1;
    }

    t = bufsiz;
    printf("system sectors - ");
    printf("%lu bytes so far",t);
    printf("\r");

    /* seek to first sector, read up to data sector */

    d_lseek(dh,0,SEEK_SET);

    if ((j = d_read(dh,buf,(size_t)ds->d_dsec)) != -1) {
        fwrite(buf,1,bufsiz,fh);
    }

    /* read cluster size at a time */

    bufsiz = (size_t)(ds->d_usize * ds->d_bsize);

    if ((buf = realloc(buf,bufsiz)) == NULL) {
        fclose(fh);
        return 1;
    }

    for (i = 2; i <= maxcluster; i++) {

        if (kbhit() && getch() == '\033')
            break;

        printf("cluster %5u",i);
        printf(" - %2d%% done",percent(i,maxcluster));
        printf(", %lu bytes so far",t);
        printf("\r");

        if ((j = d_read(dh,buf,(unsigned int)ds->d_usize)) != -1) {
            fwrite(buf,1,bufsiz,fh);
        }

        t += bufsiz;
    }

    printf("cluster %5u",i-1);
    printf(" - %2d%% done",percent(i,maxcluster));
    printf(", %lu bytes total ",t);
    printf("\r");

    free(buf);
    fclose(fh);
    return 0;
}


/*
 *  readfat     reads the FAT into an array of integers (front end)
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 */

size_t LARGE *readfat(int dh, struct d_stat *ds)
{
int i;
SIZE_T bufsiz;
size_t LARGE *buf;

    bufsiz = ((SIZE_T)ds->d_units+2) * sizeof(int);

    if ((buf = lmalloc(bufsiz)) == NULL)
        return NULL;

    d_lseek(dh,ds->d_bres,SEEK_SET);

    if (ds->d_units < 4096)
        i = readfat12(dh,ds,buf);
    else
        i = readfat16(dh,ds,buf);

    if (i != 1) {
        printf("readfat: %s error",(i == -1) ? "memory" : "read");
        lfree(buf);
        return NULL;
    }

    return buf;
}

/*
 *  readfat12   reads 12-bit FAT
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 *
 *  We can use `size_t' here rather than `SIZE_T' because FAT12
 *  drives have small FATs.
 *
 */

int readfat12(int dh, struct d_stat *ds, size_t LARGE *clusters)
{
int h,i;
size_t bufsiz;
unsigned char *buf;
unsigned short *pc,c;

    bufsiz = (size_t)(ds->d_bsize * ds->d_ftsize);

    if ((buf = malloc(bufsiz)) == NULL)
        return -1;

    if (d_read(dh,buf,(int)ds->d_ftsize) != (int)ds->d_ftsize) {
        free(buf);
        return 0;
    }

    buf += 3;

    for (h = 2; h <= ds->d_units+1; buf++)
    {
        for (i = 0; i < 2; i++)
        {
            pc = (unsigned short *)(buf++);

            if (i&1)
                c = *pc >> 4;               /* odd */
            else
                c = *pc & 0xFFF;            /* even */

            clusters[h] = (size_t)c;

            if (++h > ds->d_units+1)
                break;

        }
    }

    free(buf);
    return 1;
}

/*
 *  readfat16   reads 16-bit FAT
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 */

int readfat16(int dh, struct d_stat *ds, size_t LARGE *clusters)
{
long h,n;
SIZE_T bufsiz;
unsigned short LARGE *buf;

    bufsiz = (SIZE_T)(ds->d_bsize * ds->d_ftsize);

    if ((buf = lmalloc(bufsiz)) == NULL)
        return -1;

    if (d_read(dh,buf,(int)ds->d_ftsize) != (int)ds->d_ftsize)
    {
        lfree(buf);
        return 0;
    }

    for (h = n = 2; h <= ds->d_units+1; h++, n++)
    {
        clusters[h] = (size_t)buf[n];
    }

    lfree(buf);
    return 1;
}

/*
 * arraymax     find which number in array is largest
 *
 * Notes: Uses longs because array might be of type __huge *.
 *
 */

long arraymax(size_t LARGE *array, long max)
{
long i,n;

    for (i = n = 0; i < max; i++) {
        if (array[i]) {
            n = i;
        }
    }
    return n;
}

int percent(long m, long n)
{
float f;

    f = (float)m / n;
    f *= 100;
    return (int)f;
}

void usage(void)
{
    printf("usage: sa <d:> <file>\n\n"
    "Saves reserved sectors and up to and including the last\n"
    "allocated cluster of drive `d:' to `file'.\n"
    "\n"
    "For use with the program WF.\n"
    "\n"
    "Built with the free D_DISK library: "
    "http://www.diskwarez.com/d_disk.htm\n");
    exit(1);
}
