/*
 * COPY.C -- Internal Copy Command
 *
 * 01-Aug-98 (Rob Lake z63rrl@morgan.ucs.mun.ca)
 *      - started
 *
 * 13-Aug-1998 (John P. Price)
 * - fixed memory leak problem in copy function.
 * - fixed copy function so it would work with wildcards in the source
 *
 * 1999/01/24 ska
 * bugfix: setting time in ASCII mode before writing final ^Z
 * bugfix: copy(): if bsiz < 128: the opened files are not closed
 * chg: all error "return" statements perform the actions in the
 *      same sequence to allow the code optimizer to re-use code
 * add: support for CP/M style device names (openf.h)
 * chg/bugfix: overwrite(): use vcgetcstr(); could overflow the buffer
 * chg: there were two individual flags ASCII and BINARY to indicate
 *      which option was specified. Internally only the ASCII flag was
 *      used, if it was clear, the files were copied in binary mode.
 *      To support devices the two flags are passed unchanged to the
 *      copy() function and, if both are clear (== none specified on
 *      command line), it defaults to ASCII for devices, but will not
 *      append the ^Z character.
 *      Due to this, the destination file must be opened in ASCII mode, too;
 *      otherwise the newline character had to be handled manually.
 *
 * 1999/04/23 ska
 * bugfix: cmd_copy(): If malloc of sources fails, 'p' is not freed
 *
 * 29-Apr-1999 (John P. Price)
 * - Now works if dest is just a drive letter. (i.e "copy file a:")
 *
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <alloc.h>              //for coreleft()

#include "config.h"

#ifdef INCLUDE_CMD_COPY

#include "command.h"
#include "strings.h"

#include "copy.h"

#include "openf.h"

char *_truename(char *fn);

int isDir(char *fn)
{
	assert(fn);
	if (exist(fn) == 0)
		return 0;
  return _chmod(fn, 0) & FA_DIREC;
}

// examine a switch and modify the flags accordingly
int do_switches(char *arg, unsigned *flags)
{
	char c;

	assert(arg);
	assert(flags);

	if (stricmp(arg, "/-Y") == 0)
	{
		*flags |= PROMPT;
		*flags &= ~NPROMPT;
		return 1;
	}
	else if (strlen(arg) > 2)
	{
		display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
		return 0;
	}

	c = toupper(arg[1]);
	switch (c)
	{
		case 'V':
			*flags |= VERIFY;
			break;

		case 'A':
			*flags |= ASCII;
			*flags &= ~BINARY;
			break;

		case 'B':
			*flags |= BINARY;
			*flags &= ~ASCII;
			break;

		case 'Y':
			*flags &= ~PROMPT;
			*flags |= NPROMPT;
			break;

		default:
			error_invalid_switch(arg[1]);
			return 0;
	}
	return 1;
}

int add_file(struct files *f, char *arg, int *source, int *dest,
						 unsigned *flags)
{
	assert(f);
	assert(arg);
	assert(source);
	assert(dest);
	assert(flags);

	if (*dest)
	{
		display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
		return 0;
	}

	if (*source)
	{
		*dest = 1;
		f->flag = 0;
	}
	else
	{
		*source = 1;
		f->flag = SOURCE;
	}

	strcpy(f->file, arg);

	f->flag |= *flags & (ASCII | BINARY);

	if ((f->next = (struct files *)malloc(sizeof(struct files))) == NULL)
	{
		error_out_of_memory();
		return 0;
	}

	f = f->next;
	f->flag = 0;
	f->next = NULL;

	return 1;
}

int add_files(struct files *f, char *arg, int *source, int *dest,
							int *count, unsigned *flags)
{
	char t[128];
	int j;
	int	k;

	assert(f);
	assert(arg);
	assert(source);
	assert(dest);
	assert(count);
	assert(flags);
	if (*dest)
	{
		display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
		return 0;
	}

	j = 0;
	k = 0;

	while (arg[j] == '+')
		j++;

	while (arg[j] != '\0')
	{
    t[k] = arg[j++];
    if (t[k] == '+' || arg[j] == '\0')
    {
      if (!k)
        continue;
      if (arg[j] == '\0' && t[k] != '+')
        k++;
      t[k] = '\0';
      *count += 1;
      strcpy(f->file, t);
			*source = 1;
      f->flag |= *flags | SOURCE;
      if ((f->next = (struct files *)malloc(sizeof(struct files))) == NULL)
      {
        error_out_of_memory();
        return 0;
      }
      f = f->next;
      f->next = NULL;
      k = 0;
      f->flag = 0;
      continue;
		}
		k++;
	}
	if (arg[--j] == '+')
		*source = 0;

	return 1;
}

int get_dest(struct files *f, struct files *dest)
{
	struct files *p;
	struct files *start = f;

	assert(f);
	assert(dest);

	while (f->next != NULL)
	{
		p = f;
		f = f->next;
	}

	f = p;

	if ((f->flag & SOURCE) == 0)
	{
		free(p->next);
		p->next = NULL;
		strcpy(dest->file, f->file);
		dest->flag = f->flag;
		dest->next = NULL;
		f = start;
		return 1;
	}

	return 0;
}


int parse_cmd(struct files *f, int argc, char **arg, unsigned *flags)
{
	int i;
	int dest;
	int source;
	int count;

	assert(f);
	assert(flags);
	assert(arg);

	dest = 0;
	source = 0;
	count = 0;

	for (i = 0; i < argc; i++)
	{
		assert(arg[i]);
		if (arg[i][0] == '/')
		{
			if (!do_switches(arg[i], flags))
				return -1;
		}
		else
		{
			if (strcmp(arg[i], "+") == 0)
				source = 0;
			else if (strchr(arg[i], '+') == NULL && source)
			{
				if (!add_file(f, arg[i], &source, &dest, flags))
					return -1;
				f = f->next;
				count++;
			}
			else
			{
				if (!add_files(f, arg[i], &source, &dest, &count, flags))
					return -1;
				while (f->next != NULL)
					f = f->next;
			}
		}
	}
	dprintf(("parse_cmd: flags has %s\n", *flags & ASCII ? "ASCII" : "BINARY"));
	return count;
}

void delete_list(struct files *f)
{
	struct files *temp;

	assert(f);

	while (f != NULL)
	{
		temp = f;
		f = f->next;
		free(temp);
	}
}

int overwrite(char *fn)
{
	assert(fn);
	printf("Overwrite %s (Yes/No/All)?", fn);
	switch (vcgetcstr("YNA"))
	{
		case 'Y':
			return 1;
		case 'A':
			return 2;
	}
	/* default: */
	return 0;
}

#define BUFF_SIZE 32768U         /* 32k = max buffer size */

//int copy(struct files source, char *dest, int append, unsigned *flags)
int copy(char *source, char *dest, int append, unsigned *flags)
{
	char *buffer;
	unsigned bsiz;
	unsigned n;
	unsigned i;
	unsigned eof;
	unsigned wrote;
	unsigned attrib;
	int fd;
	int old_verify;
	struct ftime srctime;
	FILE *fsrc;
	FILE *fdest;
	int isdevice;                 /* true if source is a device */
	/* if so: the time and attributes of the destination
		 are not altered; open mode defaults to "rt" */
	int openASC;                  /* files are opened in ASCII mode */

	assert(source);
	assert(dest);
	assert(flags);

	/* we never copy the source onto destination, neither in overwrite
		 nor append mode */
	/* this check is weak -- 1999/01/24 ska */
	if (strcmp(dest, source) == 0)
	{
		display_string(TEXT_ERROR_COPIED_ONTO_SELF);
		return 0;
	}

	eof = 0;
	dprintf(("checking mode\n"));
	attrib = _chmod(source, 0);
	fd = devopen(source, O_RDONLY);
	if (fd == -1)
	{
		display_string(TEXT_ERROR_CANNOT_OPEN_SOURCE, source);
		return 0;
	}
	isdevice = isadev(fd);
	dprintf(("getting time\n"));
	getftime(fd, &srctime);
	if (close(fd) == -1)
		perror("copy");

	fsrc = fdevopen(source,
									(openASC = ((*flags & ASCII) != 0	/* open in ASCII mode */
															|| ((*flags & (ASCII | BINARY)) == 0	/* auto-select asc vs. bin */
                                  && isdevice))) != 0
	/* default to text */
	/* compiled with Borland C/C++ v5.0 to open the CON: device with
		 mode "rb" results in an input stream one cannot terminate
		 with ^Z --> it's not really useable. -- 1999/01/24 ska */
									? "rt"
									: "rb");

	if (fsrc == NULL)
	{
		display_string(TEXT_ERROR_CANNOT_OPEN_SOURCE, source);
		return 0;
	}

	dprintf(("copy: flags has %s\n", *flags & ASCII ? "ASCII" : "BINARY"));
//	dprintf(("copy: source flags has %s\n", (source.flag & ASCII) ? "ASCII" : "BINARY"));

	if (!append)
	{                             /* overwite mode */
		/* the destination file is automatically truncated if it already
			 exists */
		dprintf(("_chmod(%s, 1, 0);\n", dest));
		_chmod(dest, 1, 0);
		dprintf(("opening/creating\n"));
		fdest = fdevopen(dest, openASC ? "wt" : "wb");
	}
	else
	{
		dprintf(("opening/appending\n"));
    fdest = fdevopen(dest, openASC ? "at" : "ab");
  }

  if (fdest == NULL)
  {
    error_path_not_found();
    fclose(fsrc);
    return 0;
  }

  if ((bsiz = coreleft()) > BUFF_SIZE)
		bsiz = BUFF_SIZE;

  if (bsiz < 128 || (buffer = (char *)malloc(bsiz)) == NULL)
  {
		error_out_of_memory();
    fclose(fdest);
    fclose(fsrc);
    return 0;
  }

  strupr(source);
  puts(source);
  if (*flags & VERIFY)
  {
    old_verify = getverify();
    setverify(1);
  }
  do
  {
    n = fread(buffer, sizeof(char), bsiz, fsrc);
		if (*flags & ASCII)         /* don't use openASC here -- 1999/01/24 ska */
    {
      for (i = 0; i < n; i++)
			{
        if (buffer[i] == 0x1A)
				{
          eof = 1;
          break;
        }
      }
      n = i;
    }
    if (n == 0)
      break;
    wrote = fwrite(buffer, sizeof(char), n, fdest);
    if (wrote != n)
    {
      display_string(TEXT_ERROR_WRITING_DEST);
      free(buffer);
      fclose(fdest);
      fclose(fsrc);
      return 0;
    }
  }
  while (n && !eof);
  if (*flags & ASCII)           /* don't use openASC here -- 1999/01/24 ska */
  {
    dprintf(("appending ^Z\n"));
    putc('\x1a', fdest);
	}
	if (!isdevice)
	{
		dprintf(("setting time\n"));
		setftime(fileno(fdest), &srctime);
	}
	free(buffer);
	fclose(fdest);
	fclose(fsrc);
	if (!isdevice)
	{
		dprintf(("setting mode\n"));
		_chmod(dest, 1, attrib);
	}
	if (*flags & VERIFY)
		setverify(old_verify);
	return 1;
}

#pragma argsused
//int setup_copy(struct files *sources, char **p, int multiple,
//        char drive_d[MAXDRIVE], char dir_d[MAXDIR], char file_d[MAXFILE],
//               char ext_d[MAXEXT], int *append, unsigned *flags)

int setup_copy(struct files *sources, char **p, int multiple,
							 char *drive_d, char *dir_d, char *file_d,
							 char *ext_d, int *append, unsigned *flags)
{
	char drive_s[MAXDRIVE];
	char dir_s[MAXDIR];
	char file_s[MAXFILE];
	char ext_s[MAXEXT];
	char from_merge[MAXPATH];
	char *real_dest;
	char *real_source;
	int done;
	int	all = 0;
	int	copied = 0;
	struct ffblk ffblk;
	char *dest_dir;               /* if != NULL, destination is directory and is
																	 pointing to position to append filename to */
	int plainSource;              /* true if source is a plain file (no wildcards) */

	assert(sources);
	assert(drive_d);
	assert(dir_d);
	assert(file_d);
	assert(ext_d);
	assert(append);
	assert(flags);
	assert(p);

	real_source = (char *)malloc(MAXPATH);
	real_dest = (char *)malloc(MAXPATH);

	if (!real_source || !real_dest)
	{
		error_out_of_memory();
		delete_list(sources);
		free(real_source);
		free(real_dest);
		freep(p);
		return 0;
	}

	fnmerge(from_merge, drive_d, dir_d, file_d, ext_d);
	dest_dir = strchr(from_merge, '\0');
	if (strcmp(dir_d,"\\") != 0)
	{
		if (dest_dir[-1] == '\\')
			*--dest_dir = '\0';
		if (isDir(from_merge))
		{                             /* a name will be appended later */
			multiple = 0;
			*dest_dir++ = '\\';
		}
		else
		{
			dest_dir = NULL;
			multiple = 1;
		}
	}
	else
		multiple = 0;

	while (sources->next != NULL)
	{
		fnsplit(sources->file, drive_s, dir_s, file_s, ext_s);
		if ((plainSource = devopen(sources->file, O_RDONLY)) >= 0)
		{
			close(plainSource);
			plainSource = 1;
			done = 0;
		}
		else
		{
			plainSource = 0;
			done = FINDFIRST(sources->file, &ffblk, FA_ARCH);
			if (done)
			{
				error_file_not_found();
				delete_list(sources);
				freep(p);
				free(real_source);
				free(real_dest);
				return 0;
			}
		}
		while (!done)
		{
			if (dest_dir)             /* complete the destination filename */
				if (plainSource)
					strcpy(stpcpy(dest_dir, file_s), ext_s);
				else
					strcpy(dest_dir, ffblk.ff_name);

			strcpy(real_dest, from_merge);
			strupr(real_dest);
			if (plainSource)
				strcpy(real_source, sources->file);
			else
				fnmerge(real_source, drive_s, dir_s, ffblk.ff_name, NULL);
			printf("copying %s -> %s (%sappending%s)\n",
							 real_source, real_dest,
							 *append ? "" : "not ",
							 sources->flag & ASCII ? ", ASCII" : ", BINARY");
			if (access(real_dest, 0) == 0 && !all)
			{
				int over = overwrite(real_dest);
				if (over == 2)
					all = 1;
				else if (over == 0)
					goto next;
				else if (multiple)
					all = 1;
			}
//      if (copy(*sources, real_dest, *append, flags))
			if (copy(real_source, real_dest, *append, flags))
				copied++;
		next:
			done = plainSource || FINDNEXT(&ffblk);
			if (multiple)
				*append = 1;
		}
		sources = sources->next;
	}
	free(real_source);
	free(real_dest);
	return copied;
}

#pragma argsused
int cmd_copy(char *rest)
{
	char **p;
	char drive_d[MAXDRIVE];
	char dir_d[MAXDIR];
	char file_d[MAXFILE];
	char ext_d[MAXEXT];

	int argc;
	int dest_has_wildcards;
	int dest_found;
	int multiple;
	int append;
	int files;
	int copied;
	struct files *sources;
	struct files dest;
	struct files *start;
	unsigned flags = 0;

	assert(rest);

	// break up all the parameters into seperate strings
	p = split(rest, &argc);

	if (argc == 0)
	{
		error_req_param_missing();
		return 0;
	}

	sources = (struct files *)malloc(sizeof(struct files));
	if (!sources)
	{
		error_out_of_memory();
		freep(p);
		return 0;
	}
	sources->next = NULL;
	sources->flag = 0;

	if ((files = parse_cmd(sources, argc, p, &flags)) == -1)
	{
		delete_list(sources);
		freep(p);
		return 0;
	}
	else if (files == 0)
	{
		error_req_param_missing();
		delete_list(sources);
		freep(p);
		return 0;
	}
	start = sources;

	dest_found = get_dest(sources, &dest);
	_truename(dest.file);

	if (dest_found)
	{
		fnsplit(dest.file, drive_d, dir_d, file_d, ext_d);
		if (isDir(dest.file))
		{
			strcat(dir_d, file_d);
			strcat(dir_d, ext_d);
			file_d[0] = 0;
			ext_d[0] = 0;
		}
	}

	if (strchr(dest.file, '*') || strchr(dest.file, '?'))
    dest_has_wildcards = 1;
  else
    dest_has_wildcards = 0;

  if (strchr(rest, '+'))
    multiple = 1;
  else
    multiple = 0;

  append = 0;
  copied = 0;

  if (dest_found && !dest_has_wildcards)
  {
    copied = setup_copy(sources, p, multiple, drive_d, dir_d, file_d, ext_d, &append, &flags);
  }
  else if (dest_found && dest_has_wildcards)
  {
    display_string(TEXT_NOT_IMPLEMENTED_YET);
		delete_list(sources);
    freep(p);
    return 0;
  }
  else if (!dest_found && !multiple)
  {
    fnsplit(sources->file, drive_d, dir_d, file_d, ext_d);
		if (isDir(sources->file))
    {
      strcat(dir_d, file_d);
      strcat(dir_d, ext_d);
      file_d[0] = 0;
      ext_d[0] = 0;
    }
    copied = setup_copy(sources, p, 0, "", "", file_d, ext_d, &append, &flags);
  }
  else
	{
    fnsplit(sources->file, drive_d, dir_d, file_d, ext_d);
    if (isDir(sources->file))
		{
      strcat(dir_d, file_d);
      strcat(dir_d, ext_d);
      file_d[0] = 0;
      ext_d[0] = 0;
    }
    strupr(sources->file);
    puts(sources->file);
    append = 1;
    copied = setup_copy(sources->next, p, multiple, drive_d, dir_d, file_d, ext_d, &append, &flags) + 1;
  }

  printf("        %d file(s) copied\n", copied);
  delete_list(start);
  freep(p);
  return 1;
}

#ifdef DEBUG_STANDALONE
#pragma argsused
int main(int argc, char *argv[])
{
  dprintf(("coreleft: %u\n", coreleft()));
  if (argc == 1)
    argv[1][0] = 0;
	cmd_copy(argv[1]);
  dprintf(("coreleft: %u\n", coreleft()));
  return 1;
}
#endif                          /* DEBUG_STANDALONE */
/* #undef DEBUG */
#endif                          /* INCLUDE_CMD_COPY */
