/*
 *  dos\bios.c      INT 13h functions
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings.
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 *  1.1 grejen  06-Jun-1998 added #include <dos.h>
 *  1.0 grejen  24-May-1998 added error messages
 *
 */

#include <limits.h>     /* CHAR_BIT */
#ifdef __WATCOMC__
#include <dos.h>
#endif
#include <bios.h>

#include "bios.h"

static int adjdrv(int drv);

/*
 * disk_get_physical    get basic disk geometry (tracks, sectors, heads)
 *
 * returns -1 if successful else BIOS error code
 */

extern int disk_get_physical(int disk, int *t, int *s, int *h)
{
unsigned int tmp;
union REGS regs;

    regs.x.ax = 0x0800;
    regs.h.dl = (unsigned char)adjdrv(disk);

    int86(0x13,&regs,&regs);

    if (regs.x.cflag)
        return (int)regs.x.ax;

    *h = regs.h.dh;
    *t = regs.h.ch;             /* low bits only */
    tmp = *s = regs.h.cl;       /* has high bits of track also */
    *s &= 0x3f;                 /* mask off track high bits */
    *t += (tmp >> 6) << 8;      /* shift and add track high bits */

    return -1;
}

/*
 * disk_read_p      read sector(s)
 *
 */

extern int disk_read_p(int disk, unsigned trk, unsigned sec, unsigned head,
                       void *buf, int nsecs, int secsiz)
{
unsigned i;
struct _diskinfo_t blk;

    blk.drive = adjdrv(disk);
    blk.track = trk;
    blk.sector = sec;
    blk.head = head;
    blk.nsectors = nsecs;
    blk.buffer = buf;

    _bios_disk(_DISK_READ,&blk);
    i = _bios_disk(_DISK_STATUS,&blk);

    i >>= CHAR_BIT;     /* AH contains result; AL sectors read */

    return (i) ? i : -1;
}

/*
 * disk_write_p     write sector(s)
 *
 */

extern int disk_write_p(int disk, unsigned trk, unsigned sec, unsigned head,
                        void *buf, int nsecs, int secsiz)
{
unsigned i;
struct _diskinfo_t blk;

    blk.drive = adjdrv(disk);
    blk.track = trk;
    blk.sector = sec;
    blk.head = head;
    blk.nsectors = nsecs;
    blk.buffer = buf;

    _bios_disk(_DISK_WRITE,&blk);
    i = _bios_disk(_DISK_STATUS,&blk);

    i >>= CHAR_BIT;     /* AH contains result; AL sectors read */

    return (i) ? i : -1;
}

/* floppys: 0 - 7Fh, hards: 80h - FFh */

static int adjdrv(int drv)
{
    if (drv > 1)
        return drv + 0x80 - 2;
    return drv;
}


static struct int13_msg {
   int code;
   char *msg;
} int13_diskerr_msg[] = {
   { 0x01, "bad command" },
   { 0x02, "address mark not found" },
   { 0x03, "write-protected disk" },
   { 0x04, "sector not found" },
   { 0x05, "reset failed" },
   { 0x06, "diskette removed or changed" },
   { 0x07, "bad parameter table" },
   { 0x08, "DMA overrun" },
   { 0x09, "attempt to DMA across 64K boundary" },
   { 0x0A, "bad sector detected" },
   { 0x0B, "bad track detected" },
   { 0x0C, "unsupported track or media type not found" },
   { 0x0D, "invalid number of sectors on format" },
   { 0x0E, "control data address mark detected" },
   { 0x0F, "DMA arbitration level out of range" },
   { 0x10, "uncorrectable CRC/EEC on read" },
   { 0x11, "ECC corrected data error" },
   { 0x20, "controller failure" },
   { 0x40, "seek failed" },
   { 0x80, "timeout" },
   { 0xAA, "drive not ready" },
   { 0xBB, "undefined error" },
   { 0xCC, "write fault" },
   { 0xE0, "status error" },
   { 0xFF, "sense operation failed" },
   { 0,    "unknown error" }
};

#define nerrors (sizeof(int13_diskerr_msg)/sizeof(struct int13_msg))

extern const char *disk_error(int error)
{
int i;
char *errmsg;

    for (i = 0; i < nerrors-1; i++) {
        if (int13_diskerr_msg[i].code == error) {
            break;
        }
    }

    errmsg = int13_diskerr_msg[i].msg;

    return errmsg;
}
