/*
 *  d_disk.c    main library functions
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 *  29-Nov-1998 greggj  compiles with DISKLIB; removed hard tabs
 *  11-Oct-1998 prozac  fixed (for the most part) D_BINARY mode problem
 *  26-Sep-1998 grejen  see changelog.txt
 *  24-Apr-1998 grejen  handles reading blocks > 64K
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>

#include "d_disk.h"
#include "d_lib.h"
#include "d_error.h"

#ifndef NDEBUG
#define STATIC              /* to make non-publics public for testing */
#else
#define STATIC static
#endif

/* public data defined here */

int d_errno;

/* internal data defined here */
/*
    struct disk *_dd;

    This is a file scope, singly linked list for keeping track of
    the open disk handles. Only disk geometry information that is
    absolutely needed should put into it to keep things small and
    simple (small and simple is the number one design parameter).

    If a disk is not opened in binary mode the track, sector and
    head members (t,s,h) will be undetermined.

*/

STATIC struct disk *_dd = NULL;

/* internal functions */

STATIC void insertdisk(struct disk *d);       /* insert an open'd handle */
STATIC struct disk *removedisk(int dh);       /* remove (extract) " */
STATIC struct disk *handle_todisk(int dh);    /* handle to struct * */
STATIC int getphysical(struct disk *dd);
STATIC void setphysical(struct disk *dd);
STATIC int diskio(struct disk *dd, unsigned int nsecs, char LARGE *buffer,
          int (diskfunc(struct disk *dd, unsigned int nsecs, void *buffer)));
STATIC int readdisk(struct disk *dd, unsigned int nsecs, void *buf);
STATIC int writedisk(struct disk *dd, unsigned int nsecs, void *buf);
STATIC int disk_get_parameters(int disk, struct d_stat *ds);


/* start of code */

extern int d_open(const char *drv, int mode)
{
struct disk *dd;
struct d_stat ds;

    if (d_stat(drv,&ds) != 0)
        return -1;                  /* d_stat() error sets d_errno */

    dd = malloc(sizeof(struct disk));

    if (dd == NULL) {
        d_errno = D_ENOMEM;
        return -1;
    }

    dd->drv = (tolower)(*drv) - 'a';
    dd->mode = mode;
    dd->sec = 0;
    dd->dd = NULL;

    /* set the common values */

    dd->maxs = (int)ds.d_psecs;
    dd->maxh = (int)ds.d_pheads;
    dd->hsecs = (int)ds.d_bhid;
    dd->secsiz = (int)ds.d_bsize;

    dd->phys = dd->drv;
    dd->offs = 0;

    /* if binary get the logical -> physical translation data if not floppy */

    if (dd->mode & D_BINARY && dd->drv < 2)
        getphysical(dd);

    /* if binary and not unknown media use BIOS I/O (INT 13h) */

    if (dd->mode & D_BINARY && ds.d_media == 0)
        dd->iotype = INT13;
    else
        dd->iotype = (ds.d_size > (32L * 1024L *1024L)) ? INT25X : INT25;

    insertdisk(dd);

    return dd->dh;
}

extern int d_close(int dh)
{
struct disk *dd;

    dd = removedisk(dh);

    if (dd == NULL)
        return -1;

    free(dd);
    return 0;
}

extern long d_tell(int dh)
{
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    return dd->sec;
}

extern long d_lseek(int dh, long offset, long whence)
{
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    switch (whence) {
        case SEEK_SET:
            if (offset < 0)
                goto e_bounds;
            dd->sec = offset;
            break;
        case SEEK_CUR:
            if (dd->sec + offset < 0)
                goto e_bounds;
            dd->sec += offset;
            break;
        case SEEK_END:
        default:
            goto e_argument;
    }

    return dd->sec;

e_bounds:
e_argument:

    d_errno = D_EINVAL;
    return -1;

}

extern int d_read(int dh, void LARGE *buf, unsigned int nsecs)
{
int i;
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    /* diskio starts reading from dd->sec */

    i = diskio(dd,nsecs,(char LARGE *)buf,readdisk);

    if (i == -1) {
        dd->sec += nsecs;
        return nsecs;
    }

    d_errno = dskerr_to_errno(i,dd->iotype);
    return -1;
}

extern int d_write(int dh, void LARGE *buf, unsigned int nsecs)
{
int i;
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    if ((dd->mode & D_RDWR) == 0) {
        d_errno = D_ERDONLY;
        return -1;
    }

    /* diskio starts writing from dd->sec */

    i = diskio(dd,nsecs,(char LARGE *)buf,writedisk);

    if (i == -1) {
        dd->sec += nsecs;
        return nsecs;
    }

    d_errno = dskerr_to_errno(i,dd->iotype);
    return -1;
}

extern int d_stat(const char *path, struct d_stat *buf)
{
int i;
int disk;

    disk = tolower(*path) - 'a';

    if ((i = disk_get_parameters(disk,buf)) == -1)
        return 0;

    d_errno = dskerr_to_errno(i,0);
    return -1;
}

extern int d_hstat(int dh, struct d_stat *buf)
{
int i;
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    if ((i = disk_get_parameters(dd->drv,buf)) == -1)
        return 0;

    d_errno = dskerr_to_errno(i,0);
    return -1;
}

extern int d_bsize(int dh)
{
struct disk *dd;

    dd = handle_todisk(dh);

    if (dd == NULL)
        return -1;

    return dd->secsiz;
}


/* internals */

STATIC struct disk *handle_todisk(int dh)
{
struct disk *dd;

    dd = _dd;
    while (dd) {
        if (dd->dh == dh)
            break;
        dd = dd->dd;
    }

    if (dd == NULL)
        d_errno = D_EHANDLE;

    return dd;
}

STATIC void insertdisk(struct disk *d)
{
struct disk *dd;

    if (_dd == NULL) {
        d->dh = 0;
        _dd = d;                        /* first one */
        return;
    }

    if (_dd->dh > 0) {
        d->dh = 0;
        d->dd = _dd;
        _dd = d;                        /* prepend */
        return;
    }

    dd = _dd;
    while (dd->dd) {
        if (dd->dd->dh != dd->dh+1) {
            d->dd = dd->dd;
            d->dh = dd->dh+1;
            dd->dd = d;                /* insert */
            return;
        }
        dd = dd->dd;
    }

    d->dh = dd->dh+1;
    dd->dd = d;                        /* append */
}

STATIC struct disk *removedisk(int dh)
{
struct disk *dd,*tdd;

    if (_dd == NULL || dh < _dd->dh)    /* below */
        goto e_badf;

    if (_dd->dh == dh) {
        tdd = _dd;                      /* at start */
        _dd = _dd->dd;
        return tdd;
    }

    dd = _dd;
    while (dd->dd) {
        if (dd->dd->dh == dh) {
            tdd = dd->dd;              /* inside/at end */
            dd->dd = dd->dd->dd;
            return tdd;
        }
        dd = dd->dd;
    }

e_badf:

    d_errno = D_EHANDLE;
    return NULL;
}

/*
 *  diskio      do the disk I/O, caching I/O blocks > 64K
 *
 */

#define BLKSIZ (63U*1024U)

STATIC int diskio(struct disk *dd, unsigned int nsecs, char LARGE *buffer,
            int (diskfunc(struct disk *dd, unsigned int nsecs, void *buffer)))
{
char *buf;
unsigned int i,j,n;
unsigned long k,l,u,m,sector;

    u = (long)nsecs * dd->secsiz;   /* how much is needed in bytes */

    if (u < BLKSIZ)
        return diskfunc(dd,nsecs,buffer);

    sector = dd->sec;

    u /= BLKSIZ;                    /* how many 63K blocks needed */
    m = (long)nsecs * dd->secsiz;
    m %= BLKSIZ;                    /* how much is left over from / 63K */

    buf = malloc(BLKSIZ);
    if (buf == NULL)
        return 0;

    n = BLKSIZ / dd->secsiz;
    for (i = 0; i < (unsigned int)u; i++) {
        j = diskfunc(dd,n,buf);
        if (j != -1) {
            free(buf);
            return j;
        }
        l = BLKSIZ * (long)i;
        for (k = 0; k < BLKSIZ; k++)            /* can not use memcpy */
            buffer[k+l] = buf[k];               /*  on LARGE data */
        sector += n;
    }
    l = BLKSIZ * (long)i;

    n = (unsigned int)m / dd->secsiz;
    if (n) {
        j = diskfunc(dd,n,buf);
        for (k = 0; k < m; k++)
            buffer[k+l] = buf[k];
    }

    free(buf);

    return j;

}


STATIC void setphysical(struct disk *dd)
{
long sec;

    sec = dd->sec;
    if (!(dd->mode & D_BINARY))
        sec += dd->hsecs;

    if (dd->offs)
        sec += (dd->offs * dd->maxs * dd->maxh);

    printf("%d %d\n",dd->maxs,dd->maxh);

    dd->t  = (unsigned)(sec / (dd->maxs * dd->maxh));
    dd->s  = (unsigned)(sec % (dd->maxs)+1);
    dd->h  = (unsigned)(sec % (dd->maxs * dd->maxh));
    dd->h /= (unsigned)dd->maxs;
/*
    I figured that it would work this way, but it does not. Perhaps
    the value of ds.offs can be non-multiples of ds.s (sectors per
    track)?

    if (dd->offs)
        dd->t += dd->offs;
*/

}

/*
 * getphysical  -   get physical disk parameters
 *
 * This function figures out if a drive (e.g. "D:") is a logical
 * Partition, if it is, the `struct disk' member `phys' and `offs'
 * are set to the physical drive number (i.e. 0-N) and the number
 * of cylinders to be added to get to the partition respectively.
 *
 * NOTE: This function will not work properly if there are any
 *       non-DOS partitions in a system.
 *
 * ALSO: This function will not work properly (the BIOS call will
 *       fail) under Windows.
 */

STATIC int getphysical(struct disk *dd)
{
int i,t,s,h,d,r,p;
struct d_stat ds;

    dd->phys = dd->drv;
    dd->offs = 0;

    /* get list of all HDs and their number of tracks */
    /* get disk parameters from DOS and figure out if it's a partition */

    r = 0;
    d = 2;

    for (i = 0; i < 4; i++) {
        t = 0;
        disk_get_physical(i+2,&t,&s,&h);
        if (t > 0) {
            p = 0;
            do {
                if (disk_get_parameters(d,&ds) != -1)
                    break;
                if (d == dd->drv) {
                    dd->phys = i+2;
                    dd->offs = p;
                    break;
                }
                d++;
                p += (int)ds.d_ptracks;
            }
            while (p < t);
        }
    }
    return 0;
}

#ifndef DISKLIB

STATIC int readdisk(struct disk *dd, unsigned int nsecs, void *buf)
{
int i;

    switch (dd->iotype) {
        case INT25:
            i = disk_read(dd->drv,dd->sec,buf,nsecs,dd->secsiz);
            break;
        case INT25X:
            i = disk_read_ext(dd->drv,dd->sec,buf,nsecs,dd->secsiz);
            break;
        case INT13:
            setphysical(dd);
            i = disk_read_p(dd->phys,dd->t,dd->s,dd->h,buf,nsecs,dd->secsiz);
            break;
        default:
            i = 0;
            break;
    }
    return i;
}

STATIC int writedisk(struct disk *dd, unsigned int nsecs, void *buf)
{
int i;

    switch (dd->iotype) {
        case INT25:
            i = disk_write(dd->drv,dd->sec,buf,nsecs,dd->secsiz);
            break;
        case INT25X:
            i = disk_write_ext(dd->drv,dd->sec,buf,nsecs,dd->secsiz);
            break;
        case INT13:
            setphysical(dd);
            i = disk_write_p(dd->phys,dd->t,dd->s,dd->h,buf,nsecs,dd->secsiz);
            break;
        default:
            i = 0;
            break;
    }
    return i;
}

#else

STATIC int readdisk(struct disk *dd, unsigned int nsecs, void *buf)
{
int i;

    switch (dd->iotype) {
        case INT25:
            i = disk_read(dd->drv,dd->sec,buf,nsecs);
            break;
        case INT25X:
            i = disk_read_ext(dd->drv,dd->sec,buf,nsecs);
            break;
        case INT13:
            setphysical(dd);
            i = disk_read_p(dd->phys,dd->t,dd->s,dd->h,buf,nsecs);
            break;
        default:
            i = 0;
            break;
    }

    if (i == DISK_OK)
        i = -1;
    else
        i = 0;

    return i;
}

STATIC int writedisk(struct disk *dd, unsigned int nsecs, void *buf)
{
int i;

    switch (dd->iotype) {
        case INT25:
            i = disk_write(dd->drv,dd->sec,buf,nsecs);
            break;
        case INT25X:
            i = disk_write_ext(dd->drv,dd->sec,buf,nsecs);
            break;
        case INT13:
            setphysical(dd);
            i = disk_write_p(dd->phys,dd->t,dd->s,dd->h,buf,nsecs);
            break;
        default:
            i = 0;
            break;
    }

    if (i == DISK_OK)
        i = -1;
    else
        i = 0;

    return i;
}

#endif
