/***
*debug.c - heap checking
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Debugging of the heap.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   2.6   21-Jan-1998    HeapCheck; default list type now byte
   2.5   15-Nov-1994    changed/added __BORLANDC__ checks
   2.4   11-Sep-1994    bugfix in displaying "FREE"/"USED"
                        added __WATCOMC__
   2.3   29-Jul-1994    <more>, heapmin
   2.2   09-Mar-1994    hblock,hfrees
   2.1   13-Jan-1994    "alloc.h"
   2.0   28-Nov-1993

   Release Notes:

   Except for the references to the alloc/free call counts and the
   help text (in DATA.C), and the use of my getstr(), print(),
   curright(), charout() and pause() (all very simple), this is an
   independent module.

   Programming Notes:

   See ALLOC.H (mine) for Microsoft/Borland/Watcom for how to deal
   with their individual definitions of the HEAP macros and _heapinfo
   structure.

*/

#include <stdio.h>
#include <malloc.h>
#ifdef __WATCOMC__
#include <ctype.h>
#endif

#include "disked.h"                       /* HeapCheck */
#include "mylib.h"                        /* getstr() */
#include "console.h"                      /* input, print and cursor funcs */
#include "alloc.h"                        /* my malloc shit */

#ifndef __LARGE__
#error __FILE__ requires large data model
#endif

/* globals referenced here */

extern unsigned int blocks;               /* alloc() free() calls */
extern unsigned int nblocks;              /*  defined in ALLOC.C */
extern unsigned int hblocks;              /*  these are for statistics */
extern unsigned int frees;                /*  only -- except for these */
extern unsigned int nfrees;               /*  references, this is a */
extern unsigned int hfrees;               /*  compleletly separate unit */

extern char *debug_h[];                   /* help text, in DATA.C */
extern char *debug_d[];                   /* help text, in DATA.C */

/* NO globals defined here */


static int list = 2;
static char *list_types[] = {"","ascii","byte","word","dword"};

static void list_memory(int type);
static void heapdump(int);

extern void debug(void)
{
   for (;;)
   {
      print("\n(debug)*");
      switch(input())
      {
         case ' ':
         case 'h':
            print(" Heap status:");
            print("\n\n\tStack %u",_stackavail());
            print(", Near Heap %lu (%lu largest block)",memavail(),(unsigned long)_memmax());
            print("\n\tAlloc calls %d:%d:%d, Free calls %d:%d:%d", \
               hblocks,blocks,nblocks,hfrees,frees,nfrees);
            heapdump(0);
            break;
         case 'd':
            print(" Heap dump:");
            heapdump(1);
            break;
         case 'k':
            list++;
            if (list == 5)
                list = 1;
            print(list_types[list]);
            list_memory(list);
            break;
         case 'l':
            list_memory(0);
            break;
#ifdef __WATCOMC__
         case 'g':
            print(" growing...");
            _heapgrow();
            _nheapgrow();
            break;
#endif
#ifndef __BORLANDC__
         case 'm':
            print(" minimizing...");
            if (_heapmin() != 0)
               print(" error, _heapmin() returned -1");
            if (_nheapmin() != 0)
               print(" error, _nheapmin() returned -1");
            break;
#endif
         case 'q':
         case '.':
         case 0x1b:
            return;

         case '/':
            print(" Debug Commands:\n\n");
            print("    'h' heap status\n");
            print("    'd' heap dump\n");
            print(" 'ln:n' list memory\n");
            print("    'k' change list format\n");
#ifndef __BORLANDC__
            print("    'm' minimize heap\n");
#endif
#ifdef __WATCOMC__
            print("    'g' grow heap\n");
#endif
            break;

         case '?':
            print("command help: h,d,l,k"
#ifndef __BORLANDC__
            ",m"
#endif
#ifdef __WATCOMC__
            ",g"
#endif
            );
            switch (input())
            {
               case 'h':
                  disptext(debug_h);
                  break;
               case 'd':
                  disptext(debug_d);
                  break;
            }
            break;

         default:
            break;
      }
   }
}

/***
*heapdump   -  display heap status
* 
*  Note: See ALLOC.H for Microsoft/Borland/Watcom for how to deal
*        with their individual definitions of the HEAP macros and
*        _heapinfo structure.
****/

static void heapdump(int display)
{
struct _heapinfo hi;
int heapstatus;
int uf,un,ff,fn;           /* count things */
int lines;                 /* <more> */

   print("\n");
   uf = un = ff = fn = 0;
   hi._pentry = NULL;

   if (display)
      print("\nFar Heap:");

   lines = 2;

   while ((heapstatus = _heapwalk(&hi)) == _HEAPOK)
   {
      if (display)
      {
         print("\n\t%s",hi._useflag  == _USEDENTRY ? "USED" : "FREE");
         print(" block at %FP of size %lu\t",hi._pentry,(long)hi._size );
         if (!(++lines%24))
            pause();
      }
      if (hi._useflag == _FREEENTRY)
      {
         ++ff;
#ifdef HEAP_CHECKING
         if (display && HeapCheck)
         {
            size_t i;
            unsigned char *p;          /* to check free blocks for overwrites */
            for (p = (unsigned char *)hi._pentry, i = 0; i < hi._size; p++,i++)
               if (*p != 254)
               {
                  print(" overwrite at %FP (%02X)",p,(unsigned char)*p);
                  break;
               }
         }
#endif
      }
      else
         ++uf;
   }
   if (display)
      print("\n\tStatus: %s",heapstat(heapstatus));

#ifndef __BORLANDC__

   hi._pentry = NULL;
   if (display)
      print("\nNear Heap:");
   while ((heapstatus = _nheapwalk(&hi)) == _HEAPOK)
   {
      if (display)
      {
         print("\n\t%s",hi._useflag  == _USEDENTRY ? "USED" : "FREE");
         print(" block at %FP of size %u\t",hi._pentry,hi._size );
         if (!(++lines%24))
            pause();
      }
      if (hi._useflag == _FREEENTRY)
      {
         ++fn;
#ifdef HEAP_CHECKING
         if (display && HeapCheck)
         {
            size_t i;
            unsigned char *p;          /* to check free blocks for overwrites */
            for (p = (unsigned char *)hi._pentry, i = 0; i < hi._size; p++,i++)
               if (*p != 254)
               {
                  print(" overwrite at %FP (%02X)",p,(unsigned char)*p);
                  break;
               }
         }
#endif
      }
      else
         ++un;
   }
   if (display)
      print("\n\tStatus: %s\n",heapstat(heapstatus));

#endif   /* near heap check */

   if (!display)
      print("\tUsed blocks %d:%d, Free blocks %d:%d.\n",uf,un,ff,fn);
}

#ifdef _MSC_VER
#pragma check_pointer(off)
#endif

/* dump memory, call with 0 to "set" format */

static void list_memory(int type)
{
int i,j;
static UINT8  *cp;
static UINT16  *ip;
static UINT32 *lp;
static int size = 1;
char inbuf[20];

   if (type)
   {
      size = type;
      return;
   }
   print("l");
   if ((i = getstr(inbuf,9,_HEX|_PUNCT)) > 0)
   {
      i = j = 0;
      sscanf(inbuf,"%x:%x",&i,&j);
      cp = (UINT8 *) ((((long)i)<<16) + (long)j);
      ip = (UINT16 *) ((((long)i)<<16) + (long)j);
      lp = (UINT32 *) ((((long)i)<<16) + (long)j);
   }
   else if (i == ABORT)
      return;

   print("\n");

   for (i = 0; i < 8; i++)
   {
      if (size == 1)
      {
         print("\n%Fp ",cp);
         for (j = 0; j < 80-21; j++)
         {
            if (*cp == 0 || *cp == 255)
               charout(' ');                 /* charout() displays all */
            else                             /*  254 characters */
               charout(*cp);
            cp++;
            curright();                      /* doesn't move cursor though */
         }
      }
      else if (size == 2)
      {
         print("\n%Fp ",cp);
         for (j = 0; j < 20; j++)
            print("%02x ",(int)*cp++);
      }
      else if (size == 3)
      {
         print("\n%Fp ",ip);
         for (j = 0; j < 12; j++)
            print("%04x ",*ip++);
      }
      else if (size == 4)
      {
         print("\n%Fp ",lp);
         for (j = 0; j < 6; j++)
            print("%04x:%04x ",(int)((*lp)>>16),(int)*lp++);
      }
   }
   if (size == 1 || size == 2)
   {
      ip = (UINT16 *)cp;
      lp = (UINT32 *)cp;
   }
   else if (size == 3)
   {
      cp = (UINT8 *)ip;
      lp = (UINT32 *)ip;
   }
   else if (size == 4)
   {
      ip = (UINT16 *)lp;
      cp = (UINT8 *)lp;
   }
   print("\n");
}
