/***************************************************************************
*
*	FSCK.C	by: Andreas Rosenberg
*               created: 1994.01.07 updated: 2000.12.13
*               eMail: andy@rosen-berg.de
*                      andreas.rosenberg@t-online.de
*
*   A MSDOS like CHKDSK utility,to show info about the disk
*   and allows to free lost clusters. Under MSDOS compile with
*   compact memory model.
*
*   1994.04.03 released V1.0a - fixed a segment limit violation bug
*   1994.11.10 released V1.0b - fixed bug with fats >= 0x8000 bytes
*                               fixed bug in show_boot (capacity of drive)
*   2000.12.23 released V1.0c - minor changes (compatibility with CHKDSK)
*                               cosmetical stuff
*
*   If you should find any bugs it would be nice to inform me.
*
***************************************************************************/
#define VERSION "V1.0c 2000.12.23"

#include "disk.h"
#define malloc farmalloc
#define free farfree

typedef struct {
    ulong next;
    ulong parent;
    int cnt;} PART_SEC;

typedef struct {
    ulong sectnum;
    unsigned int sectcnt;
    void far *addr;
    } DISK_PACK;
/* Function prototypes */
/*int show_root(int dev_num,int mode, unsigned long sector);*/
int show_boot(int drv_num,int mode);
int show_fat(int drv_num,int mode);
int scan_fdb(int drv,int mode);
int scan_rdir(int drv,int mode,FAT_TABLE *ft);
int get_flat_fat(int drv,FAT_TABLE *ft);
int read_boot(int drv,byte *b);
void print_ferr(int err,word cluster,char *fn);
void print_derr(int err,char *fn);
void print_discerr(int err);
int backup(int drv,int mode);
int restore(int drv,int mode);
int show_root(int dev,int mode);
int show_root_i(int dev,int mode, PART_SEC *ps);

/* Global variables */
int MERROR;
int CPU=1;    /* CPU = 1 means Intel byte order - relic from earlier version */
              /* CPU = 0 means Motorola CPU */

int DRV=PHSDISK;                 /* first hard drive */
INT13_TAB int13_tab[MAX_DISKS];  /* we support up to 8 fixed disks */
BOOT_SECTOR boot;                /* buffer to hold boot sector */
ROOT_SECTOR root;                /* buffer to hold root sector */
BPB XBPB;                        /* BIOS parameter block for this drive */

word im(byte huge *x)            /* intel to motorola & v.v.*/
	{
	return (*(word huge *)x);
	}

void siml(byte *adr,ulong val)   /* store ulong at adr */
	{
	*(ulong *)adr = val;
	}

void init(void)                  /* initialize disk table */
    {
    int i;
    for (i=0;i<MAX_DISKS;i++)
        int13_tab[i].valid = 0;
    }

long Rw_abs_sector(int mode,void huge *buff,int cnt,ulong recnr,int dev)
	{
    int cyl,head,sect;
    int mdev = dev-PHSDISK;
    ulong cs,r;

    if (!int13_tab[mdev].valid)  /* Get Drive Parameters */
        {
        union REGS regs;

        regs.h.ah = 0x08;
        regs.h.dl = dev;
        int86(0x13,&regs,&regs);
        if (regs.x.cflag)
	        return(E_DISCGENL);
        int13_tab[mdev].dev = dev;
        int13_tab[mdev].heads = regs.h.dh + 1;
        int13_tab[mdev].sect = regs.h.cl&0x3f;
        int13_tab[mdev].cyl = (word)regs.h.ch + ((word)(regs.h.cl&0xC0)<<2);
        int13_tab[mdev].valid = 1;
        }
    cs = (ulong)int13_tab[mdev].heads * (ulong)int13_tab[mdev].sect;
    cyl = (int)(recnr/cs);
    r = recnr - (cyl*cs);
    head = (int)(r/int13_tab[mdev].sect);
    sect = (int)(recnr - (cyl*cs) - (head*int13_tab[mdev].sect) + 1);
    /*printf("Abs read c:%d,h:%d,s:%d, cs:%ld, r:%ld,rec:%ld\n",cyl,head,sect,cs,r,recnr);*/
	if ((mode == MD_READ) && (cyl <= int13_tab[mdev].cyl))
        {
		return(biosdisk(2,dev,head,cyl,sect,cnt,buff));
        }
	return(E_DISCGENL);
	}

long Rw_rel_sector(int mode,void huge *buff,int cnt,ulong recnr,int dev)
	{
   /*DISK_PACK dp;
   unsigned int al_reg;
   void far* m=(void far *)&dp;*/
	if (mode == MD_READ)
		return(absread(dev,cnt,recnr,buff));
       /* {
        dp.sectnum = recnr;               Don't know why this asm code doesn't work
        dp.sectcnt = cnt;                 weird things are happening
        dp.addr = buff;

        _asm{
          mov ax,WORD PTR m+2
          mov ds,ax
          mov bx,WORD PTR m
          mov al,BYTE PTR dev
          mov cx,0xFFFF
          int 0x025
          pushf
          pop bx
          add sp,2
          mov BYTE PTR al_reg,al
          push bx
          popf
          jnc nocarry1
          }
          return (E_DISCREAD);
        nocarry1:
        return(0L);
        }*/

	if (mode == MD_WRITE)
		return(abswrite(dev,cnt,recnr,buff));
      /*{
        dp.sectnum = recnr;
        dp.sectcnt = cnt;
        dp.addr = buff;

        _asm{
          mov ax,WORD PTR m+2
          mov ds,ax
          mov bx,WORD PTR m
          mov al,BYTE PTR dev
          mov cx,0xFFFF
          int 0x26
          pushf
          pop bx
          add sp,2
          mov BYTE PTR al_reg,al
          push bx
          popf
          jnc nocarry2
          }
          return (E_DISCWRIT);
        nocarry2:
        return(0L);
        }*/

	return(E_DISCGENL);
	}

BPB* Getbpb(int dev)
	{
	BOOT_SECTOR b;
	int res;
	word numcl;
	struct dfree cfree;

	res = (int)Rw_rel_sector(MD_READ,&b,1,0L,dev);
	if (res == 0)
	   {
	   XBPB.recsiz = b.b_bps;
	   XBPB.clsiz  = b.b_spc;
	   XBPB.clsizb = XBPB.recsiz*XBPB.clsiz;
	   XBPB.rdlen  = b.b_dir*32/XBPB.recsiz;
	   XBPB.fsiz   = b.b_spf;
	   XBPB.fatrec = 1;  /* Atari always used the second fat */
      XBPB.nfat   = b.b_fats;
	   XBPB.datrec = XBPB.fatrec+(XBPB.fsiz*b.b_fats)+XBPB.rdlen;
	   if (dev <2)
		  numcl = (b.b_sec-XBPB.datrec)/XBPB.clsiz;
	   else
		  {
		  getdfree(dev+1,&cfree);
		  numcl = cfree.df_total;
		  }
	   XBPB.numcl  = numcl;
	   XBPB.bflags = (numcl <= 0x0fef)?0:1;
	   return(&XBPB);
	   }
	return(NULL);
	}

word Drvmap(void)
	{
	int ad,i;
	word drvmap=0,maxdrv;

	ad = _getdrive();
	_dos_setdrive((unsigned)ad,&maxdrv);
	for (i=0;i<maxdrv;i++)
		drvmap |= 1<<i;
	return(drvmap);
	}

int main(int argc,char *argv[])
	{
	char *s,drive_char=-1,opt;
	int drive=-1,res,mode=0;
	int (*func)(int,int)=NULL;

    init();
	while (--argc > 0)
		{
		if ((opt=**++argv) == '-' || opt == '/')
			{
			for (s = argv[0]+1; *s != '\0'; s++)
				{
				switch(*s)
					{
					case 's':               /* -s: show something */
						switch (*++s)
							{
							case 'r':       /* r - root sector */
								func = show_root;
								break;
							case 'b':       /* b - boot sector */
								func = show_boot;
                        mode = 0;
								break;
							case 'd':       /* d - show root dir */
								func = scan_fdb;
								mode |= MD_SHOWDIR;
								break;
							case 'f':       /* d - show fat */
								func = show_fat;
								mode |= MD_SHOWFAT;
								break;
							default:
								func = NULL;
								break;
							}
						break;
					case 'c':      /* check something */
						{
						func = scan_fdb;
						switch (*++s)
							{
							case 'f':
								mode |= MD_CHECKFAT;
								break;
							case 'd':
								mode |= MD_CHECKDIR;
								break;
							}
						}
						break;
					case 'w':               /* write something */
               case 'f':               /* compatibility to CHKDSK */
                  if (mode == 0) mode = MD_CHECKFAT;
						mode |= MD_WRITECHG;
						break;
					case 'b':               /* -b: backup something */
                        func = backup;
                        break;
					case 'r':               /* -r: restore something */
                        func = restore;
                        break;
                    case 'o':               /* -o options */
						switch (*++s)
							{
							case 'd':       /* device */
								DRV += atoi(++s);
								break;
                            }
                        break;
					default:
						fprintf(stdout,"Usage: FSCK [DRIVE:] [OPTIONS]\n");
						fprintf(stdout," OPTION: -sr show root sector info\n");
						fprintf(stdout,"         -sb show boot sector info\n");
						fprintf(stdout,"         -sd show dir info\n");
						fprintf(stdout,"         -sf show fat info\n");
						fprintf(stdout,"         -cd check directories\n");
						fprintf(stdout,"         -cf check fat\n");
						fprintf(stdout,"         -b backup MBR&BOOT sector\n");
						fprintf(stdout,"         -r restore MBR&BOOT sector\n");
						return(0);
					} /* end switch */
				} /* end for */
			}
		else
			{
			if (*(*argv+1) == ':')
				{
				drive_char = toupper(**argv);
				if (drive_char < 'A' || drive_char > 'Z')
					{
					fprintf(stderr,"Drive name must be between A and Z \n");
					return(1);
					}
				}
			else
				{
				fprintf(stderr,"Drive name must look like this: A:,B:,C: ...\n");
				return(1);
				}
			}
		} /* end while */

	if (drive_char < 0)
		drive = getdisk();
	else
		drive = drive_char-'A';
	if (func == NULL)
		func = scan_fdb;
	if (mode == 0)
		mode = MD_CHECKFAT;
	if ((drive > -1) && func != NULL)
		{
		if ((1<<drive) & Drvmap())    /* test if drive is known */
			{
			if ((res = read_boot(drive,(byte *)&boot)) >= 0)
				{
				fprintf(stdout,"FSCK %s\n",VERSION);
				res = (func)(drive,mode); /* perform function with drive */
				}
			if (res != 0)
				{
				if (res < 0)
					{
					if (res > -16)
						print_ferr(res,0,"");
					if (res > -24)
						print_derr(res,"");
					if (res > -32)
						print_discerr(res);
					}
                else
                    fprintf(stderr,"DOS Error %02x\n",res);
				}
			}
		else
			{
			fprintf(stderr,"Drive not known to BIOS!\n");
			return(1);
			}
		}
	return(res);
	}

int read_boot(int drv,byte *b)
	{
	int res,i;

	res = (int)Rw_rel_sector(MD_READ,b,1,0L,drv);
	if (res < 0)
		return(E_DISCREAD);
	i = 0;
	while (i<SECT_SIZE)
		{
		if (*((char *)b+i) != 0)
		   break;
		i++;
		}
	if (i == SECT_SIZE)
		{
		fprintf(stderr,"Seems to me drive %c: is not formatted!\n",'A'+drv);
		return(E_DISCNOFS);
		}
	return(0);
	}

int show_root(int dev,int mode)
    {
    return (show_root_i(dev,mode,NULL));
    }

int show_root_i(int dev,int mode, PART_SEC *ps)
	{
	ROOT_SECTOR root,*r;
	PART_INFO *p;
    PART_SEC ext_ps;
	int res,drv,i,nr;
	long used;
    unsigned long lsector;

    memset(&ext_ps,0,sizeof(PART_SEC));
	drv = DRV;
	r = &root;
    if (mode)
        {
        lsector = ps->next+ps->parent;
        }
    else
        {
        lsector = 0L;
        }
	res = (int)Rw_abs_sector(MD_READ,r,1,lsector,drv);
	i = 0;
	while (i<SECT_SIZE)
		{
		if (*((char *)r+i) != 0)
		   break;
		i++;
		}
	if (i == SECT_SIZE)
	   {
	   fprintf(stderr,"Seems to me the disk is not formatted!");
	   return(E_DISCNOFS);
	   }

	if (res<0)
		return(res);
    if (!mode)
        {
	    fprintf(stdout,"Partition info for drive: %2x\n",drv);
        }
	p = &(root.part_0);
	used = 0L;
    nr = 0;
	for (i=0;i<4;i++)
		{
        if (p->id)
            {
            nr=i+1;
            if (p->id == P_ID_EXT)
                {
                if (!mode)
                    {
                    ext_ps.parent = p->start_sec;
                    ext_ps.cnt = 1;
                    }
                else
                    {
                    ext_ps.parent = ps->parent;
                    ext_ps.cnt = ps->cnt+1;
                    ext_ps.next = p->start_sec;
                    }
                nr = 4 + ext_ps.cnt;
                }
		    fprintf(stdout,"%2d Flags:$%02x ",nr,p->flags);
		    fprintf(stdout,"ID:%02x ",p->id);
		    fprintf(stdout,"Start:%8ld,$%8lx ",p->start_sec,p->start_sec);
		    fprintf(stdout,"Size:%8ld,$%8lx ",p->size,p->size);
		    fprintf(stdout,"(%6.1f MB)\n",(float)p->size/2048);
		    used += p->size;
		    p++;
            }
		}
    if (ext_ps.next || ext_ps.parent)
        {
        res = show_root_i(drv,1,&ext_ps);
        }
	return(res);
	}

int show_boot(int drv,int mode)
	{
	BOOT_SECTOR *b;
	long unused;

	b = &boot;
	fprintf(stdout,"Drive information for drive: %c\n",'A'+drv);
	fprintf(stdout,"Serial number    : %01x%01x%01x\n",b->b_serial[0],b->b_serial[1],b->b_serial[2]);
	fprintf(stdout,"Bytes per sector : %-4d,$%-04x\n",b->b_bps,b->b_bps);
	fprintf(stdout,"Sectors/cluster  : %-4d,$%-02x\n",b->b_spc,b->b_spc);
	fprintf(stdout,"# of res. sectors: %-4d,$%-04x\n",b->b_res,b->b_res);
	fprintf(stdout,"# of Fats        : %-4d,$%-02x\n",b->b_fats,b->b_fats);
	fprintf(stdout,"# of rootdirentr.: %-4d,$%-04x\n",b->b_dir,b->b_dir);
	fprintf(stdout,"Sectors on disc  : %-5d,$%-04x\n",b->b_sec,b->b_sec);
	fprintf(stdout,"Media byte       : %-4d,$%01x\n",b->b_media,b->b_media);
	fprintf(stdout,"Sectors/Fat      : %-4d,$%-04x\n",b->b_spf,b->b_spf);
	fprintf(stdout,"Sectors/track    : %-4d,$%-04x\n",b->b_spt,b->b_spt);
	fprintf(stdout,"# of heads       : %-4d,$%-04x\n",b->b_heads,b->b_heads);
	fprintf(stdout,"# of hidden sec. : %-4d,$%-04x\n",b->b_hidden,b->b_hidden);
	fprintf(stdout,"Chksum field     : $%04x\n",b->b_chksum);
	unused = (long)b->b_sec - b->b_res -
		b->b_hidden - (b->b_spf*2) - (b->b_dir/16);
	if (unused > 0)
		fprintf(stdout,"The total net capacity on this drive is %ld Bytes (%lf MB)\n",
			unused*SECT_SIZE,(float)unused/2048);
	return(0);
	}


char *ferr_msg[]={
	"Error: Fat entry %04x out of range marking %s\n",
	"Error: Invalid fat entry %04x marking %s\n",
	"Warning: Bad blocks (%04x) encountered marking %s\n",
	"Error: Cross linked cluster (%04x) marking %s\n",
	"Warning: Cluster %04x is lost!%s\n",
	"",
	"Warning: too much clusters assigned to (%04x) %s\n",
	"Error: too less clusters assigned to (%04x) %s\n",
	"Error: cluster (%04x) used by file but is not marked used %s\n"
	};

void print_ferr(int err,word cluster,char *fn)
	{
	fprintf(stderr,ferr_msg[abs(err)-1],cluster,fn);
	}

char *derr_msg[]={
	"Warning: No clusters assigned to %s\n",
	"Warning: Bad character(s) in name %s\n",
	"Error: Illegal flag byte for %s\n",
	"Warning: Reserved area not empty for %s\n"};

void print_derr(int err,char *fn)
	{
	fprintf(stderr,derr_msg[abs(err)-16],fn);
	}

char *discerr_msg[]={
	"Error reading from drive.\n",
	"Error writing to drive.\n",
	"Error: general error accessing drive.\n",
	"Error: no valid DOS filesystem (bad/corrupted/not initialized).\n"};

void print_discerr(int err)
	{
	fprintf(stderr,discerr_msg[abs(err)-24]);
	}

int isValid(byte c)
	{
	return(c >= 0x1f && c < 0xe5 && c!= '.');
	}

int isValidFnChar(char c)
	{
	return(isalnum(c)||
		c== ' ' || c== '!' || c== '#' || c== '$' || c== '%' ||
		c== '^' || c== '&' || c== '(' || c== ')' || c== '-' ||
		c== '_' || c== '{' || c== '}' || c== '~');
	}

int check_dir_entry(DIR_ENTRY *d,int mode)
	{
	int i;

	if (im((byte *)&d->d_fclust) == 0)
		return(ED_NOCLSTS);
	i = 11;
	while (i--)
		if (!isValidFnChar(d->d_fname[i]))
			return(ED_BADCHAR);
	i = 10;
	while (i--)
		if (d->d_res[i])
			return(ED_BADRSVD);
	if (d->d_attr & 0xc0)
		return(ED_ILLFLAG);
	return(0);
	}

long init_fat_table(int drv,int mode,FAT_TABLE *ft)
	{
	long fs,gs;
	BPB *bpb;

	MERROR = 0;
	memset(ft,0,sizeof(FAT_TABLE));
	bpb = Getbpb(drv);
	if (bpb == NULL)
		return(E_DISCREAD);
	ft->fat_type = bpb->bflags;
	ft->bpb = bpb;
	fs = (ft->fat_type == 0)?3:4;
	gs = ((long)bpb->fsiz*bpb->recsiz)*2/fs;
	ft->csize=bpb->numcl;
	ft->tsize=ft->csize+FAT_OFFSET; /* two entries in fat are used but not counted */
	if (ft->tsize>(long)gs)
		{
		gs = MAX(gs,(long)ft->tsize);
		printf("WARNING: BPB->NUMCL(%u) > FAT-SPACE(%f) ON DISK!!!!\n",ft->tsize,gs);
		}
	return(gs);
	}

int show_fat(int drv,int mode)
	{
	FAT_TABLE Ft,*ft;
	word l,maxc,i,x;
	int res;
	long gs;
   double cl_per_char;
	char *ct,c;

	ft = &Ft;
	gs = init_fat_table(drv,mode,ft);
	if (gs < 0)
		return((int)gs);
	ct = malloc(1000);
	if (!ct)
		return(ENOMEM);
	if (mode & MD_SHOWFAT)
		{
		ft->flat_fat = malloc((long)gs*2+4);
		if (!ft->flat_fat)
			{
			return(ENOMEM);
			}
		ft->check_fat = malloc((long)gs+2);
		if (!ft->check_fat)
			{
			free(ft->flat_fat);
			return(ENOMEM);
			}
		memcpy(ft->flat_fat+ft->tsize,"ZAZB",4);
		memcpy(ft->check_fat+ft->tsize,"ZB",2);
		memset(ft->flat_fat,0,ft->tsize*2+2);
		memset(ft->check_fat,0,ft->tsize+1);
		}
	res = get_flat_fat(drv,ft);
	if (res == 0)
		{
		maxc = ft->csize;
		if (maxc <= 1000)
			cl_per_char = 1;
		else
			cl_per_char = maxc/1000;
		memset(ct,0,1000);
		for (i=0;i<maxc;i++)
			{
			x = *(ft->flat_fat+i+FAT_OFFSET);
			if (x > 0)
				if (x > 0xffef && x < 0xfff8)
					*(ct+(word)((double)i/cl_per_char)) = -1;
				else
					(*(ct+(word)((double)i/cl_per_char)))++;
			}
		for (i=0;i<MIN(maxc,1000);i++)
			{
			c = *(ct+i);
			if (c == 0)
				*(ct+i) = '.';
			else
				if (c < 0)
					*(ct+i) = 'X';
				else
					*(ct+i) = '0'+MIN((c*10/cl_per_char),9);
			}
		l = 0;
		fprintf(stdout,"Graphical FAT map for drive %c:\n",'A'+drv);
		for (i=0;i<12 && l<maxc;i++)
			{
			fprintf(stdout,"\n ");
			for (x=0;x<75 && l<maxc;x++)
				{
				fputc(*(ct+l++),stdout);
				}
			}
		fprintf(stdout,"\n\nLegend: '.' = Free cluster(s), 0-9 = Occupied cluster(s), X = Bad cluster(s)\n");
		fprintf(stdout,"        One character represents %.1lf cluster(s)\n",cl_per_char);
		fprintf(stdout,"        Numbers 0-9 represent grade of filling\n");
		fprintf(stdout,"        Total clusters on drive %ld\n",(long)maxc);
		}
	return(0);
	}

int get_flat_fat(int drv,FAT_TABLE *ft)
	{
	byte huge *fat,huge *tmp,t;
	word huge *fe,huge *ff,w,v,entries;
	long res;
	BPB *bpb;

	bpb = ft->bpb;
	if (CPU == 0 || ft->fat_type == 0)
		fat = malloc(bpb->fsiz*bpb->recsiz);
	else
		fat = (byte huge *)ft->flat_fat;
	if (!fat)
		return(ENOMEM);
	if ((ulong)bpb->fsiz*bpb->recsiz > 0x10000L)
		{
		tmp = fat;
		w = (word) (0x10000L / bpb->recsiz);
		res = Rw_rel_sector(MD_READ,tmp,w,(ulong)bpb->fatrec,drv);
		tmp += 0x10000L;
		v = bpb->fsiz - w;
		res |= Rw_rel_sector(MD_READ,tmp,v,(ulong)bpb->fatrec+w,drv);
		}
	else
		res = Rw_rel_sector(MD_READ,fat,bpb->fsiz,(ulong)bpb->fatrec,drv);
	if (res < 0)
		{
		free(fat);
		return(E_DISCREAD);
		}
	entries = ft->tsize;
	switch (ft->fat_type)
		{
		case 0:
			tmp = (byte huge *)fat;
			ff = ft->flat_fat;
			t = 1;
			while (entries--)
				{
				if (t)
					{
					w = *tmp;
					w |= (*(tmp+1)&0x0f)<<8;
					tmp++;
					}
				else
					{
					w = *tmp>>4;
					w |= *(tmp+1)<<4;
					tmp +=2;
					}
				t ^=1;
				if (w > 0x0FEF)		/* make 12 and 16 bit fats compatible */
					w |= 0xf000;
				*ff++ = w;
				}
			break;
		case 1:
			if (CPU == 0)
				{
				fe = (word *)fat;
				ff = ft->flat_fat;
				while (entries--)
					{
					*ff++ = im((byte *)fe);
					fe++;
					}
				}
			break;
		}
	if (CPU == 0)
		free(fat);
	return(0);
	}

int make_disk_fat(int drv,FAT_TABLE *ft,int mode)
	{
	byte huge *fat,huge *tmp;
	word huge *fe,huge *ff,w,v,fat_bytes;
	long res,entries,s_sec;
	BPB *bpb;

	bpb = ft->bpb;
	fat_bytes = bpb->fsiz*bpb->recsiz;
	if (CPU == 0 || ft->fat_type == 0)
		{
		fat = malloc(fat_bytes);
		memset(fat,0,fat_bytes);
		}
	else
		fat = (byte huge *)ft->flat_fat;
	if (!fat)
		return(ENOMEM);
	entries = ft->tsize;
	switch (ft->fat_type)
		{
		case 0:
			tmp = (byte huge *)fat;
			ff = ft->flat_fat;
			while (entries>0)
				{
				entries--;
				entries--;
				w = *ff++;
				v = *ff++;
				*tmp++ = (byte)(w & 0xff);
				*tmp++ = (byte)(((w>>8)&0xf)|((v&0xf)<<4));
				*tmp++ = (byte)((v>>4)&0xff);
				}
			break;
		case 1:
			if (CPU == 0)
				{
				fe = (word *)fat;
				ff = ft->flat_fat;
				while (entries--)
					{
					*fe++ = im((byte *)ff++);
					}
				}
			break;
		}
	res = 0;
	if (mode & MD_WRITECHG)
		{
      int i;
      s_sec = (ulong)bpb->fatrec;
      for (i=0;i<(bpb->nfat);i++)
         {
		   if ((ulong)bpb->fsiz*bpb->recsiz > 0x10000L)
			   {
			   tmp = fat;
			   w = (word) (0x10000L / bpb->recsiz);
            res = Rw_rel_sector(MD_WRITE,tmp,w,s_sec,drv);
			   tmp += 0x10000L;
			   v = bpb->fsiz - w;
			   res |= Rw_rel_sector(MD_WRITE,tmp,v,s_sec+w,drv);
			   }
		   else
			   {
			   res = Rw_rel_sector(MD_WRITE,fat,bpb->fsiz,s_sec,drv);
			   }
         s_sec += bpb->fsiz;
         }
		}
	if (CPU == 0 || ft->fat_type == 0)
		free(fat);
	return((int) res);
	}

int mark_clusters(word scluster,FAT_TABLE *ft,ulong *fsize)
	{
	word cluster,v,fclust;
	long slack;

	cluster = scluster;
	if (cluster == 0x0000)
	   return(EF_NOCLSTS);
	fclust = 0;
	while (cluster <0xfff8)
		{
		if (cluster > (ft->csize+1))
			return(EF_OUTSIDE);                /* link pointer outsize fat */
		v = *(ft->flat_fat+cluster);
		if (v == 0x0001)
			return(EF_INVALID);
		if (v >= 0xfff0 && v <0xfff8)
			return(EF_BADBLCK);                /* enounterd bad blocks*/
		if ((*(ft->check_fat+cluster))++ > 0)  /* increment usage */
			{
			/* printf("%04x %04x\n",*(ft->check_fat+cluster),cluster); */
			return(EF_CRSSLNK);                /* cross linked cluster*/
			}
		cluster = v;
		fclust++;
		}
	if (*fsize > 0L)
		{
		slack = (ulong)fclust*ft->bpb->clsizb - *fsize;    /* Test if the right # */
		if (slack < 0)
			{
			*fsize = (ulong)fclust*ft->bpb->clsizb;
			return(EF_WRGSIZL);
			}
		else
			if (slack > ft->bpb->clsizb)                 /* of clusters is assigned */
				{
				*fsize = (ulong)fclust*ft->bpb->clsizb;
				return(EF_WRGSIZG);
				}
		}
	return(0);
	}

word get_lost_clusters(FAT_TABLE *ft,int mode)
	{
	word huge *o,l,e;
	byte *c;

	o = ft->flat_fat;
	c = ft->check_fat;
	l = 0;
	c +=FAT_OFFSET;                   /* don't check cluster zero and one*/
	o +=FAT_OFFSET;
	e = FAT_OFFSET;
	while (e<ft->tsize)
		 {
		 if (*o >= 0xfff0 && *o <0xfff8)
			 {
			 ft->bad_cl++;
			 (*(ft->check_fat+e))++;    /* also mark bad clusters */
			 }
		 else
			 if (*o == 0)
				 ft->unused_cl++;
			 else
				 ft->used_cl++;
		 if (*o > 0 && *c == 0)        /* lost cluster */
			 {
			 l++;
			 print_ferr(EF_LOSTCST,e,"");
			 if (mode & MD_WRITECHG)
				 *o=0;
			 }
		 if (*o == 0 && *c > 0)       /* cluster used by file but not marked in FAT */
			 {
			 print_ferr(EF_USEDFRE,e,"");
			 if (mode & MD_WRITECHG)
				 *o = 0xFFFF;
			 }
		 e++;
		 c++;
		 o++;
		 }
	return(l);
	}

int scan_dir(int drv,byte *b,word dcluster,int mode,FAT_TABLE *ft)
	{
	DIR_ENTRY *d;
	char string[13];
	byte *bb,dirty;
	int s_size,res=0,mres,goon=1,count=0;
	word cluster;
	ulong sector;
	ulong fsize;
	BPB *bpb;

	bpb = ft->bpb;
	cluster = dcluster;
	s_size = bpb->recsiz;
	string [8] = '.';
	string [12] = '\0';

	if ((mode & MD_CHECKFAT) && cluster != 0)
		{
		fsize = 0;
		mres = mark_clusters(cluster,ft,&fsize);
		if (mres < 0)
			print_ferr(mres,cluster,"directory");
		}
	while (goon)
		{
		if (cluster == 0)    /* hacky, but rootdir is not in cluster list */
			{
			sector = bpb->datrec-bpb->rdlen+count;
			count++;
			d = (DIR_ENTRY *)b;
			if (count >= bpb->rdlen)
				break;
			}
		else
			{
			sector = (ulong)(cluster - FAT_OFFSET) * bpb->clsiz + bpb->datrec + count;
			count++;
			d = (DIR_ENTRY *)b;
			}
		res = (int)Rw_rel_sector(MD_READ,b,1,sector,drv);
		dirty = 0;
		if (res < 0)
		   return(E_DISCREAD);
	    while ((byte *)d-b < s_size)
	        {
	        /* don't care about back-links and deleted files */
	        if (isValid(d->d_fname[0]) && !(d->d_attr & FA_LABEL))
	            {
                fsize = (ulong)im((byte *)&(d->d_size))+((ulong)im((byte*)&(d->d_size)+2)<<16);
                if (d->d_attr & FA_DIREC)
                    ft->d_index++;
                else
                    {
	                ft->f_index++;
                    if (d->d_attr & FA_HIDDEN)
                        ft->h_index++;
	                ft->f_size += fsize;
                    }
		        if (mode & MD_SHOWDIR)
		            {
		            strncpy(string,d->d_fname,8);
		            strncpy(&(string[9]),&(d->d_fname[8]),3);
		            printf("%s :%7ld\n",string,fsize);
		            }
		        if ((mode & MD_CHECKFAT) && !(d->d_attr & FA_DIREC))
		            {
		            fprintf(stdout,"Checking cluster %04x\r",im((byte *)&d->d_fclust));
                    mres = mark_clusters(im((byte *)&d->d_fclust),ft,&fsize);
                    if (mres < 0)
						{
                        strncpy(string,d->d_fname,8);
                        strncpy(&(string[9]),&(d->d_fname[8]),3);
                        print_ferr(mres,im((byte *)&d->d_fclust),string);
                        if (mode & MD_WRITECHG)
                            {
                            siml((byte*)&d->d_size,fsize);    /* store long in intel format */
                            dirty = 1;
                            }
                        }
		            }
		        if (mode & MD_CHECKDIR)
                    {
                    mres = check_dir_entry(d,mode);
                    if (mres <0)
                        {
                        strncpy(string,d->d_fname,8);
                        strncpy(&(string[9]),&(d->d_fname[8]),3);
                        print_derr(mres,string);
                        }
                    }
		        if (d->d_attr & FA_DIREC)
                    {
                    bb = malloc(bpb->recsiz);
                    if (bb)
                        {
                        scan_dir(drv,bb,im((byte *)&d->d_fclust),mode,ft);
						free(bb);
                        }
                    else
                        {
                        MERROR=1;
                        goon=0;
                        }
                    }
		        }
		    d++;
	        }
	    if (dirty && (mode & MD_WRITECHG))
	        res = (int)Rw_rel_sector(MD_WRITE,b,1,sector,drv);
        if (cluster != 0 && count >= bpb->clsiz)
            {
            count = 0;
            if (mode & MD_CHECKFAT)
                cluster = *(ft->flat_fat+cluster);
            else
                goon = 0;
            if (cluster >= 0xfff8)
               goon = 0;
            }
	    }
    fprintf(stdout,"                     \r");
    return(res);
    }

int scan_rdir(int drv,int mode,FAT_TABLE *ft)
    {
    byte *b;
    int res=0;

    b = malloc(ft->bpb->recsiz);
    if (!b)
        return(ENOMEM);
    if (mode & MD_CHECKFAT)
        res = get_flat_fat(drv,ft);
    if (res == 0)
        {
        res = scan_dir(drv,b,0,mode,ft);
        }
    free(b);
    return(res);
    }

int cmp_fats(int drv,int mode,FAT_TABLE *ft)
    {
    byte *b1,*b2;
    word f,numfats,cmp;
    ulong fatsector;
    int res;
    long tocmp;

	b1 = malloc(ft->bpb->recsiz);
    if (!b1)
        return(ENOMEM);
    b2 = malloc(ft->bpb->recsiz);
    if (!b2)
        {
        free(b1);
        return(ENOMEM);
        }
    numfats = boot.b_fats;
    cmp = 0;
    res = 0;
    if (numfats >= 2)
        {
        tocmp = (long)((double)ft->tsize*((ft->bpb->bflags==0)?1.5:2.0));
        fatsector = ft->bpb->fatrec;
        if (fatsector == 1)               /* MSDOS uses first FAT */
            fatsector += ft->bpb->fsiz;
        for (f=0;f<ft->bpb->fsiz;f++)
            {
            res |= (int)Rw_rel_sector(MD_READ,b1,1,fatsector+f,drv);
			res |= (int)Rw_rel_sector(MD_READ,b2,1,fatsector+f-ft->bpb->fsiz,drv);
            if (res)
                break;
            else
				if (memcmp(b1,b2,(int)MIN((long)ft->bpb->recsiz,tocmp)) == 0)
                    {
					tocmp -= ft->bpb->recsiz;
                    cmp++;
                    }
            }
        res = (cmp==ft->bpb->fsiz)?0:2;
        }
    else
        res = 1;
	free(b1);
    free(b2);
    return((int)res);
	}

int scan_fdb(int drv,int mode)
    {
    FAT_TABLE Ft,*ft;
    BPB *bpb;
    int res;
    word l;
    double gs,fs;

    ft = &Ft;
    gs = init_fat_table(drv,mode,ft);
    bpb = ft->bpb;
    if (mode & MD_CHECKFAT)
        {
        ft->flat_fat = malloc((long)gs*2+4);
		if (!ft->flat_fat)
            {
            return(ENOMEM);
            }
        ft->check_fat = malloc((long)gs+2);
        if (!ft->check_fat)
            {
            free(ft->flat_fat);
            return(ENOMEM);
            }
        memcpy(ft->flat_fat+ft->tsize,"ZAZB",4);
        memcpy(ft->check_fat+ft->tsize,"ZB",2);
        memset(ft->flat_fat,0,ft->tsize*2+2);
        memset(ft->check_fat,0,ft->tsize+1);
        }
    res = cmp_fats(drv,mode,ft);
    if (res < 0)
        {
        fprintf(stderr,"Error: Unable to read BOOT/FAT sectors!\n");
        fprintf(stderr,"Error: Unable to proceed with test!\n");
        return(res);
        }
    if (res > 1)
        return(res);
    if (res == 1)
        fprintf(stderr,"Warning: Fats are not in sync!\n");
    scan_rdir(drv,mode,ft);
    fprintf(stdout,"File system check for drive %c:\n",'A'+drv);
    fprintf(stdout," %11ld bytes (%.2lf MB) total capacity\n",(ulong)ft->csize*bpb->clsizb,(ulong)ft->csize*bpb->clsizb/1048576.0);
	fprintf(stdout," %11ld bytes (%.2lf MB) used by files\n",ft->f_size,(double)ft->f_size/1048576L);
    fprintf(stdout," %11ld files\n",ft->f_index);
    fprintf(stdout," %11ld files of the above are hidden.\n",ft->h_index);
    fprintf(stdout," %11ld directories\n",ft->d_index);

    if (mode & MD_CHECKFAT)
        if (!MERROR)
            {
            l = get_lost_clusters(ft,mode);
            fprintf(stdout," %11ld clusters occupying\n",ft->used_cl);
            fs = ft->f_size;
            gs = ft->used_cl*bpb->clsizb;
            if (gs != 0)
                fs = (gs-fs)*100/gs;
            fprintf(stdout," %11.0lf bytes (=%.2f%% slack).\n",gs,fs);
            fprintf(stdout," %11ld bad clusters occupying\n",ft->bad_cl);
            fprintf(stdout," %11ld bytes.\n\n",ft->bad_cl*bpb->clsizb);
            fprintf(stdout," %11ld free clusters leaving\n",ft->unused_cl);
            fprintf(stdout," %11ld bytes on this drive (%.2lf MB).\n\n",ft->unused_cl*bpb->clsizb,
                (double)ft->unused_cl*bpb->clsizb/1048576L);
            if (l>0)
                {
                fprintf(stdout,"%d lost clusters found!\n",l);
                if (mode & MD_WRITECHG)
                    {
                    if (make_disk_fat(drv,ft,mode) != 0)
                        fprintf(stderr,"Could not write new FAT!\n");
                    else
						      fprintf(stdout,"Successfully wrote changes.\n");
                    }
                else
                    fprintf(stdout,"Option -f not given. Changes will not be written.\n");
                }
            }
        else
            {
            fprintf(stderr,"Could not allocate enough memory to perform all tests.\n");
            fprintf(stderr,"Free up memory and try again!\n");
            if (mode & MD_CHECKFAT)
                {
                free(ft->check_fat);
                free(ft->flat_fat);
                }
            return(ENOMEM);
            }
    if (mode & MD_CHECKFAT)
        {
        free(ft->check_fat);
        free(ft->flat_fat);
        }
    return(0);
	}

int backup(int drv,int mode)
    {
    FILE *fp;
	ROOT_SECTOR root,*r;
    BOOT_SECTOR boot,*b;
    int res;

	r = &root;
	res = (int)Rw_abs_sector(MD_READ,r,1,0,0x80);
	if (res<0)
		return(res);
    b = &boot;
	res = (int)Rw_rel_sector(MD_READ,b,1,0L,drv);
	if (res < 0)
		return(E_DISCREAD);
    fp = fopen("fsck_bak.bin","w+b");
    if (!fp)
        {
        fprintf(stderr,"Error creating backup file (fsck_bak.bin)\n");
        }
    res = fwrite(&root,sizeof(ROOT_SECTOR),1,fp);
    if (res != 1)
        {
        fprintf(stderr,"Error writing backup file (fsck_bak.bin)\n");
        fclose(fp);
        }
    res = fwrite(&boot,sizeof(BOOT_SECTOR),1,fp);
    if (res != 1)
        {
        fprintf(stderr,"Error writing backup file (fsck_bak.bin)\n");
        fclose(fp);
        }
    fclose(fp);
    return (0);
    }

int restore(int drv,int mode)
    {
    FILE *fp;
	ROOT_SECTOR root,*r;
    BOOT_SECTOR boot,*b;
    int res=0;
    char c;

    fprintf(stdout,"Writing a new MBR&BOOT sector may leave your hard disk\n");
    fprintf(stdout,"in an inaccessible state! Are you sure to do this?\n");

    c = getch();
    if (c == 'y' || c == 'Y')
        {
        fp = fopen("fsck_bak.bin","r+b");
        if (!fp)
            fprintf(stderr,"Error reading backup file (fsck_bak.bin)\n");
    	r = &root;
        res = fread(&root,sizeof(ROOT_SECTOR),1,fp);
        if (res != 1)
            {
            fprintf(stderr,"Error reading backup file (fsck_bak.bin)\n");
            fclose(fp);
            }
        b = &boot;
        res = fread(&boot,sizeof(BOOT_SECTOR),1,fp);
        if (res != 1)
            {
            fprintf(stderr,"Error reading backup file (fsck_bak.bin)\n");
            fclose(fp);
            }
        fclose(fp);
	    res = (int)Rw_abs_sector(MD_WRITE,r,1,0,0x80);
    	if (res<0)
	    	return(res);
    	res = (int)Rw_rel_sector(MD_WRITE,b,1,0L,drv);
    	if (res < 0)
    		return(E_DISCWRIT);
        fprintf(stdout,"Successfully wrote sectors. You should reboot now!\n");
        }
    else
        fprintf(stdout,"Abort.\n");
    return (0);
    }
