/* CTmouse wheel api docs:
---------------------------------------------------------------------

API version 1.0

Summary:

This document describes an extension to the commonly used INT 33h Mouse
API to add wheel (Z axis) support. This draft introduces extra functions
and additions to the standard INT 33 API. These new and changed functions
are mentioned in the technote.txt as the WheelAPI.

---------------------------------------------------------------------
New functions:

INT 33/0011 - Check wheel support and get capabilities flags
	AX = 0011h
Return:	AX = 574Dh ('WM' in assembly) if Wheel API is supported by driver
	BX = Capabilities flag (all bits reserved)
	CX = Capabilities flag
		Bit(s)	Description
		------	-----------
		 0	1=Pointing device supports wheel
		 1-15	Reserved
Note:	this function should be examined before accessing wheel features

---------------------------------------------------------------------
Changes in the original INT 33h functions:

INT 33/0000 - Reset driver and read status
Note:	this call clears the wheel movement counter

INT 33/0003 - Get cursor position, buttons status and wheel counter
	AX = 0003h
Return:	BL = buttons status
	BH = 8-bit signed counter of wheel movement since last call
	CX = column
	DX = row
Notes:	returned wheel counter contains all wheel movements accumulated since
	  the last call to INT 33/AX=0003h, INT 33/AX=0005h/BX=-1 or
	  INT 33/AX=0006h/BX=-1
	positive value of wheel counter means downward wheel movement
	this call clears the wheel movement counter

INT 33/0005 - Get button press or wheel movement data
	AX = 0005h
	BX = button number or -1 for wheel
Return:	AL = state of buttons
	AH = 8-bit signed counter of wheel movement
	---button info---
	BX = number of times specified button has been pressed since last call
	CX = column where specified button was last pressed
	DX = row where specified button was last pressed
	---wheel info---
	BX = 16-bit signed counter of wheel movement since last call
	CX = column where wheel was last moved
	DX = row where wheel was last moved
Notes:	returned wheel counters contain all wheel movements accumulated since
	  the last call to INT 33/AX=0003h, INT 33/AX=0005h/BX=-1 or
	  INT 33/AX=0006h/BX=-1
	positive value of wheel counter means downward wheel movement
	this call clears the wheel movement counter for BX=-1

INT 33/0006 - Get button release or wheel movement data
	AX = 0006h
	BX = button number or -1 for wheel
Return:	AL = state of buttons
	AH = 8-bit signed counter of wheel movement
	---button info---
	BX = number of times specified button has been released since last call
	CX = column where specified button was last released
	DX = row where specified button was last released
	---wheel info---
	BX = 16-bit signed counter of wheel movement since last call
	CX = column where wheel was last moved
	DX = row where wheel was last moved
Notes:	returned wheel counters contain all wheel movements accumulated since
	  the last call to INT 33/AX=0003h, INT 33/AX=0005h/BX=-1 or
	  INT 33/AX=0006h/BX=-1
	positive value of wheel counter means downward wheel movement
	this call clears the wheel movement counter for BX=-1

INT 33/000C - Define User Interrupt Routine
INT 33/0014 - Exchange User Interrupt Routines
Notes:	on entry, bit 7 of CX (call mask) indicates that the user routine
	  will be called on a wheel movement
	the user routine will be called with BH holding the 8-bit signed
	  counter of wheel movement since the last call to the routine

INT 33/0021 - Software reset
Note:	this call clears the wheel movement counter
*/

#define _GPM_USE_INTERNAL

#include <sys/farptr.h>
#include <sys/fsext.h>
#include "keyboard.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <crt0.h>
#include <dpmi.h>
#include <go32.h>
#include "gpm.h"
#include <pc.h>

#define GPM_HANDLER_QUEUE_MAX 20

#ifdef GPM_EXTREME_DEBUG
#define GPM_DEBUG
int crt0_startup_flags = _CRT0_FLAG_LOCK_MEMORY;
#endif
#ifdef GPM_DEBUG
FILE *dbgfile = NULL;
#endif

int gpm_flag = 0;
int gpm_tried = 0;
int gpm_fd = -1;
int gpm_zerobased = 0;
int gpm_visiblepointer = 1; /* Always visible in this implementation */
int gpm_hflag = 0;
int gpm_morekeys = 0;

Gpm_Handler *gpm_handler = NULL;
void        *gpm_data = NULL;

static int gpm_buttons;
static int gpm_wheel = 0;
static int gpm_available = 0;

static Gpm_Connect *gpm_fds[ 20 ];
static int          gpm_nfd = -1;
static int          gpm_handler_installed = 0;

static void gpm_mousexy_to_gpmxy( short mx, short my, short *gx, short *gy );

#ifdef GPM_DEBUG
static void eventprintf( Gpm_Event *e )
{
    fputs( "Gpm_GetEvent ", dbgfile );
    if( e->type == GPM_UP ) fputs( "returning GPM_UP event\n", dbgfile );
    else if( e->type == GPM_DOWN ) fputs( "returning GPM_DOWN event\n", dbgfile );
    else fputs( "returning an unknown event\n", dbgfile );
}
#endif

/* GPM_HANDLER section */
static _go32_dpmi_registers h_regs;
static _go32_dpmi_seginfo   h_info;
static int gpm_handler_entered = 1;
static int gpm_fd_ready = 0;
static Gpm_Event gpm_handler_queue[ GPM_HANDLER_QUEUE_MAX ];  /* We'll keep up to 20 items in queue */
static int gpm_handler_nqueue = -1;
static int gpm_queue_lbp = 0;
static Gpm_Event *gpm_curhq;
static int gpm_max_x;
static int gpm_max_y;
static int temp1, temp2;
static int end_gpm_internal_handler_data;
static int old_handler_cx;

static void gpm_shift_queue( int reading );

static void gpm_internal_handler( struct _gpm_handler_info *r )
{
    if( gpm_handler_entered ) return;
    gpm_handler_entered = 1;
#ifdef GPM_EXTREME_DEBUG
    fputs( "Gpm handler entered\n", dbgfile );
#endif
    gpm_fd_ready++;
    gpm_shift_queue( 0 );
    gpm_curhq = &gpm_handler_queue[ gpm_handler_nqueue ];

    gpm_curhq->wdx = gpm_curhq->margin = gpm_curhq->modifiers = gpm_curhq->vc = 0;
    gpm_curhq->dx = gpm_max_x;
    gpm_curhq->dy = gpm_max_y;
    gpm_mousexy_to_gpmxy( r->col, r->row, &gpm_curhq->x, &gpm_curhq->y );
    gpm_curhq->clicks = 1;

    if( r->bstate ) { /* A button was pressed */
#ifdef GPM_EXTREME_DEBUG
        fputs( "Gpm handler called with a GPM_DOWN event\n", dbgfile );
#endif
        gpm_curhq->type = GPM_DOWN;
        gpm_curhq->buttons = r->bstate;
        gpm_queue_lbp = r->bstate;
    } else { /* A button was released */
        if( !gpm_queue_lbp ) gpm_fd_ready--; /* Wait for a complete down/up event */
        else {
#ifdef GPM_EXTREME_DEBUG
            fputs( "Gpm handler called with a GPM_UP event\n", dbgfile );
#endif
            gpm_curhq->type = GPM_UP;
            gpm_curhq->buttons = gpm_queue_lbp;
            gpm_queue_lbp = 0;
        }
    }
#ifdef GPM_EXTREME_DEBUG
    fputs( "Leaving the Gpm handler\n", dbgfile );
    fprintf( dbgfile, "gpm_fd_ready is currently at %i\n", gpm_fd_ready );
#endif
    gpm_handler_entered = 0;
//    _go32_dpmi_simulate_fcall(
}

static void gpm_mousexy_to_gpmxy( short mx, short my, short *gx, short *gy )
{
    *gx = ( mx / 8 ) + 1;
    *gy = ( my / 8 ) + 1;
}

/* So we can lock our memcpy */
static void gpm_handler_memcpy( char *ptr1, char *ptr2, size_t s )
{
    for( temp1 = 0; temp1 < s; temp1++ ) ptr1[ temp1 ] = ptr2[ temp1 ];
}

static void gpm_shift_queue( int reading )
{
    if( !reading ) gpm_handler_nqueue++;
    if( gpm_handler_nqueue >= GPM_HANDLER_QUEUE_MAX || reading ) {
        for( temp2 = 0; temp2 < ( GPM_HANDLER_QUEUE_MAX - 1 ); temp2++ ) {
            gpm_handler_memcpy( ( void * )&gpm_handler_queue[ temp2 ], ( void * )&gpm_handler_queue[ temp2 + 1 ], sizeof( Gpm_Event ) );
        }
        gpm_handler_nqueue--;
    }
}

static void end_gpm_internal_handler( void ) {}

static void gpm_clear_queue( void )
{
    gpm_handler_nqueue = -1;
    memset( &gpm_handler_queue, 0, sizeof( Gpm_Event ) * 20 );
}

static int gpm_register_internal_handler( void )
{
    __dpmi_regs r;

    h_info.pm_offset = ( long )gpm_internal_handler;
    if( _go32_dpmi_lock_code( gpm_internal_handler, ( long )end_gpm_internal_handler -
                                                    ( long )gpm_internal_handler ) ) return( -1 );
    if( _go32_dpmi_lock_data( &h_regs, ( long )( &end_gpm_internal_handler_data ) -
                                       ( long )( &h_regs ) ) ) return( -1 );
    if( _go32_dpmi_allocate_real_mode_callback_retf( &h_info, &h_regs ) ) return( -1 );

    r.x.ax = 0x0014;
    r.x.cx = ~0x01;
    r.x.es = h_info.rm_segment;
    r.x.dx = h_info.rm_offset;
    __dpmi_int( 0x33, &r );
    h_info.rm_segment = r.x.es;
    h_info.rm_offset = r.x.dx;
    old_handler_cx = r.x.cx;
    gpm_handler_entered = 0;
    gpm_handler_installed = 1;

    return( 0 );
}

static void gpm_unregister_internal_handler( void )
{
    __dpmi_regs r;

    gpm_handler_entered = 1;
    r.x.ax = 0x000C;
    r.x.cx = old_handler_cx;
    r.x.es = h_info.rm_segment;
    r.x.dx = h_info.rm_offset;
    __dpmi_int( 0x33, &r );
    _go32_dpmi_free_real_mode_callback( &h_info );
}

static void gpm_set_cursor_range( void )
{
    __dpmi_regs r;
    r.x.ax = 0x0007;
    r.x.cx = 0;
    r.x.dx = gpm_max_x * 8;
    __dpmi_int( 0x33, &r );
    r.x.ax = 0x0008;
    r.x.cx = 0;
    r.x.dx = gpm_max_y * 8;
    __dpmi_int( 0x33, &r );
}

static void gpm_show_mouse( void )
{
    __dpmi_regs r;
    r.x.ax = 0x0001;
    __dpmi_int( 0x33, &r );
}

static void gpm_hide_mouse( void )
{
    __dpmi_regs r;
    r.x.ax = 0x0002;
    __dpmi_int( 0x33, &r );
}

#if 0 /* Not used yet */
static int gpm_get_shift( void )
{
    __dpmi_regs r;
    r.h.ah = 0x02;
    __dpmi_int( 0x16, &r );
    return( r.h.al );
}
#endif

static int gpm_mouse_available( void )
{
    __dpmi_raddr mouseaddr;
    __dpmi_regs  r;
    short        intbyte;

    if( gpm_tried ) return( 0 );
    if( gpm_available ) return( 1 );

    __dpmi_get_real_mode_interrupt_vector( 0x33, &mouseaddr );
    intbyte = _farpeekb( _dos_ds, ( ( long )mouseaddr.segment * 16 ) + ( long )mouseaddr.offset16 );
    if( ( !mouseaddr.segment && !mouseaddr.offset16 ) || intbyte == 0xCF ) return( 0 );

    r.x.ax = 0;
    if( __dpmi_int( 0x33, &r ) == -1 || r.x.ax != 0xFFFF ) return( 0 );
    gpm_buttons = r.x.bx;
    if( gpm_buttons == 0xFFFF ) gpm_buttons = 2;
    if( gpm_buttons == 0      ) gpm_buttons = 1;

    gpm_flag = 1;

    r.x.ax = 0x0011;
    __dpmi_int( 0x33, &r );
    if( r.x.ax == 0x574D ) gpm_wheel = 1;

    gpm_set_cursor_range();
    gpm_available = 1;
    gpm_max_y = ScreenRows();
    gpm_max_x = ScreenCols();

    return( 1 );
}

int Gpm_GetEvent( Gpm_Event *e )
{
    if( gpm_nfd == -1 ) return( 0 );
    if( gpm_fd_ready ) {
        memcpy( e, &gpm_handler_queue[ 0 ], sizeof( Gpm_Event ) );
#ifdef GPM_DEBUG
        eventprintf( e );
#endif
        gpm_shift_queue( 1 );
        gpm_fd_ready--;

        return( 1 );
    }

    return( -1 );
}

int gpm_fileext( __FSEXT_Fnumber num, int *rv, va_list args )
{
    if( num == __FSEXT_ready ) {
        if( gpm_fd_ready ) {
#ifdef GPM_DEBUG
            fprintf( dbgfile, "gpm_fileext called and gpm_fd_ready == %i\n", gpm_fd_ready );
#endif
            *rv = __FSEXT_ready_read;
        } else *rv = 0;

        return( 1 );
    }

    return( 0 );
}

int Gpm_Open( Gpm_Connect *c, int con )
{
#ifdef GPM_DEBUG
    {
        char *df = getenv( "GPM_DBG" );

        if( df == NULL ) df = "NUL";
        dbgfile = fopen( df, "a+" );
        setvbuf( dbgfile, NULL, _IONBF, 0 );
        fprintf( dbgfile, "Opened the log file\n" );
    }
#endif
    if( !gpm_mouse_available() ) {
        gpm_tried = 1;
        return( -1 );
    }
    if( con != 0 ) return( -1 );
    if( gpm_nfd == -1 ) {
        if( ( gpm_fd = dup( STDIN_FILENO ) ) == -1 ) return( -1 );
        __FSEXT_set_function( gpm_fd, gpm_fileext );
        gpm_clear_queue();
        gpm_queue_lbp = 0;
        if( !gpm_handler_installed ) {
            if( gpm_register_internal_handler() == -1 ) {
                close( gpm_fd );
                return( -1 );
            }
            atexit( gpm_unregister_internal_handler );
        }
    }

    gpm_handler_entered = 0;
    gpm_nfd++;
    gpm_fds[ gpm_nfd ] = c;

#ifdef GPM_DEBUG
    fputs( "Showing the mouse\n", dbgfile );
#endif
    gpm_show_mouse();

    return( gpm_fd );
}

int Gpm_Close( void )
{
    if( gpm_nfd == -1 ) return( 0 );
    if( gpm_nfd == 0 ) {
        gpm_handler_entered = 1;
        close( gpm_fd );
        gpm_fd = -1;
        gpm_clear_queue();
        gpm_fd_ready = 0;
#ifdef GPM_DEBUG
        fputs( "Closing the debug file\n", dbgfile );
        fclose( dbgfile );
#endif
    }

    gpm_fds[ gpm_nfd ] = NULL;
    gpm_nfd--;

    gpm_hide_mouse();

    if( gpm_nfd == -1 ) return( 0 );
    return( -1 );
}

/* With ncurses, by default it only enables gpm on terminals containing "linux", unless found in this envitonment variable */
/* Also, TTY_SCREEN_INTFACE=DIRECT seems to work best with ncurses */
static void __attribute__((constructor)) gpm_enable_ncurses_mouse( void )
{
    putenv( "NCURSES_GPM_TERMS=djgpp|djgpp-nc|djgpp-m|djgpp-mono|ansi|ansi.sys|cons25|cygwin|linux|nansi.sys|pcansi-m|pcansi-mono|dumb|ansi-m|ansi-mono|unknown" );
    putenv( "TTY_SCREEN_INTFACE=DIRECT" );
}

