/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5warnings_H
#define H5warnings_H

#define H5_WARN_JOINSTR(x, y) x y
#define H5_WARN_DO_PRAGMA(x)  _Pragma(#x)
#define H5_WARN_PRAGMA(x)     H5_WARN_DO_PRAGMA(GCC diagnostic x)

#ifdef H5_SHOW_ALL_WARNINGS
    #define H5_WARN_OFF(x)
    #define H5_WARN_ON(x)
#else
    #if (((__GNUC__ * 100) + __GNUC_MINOR__) >= 406) || defined(__clang__)
        
        #define H5_WARN_OFF(x) H5_WARN_PRAGMA(push) H5_WARN_PRAGMA(ignored H5_WARN_JOINSTR("-W", x))
        #define H5_WARN_ON(x)  H5_WARN_PRAGMA(pop)
    #endif
#endif

#if defined(__clang__)
#define H5_WARN_C11_EXTENSIONS_OFF H5_WARN_OFF("c11-extensions")
#define H5_WARN_C11_EXTENSIONS_ON  H5_WARN_ON("c11-extensions")
#else
#define H5_WARN_C11_EXTENSIONS_OFF
#define H5_WARN_C11_EXTENSIONS_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_CAST_ALIGNMENT_OFF H5_WARN_OFF("cast-align")
#define H5_WARN_CAST_ALIGNMENT_ON  H5_WARN_ON("cast-align")
#else
#define H5_WARN_CAST_ALIGNMENT_OFF
#define H5_WARN_CAST_ALIGNMENT_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_CAST_AWAY_CONST_OFF H5_WARN_OFF("cast-qual")
#define H5_WARN_CAST_AWAY_CONST_ON  H5_WARN_ON("cast-qual")
#else
#define H5_WARN_CAST_AWAY_CONST_OFF
#define H5_WARN_CAST_AWAY_CONST_ON
#endif

#if defined(__GNUC__) && !defined(__clang__)
#define H5_WARN_DUPLICATED_BRANCHES_OFF H5_WARN_OFF("duplicated-branches")
#define H5_WARN_DUPLICATED_BRANCHES_ON  H5_WARN_ON("duplicated-branches")
#else
#define H5_WARN_DUPLICATED_BRANCHES_OFF
#define H5_WARN_DUPLICATED_BRANCHES_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_FLOAT_CONVERSION_OFF H5_WARN_OFF("float-conversion")
#define H5_WARN_FLOAT_CONVERSION_ON  H5_WARN_ON("float-conversion")
#else
#define H5_WARN_FLOAT_CONVERSION_OFF
#define H5_WARN_FLOAT_CONVERSION_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_FLOAT_EQUAL_OFF H5_WARN_OFF("float-equal")
#define H5_WARN_FLOAT_EQUAL_ON  H5_WARN_ON("float-equal")
#else
#define H5_WARN_FLOAT_EQUAL_OFF
#define H5_WARN_FLOAT_EQUAL_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_FORMAT_NONLITERAL_OFF H5_WARN_OFF("format-nonliteral")
#define H5_WARN_FORMAT_NONLITERAL_ON  H5_WARN_ON("format-nonliteral")
#else
#define H5_WARN_FORMAT_NONLITERAL_OFF
#define H5_WARN_FORMAT_NONLITERAL_ON
#endif

#if defined(__GNUC__) && !defined(__clang__)
#define H5_WARN_FORMAT_TRUNCATION_OFF H5_WARN_OFF("format-truncation")
#define H5_WARN_FORMAT_TRUNCATION_ON  H5_WARN_ON("format-truncation")
#else
#define H5_WARN_FORMAT_TRUNCATION_OFF
#define H5_WARN_FORMAT_TRUNCATION_ON
#endif

#if defined(__clang__)
#define H5_WARN_IMPLICIT_FALLTHROUGH_OFF H5_WARN_OFF("implicit-fallthrough")
#define H5_WARN_IMPLICIT_FALLTHROUGH_ON  H5_WARN_ON("implicit-fallthrough")
#else
#define H5_WARN_IMPLICIT_FALLTHROUGH_OFF
#define H5_WARN_IMPLICIT_FALLTHROUGH_ON
#endif

#if defined(__clang__)

#define H5_WARN_LARGE_STACK_OBJECTS_OFF H5_WARN_OFF("overlength-strings")
#define H5_WARN_LARGE_STACK_OBJECTS_ON  H5_WARN_ON("overlength-strings")
#elif defined(__GNUC__)

#define H5_WARN_LARGE_STACK_OBJECTS_OFF H5_WARN_OFF("larger-than=")
#define H5_WARN_LARGE_STACK_OBJECTS_ON  H5_WARN_ON("larger-than=")
#else
#define H5_WARN_LARGE_STACK_OBJECTS_OFF
#define H5_WARN_LARGE_STACK_OBJECTS_ON
#endif

#if defined(__GNUC__) && !defined(__clang__)
#define H5_WARN_MPI_STATUSES_IGNORE_OFF H5_WARN_OFF("stringop-overflow")
#define H5_WARN_MPI_STATUSES_IGNORE_ON  H5_WARN_ON("stringop-overflow")
#else
#define H5_WARN_MPI_STATUSES_IGNORE_OFF
#define H5_WARN_MPI_STATUSES_IGNORE_ON
#endif

#if defined(__clang__)
#define H5_WARN_NONSTD_SUFFIX_OFF H5_WARN_OFF("pedantic")
#define H5_WARN_NONSTD_SUFFIX_ON  H5_WARN_ON("pedantic")

#elif defined(__GNUC__) && __GNUC__ >= 14
#define H5_WARN_NONSTD_SUFFIX_OFF H5_WARN_OFF("c11-c23-compat")
#define H5_WARN_NONSTD_SUFFIX_ON  H5_WARN_ON("c11-c23-compat")

#elif defined(__GNUC__) && __GNUC__ >= 9
#define H5_WARN_NONSTD_SUFFIX_OFF                                                                            \
    H5_WARN_OFF("pedantic")                                                                                  \
    H5_WARN_OFF("c11-c2x-compat")
#define H5_WARN_NONSTD_SUFFIX_ON                                                                             \
    H5_WARN_ON("c11-c2x-compat")                                                                             \
    H5_WARN_ON("pedantic")
#else

#define H5_WARN_NONSTD_SUFFIX_OFF
#define H5_WARN_NONSTD_SUFFIX_ON
#endif

#if defined(__GNUC__) && !defined(__clang__)
#define H5_WARN_OBJ_FXN_POINTER_CONVERSION_OFF H5_WARN_OFF("pedantic")
#define H5_WARN_OBJ_FXN_POINTER_CONVERSION_ON  H5_WARN_ON("pedantic")
#else
#define H5_WARN_OBJ_FXN_POINTER_CONVERSION_OFF
#define H5_WARN_OBJ_FXN_POINTER_CONVERSION_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_USELESS_COMPARISON_OFF H5_WARN_OFF("type-limits")
#define H5_WARN_USELESS_COMPARISON_ON  H5_WARN_ON("type-limits")
#else
#define H5_WARN_USELESS_COMPARISON_OFF
#define H5_WARN_USELESS_COMPARISON_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_AGGREGATE_RETURN_OFF H5_WARN_OFF("aggregate-return")
#define H5_WARN_AGGREGATE_RETURN_ON  H5_WARN_ON("aggregate-return")
#else
#define H5_WARN_AGGREGATE_RETURN_OFF
#define H5_WARN_AGGREGATE_RETURN_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_MISSING_PROTOTYPE_OFF H5_WARN_OFF("missing-prototypes")
#define H5_WARN_MISSING_PROTOTYPE_ON  H5_WARN_ON("missing-prototypes")
#else
#define H5_WARN_MISSING_PROTOTYPE_OFF
#define H5_WARN_MISSING_PROTOTYPE_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_SIGN_CONVERSION_OFF H5_WARN_OFF("sign-conversion")
#define H5_WARN_SIGN_CONVERSION_ON  H5_WARN_ON("sign-conversion")
#else
#define H5_WARN_SIGN_CONVERSION_OFF
#define H5_WARN_SIGN_CONVERSION_ON
#endif

#if defined(__clang__) || defined(__GNUC__)
#define H5_WARN_UNUSED_PARAMETER_OFF H5_WARN_OFF("unused-parameter")
#define H5_WARN_UNUSED_PARAMETER_ON  H5_WARN_ON("unused-parameter")
#else
#define H5_WARN_UNUSED_PARAMETER_OFF
#define H5_WARN_UNUSED_PARAMETER_ON
#endif

#endif 
