### Name: match-utils
### Title: Utility functions related to pattern matching
### Aliases: match-utils nmismatchStartingAt nmismatchEndingAt
###   nmismatchStartingAt,character-method
###   nmismatchEndingAt,character-method nmismatchStartingAt,XString-method
###   nmismatchEndingAt,XString-method isMatching
###   isMatching,character-method isMatching,XString-method mismatch
###   mismatch,XStringViews-method coverage coverage,IRanges-method
###   coverage,MaskCollection-method coverage,XStringViews-method
###   coverage,MaskedXString-method coverage,MIndex-method
### Keywords: methods

### ** Examples

  ## ---------------------------------------------------------------------
  ## nmismatchStartingAt() / isMatching()
  ## ---------------------------------------------------------------------
  subject <- DNAString("GTATA")

  ## Pattern "AT" matches subject "GTATA" at position 3 (exact match)
  nmismatchStartingAt("AT", subject, starting.at=3)
  isMatching("AT", subject, start=3)

  ## ... but not at position 1
  nmismatchStartingAt("AT", subject)
  isMatching("AT", subject)

  ## ... unless we allow 1 mismatching letter (inexact match)
  isMatching("AT", subject, max.mismatch=1)

  ## Here we look at 6 different starting positions and find 3 matches if
  ## we allow 1 mismatching letter
  isMatching("AT", subject, start=0:5, max.mismatch=1)

  ## No match
  nmismatchStartingAt("NT", subject, starting.at=1:4)
  isMatching("NT", subject, start=1:4)

  ## 2 matches if N is interpreted as an ambiguity (fixed=FALSE)
  nmismatchStartingAt("NT", subject, starting.at=1:4, fixed=FALSE)
  isMatching("NT", subject, start=1:4, fixed=FALSE)

  ## max.mismatch != 0 and fixed=FALSE can be used together
  nmismatchStartingAt("NCA", subject, starting.at=0:5, fixed=FALSE)
  isMatching("NCA", subject, start=0:5, max.mismatch=1, fixed=FALSE)

  some_starts <- c(10:-10, NA, 6)
  subject <- DNAString("ACGTGCA")
  is_matching <- isMatching("CAT", subject, start=some_starts, max.mismatch=1)
  some_starts[is_matching]

  ## ---------------------------------------------------------------------
  ## mismatch()
  ## ---------------------------------------------------------------------
  m <- matchPattern("NCA", subject, max.mismatch=1, fixed=FALSE)
  mismatch("NCA", m)

  ## ---------------------------------------------------------------------
  ## coverage()
  ## ---------------------------------------------------------------------
  coverage(m)

  x <- IRanges(start=c(-2L, 6L, 9L, -4L, 1L, 0L, -6L, 10L),
               width=c( 5L, 0L, 6L,  1L, 4L, 3L,  2L,  3L))
  coverage(x, start=-6, end=20)  # 'start' and 'end' must be specified for
                                 # an IRanges object.
  coverage(shift(x, 2), start=-6, end=20)
  coverage(restrict(x, 1, 10), start=-6, end=20)
  coverage(reduce(x), start=-6, end=20)
  coverage(gaps(x, start=-6, end=20), start=-6, end=20)

  mask1 <- Mask(mask.width=29, start=c(11, 25, 28), width=c(5, 2, 2))
  mask2 <- Mask(mask.width=29, start=c(3, 10, 27), width=c(5, 8, 1))
  mask3 <- Mask(mask.width=29, start=c(7, 12), width=c(2, 4))
  mymasks <- append(append(mask1, mask2), mask3)
  coverage(mymasks)

  ## See ?matchPDict for examples of using coverage() on an MIndex object...



