\name{GetColor}
\alias{GetColor}
\alias{dChip.colors}
\alias{greenred.colors}
\title{ A function to get the Red-Blue color scheme used by dChip }
\description{
 A simple, vectorized function that computes a Red/Blue color for
 plotting microarray expression data.
}
\usage{
GetColor(value, GreenRed=FALSE, DisplayRange=3)
dChip.colors(n)
greenred.colors(n)
}

\arguments{
  \item{value}{The vector of expression values. }
  \item{GreenRed}{If \code{TRUE} the Green-Red colors are produced,
    otherwise Red-Blue are procduced. }
  \item{DisplayRange}{ A parameter controlling the range of
    \code{value}'s that will be plotted. }
  \item{n}{An integer saying how many colors to be in the palette.}
}
\details{
  \code{GetColor} is a simple mapping into RGB land provided by Cheng
  Li.
  \code{dChip.colors} provides functionality similar to that of
  \code{\link{topo.colors}} for the red--blue colors used for genome
  plots. \code{greenred.colors} does the same for the green-black-red gradient. 
}
\value{
 A vector of RGB colors suitable for plotting in R.
}

\author{R. Gentleman, based on an original by C. Li. }

\examples{
  set.seed(10)
  x <- rnorm(10)
  GetColor(x)
  dChip.colors(10)
}
\keyword{ manip }


\eof
\name{Makesense}
\alias{Makesense}
\alias{Makesense,matrix-method}
\alias{Makesense,exprSet-method}
\title{Produce Smoothed Sense/Anti-sense For All Chromosomes}
\description{
  'Makesense' takes either an exprSet object or a matrix of gene
  expressions and will produce a smoothed positive and negative strands
  for all chromosomes.
}
\usage{
Makesense(expr, lib, f)
}
\arguments{
  \item{expr}{Either an exprSet object or a matrix of gene expressions
    with genes as rows and columns as samples.}
  \item{lib}{The name of the Bioconductor annotation data package that
    will be used to provide mappings from probes to chromosomal
    locations, such as \code{hgu95av2} or \code{hgu133a}.  If
    \code{expr} is an \code{exprSet}, the argument defaults to the
    \code{annotation} slot of the \code{exprSet}.}
  \item{f}{the smoother span to be passed to 'lowess'. Its value should
    be in the interval of (0,1). This gives the proportion of points in
    the plot which influence the smooth at each value. Larger values
    give more smoothness.  The default value for this argument is 1/10.}
}
\details{
  The \code{expr} argument can either be of class \code{exprSet} or
  \code{matrix}, where the latter represents the matrix of gene
  expressions.

  If the \code{expr} argument is an \code{exprSet}, the \code{lib}
  argument will use the \code{annotation} slot.  Users can override this
  behaviour and supply their own \code{lib} argument if they wish.  If
  the \code{exprSet} has no value associated with the \code{annotation}
  slot (which should not happen, but is possible) then the user must
  supply the \code{lib} argument manually or the function will throw an
  error.
  }
\value{
A list of 2 components:
\item{ans2}{a list, whose components correspond to samples 
in the same order as appearing in the columns of 
'expr'. Each component is also a list, named by chromosomes 
"1"-"22", "X" and "Y". Each named component is again a list 
with two elements named "posS" and "negS", corresponding to the
positive and negative strands of a chromosome, each of which isan object returned by 'lowess'.}
\item{lib}{as passed in.}
}
\author{Robert Gentleman and Xiaochun Li}

\seealso{ \code{\link{plotChr}}}
\examples{
require("hgu133a") ||
stop("\nThis example requires the hgu133a package which is not installed.",
     "\nPlease install the package (perhaps using: ",
     "install.packages2(\"hgu133a\")) and try again")

data(eset133a)

esetobj <- Makesense(exprs(eset133a), "hgu133a")

esetobj2 <- Makesense(eset133a, "hgu133a")
}
\keyword{dplot}

\eof
\name{alongChrom}
\alias{alongChrom}
\alias{buildACMainLabel}
\alias{doACImagePlot}
\alias{fixACPhysPoints}
\alias{getACGeneSyms}
\alias{limitACXRange}
\alias{cullACXPoints}
\alias{doACLocalPlot}
\alias{getACClosestPos}
\alias{getACPlotLabs}
\alias{scaleACData}
\alias{dispACXaxis}
\alias{doACMatPlot}
\alias{getACDataEnv}
\alias{getACStrandVals}
\alias{doACCumPlot}
\alias{emptyACPlot}
\alias{getACExprs}
\alias{highlightACDups}
\title{A function for plotting expression data from an exprset for a
  given chromosome. }
\description{
  Given a particular exprset object, a chromLocation object, and a
  chromosome name, will plot selected exprSet data using various methods.
}
\usage{
   alongChrom(eSet, chrom, specChrom,  xlim, whichGenes, 
   plotFormat=c("cumulative", "local","image"),
   xloc=c("equispaced", "physical"), 
   scale=c("none","zscale","rankscale","rangescale","zrobustscale"),
   geneSymbols=FALSE, byStrand=FALSE, colors="red", lty=1, type="S",
   ...)}
\arguments{
  \item{eSet}{The exprset object to be used.}
  \item{chrom}{The desired chromosome.}
  \item{specChrom}{An object of type chromLocation for the species being
    represented.}
  \item{xlim}{A pair of values - either character or integer, which will
    denote the range of genes to display (based on base pair: either directly
    in the case of integers, or using the locations of the named genes if
    character).  If not supplied, the entire chromosome is used.}
  \item{whichGenes}{If supplied, will limit the displayed genes to the ones
    provided in this vector.}
  \item{xloc}{Determines whether the X axis points (gene names) will be
    displayed according to their relative position on the chromosome
    (physical), or spaced evenly (equispaced).  Default is equispaced.}
  \item{plotFormat}{Determines the method which to plot the data.}
  \item{scale}{Determines what method of scaling will be applied to the
    data.  Default is none.}
  \item{geneSymbols}{Notes whether to use Affy IDs or Gene Symbols,
    default is Affy IDs}
  \item{byStrand}{Determines whether to show the entire plot at once, or
    a split plot by strands.  Default is a singular plot}
  \item{lty}{A vector of line types, which will be cycled. }
  \item{type}{Plot type, from par.  Defaults to "S".}
  \item{colors}{A vector of colors for the plots, which will be cycled.}
  \item{...}{Any remaining graphics commands may be passed along as per plot()}
}
\details{
  The genes on the chromosome of interest are extracted from the
  \code{chromLocation} object passed in, which are then intersected with the
  genes listed in the exprset.  These remaining genes will then be
  plotted according to the \code{plotFormat} argument.  If \code{image} is
  specified, an image plot is created showing the expression levels of
  the samples by gene, using a colour map to denote the levels.  If
  \code{cumulative} is chosen, the cumulative expression level is plotted
  against the genes for each sample.  Likewise, if \code{local} is used, the
  raw data is plotted for each sample against the genes using a boxplot format.

  Not all parameters are honored for all plotformats.  \code{xloc},
  \code{lty}, and \code{type} are only used with the \code{cumulative}
  plotformat.
}
\author{Jeff Gentry}
\examples{
   data(eset)
   ## A bit of a hack to not have a package dependency on hgu95av2
   ## but need to fiddle w/ the warn level to not fail the example anyways.
   curWarn <- options(warn=0)
   on.exit(options(warn=curWarn$warn), add=TRUE)
   if (require(hgu95av2)) {
      z <- buildChromLocation("hgu95av2")
      lty <- c(1, 2, 3, 4, 5)
      cols <- c("red", "green", "blue", "orange", "magenta", "black")
      cols <- cols[eset$cov3]
      if (interactive()) {
         par(ask=TRUE)
      }

      ## Here we're using xlim to denote a physical region to display
      xlim <- c(87511280,127717880)
      for (xl in c("equispaced", "physical"))
        for (sc in c("none","rangescale"))
        {
           alongChrom(eset, "1", z, xlim=xlim, xloc=xl,
              plotFormat="cumulative", scale=sc,lty=lty, col=cols)
        }

     ## Here we're looking for specific genes
     which <- c("31540_at","31583_at", "31508_at", "31529_at", "31439_f_at",
               "31729_at")
     ## Gene "31529_at" does not exist in the current set of genes,
     ## here it demonstrates how genes not available are dropped.
      for (xl in c("equispaced", "physical"))
        for (sc in c("none","rangescale"))
        {
            alongChrom(eset, "1", z, which=which, xloc=xl,
               plotFormat="cumulative", scale=sc,lty=lty, col=cols)
         }

      ## Do an image plot
      for (bs in c(TRUE,FALSE))
         alongChrom(eset, "1",z, xlim=xlim, plotFormat="image",
                   scale="zscale", byStrand=bs)

      ## A boxplot
      for (st in c(TRUE,FALSE))
         alongChrom(eset, "1", z, plotFormat="local",
                    colors=cols, byStrand=st)
    } else print("Example can not be run without the hgu95av2 data package")
}
  
\keyword{utilities}


\eof
\name{amplicon.plot}
\alias{amplicon.plot}
\title{Create an amplicon plot  }
\description{
  Given a two-sample test statistic and an exprSet this function plots
  regions of the genome that are either highly expressed (in red) or
  have low expression (blue) differentially in the two groups.
}
\usage{
amplicon.plot(ESET, FUN, genome)
}
\arguments{
  \item{ESET}{an object of class \code{exprSet}}
  \item{FUN}{A two sample test function suitable for \code{\link{esApply}}. }
  \item{genome}{A character string of the base name for the annotation. }
}
\details{
  In some genetic studies we are interested in finding regions of the
  genome where there are a set of highly expressed genes in some
  subgroup of the population. This set of highly (or lowly) expressed
  genes is often of great interest. For example in breast cancer the
  HER--2 gene is on an amplicon. In some patients approximately 5 genes
  located near HER--2 are all amplified.

  These plot should help in the search for such regions.
}
\value{
  No value is returned. This function is executed purely for side
  effect.
}

\author{Robert Gentleman }

\seealso{\code{\link{esApply}}, \code{\link{make.chromOrd}}}

\examples{
  ##none yet; takes too long
}
\keyword{ hplot }

\eof
\name{cColor}
\alias{cColor}
\title{A function for marking specific probes on a cPlot. }
\description{
  Given a set of probes, will highlight them in the color desired on
  a plot which has already been created via the function cPlot().
}
\usage{
cColor(probes, color, plotChroms, scale=c("relative","max"),glen=0.4)
}
\arguments{
  \item{probes}{The probes that are being highlighted.}
  \item{color}{The color to highlight the probes.}
  \item{plotChroms}{An object of type \code{chromLocation} which contains all
    the gene information to be plotted. }
  \item{scale}{Whether to plot the graph scaled absolutely or relative
    by chromosome.  Default is absolute.}
  \item{glen}{The length of the gene line plotted}
}
\details{
  It is important to call the function \code{cPlot()} first.  This function
  will then search for the specific locations of the probes desired,
  which are contained within the \code{plotChroms} instance of a
  \code{chromLocation} class.  It will then pass these on to the
  plotting routine to highlight the desired locations.  NOTE:  It
  is important that \code{plotChroms}, \code{scale} and \code{glen}
  parameters are the same as used for \code{cPlot()}.
}
\author{Jeff Gentry}
\seealso{
  \code{\link{cPlot}}, \code{\link{chromLocation-class}}
}
\examples{
   ## A bit of a hack to not have a package dependency on hgu95av2
   ## but need to fiddle w/ the warn level to not fail the example anyways.
   curWarn <- options(warn=0)
   on.exit(options(warn=curWarn), add=TRUE)
   if (require(hgu95av2)) {

     z <- buildChromLocation("hgu95av2")
     cPlot(z)
     probes <- c("266_s_at", "31411_at", "610_at", "failExample")
     cColor(probes, "red", z)
     probes2 <- c("960_g_at", "41807_at", "931_at", "39032_at")
     cColor(probes2, "blue", z)
 } else print("Need hgu95av2 data package to run example")
}
\keyword{utilities}


\eof
\name{cPlot}
\alias{cPlot}
\title{A plotting function for chromosomes. }
\description{
  Given a chromLocation object, will plot all the gene locations from
  that object.
}
\usage{
cPlot(plotChroms, useChroms=chromNames(plotChroms),
      scale=c("relative","max"), fg="white", bg="lightgrey", glen=0.4)
}
\arguments{
  \item{plotChroms}{An object of type chromLocation which contains all
    the gene information to be plotted. }
  \item{useChroms}{A vector of chromosome names to be used in the plot.
  Default is to use all the chromosomes from the plotChroms object.}
  \item{scale}{Passed on to cScale as it's scale argument.  Determines
    whether the graph is scaled on a relative or absolute basis.}
  \item{fg}{The colour to be used for the genes.  Default is white.}
  \item{bg}{The colour to be used for the background of the plot.
    Defaults to lightgrey/}
  \item{glen}{A scaling factor applied to the plotted length of each
    gene.  Defaults to 0.4 - it is recommended that this not be set
    larger then 0.5 as it will cause overlap between chromosomes.}
}
\details{
  This function will first use the lengths of the chromosomes, stored in
  the object to create scaling factors for the X axis.  Once the
  scaling factors are determined, the \code{chromLocation} object which is
  passed in is used to determine all the gene locations/strand
  information/etc, which is then plotted for the user.
}
\author{Jeff Gentry}
\seealso{\code{\link{cScale}}, \code{\link{cColor}},
  \code{\link{chromLocation-class}}}
\examples{
   ## A bit of a hack to not have a package dependency on hgu95av2
   ## but need to fiddle w/ the warn level to not fail the example anyways.
   curWarn <- options(warn=0)
   on.exit(options(warn=curWarn), add=TRUE)
   if (require(hgu95av2)) {
    z <- buildChromLocation("hgu95av2")

     if (interactive()) {
       par(ask=TRUE)
     }

     for (sc in c("max","relative"))
       cPlot(z,c("1","5","10","X","Y"),sc)
  } else print("This example can not be run without hgu95av2 data package")
}
\keyword{utilities}


\eof
\name{cScale}
\alias{cScale}
\title{A function for mapping chromosome length to a number of points. }
\description{
  Given a number of points (generally representing the number of points
  on a plot's axis), and a vector of chromosome lengths - will generate
  a vector of the same length as the one passed in containing scaling
  factors for each chromosome.
}
\usage{
cScale(points, cLengths, method=c("max", "relative"), chrom)
}
\arguments{
  \item{points}{The number of points to scale the chromosome length to.}
  \item{cLengths}{A vector of chromosome lengths. }
  \item{method}{Determines whether to use relative or absolute scaling.
    Default is "max" (absolute).}
  \item{chrom}{Which chrom to determine the scale for}
}
\details{
  The scale factor is calculated in a manner based on the \code{method}
  argument.  If method is \code{max}, the factor is derived by dividing the
  points argument by each chromosome's length (in base pairs).  If the
  method chosen is \code{relative}, then the scale is determined by dividing
  the points argument by the maximum chromsome length, and applying that
  value to each chromosome.
}
\author{Jeff Gentry}
\seealso{\code{\link{cPlot}}}
\examples{

   ## A bit of a hack to not have a package dependency on hgu95av2
   ## but need to fiddle w/ the warn level to not fail the example anyways.
   curWarn <- options(warn=0)
   on.exit(options(warn=curWarn), add=TRUE)
   if (require(hgu95av2)) {
      z <- buildChromLocation("hgu95av2")

     for (sc in c("max","relative"))
       scale <- cScale(1000, chromLengths(z),sc,"Y")
   } else print("This example needs the hgu95av2 data package")
}
\keyword{utilities}


\eof
\name{colorRampPalette}
\alias{colorRampPalette}
\alias{colorRamp}
\title{Extend a color palette to a color ramp }
\description{
These functions are useful for converting hand-designed `sequential' or `diverging' color schemes into continous color ramps eg for image and filled contour plots.
}
\usage{
colorRampPalette(palette, bias = 1,method=c("linear","spline"))
colorRamp(palette, bias=1,method=c("spline","linear"))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{palette}{A vector of colors}
  \item{bias}{Bias >1 puts more colors at high values, <1 puts more at low values}
\item{method}{Interpolation method. Spline interpolation may give smoother results but can end up outside the valid RGB space, especially with highly saturated palettes or those based on a small starting set.}
}

\value{
\code{colorRampPalette} returns a palette generating function like \code{\link{terrain.colors}} or \code{\link{heat.colors}} that takes an integer argument and generates a palette with that many colors.

\code{colorRamp} returns a color generating function, that takes a number between 0 and 1 and generates the corresponding color.

}
\author{Thomas Lumley}

\seealso{\url{www.colorbrewer.org} }

\examples{
   m<-outer(1:20,1:20,function(x,y) sin(sqrt(x*y)/3))

   redblue<-colorRampPalette(c("red","orange","blue"))
   filled.contour(m,color.palette=redblue)

   yb<-colorRampPalette(c("yellow2","goldenrod","darkred"))
   filled.contour(m,color.palette=yb)

   yb<-colorRampPalette(c("yellow2","goldenrod","darkred"),bias=0.5)
   filled.contour(m,color.palette=yb)

   ## Two ColorBrewer sequential palettes
   ylOrBn4<-c("#FFFFD4", "#FED98E", "#FE9929","#CC4C02")
   ylGnBl5<-c("#FFFFCC","#C7E9B4","#7FCDBB","#40B6C4","#2C7FB8" ,"#253494")

   ylOrBn<-colorRampPalette(ylOrBn4)
   ylGnBl<-colorRampPalette(ylGnBl5,method="spline")
   filled.contour(m,color.palette=ylOrBn)
   filled.contour(m,color.palette=ylGnBl)

   if(require("RColorBrewer")){
	RdBu7<-colorRampPalette(brewer.pal(7,"RdBu"),method="spline")
        RWB<-colorRampPalette(c("red","white","blue"))
        redblue<-colorRampPalette(c("red","blue"))
	filled.contour(m,color.palette=redblue,main="naive red-blue")
        filled.contour(m,color.palette=RWB,main="naive red-white-blue")
        filled.contour(m,color.palette=RdBu7,main="ColorBrewer RdBu7")
   }

}
\keyword{color}% at least one, from doc/KEYWORDS

\eof
\name{eset133a}
\alias{eset133a}
\docType{data}
\title{A dataset of class exprSet}
\description{
	An artificial Affymetrix hgu133a dataset, with one 
covariate 'cov1'.
}
\usage{data(eset133a)}
\format{
 The data are artifical. There are 6 cases labeled 1 to
 6 and and 22283 genes as in an Affymetrix U133a chips. 
 There is one covariate (factor) whose values are "type 1"
 for the first 3 samples and "type 2" for the last 3 samples.
}
\examples{
data(eset133a)
}
\keyword{datasets}

\eof
\name{identifyLines}
\alias{identifyLines}
\title{A function for identifying sample lines on an alongChrom()
  plot. }
\description{
  Given an environment returned from alongChrom(), will call identify()
  on the alongChrom() plot.
}
\usage{
identifyLines(identEnvir,...)
}
\arguments{
  \item{identEnvir}{An environment created by alongChrom.}
  \item{...}{Any extra arguments are passed on to identify().}
}
\details{
  The environment contains two vectors of points - "X" and "Y" are their
  labels, and contain the X and Y points respectively.  These vectors
  are extracted out of the environment and fed into an identify() call
  (with some extra processing to get the labeling correct.)  As with
  identify(), one need only to right click to exit from the function.
}
\author{Jeff Gentry}
\seealso{
  \code{\link{alongChrom}}, \code{\link{identify}}.
}
\examples{
   data(eset)

   ## A bit of a hack to not have a package dependency on hgu95av2
   ## but need to fiddle w/ the warn level to not fail the example anyways.
   curWarn <- options(warn=0)
   on.exit(options(warn=curWarn), add=TRUE)
   if (require(hgu95av2)) {
     z <- buildChromLocation("hgu95av2")
     lty <- c(1, 2, 3, 4, 5)
     cols <- c("red", "green", "blue", "orange", "magenta", "black")
     identEnv <- alongChrom(eset,"1", z, xloc="equispaced",
                          plotFormat="cumulative", scale="none",
                          lty,cols)
     if (interactive()) {
       identifyLines(identEnv)
     }
  } else print("This example needs the hgu95av2 data package")
}
\keyword{utilities}

\eof
\name{make.chromOrd}
\alias{make.chromOrd}
\title{Make a chromOrd object  }
\description{
  This function makes a chromOrd object.
}
\usage{
make.chromOrd(genome, gnames)
}
\arguments{
  \item{genome}{A character string. }
  \item{gnames}{A character vector of the genes to be selected. }
}
\details{
  This function reads in a lot of annotation data and creates a list
  with one element for each chromosome. The elements of this list are
  indices indicating the order of the genes that are on that chromosome
  (and in the annotation data set being used).
}
\value{
 A list of chromOrd type. One element for each chromosome. Suitable for
 reordering other values according to the chromosomal location.
}

\author{ Robert Gentleman }

\seealso{ \code{\link{amplicon.plot}} }

\examples{
  data(eset)
  make.chromOrd("hgu95A", geneNames(eset))
}
\keyword{ utilities }

\eof
\name{plotChr}
\alias{plotChr}
\title{Plot Smoothed Sense/Anti-sense of Specified Chromosomes}
\description{
For a given chromosome, plot the smooths of the sense and the 
anti-sense from 5' to 3' (left to right on x-axis). 
}
\usage{
plotChr(chrN, senseObj, cols = rep("blue", length(senseObj[[1]])), log = FALSE, xloc = c("equispaced", "physical"), geneSymbols = FALSE, ngenes = 20, lines.at = NULL, lines.col = "red")
}
\arguments{
  \item{chrN}{The desired chromosome, e.g. for humans it would be a character string in the set of c(1:22, "X", "Y").}
  \item{senseObj}{The result of \code{Makesense}.}
  \item{cols}{A vector of colors for the lines in the plot, typically specified according to a certain pheotype of samples.}
  \item{log}{Logical, whether log-transformation should be taken on the smoothed expressions.}
  \item{xloc}{Determines whether the "Representative Genes" 
	will be displayed according to their relative 
	positions on the chromosome (physical), or spaced 
	evenly (equispaced).  Default is equispaced.}
  \item{geneSymbols}{Logical, whether to use Affy IDs or 
	Gene Symbols for "Representative Genes", default is 
	Affy IDs.}
  \item{ngenes}{Desired number of "Representative Genes". The 
	number of actual displayed genes may differ.}
  \item{lines.at}{A vector of Affy IDs. Vertical lines will 
	be drawn at specified genes.}
  \item{lines.col}{A vector of colors associated with 
	\code{lines.at}.}
}
\author{Robert Gentleman and Xiaochun Li}
\seealso{\code{\link{Makesense}}}
\examples{
example(Makesense)

op <- par(ask=TRUE)

cols <- c("red", "green")[unclass(eset133a$cov1)]
plotChr("21", esetobj, cols)

# plot on log-scale:

plotChr("21", esetobj, cols, log=TRUE)

# genesymbol instead of probe names:

plotChr("21", esetobj, cols, log=TRUE, geneSymbols=TRUE)

# add vertical lines at genes of interest:

gs <- c("220372_at", "35776_at", "200943_at")
plotChr("21", esetobj, cols, log=TRUE, geneSymbols=FALSE, lines.at=gs)

# add vertical lines at genes of interest
# with specified colors:

gs <- c("220372_at", "35776_at", "200943_at")
cc <- c("blue", "cyan","magenta")
plotChr("21", esetobj, cols, log=TRUE, geneSymbols=FALSE, lines.at=gs,
lines.col=cc)
par(op)
}
\keyword{hplot} 

\eof
\name{plotExpressionGraph}
\alias{plotExpressionGraph}
\alias{getPlotExpressionColors}
\alias{defMapFun}
\title{A function to plot a graph colored by expression data}
\description{
  Given a graph and expression data for one entity, will plot the graph
  with the nodes colored according to the expression levels provided.
}
\usage{
plotExpressionGraph(graph, nodeLLmap, exprs, LOCUSIDenvir, mapFun, log = FALSE, nodeAttrs = list(), ...)
}
\arguments{
  \item{graph}{The graph to plot}
  \item{nodeLLmap}{A list with element names being node names and the
    elements being LocusLink IDs corresponding to those node names.}
  \item{exprs}{A vector of expression data, with names being Affymetrix
    IDs and values being the expression level.}
  \item{LOCUSIDenvir}{An environment mapping Affymetrix IDs to LocusLink
  IDs, such as the ones provided in the xxx2LOCUSID environments from
  the Bioconductor data packages (where xxx) is a data package)}
  \item{mapFun}{A function to map expression levels to colors}
  \item{log}{Whether or not the expression data }
  \item{nodeAttrs}{A list of node attributes, as per \code{plot.graph}}
  \item{\dots}{ Any extra arguments to be passed to \code{plot.graph} }
}
\details{
  This function can be used to plot a graph and have the nodes colored
  according to expression levels provided by the user.  The
  \code{graph} parameter is a \code{graph} object from the \code{graph}
  package.

  The \code{nodeLLmap} parameter is a list that maps the nodes of the
  graphs to LocusLink IDs.  An example of this is the
  \code{IMCALocusLink} object in the
  \code{integrinMediatedCellAdhesion} data set in the
  \code{graph} package.

  The \code{exprs} argument is a vector mapping expression levels to
  Affymetrix IDs.  One way to generate an appropriate vector is to
  extract a single column from an \code{exprSet}.

  The \code{LOCUSIDenvir} environment maps Affymetrix IDs to LocusLink
  IDs.  The simplest way to provide this argument is to load the
  preferred Bioconductor data package (e.g. \code{hgu95av2}) and pass in
  that package's \code{xxx2LOCUSID}, where \code{xxx} is the name of the
  package.

  The \code{mapFun} function defaults to the function \code{defMapFun},
  which maps nodes to be either blue, green or red depending for
  expression ranges of 0-100, 101-500, and 501+.  In the case where
  \code{log} is \code{TRUE} these ranges are modified with
  \code{\link{log2}}.  Custom versions of this function can be supplied
  by the user - it must take two parameters, first the expression vector
  and a boolean value (\code{log}) specifying if the data has had a
  \code{log2} applied to it.  The function must return a vector with the
  same names as the expression vector, but the values of the vector will
  be color strings.

  The \code{nodeAttrs} list can be specified if any other node
  attributes are desired to be set by the user.  Please see the
  \code{\link{plot.graph}} man page for more information on this.  The
  other attribute list (\code{attrs} and \code{edgeAttrs}) can be passed
  in via the \code{...} parameter.
}
\author{Jeff Gentry}
\seealso{\code{\link{plot.graph}}, \code{integrinMediatedCellAdhesion}}
\examples{
   if ((require("Rgraphviz")) && (require("hgu95av2")) &&
        (require("fibroEset")) && (interactive())) {     
            data(integrinMediatedCellAdhesion)
            data(fibroEset)
            attrs=getDefaultAttrs()
            attrs$graph$rankdir <- "LR"
            plotExpressionGraph(IMCAGraph, IMCALocusLink,
                                exprs(fibroEset)[,1],
                                hgu95av2LOCUSID, attrs=attrs)
   }
}
\keyword{utilities}
\keyword{graphs}
\keyword{hplot}

\eof
