autoPkgBuilder <- function(basePath ="/scratch/homes/jzhang/affymappings",
                           baseMapType = "gb",
                           pkgPath = "/scratch/homes/jzhang/datapkgs",
                           version = "1.2.1",
                           author = list(author = "Jianhua Zhang",
                           maintainer = "jzhang@jimmy.harvard.edu"),
                           pattern = "_.*\\.txt$", pkgOnly = FALSE){
    srcs <- getPkgFileNames(basePath, pattern = pattern)

    for(i in names(srcs)){

        baseName <- srcs[[i]]["ID"]
        otherSrc <- srcs[[i]][names(srcs[[i]]) != "ID"]
        organism <- getOrganism(i)
        srcUrls <- getSrcUrl("all", organism = organism)

        ABPkgBuilder(baseName = baseName , srcUrls = srcUrls,
                     baseMapType = baseMapType, otherSrc = otherSrc,
                     pkgName = i, pkgPath = pkgPath,
                     organism = organism, version = version,
                     author = author)
    }
    if(!pkgOnly){
        for(i in c("human", "mouse", "rat")){
            chrLengths <- findChrLength(i, getSrcUrl("GP", i))
            for(j in getPkgNameByOrg(pkgPath, i)){
                writeOrganism(j, pkgPath, i)
                writeChrLength(j, pkgPath, chrLengths)
                writeManPage(j, pkgPath, "CHRLENGTHS", organism = i,
                             src = "gp", isEnv = FALSE)
                writeManPage(j, pkgPath, "ORGANISM", organism = i,
                             src = NA, isEnv = FALSE)
            }
        }
        GOPkgBuilder(pkgPath = pkgPath, version = version, author = author)
        KEGGPkgBuilder(pkgPath = pkgPath, version = version, author = author)
    }
}

getPkgFileNames <- function(basePath, pattern = "_.*\\.txt$"){
    # Create a list for holding the data
    idNSrcs <- list()

    pkgNames <- list.files(basePath, pattern = pattern, full.names = TRUE)

    for(i in unique(gsub("_.*\\.txt", "", pkgNames))){
        files4OnePkg <- pkgNames[grep(paste("^", i, "_.*", sep = ""),
                                     pkgNames)]
        if(length(grep("_ID", files4OnePkg)) > 0){
            files4OnePkg <- c(files4OnePkg[grep("_ID", files4OnePkg)],
                              files4OnePkg[-grep("_ID", files4OnePkg)])
            names(files4OnePkg) <- c("ID",
                                  LETTERS[1:length(files4OnePkg) - 1])
            idNSrcs[[basename(i)]] <- files4OnePkg
        }
    }
    return(idNSrcs)
}

getOrganism <- function(orgChar){
    orgChar <- substr(orgChar, 1, 2)
    switch(toupper(orgChar),
           HG = ,
           HU = return("human"),
           MO = ,
           MG = return("mouse"),
           RA = ,
           RG = return("rat"),
           stop(paste("Unknown entry for orgChar:", orgChar)))
}

getPkgNameByOrg <- function(pkgPath, organism){
    switch(toupper(organism),
           HUMAN = init <- "h",
           MOUSE = init <- "mg\|mo",
           RAT = init <- "rg\|ra",
           stop(paste("Ubknown organism", organism)))

    pkgNames <- list.files(pkgPath, pattern = init)

    return(setdiff(pkgNames, pkgNames[grep("\\.xml$", pkgNames)]))
}

putDPkgsToWeb <- function(pkgPath = "/scratch/homes/jzhang/datapkgs",
                    webPath = "/scratch/homes/jzhang/bioC/webstuff/data",
                    organism = "human", version = "1.2.1", what = "gz"){

    if(organism == "GO" || organism == "KEGG"){
        destDir <- file.path(webPath, "misc", "files", version)
    }else{
        destDir <- file.path(webPath, organism, "files", version)
    }

    if(!file.exists(destDir)){
        dir.create(destDir)
    }

    system(paste("cp", file.path(pkgPath, paste(substr(organism, 1, 1),
                           paste("*.", what, sep = ""), sep = "")), destDir))
}

writeDataHTML <- function(cin =
    "/scratch/homes/jzhang/madman/Rpacks/AnnBuilder/inst/scripts/data.html",
    cout = "/scratch/homes/jzhang/bioC/webstuff/data/data.html",
                          symbolValues = list(VERSION = "1.2.1")){
    copySubstitute(cin, cout, symbolValues)
}

checkNBuildPkgs <- function(pkgPath, check = TRUE){
    curDir <- getwd()
    setwd(pkgPath)
    pkgs <- list.files(pkgPath)
    # Get rid of any file with an extension
    pkgs <- pkgs[-grep("\\.", pkgs)]
    if(check){
        for(i in pkgs){
            system(paste("R CMD INSTALL", i))
            library(i, character.only = TRUE)
            # Check for probe missmatch for probe based envs
            pBased <- getPBasedEnv()
            for(k in pBased[-1]){
                if(any(!is.element(ls(get(paste(i, pBased[1], sep = ""))),
                                   ls(get(paste(i, k, sep = "")))))){
                    stop(paste("Checking for probe missmatch failed for",
                               pBased[1], "and", k))
                }
            }
            # checking for probe missmatch for non-probe based envs
            for(k in getNPEnvs()){
                probes <- ls(get(paste(i, "ACCNUM", sep = "")))
                toMatch <- unique(unlist(multiget(ls(get(paste(i, k,
                                sep = ""))), get(paste(i, k, sep = ""))),
                                         use.names = FALSE))
                toMatch <- toMatch[!is.na(toMatch)]
                if(any(!is.element(toMatch, probes))){
                    stop(paste("Checking for", paste(i, k, sep = ""),
                               "failed"))
                }
            }
        }
    }
    # If checking is successful, build the packages
    for(i in pkgs){
        for(j in getPkgNameByOrg(pkgPath, i)){
            system(paste("R CMD build -f", j))
        }
    }
    setwd(curDir)
}

getNPEnvs <- function(){
    return(c("ENZYME2PROBE", "GO2ALLPROBES", "GO2PROBE", "PATH2PROBE",
           "PMID2PROBE"))
}

getPBasedEnv <- function(){
    return(c("ACCNUM", "CHR", "CHRLOC" , "GENENAME", "GO", "HGID",
             "LOCUSID", "MAP", "PATH", "PMID", "SUMFUNC",
             "SYMBOL", "UNIGENE", "OMIM", "NP", "NM"))
}
