.packageName <- "tkWidgets"
# This function lists all the rda files in the data directory of a
# Bioconductor data package and thus allow users to explorer the data
# sets.
#
# Copyrihgt, 2003, J. Zhang, all rights reserved
#

DPExplorer <- function (pkgName = "",
                        title = "BioC Data Package Explorer"){
    on.exit(tkdestroy(base))

    quit <- FALSE
    dataName <- NULL
    keyName <- NULL
    keySelection <- NULL

    if(typeof(pkgName) != "character"){
        tkmessageBox(title = "Argument Error",
                         message = "Package name must be a character string",
                         icon = "error",
                         type = "ok")
        stop()
    }
    # multiget is copied form Biobase to remove the dependency of
    # tkWidgets on Biobase
    multiget <- function(x, pos=-1, envir=as.environment(pos), mode =
                         "any",inherits = TRUE, iffail=NA){
        lenx <- length(x)
        ans <- vector("list", length=lenx)
        if( ! is.environment(envir) )
            stop("envir argument is not an environment")
        options(show.error.messages = FALSE)
        on.exit(options(show.error.messages = TRUE))
        for(i in 1:lenx)
            if( is.list(x) )
                ans[[i]] <- try(get(x[[i]],pos,envir, mode, inherits))
            else
                ans[[i]] <- try(get(x[i],pos,envir, mode, inherits))
        options(show.error.messages = TRUE)
        on.exit(NULL)

        failfun <- function(x) {
            cx <- class(x)
            if( !is.null(cx) && cx == "try-error")
                TRUE
            else
                FALSE
        }
        failed <- sapply(ans, failfun)
        ans[failed] <- iffail

        names(ans) <- x
        ans
    }
    loadDP <- function(){
        pkgName <<- tclvalue(nameVar)
        if(pkgName == ""){
            tkmessageBox(title = "No Entry Error",
                         message = "Please enter a package name",
                         icon = "error",
                         type = "ok")
        }else{
            rdas <- loadDataPkg(pkgName)
            if(!is.null(rdas)){
                writeList(dataNameList, rdas)
            }
        }
    }
    activeLoadBut <- function(){
        tkconfigure(loadBut, state = "normal")
    }
    dataSelected <- function(){
        dataName <<- as.character(tkget(dataNameList,
                                       tkcurselection(dataNameList)))
        keys <- ls(get(dataName, pos = grep(pkgName, search())))
        writeList(keyList, keys)
    }
    keySelected <- function(){
        keyName <<- as.character(tkget(keyList,
                                       tkcurselection(keyList)))
        values <- get(keyName, get(dataName, pos = grep(pkgName, search())))
        writeList(valueList, values)
        tkconfigure(selectBut, state = "normal")
    }
    select <- function(){
        keySelection <<- unique(c(keySelection, keyName))
        writeList(selectionList, keySelection)
        tkconfigure(selectBut, state = "disabled")
        tkconfigure(clearBut, state = "normal")
    }
    drop <- function(){
        keySelection <<- keySelection[keySelection != keyName]
        writeList(selectionList, keySelection)
        tkconfigure(dropBut, state = "disabled")
    }
    clear <- function(){
        keySelection <<- NULL
        writeList(selectionList, NULL)
        tkconfigure(clearBut, state = "disabled")
    }
    selectionSelected <- function(){
        keyName <<- as.character(tkget(selectionList,
                                       tkcurselection(selectionList)))
        tkconfigure(dropBut, state = "normal")
    }
    cancel <- function(){
        quit <- TRUE
        tkdestroy(base)
    }
    finish <- function(){
        tkdestroy(base)
    }

    if(pkgName != ""){
        nameVar <- tclVar(pkgName)
    }else{
        nameVar <- tclVar("")
    }

    base <- getTopLevel(title)
    # Pack the top frame with an entry button for package name
    nameFrame <- tkframe(base)
    label <- tklabel(nameFrame, text = "Data Package: ")
    entry <- tkentry(nameFrame, width = 15, textvariable = nameVar)
    tkbind(entry, "<KeyPress>", activeLoadBut)
    tkbind(entry, "<Return>", loadDP)
    loadBut <- tkbutton(nameFrame, width = 8, text = "Load",
                        command = loadDP, state = "disabled")
    tkpack(label, side = "left")
    tkpack(entry, side = "left", expand = TRUE, fill = "x")
    tkpack(loadBut, side = "left")
    tkpack(nameFrame, expand = TRUE, fill = "x", pady = 5, padx = 5)
    # Pack the frame with lists for data
    dataFrame <- tkframe(base)
    # Frame for a list for names of data sets
    dataListFrame <- tkframe(dataFrame)
    dataLabel <- tklabel(dataListFrame, text = "Data:")
    tempFrame <- tkframe(dataListFrame)
    dataNameList <- makeViewer(tempFrame, vWidth = NULL, vHeight = NULL,
                        hScroll = FALSE, vScroll = TRUE,
                        what = "list", side = "bottom")
    tkconfigure(dataNameList, exportselection = FALSE)
    tkbind(dataNameList, "<B1-ButtonRelease>", dataSelected)
    tkpack(dataLabel, side = "top")
    tkpack(tempFrame, side = "bottom", expand = TRUE, fill ="both")
    tkpack(dataListFrame, side = "left", expand = TRUE, fill = "both")
    # Frame for a list for names of keys
    keyListFrame <- tkframe(dataFrame)
    keyLabel <- tklabel(keyListFrame, text = "Keys:")
    tempFrame <- tkframe(keyListFrame)
    keyList <- makeViewer(tempFrame, vWidth = NULL, vHeight = NULL,
                        hScroll = FALSE, vScroll = TRUE,
                        what = "list", side = "bottom")
    tkconfigure(keyList, exportselection = FALSE)
    tkbind(keyList, "<B1-ButtonRelease>", keySelected)
    tkpack(keyLabel, side = "top")
    tkpack(tempFrame, side = "bottom", expand = TRUE, fill = "both")
    tkpack(keyListFrame, side = "left", expand = TRUE, fill = "both")
    # Frame for a list for displaying values of selected keys
    valueFrame <- tkframe(dataFrame)
    valueLabel <- tklabel(valueFrame, text = "Value(s)")
    temp <- tkframe(valueFrame)
    valueList <- makeViewer(temp, vWidth = NULL, vHeight = NULL,
                        hScroll = FALSE, vScroll = TRUE,
                        what = "list", side = "bottom")
    selectBut <- tkbutton(valueFrame, text = "Select Key", width = 16,
                          command = select, state = "disabled")
    dropBut <- tkbutton(valueFrame, text = "Drop Key", width = 16,
                         command = drop, state = "disabled")
    clearBut <- tkbutton(valueFrame, text = "Clear Selection", width = 16,
                          comman = clear, state = "disabled")
    tkpack(clearBut, side = "bottom", expand = TRUE, fill = "x")
    tkpack(dropBut, side = "bottom", expand = TRUE, fill = "x")
    tkpack(selectBut, side = "bottom", expand = TRUE, fill = "x")
    tkpack(temp, side = "bottom", expand = TRUE, fill = "both")
    tkpack(valueLabel, side = "bottom")
    tkpack(valueFrame, side = "left", expand = TRUE, fill = "both")
    # Frame for a list for selected keys
    selectionListFrame <- tkframe(dataFrame)
    selectionLabel <- tklabel(selectionListFrame, text = "Selected keys:")
    viewFrame <- tkframe(selectionListFrame)
    selectionList <- makeViewer(viewFrame, vWidth = NULL, vHeight = NULL,
                        hScroll = FALSE, vScroll = TRUE,
                        what = "list", side = "bottom")
    tkbind(selectionList, "<B1-ButtonRelease>", selectionSelected)
    tkpack(selectionLabel, side = "top")
    tkpack(viewFrame, side = "bottom", expand = TRUE, fill = "both")
    tkpack(selectionListFrame, side = "left", expand = TRUE, fill = "both")
    # Pack the frame for data lists
    tkpack(dataFrame, expand = TRUE, fill = "both", padx = 5)
    # Frame for the buttons
    butFrame <- tkframe(base)
    cancelBut <- tkbutton(butFrame, text = "Cancel", width = 8,
                          command = cancel)
    finishBut <- tkbutton(butFrame, text = "Finish", width = 8,
                          comman = finish)
    tkgrid(cancelBut, finishBut, padx = 10)
    tkpack(butFrame, side = "top", expand = TRUE, fill = "x")

    if(pkgName != ""){
        loadDP()
        activeLoadBut()
    }

    tkwait.window(base)

    if(quit){
        return(NULL)
    }else{
        if(is.null(keySelection)){
            return(NULL)
        }else{
            if(length(keySelection) == 1){
                return(get(keySelection,
                           get(dataName, pos = grep(pkgName, search()))))
            }else{
                return(multiget(keySelection,
                           get(dataName, pos = grep(pkgName, search()))))
            }
        }
    }
}

getTopLevel <- function(title){
    base <- tktoplevel()
    tktitle(base) <- title
    return(base)
}


loadDataPkg <- function(pkgName){
    where <- grep(pkgName, search())
    if(length(where) == 0){
        tkmessageBox(title = "Invalid Package Name",
                         message = paste("Package", pkgName,
                         "is not valid or has not been loaded yet!"),
                     icon = "error",
                     type = "ok")
        return(NULL)
    }else{
        rdas <- gsub("\\.rda$", "", ls(where))
        rdas <- setdiff(rdas, c(pkgName,
                                     paste(pkgName, "QC", sep = "")))
        return(rdas)
    }
}

# This function takes a path and returns a vector containing names of
# files. A platform file separator will be appended to the end of
# subdirectory names.
#

appendSepDir <- function(path = ".") {

     toCheck <- list.files(path)

     if (length(toCheck) == 0) return("")

     isd <- file.info(file.path(path, toCheck))
     # Separate dirs and files into two groups for better presentation
     dirs <- paste(toCheck[isd$isdir], .Platform$file.sep, sep = "")
     toCheck <- c(dirs, toCheck[!isd$isdir])
     return(toCheck)
}


















# This function reads the formal arguments of an R function and then
# generates an XML document containing those arguments
#
# Copyright 2002, J. Zhang, all rights reserved
#

args2XML<- function(fun, xml.name = "", full.names = NULL,
                      priority = NULL){

    if(xml.name == ""){
        xml.name = paste(deparse(substitute(fun)), "arg.xml", sep = "")
    }

    writeXML <- function(text){
        write(text, file = xml.name, append = TRUE)
    }

    args <- formals(fun)
    write(paste("<!DOCTYPE tkWidgets: SYSTEM \"http://",
                "www.bioconductor.org/datafiles/dtds/args.dtd\">",
                sep = ""), file = xml.name)
    writeXML(paste("<tkWidgets:Arguments xmlns:AnnBuilder='http://",
                "www.bioconductor.org/tkWidgets/'>", sep = ""))

    for(i in 1:length(args)){
        writeXML("<tkWidgets:Argument>")
        writeXML(paste("<tkWidgets:ArgName value = \"",
                       names(args[i]), "\"/>", sep = ""))
        writeXML(paste("<tkWidgets:ArgVal value = \"", args[i],
                       "\"/>", sep = ""))
        writeXML(paste("<tkWidgets:FullName value = \"",
                       ifelse(is.null(full.names), "", full.names[i]),
                       "\" />", sep = ""))
        writeXML(paste("<tkWidgets:Priority value = \"",
                       ifelse(is.null(priority), "", priority[i]),
                       "\" />", sep = ""))
        write("</tkWidgets:Argument>")
    }

    writeXML("</tkWidgets:Arguments>")

    return(xml.name)
}
# This function takes a argument list returned by formals(R function)
# and creates a widget that allows users to manipulate the values
# using an interface.
#
# Copyright 2002, J. Zhang, all rights reserved
#

argsWidget <- function(argsList, defaultNames = c("OK", "Cancel"),
                       inst = ""){

#    if(is.pairlist(argsList)){
        # as.list does not work.
#        temp <- list()
#        for(i in names(argsList)){
#            temp[[i]] <- argsList[[i]]
#        }
#        argsList <- temp
#    }

    # Arguments that are functions
    funcs <- getSymbol(argsList)
    # Conver functions to characters
    argsList <- sapply(funcs2Char(argsList, funcs), formatArg)
    # Constructs the interface
    # Sets the working environment
    PWEnv <- new.env(hash = TRUE, parent = NULL)
    pWidgets <- getPWidget(argsList, PWEnv, inst)
    widget <- widget(wTitle = "BioC Arguments Widget", pWidgets,
                     funs = list(), preFun = function() {},
                     postFun = function() {}, env = PWEnv,
                     defaultNames = defaultNames)
    if(!is.null(widget)){
        # Returns the input values
        for(i in names(argsList)){
            if(any(i == names(funcs))){
                argsList[[i]] <-
                    get(wValue(pWidgets(widget)[[i]][["entry"]])[[1]])
            }else{
                argsList[[i]] <-
                    formatArg(wValue(pWidgets(widget)[[i]][["entry"]])[[1]])
            }
        }
        return(argsList)
    }else{
        return(widget)
    }
#    return(argsList)
}
# Creates the primary widget list for building the interface
getPWidget <- function(argsList, PWEnv, inst = ""){
    # Figures out the width for lables
    lWidth <- max(nchar(names(argsList)))
    pWidgets <- list()

    if(inst != ""){
        label <- label(wName = "label", wValue = inst,
                       wWidth = nchar(inst), wEnv = PWEnv)
        tempList <- list()
        tempList[["label"]] <- label
        pWidgets[["inst"]] <- tempList
    }

    for(i in names(argsList)){
        tempList <- list()
        # Creates radio buttons with TRUE and FALSE if the default
        # value for an argument is either a TRUE or FALSE
        if(is.logical(argsList[[i]]) && !is.na(argsList[[i]])){
            eval(substitute(j <- radioButton(wName = j,
                     wValue = c("TRUE" = TRUE,"FALSE" = FALSE),
                     wEnv = PWEnv), list(j = i)))
        }else{
            eval(substitute(j <- entryBox(wName = j,
                                 wValue = argsList[[j]],
                                 wWidth = 15, wEnv = PWEnv), list(j = i)))

        }
        label <- label(wName = "label", wValue = i,
                                             wWidth = lWidth, wEnv = PWEnv)
        tempList[["label"]] <- label
        tempList[["entry"]] <- get(i)
        pWidgets[[i]] <- tempList
    }
    return(pWidgets)
}

# This function fomats the arguments obtained from a widget
formatArg <- function(toFormat){

    # Turns off any warnings when checking for NULL, NA, and boolean
    options(warn = -1)
    if(is.null(toFormat)){
        options(warn = 0)
        return(toFormat)
    }
    if(is.na(toFormat)){
        options(warn = -1)
        return("NA")
    }
    if(is.logical(toFormat)){
        options(warn = 0)
        return(toFormat)
    }
    if(toFormat == "TRUE"){
        return(TRUE)
    }
    if(toFormat == "FALSE"){
        return(FALSE)
    }
    options(warn = 0)
    if(toFormat == ""){
        return(toFormat)
    }
    # expression and negative numbers can be "language"
    if(is.language(toFormat)){
        return(formula(toFormat))
    }
    options(warn = -1)
    temp <- as.numeric(toFormat)
    options(warn = 0)
    if(is.na(temp)){
        return(getTrueNullNa(toFormat))
    }else{
        return(temp)
    }
}

# All functions are of type "symbol", "builtin", in the list returned
# by formals
getSymbol <- function(args){
    temp <- sapply(args, typeof)
    temp <- args[names(
           temp[temp == "symbol" | temp == "closure" | temp == "builtin"])]
    temp <- temp[temp != ""]
    return(temp)
}

funcs2Char <- function(args,funcs){
    for(i in names(funcs)){
        args[[i]] <- as.character(funcs[i])
    }
    return(args)
}

getTrueNullNa <- function(toFormat){
     switch(tolower(toFormat),
                   "t" = ,
                   "true" = return(TRUE),
                   "f" = ,
                   "false" = return(FALSE),
                   "na" = return(NA),
                   "null" = return(NULL),
                   return(toFormat))
}

# This function creates a widget with a top frame that will be filled
# with a widget passed as one of the arguments and bottom frame with
# buttons.
#
# Copyright 2002, J. Zhang. All right reserved.
#
dataViewer <- function(data, caption = "", save = TRUE){
    on.exit(exit())
    exit <- function() tkdestroy(base)

    base <- tktoplevel()
    tktitle(base) <- "BioC tkWidgets"

    if(caption != ""){
        label <- tklabel(base, text = caption)
        tkpack(label)
    }

    boxFrame <- tkframe(base)

    # Insert data into the canvas
    innerFrame <- makeViewer(boxFrame,
                             vScroll = TRUE, side = "top",
                             hScroll = TRUE, what = "canvas")

    dataFrame <- tkframe(innerFrame)
    for(i in 1:ncol(data)){
        tempFrame <- tkframe(dataFrame)
        if(!is.null(colnames(data))){
            tempName <- tkbutton(tempFrame, text = colnames(data)[i],
                                 width = 0)
            tkpack(tempName, expand = TRUE, fill = "x")
        }
        tempList <- tklistbox(tempFrame, width = 0,
                              height = 0, background = "white")
        writeList(tempList, as.vector(data[,i]))
        tkpack(tempList, side = "left", expand = TRUE, fill = "both")
        tkpack(tempFrame, side = "left", expand = TRUE, fill = "both")
    }

    tkcreate(innerFrame, "window", 0, 0, anchor = "nw", window = dataFrame)

    tkpack(boxFrame, side = "top", expand = TRUE, fill = "both")

    botFrame <- tkframe(base)
    if(save){
        save <- function(){
            name <- tclvalue(tkgetSaveFile())
            write.table(data, name, quote = FALSE, sep = "\t",
                        row.names = FALSE, col.names = FALSE)
            exit()
        }
        saveBut <- tkbutton(botFrame, text = "Save", width = 8,
                            command = save)
        tkpack(saveBut, side = "left")
    }
    exitBut <- tkbutton(botFrame, text = "Exit", width = 8, command = exit)
    tkpack(exitBut, side = "left")
    tkpack(botFrame, side = "top")

    tkwait.window(base)
}


# This function provides the interface for the inputs for making the
# connection to a database.
#
# Copyright 2002, J. Zhang, all rights reserved
#

dbArgsWidget <- function(){

    # Sets the working environment
    PWEnv <- new.env(hash = TRUE, parent = NULL)
    # Defines the widget components
    label1 <- label(wName = "label1", wValue = "Database: ", wWidth = 15,
                    wEnv = PWEnv)
    db <- entryBox(wName = "db", wWidth = 20, wEnv = PWEnv)
    label2 <- label(wName = "label2", wValue = "User name: ", wWidth = 15,
                    wEnv = PWEnv)
    un <- entryBox(wName = "un", wWidth = 20, wEnv = PWEnv)
    label3 <- label(wName = "label3", wValue = "Password: ", wWidth = 15,
                    wEnv = PWEnv)
    pw <- entryBox(wName = "pw", wWidth = 20, wEnv = PWEnv)
    label4 <- label(wName = "label4", wValue = "Host: ", wWidth = 15,
                    wEnv = PWEnv)
    hs <- entryBox(wName = "hs", wValue = "localhost", wWidth = 20,
                   wEnv = PWEnv)
    label5 <- label(wName = "label5", wValue = "Table name: ", wWidth = 15,
                    wEnv = PWEnv)
    tn <- entryBox(wName = "tn", wWidth = 20, wEnv = PWEnv)
    # Makes a list with the layout defined
    pWidgets <- list(dbName = list(label1 = label1, db = db),
                     tableName = list(lable5 = label5, tn = tn),
                     userName = list(label2 = label2, un = un),
                     password = list(label3 = label3, pw = pw),
                     host = list(label4 = label4, hs = hs))
    # Constructs the interface
    widget <- widget(wTitle = "BioC DB Inputs Widget", pWidgets,
                     funs = list(), preFun = function() {},
                     postFun = function() {}, env = PWEnv)

    # Returns the input values
    if(wValue(pWidgets(widget)[["dbName"]][["db"]]) == ""){
       stop("Database name can not be an empty string!")
    }
    inputs <- list(dbname = wValue(pWidgets(widget)[["dbName"]][["db"]]),
                   host = wValue(pWidgets(widget)[["host"]][["hs"]]))
    if(wValue(pWidgets(widget)[["userName"]][["un"]]) != ""){
        inputs[["user"]] <- wValue(pWidgets(widget)[["userName"]][["un"]])
    }
    if(wValue(pWidgets(widget)[["password"]][["pw"]]) != ""){
        inputs[["password"]] <- wValue(pWidgets(widget)[["password"]][["pw"]])
    }
    if(wValue(pWidgets(widget)[["tableName"]][["tn"]]) != ""){
        inputs[["tableName"]] <- wValue(pWidgets(widget)[["tableName"]][["tn"]])
    }

    return(inputs)
}
# This widget allows users to pick filters in the order they are going
# to be used to filer genes and set the parameters for
# each filter.
#
# Copyright 2003, J. Zhang. All rights reserved.
#

eSetFilter <- function(eSet){
    require(Biobase) || stop(paste("eSetFilter requires the Biobase",
                                   "package. Please have if installed"))
    require(genefilter) || stop(paste("eSetFilter requires the genefilter",
                                "package. Please have it installed"))

    descList <- getFuncDesc()

    buildGUI <- function(){
        END <<- FALSE

        selectedNames <- NULL
        filterWithArgs <- list()

        setFilter <- function(){
            currentFilter <- as.character(tkget(filters,
                                             (tkcurselection(filters))))
            args <- setESetArgs(currentFilter)
            if(!is.null(args)){
                expression <- paste(currentFilter, "(",
                    paste(names(args), args, sep = "=", collapse = ","),
                    ")", sep = "")
                filterWithArgs[[currentFilter]] <<- eval(parse(text =
                                                               expression))
                selectedNames <<- unique(c(selectedNames, currentFilter))
                writeList(pickedF, selectedNames)
                tkconfigure(selectBut, state = "disabled")
            }
        }
        cancel <- function(){
            tkdestroy(base)
        }
        finish <- function(){
            END <<- TRUE
            tkdestroy(base)
        }
        viewFilter <- function(){
            currentFilter <- as.character(tkget(filters,
                                             (tkcurselection(filters))))
            tkconfigure(description, state = "normal")
            writeText(description, descList[[currentFilter]])
            tkconfigure(description, state = "disabled")
            tkconfigure(selectBut, state = "normal")
        }
        pickedSel <- function(){
            tkconfigure(remBut, state = "normal")
        }
        remove <- function(){
             filter <- as.character(tkget(pickedF,
                                             (tkcurselection(pickedF))))
             selectedNames <<- setdiff(selectedNames, filter)
             writeList(pickedF, selectedNames)
             tkconfigure(remBut, state = "disabled")

        }

        base <- tktoplevel()
        tktitle(base) <- "BioC Filter Master"
        # Pack the top frame with a brief description
        introText <- tktext(base, width = 30, height = 4, wrap = "word")
        text <- paste("Bioconductor's gene filtering functons are",
                      "listed below. Select one from the list to view the",
                      "description and formal arguments for each filter.",
                      "A filter can be selected to the set of filters",
                      "for filtering genes using the select button.")
        writeText(introText, text)
        tkconfigure(introText, state = "disabled")
        tkpack(introText, expand = FALSE, fill  = "both", padx = 5)
        # Pack a frame with a list box for selected filters and
        # buttons manipulate the selected filters
        infoFrame <- tkframe(base)
        filterFrame <- tkframe(infoFrame)
        tkpack(tklabel(filterFrame, text = "Filters"), expand = FALSE,
               fill = "x")
        listFrame <- tkframe(filterFrame)
        filters <- makeViewer(listFrame, vHeight = 10, vWidth = 12,
                                vScroll = TRUE, hScroll = TRUE,
                                what = "list")
        tkbind(filters, "<B1-ButtonRelease>", viewFilter)
        tkbind(filters, "<Double-Button-1>", setFilter)
        writeList(filters, getFilterNames())
        tkpack(listFrame, expand = TRUE, fill = "both")
        selectBut <- tkbutton(filterFrame, text = "Select",
                              command = setFilter, state = "disabled")
        tkpack(selectBut, expand = FALSE, fill = "x")
        tkpack(filterFrame, side = "left", expand = FALSE, fill = "both")
        descFrame <- tkframe(infoFrame)
        tkpack(tklabel(descFrame, text = "Description"), expand = FALSE,
               fill = "x")
        dListFrame <- tkframe(descFrame)
        description <- makeViewer(dListFrame, vHeight = 10, vWidth = 30,
                                vScroll = TRUE, hScroll = TRUE,
                                what = "text")
        tkconfigure(description, wrap = "word", state = "disabled")
        tkpack(dListFrame, expand = TRUE, fill = "both")
        tkpack(descFrame, side = "left", expand = TRUE, fill = "both")
        selFrame <- tkframe(infoFrame)
        tkpack(tklabel(selFrame, text = "Selected"),
               expand = FALSE, fill = "x")
        selFFrame <- tkframe(selFrame)
        pickedF <- makeViewer(selFFrame, vHeight = 10, vWidth = 12,
                                vScroll = TRUE, hScroll = TRUE,
                                what = "list")
        tkbind(pickedF, "<B1-ButtonRelease>", pickedSel)
        tkbind(pickedF, "<Double-Button-1>", remove)
        tkpack(selFFrame, expand = TRUE, fill = "both")
        remBut <- tkbutton(selFrame, text = "Remove", command = remove,
                           state = "disabled")
        tkpack(remBut, expand = FALSE, fill = "x")
        tkpack(selFrame, expand = FALSE, fill = "both")
        tkpack(infoFrame, expand = TRUE, fill = "both", padx = 5)
        # Pack the bottom frame with cancel and finish buttons
        endFrame <- tkframe(base)
        cancelBut <- tkbutton(endFrame, width = 8, text = "Cancel",
                           command = cancel)
        tkpack(cancelBut, side = "left", expand = TRUE, fill = "x",
               padx = 10)
        finishBut <- tkbutton(endFrame, width = 8, text = "finish",
                           command = finish)
        tkpack(finishBut, side = "left", expand = TRUE, fill = "x",
               padx = 10)
        tkpack(endFrame, expand = FALSE, fill = "x", pady = 5)

        showESet(eSet)
        tkwait.window(base)

        if(END){
            tempList <- list()
            for(i in  selectedNames){
                tempList[[i]] <- filterWithArgs[[i]]
            }
            return(tempList)
        }else{
            return(NULL)
        }
    }

#    if(class(eSet) != "exprSet"){
#        tkmessageBox(title = "Wrong Data type",
#                     message = paste("The object passed is not an object",
#                               "of exprSet"),
#                     icon = "warning",
#                     type = "ok")
#        stop()
#    }else{
    filters <- buildGUI()
    if(!is.null(filters)){
        filters <- filterfun(unlist(filters))
        return(genefilter(exprs(eSet), filters))
    }else{
        return(NULL)
    }
#    }
}

getFilterNames <- function(){
    return(sort(c("Anova", "coxfilter", "cv", "gapFilter", "kOverA",
    "maxA", "pOverA", "ttest")))
}

getFuncDesc <- function(lib = "genefilter", funcs = getFilterNames()){
    descList <- list()

    lines <- getRdAsText(lib)
    for(i in funcs){
        rd <- lines[grep(paste("\\\\name{", i, "}", sep = ""), lines)]
        desc <- parseDesc(rd)
        args <- parseArgs(rd)
        if(length(args) > 0){
            temp <- "\n\nArguments:"
            for(j in names(args)){
                temp <- c(temp, paste(j, "-", args[[j]]))
            }
            args <- paste(temp, sep = "", collapse = "\n")
        }
        descList[[i]] <- paste(desc, args, sep = "", collapse = "")
    }
    return(descList)
}

getRdAsText <- function(lib){
    fileName <- file.path(.path.package(lib), "man",
                          paste(lib, ".Rd", sep = ""))
    lines <- readLines(fileName)

    lines <- paste(lines, sep = "", collapse = " ")
    lines <- unlist(strsplit(lines, "\\\\eof"))
    return(lines)
}

parseDesc <- function(text){
    descRExp <- ".*\\\\description{(.*)}.*\\\\usage{.*"
    text <- gsub(descRExp, "\\1", text)
    text <- gsub("(\\\\[a-zA-Z]*{|})", "", text)
    return(text)
}

parseArgs <- function(text){
    argsList <- list()
    text <- gsub(".*\\\\arguments{(.*)}.*\\\\details{.*", "\\1", text)
    text <- gsub(".*\\\\arguments{(.*)}.*\\\\value{.*", "\\1", text)
    text <- unlist(strsplit(text, "\\\\item{"))
    text <- gsub("(\\\\[a-zA-Z]*{|})", "", text)
    for(i in text){
        i <- unlist(strsplit(i, "{"))
        if(length(i) > 1){
            argsList[[i[1]]] <- i[2]
        }
    }
    return(argsList)
}

showESet <- function(eSet){
    end <- function(){
        tkdestroy(base)
    }
    if(!isESet(eSet)){
        stop()
    }
    colNRow <- dim(exprs(eSet))
    vl <- varLabels(eSet)
    text <- c(paste("Genes: ", colNRow[1]),
              paste("Samples: ", colNRow[2], sep = ""),
              "Variable labels:",
              paste(names(vl), ": ", vl[1:length(vl)], sep = ""))

    base <- tktoplevel()
    tktitle(base) <- "BioC exprSet viewer"
    dataDescFrame <- tkframe(base)
    data <- makeViewer(dataDescFrame, vHeight = 10, vWidth = 25,
                       vScroll = TRUE, hScroll = TRUE,
                       what = "list")
    writeList(data, text)
    tkpack(dataDescFrame, expand = TRUE, fill = "both")
    endBut <- tkbutton(base, text = "Finish", command = end)
    tkpack(endBut, expand = FALSE, fill = "x", pady = 5)
}

setESetArgs <- function(filter){
    on.exit(tkdestroy(base))

    cancel <- function(){
        tkdestroy(base)
    }
    end <- function(){
        END <<- TRUE
        tkdestroy(base)
    }
    END <- FALSE
    argsVar <- list()
    desc <- list()
    entries <- list()
    ftFun <- list()

    args <- getRdAsText("genefilter")
    args <- args[grep(paste("\\\\name{", filter, "}", sep = ""), args)]
    args <- parseArgs(args)
    argValues <- formals(filter)

    base <- tktoplevel()
    tktitle(base) <- "BioC Filter Argument input"

    tkgrid(tklabel(base, text = "Arguments"),
           tklabel(base, text = "Descriptions"),
           tklabel(base, text = "Values"))
    for(i in names(args)){
        argsVar[[i]] <- tclVar(as.character(argValues[[i]]))
        tempFrame <- tkframe(base)
        desc[[i]] <- makeViewer(tempFrame, vHeight = 3, vWidth = 55,
                                vScroll = FALSE, hScroll = FALSE,
                                what = "text")
        writeText(desc[[i]], args[[i]])
        tkconfigure(desc[[i]], wrap = "word", state = "disabled")
        entries[[i]] <- tkentry(base, textvariable = argsVar[[i]],
                                    width = 10)
        tkgrid(tklabel(base, text = i), tempFrame, entries[[i]])
        if(any(as.character(argValues[[i]]) == c("FALSE", "TRUE"))){
             ftFun[[i]] <- function(){}
             body <- list(as.name("{"),
                          substitute(eval(if(tclvalue(argsVar[[j]]) ==
                                                      "TRUE"){
                              writeList(entries[[j]], "FALSE")}else{
                                  writeList(entries[[j]], "TRUE")}),
                                     list(j = i)))
             body(ftFun[[i]]) <- as.call(body)
             tkbind(entries[[i]],"<B1-ButtonRelease>", ftFun[[i]])
        }
        tkgrid.configure(tempFrame, sticky = "eswn")
    }

    butFrame <- tkframe(base)
    canBut <- tkbutton(butFrame, text = "cancel", width = 8,
                       command = cancel)
    endBut <- tkbutton(butFrame, text = "Finish", width = 8,
                       comman = end)
    tkpack(canBut, side = "left", expand = FALSE, fill = "x")
    tkpack(endBut, side = "left", expand = FALSE, fill = "x")
    tkgrid(butFrame, columnspan = 3)

    tkwait.window(base)
    if(END){
        for(i in names(argValues)){
            argValues[[i]] <- formatArg(tclvalue(argsVar[[i]]))
        }
        return(argValues)
    }else{
        return(NULL)
    }
}

isESet <- function(eSet){
    if(missing(eSet) || class(eSet) != "exprSet"){
        tkmessageBox(title = "Input Error",
                     message = paste("filterMaster has to take",
                     "an object of exprSet"),
                     icon = "warning",
                     type = "ok")
        return(FALSE)
    }else{
        return(TRUE)
    }
}
# This function provides the interface to browse files in the system
# and returns all or a given number of selected file names.
#

fileBrowser <- function (path = "", testFun = function(x) TRUE,
                         prefix = NULL, suffix = NULL,
                         textToShow = "Select file(s)", nSelect = -1)
{
    on.exit(end())

    LABELFONT1 <- "Helvetica 12 bold"
    LABELFONT2 <- "Helvetica 11"
    BUTWIDTH <- 8
    CANWIDTH <- 500
    CANHEIGHT <- 420
    BOXHEIGHT <- 16
    BOXWIDTH <- 29
    OFFSET <- 10
    LABELHEIGHT <- 20

#    currentNode <- NULL
#    nodes <- NULL
    fileSelected <- NULL
#    currentDir <- NULL
    currentFile <- NULL
    selIndex <- NULL
    fileIndex <- NULL
    nextY <- NULL

    # Close the window. Warn users if the number of selected files is wrong
    end <- function(){
        if(nSelect == -1){
            tkdestroy(base)
        }else{
            if(nSelect != length(fileSelected)){
                tkmessageBox(title = "Wrong number", message =
                       paste("You can only select", nSelect, "file(s)"),
                       icon = "warning", type = "ok")
            }else{
                tkdestroy(base)
            }
        }
        setwd(oldDir)
    }
    # When a user double clicks a file name in the list box for file names,
    # remembers the file name and path for that name and write the
    # contents to the list box if the name is a directory.
    inList <- function(){
        selectedObj <- as.character(tkget(listView,
                                          (tkcurselection(listView))))
        if(regexpr(.Platform$file.sep, selectedObj[1]) >= 1){
      setwd(file.path(getwd(), selectedObj))
            write2List(listView,
                     pickFiles(appendSepDir(getwd()), testFun,
                               prefix, suffix))
            writeCap(getwd())
            if(length(unlist(strsplit(getwd(), getSep()))) >= 1)
                tkconfigure(upBut, state = "normal")
            tkconfigure(selectBut, state = "disabled")
        }else{
            selIndex <<- unlist(strsplit(
                                as.character(tkcurselection(listView)), ""))
            selectAFile()
        }
    }
    # Remember the index of the file name when a user single clicked a
    # file name in the box for file names.
    selInDir <- function (){
        fileIndex <<- NULL
        tkconfigure(selectBut, state = "normal")
        selIndex <<- unlist(strsplit(
                            as.character(tkcurselection(listView)), " "))
    }
    # Remember the index of the file name when a user single clicked a
    # file name in the box for selected file names.
    selInSelection <- function(){
        selIndex <<- NULL
        if(as.character(tkcurselection(selectView))[1] != ""){
            fileIndex <<-
              unlist(strsplit(as.character(tkcurselection(selectView)), " "))
            tkconfigure(remBut, state = "normal")
        }else
            tkconfigure(remBut, state = "disabled")
    }
    # Write a selected file name to the list box for selected files
    # when a user clicked the select button after clicking a file name
    #in the box for file names.
    selectAFile <- function(){
        if(length(selIndex) > 0){
            for(i in selIndex){
                selObj <- as.character(tkget(listView, i))
                    fileSelected <<- c(fileSelected,
                                    file.path(getwd(), selObj))
            }
            fileSelected <<- unique(fileSelected)
            writeToView(selectView, fileSelected)
            selIndex <<- NULL
        }
    }
    # Clear the selected file names
    clearSelection <- function(){
        fileSelected <<- NULL
        writeToView(selectView, NULL)
    }
    # Remove a clicked file name from the box for selected file names
    # after a user clicked the remove button.
    dropSelection <- function(){
        toRemove <- NULL
        if(length(fileIndex) > 0){
            for(i in fileIndex)
                toRemove <- c(toRemove, -(as.numeric(i) + 1))
            fileSelected <<- fileSelected[toRemove]
            writeToView(selectView, fileSelected)
            fileIndex <<- NULL
         }
    }
    # Remember the path the browser has traveled.
#    doPath <- function(){
# nodes <<- unlist(strsplit(path, .Platform$file.sep))
#        currentNode <<- length(nodes)
#    }

    write2List <- function (listBox, toWrite){
        writeList(listBox, toWrite, clear = TRUE)
    }

    # Move the browser one level up the directory path
    goUp <- function(){
        tkconfigure(selectBut, state = "disabled")
        sep <- getSep()
        temp  <- unlist(strsplit(getwd(), sep))
        if(length(temp) >= 2){
            if(length(temp) > 2){
                temp <- paste(temp[1:(length(temp) - 1)],
                                  sep = "", collapse = .Platform$file.sep)
                if(length(temp) == 3 && .Platform$OS.type == "unix"){
                    tkconfigure(upBut, state = "disabled")
                }
            }else{
                temp <- paste(temp[1], sep, sep = "", collapse = "")
                tkconfigure(upBut, state = "disabled")
            }
            setwd(temp)
            write2List(listView, pickFiles(appendSepDir(getwd()), testFun,
                                         prefix, suffix))
            writeCap(getwd())
        }
    }
    # Returns the file separator
    getSep <- function(){
            return(.Platform$file.sep)
    }
    # Write the content of toWrite to a list given box
    writeToView <- function(aView, toWrite){
    tkdelete(aView, 0, "end")
        if(!is.null(toWrite)){
            for (i in toWrite){
                if(substr(i, nchar(i), (nchar(i) + 1))
                      == .Platform$file.sep)
                    i <- substr(i, 0, (nchar(i) - 1 ))
                tkinsert(aView, "end",
                     gsub(paste(".*", .Platform$file.sep, "(.*)",
                                sep = ""), "\\1", i))
            }
        }
        if(length(toWrite) > 0)
            tkconfigure(clearBut, state = "normal")
        else{
            tkconfigure(clearBut, state = "disabled")
            tkconfigure(remBut, state = "disabled")
        }
    }
    # Refresh the path shown on the widget
    writeCap <- function(toWrite)
  tkconfigure(caption, text = toWrite)

    oldDir <- getwd()
    if(path != "")
        setwd(path)

#    doPath()


    ## Set up the interface
    base <- tktoplevel()
    tktitle(base) <- paste("File Browser")
    # Writes the directory name
    topFrame <- tkframe(base)
    instruct <- tklabel(topFrame, text = textToShow, font = LABELFONT1)
    dir <- tklabel(topFrame, text = "Directory: ", font = LABELFONT2)
    caption <- tklabel(topFrame, text = getwd(), font = LABELFONT2)
    tkgrid(instruct, columnspan = 2)
    tkgrid(dir, caption)
    tkgrid(topFrame, columnspan = 2, padx = 10)
    # Put the list box for file names in a directory and the
    # associated buttons
    leftFrame <- tkframe(base)
    dirLabel <- tklabel(leftFrame, text = "Files in directory",
                        font = "Helvetica 11")
    tkgrid(dirLabel, columnspan = 2)
    dirLFrame <- tkframe(leftFrame)

    listView <- makeViewer(dirLFrame, vWidth = BOXWIDTH,
                         vHeight = BOXHEIGHT)
    tkgrid(dirLFrame, columnspan = 2)
    upBut <- tkbutton(leftFrame, text = "Up", width = BUTWIDTH,
          command = goUp)
    selectBut <- tkbutton(leftFrame, text = "Select >>", width = BUTWIDTH,
          state = "disabled", command = selectAFile)
    tkgrid(upBut, selectBut)
    tkgrid.configure(upBut, sticky = "e")
    tkgrid.configure(selectBut, sticky = "w")
    tkconfigure(listView, selectmode = "extended", font = LABELFONT2)
    tkbind(listView, "<Double-Button-1>", inList)
    tkbind(listView, "<B1-ButtonRelease>", selInDir)
    write2List(listView, pickFiles(appendSepDir(getwd()), testFun,
                                 prefix, suffix))
    # Put the list box for selected file names and the associated buttons
    rightFrame <- tkframe(base)
    selLabel <- tklabel(rightFrame, text = "Files selected",
                        font = "Helvetica 11")
    tkgrid(selLabel, columnspan = 2)
    selLFrame <- tkframe(rightFrame)
    selectView <- makeViewer(selLFrame, vWidth = BOXWIDTH,
                           vHeight = BOXHEIGHT)
    tkgrid(selLFrame, columnspan = 2)
    tkconfigure(selectView, selectmode = "extended", font = LABELFONT2)
    tkbind(selectView, "<B1-ButtonRelease>", selInSelection)
    remBut <- tkbutton(rightFrame, text = "<< Remove", width = BUTWIDTH,
          state = "disabled", command = dropSelection)
    clearBut <- tkbutton(rightFrame, text = "Clear", width = BUTWIDTH,
          state = "disabled", command = clearSelection)
    tkgrid(remBut, clearBut)
    tkgrid.configure(remBut, sticky = "e")
    tkgrid.configure(clearBut, sticky = "w")

    tkgrid(leftFrame, rightFrame)
    # Put the end button
    endBut <- tkbutton(base, text = "Finish", width = BUTWIDTH,
           command = end)
    tkgrid(endBut, columnspan = 2)

    tkwait.window(base)

    return(fileSelected)
}















# This function provides preliminary visual data inport.
#
# Copyright 2002, J. Zhang, all rights reserved.

fileWizard <- function(filename = "", fun = read.table, file = "file",
                       basic = c("header", "sep")){
    on.exit(end)

    BOLD12 <- "Helvetica 12 bold"
    NORMAL11 <- "Helvetica 11"
    BOXW <- 15

    args <- formals(fun)
    rest <- setdiff(names(args), c(file, basic))
    boxes <- vector("list")
    dList <- vector("list")
    fileRead <- NULL

    # Destroy the window
    end <- function(){
         tkdestroy(top)
    }
    # Write toPop to a given entry box
#    writeEntry <- function(name, toPop){
#        tkdelete(name, "0", "end")
#        tkinsert(name, "end", toPop)
#    }
    # Initialize the list box if fileWizard is called with a file name
    init <- function(){
        if(!is.null(filename) && !is.na(filename) && filename != ""){
            tkinsert(nameEntry, "0", filename)
            writeList(fileText, readLines(filename))
            makeGuess(filename)
        }else{
            tkconfigure(finishBut, state = "disabled")
        }
    }
    # Browse directories for a file
    brows <- function(){
        filename <<- tclvalue(tkcmd("tk_getOpenFile"))
        args$file <<- filename
        writeEntry(fileText, readLines(filename))
        writeEntry(nameEntry, args$file)
        makeGuess(args$file)
        tkconfigure(finishBut, state = "normal")
    }
    # Take a guess at the delimiter and header of a file
    makeGuess <- function(filename){
        headerNSep <- guess.sep(filename)
        args$header <<- headerNSep$header
        args$sep <<- headerNSep$separator
        if(is.element("header", basic)){
            writeEntry(boxes[["header"]], args$header)
        }
        if(is.element("sep", basic)){
            writeEntry(boxes[["sep"]], .sep2Entry(args$sep))
        }
    }
    # View the file after setting some of the arguments. Willl be
    # functioning later.
    view <- function(){
        fileRead <- read.table(file = args$file, head = args$header,
                               sep = args$sep, as.is = TRUE)

        fileRead <- as.matrix(fileRead)
        tkdelete(fileText, 0, "end")

        tkinsert(fileText, "end", fileRead[1:nrow(fileRead),])
    }
    # Read the file in before ending
    finish <- function(){
        for(i in names(boxes)){
            args[[i]] <<- .entry2Arg(tclvalue(tkget(boxes[[i]])))
        }
        fileRead <<- read.table(file = args$file, head = args$header,
                               sep = args$sep, as.is = TRUE)
        end()
    }
    # Not for this release yet
    showMore <- function(){
        writeEntry(restList, rest)
    }
    # Not for this release yet
    update <- function(){
        cat("workds")
    }
    # Not for this release yet
    argSelected <- function(){
        selectedArg <<-
            tclvalue(tkget(restList,(tkcurselection(restList))))
        writeEntry(restEntry, args[[selectedArg]])
        tkconfigure(restUp, state = "normal")
    }
    # Set up the interface
    top <- tktoplevel()
    tktitle(top) <- "BioC Data Import Wizard"
    # The top frame that has an entry box for a file name
    nameFrame <- tkframe(top)
    nameLabel <- tklabel(nameFrame, text = "File name:", font = NORMAL11)
    nameEntry <- tkentry(nameFrame, width = 60)
    browsBut <- tkbutton(nameFrame, text = "Browse", command = brows)
    tkpack(nameLabel, side = "left", padx = 2, pady = 4)
    tkpack(nameEntry, side = "left", padx = 2, pady = 4)
    tkpack(browsBut, side = "left", padx = 2, pady = 4)
    tkgrid(nameFrame, columnspan = 2, sticky = "w")
    # Put the entry boxes for basic arguments
    argFrame <- tkframe(top)
    i <- 1
    while(i <= length(basic)){
        if(i != length(basic)){
            frame1 <- tkframe(argFrame)
            boxes[[basic[i]]] <- tkentry(frame1, width = BOXW)
            tempLabel <- tklabel(frame1, text = paste(basic[i], ": ",
                            sep = ""), font = NORMAL11)
            tkgrid(tempLabel, boxes[[basic[i]]])
            frame2 <- tkframe(argFrame)
            boxes[[basic[i + 1]]] <- tkentry(frame2, width = BOXW)
            tempLabel <- tklabel(frame2, text = paste(basic[i + 1],
                                   ": ", sep = ""),
                                 font =NORMAL11)
            tkgrid(tempLabel, boxes[[basic[i + 1]]])
            tkgrid(frame1, frame2, padx = 10)
            tkgrid.configure(frame1, sticky = "w")
            tkgrid.configure(frame2, sticky = "e")
            writeEntry(boxes[[basic[i]]], args[[basic[i]]])
            writeEntry(boxes[[basic[i + 1]]], args[[basic[i + 1]]])
        }else{
            # Write the one that is left if the total is an odd number
            frame1 <- tkframe(argFrame)
            boxes[[basic[i]]] <- tkentry(frame1, width = BOXW)
            tempLabel <- tklabel(frame1, text = paste(basic[i], ": ",
                            sep = ""), font = NORMAL11)
            tkgrid(tempLabel, boxes[[basic[i]]])
            tkgrid.configure(tempLabel, sticky = "w")
            tkgrid.configure(boxes[[basic[i]]], sticky = "e")
            frame2 <- tkframe(argFrame)
            frame2 <- tkframe(argFrame)
            tkgrid(frame1, frame2, padx = 10)
            writeEntry(boxes[[basic[i]]], args[[basic[i]]])
        }
        i <- (i + 2)
    }
    tkgrid(argFrame, columnspan = 2)
    # Put in the rest of the arguments. Not for this release
#    restFrame <- tkframe(top)
#    bFrame <- tkframe(restFrame)
#    restButton <- tkbutton(bFrame, text = "More", width = 10,
#                           command = showMore)
#    tkpack(restButton)
#    restUp <- tkbutton(bFrame, text = "Update", width = 10,
#                       command = update)
#    tkpack(restUp)
#    tkpack(bFrame, side = "left", padx = 10)
#    listFrame <- tkframe(restFrame)
#    restList <- makeViewer(listFrame, vHeight = 2, vWidth = 20,
#                           vScroll = TRUE)
#    tkpack(listFrame, side = "left")
#    tkbind(restList, "<B1-ButtonRelease>", argSelected)
#    dFrame <- tkframe(restFrame)
#    restLabel <- tklabel(dFrame, text = "Default Value:")
#    tkpack(restLabel)
#    restEntry <- tkentry(dFrame, width = 20)
#    tkpack(restEntry)
#    tkpack(dFrame, side = "left", padx = 10)
#    tkgrid(restFrame, columnspan = 2, pady = 10)
#    tkconfigure(restUp, state = "disabled")
    # Put the window for viewing a file
    fileFrame <- tkframe(top)
    fileLabel <- paste("Preview of file:", filename)
    previewLabel <- tklabel(fileFrame, text = fileLabel,
                            font = NORMAL11)
    tkgrid(previewLabel, pady = 4, padx = 2, sticky = "nw")
    textFrame <- tkframe(fileFrame)
    fileText <- makeViewer(textFrame, vWidth = 20, vHeight = 15,
                           vScroll = TRUE, hScroll = TRUE,
                           what = "list")
    tkgrid(textFrame)
    tkgrid(fileFrame, columnspan = 2, padx = 2)
    # Put the buttons for different functions
    butFrame <- tkframe(top)
    cancelBut <- tkbutton(butFrame, text = "Cancel", width = 10,
                          command = end)
#    backBut <- tkbutton(butFrame, text = "< Back", width = 10,
#                        command = init)
#    viewBut <- tkbutton(butFrame, text = "View", width = 10,
#                        command = view)
    finishBut <- tkbutton(butFrame, text = "Finish", width = 10,
                          command = finish)
    tkpack(cancelBut, finishBut, side = "left",
           padx = 2)
    tkgrid(butFrame, columnspan = 2, pady = 4)

    init()

    tkwait.window(top)

    return(fileRead)
}
# Format a delimiter so that it can be displayed in an entry box
.sep2Entry <- function(sep){
    switch(sep,
           "\t" = return("\\t"),
           " " = return("\" \""),
           "\n" = return("\\n"),
           return(sep))
}

.entry2Arg <- function(entry){
    switch(entry,
           "T" = ,
           "TRUE" = ,
           "True" = ,
           "true" = return(TRUE),
           "F" = ,
           "FALSE" = ,
           "False" = ,
           "false" = return(FALSE),
           "\\t" = return("\t"),
           "\"\"" = return(""),
           "\" \"" = return(" "),
           return(entry))
}

# A function to create a light wight tk widget with a text.
# x and y - the location for the left upper coner of the widget to
#           appear on the screen
getLightTW <- function(x, y, text){
    on.exit(end())

    end <- function(){
        tkdestroy(base)
    }
    # Takes out the frame and title bar
    tkwm.overrideredirect(base <- tktoplevel(), TRUE)
    # Put the TW in the right place
    tkwm.geometry(base, paste("+", x, "+", y, sep = ""))
    text <- tklabel(base, text = text)
    tkpack(text)
    tkbind(base, "<ButtonPress>", end)

    tkwait.window(base)

    return(invisible())
}
# This function returns a list containing the values for widgets on a
# widget created by function widgetRender. It takes a list defining
# the widget and returns a named list containing the values for each
# of the widgets.
#
# Copyright J. Zhang 2002, all rights reserved.
#

getWvalues <- function (W){
    wList <- WwList(W)
    lenW <- length(wList)
    if( is.null(wList) ) return(NULL)
    rList <- vector("list", length = lenW)
    names(rList) <- names(wList)
    for(i in 1:lenW)
        rList[[i]] <- WValue(wList[[i]])
    return(rList)
}
# This function reads a few lines from a given file and then guesses
# if there is a header line, the separator used to separate columns,
# and the data type of each column.
#
# Copyright 2002, J. Zhang, all rights reserved
#

guess.sep <- function(file.name, numLine = 5, seps = "", isFile = TRUE){

    separator <- ""
    header <- FALSE

    if(seps == ""){
        seps <- c(" ", ",", ";", "\t")
    }else{
        seps <- c(" ", ",", ";", "\t", seps)
    }

    if(isFile){
        toCheck <- readLines(file.name, n = numLine)
    }else{
        toCheck <- file.name
    }

    w<-NULL

    for(i in seps)
         w[[i]] <- strsplit(toCheck[2:length(toCheck)], i)

    v <- lapply(w, function(x) sapply(x, length))

    good <- function(x) all(x==x[1]) && x[1] > 1
    found <- sapply(v, good)

    sep <- names(found[found])
    if(length(sep) == 1){
        separator <- sep
        if(length(unlist(strsplit(toCheck[1], separator)))
                                              == v[[separator]][1] - 1){
            header <- TRUE
        }else{
            header <- guess.header(toCheck[1:2], separator)
        }

        type <- find.type(toCheck[2:length(toCheck)],separator, header)
        return(list(header = header, separator = separator, type = type))
    }else{
        # New line is always the separator
        return(list(header = FALSE, separator = "\n",
                    type = find.type(toCheck[2:length(toCheck)],
                    header = FALSE)))
    }
}

guess.header <- function(twoLines, sep){

    on.exit(options(warn = 1))

    if(!is.null(sep)){
        firstLine <- unlist(strsplit(twoLines[1], sep))
        scndLine <- unlist(strsplit(twoLines[2], sep))
    }else{
        firstLine <- twoLines[1]
        scndLine <- twoLines[2]
    }

    options(warn = -1)
    firstLine <- as.numeric(firstLine)
    scndLine <- as.numeric(scndLine)
    options(warn = 1)

    firstLine[!is.na(firstLine)] <- "num"
    scndLine[!is.na(scndLine)] <- "num"

    if(!setequal(firstLine, scndLine)){
        return(TRUE)
    }else{
        if(any(!is.na(firstLine))){
            return(FALSE)
        }
        return(FALSE)
    }
}

find.type <- function(line, sep, header = FALSE){

     types <- NULL
     if(!missing(sep)){
         for(i in line){
             temp <- unlist(strsplit(i, sep))
             types <- rbind(types, charOrNum(temp))
         }
     }else{
         for(i in line){
             types <- rbind(types, charOrNum(i))
         }
     }
     if(nrow(unique(types)) == 1){
         if(header){
             return(types[1,][2:length(types[1,])])
         }else{
             return(types[1,])
         }
     }else{
         return("Not detected")
     }
}

charOrNum <- function(vect){
    options(warn = -1)
    temp <- as.numeric(vect)
    options(warn = 1)

    temp[is.na(temp)] <- "Character"
    temp[!is.na(temp) & temp != "Character"] <- "Numeric"
    return(temp)
}
# This function returns TRUE if a given prefix, suffix, or characters
# matches a character string or FALSE otherwise.
#
# Copyright 2002, J. Zhang. All rights reserved.
#

hasPrefix <- function(aPrefix){
    hasChar(aPrefix, "prefix")
}

hasSuffix <- function(aSuffix){
    hasChar(aSuffix, "suffix")
}

hasChar <- function (toCheck, what = ""){

    if(!is.character(toCheck) || nchar(toCheck)  < 1)
        stop(paste("Bad value:", toCheck))

    function(x){
        if(what == "prefix"){
            pattern <- paste("^", toCheck, sep = "")
        }else if(what == "suffix"){
            pattern <- paste(toCheck, "$", sep = "")
        }else{
            pattern <- toCheck
        }

        if(regexpr(pattern, x) > 0 ){
           return(TRUE)
        }else{
            return(FALSE)
        }
    }
}


# This function provides data import interfaces by mimicing MS Excel's
# Text Wizard. read.table will be used to import the data.
#
# filename - an optional character string for the name of the file to
# be imported
# maxRow - an integer for the maximum number of rows to be
# displayed. Large row numbers may slow down the machine
#
# Copyright 2002, J. Zhang, all rights reserved.

importWizard <- function(filename, maxRow = 400){

    # Creates an environment to make some variables available to all
    # related functions
    workEnv <- new.env(hash = TRUE, parent = NULL)
    # A string to keep track of the current state
    assignCState("state1", env = workEnv)
    # Number of row to be displayed
    assignShowNum(maxRow, env = workEnv)

    if(!missing(filename)){
        # Do this if a file name is given
        argsSet <- setArgsList(filename, workEnv)
    }else{
        # Otherwise, assign an empty list to argsList and colInfo
        assignArgs(list(), workEnv)
        setColInfos(env = workEnv)
        argsSet <- TRUE
    }
    if(argsSet){
        # Initializes the interface
        initImportWizard(workEnv)
    }
}
# Using function guess.sep to figure out the the header, sep, and data
# type of a file and sets the argument list and colInfo
setArgsList <- function(filename, env, isFile = TRUE, init = TRUE){
    options(show.error.messages = FALSE)
    fileInfo <- try(guess.sep(file.name = filename, numLine = 40,
                              isFile = isFile))
    options(show.error.messages = TRUE)
    if(inherits(fileInfo, "try-error")){
        tkmessageBox(title = "Incorrect File Name",
                 message = paste("An error message:\n\n", fileInfo,
                           "\nwas generated while reading file",
                           filename, "."), icon = "error", type = "ok")
        return(FALSE)
    }else{
        if(init){
            argsList <- list()
            temp <- formals("read.table")
            temp[["file"]] <- filename
            temp[["header"]] <- fileInfo[["header"]]
            temp[["sep"]] <- fileInfo[["separator"]]
            temp[["quote"]] <- ""
            # Reassign fill with the value of blank.lines.skip
            temp[["fill"]] <- !temp[["blank.lines.skip"]]
            argsList[["state1"]] <- as.list(temp)
        }else{
            argsList <- getArgs(env)
            argsList[["state1"]][["header"]] <- fileInfo[["header"]]
            argsList[["state1"]][["sep"]] <- fileInfo[["separator"]]
        }
        assignArgs(argsList, env)
        setColInfos(fileInfo[["type"]], env)
        if(isFile){
            assignLineData(readLines(filename, n = getShowNum(env)), env)
        }else{
            assignLineData(filename, env)
        }
        return(TRUE)
    }
}
# Set the temp data read as lines with a maxmun number
assignLineData <- function(lineData, env){
    assign("lineData", lineData, env = env)
}
# Get the temp data stroed as lines with a maxmun number
getLineData <- function(env){
    get("lineData", env = env)
}
# Set and get methods for argument list
assignArgs <- function(value, env){
    assign("argsList", value, env = env)
}
getArgs <- function(env){
    get("argsList", env = env)
}
# Set and get methods for number to show in the interface
assignShowNum <- function(value, env){
    assign("showNum", value, env)
}
getShowNum <- function(env){
    get("showNum", env)
}
# Set and get methods for current state to keep track of the state
assignCState <- function(value, env){
    assign("currentState", value, env)
}
getCState <- function(env){
    get("currentState", env)
}
# Set and get methods for colInfo that is a list of colInfo objects to
# keep column name, type, and drop info
assignColInfo <- function(value, env){
    assign("colInfos", value, env)
}
getColInfo <- function(env){
    get("colInfos", env)
}
# Creates colInfo objects and sets the value of 'colInfos' list
setColInfos <- function(types, env){
#    initColInfo()
    if(missing(types)){
        assignColInfo(list(), env)
    }else{
        temp <- list()
        for(i in 1:length(types)){
            # name = "", type = types[i], dropOrNot = FALSE
            temp[[i]] <- colInfo("", types[i], FALSE)
        }
        assignColInfo(temp, env)
    }
}
# This function initializes the interface for importWizard by creating
# a widget with an empty top canvas and bottom frame filled with four buttons
initImportWizard <- function(env){
    # A list to be returned that contains an argument list and data
    # imported using read.table
    dataList <- NULL
    # A variable to keep the frame that is currently displayed
    currentFrame <- NULL

    on.exit(end())
    # Destroy the window
    end <- function(){
        tkdestroy(top)
    }
    nextState <- function(){
        args <- getArgs(env)
        if(is.null(args[["state1"]][["file"]])){
            tkmessageBox(title = "Import Error",
                         message = "I do not know what file to import!",
                         icon = "error",
                         type = "ok")
        }else{
            tempFrame <- changeState(canvas, backBut, nextBut, env, TRUE,
                                 endBut, viewBut)
            tkdestroy(currentFrame)
            tkpack(tempFrame, fill = "both", expand = TRUE)
            currentFrame <<- tempFrame
        }
    }
    preState <- function(){
        tempFrame<- changeState(canvas, backBut, nextBut, env, FALSE,
                                endBut, viewBut)
        tkdestroy(currentFrame)
        tkpack(tempFrame, fill = "both", expand = TRUE)
        currentFrame <<- tempFrame
    }
    redraw <- function(){
        tempFrame <- getAFrame(canvas, env)
        tkdestroy(currentFrame)
        tkpack(tempFrame, fill = "both", expand = TRUE)
        currentFrame <<- tempFrame
    }
    finishClicked <- function(){
        dataList <<- finish(env)
        if(!is.null(dataList)){
            end()
        }
    }

    ## Set up the interface
    top <- tktoplevel()
    tktitle(top) <- "BioC Data Import Wizard"
    # Set the empty canvas that will be filled later
    canvas <- getTopCan(top, env)
    # Sets current frame to state1 now
    currentFrame <- getAFrame(canvas, env)
    tkpack(currentFrame, fill = "both", expand = TRUE)
#    tkcreate(canvas, "window", 0, 0, anchor = "nw", window = currentFrame)
    ## The bottom frame contains the buttons that allow users to
    ## navigate the importing process
    butFrame <- tkframe(top)
    viewBut <- tkbutton(butFrame, text = "View", width = 8,
                        state = "disabled", command = redraw)
    canBut <- tkbutton(butFrame, text = "Cancel", width = 8,
                       command = end)
    backBut <- tkbutton(butFrame, text = "< Back", width = 8,
                        state = "disabled", command = preState)
    nextBut <- tkbutton(butFrame, text = "Next >", width = 8,
                        command = nextState)
    endBut <- tkbutton(butFrame, text = "Finish", width = 8,
                       state = "disabled", command = finishClicked)
    tkpack(canBut, backBut, nextBut, viewBut, endBut, side = "left")
    tkpack(butFrame, pady = 10, fill = "y", expand = TRUE)

    args <- getArgs(env)
    tkwait.window(top)
    return(invisible(dataList))
}
# Creates a top frame of an empty canvas
getTopCan <- function(base, env){
    WIDTH <- 730
    HEIGHT <- 400
    ## The canvas has widgets for the relevant
    ## arguments and preview of the original data. The content various
    ## depending on the state
    canvas <- makeViewer(base, vWidth = WIDTH, vHeight = HEIGHT,
                           vScroll = FALSE, hScroll = FALSE,
                           what = "canvas", side = "top")

    return(canvas)
}
# Changes the state and thus the interface
changeState <- function(canvas, backBut, nextBut, env, forward = TRUE,
                        endBut, viewBut){
    # Sets the current state
    setNewState(env, backBut, nextBut, forward, endBut, viewBut)
    if(forward){
        addArgs(env)
    }else{
        dropArgs(env)
    }
    return(getAFrame(canvas, env))
#    tkcreate(canvas, "window", 0, 0, anchor = "nw", window = newFrame)
}
# Sets the string for the new state (next or previous) and
# actviates/inactivates buttons depending on the state
setNewState <- function(env, backBut, nextBut, forward = TRUE,
                        endBut, viewBut){
    if(forward){
        if(getCState(env) == "state1"){
            assignCState("state2", env)
            tkconfigure(backBut, state = "normal")
            tkconfigure(viewBut, state = "normal")
        }else{
            assignCState("state3", env)
            tkconfigure(nextBut, state = "disabled")
            tkconfigure(endBut, state = "normal")
            tkconfigure(viewBut, state = "disabled")
        }
    }else{
        if(getCState(env) == "state2"){
            assignCState("state1", env)
            tkconfigure(nextBut, state = "normal")
            tkconfigure(backBut, state = "disabled")
            tkconfigure(viewBut, state = "disabled")
        }else{
            assignCState("state2", env)
            tkconfigure(nextBut, state = "normal")
            tkconfigure(endBut, state = "disabled")
            tkconfigure(viewBut, state = "normal")
        }
    }
}
# Add a new state arguments list to argsList
addArgs <- function(env){
    temp <- getArgs(env)
    if(length(temp) == 1){
        temp[["state2"]] <- temp[[length(temp)]]
        assignArgs(temp, env)
    }else{
        temp[["state3"]] <- temp[[length(temp)]]
        assignArgs(temp, env)
    }
}
# Drop a state arguments list from argsList when the back button is clicked
dropArgs <- function(env){
    temp <- getArgs(env)
    if(length(temp) > 1){
        temp <- temp[-length(temp)]
        assignArgs(temp, env)
    }
}
# Gets a frame based on which state is of interest
getAFrame <- function(base, env){
    switch(getCState(env),
           "state1" = return(getState1Frame(base, env)),
           "state2" = return(getState2Frame(base, env)),
           "state3" = return(getState3Frame(base, env)))
}
# The importing process ends. Return a list with argument list and
# data read using read.table as elements
finish <- function(env){
    args <- getArgs(env)[["state3"]]
    if(is.null(args[["quote"]])){
        args[["quote"]] <- "\"'"
    }
    dataName <- getName4Data(args[["file"]])
    options(show.error.messages = FALSE)
    dataFile <- try(do.call("read.table", args))
    options(show.error.messages = TRUE)
    if(inherits(dataFile, "try-error")){
        tkmessageBox(title = "Import Error",
                     message = paste("An error message:\n\n", dataFile,
                     "\nwas generated while reading file ",
                     args[["file"]], "."), icon = "error", type = "ok")
        return(NULL)
    }else{
        colInfos <- getColInfo(env)
        colNames <- NULL
        colToDrop <- NULL
        for(i in 1:length(colInfos)){
            if(dropOrNot(colInfos[[i]])){
                colToDrop <- c(colToDrop, i)
            }else{
                switch(colType(colInfos[[i]]),
                   "Character" = dataFile[, i] <- as.character(dataFile[, i]),
                   "Numeric" = dataFile[, i] <- as.numeric(dataFile[, i]))
                colNames <- c(colNames, colName(colInfos[[i]]))
            }
        }
        # Drop the columns
        if(!is.null(colToDrop)){
            dataFile <- dataFile[, -colToDrop]
        }
        # In case there is only one column left
        if(is.null(ncol(dataFile))){
            dataFile <- data.frame(matrix(dataFile, ncol = 1))
            names(dataFile) <- colNames
        }else{
            names(dataFile) <- colNames
        }
        if(!is.null(dataName)){
            assign(dataName, dataFile, env = .GlobalEnv)
        }
        return(list(args = args, data = dataFile))
    }
}
# Gets the frame containing the interface for the top frame of
# importWizard for state1
getState1Frame <- function(base, env){
    # A frame containing the interface that will be returned
    frame <- tkframe(base)
    # The bottom frame contains a list box showing the data. The
    # bottom frame is set first to make the list box available for
    # updating by the top frame
    bottomFrame <- tkframe(frame)
    dataViewer <- setState1BFrame(bottomFrame, env)
    # The mid frame contains the delimiter and number line
    # information.
    midFrame <- tkframe(frame)
    delims <- setState1MFrame(midFrame, env, dataViewer)
    # The top frame contains a entry box and a browse button that
    # allows for browing directories for a file name
    topFrame <- tkframe(frame)
    setState1TFrame(topFrame, dataViewer, delims, env)
    tkpack(topFrame, pady = 5, padx = 5, fill = "both", expand = TRUE)
    tkpack(midFrame, padx = 5, fill = "both", expand = TRUE)
    # Pack the bottom frame last
    tkpack(bottomFrame, pady = 5, padx = 5, fill = "both", expand = TRUE)
    return(frame)
}
# Sets the botton frame for state1
setState1BFrame <- function(frame, env){
    # A list box to show the original data
    viewFrame <- tkframe(frame)
    dataViewer <- makeViewer(viewFrame, vWidth = 50, vHeight = 10,
                            vScroll = TRUE, hScroll = TRUE,
                            what = "list", side = "top")
    tkpack(viewFrame, anchor = "w", pady = 10, fill = "both",
                                                      expand = TRUE)
    return(dataViewer)
}
# Sets the top frame for state1
setState1TFrame <- function(frame, viewer, delims, env){
    fName <- tclVar()
    # Populate the entry box for file name when the brose button is
    # clicked
    browse <- function(){
        filename <- tclvalue(tkgetOpenFile())
        writeList(nameEntry, filename, clear = TRUE)
        argsSet <- setArgsList(filename, env)
        if(argsSet){
            showData4State1(viewer, env)
            if(!is.null(getArgs(env)[["state1"]][["sep"]])){
                tkselect(delims[["delimit"]])
            }
        }
    }
    # Get the file
    getFile <- function(){
        argsSet <- setArgsList(tclvalue(fName), env)
        if(argsSet){
            showData4State1(viewer, env)
            if(!is.null(getArgs(env)[["state1"]][["sep"]])){
                tkselect(delims[["delimit"]])
            }
        }
    }

    # Frame to hole the widgets
    nameFrame <- tkframe(frame)
    label1 <- tklabel(nameFrame, text = "File name: ")
    tkpack(label1, side = "left")
    # An entry box to hold the result of fileBrowser
    nameEntry <- tkentry(nameFrame, width = 20, textvariable = fName)
    # If a file name is given, fill the widget with data
    if(!is.null(getArgs(env)[["state1"]][["file"]])){
        writeList(nameEntry, getArgs(env)[["state1"]][["file"]], clear = TRUE)
        showData4State1(viewer, env)
        if(length(getArgs(env)[["state1"]][["sep"]]) != 0){
            tkselect(delims[["delimit"]])
        }
    }
    tkpack(nameEntry, side = "left", fill = "x", expand = TRUE)
    # A button to envoke fileBrowser
    browseBut <- tkbutton(nameFrame, width = 6, text = "Browse",
                          command = browse)
    getBut <- tkbutton(nameFrame, width = 6, text = "Get",
                       command = getFile)
    tkpack(browseBut, side = "left", fill = "x")
    tkpack(getBut, side = "left", fill = "x")
    tkpack(nameFrame, fill = "both", expand = TRUE)
}
# Show the data read in using readLines for state1
showData4State1 <- function(widget, env){
     skip <- getArgs(env)[["state1"]][["skip"]]
     if(!is.null(skip)){
         dataFile <- getLineData(env)
         showNum <- getShowNum(env)
         if(length(dataFile) > showNum){
             dataFile <- dataFile[(skip + 1):showNum]
         }else{
             dataFile <- dataFile[(skip + 1):length(dataFile)]
         }
     }else{
         dataFile <- getLineData(env)
     }
     # Preventing the header to be shown
     if(getArgs(env)[["state1"]][["header"]]){
         dataFile <- dataFile[2:length(dataFile)]
     }
     # determines how many lines to show
     if(length(dataFile) > getShowNum(env)){
         writeList(widget, paste(1:getShowNum(env), ": ",
                             dataFile[1:getShowNum(env)], sep = ""), TRUE)
     }else{
         writeList(widget, paste(1:length(dataFile), ": ",
                                                dataFile, sep = ""), TRUE)
     }
}
# Sets the mid frame for state1
setState1MFrame <- function(frame, env, dataViewer){
    # Executed when values in start at row list box is clicked
    startClicked <- function(){
        setSkip(startList, env)
        args <- getArgs(env)
        skip <- as.numeric(args[["state1"]][["skip"]])
        assignShowNum((getShowNum(env) + skip), env)
        dataFile <- getLineData(env)
        showNum <- getShowNum(env)
        if(length(dataFile) > showNum){
            dataFile <- dataFile[(skip + 1):showNum]
        }else{
            dataFile <- dataFile[(skip + 1):length(dataFile)]
        }
        setArgsList(dataFile, env, FALSE, FALSE)
#        showData4State1(dataViewer, env)

    }
    leftPan <- tkframe(frame)
    delimit <- tclVar()
    delimitRadio <- tkradiobutton(leftPan, text = paste("Delimited",
                                  " - Files are separated by a character",
                                  " such as a comma, tab ...", sep =""),
                                  value = "delim", variable = delimit,
                                  anchor = "nw")
    tkpack(delimitRadio, anchor = "w", expand = TRUE, fill = "x")
    fixedRadio <- tkradiobutton(leftPan, text = paste("Fixed",
                                " width - Fields are aligned in columns",
                                " with spaces between fields", sep = ""),
                                value = "fixed", variable = delimit,
                                anchor = "nw")
    tkpack(fixedRadio, anchor = "w", expand = TRUE, fill = "x")
    tkpack(leftPan, side = "left", anchor = "w", fill = "x", expand = TRUE)
    rightPan <- tkframe(frame)
    paraLabel2 <- tklabel(rightPan, text = "Start import at row:")
    tkpack(paraLabel2, side = "left", anchor = "ne")
    startFrame <- tkframe(rightPan)
    startList <- makeViewer(startFrame, vWidth = 2, vHeight = 1,
                            what  = "list", side = "top")
    tkconfigure(startList, selectmode = "single")
    tkbind(startList, "<B1-ButtonRelease>", startClicked)
    writeList(startList, 1:99, clear = TRUE)
    tkpack(startFrame, anchor = "w", side = "left",
                                          fill = "x", expand = TRUE)
    tkpack(rightPan, side = "left", padx = 7, expand = TRUE, fill = "x")
    return(list(delimit = delimitRadio, fixed = fixedRadio))
}

# Sets the value for skip when user selects line to start in
# state1
setSkip <- function(widget, env, state = "state1"){
    temp <- getArgs(env)
    temp[[state]][["skip"]] <-
                    as.numeric(tkget(widget, tkcurselection(widget))) - 1
    assignArgs(temp, env)
}
# Gets a frame for state2
getState2Frame <- function(base, env, state = "state2", reset = FALSE){
    frame <- tkframe(base)
    # Shows the name of the file
    label1 <- tklabel(frame, text = paste("File:",
                      getArgs(env)[[state]][["file"]]),
                      font = "Helvetica 11 bold")
    tkpack(label1, pady = 5, padx = 5)
    midFrame <- tkframe(frame)
    setState2MFrame(midFrame, env)
    tkpack(midFrame, pady = 5, padx = 5, fill = "x", expand = TRUE)
    bottomFrame <- tkframe(frame)
    setState2BFrame(bottomFrame, env)
    tkpack(bottomFrame, fill = "both", expand = TRUE)
    return(frame)
}
# Sets the state2 mid frame containing radio buttons for delimiters
# and a list box for quote selection
setState2MFrame <- function(frame, env){
    # Radio buttons for delimiters
    leftFrame <- tkframe(frame)
    setSepRadios(leftFrame, env)
    # A list for quote selecttion (" or/and ')
    rightFrame <- tkframe(frame)
    setQuoteList(rightFrame, env)
    tkpack(leftFrame, side = "left", anchor = "w", fill = "x",
           expand  = TRUE)
    tkpack(rightFrame, side = "left", fill = "x", expand = TRUE)
}
# Sets the radio buttons for separators for state2 mid frame
setSepRadios <- function(frame, env, state = "state2"){
    labelFrame <- tkframe(frame)
    label <- tklabel(labelFrame, text = "File Delimiter:")
    tkpack(label, side = "left", anchor = "nw")
    tkpack(labelFrame, side = "left")
    sepFrame <- tkframe(frame)
    sepVar <- tclVar()
    sepButs <- list()
    sepButFun <- function(){
        if(tclvalue(sepVar) != "other"){
            temp <- getArgs(env)
            temp[[state]][["sep"]] <- tclvalue(sepVar)
            assignArgs(temp, env)
            tkconfigure(otherEntry, state = "disabled")
        }else{
            tkconfigure(otherEntry, state = "normal")
        }
    }
    sepEntered <- function(){
        tkselect(sepButs[["other"]])
        temp <- getArgs(env)
        temp[[state]][["sep"]] <- as.character(tkget(otherEntry))
        assignArgs(temp, env)
    }
    sepButs[["tab"]] <- tkradiobutton(sepFrame, text = "Tab",
                              variable = sepVar, width = 9,
                              value = "\t", anchor = "nw",
                                      command = sepButFun)
    sepButs[["semi"]] <- tkradiobutton(sepFrame, text = "Semicolon",
                               variable = sepVar, width = 9,
                               value = ";", anchor = "nw",
                                       command = sepButFun)
    sepButs[["comma"]] <- tkradiobutton(sepFrame, text = "Comma",
                              variable = sepVar, value = ",",
                              width = 9, anchor = "nw",
                                        command = sepButFun)
    sepButs[["space"]] <- tkradiobutton(sepFrame, text = "Space",
                              variable = sepVar, value = "\"\"",
                              width = 9, anchor = "nw",
                                        command = sepButFun)
    # Puts the buttons in two rows. First row now
    tkgrid(sepButs[["tab"]], sepButs[["semi"]], sepButs[["comma"]],
           sepButs[["space"]])
    sepButs[["newline"]] <- tkradiobutton(sepFrame, text = "Newline",
                              variable = sepVar, value = "\n",
                              width = 9, anchor = "nw",
                                        command = sepButFun)
    sepButs[["other"]] <- tkradiobutton(sepFrame, text = "Other:",
                              variable = sepVar, value = "other",
                              width = 9, anchor = "nw",
                                        command = sepButFun)
    otherEntry <- tkentry(sepFrame, width = 11, state = "disabled")
    tkbind(otherEntry, "<KeyRelease>", sepEntered)
    # Second row with an entry box for delimiters other those given here
    tkgrid(sepButs[["newline"]], sepButs[["other"]], otherEntry)
    tkpack(sepFrame, side = "left", anchor = "ne", fill = "x",
           expand = TRUE)
    if(!is.null(getArgs(env)[[state]][["sep"]])){
        tkselect(sepButs[[whatDeli(getArgs(env)[[state]][["sep"]])]])
    }
}

# Sets the list box for quotes for state2 mid frame
setQuoteList <- function(frame, env){
    quoteSelected <- function(){
        setQuote(quoteList, env)
    }
    label1 <- tklabel(frame, text = "     Quote:")
    tkpack(label1, side = "left", anchor = "ne")
    quoteFrame <- tkframe(frame)
    quoteList <- makeViewer(quoteFrame, vWidth = 8, vHeight = 1,
                            what  = "list", side = "top")
    tkconfigure(quoteList, selectmode = "extended")
    tkbind(quoteList, "<B1-ButtonRelease>", quoteSelected)
    writeList(quoteList, c("\"", "'"), clear = TRUE)
    tkpack(quoteFrame, anchor = "w", fill = "x", expand = TRUE)
}

# Sets the value for quote when user selects quote in the list for
# quotes in state2
setQuote <- function(listBox, env, state = "state2"){
    quotes <- ""
    # Quote can be multiple (" and/or ')
    selIndex <- unlist(strsplit(as.character(tkcurselection(listBox)), " "))
    for(i in selIndex){
        quotes <- paste(quotes, tkget(listBox, i), sep = "")
    }
    temp <- getArgs(env)
    temp[[state]][["quote"]] <- quotes
    assignArgs(temp, env)
}
# Sets the canvas holding the preview of data for state2
setState2BFrame <- function(frame, env){
    viewFrame <- tkframe(frame)
    dataView2 <- makeViewer(viewFrame, vScroll = TRUE, hScroll = TRUE,
                           what = "canvas", side = "top")
    tkpack(viewFrame, anchor = "w", padx = 5, pady = 5, fill = "both",
           expand = TRUE)
    showData4State2(dataView2, env)
}
# Populates the data preview list of state2
showData4State2 <- function(canvas, env, state = "state2"){
    writeColList <- function(i){
        tempList <- tklistbox(tempFrame, height = 0, width = 0,
                              background = "white")
        writeList(tempList, dataFile[,i])
        #tkinsert(tempList, "end", dataFile[,i])
        tkpack(tempList, side = "left", fill = "both", expand = TRUE)
    }

    # Only show the number of rows defined
    temp <- getArgs(env)[[state]]
    tempFrame <- tkframe(canvas)
    # Puts n in temporaly
    temp[["nrows"]] = getShowNum(env)
    dataFile <- do.call("read.table", temp)
    # For data without a separater
    if(is.null(ncol(dataFile))){
        writeColList(1)
    }else{
        for(i in 1:ncol(dataFile)){
            writeColList(i)
        }
    }
    tkcreate(canvas, "window", 0, 0, anchor = "nw", window = tempFrame)
}

# Gets the frame containing the interface for the top frame of
# importWizard for state3
getState3Frame <- function(base, env){
    # A frame containing the interface that will be returned
    frame <- tkframe(base)
    label1 <- tklabel(frame, text = paste("File:",
                             getArgs(env)[["state3"]][["file"]]),
                      font ="Helvetica 11 bold")
    tkpack(label1, pady = 5)
    topFrame <- tkframe(frame)
    if(!is.null(getArgs(env)[["state3"]][["sep"]])){
        setState3TFrame(topFrame, env)
        tkpack(topFrame, anchor = "nw", fill = "x", expand = TRUE)
    }
    bottomFrame <- tkframe(frame)
    setState3BFrame(bottomFrame, env)
    tkpack(bottomFrame, padx = 5, pady = 5, fill = "both", expand = TRUE)
    return(frame)
}
# Creates the left bottom portion of state3 frame
setState3TFrame <- function(frame, env){
    # More arguments for read.table
    moreClicked <- function(){
        moreArgs(env)
    }
    label <- tklabel(frame, text = paste("Editable column names are",
                            " shown in the first entry box on top of data",
                            " columns.\nEditable data type of columns is",
                            " shown in the entry box following coloumn",
                            " names\nClick the check box on top of a column",
                            " to drop the column.\nClik 'More Args' button",
                            " for more arguments.", sep = ""),
                       width = 80, height = 4, justify = "left")
    tkpack(label, anchor = "w", pady = 5, side = "left", padx = 5)
    moreBut <- tkbutton(frame, text = "More Args...", width = 10,
                                                 command = moreClicked)
    tkpack(moreBut, side = "left", padx = 5, pady = 15)
}
# Read the other arguments from a widget for read.table arguments
moreArgs <- function(env){
    temp <- getArgs(env)
    moreArgs <- getMoreArgs()
    for(i in names(moreArgs)){
        temp[[getCState(env)]][[i]] <- moreArgs[[i]]
    }
    assignArgs(temp, env)
}
# Creates the right bottom portion of state3 frame
setState3BFrame <- function(frame, env){
    # Creat a canvas to hold the other widget elements
    rCanv <-  makeViewer(frame, vWidth = 700, vHeight = 280,
                       vScroll = TRUE, hScroll = TRUE,
                       what = "canvas", side = "top")
    tempFrame <- tkframe(rCanv)
    argsList <- getArgs(env)[["state3"]]
    argsList[["nrows"]] <- getShowNum(env)
    dataFile <- do.call("read.table", argsList)
    # Cut to right size of file if longer than maxRow
#    if(nrow(dataFile) > getShowNum(env)){
#        dataFile <- dataFile[1:getShowNum(env),]
#    }
    # Finds the data type for columns
    colInfos <- getColInfo(env)
    writeCol4Matrix(tempFrame, dataFile, colInfos, env)

    tkcreate(rCanv, "window", 0, 0, anchor = "nw", window = tempFrame)
}
# Create a group of list boxes with entry boxes and a radio button on
# top to allow for user inputs.
writeCol4Matrix <- function(tempFrame, dataFile, colInfos, env){
    writeDataCol <- function(i, data){
        colFrame <- tkframe(tempFrame)
        dropCMD[[i]] <- function(){}
        body <- list(as.name("{"),
                     substitute(eval(dropColumn(j, env)), list(j = i)))
        body(dropCMD[[i]]) <- as.call(body)
        var <- tclVar()
        dropCheck[[i]] <- tkcheckbutton(colFrame, text = "Drop",
                                   variable = var, command = dropCMD[[i]])
        tkpack(dropCheck[[i]], side = "top", fill = "x", expand = TRUE)
        nameEntry[[i]] <- tkentry(colFrame, width = 0)
        # Also updates the value of colInfos
        temp <- colInfos[[i]]
        if(!is.null(colnames(data))){
            writeList(nameEntry[[i]], colnames(data))
            colName(temp) <- colnames(data)
        }else{
            writeList(nameEntry[[i]], paste("V", i, sep = ""))
            colName(temp) <- paste("V", i, sep = "")
        }
        colInfos[[i]] <<- temp
        nameCMD[[i]] <- function(){}
        body <- list(as.name("{"), substitute(eval(setColName(j,
                                   nameEntry[[j]], env)), list(j = i)))
        body(nameCMD[[i]]) <- as.call(body)
            tkbind(nameEntry[[i]], "<KeyRelease>", nameCMD[[i]])
        tkpack(nameEntry[[i]], side = "top", fill = "x", expand = TRUE)
        typeEntry[[i]] <- tkentry(colFrame, width = 0)
        writeList(typeEntry[[i]], colType(colInfos[[i]]))
        typeCMD[[i]] <- function(){}
        body <- list(as.name("{"), substitute(eval(setColType(j,
                                      typeEntry[[j]], env)), list(j = i)))
        body(typeCMD[[i]]) <- as.call(body)
        tkbind(typeEntry[[i]], "<KeyRelease>", typeCMD[[i]])
        tkpack(typeEntry[[i]], side = "top", fill = "x", expand = TRUE)
        colList[[i]] <- tklistbox(colFrame, width = 0,
                                  height = 0, background = "white")
        writeList(colList[[i]], data)
#        tkinsert(colList[[i]], "end", data)
        tkpack(colList[[i]], side = "top", fill = "x", expand = TRUE)
        tkpack(colFrame, side = "left", fill = "both", expand = TRUE)
    }
    typeEntry <- list()
    dropCheck <- list()
    nameEntry <- list()
    # Lists to keep the command associated with the radio buttons of
    # entry boxex
    dropCMD <- list()
    nameCMD <- list()
    typeCMD <- list()
    colList <- list()
    if(is.null(ncol(dataFile))){
        writeDataCol(1, dataFile)
    }else{
        for(i in 1:ncol(dataFile)){
            writeDataCol(i, dataFile[,i])
        }
    }
    # Sets values for colInfo object
    assignColInfo(colInfos, env)
}

# Set the value of slot 'drop' of a colInfo object
dropColumn <- function(index, env){
    colInfos <- getColInfo(env)
    temp <- colInfos[[index]]
    if(dropOrNot(colInfos[[index]])){
        temp <- colInfos[[index]]
        dropOrNot(temp) <- FALSE
    }else{
        dropOrNot(temp) <- TRUE
    }
    colInfos[[index]] <- temp
    assignColInfo(colInfos, env)
}
# Set the value of slot (column) 'name' of a colInfo object
setColName <- function(index, entryBox, env){
    colInfos <- getColInfo(env)
    entry <- as.character(tkget(entryBox))
    temp <- colInfos[[index]]
    colName(temp) <- entry
    colInfos[[index]] <- temp
    assignColInfo(colInfos, env)
}
# Set the value of slot 'type' of a colInfo object
setColType <- function(index, entryBox, env){
    colInfos <- getColInfo(env)
    entry <- as.character(tkget(entryBox))
    temp <- colInfos[[index]]
    colType(temp) <- entry
    colInfos[[index]] <- temp
    assignColInfo(colInfos, env)
}
# Gets the word representation of delimiters
whatDeli <- function(delimiter){
    switch(delimiter,
           "\t" = return("tab"),
           ";" = return("semi"),
           " " = return("space"),
           "," = return("comma"),
           "\n" = return("newline"),
           stop("Unknown delimiter"))
}
# This function generates a widget using widgetTools to collect all
# the arguments for read.table that are not yet collected by importWizard
getMoreArgs <- function(){
    args <- formals(read.table)

    args <- args[setdiff(names(args),
                         c("file", "header", "sep", "skip","quote"))]

    # Argument fill has to be defined using the value of
    # blank.lines.skip.
    args[["fill"]] <- !args[["blank.lines.skip"]]
    return(argsWidget(args))
}
# This function provides the interface for uers to decide whether to
# save the imported data in the global environment
getName4Data <- function(filename){
    # Gets ride of the separaters
    temp <- gsub(paste("^.*", .Platform$file.sep,
                                  "(.*)", sep = ""), "\\1", filename)
    # Gets ride of the extensions
    temp <- strsplit(temp, "\\.")[[1]][1]
    var <- tclVar(temp)
    # Destroy the window
    end <- function(){
        tkdestroy(top)
    }
    # Save the data
    save <- function(){
        temp <<- tclvalue(var)
        end()
    }
    # Do not save the data
    noSave <- function(){
        temp <<- NULL
        end()
    }
    ## Set up the interface
    top <- tktoplevel()
    nameFrame <- tkframe(top)
    tktitle(top) <- "BioC Data Import Wizard"
    inst <- tklabel(nameFrame, text = paste("Save data in .GlobalEnv as:"))
    tkpack(inst, side = "left")
    nameEntry <- tkentry(nameFrame, width = 10, textvariable = var)
    tkpack(nameEntry, side = "left")
    tkpack(nameFrame, side = "top", pady = 5, padx = 5)
    butFrame <- tkframe(top)
    noSaveBut <- tkbutton(butFrame, text = "Do'nt Save", width = 10,
                       command = noSave)
    saveBut <- tkbutton(butFrame, text = "Save", width = 10,
                        command = save)
    tkpack(noSaveBut, saveBut, side = "left")
    tkpack(butFrame, side = "top", pady = 5, padx = 5)

    tkwait.window(top)

    return(temp)
}
# Functions that initializes or constructs a colInfo class/object
# name - the name for a column
# type - data type for a column
# dropOrNot - a boolean indicating whether the column will be droped

    setClass("colInfo", representation(colName = "character",
                                       colType = "character",
                                       dropOrNot = "logical"))
    # Set the get methods
    if(!isGeneric("colName")){
        setGeneric("colName",
                   function(object) standardGeneric("colName"))
    }
    setMethod("colName", "colInfo",
              function(object) object@colName)
    if(!isGeneric("colType")){
        setGeneric("colType",
                   function(object) standardGeneric("colType"))
    }
    setMethod("colType", "colInfo",
              function(object) object@colType)
    if(!isGeneric("dropOrNot")){
        setGeneric("dropOrNot",
                   function(object) standardGeneric("dropOrNot"))
    }
    setMethod("dropOrNot", "colInfo",
              function(object) object@dropOrNot)
    # Define the replace methods
    if(!isGeneric("colName<-")){
        setGeneric("colName<-", function(object, value)
                   standardGeneric("colName<-"))
    }
    setReplaceMethod("colName", "colInfo", function(object, value){
                  object@colName <- value; object})
    if(!isGeneric("colTyype<-")){
        setGeneric("colType<-", function(object, value)
                   standardGeneric("colType<-"))
    }
    setReplaceMethod("colType", "colInfo", function(object, value){
                  object@colType <- value; object})
    if(!isGeneric("dropOrNot<-")){
        setGeneric("dropOrNot<-", function(object, value)
                   standardGeneric("dropOrNot<-"))
    }
    setReplaceMethod("dropOrNot", "colInfo", function(object, value){
                  object@dropOrNot <- value; object})

colInfo <- function(name, type, drop){
    new("colInfo", colName = name, colType = type, dropOrNot = drop)
}
# This function makes a list of items for selection denoted by a yes
# and no radio button.
#
# Copyright 2002, J. Zhang. All rights reserved
#
listSelect <-
    function(aList, topLbl = "Select Elements From The Following List",
             typeFun = stdType, valueFun = stdView)
{
    if(is.null(aList) || length(aList) < 1)
        stop("Invalid `aList'")

    returnList <- list()
    end <- FALSE

    for(i in names(aList))
        i <- tclVar(TRUE)

    clear <- function(){
        for(i in names(aList))
            tclvalue(i) <<- 1
    }

    cancel <- function(){
        tkdestroy(base)
    }

    finish <- function(){
        for(i in names(aList)){
            if(tclvalue(i) == 1)
                returnList[[i]] <<- TRUE
            else
                returnList[[i]] <<- FALSE
        }
        end <<- TRUE
        cancel()
    }

    butList <- list(Clear = clear, Cancel = cancel,
                    Finish = finish)

    base <- tktoplevel()
    tkwm.title(base,"BioC Widget")

    can <- tkcanvas(base, width = 300, height = 200)
    scr <- tkscrollbar(base, repeatinterval=5,
                       command=function(...)tkyview(can,...))
    tkconfigure(can, yscrollcommand=function(...)tkset(scr,...))

##FIXME: please try to document what is going on in here
##these long sets of operations are not easy to understand and should
##be documented somewhat

    topFrame <- tkframe(can)
    titlelbl <- tklabel(topFrame, text = topLbl, font = "Helvetica 12")
    tkpack(titlelbl, side = "top", fill = "both", expand = TRUE)
    selFrame <- tkframe(topFrame, borderwidth = 5)
    writeSelBox(selFrame, aList, typeFun, valueFun)
    tkpack(selFrame, side = "top")
    butFrame <- tkframe(topFrame, borderwidth = 5)
    writeBut(butFrame, butList)
    tkpack(butFrame, side = "top")
    tkcreate(can, "window", 5,5, anchor = "nw", window = topFrame)

    tkpack(can, side="left", fill="both", expand = TRUE)
    tkpack(scr, side="right", fill = "y", expand = TRUE)

    tkwait.window(base)

    if(end)
        return(returnList)
    else
        return(aList)
}

writeSelBox <- function(baseW, aList, typeFun = NULL, valueFun = NULL){

    LABELFONT <- "Helvetica 12"

    writeLabel(baseW, typeFun, valueFun)

    for (i in names(aList)){
        tempName <- tklabel(baseW, text = paste(i, ":", sep = ""),
                            font = LABELFONT, padx = 2)
        if(!is.null(typeFun))
            tempType <- tklabel(baseW,
                                text = eval(call(paste(quote(typeFun)),
                                eval(substitute(aList[[i]], list(i = i))))),
                                font = LABELFONT, padx = 2)
        else
            tempType <- tklabel(baseW, text = "")
        if(!is.null(valueFun)){
            fun <- function() {}
            body <- list(as.name("{"),
                         substitute(valueFun(aList[[i]]), list(i = i))

                         )
            body(fun) <- as.call(body)
            viewBut <- tkbutton(baseW, text = "View", command = fun)
        }
        else
            viewBut <- tklabel(baseW, text = "")
        tempCheck <-tkcheckbutton(baseW, text = "Accept", variable = i)
        tkgrid(tempName, tempType, viewBut, tempCheck)
        tclvalue(i) <- 1
    }
}

writeBut <- function(baseW, butList, butWidth = 6){

    butFrame <- tkframe(baseW, borderwidth = 5)
    for(i in 1:length(butList)){
        button <- tkbutton(butFrame, text= names(butList)[i],
                           width=butWidth, command = butList[[i]])
        tkpack(button, side = "left")
    }
    tkgrid(butFrame, columnspan = 4)
}

writeLabel <- function(baseW, typeFun, valueFun){
    LABELFONT <- "Helvetica 12"
    name <- tklabel(baseW, text = "Name", font = LABELFONT, padx = 2)
    if(!is.null(typeFun))
        type <- tklabel(baseW, text = "Type", font = LABELFONT, padx = 2)
    else
        type <- tklabel(baseW, text = "", font = LABELFONT, padx = 2)
    if(!is.null(valueFun))
        view <- tklabel(baseW, text = "Value", font = LABELFONT, padx = 2)
    else
        view <- tklabel(baseW, text = "", font = LABELFONT, padx = 2)
    option <- tklabel(baseW, text = "Option", font = LABELFONT, padx = 2)

    tkgrid(name, type, view, option)
}


neighborGeneFinder <- function(geneData, keyName = c("unigene", "locuslink"),
                               organism = c("human",  "mouse", "rat")){

    require(annotate) || stop(paste("neighbotGeneFinder requires the",
                                    "annotate package. Please have",
                                    "it installed"))
    organism <- tolower(match.arg(organism))
    keyName <- tolower(match.arg(keyName))
    collection <- list()
    neighbors <- NULL

    if(keyName == "unigene"){
        if(!require(paste(organism, "LLMappings", sep = ""),
                   character.only = TRUE)){
            stop(paste("Package ", organism, "LLMappings unavailable",
                      sep = ""))
        }
    }
    if(!require(paste(organism, "CHRLOC", sep = ""),
                character.only = TRUE)){
        stop(paste("Package ", organism, "CHRLOC unavailable",  sep = ""))
    }
    if(!is.element(keyName, colnames(geneData))){
        stop(paste("Key name", keyName, "is not one of the columns",
                   "of argument geneData"))
    }
    elementSelected <- function(index){
        for(i in 1:length(geneList)){
            if(i != index){
                tkselection.clear(geneList[[i]], 0, tksize(geneList[[i]]))
                tkselection.set(geneList[[i]],
                                tkcurselection(geneList[[index]]))
            }
        }
    }
    bindLists <- function(){
        for(i in 1:length(geneList)){
             selFun <- function(){}
             body <- list(as.name("{"),
                          substitute(elementSelected(j), list(j = i)),
                          substitute(findUDGenes()))
             body(selFun) <- as.call(body)
             tkbind(geneList[[i]], "<B1-ButtonRelease>", selFun)
         }
    }
    upDateNames <- function(){
        for(i in colnames(geneData)){
            writeList(geneList[[i]], geneData[,i], clear = TRUE)
        }
    }
    end <- function(){
        tkdestroy(base)
    }
    collect <- function(){
        key <- tclvalue(tkget(geneList[[keyName]],
                              (tkcurselection(geneList[[keyName]]))))
        if(!is.null(neighbors)){
            collection[[key]] <<- neighbors
        }
        tkconfigure(collectBut, state = "disabled")
    }
    findUDGenes <- function(){
        # Clean the list boxes
        tkdelete(upNeighbors, 0, "end")
        tkdelete(downNeighbors, 0, "end")
        key  <- tclvalue(tkget(geneList[[keyName]],
                              (tkcurselection(geneList[[keyName]]))))
        if(keyName == "unigene"){
            options(show.error.messages = FALSE)
            tempKey <- try(get(key, humanLLMappingsUG2LL))
            options(show.error.messages = TRUE)
            if(inherits(tempKey, "try-error")){
                tkmessageBox(title = "No map  error",
                             message = paste("Can not map", key,
                             " to a gene"), icon = "warning",
                             type = "ok")
                return(invisible())
            }else{
                key <- tempKey
            }
        }
        neighbors <<- findNeighbors(paste(organism, "CHRLOC", sep = ""),
                                   key, upBase = tclvalue(upBase),
                                   downBase = tclvalue(downBase),
                                   mergeOrNot = FALSE)
        upstream <- NULL
        downstream <- NULL
        for(i in names(neighbors)){
            upstream <- c(upstream, c(paste("chromosome", i),
                              as.vector(neighbors[[i]]$upstream), "  "))
            downstream <- c(downstream, c(paste("chromosome", i),
                              as.vector(neighbors[[i]]$downstream), "  "))
        }
        writeList(upNeighbors, upstream)
        writeList(downNeighbors, downstream)

        tkconfigure(collectBut, state = "normal")
        tkconfigure(locatBut, state = "normal")
    }

    base <- tktoplevel()
    on.exit(tkdestroy(base))
    tktitle(base) <- "BioC Neighbor Genes Finder"

    conFrame <- tkframe(base)

    leftFrame <- tkframe(conFrame)
    tkpack(tklabel(leftFrame, text = "Gene names"),
           expand = FALSE, fill = "x", pady = 5)
    geneFrame <- tkframe(leftFrame)
    geneList <- oneVScrList(geneFrame, geneData)
    tkconfigure(geneList[[keyName]], exportselection = FALSE)
#    tkbind(geneList[[keyName]], "<B1-ButtonRelease>", findUDGenes)
    bindLists()
    tkpack(geneFrame, expand = TRUE, fill = "y")

    tkpack(leftFrame, padx = 4, side = "left", expand = FALSE, fill = "y")

    rightFrame <- tkframe(conFrame)
    tkpack(tklabel(rightFrame, text = "Neighboring genes within"),
           side = "top", fill = "x", pady = 5)
    neighborFrame <- tkframe(rightFrame)
    upFrame <- tkframe(neighborFrame)
    baseFrame <- tkframe(upFrame)
    upBase <- tclVar(50000)
    tkpack(tkentry(baseFrame, textvariable = upBase, width = 10),
           expand = TRUE, fill = "x", side = "left")
    tkpack(tklabel(baseFrame, text = "bases up"), side = "left",
           expand = FALSE, fill = "x")
    tkpack(baseFrame, expand = FALSE, fill = "x")
    UNFrame <- tkframe(upFrame)
    upNeighbors <- makeViewer(UNFrame, vHeight = 10, vWidth = 15,
                          vScrol = TRUE, hScrol = FALSE, what = "list")
    tkpack(UNFrame, expand = TRUE, fill = "both")
    tkpack(upFrame, side = "left", expand = TRUE, fill = "y", padx = 5)

    downFrame <- tkframe(neighborFrame)
    baseFrame <- tkframe(downFrame)
    downBase <- tclVar(50000)
    tkpack(tkentry(baseFrame, textvariable = downBase, width = 10),
           expand = TRUE, fill = "x", side = "left")
    tkpack(tklabel(baseFrame, text = "bases down"), side = "left",
           expand = FALSE, fill = "x")
    tkpack(baseFrame, expand = FALSE, fill = "x")
    DNFrame <- tkframe(downFrame)
    downNeighbors <- makeViewer(DNFrame, vHeight = 10, vWidth = 15,
                          vScrol = TRUE, hScrol = FALSE, what = "list")
    tkpack(DNFrame, expand = TRUE, fill = "both")
    tkpack(downFrame, side = "left", expand = TRUE, fill = "y",
           padx = 5)
    tkpack(neighborFrame, expand = TRUE, fill = "both")

    butsFrame <- tkframe(rightFrame)
    locatBut <- tkbutton(butsFrame,
                         text = "Locate genes with new settings",
                         command = findUDGenes, state = "disabled")
    collectBut <- tkbutton(butsFrame, text = "Collect data",
                           command = collect, state = "disabled")
    tkgrid(locatBut, collectBut, padx = 8)
    tkpack(butsFrame, expand = FALSE, fill = "x")

    tkpack(rightFrame, padx = 4, expand = TRUE, fill = "both")
    tkpack(conFrame, expand = TRUE, fill = "both")

    exitBut <- tkbutton(base, text = "Exit", width = 20, command = end)
    tkpack(exitBut, pady = 5)

    tkwait.window(base)

    return(collection)
}
# This function gets a vector containing the names of R objects and
# returns a list of lists with name and object pairs. If a name is for
# a package, the content of the package will be associated with the
# package name.
#
# J. Zhang, copyright 2002, all rights reserved.
#
objNameToList <- function (objNames, env){
    returnList <- list()
    for(i in 1:length(objNames)){
        # if it is a package, the list contains the contents of
        # the package
        if(regexpr("^package", objNames[i]) > 0){
            returnList[[objNames[i]]] <- package.contents(gsub("(^package:)",
                                                          "\\", objNames[i]))
        }else{
            returnList[[objNames[i]]] <- get(objNames[i], env = env)
        }
    }
    return(returnList)
}
# This function creates a widget with a scrollable text box to show
# R object passed in the text box in a predefined format.
#
# Copyright 2002, J. Zhang. All right reserved.
#
objViewer <- function(toView, width = 40, height = 10){

    ok <- function() tkdestroy(base)

    base <- tktoplevel()
    tktitle(base) <- "Bioconductor Object Viewer"

    boxFrame <- tkframe(base)
    boxView <- makeViewer(boxFrame, vWidth = width,
                        vHeight = height, vScroll = TRUE,
                        hScroll = TRUE)
    writeList(boxView, toView)
    tkpack(boxFrame, side = "top")

    okBut <- tkbutton(base, text = "OK", command = ok)
    tkpack(okBut,side = "bottom")

    tkwait.window(base)
}
# This function provides the interface to browse objects in the work
# place and allows users to pick given number of objects to be
# returned.
# fun = a function that filters objects to be shown;
# textToShow = text to be shown as the title of the widget;
# nSelect = number of selection can be made (default to no limit).
#
#Copyright 2002, J. Zhang, all rights reserved
#

objectBrowser<- function (env = .GlobalEnv,
                          fun = noAuto, textToShow = "Select object(s)",
                          nSelect = -1){

    on.exit(end(), add = TRUE)

    LABELFONT1 <- "Helvetica 12 bold"
    LABELFONT2 <- "Helvetica 11"
    BUTWIDTH <- 8

    selectedObj <- NULL
    isPack <- FALSE
    returnObj <- NULL
    returnList <- NULL
    selIndex <- NULL
    objIndex <- NULL
    objsInSel <- NULL
    tempObj <- NULL
    currentEnv <- env

    # close window
    end <- function(){
        tkdestroy(base)
    }

    # Executed when a user clicks the end button. returnList that
    # contains names of selected objects will be updated before the
    # window closes.
    finish <- function(){
        if(length(objsInSel) != 0){
            if(nSelect == -1){
                returnList <<- objNameToList(objsInSel, currentEnv)
                end()
            }else{
                if(nSelect != length(objsInSel)){
                    tkmessageBox(title = "Wrong number", message =
                       paste("You can only select", nSelect, "object(s)"),
                       icon = "warning", type = "ok")
                }else{
                    returnList <<- objNameToList(objsInSel, currentEnv)
                    end()
                }
            }
        }else{
            returnList <<- NULL
            end()
        }
    }

    # Write the content of the global environment to the list box for
    # object names
    viewEnv <- function(env){
        writeList(listView, pickObjs(objNames = ls(env = env,
                                    all = TRUE), fun = fun), clear = TRUE)
        writeCap(".GlobalEnv")
    }
    # Executed when a user double clicks an object that is an R
    # environment. List object names in an enviroment to the list
    # boxes for objects.
    doEnv <- function (item){
        writeList(listView,  pickObjs(objNames = ls(env = get(item)),
                                      fun = fun), clear = TRUE)
        writeCap(item)
        if(!is.null(parent.env(get(item))))
            tkconfigure(upBut, state = "normal")
    }
    # Executed when a user couble clicks an object that is an R
    # package. List all object names in the list box for objects.
    doPack <- function (index, pack){
        whichOne <- as.numeric(index) + 1
        writeList(listView, ls(pos = whichOne), clear = TRUE)
        isPack <<- TRUE
        writeCap(pack, asis = TRUE)
        tkconfigure(upBut, state = "normal")
    }
    # Will be done for other objects
    doElse <- function(){
        # This a temp function for now. More checking will be implemented
    }
    # Executed when a user double clicks an ojbect that is a
    # list. Shows the number of columns, number of rows and column names.
    doList <- function (aList){

        if(is.null(ncol(get(aList))))
            towrite <- c("Type: List",
                         paste("Length:", length(get(aList))))


        toWrite <- c("Type: List",
                     paste("Number of columns:", ncol(get(aList))),
                     paste("Number of row(s):", nrow(get(aList))),
                     paste("Column Name(s):"),names(get(aList)))

        writeList(listView, toWrite, clear = TRUE)
        writeCap(aList)
    }

    # Executed when a user double clicks an object name in a list box.
    # Shows the content of the object if there is any
    dClick <- function (){
        selectedObj <<- as.character(tkget(listView,
                                           tkcurselection(listView)))
        goin()
    }

    # Determines the type of an object that is double clicked to
    # decide what to do
    goin <- function (){
        if(!is.null(selectedObj)){
            if(regexpr("^package", selectedObj) > 0){
                objType <- "package"
            }else{
                objType <- typeof(get(selectedObj))
            }
            switch(objType,
               "environment" = doEnv(selectedObj),
               "package" =  doPack(tkcurselection(listView), selectedObj),
               "list" = doList(selectedObj),
               doElse()
            )
            tkconfigure(selectBut, state = "disabled")
            selectedObj <<- NULL
        }
    }

    # Response to a single click to an object name in the list box for
    # object names
    sClick <- function () {
        selectedObj <<- NULL
        tkconfigure(selectBut, state = "normal")
        selIndex <<- unlist(strsplit(
                              as.character(tkcurselection(listView)), " "))
        if(length(selIndex) == 1){
            tempObj <<- as.character(tkget(listView, selIndex))
        }else{
            for(i in selIndex){
                tempObj <<- c(tempObj,
                                  as.character(tkget(listView, i)))
            }

        }
    }

    # When the global environment is clicked and shows the content
    getAct <- function(){
        selectedObj <<- ".GlobalEnv"
        goin()
    }

    # Moves one step back the path of objects browsered
    up <- function(){
        options(show.error.messages = FALSE)
        tryMe <- try(parent.env(get(selectedObj)))
        options(show.error.messages = TRUE)

        if(isPack || selectedObj == ".GlobalEnv" ||
           inherits(tryMe, "try-error")){
            writeList(listView, pickObjs(objNames = search(),
                                             fun = fun), clear = TRUE)
            writeCap("Top Level")
            tkconfigure(upBut, state = "disabled")
        }else{
            writeList(listView,
                    pickObjs(objNames = ls(env = get(selectedObj)),
                                             fun = fun), clear = TRUE)
            writeCap(selectedObj)

        }
        tkconfigure(selectBut, state = "disabled")
    }
    # Writes the name of an object in the list box for object names to
    # the list box for selected objects.
    selectObj <- function (){
        objsInSel <<- c(objsInSel, tempObj)
        objsInSel <<- unique(objsInSel)
        writeSelection(objsInSel)
        tkconfigure(clearBut, state = "normal")
        tkconfigure(selectBut, state = "disabled")
    }
    # Removes everything from the list box for selected ojbects
    clearSelection <- function(){
        objsInSel <<- NULL
        tkdelete(selectView, 0, "end")
        tkconfigure(clearBut, state = "disabled")
        tkconfigure(removeBut, state = "disabled")
        tkconfigure(selectBut, state = "disabled")
    }
    # Destroy the widow and returns NULL
    cancel <- function (){
        objsInSel <<- NULL
        finish()
    }
    # Removes items from the list box for selected objects
    removeSelection <- function (){
        toRemove <- NULL
        if(length(objIndex) > 0){
            for(i in objIndex)
                toRemove <- c(toRemove, -(as.numeric(i) + 1))

            objsInSel <<- objsInSel[toRemove]
            writeSelection(objsInSel)
        }
        tkconfigure(removeBut, state = "disabled")
    }
    # When an object in the box for selected objects is clicked, the
    # index of the object name is remembered
    selClick <- function (){
        objIndex <<- NULL
        tkconfigure(removeBut, state = "normal")
        objIndex <<- unlist(strsplit(
                      as.character(tkcurselection(selectView)), " "))
    }
    # Write to the list box for selected objects
    writeSelection <- function (toWrite){
        writeList(selectView, toWrite, clear = TRUE)
        #tkdelete(selectView, 0, "end")
        #for(i in toWrite)
        #    tkinsert(selectView, "end", i)
        fileIndex <<- NULL
    }
    # Writes to the top of the widget to indicate the current environment
    writeCap <- function(objName, asis = FALSE){
        if(asis){
            tkconfigure(labl1, text = objName)
        }else{
            if(objName == "Top Level"){
                tkconfigure(labl1, text = "Top level")
            }else{
                tkconfigure(labl1, text = objName)
            }
        }
    }

    ## This portion sets up the interface
    base <- tktoplevel()
    tktitle(base) <- paste("Bioconductor Object Browser")
    # Set up the labels for list boxes of objects
    capFrame <- tkframe(base)
    noteLabel <- tklabel(capFrame, text = textToShow, font = LABELFONT1)
    labl1 <- tklabel(capFrame, text = " ", font = LABELFONT2)
    labl2 <- tklabel(capFrame, text = "Selected", font = LABELFONT2)
    dummyLabel <- tklabel(capFrame, text = "           ")
    tkgrid(noteLabel, columnspan = 3)
    tkgrid(labl1, dummyLabel, labl2)
    tkgrid(capFrame, columnspan = 2, padx = 10)
    # Sets up the list boxes. One for objects to browse and one for selected
    leftFrame <- tkframe(base)
    listFrame <- tkframe(leftFrame)
    listView <- makeViewer(listFrame)
    tkgrid(listFrame, columnspan = 2)
    tkconfigure(listView, selectmode = "extended", font = LABELFONT2)
    tkbind(listView, "<Double-Button-1>", dClick)
    tkbind(listView, "<B1-ButtonRelease>", sClick)
    # Puts in the buttons for the list box for object names
    butFrame <- tkframe(leftFrame)
    upBut <- tkbutton(butFrame, text = "Up", width = BUTWIDTH,
		      command = up)
    activeBut <- tkbutton(butFrame, text = "Back", width = BUTWIDTH,
		      command = getAct)
    selectBut <- tkbutton(butFrame, text = "Select >>", width = BUTWIDTH,
                          command = selectObj, state = "disabled")
    tkgrid(upBut, selectBut)
    tkgrid(activeBut, columnspan = 2)
    tkgrid(butFrame)
    # puts in the buttons for list box for selected objects
    rightFrame <- tkframe(base)
    selectFrame <- tkframe(rightFrame)
    selectView <- makeViewer(selectFrame)
    tkconfigure(selectView, selectmode = "extended", font = LABELFONT2)
    tkgrid(selectFrame, columnspan = 2)
    tkbind(selectView, "<B1-ButtonRelease>", selClick)

    butFrame2 <- tkframe(rightFrame)
    removeBut <- tkbutton(butFrame2, text = "<< Remove", width = BUTWIDTH,
		      command = removeSelection, state = "disabled")
    clearBut <- tkbutton(butFrame2, text = "Clear", width = BUTWIDTH,
		      command = clearSelection, state = "disabled")
    canBut <- tkbutton(butFrame2, text = "Cancel", width = BUTWIDTH,
                       command = cancel)
    endBut <- tkbutton(butFrame2, text = "Finish", width = BUTWIDTH,
		       command = finish)
    tkgrid(removeBut, clearBut)
    tkgrid(canBut, endBut)
    tkgrid(butFrame2)

    tkgrid(leftFrame, rightFrame)

    viewEnv(env)

    tkwait.window(base)

    return(returnList)
}

# This function checks a vector of file names from a directory and
# determins what to be sent for desplay on the fileBrowser  based
# on the default and user speficied parameters.
#
# Copyright 2002, J. Zhang. All rights reserved.
#
pickFiles <- function (fileNames, fun = function(x) TRUE,
                       prefix = NULL, suffix = NULL,
                       exclude = .Platform$file.sep){

    unTouched <- fileNames[regexpr(exclude, fileNames) > 0]
    rest <- setdiff(fileNames, unTouched)
    if(length(rest) > 0){
        rest <- rest[sapply(rest, fun)]

        if(!is.null(prefix)){
            tryMe <- hasChar(prefix, "prefix")
            rest <- rest[sapply(rest, tryMe)]
        }
        if(!is.null(suffix)){
            tryMe <- hasChar(suffix, "suffix")
            rest <- sapply(rest, tryMe)
        }
        return(c(unTouched, rest))
    }else{
        return(unTouched)
    }
}


# This function provides the interface for users to select items from
# a set of source elements.
#
# items - a vector of character strings for the items to be picked
# from.
#
# Copyright 2002 J. Zhang, all rights reserved
#
pickItems <- function (items){

    on.exit(end())

    columns <- NULL
    colIndex <- NULL
    indexInSel <- NULL
    text2 <- "Select item(s) from the list box on the left"

    end <- function(){
        tkdestroy(base)
    }

    # When a user double clicks a column name in the list box, put the
    # library name in the vector for selected SAGE libraries.
    colDClicked <- function(){
        column <- as.character(tkget(colView,(tkcurselection(colView))))
        columns <<- unique(c(columns, column))
        writeList(selectView, columns)
        tkconfigure(clearBut, state = "normal")
        tkconfigure(selectBut, state = "disabled")
    }
    # When a user single clicked a column name, remember that name and
    # activate the select button
    colSClicked <- function(){
        colIndex <<- unlist(strsplit(
                              as.character(tkcurselection(colView)), " "))
        tkconfigure(selectBut, state = "normal")
    }
    # When a user click the select button, put the selected column name
    # in the list box for selected columns
    selectCol <- function(){
        for(i in colIndex){
            columns <<- c(columns, as.character(tkget(colView, i)))
        }
        writeList(selectView, unique(columns))
        tkconfigure(selectBut, state = "disabled")
        tkconfigure(clearBut, state = "normal")
    }
    # Remember the column name when a name in the selected column names list
    # box is single clicked
    selSClick <- function(){
        indexInSel <<- unlist(strsplit(
                              as.character(tkcurselection(selectView)), " "))
        tkconfigure(remBut, state = "normal")
    }
    # Removes the selected column name from the list box for selected
    # column names
    dropSelCol <- function(){
        for(i in indexInSel){
            columns <<- columns[columns != as.character(tkget(selectView, i))]
        }
        writeList(selectView, unique(columns))
        tkconfigure(remBut, state = "disabled")
    }
    # Remove everything from the list box for selected column names
    clearSelCol <- function(){
        writeList(selectView, "")
        columns <<- NULL
        tkconfigure(clearBut, state = "disabled")
    }

    base <- tktoplevel()
    tktitle(base) <- "BioC tkWidgets"

    # Lists for column names
    midFrame <- tkframe(base)
    text2Label <- tklabel(midFrame, text = text2, font = "Helvetica 12")
    tkgrid(text2Label, columnspan = 2, pady = 2)
    # Label for available SAGE libs
    leftFrame <- tkframe(midFrame)
    label1 <- tklabel(leftFrame, text = "Items to pick",
                        font = "Helvetica 11")
    tkpack(label1)
    # List box showing the available SAGE libs
    colFrame <- tkframe(leftFrame)
    colView <- makeViewer(colFrame, vWidth = 40, vHeight = 15,
                           hScroll = TRUE)
    tkbind(colView, "<Double-Button-1>", colDClicked)
    tkbind(colView, "<B1-ButtonRelease>", colSClicked)
    tkpack(colFrame, padx = 5)
    selectBut <- tkbutton(leftFrame, text = "Select >>", width = 12,
		      state = "disabled", command = selectCol)
    tkpack(selectBut)
    tkconfigure(colView, selectmode = "extended")
    # Put the list box for selected SAGE libs and the associated buttons
    rightFrame <- tkframe(base)
    label2 <- tklabel(rightFrame, text = "Picked items",
                        font = "Helvetica 11")
    tkgrid(label2, columnspan = 2)
    selLFrame <- tkframe(rightFrame)
    selectView <- makeViewer(selLFrame, vWidth = 40, vHeight = 15,
                             hScroll = TRUE)
    tkgrid(selLFrame, columnspan = 2, padx = 5)
    tkconfigure(selectView, selectmode = "extended")
    tkbind(selectView, "<B1-ButtonRelease>", selSClick)
    remBut <- tkbutton(rightFrame, text = "<< Remove", width = 12,
		      state = "disabled", command = dropSelCol)
    clearBut <- tkbutton(rightFrame, text = "Clear", width = 12,
		      state = "disabled", command = clearSelCol)
    tkgrid(remBut, clearBut)
    tkgrid.configure(remBut, sticky = "e")
    tkgrid.configure(clearBut, sticky = "w")

    tkgrid(leftFrame, rightFrame)
    tkpack(midFrame)

    # Put the end button
    endBut <- tkbutton(base, text = "Finish", width = 12, command = end)
    tkpack(endBut, pady = 5)

    writeList(colView, items)

    tkwait.window(base)
    return(columns)
}









# This function takes a vector of strings for object names is an
# environment specified by the calling function and returns a vactor
# of strings for object names selected by the function passed.
#
# Copyright 2002, J. Zhang. All rights reserved.
#

pickObjs <- function (objNames, fun = noAuto){

    if(length(objNames) == 0){
        return(objNames)
    }

    whichOnes <- NULL

    for(i in objNames){
        whichOnes <- c(whichOnes, fun(i))
    }
    return (objNames[whichOnes])
}

noAuto <-  function(x) {
    if(regexpr("^Autoloads", x) > 0 ){
        return(FALSE)
    }else{
        return(TRUE)
    }
}
# This function prints the values for the Name, Value, canEdit, and
# buttonText of all the value containing widget elements on the widget
# created by using the function widgetRender.
#
# Copyright J. Zhang 2002, all rights reserved.
#
print.Widget <- function (x, ...){

    wList <- WwList(x)
    for(i in names(wList)) {
        pW <- wList[[i]]
        print(paste("widget_", i, ":", sep = ""))
        class(pW) <- "pWidget"
        print.pWidget(pW)
    }
    # for now we just the other parameters as a vector. Will do more later.
    if(length(unlist(list(...))) > 0)
        unlist(list(...))
}
# This function prints the values for the Name, Value, canEdit, and
# buttonText of a value containing widget element on the widget
# created by using the function widgetRender.
#
# Copyright J. Zhang 2002, all rights reserved.
#
print.pWidget <- function (x, ...){
    print(paste("Name = ", WName(x), ";", sep = ""))
    print(paste("Value = ", WValue(x), ";", sep = ""))
    print(paste("canEdit = ", WcanEdit(x), ";", sep = ""))
    if(!is.null(WbuttonFun(x)))
       print(paste("buttonText = ", WbuttonText(x), ";", sep = ""))
    # For now we just print the other prameters as a vector. Will do
    # more later.
    if(length(unlist(list(...))) > 0)
        unlist(list(...))
}
# Function widgetRender takes a list of a widget list that has a name,
# value, ... buttonText, and buttonFun as shown below plus a
# prefunction, postfunction, and a few buttons. The functions listed
# here provide a defined interface for accessing and manipulate these
# values. The functions are currently temporary and will be modified
# for better names.

WName <- function(x) x$Name
WValue <- function(x) x$Value
"WValue<-" <- function(x, value) {x$Value <- value; x}
WtoText <- function(x) x$toText
WfromText <- function(x) x$fromText
WcanEdit <- function(x) x$canEdit
WbuttonText <- function(x) x$buttonText
WbuttonFun <- function(x) x$buttonFun

#get and set the widget list
##FIXME: shouldn't these be getWidgetList and setWidgetList?
WwList <- function(x) x$wList
"WwList<-" <- function(x, value) {x$wList <- value; x}

##what about getIndexValuepWidget and setIndexValuepWidget
#get and set index' pWidgets value
WLValue <- function(x, index) WValue(WwList(x)[[index]])
"WLValue<-" <- function(x, index, value)
    {WValue(WwList(x)[[index]]) <- value; x}


WpreFun <- function(x) x$preFun
WpostFun <- function(x) x$postFun
WRButtons <- function(x) x$buttons
WEnd <- function(x) x$end

# These functions provide standard outputs that are going to be
# rendered to a widget to indicate the type or value of the R object
# shown on the widget.
#
# Copyright 2002, J. Zhang. All rights reserved
#

##what is formular????
##is this really what you want?? shouldn't it always use a class if 
##there is one? 
stdType <- function(toCheck){
    if(inherits(toCheck, "formular")){
        return("formula")
    }else{
        return(mode(toCheck))
    }
}

stdView <- function(toView){
    if(inherits(toView, "formular")){
        toView <- format(toView)
    }
    objViewer(toView)
}

tkMIAME <- function(){
  tt <- tktoplevel(width = 30)
  tkconfigure(tt, background = "white")
  tktitle(tt) <- "MIAME Information"

  tbar.fr <- tkframe(tt,relief="sunken",borderwidth=5,width="5i")
  win.fr <- tkframe(tt,relief="sunken",borderwidth=5,width="5i",height="7i")
  tkconfigure(win.fr, background = "white")
  tkpack(tbar.fr,side="top",fill="x")
  tkpack(win.fr,side="top", fill="both")

  space.label1 <- tklabel(win.fr, text = "")
  tkconfigure(space.label1, background = "white")
  space.label2 <- tklabel(win.fr, text = "")
  tkconfigure(space.label2, background = "white")
  space.label3 <- tklabel(win.fr, text = "")
  tkconfigure(space.label3, background = "white")
  space.label4 <- tklabel(win.fr, text = "")
  tkconfigure(space.label4, background = "white")
  space.label5 <- tklabel(win.fr, text = "")
  tkconfigure(space.label5, background = "white")
  space.label6 <- tklabel(win.fr, text = "")
  tkconfigure(space.label6, background = "white")
  space.label7 <- tklabel(win.fr, text = "")
  tkconfigure(space.label7, background = "white")
  space.label8 <- tklabel(win.fr, text = "")
  tkconfigure(space.label8, background = "white")
  space.label9 <- tklabel(win.fr, text = "")
  tkconfigure(space.label9, background = "white")
  space.label10 <- tklabel(win.fr, text = "")
  tkconfigure(space.label10, background = "white")

  ExpName.label <- tklabel(win.fr, text="Experimenter's Name: ")
  tkconfigure(ExpName.label, background = "white")
  ExpName.var <- tclVar()
  tclvalue(ExpName.var) <- ""
  ExpName.entry <- tkentry(win.fr, width = 25,textvariable = ExpName.var)
  tkconfigure(ExpName.entry, background = "white")

  LabName.label <- tklabel(win.fr, text="Laboratory: ")
  tkconfigure(LabName.label, background = "white")
  LabName.var <- tclVar()
  tclvalue(LabName.var) <- ""
  LabName.entry <- tkentry(win.fr, width = 25,textvariable = LabName.var)
  tkconfigure(LabName.entry, background = "white")

  Contact.label <- tklabel(win.fr, text="Contact Information: ")
  tkconfigure(Contact.label, background = "white")
  Contact.txt <- tktext(win.fr, width = 25, height =5)
  Contact.var <- tclVar()
  tkconfigure(Contact.txt, background = "white")

  ExpTitle.label <- tklabel(win.fr, text="Experiment Title: ")
  tkconfigure(ExpTitle.label, background = "white")
  ExpTitle.var <- tclVar()
  tclvalue(ExpTitle.var) <- ""
  ExpTitle.entry <- tkentry(win.fr, width = 25,textvariable = ExpTitle.var)
  tkconfigure(ExpTitle.entry, background = "white")

  Desc.label <- tklabel(win.fr, text="Experiment Description: ")
  tkconfigure(Desc.label, background = "white")
  Desc.txt <- tktext(win.fr, width = 25, height =10)
  Desc.var <- tclVar()
  tkconfigure(Desc.txt, background = "white")


  URL.label <- tklabel(win.fr, text="URL: ")
  tkconfigure(URL.label, background = "white")
  URL.var <- tclVar()
  tclvalue(URL.var) <- ""
  URL.entry <- tkentry(win.fr, width = 25,textvariable = URL.var)
  tkconfigure(URL.entry, background = "white")



  Exit.but <- tkbutton(win.fr, text = "Finish", command = function(){
    tclvalue(Contact.var) <- tclvalue(tkget(Contact.txt,"0.0","end"))
    tclvalue(Desc.var) <- tclvalue(tkget(Desc.txt,"0.0","end"))
    tkdestroy(tt)
  })

  tkconfigure(Exit.but, background = "white")


  tkgrid(space.label1, row =1)
  tkgrid(space.label2, row =2)
  tkgrid(ExpName.label, row = 3, column = 1, sticky ="e")
  tkgrid(ExpName.entry, row = 3, column =2, sticky = "w")
  tkgrid(space.label3, row =4)
  tkgrid(LabName.label, row = 5, column = 1, sticky = "e")
  tkgrid(LabName.entry, row = 5, column =2, sticky = "w")
  tkgrid(space.label4, row =6)
  tkgrid(Contact.label, row = 7, column = 1, sticky = "e")
  tkgrid(Contact.txt, row = 7, column =2, sticky = "w")
  tkgrid(space.label5, row =8)
  tkgrid(ExpTitle.label, row = 9, column = 1, sticky = "e")
  tkgrid(ExpTitle.entry, row = 9, column =2, sticky = "w")
  tkgrid(space.label6, row =10)
  tkgrid(Desc.label, row = 11, column = 1, sticky = "e")
  tkgrid(Desc.txt, row = 11, column =2, sticky = "w")
  tkgrid(space.label7, row =12)
  tkgrid(URL.label, row = 13, column = 1, sticky = "e")
  tkgrid(URL.entry, row = 13, column =2, sticky = "w")
  tkgrid(space.label8, row =14)
  tkgrid(space.label9, row =15)
  tkgrid(Exit.but, row = 16, column =2, sticky = "e")
  tkgrid(space.label10, row =17)

  tkwait.window(tt)

  miame.lst <- list(tclvalue(ExpName.var), tclvalue(LabName.var),tclvalue(Contact.var),tclvalue(ExpTitle.var),tclvalue(Desc.var),tclvalue(URL.var))
  names(miame.lst) <- c("ExperimentName","LabName","ContactInfo","ExperimentTitle","Description","URL")

  return(miame.lst)
}



# pW1 <- list(Name="Experiments's Name: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# pW2 <- list(Name="Laboratory: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# pW3 <- list(Name="Contact Information: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# pW4 <- list(Name="Experimental Title: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# pW5 <- list(Name="Experiment Description: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# pW6 <- list(Name="URL: ", Value="", toText=NULL, fromText=NULL, canEdit=TRUE, buttonFun = NULL, buttonText = NULL)

# widget1 <- list(wList = list(a = pW1, b = pW2, c = pW3, d = pW4, e = pW5, f = pW6), preFun = NULL, postFun = NULL)

# x <- widgetRender(widget1, "MIAME Information")

tkSampleNames <- function(...,filenames=character(0)){


  auxnames <- as.list(substitute(list(...)))[-1]
  filenames <- .Primitive("c")(filenames, auxnames)

  NumSamples <- length(filenames)

  sample.names <- array("",NumSamples)

  for(i in 1:NumSamples){
    s2 <-strsplit(strsplit(filenames[[i]],"/")[[1]][length(strsplit(filenames[[i]],"/")[[1]])],"\\.")[[1]]
    sample.names[i] <- s2[1]
  }

  tt <- tktoplevel(width = 30)
  tkconfigure(tt, background = "white")
  tktitle(tt) <- "Sample Information"

  tbar.fr <- tkframe(tt,relief="sunken",borderwidth=5,width="5i")
  win.fr <- tkframe(tt,relief="sunken",borderwidth=5,width="5i",height="7i")
  tkconfigure(win.fr, background = "white")
  tkpack(tbar.fr,side="top",fill="x")
  tkpack(win.fr,side="top", fill="both")


  array.name.lst <- list()
  array.entry.lst <- list()


  var.lst <- list()
  for(i in 1:NumSamples) {
    var.lst[[i]] <-tclVar()
    tclvalue(var.lst[[i]]) <- sample.names[i]
  }

  desc.lst <- list()
  for(i in 1:NumSamples) {
    desc.var <- tclVar("")
    desc.lst <- c(desc.lst,list(desc.var))
  }


  array.var <- list()

  space.label1 <- tklabel(win.fr, text = "")
  tkconfigure(space.label1, background = "white")
  space.label2 <- tklabel(win.fr, text = "")
  tkconfigure(space.label2, background = "white")
  space.label3 <- tklabel(win.fr, text = "")
  tkconfigure(space.label3, background = "white")
  tkgrid(space.label1,row = 1)

  array.name.lst <- tklabel(win.fr, text="File PathNames")
  tkconfigure(array.name.lst, background = "white")
  tkgrid(array.name.lst, row = 2, column = 1)

  array.entry.lst <- tklabel(win.fr, text="Sample Names")
  tkconfigure(array.entry.lst, background = "white")
  tkgrid(array.entry.lst, row = 2, column = 2)

  desc.entry.lst <- tklabel(win.fr, text="Description")
  tkconfigure(desc.entry.lst, background = "white")
  tkgrid(desc.entry.lst, row = 2, column = 3)



  for(i in 1:NumSamples){
    array.name.lst[[i]] <-tklabel(win.fr, text=filenames[[i]])
    tkconfigure(array.name.lst[[i]], background = "white")
    tkgrid(array.name.lst[[i]], row = (i+2), column = 1)
    tkgrid.configure(array.name.lst[[i]], sticky = "e")

    array.entry.lst [[i]] <- tkentry(win.fr, width = 30, textvariable
    = var.lst[[i]])
    tkconfigure(array.entry.lst[[i]], background = "white")
    tkgrid(array.entry.lst[[i]], row = (i+2), column = 2)
    tkgrid.configure(array.entry.lst[[i]], sticky = "e")

    desc.entry.lst [[i]] <- tkentry(win.fr, width = 30, textvariable = desc.lst[[i]])
    tkconfigure(desc.entry.lst[[i]], background = "white")
    tkgrid(desc.entry.lst[[i]], row = (i+2), column = 3)
    tkgrid.configure(desc.entry.lst[[i]], sticky = "w")




  }


  tkgrid(space.label3,row=(NumSamples+7))
  finish.but <- tkbutton(win.fr, text = "Continue", command = function(){tkdestroy(tt)})
  tkconfigure(finish.but, background = "white")
  tkgrid(finish.but, row = (NumSamples + 8), column = 2)
  tkgrid.configure(finish.but, sticky="e")

  tkwait.window(tt)

  sam.vec <- matrix("",nrow=NumSamples,ncol=2)
  colnames(sam.vec) <- c("Sample Names","Description")
  for(i in 1:NumSamples){
    sam.vec[i,1] <- tclvalue(var.lst[[i]])
    sam.vec[i,2] <- tclvalue(desc.lst[[i]])}
  return(sam.vec)
}



tkphenoData <- function(sampleNames){
  eerieEnv <- new.env()

  sample.length <- length(sampleNames)

#############################################
#hierarchy of widgets
#############################################

  NumWidgets <- 5 #number of widgets
  hierarchy <- list()
  for(i in 1:NumWidgets){
    hierarchy[[i]] <-function(){}
  }


#############################################
###third widget -- Main widget
#############################################
  hierarchy[[3]] <- function(){



    tt3 <- tktoplevel(width = 30)
    tkconfigure(tt3, background = "white")
    tktitle(tt3) <- "Pheno Data"

    tbar.fr <- tkframe(tt3,relief="sunken",borderwidth=5,width="5i")
    win.fr <- tkframe(tt3,relief="sunken",borderwidth=5,width="5i",height="7i")
    tkconfigure(win.fr, background = "white")
    tkpack(tbar.fr,side="top",fill="x")
    tkpack(win.fr,side="top", fill="both")

    assign("NumSamples", sample.length,eerieEnv)
    assign("NumCovar",get("NumCovariates",eerieEnv),eerieEnv)
    assign("NewNumCovariates",as.numeric(tclvalue(get("NewNumCovariates.var",eerieEnv))),eerieEnv)
    assign("OldNumCovariates1",as.numeric(tclvalue(get("OldNumCovariates1.var",eerieEnv))),eerieEnv)

    array.name.lst <-list()
    array.entry.lst <- list()

    cb.lst <- list()
    if(!is.na(match("var.lst",ls(eerieEnv)))){
      assign("var.lst", get("var.lst",eerieEnv),eerieEnv)}
    else{
      assign("var.lst", list(),eerieEnv)
      for(i in 1:get("NumSamples",eerieEnv)) {
        assign("var",tclVar(sampleNames[i]),eerieEnv)
        assign("var.lst",c(get("var.lst",eerieEnv),list(get("var",eerieEnv))),eerieEnv)
        ## assign(tclvalue(get("var.lst",eerieEnv)[[i]]),sampleNames[i],eerieEnv)
      }}
    if(get("OldNumCovariates1",eerieEnv) != get("NumCovariates",eerieEnv)){
      assign("cov.lst",list(),eerieEnv)

      for(i in 1:get("NumSamples",eerieEnv)){
        assign("dummy.lst",list(),eerieEnv)
        for(j in 1:get("NumCovar",eerieEnv)){
          assign("dummy.var",tclVar(),eerieEnv)
          assign("dummy.lst",c(get("dummy.lst",eerieEnv),list(get("dummy.var",eerieEnv))),eerieEnv)}
        assign("cov.lst",c(get("cov.lst",eerieEnv),list(get("dummy.lst",eerieEnv))),eerieEnv)
      }


      ##for(i in 1:get("NumSamples",eerieEnv)){
      ##  assign(get("cov.lst",eerieEnv)[[i]],list(),eerieEnv)
      ##  for(j in 1:get("NumCovar",eerieEnv)){
      ##       assign("array.cov",tclVar(),eerieEnv)
      ##       assign(get("cov.lst",eerieEnv)[[i]],c(get("cov.lst",eerieEnv)[[i]],list(get("array.cov",eerieEnv))),eerieEnv)
      ##     }
    }
    else{
      if(!is.na(match("cov.lst",ls(eerieEnv)))){
        assign("cov.lst","cov.lst",eerieEnv)}
      else{
        assign("cov.lst",list(),eerieEnv)

        for(i in 1:get("NumSamples",eerieEnv)){
          assign("dummy.lst",list(),eerieEnv)
          for(j in 1:get("NumCovar",eerieEnv)){
            assign("dummy.var",tclVar(),eerieEnv)
            assign("dummy.lst",c(get("dummy.lst",eerieEnv),list(get("dummy.var",eerieEnv))),eerieEnv)}
          assign("cov.lst",c(get("cov.lst",eerieEnv),list(get("dummy.lst",eerieEnv))),eerieEnv)
        }


        ##for(i in 1:get("NumSamples",eerieEnv)){
        ##  assign(get("cov.lst",eerieEnv)[[i]],list(),eerieEnv)
        ##  for(j in 1:get("NumCovar",eerieEnv)){
        ##       assign("array.cov",tclVar(),eerieEnv)
        ##       assign(get("cov.lst",eerieEnv)[[i]],c(get("cov.lst",eerieEnv)[[i]],list(get("array.cov",eerieEnv))),eerieEnv)
        ##q()

      }}
    ##array.var <<- list()

    space.label1 <- tklabel(win.fr, text = "")
    tkconfigure(space.label1, background = "white")
    space.label2 <- tklabel(win.fr, text = "")
    tkconfigure(space.label2, background = "white")
    space.label3 <- tklabel(win.fr, text = "")
    tkconfigure(space.label3, background = "white")
    space.label4 <- tklabel(win.fr, text = "")
    tkconfigure(space.label4, background = "white")
    space.label5 <- tklabel(win.fr, text = "")
    tkconfigure(space.label5, background = "white")
    tkgrid(space.label1,row = 1)

    array.entry.lst <- tklabel(win.fr, text="Sample Names")
    tkconfigure(array.entry.lst, background = "white")
    tkgrid(array.entry.lst, row = 2, column = 2)


    for(j in 1:get("NumCovar",eerieEnv)) {
      cb.lst <- tklabel(win.fr, text = tclvalue(get("cov.name.lst",eerieEnv)[[j]]))
      tkconfigure(cb.lst, background = "white")
      tkgrid(cb.lst, row = 2, column = (j+3))
    }

    for(i in 1:get("NumSamples",eerieEnv)){
      array.name.lst <-tklabel(win.fr, text=paste("Array",i))
      tkconfigure(array.name.lst, background = "white")
      tkgrid(array.name.lst, row = (i+2), column = 1)


      array.entry.lst [[i]]<- tkentry(win.fr, width = 30, textvariable
      = get("var.lst",eerieEnv)[[i]], state = "disabled")
      tkconfigure(array.entry.lst[[i]], background = "white")
      tkgrid(array.entry.lst[[i]], row = (i+2), column = 2)
      tkgrid.configure(array.entry.lst[[i]], sticky = "e")

      for(j in 1:get("NumCovar",eerieEnv)){
        cb.lst[[j]] <- tkentry(win.fr, width=6,textvariable = get("cov.lst",eerieEnv)[[i]][[j]])
        tkconfigure(cb.lst[[j]], background = "white")
        tkgrid(cb.lst[[j]], row = (i+2), column =(j+3))
      }
    }
    tkgrid(space.label3)
#    tkgrid(space.label4)
#    tkgrid(space.label5)

#    tkgrid(space.label3,row=(get("NumSamples",eerieEnv)+3))
#    tkgrid(space.label4,row=(get("NumSamples",eerieEnv)+4))
#    tkgrid(space.label5,row=(get("NumSamples",eerieEnv)+5))

    ###### JZ add this to save the phenoData object to .Global Env
    saveFrame <- tkframe(win.fr, background = "white")
    tkpack(tklabel(saveFrame, text = "Save phenoData as", background = "white"),
           side = "left", expand = FALSE)
    pdName <- tclVar("")
    pdNameEntry <- tkentry(saveFrame, width = 40, textvariable = pdName,
                           background = "white")
    tkpack(pdNameEntry, side = "left", expand = TRUE, fill = "x")
    tkgrid(saveFrame, pady = 5, columnspan = j + 4)
    ######

    ## JZ modified the layout a little bit
    butFrame <- tkframe(win.fr)
    back.but <- tkbutton(butFrame, width = 8, text ="Back", command = function(){
      tkdestroy(tt3)
      hierarchy[[2]]()})
    tkconfigure(back.but, background = "white")
    finish.but <- tkbutton(butFrame, text = "Continue", command = function(){

      for(i in 1:get("NumSamples",eerieEnv)){
        for(j in 1:get("NumCovar",eerieEnv)){
          if(tclvalue(get("cov.lst",eerieEnv)[[i]][[j]])==""){
            assign("error", "Missing Entry",eerieEnv)
          }
        }
      }

      if(get("error",eerieEnv)=="Missing Entry"){
        assign("error","just for a change!!",eerieEnv)
        error.fr <-tkframe(tt3, borderwidth = 5, width = "5i", height = "2i")
        tkconfigure(error.fr, background = "yellow")
        tkpack(error.fr,side="top", fill="both")
        error.lbl <- tklabel(error.fr, text = "Error! Missing Entry in Phenodata")
        tkconfigure(error.lbl, background = "yellow")
        tkpack(error.lbl)
        error.but <- tkbutton(error.fr,text="OK",command = function(){tkdestroy(error.fr)})
        tkconfigure(error.but, background = "yellow")
        tkpack(error.but)
      }

      else{
        tkdestroy(tt3)
        tkdestroy(win.fr)
      }

    })

    tkconfigure(finish.but, background = "white")
    tkpack(back.but, side = "left", expand = FALSE)
    tkpack(finish.but, side = "left", expand = FALSE)

#    tkgrid(back.but, row = (get("NumSamples",eerieEnv) + 7),column = 2)
#    tkgrid(finish.but, row = (get("NumSamples",eerieEnv) + 7), column = 3)
#    tkgrid.configure(back.but, sticky = "e")
#    tkgrid.configure(finish.but, sticky = "w")
    tkgrid(butFrame, columnspan = j + 4)


    tkwait.window(tt3)

    pd.matrix <- matrix(NA,nrow=length(sampleNames),ncol=get("NumCovar",eerieEnv))
    rownames(pd.matrix) <- sampleNames

    c.names <- array("",get("NumCovar",eerieEnv))
    for(j in 1:get("NumCovar",eerieEnv)){
      c.names[j] <- tclvalue(get("cov.name.lst",eerieEnv)[[j]])}
    colnames(pd.matrix) <- c.names
    for(i in 1:length(sampleNames)){
      for(j in 1:get("NumCovar",eerieEnv)){
        pd.matrix[i,j] <- tclvalue(get("cov.lst",eerieEnv)[[i]][[j]])}}

    desc.matrix <- matrix("",nrow=get("NumCovar",eerieEnv),ncol=1)
    rownames(desc.matrix) <- c.names
    colnames(desc.matrix) <- "Description"

    for(i in 1:get("NumCovar",eerieEnv)){
      desc.matrix[i,1] <- tclvalue(get("desc.lst",eerieEnv)[[i]])}
    pd.info <- list(pd.matrix,desc.matrix)
    names(pd.info) <- c("pData", "varLabels")

    assign("pd.info",pd.info,eerieEnv)
    if(tclvalue(pdName) != ""){
        assign(tclvalue(pdName), pd.info, .GlobalEnv)
    }
}





##############################################
### second widget
##############################################

  hierarchy[[2]] <- function(){

    assign("backCount1",0,eerieEnv)

    if(!is.na(match("NewNumCovariates.var",ls(eerieEnv)))){
      assign("OldNumCovariates.var",get("NewNumCovariates.var",eerieEnv),eerieEnv)
    }



    tt2 <- tktoplevel(width = 70)
    tktitle(tt2) <- "Covariate Names"
    tkconfigure(tt2, background = "white")

    tbar.fr <- tkframe(tt2,relief="sunken",borderwidth=5,width="5i")
    win.fr <- tkframe(tt2,relief="sunken",borderwidth=5,width="5i",height="7i")
    tkconfigure(win.fr, background = "white")
    tkpack(tbar.fr,side="top",fill="x")
    tkpack(win.fr,side="top", fill="both")

    assign("NumCovariates", as.numeric(tclvalue(get("NumCovariates.var",eerieEnv))),eerieEnv)
    assign("OldNumCovariates",as.numeric(tclvalue(get("OldNumCovariates.var",eerieEnv))),eerieEnv)

    if(!is.na(match("OldNumCovariates1.var",ls(eerieEnv)))){
      assign("OldNumCovariates1.var", get("OldNumCovariates1.var",eerieEnv),eerieEnv)}
    else{
      assign("OldNumCovariates1.var", tclVar(0),eerieEnv)
    }
    assign("cov.name.lst",list(),eerieEnv)
    if(get("OldNumCovariates",eerieEnv) != get("NumCovariates",eerieEnv)){
      for(i in 1:get("NumCovariates",eerieEnv)){
        assign("cov.name", tclVar(""),eerieEnv)
        assign("cov.name.lst",c(get("cov.name.lst",eerieEnv),list(get("cov.name",eerieEnv))),eerieEnv)
      }}
    else{
      if(!is.na(match("cov.name.lst",ls(eerieEnv)))){
        assign("cov.name.lst", cov.name.label,eerieEnv)}
      else{
        for(i in 1:get("NumCovariates",eerieEnv)){
          assign("cov.name", tclVar(""),eerieEnv)
          assign("cov.name.lst",c(get("cov.name.lst",eerieEnv),list(get("cov.name",eerieEnv))),eerieEnv)
        }  }}

    cov.entry.lst <- list()

    assign("desc.lst",list(),eerieEnv)
    if(get("OldNumCovariates",eerieEnv) != get("NumCovariates",eerieEnv)){
      for(i in 1:get("NumCovariates",eerieEnv)){
        assign("descvar",tclVar(""),eerieEnv)
        assign("desc.lst",c(get("desc.lst",eerieEnv),list(get("descvar",eerieEnv))),eerieEnv)}
    }
    else{
      if(!is.na(match("desc.lst",ls(eerieEnv)))){
        assign("desc.lst",desc.lst,eerieEnv)}
      else{
        for(i in 1:get("NumCovariates",eerieEnv)){
          assign("descvar",tclVar(""),eerieEnv)
          assign("desc.lst",c(get("desc.lst",eerieEnv),list(get("descvar",eerieEnv))),eerieEnv)}
      }}

    desc.entry <- list()

    space.label1 <- tklabel(win.fr, text = "")
    tkconfigure(space.label1, background = "white")
    space.label2 <- tklabel(win.fr, text = "")
    tkconfigure(space.label2, background = "white")
    space.label3 <- tklabel(win.fr, text = "")
    tkconfigure(space.label3, background = "white")


    tkgrid(space.label1, row = 1)

    for(i in 1:get("NumCovariates",eerieEnv)){

      cov.label <-tklabel(win.fr, text=paste("Cov",i))
      tkconfigure(cov.label, background = "white")
      tkgrid(cov.label, row = (i+2), column = 1)

      cov.entry.lst[[i]] <- tkentry(win.fr, width = 30, textvariable = get("cov.name.lst",eerieEnv)[[i]])
      tkconfigure(cov.entry.lst[[i]], background = "white")
      tkgrid(cov.entry.lst[[i]], row = (i+2), column = 2)

      covtop.label <-tklabel(win.fr, text="Covariate Names")
      tkconfigure(covtop.label, background = "white")
      tkgrid(covtop.label, row=2, column = 2)

      desc.label <- tklabel(win.fr, text = "Description")
      tkconfigure(desc.label, background = "white")
      tkgrid(desc.label, row=2, column = 3)

      desc.entry[[i]] <- tkentry(win.fr, width = 30, textvariable = get("desc.lst",eerieEnv)[[i]])
      tkconfigure(desc.entry[[i]], background = "white")
      tkgrid(desc.entry[[i]], row = (i+2), column = 3)
    }


    but1 <- tkbutton(win.fr, width = 8,text = "Back", command = function(){
      assign("backCount1",1,eerieEnv)
      tkdestroy(tt2)
      hierarchy[[1]]()})
    tkconfigure(but1, background = "white")
    but2 <- tkbutton(win.fr,text = "Continue",command= function(){

      for(i in 1:get("NumCovariates",eerieEnv)){
        if(tclvalue(get("cov.name.lst",eerieEnv)[[i]]) == ""){
          assign("error","Missing Entry",eerieEnv)}
      }

      if(get("error",eerieEnv)=="Missing Entry"){
        assign("error", "just for a change1!!",eerieEnv)
        error.fr <-tkframe(tt2, borderwidth = 5, width = "5i", height = "2i")
        tkconfigure(error.fr, background = "yellow")
        tkpack(error.fr,side="top", fill="both")
        error.lbl <- tklabel(error.fr, text = "Error! Missing Entry in Covariate Names")
        tkconfigure(error.lbl, background = "yellow")
        tkpack(error.lbl)
        error.but <- tkbutton(error.fr,text="OK",command = function(){tkdestroy(error.fr)})
        tkconfigure(error.but, background = "yellow")
        tkpack(error.but)
      }else{
        assign("NewNumCovariates.var",get("OldNumCovariates.var",eerieEnv),eerieEnv)
        assign("OldNumCovariates.var", get("NumCovariates.var",eerieEnv),eerieEnv)
        tkdestroy(tt2)
        hierarchy[[3]]()
      }})

    tkconfigure(but2, background = "white")

    tkgrid(space.label2,row = (get("NumCovariates",eerieEnv) + 5))
    tkgrid(space.label3,row = (get("NumCovariates",eerieEnv) + 6))
    tkgrid(but1,row = (get("NumCovariates",eerieEnv) + 7),column = 2)
    tkgrid(but2,row = (get("NumCovariates",eerieEnv) + 7), column = 3)
    tkgrid.configure(but1,sticky="e")
    tkgrid.configure(but2,sticky="w")


    tkwait.window(tt2)

    assign("OldNumCovariates1", as.numeric(tclvalue(get("OldNumCovariates1.var",eerieEnv))),eerieEnv)
  }

###############################################
###first widget
###############################################


  hierarchy[[1]] <- function(){


    tt1 <- tktoplevel(width = "5i")
    tkconfigure(tt1, background = "white")
    tktitle(tt1) <- "Number of Covariates"

    tbar.fr <- tkframe(tt1,relief="sunken",borderwidth=5,width="5i")
    win.fr <- tkframe(tt1,relief="sunken",borderwidth=5,width="5i",height="7i")
    tkconfigure(win.fr, background = "white")
    tkpack(tbar.fr,side="top",fill="x")
    tkpack(win.fr,side="top", fill="both")

    NumCovariates.label <- tklabel(win.fr, text="Enter the Number Of Covariates")
    tkconfigure(NumCovariates.label, background = "white")

    ##taking care of the back button( ie. once some value is assigned it remains unless changed by th user)
    if(!is.na(match("OldNumCovariates.var",ls(eerieEnv)))){
      assign("OldNumCovariates.var",get("OldNumCovariates.var",eerieEnv),eerieEnv)}
    else{
      assign("OldNumCovariates.var", tclVar(0),eerieEnv)
    }
    if(!is.na(match("NumCovariates.var",ls(eerieEnv)))){
      assign(" NumCovariates.var",get("NumCovariates.var",eerieEnv),eerieEnv)}
    else{
      assign("NumCovariates.var", tclVar(),eerieEnv)
    }

    NumCovariates.entry <- tkentry(win.fr, width = 6,textvariable = get("NumCovariates.var",eerieEnv))
    tkconfigure(NumCovariates.entry, background = "white")

    but1 <- tkbutton(win.fr,text = "Continue",command=function(){
      assign("error", "none", eerieEnv)
      if(tclvalue(get("NumCovariates.var",eerieEnv)) < 1){
        assign("error","Error!! Number of covariates should be atleast one",eerieEnv)
        error.fr <-tkframe(tt1, borderwidth = 5, width = "5i", height = "2i")
        tkconfigure(error.fr, background = "yellow")
        tkpack(error.fr,side="top", fill="both")
        error.lbl <- tklabel(error.fr, text = get("error",eerieEnv))
        tkconfigure(error.lbl, background = "yellow")
  tkpack(error.lbl)
        error.but <- tkbutton(error.fr,text="OK",command = function(){tkdestroy(error.fr)})
        tkconfigure(error.but, background = "yellow")
        tkpack(error.but)
      }
      else {
        tkdestroy(win.fr)
        tkdestroy(tt1)
      }})

    tkconfigure(but1, background = "white")

    space.label1 <- tklabel(win.fr, text = "")
    tkconfigure(space.label1, background = "white")
    space.label2 <- tklabel(win.fr, text = "")
    tkconfigure(space.label2, background = "white")
    space.label3 <- tklabel(win.fr, text = "")
    tkconfigure(space.label3, background = "white")
    space.label4 <- tklabel(win.fr, text = "")
    tkconfigure(space.label4, background = "white")
    space.label5 <- tklabel(win.fr, text = "")
    tkconfigure(space.label5, background = "white")


    tkgrid(space.label1, row =1)
    tkgrid(space.label2, row =2)
    tkgrid(NumCovariates.label, row = 3, column = 1)
    tkgrid(NumCovariates.entry, row = 3, column =2)
    tkgrid(space.label3, row =4)
    tkgrid(space.label4, row =5)
    tkgrid(but1, row = 6, column = 1)
    tkgrid(space.label5, row =7)

    tkwait.window(tt1)
    hierarchy[[2]]()



  }


  hierarchy[[1]]()



  return(get("pd.info",eerieEnv))
}







# This function lists all the R packages in the library that have a
# Bioconductor vignettes and waits for the user to selecte one to
# explore.
#
# Copyrihgt, 2002, J. Zhang, all rights reserved
#

vExplorer <- function (title = "BioC Vignettes Explorer",
                       pkgName ="", font = ifelse(.Platform$OS.type
                                    == "unix", "arial 14", "arial 11")){

#    require(Biobase) || stop("Package Biobase not available!")
    require(tools) || stop("Package tools not available!")
#    require(DynDoc) || stop("Package DynDoc not available!")

    on.exit(tkdestroy(base))
    PLATFORM <- .Platform$OS.type
    selectedPkg <- NULL
#    selectedVig <- NULL
    vigList <- NULL

    end <- function(){
        tkdestroy(base)
    }

    # Executes when a user clicks a package name in the list of pkg names
    packSelected <- function(){
         selectedPkg <<-
             tclvalue(tkget(packViewer,(tkcurselection(packViewer))))

         write2VigList(selectedPkg)
    }
    # Writes vignette names to the list box for vignettes
    write2VigList <- function(selectedPkg){
         checkMe <- .getPackNames(selectedPkg)
         if(is.null(checkMe)){
             tkmessageBox(title = "No vignette found",
                     message = paste("Package", selectedPkg,
                     "has no vignette"))
         }else{
             vigList <<- vignette(selectedPkg)
             tkdelete(vigViewer, 0, "end")
             for(i in names(vigList)){
                 if(!inherits(chunkList, "try-error")){
                     tkinsert(vigViewer, "end", i)
                 }
             }
#             tkconfigure(vButton, state = "disabled")
         }
    }

    # Executes when a user clicks a vignette name in the list of Rnw files
    vigSelected <- function(){
        selectedVig <-
             tclvalue(tkget(vigViewer,(tkcurselection(vigViewer))))
        writeList(statusBar, paste("Loading", selectedVig), TRUE)
        # Make sure widget gets redrawn
        Sys.sleep(1)
        viewVignette(title, selectedPkg, vigList[[selectedVig]]$VigPath,
                     vigList[[selectedVig]]$PDFpath, font)
        writeList(statusBar, "", TRUE)
#        tkconfigure(vButton, state = "normal")
    }

    # Executes when a user clicks the view button after selecting a
    # vignette
#    viewVig <- function(){
#        viewVignette(title, selectedPkg, vigList[[selectedVig]]$VigPath,
#                     vigList[[selectedVig]]$PDFpath, font)
#    }


    base <- tktoplevel()
    tktitle(base) <- title

    listFrame <- tkframe(base)

    # List box showing all the packages that have a vignette
    packFrame <- tkframe(listFrame)
    packName <- tklabel(packFrame, text = "Package List", font = font)
    tkpack(packName, side = "top")
    packViewer <- makeViewer(packFrame, hScroll = TRUE, vScroll = TRUE)
    tkconfigure(packViewer, selectmode = "browse")
    tkconfigure(packViewer, font = font)
    tkconfigure(packViewer, exportselection = FALSE)
    tkbind(packViewer, "<B1-ButtonRelease>", packSelected)
    tkpack(packFrame, side = "left", expand = TRUE, fill = "both")

    # List box for vignettes of a given package
    vigFrame <- tkframe(listFrame)
    vigName <- tklabel(vigFrame, text = "Vignette List", font = font)
    tkpack(vigName, side = "top")
    vigViewer <- makeViewer(vigFrame, hScroll = TRUE, vScroll = TRUE)
    tkconfigure(vigViewer, selectmode = "browse")
    tkconfigure(vigViewer, font = font)
    tkconfigure(vigViewer, exportselection = FALSE)
    tkbind(vigViewer, "<B1-ButtonRelease>", vigSelected)
    tkpack(vigFrame, side = "left", expand = TRUE, fill = "both")

    # Buttons to view a vignette and end the widget
#    vButton <- tkbutton(listFrame, text = "View", width = 8,
#                        font = font, command = viewVig)
#    tkpack(vButton, side = "left")

    # Put the three frames in
    tkpack(listFrame, expand = TRUE, fill = "both", pady = 4, padx = 4)
    # put in the end button
    endButton <- tkbutton(base, text = "End", width = 8,
                          font = font, command = end)
    tkpack(endButton, pady = 4)

    # Put a status
    tkpack(statusBar <- tkentry(base, foreground = "Red"),
           expand = FALSE, side = "bottom", fill = "x")

    # Populates the list box for package names
    .popPackList(packViewer, pkgName)
    if(pkgName != ""){
         write2VigList(pkgName)
     }

#    tkconfigure(vButton, state = "disabled")

    tkwait.window(base)
}

# Check for the availability of vignettes and populate the list box
# for packages that have a vignette
.popPackList <- function(packViewer, packName){
    if(packName == ""){
        packs <- sort(.packages(all = TRUE))
    }else{
        packs <- packName
    }
    tkdelete(packViewer, 0, "end")
    for(i in packs){
        if(!is.null(pkgVignettes(i)) &&
           length(pkgVignettes(i)$docs) > 0){
            tkinsert(packViewer, "end", i)
        }
    }
}

# Returns package names that have a vignette
.getPackNames <- function(packName = ""){
    if(packName == ""){
        packNames <- .packages(all = TRUE)
    }else{
        packNames <- packName
    }
    goodNames <- NULL
    for(i in packNames){
        eval(call(deparse(substitute(require)), i, TRUE))
        path <- .path.package(i)
        options(warn = -1)
        tryMe <- getPkgVigList(path)
        options(warn = 1)
        if(!is.null(tryMe))
        goodNames <- c(goodNames, i)
    }
    return(goodNames)
}

# This window is called by vExplorer for interacting with the code
# chunks of a vignette
viewVignette <- function(title, packName, vigPath, pdfPath,
                         font = "arial 11"){

    on.exit(end)

    selectedChunk <- NULL
    buts <- vector("list")
    codeVersion <- NULL
    newCode <- FALSE
    executed <- NULL
    chunkOrNot <- "Code Chunk"

    chunkList <- getVignetteCode(vigPath)

    if(is.null(chunkList)){
        chunkOrNot <- "No Code Chunk"
        nameNCode <- NULL
#        tkmessageBox(title = "No code chunk found",
#                     message = paste(gsub(".*/(.*)", "\\1", vigPath),
#                     "does not contain any code chunk"))
#        return()
    }else{
        nameNCode <- .getNameNCode(chunkList)
#        codeVersion[[1]] <- chunkList
    }

    end <- function(){
        tkdestroy(base)
    }
    # Not implemented now but will be later
    back <- function(){
        if(length(codeVersion) > 1){
            codeVersion <<- codeVersion[[-length(codeVersion)]]
        }
        if(length(codeVersion) == 1){
            tkconfigure(backButton, state = "disabled")
        }
        tempCode <- chunk(getChunk(codeVersion[[length(codeVersion)]]))
        tkdelete(editViewer, "1.0", "end")
        for(i in tempCode){
            tkinsert(editViewer, "end", paste(i, "\n", sep = ""))
        }
        tkdelete(resultViewer, 0, "end")
    }

    # Executed when a user clicks the view PDF button
    viewPDF <- function(){
        openPDF(pdfPath)
    }
    # Shows the code chunk in a text box that allows the user to
    # editor the code chunk in the box but not to the actual code chunk
    showCode <- function(chunkName){
        tkdelete(editViewer, "1.0", "end")
        tkconfigure(resultViewer, state = "normal")
        tkdelete(resultViewer, "1.0", "end")
        tkconfigure(resultViewer, state = "disabled")
        for(i in nameNCode[[chunkName]]){
            tkinsert(editViewer, "end", paste(i, "\n", sep = ""))
        }
        tkconfigure(buts[[chunkName]], state = "active")
        if(chunkName != 1){
            tkconfigure(buts[[chunkName - 1]], state = "normal")
        }
        tkconfigure(execButton, state = "normal")
        tkconfigure(clearButton, state = "normal")
        tkconfigure(expoButton, state = "normal")
    }
    # Export code chunk to the R session
    export <- function(){
        temp <- objectBrowser(evalEnv(chunkList))
        for(i in names(temp)){
            assign(i, temp[[i]], env = parent.frame(2))
        }
    }
    # Executes whatever that is in the text box for code chunk
    execute <- function(){
#        if(selectedChunk == 1 || any(executed == (selectedChunk - 1))){
            if(newCode){
                tempCode <- tclvalue(tkget(editViewer, "1.0", "end"))
                tempChunk <-
                    editVignetteCode(chunkList,
                                     selectedChunk,tempCode)
                result <- evalChunk(tempChunk, selectedChunk)
                newCode <<- FALSE
#                tkconfigure(backButton, state = "normal")
            }else{
                result <- evalChunk(chunkList, selectedChunk)
#                modButton(buts[[selectedChunk]], "libhtblue")
                tkconfigure(buts[[selectedChunk]], relief = "sunken",
                            state = "active")
                if(selectedChunk < length(buts)){
                    tkconfigure(buts[[selectedChunk + 1]], state = "normal")
                }

            }
            tkdelete(resultViewer, "", "end")
            tkconfigure(resultViewer, state = "normal")
            tkinsert(resultViewer, "end", result)
            tkconfigure(resultViewer, state = "disabled")
#            modButton(buts[[selectedChunk]], "blue")
#            executed <<- unique(c(executed, selectedChunk))
#        }else{
#             tkmessageBox(title = "Execution failed",
#                     message = "Code chunks need to be executed in order")
#        }
    }

    # keeps track of code modification done
    codeChanged <- function(){
        newCode <<- TRUE
    }

    # Cleans the boxes for code chunk and result of execution
    clear <- function(){
        tkdelete(editViewer, "1.0", "end")
        tkconfigure(resultViewer, state = "normal")
        tkdelete(resultViewer, "1.0", "end")
        tkconfigure(resultViewer, state = "disabled")
        for(i in 1:length(buts)){
            tkconfigure(buts[[i]], state = "disabled", relief = "raised")
#            modButton(buts[[i]], "green")
        }
        tkconfigure(buts[[1]], state = "normal")
        tkconfigure(execButton, state = "disabled")
        tkconfigure(expoButton, state = "disabled")
        tkconfigure(clearButton, state = "disabled")
        executed <<- NULL
        codeVersion[[1]] <<- chunkList
#        tkconfigure(backButton, state = "disabled")
    }

    # Changes the background colour of a widget
    modButton <- function(but, what){
        tkconfigure(but, background = what)
    }

    # Initilizes the buttons for code chunks
    popChunks <- function(){
        if(!is.null(nameNCode)){
            k <- 1
            for(i in names(nameNCode)){
                # Create button functions
                tempBut <- substitute(buts[[j]], list(j = k))

                fun <- function() {}
                body <- list(as.name("{"),
                             substitute(showCode(j),
                                        list(j = k)),
                             substitute(selectedChunk <<- j,
                                        list(j = k)))
                body(fun) <- as.call(body)
                assign(paste("chunkList",k,sep=""), fun)

                buts[[k]] <<- tkbutton(base, text= i, width = 13,
                                       state = "disabled", font = font,
                                       command = get(paste("chunkList",
                                       k, sep = "")))
                tkbind(buts[[k]], "<Double-Button-1>", execute)
                tkwindow.create(chunkText, "end", window = buts[[k]])
#                tkcreate(chunkText, "window", 5, space, anchor = "nw",
#                 window = buts[[k]])
                tkinsert(chunkText, "end", "\n")
                k <- k + 1
            }
            tkconfigure(buts[[1]], state = "normal")
        }
    }

    base <- tktoplevel()
    tktitle(base) <- title
    # Write package and vignette names
    pNvNames <- paste("Package:", packName, "   Vignette:",
                      gsub(paste(".*", .Platform$file.sep, "(.*)",
                                 sep = ""), "\\1", vigPath))
    tkpack(tklabel(base, text = pNvNames, font = font), pady = 4)

    listFrame <- tkframe(base)
    # Create a text widgets for code chunks
    chunkFrame <- tkframe(listFrame)
    tkpack(tklabel(chunkFrame, text = chunkOrNot, font = font))
    chunkText <- makeViewer(chunkFrame, vWidth = 18, vHeight = NULL,
                      hScroll = TRUE, vScroll = TRUE, what = "text")
    tkconfigure(chunkText)
    popChunks()
    tkpack(chunkFrame, side = "left", anchor = "nw", expand = FALSE,
           fill = "y")

    # Create the viewers for code and results of execution
    codeNRelFrame <- tkframe(listFrame)
    editFrame <- tkframe(codeNRelFrame)
    tkpack(tklabel(editFrame, text = "R Source Code", font = font))
    eViewerFrame <- tkframe(editFrame)
    editViewer <- makeViewer(eViewerFrame, vWidth = 40, vHeight = 4,
                      hScroll = TRUE, vScroll = TRUE, what = "text")
    tkconfigure(editViewer, font = font)
    tkbind(editViewer, "<KeyRelease>", codeChanged)
    tkpack(eViewerFrame, expand = TRUE, fill = "both")
    tkpack(tklabel(editFrame, text = "Results of Execution", font = font))
    rViewerFrame <- tkframe(editFrame)
    resultViewer <-  makeViewer(rViewerFrame, vWidth = 40, vHeight = 4,
                      hScroll = TRUE, vScroll = TRUE, what = "text")
    tkconfigure(resultViewer, font = font)
    tkpack(rViewerFrame, expand = TRUE, fill = "both")
    tkpack(editFrame, expand = TRUE, fill = "both")
    tkpack(codeNRelFrame, side = "left", expand = TRUE, fill = "both")

    tkpack(listFrame, side = "top", expand = TRUE, fill = "both", padx
    = 4, pady = 6)

    # Put the buttons in
    butFrame <- tkframe(base)
    pdfButton <- tkbutton(butFrame, text = "View PDF", width = 12,
                          font = font, command = viewPDF)
    execButton <- tkbutton(butFrame, text = "Execute Code", width = 12,
                           font = font, command = execute)
    tkconfigure(execButton, state = "disabled")
    expoButton <- tkbutton(butFrame, text = "Export to R", width = 12,
                           font = font, command = export)
    tkconfigure(expoButton, state = "disabled")
    clearButton <- tkbutton(butFrame, text = "Clear", width = 12,
                            font = font, command = clear)
    tkconfigure(clearButton, state = "disabled")
    tkpack(pdfButton, execButton, expoButton, clearButton, side = "left")
    tkpack(butFrame, pady = 6)
    # Put end button separately to avoid accidents
    endButton <- tkbutton(base, text = "End", width = 12,
                          font = font, command = end)
    tkpack(endButton)

    if(is.null(pdfPath) || is.na(pdfPath) || length(pdfPath) == 0){
        tkconfigure(pdfButton, state = "disabled")
    }
#    tkconfigure(resultViewer, state = "disabled")

#    tkwait.window(base)
}

.getNameNCode <- function(chunkList){
    chunkNum <- numChunks(chunkList)
    nameNCode <- list()
    for(i in 1:chunkNum){
        name <- chunkName(getChunk(chunkList, i))
        if(length(name) == 0){
            name <- paste("Code chunk", i)
        }
        nameNCode[[name]] <- chunk(getChunk(chunkList, i))
    }
    return(nameNCode)
}
# This function prints the values for all the value containing widgets
# elements on a widget created using the function widgetRender.
#
# Copyright J. Zhang 2002, all rights reserved.
#
values.Widget <- function (x){

    wList <- WwList(x)
    returnList <- vector("list", length = length(names(wList)))
    counter <- 1
    for(i in names(wList)) {
        pW <- wList[[i]]
        returnList[[counter]] <- list(Entry = i, Value = WValue(pW))
        counter <- counter + 1
    }
    return(returnList)
}
# This function creates a widget based on the requirements passed as a
# list of lists.
# iWidget: a list of lists defining the appearance and behavour of the
# primitive widgets such entry boxex, buttons, ...
# tkTitle: text to be appear in the title bar of the widgets.
#
widgetRender <- function (iWidget, tkTitle)
{
    LABELFONT <- "Helvetica 12"
    ENTRYWIDTH <- 50
    BUTWIDTH <- 10
    BUTTONLAST <- NULL

    savediWidget <- iWidget
    wList <- WwList(iWidget)

    if(is.null(wList) || is.na(wList) )
        stop("Either wList or/and funName is null or empty")

    if(!is.null(WpreFun(iWidget)))
        tt <- eval(WpreFun(iWidget)())
    if(!is.null(WpostFun(iWidget)))
        on.exit(eval(WpostFun(iWidget)()))

    PFRAME <- parent.frame(1)
    CANCEL <- FALSE
    END <- FALSE
    cancel <- function() {
        CANCEL <<- TRUE
        tkdestroy(base)
    }
    end <- function() {
        END <<- TRUE
        tkdestroy(base)
    }

    base <- tktoplevel()
    tktitle(base) <- tkTitle
    eFrame <- tkframe(base)

    ## function that gets called at the end and updates the
    ## values of iWidget using data from the entry boxes
    getEntryValues <- function(){
        for(i in 1:length(wList) ) {
            if(!BUTTONLAST[[i]]){
                if(!is.null(WfromText(wList[[i]]))){
#                     eval(substitute(WLValue(iWidget, i) <<-
#                                     WfromText(wList[[i]])
#                                     (tclvalue(entryValue[[i]])),
#                                     list(i = i)))
                     WLValue(iWidget, i) <<- WfromText(wList[[i]])
                                     (tclvalue(entryValue[[i]]))
                }else{
                     WLValue(iWidget, i) <<- tclvalue(entryValue[[i]])
                }
            }
        }
    }

    ##build button functions
    for(i in 1:length(wList)) {
        fun <- function() {}
        body <- list(as.name("{"),
                     substitute(rval <-
                                eval(as.call(list(WbuttonFun(wList[[j]]))),
                                             env=PFRAME),
                                     list(j=i)),
                     substitute(mytext <- WtoText(wList[[j]])(rval),
                                list(j=i)),
                     substitute(if(!WcanEdit(wList[[j]]))
                                tkconfigure(entryList[[j]],state = "normal"),
                                list(j=i)),
                     substitute(tkdelete(entryList[[j]], 0, "end"),
                                list(j=i)),
                     substitute(tkinsert(entryList[[j]], 0, mytext),
                                list(j=i)),
                     substitute(if(!WcanEdit(wList[[j]]))
                                tkconfigure(entryList[[j]],
                                            state = "disabled"),
                                list(j=i)),

                     substitute(BUTTONLAST[[j]] <<- TRUE, list(j=i)),

                     substitute(WLValue(iWidget, j) <<- rval,
                                list(j = i))
                     )

        body(fun) <- as.call(body)
        assign(paste("funList",i,sep=""), fun)
    }

    # set the functions for user defined control buttons
    butFuns <- WRButtons(iWidget)
    for(i in 1:length(butFuns)) {
        fun <- function() {}
        body <- list(as.name("{"),
                     substitute(rval <-
                                eval(as.call(list(WbuttonFun(butFuns[[j]]))),
                                             env=PFRAME),
                                     list(j=i)),
                     )

        body(fun) <- as.call(body)
        assign(paste("userFunList",i,sep=""), fun)
    }

    ##initialize the buttons/boxes
    entryList <- vector("list", length = length(wList))
    entryValue <- vector("list", length = length(wList))
    for(i in 1:length(wList) ) {
        pW <- wList[[i]]
        label <- tklabel(eFrame, text = WName(pW), font = LABELFONT)
        entryValue[[i]] <- tclVar()
        BUTTONLAST[[i]] <- TRUE

        entryList[[i]] <- tkentry(eFrame,
                                  textvariable = entryValue[[i]],
                                  width=ENTRYWIDTH)

        eval(substitute(tkbind(entryList[[i]], "<KeyPress>",
               kpress <- function() BUTTONLAST[[i]] <<- FALSE), list(i = i)))

        if( !is.null(WValue(pW))){
            tkinsert(entryList[[i]], 0, WValue(pW))
        }

        if(!WcanEdit(wList[[i]]))
           tkconfigure(entryList[[i]], state = "disabled")

        if(is.null(WbuttonFun(pW)))
            browse <- tklabel(eFrame, text = "  ")
        else
            browse <- tkbutton(eFrame, text= WbuttonText(pW),
                               width=BUTWIDTH,
                               command=get(paste("funList",i,sep="")))
        tkgrid(label, entryList[[i]], browse)
    }

    tkpack(eFrame)

    # make a list with all user defined and default buttons
    butFuns[["cancel"]] <-  list(buttonText = "Cancel",
                             buttonFun = cancel)
    butFuns[["end"]] <- list(buttonText = "Finish", buttonFun = end)

    # Put user defined buttons in
    butFrame <- tkframe(base)

    for(i in 1:length(butFuns)){
        aBut <- butFuns[[i]]
        button <- tkbutton(butFrame, text = WbuttonText(aBut),
                           width = BUTWIDTH,
                               command = WbuttonFun(aBut))
        tkpack(button, side = "left")
    }
    tkpack(butFrame)

    tkwait.window(base)

    if(END){
        getEntryValues()
        iWidget$end <- "END"
        class(iWidget) <- "Widget"
        return(iWidget)
    }else{
        savediWidget$end <- "CANCEL"
        class(savediWidget) <- "Widget"
        return(savediWidget)
    }
}















