.packageName <- "marrayClasses"
##########################################################################
# maClasses.R
#
# Basic class definitions for two-color cDNA microarrays
#
###########################################################################

###########################################################################
# marrayInfo
# Class for describing either samples hybed to array, spots, or other objects
## Jean: Add prototype June-19,03

setClass("marrayInfo",
         representation(maLabels="character",
			maInfo="data.frame",
			maNotes="character"),
         prototype=list(maInfo=data.frame()))

##########################################################################
# marrayLayout
# Class for the microarray layout:  spot coords, grid coords, print-tip group,
# plate, controls (e.g. with expected log-ratio M=0), etc.

setClass("marrayLayout",
         representation(maNgr="numeric",
			maNgc="numeric",
			maNsr="numeric",
                        maNsc="numeric",
			maNspots="numeric",
			maSub="logical",
                        maPlate="factor",
			maControls="factor",
			maNotes="character"),
	prototype=list(maSub=TRUE, maPlate=factor(numeric(0)),
			maControls=factor(numeric(0))))

###########################################################################
# marraySpots
# Class describing spots hybridized to the array

setClass("marraySpots",
         representation(maGnames="marrayInfo", maLayout="marrayLayout"))

###########################################################################
# marrayTwo
# Class containing pairs of spot statistics

setClass("marrayTwo",
         representation(maX="matrix",
			maY="matrix",
			maLayout="marrayLayout",
			maTargets="marrayInfo"))

###########################################################################
# marrayRaw
# Class for pre-normalization intensity data and layout for several arrays

setClass("marrayRaw",
         representation(maRf="matrix",
 			maGf="matrix",
			maRb="matrix",
			maGb="matrix",
                        maW="matrix",
			maLayout="marrayLayout",
                        maGnames="marrayInfo",
			maTargets="marrayInfo",
			maNotes="character"))



###########################################################################
# marrayNorm
# Class used for normalization of intensity data (before and after)
# Use match.call() inside normalization function to return the call

setClass("marrayNorm",
         representation(maA="matrix",
			maM="matrix",
			maMloc="matrix",
			maMscale="matrix",
                        maW="matrix",
			maLayout="marrayLayout",
                        maGnames="marrayInfo",
			maTargets="marrayInfo",
			maNotes="character",
			maNormCall="call"))

###########################################################################
# Coerce object from class marrayRaw to marrayNorm

setAs("marrayRaw", "marrayNorm", function(from)
{
    mnorm<-new("marrayNorm", maA=maA(from), maM=maM(from), maW=maW(from),
	maLayout=maLayout(from), maGnames=maGnames(from),
	maTargets=maTargets(from), maNotes=maNotes(from))
     mnorm
})

#########################
# Coerce to exprSet

setAs("marrayRaw", "exprSet", function(from)
{
   eset<-new("exprSet", exprs=maM(from), se.exprs=maW(from), notes=maNotes(from) )

   targets<-maInfo(maTargets(from))
   pdata<-new("phenoData", pData=targets, varLabels=as.list(names(targets)))
   slot(eset, "phenoData")<-pdata
   eset
})

setAs("marrayNorm", "exprSet", function(from)
{
   eset<-new("exprSet", exprs=maM(from), se.exprs=maW(from), notes=maNotes(from) )

   targets<-maInfo(maTargets(from))
   pdata<-new("phenoData", pData=targets, varLabels=as.list(names(targets)))
   slot(eset, "phenoData")<-pdata
   eset
})

#########################
# Coerce to marraySpots

setAs("marrayRaw", "marraySpots", function(from)
	 new("marraySpots", maGnames=maGnames(from), maLayout=maLayout(from)))

setAs("marrayNorm", "marraySpots", function(from)
	 new("marraySpots", maGnames=maGnames(from), maLayout=maLayout(from)))

##########################################################################
# maGet.R
#
# Accessor methods for two-color cDNA microarrays classes
#
###########################################################################

###########################################################################
# Accessor methods for marrayInfo class

if(!isGeneric("maLabels"))
   setGeneric("maLabels", function(object)
   standardGeneric("maLabels"))

setMethod("maLabels", "marrayInfo", function(object) slot(object, "maLabels"))

if(!isGeneric("maInfo"))
   setGeneric("maInfo", function(object) standardGeneric("maInfo"))
setMethod("maInfo", "marrayInfo", function(object) slot(object, "maInfo"))

if(!isGeneric("maNotes"))
   setGeneric("maNotes", function(object) standardGeneric("maNotes"))
setMethod("maNotes", "marrayInfo", function(object) slot(object,
   "maNotes") )

##########################################################################
# Accessor methods for marrayLayout class

if(!isGeneric("maNspots"))
   setGeneric("maNspots", function(object)
   standardGeneric("maNspots"))

setMethod("maNspots", "marrayLayout",
	function(object) {
		n<-slot(object, "maNspots")
		if(length(n)==0)
		  n<-maNgr(object)*maNgc(object)*maNsr(object)*maNsc(object)
		n
	})

if(!isGeneric("maNgr"))
   setGeneric("maNgr", function(object) standardGeneric("maNgr"))
setMethod("maNgr", "marrayLayout", function(object) slot(object, "maNgr"))

if(!isGeneric("maNgc"))
   setGeneric("maNgc", function(object) standardGeneric("maNgc"))
setMethod("maNgc", "marrayLayout", function(object) slot(object, "maNgc"))

if(!isGeneric("maNsr"))
   setGeneric("maNsr", function(object) standardGeneric("maNsr"))
setMethod("maNsr", "marrayLayout", function(object) slot(object, "maNsr"))

if(!isGeneric("maNsc"))
   setGeneric("maNsc", function(object) standardGeneric("maNsc"))
setMethod("maNsc", "marrayLayout", function(object) slot(object, "maNsc"))

if(!isGeneric("maSub"))
   setGeneric("maSub", function(object) standardGeneric("maSub"))
setMethod("maSub", "marrayLayout", function(object) slot(object, "maSub"))

if(!isGeneric("maPlate"))
   setGeneric("maPlate", function(object) standardGeneric("maPlate"))
setMethod("maPlate", "marrayLayout", function(object)
          slot(object, "maPlate"))

if(!isGeneric("maControls"))
   setGeneric("maControls", function(object) standardGeneric("maControls"))
setMethod("maControls", "marrayLayout",
          function(object) slot(object, "maControls"))

if(!isGeneric("maNotes"))
   setGeneric("maNotes", function(object) standardGeneric("maNotes"))
setMethod("maNotes", "marrayLayout", function(object) slot(object, "maNotes"))

#########################
# Methods for quantities that are not slots of marrayLayout

if(!isGeneric("maGridRow"))
   setGeneric("maGridRow", function(object) standardGeneric("maGridRow"))
setMethod("maGridRow", "marrayLayout",
	function(object) {
		ngr<-maNgr(object)
		ngc<-maNgc(object)
		nsr<-maNsr(object)
		nsc<-maNsc(object)
                gr<-numeric(0)
 		if(length(ngr*ngc*nsr*nsc)!=0)
		{
		  gr <- rep(1:ngr, rep(nsr * nsc * ngc, ngr))
		  gr <- gr[as.logical(maSub(object))]
                }
		gr
	})

if(!isGeneric("maGridCol"))
   setGeneric("maGridCol", function(object) standardGeneric("maGridCol"))
setMethod("maGridCol", "marrayLayout",
	function(object) {
		ngr<-maNgr(object)
		ngc<-maNgc(object)
		nsr<-maNsr(object)
		nsc<-maNsc(object)
		gc<-numeric(0)
		if(length(ngr*ngc*nsr*nsc)!=0)
		{
		  gc <- rep(rep(1:ngc, rep(nsr * nsc, ngc)), ngr)
		  gc <- gc[as.logical(maSub(object))]
		}
		gc
   	})


if(!isGeneric("maSpotRow"))
   setGeneric("maSpotRow", function(object) standardGeneric("maSpotRow"))
setMethod("maSpotRow", "marrayLayout",
	function(object) {
		ngr<-maNgr(object)
		ngc<-maNgc(object)
		nsr<-maNsr(object)
		nsc<-maNsc(object)
		sr<-numeric(0)
		if(length(ngr*ngc*nsr*nsc)!=0)
		{
		  sr <- rep(rep(1:nsr, rep(nsc, nsr)), ngc * ngr)
		  sr <- sr[as.logical(maSub(object))]
		}
		sr
	})

if(!isGeneric("maSpotCol"))
   setGeneric("maSpotCol", function(object) standardGeneric("maSpotCol"))
setMethod("maSpotCol", "marrayLayout",
	function(object) {
		ngr<-maNgr(object)
		ngc<-maNgc(object)
		nsr<-maNsr(object)
		nsc<-maNsc(object)
		sc<-numeric(0)
		if(length(ngr*ngc*nsr*nsc)!=0)
		{
  		  sc <- rep(1:nsc, nsr * ngc * ngr)
		  sc <- sc[as.logical(maSub(object))]
		}
		sc
	})


if(!isGeneric("maPrintTip"))
   setGeneric("maPrintTip", function(object) standardGeneric("maPrintTip"))
setMethod("maPrintTip", "marrayLayout",
	function(object) {
		ngr<-maNgr(object)
		ngc<-maNgc(object)
		nsr<-maNsr(object)
		nsc<-maNsc(object)
		pt<-numeric(0)
		if(length(ngr*ngc*nsr*nsc)!=0)
		{
		  gr<- rep(1:ngr, rep(nsr * nsc * ngc, ngr))
		  gc <- rep(rep(1:ngc, rep(nsr * nsc, ngc)), ngr)
		  pt<-(gr-1) * ngc + gc
		  pt<-pt[as.logical(maSub(object))]
		}
		pt
	})

###########################################################################
# Accessor methods for marraySpots class

if(!isGeneric("maGnames"))
   setGeneric("maGnames", function(object) standardGeneric("maGnames"))
setMethod("maGnames", "marraySpots", function(object) slot(object, "maGnames"))


#########################
# maLayout slots

if(!isGeneric("maLayout"))
   setGeneric("maLayout", function(object) standardGeneric("maLayout"))
setMethod("maLayout", "marraySpots", function(object) slot(object, "maLayout"))

if(!isGeneric("maNspots"))
   setGeneric("maNspots", function(object) standardGeneric("maNspots"))
setMethod("maNspots", "marraySpots", function(object) maNspots(maLayout(object)))

if(!isGeneric("maNgr"))
   setGeneric("maNgr", function(object) standardGeneric("maNgr"))
setMethod("maNgr", "marraySpots", function(object) maNgr(maLayout(object)))

if(!isGeneric("maNgc"))
   setGeneric("maNgc", function(object) standardGeneric("maNgc"))
setMethod("maNgc", "marraySpots", function(object) maNgc(maLayout(object)))

if(!isGeneric("maNsr"))
   setGeneric("maNsr", function(object) standardGeneric("maNsr"))
setMethod("maNsr", "marraySpots", function(object) maNsr(maLayout(object)))

if(!isGeneric("maNsc"))
   setGeneric("maNsc", function(object) standardGeneric("maNsc"))
setMethod("maNsc", "marraySpots", function(object) maNsc(maLayout(object)))

if(!isGeneric("maGridRow"))
   setGeneric("maGridRow", function(object) standardGeneric("maGridRow"))
setMethod("maGridRow", "marraySpots", function(object) maGridRow(maLayout(object)))

if(!isGeneric("maGridCol"))
   setGeneric("maGridCol", function(object) standardGeneric("maGridCol"))
setMethod("maGridCol", "marraySpots", function(object) maGridCol(maLayout(object)))

if(!isGeneric("maSpotRow"))
   setGeneric("maSpotRow", function(object) standardGeneric("maSpotRow"))
setMethod("maSpotRow", "marraySpots", function(object) maSpotRow(maLayout(object)))

if(!isGeneric("maSpotCol"))
   setGeneric("maSpotCol", function(object) standardGeneric("maSpotCol"))
setMethod("maSpotCol", "marraySpots", function(object) maSpotCol(maLayout(object)))

if(!isGeneric("maPrintTip"))
   setGeneric("maPrintTip", function(object) standardGeneric("maPrintTip"))
setMethod("maPrintTip", "marraySpots", function(object) maPrintTip(maLayout(object)))

if(!isGeneric("maSub"))
   setGeneric("maSub", function(object) standardGeneric("maSub"))
setMethod("maSub", "marraySpots", function(object) maSub(maLayout(object)))

if(!isGeneric("maPlate"))
   setGeneric("maPlate", function(object) standardGeneric("maPlate"))
setMethod("maPlate", "marraySpots", function(object) maPlate(maLayout(object)))

if(!isGeneric("maControls"))
   setGeneric("maControls", function(object) standardGeneric("maControls"))
setMethod("maControls", "marraySpots", function(object) maControls(maLayout(object)))

###########################################################################
# Accessor methods for marrayTwo class

if(!isGeneric("maX"))
  setGeneric("maX", function(object) standardGeneric("maX"))
setMethod("maX", "marrayTwo", function(object) slot(object, "maX"))

if(!isGeneric("maY"))
  setGeneric("maY", function(object) standardGeneric("maY"))
setMethod("maY", "marrayTwo", function(object) slot(object, "maY"))

if(!isGeneric("maTargets"))
   setGeneric("maTargets", function(object) standardGeneric("maTargets"))
setMethod("maTargets", "marrayTwo", function(object) slot(object, "maTargets"))


if(!isGeneric("maNsamples"))
   setGeneric("maNsamples", function(object) standardGeneric("maNsamples"))
setMethod("maNsamples", "marrayTwo", function(object) ncol(maX(object)))


#########################
# maLayout slots

if(!isGeneric("maLayout"))
   setGeneric("maLayout", function(object) standardGeneric("maLayout"))
setMethod("maLayout", "marrayTwo", function(object) slot(object, "maLayout"))

if(!isGeneric("maNspots"))
   setGeneric("maNspots", function(object) standardGeneric("maNspots"))
setMethod("maNspots", "marrayTwo", function(object) maNspots(maLayout(object)))

if(!isGeneric("maNgr"))
   setGeneric("maNgr", function(object) standardGeneric("maNgr"))
setMethod("maNgr", "marrayTwo", function(object) maNgr(maLayout(object)))

if(!isGeneric("maNgc"))
   setGeneric("maNgc", function(object) standardGeneric("maNgc"))
setMethod("maNgc", "marrayTwo", function(object) maNgc(maLayout(object)))

if(!isGeneric("maNsr"))
   setGeneric("maNsr", function(object) standardGeneric("maNsr"))
setMethod("maNsr", "marrayTwo", function(object) maNsr(maLayout(object)))

if(!isGeneric("maNsc"))
   setGeneric("maNsc", function(object) standardGeneric("maNsc"))
setMethod("maNsc", "marrayTwo", function(object) maNsc(maLayout(object)))

if(!isGeneric("maGridRow"))
   setGeneric("maGridRow", function(object) standardGeneric("maGridRow"))
setMethod("maGridRow", "marrayTwo", function(object) maGridRow(maLayout(object)))

if(!isGeneric("maGridCol"))
   setGeneric("maGridCol", function(object) standardGeneric("maGridCol"))
setMethod("maGridCol", "marrayTwo", function(object) maGridCol(maLayout(object)))

if(!isGeneric("maSpotRow"))
   setGeneric("maSpotRow", function(object) standardGeneric("maSpotRow"))
setMethod("maSpotRow", "marrayTwo", function(object) maSpotRow(maLayout(object)))

if(!isGeneric("maSpotCol"))
   setGeneric("maSpotCol", function(object) standardGeneric("maSpotCol"))
setMethod("maSpotCol", "marrayTwo", function(object) maSpotCol(maLayout(object)))

if(!isGeneric("maPrintTip"))
   setGeneric("maPrintTip", function(object) standardGeneric("maPrintTip"))
setMethod("maPrintTip", "marrayTwo", function(object) maPrintTip(maLayout(object)))

if(!isGeneric("maSub"))
   setGeneric("maSub", function(object) standardGeneric("maSub"))
setMethod("maSub", "marrayTwo", function(object) maSub(maLayout(object)))

if(!isGeneric("maPlate"))
   setGeneric("maPlate", function(object) standardGeneric("maPlate"))
setMethod("maPlate", "marrayTwo", function(object) maPlate(maLayout(object)))

if(!isGeneric("maControls"))
   setGeneric("maControls", function(object) standardGeneric("maControls"))
setMethod("maControls", "marrayTwo", function(object) maControls(maLayout(object)))


###########################################################################
# Accessor methods for marrayRaw class

if(!isGeneric("maRf"))
   setGeneric("maRf", function(object) standardGeneric("maRf"))
setMethod("maRf", "marrayRaw", function(object) slot(object, "maRf"))

if(!isGeneric("maGf"))
   setGeneric("maGf", function(object) standardGeneric("maGf"))
setMethod("maGf", "marrayRaw", function(object) slot(object, "maGf"))

if(!isGeneric("maRb"))
   setGeneric("maRb", function(object) standardGeneric("maRb"))
setMethod("maRb", "marrayRaw", function(object) slot(object, "maRb"))

if(!isGeneric("maGb"))
   setGeneric("maGb", function(object) standardGeneric("maGb"))
setMethod("maGb", "marrayRaw", function(object) slot(object, "maGb"))

if(!isGeneric("maW"))
   setGeneric("maW", function(object) standardGeneric("maW"))
setMethod("maW", "marrayRaw", function(object) slot(object, "maW"))

#########################
# Methods for quantities that are not slots of marrayRaw

if(!isGeneric("maLR"))
   setGeneric("maLR", function(object) standardGeneric("maLR"))
setMethod("maLR", "marrayRaw",
          function(object)
          {
            r<-maRf(object)
            if(length(maRf(object))!=0)
	    {
              if(length(maRb(object))!=0)
                r<-maRf(object) - maRb(object)
              else
                r<-maRf(object)
              r<-log(ifelse(r>0, r, NA),2)
	    }
      	   r
          }
          )



if(!isGeneric("maLG"))
   setGeneric("maLG", function(object) standardGeneric("maLG"))
setMethod("maLG", "marrayRaw",
          function(object)
          {
            g<-maGf(object)
            if(length(maGf(object))!=0)
	    {
              if(length(maGb(object))!=0)
                g<-maGf(object) - maGb(object)
              else
                g<-maGf(object)
              g<-log(ifelse(g>0, g, NA),2)
	    }
      	   g
          }
          )


if(!isGeneric("maM"))
   setGeneric("maM", function(object) standardGeneric("maM"))
setMethod("maM", "marrayRaw",
	function(object)
	{
           M<-matrix(nr=0,nc=0)
	   if((length(maLR(object))!=0) & (length(maLG(object))!=0))
 	      M<-maLR(object)-maLG(object)
	   M
        }
	)


if(!isGeneric("maA"))
   setGeneric("maA", function(object) standardGeneric("maA"))
setMethod("maA", "marrayRaw",
	function(object)
	{
           A<-matrix(nr=0,nc=0)
           if((length(maLR(object))!=0) & (length(maLG(object))!=0))
             A<-(maLR(object)+maLG(object))/2
	   A
        }
	)


#########################
# maLayout slots

if(!isGeneric("maLayout"))
   setGeneric("maLayout", function(object) standardGeneric("maLayout"))
setMethod("maLayout", "marrayRaw", function(object) slot(object, "maLayout"))

if(!isGeneric("maNspots"))
   setGeneric("maNspots", function(object) standardGeneric("maNspots"))
setMethod("maNspots", "marrayRaw", function(object) maNspots(maLayout(object)))

if(!isGeneric("maNgr"))
   setGeneric("maNgr", function(object) standardGeneric("maNgr"))
setMethod("maNgr", "marrayRaw", function(object) maNgr(maLayout(object)))

if(!isGeneric("maNgc"))
   setGeneric("maNgc", function(object) standardGeneric("maNgc"))
setMethod("maNgc", "marrayRaw", function(object) maNgc(maLayout(object)))

if(!isGeneric("maNsr"))
   setGeneric("maNsr", function(object) standardGeneric("maNsr"))
setMethod("maNsr", "marrayRaw", function(object) maNsr(maLayout(object)))

if(!isGeneric("maNsc"))
   setGeneric("maNsc", function(object) standardGeneric("maNsc"))
setMethod("maNsc", "marrayRaw", function(object) maNsc(maLayout(object)))

if(!isGeneric("maGridRow"))
   setGeneric("maGridRow", function(object) standardGeneric("maGridRow"))
setMethod("maGridRow", "marrayRaw", function(object) maGridRow(maLayout(object)))

if(!isGeneric("maGridCol"))
   setGeneric("maGridCol", function(object) standardGeneric("maGridCol"))
setMethod("maGridCol", "marrayRaw", function(object) maGridCol(maLayout(object)))

if(!isGeneric("maSpotRow"))
   setGeneric("maSpotRow", function(object) standardGeneric("maSpotRow"))
setMethod("maSpotRow", "marrayRaw", function(object) maSpotRow(maLayout(object)))

if(!isGeneric("maSpotCol"))
   setGeneric("maSpotCol", function(object) standardGeneric("maSpotCol"))
setMethod("maSpotCol", "marrayRaw", function(object) maSpotCol(maLayout(object)))

if(!isGeneric("maPrintTip"))
   setGeneric("maPrintTip", function(object) standardGeneric("maPrintTip"))
setMethod("maPrintTip", "marrayRaw", function(object) maPrintTip(maLayout(object)))

if(!isGeneric("maSub"))
   setGeneric("maSub", function(object) standardGeneric("maSub"))
setMethod("maSub", "marrayRaw", function(object) maSub(maLayout(object)))

if(!isGeneric("maPlate"))
   setGeneric("maPlate", function(object) standardGeneric("maPlate"))
setMethod("maPlate", "marrayRaw", function(object) maPlate(maLayout(object)))

if(!isGeneric("maControls"))
   setGeneric("maControls", function(object) standardGeneric("maControls"))
setMethod("maControls", "marrayRaw", function(object) maControls(maLayout(object)))

if(!isGeneric("maNsamples"))
   setGeneric("maNsamples", function(object) standardGeneric("maNsamples"))
setMethod("maNsamples", "marrayRaw", function(object) ncol(maRf(object)))

#########################
if(!isGeneric("maGnames"))
   setGeneric("maGnames", function(object) standardGeneric("maGnames"))
setMethod("maGnames", "marrayRaw", function(object) slot(object, "maGnames"))

if(!isGeneric("maTargets"))
   setGeneric("maTargets", function(object) standardGeneric("maTargets"))
setMethod("maTargets", "marrayRaw", function(object) slot(object, "maTargets"))

if(!isGeneric("maNotes"))
   setGeneric("maNotes", function(object) standardGeneric("maNotes"))
setMethod("maNotes", "marrayRaw", function(object) slot(object, "maNotes"))

###########################################################################
# Accessor methods for marrayNorm class

if(!isGeneric("maA"))
   setGeneric("maA", function(object) standardGeneric("maA"))
setMethod("maA", "marrayNorm", function(object) slot(object, "maA"))

if(!isGeneric("maM"))
   setGeneric("maM", function(object) standardGeneric("maM"))
setMethod("maM", "marrayNorm", function(object) slot(object, "maM"))

if(!isGeneric("maMloc"))
   setGeneric("maMloc", function(object) standardGeneric("maMloc"))
setMethod("maMloc", "marrayNorm", function(object) slot(object, "maMloc"))

if(!isGeneric("maMscale"))
   setGeneric("maMscale", function(object) standardGeneric("maMscale"))
setMethod("maMscale", "marrayNorm", function(object) slot(object, "maMscale"))

if(!isGeneric("maW"))
   setGeneric("maW", function(object) standardGeneric("maW"))
setMethod("maW", "marrayNorm", function(object) slot(object, "maW"))

#########################
# maLayout slots

if(!isGeneric("maLayout"))
   setGeneric("maLayout", function(object) standardGeneric("maLayout"))
setMethod("maLayout", "marrayNorm", function(object) slot(object, "maLayout"))

if(!isGeneric("maNspots"))
   setGeneric("maNspots", function(object) standardGeneric("maNspots"))
setMethod("maNspots", "marrayNorm", function(object) maNspots(maLayout(object)))

if(!isGeneric("maNgr"))
   setGeneric("maNgr", function(object) standardGeneric("maNgr"))
setMethod("maNgr", "marrayNorm", function(object) maNgr(maLayout(object)))

if(!isGeneric("maNgc"))
   setGeneric("maNgc", function(object) standardGeneric("maNgc"))
setMethod("maNgc", "marrayNorm", function(object) maNgc(maLayout(object)))

if(!isGeneric("maNsr"))
   setGeneric("maNsr", function(object) standardGeneric("maNsr"))
setMethod("maNsr", "marrayNorm", function(object) maNsr(maLayout(object)))

if(!isGeneric("maNsc"))
   setGeneric("maNsc", function(object) standardGeneric("maNsc"))
setMethod("maNsc", "marrayNorm", function(object) maNsc(maLayout(object)))

if(!isGeneric("maGridRow"))
   setGeneric("maGridRow", function(object) standardGeneric("maGridRow"))
setMethod("maGridRow", "marrayNorm", function(object) maGridRow(maLayout(object)))

if(!isGeneric("maGridCol"))
   setGeneric("maGridCol", function(object) standardGeneric("maGridCol"))
setMethod("maGridCol", "marrayNorm", function(object) maGridCol(maLayout(object)))

if(!isGeneric("maSpotRow"))
   setGeneric("maSpotRow", function(object) standardGeneric("maSpotRow"))
setMethod("maSpotRow", "marrayNorm", function(object) maSpotRow(maLayout(object)))

if(!isGeneric("maSpotCol"))
   setGeneric("maSpotCol", function(object) standardGeneric("maSpotCol"))
setMethod("maSpotCol", "marrayNorm", function(object) maSpotCol(maLayout(object)))

if(!isGeneric("maPrintTip"))
   setGeneric("maPrintTip", function(object) standardGeneric("maPrintTip"))
setMethod("maPrintTip", "marrayNorm", function(object) maPrintTip(maLayout(object)))

if(!isGeneric("maSub"))
   setGeneric("maSub", function(object) standardGeneric("maSub"))
setMethod("maSub", "marrayNorm", function(object) maSub(maLayout(object)))

if(!isGeneric("maPlate"))
   setGeneric("maPlate", function(object) standardGeneric("maPlate"))
setMethod("maPlate", "marrayNorm", function(object) maPlate(maLayout(object)))

if(!isGeneric("maControls"))
   setGeneric("maControls", function(object) standardGeneric("maControls"))
setMethod("maControls", "marrayNorm", function(object) maControls(maLayout(object)))

if(!isGeneric("maNsamples"))
   setGeneric("maNsamples", function(object) standardGeneric("maNsamples"))
setMethod("maNsamples", "marrayNorm", function(object) ncol(maM(object)))

#########################
if(!isGeneric("maGnames"))
   setGeneric("maGnames", function(object) standardGeneric("maGnames"))
setMethod("maGnames", "marrayNorm", function(object) slot(object, "maGnames"))

if(!isGeneric("maTargets"))
   setGeneric("maTargets", function(object) standardGeneric("maTargets"))
setMethod("maTargets", "marrayNorm", function(object) slot(object, "maTargets"))

if(!isGeneric("maNotes"))
   setGeneric("maNotes", function(object) standardGeneric("maNotes"))
setMethod("maNotes", "marrayNorm", function(object) slot(object, "maNotes"))

if(!isGeneric("maNormCall"))
   setGeneric("maNormCall", function(object) standardGeneric("maNormCall"))
setMethod("maNormCall", "marrayNorm", function(object) slot(object, "maNormCall"))

##########################################################################
# maSet.R
#
# Assignment methods for two-color cDNA microarrays classes
#
###########################################################################

###########################################################################
# Assignment methods for maLabels slot of marrayInfo class

if( !isGeneric("maLabels<-") )
      setGeneric("maLabels<-", function(object, value)
               standardGeneric("maLabels<-"))

setReplaceMethod("maLabels", signature("marrayInfo","character"),
             function(object, value){
               slot(object, "maLabels")<-value
               object
             })

setReplaceMethod("maLabels", signature("marrayInfo", "numeric"),
             function(object, value){
               info <- maInfo(object)
               if(length(value)==1)
		 labels<-info[,value]
	       else
                 labels <- apply(info[,value], 1, paste, collapse=" ")
               slot(object, "maLabels") <- labels
               object
             })

###########################################################################
# Assignment methods for maInfo slot of marrayInfo class

if( !isGeneric("maInfo<-") )
      setGeneric("maInfo<-", function(object, value)
               standardGeneric("maInfo<-"))

setReplaceMethod("maInfo", signature("marrayInfo","data.frame"),
             function(object, value) {
               slot(object, "maInfo")<-value
               object
             })

###########################################################################
# Assignment methods for maNgr slot of marrayLayout class

if( !isGeneric("maNgr<-") )
      setGeneric("maNgr<-", function(object, value)
               standardGeneric("maNgr<-"))

setReplaceMethod("maNgr", signature("marrayLayout","numeric"),
  	function(object, value) {
     	  slot(object,"maNgr")<- value
	  object
	})

setReplaceMethod("maNgr", signature("marrayRaw","numeric"),
	function(object, value) {
	  maNgr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgr", signature("marrayNorm","numeric"),
  	function(object, value) {
	  maNgr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgr", signature("marraySpots","numeric"),
  	function(object, value) {
	  maNgr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgr", signature("marrayTwo","numeric"),
  	function(object, value) {
	  maNgr(slot(object,"maLayout"))<- value
	  object
	})

###########################################################################
# Assignment methods for maNgc slot of marrayLayout class

if( !isGeneric("maNgc<-") )
      setGeneric("maNgc<-", function(object, value)
               standardGeneric("maNgc<-"))

setReplaceMethod("maNgc", signature("marrayLayout","numeric"),
  	function(object, value) {
     	  slot(object,"maNgc")<- value
	  object
	})

setReplaceMethod("maNgc", signature("marrayRaw","numeric"),
	function(object, value) {
	  maNgc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgc", signature("marrayNorm","numeric"),
  	function(object, value) {
	  maNgc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgc", signature("marraySpots","numeric"),
  	function(object, value) {
	  maNgc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNgc", signature("marrayTwo","numeric"),
  	function(object, value) {
	  maNgc(slot(object,"maLayout"))<- value
	  object
	})

###########################################################################
# Assignment methods for maNsr slot of marrayLayout class

if( !isGeneric("maNsr<-") )
      setGeneric("maNsr<-", function(object, value)
               standardGeneric("maNsr<-"))

setReplaceMethod("maNsr", signature("marrayLayout","numeric"),
  	function(object, value) {
     	  slot(object,"maNsr")<- value
	  object
	})

setReplaceMethod("maNsr", signature("marrayRaw","numeric"),
	function(object, value) {
	  maNsr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsr", signature("marrayNorm","numeric"),
  	function(object, value) {
	  maNsr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsr", signature("marraySpots","numeric"),
  	function(object, value) {
	  maNsr(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsr", signature("marrayTwo","numeric"),
  	function(object, value) {
	  maNsr(slot(object,"maLayout"))<- value
	  object
	})

###########################################################################
# Assignment methods for maNsc slot of marrayLayout class

if( !isGeneric("maNsc<-") )
      setGeneric("maNsc<-", function(object, value)
               standardGeneric("maNsc<-"))

setReplaceMethod("maNsc", signature("marrayLayout","numeric"),
  	function(object, value) {
     	  slot(object,"maNsc")<- value
	  object
	})

setReplaceMethod("maNsc", signature("marrayRaw","numeric"),
	function(object, value) {
	  maNsc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsc", signature("marrayNorm","numeric"),
  	function(object, value) {
	  maNsc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsc", signature("marraySpots","numeric"),
  	function(object, value) {
	  maNsc(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNsc", signature("marrayTwo","numeric"),
  	function(object, value) {
	  maNsc(slot(object,"maLayout"))<- value
	  object
	})

###########################################################################
# Assignment methods for maNspots slot of marrayLayout class

if( !isGeneric("maNspots<-") )
      setGeneric("maNspots<-", function(object, value)
               standardGeneric("maNspots<-"))

setReplaceMethod("maNspots", signature("marrayLayout","numeric"),
  	function(object, value) {
     	  slot(object,"maNspots")<- value
	  object
	})

setReplaceMethod("maNspots", signature("marrayRaw","numeric"),
	function(object, value) {
	  maNspots(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNspots", signature("marrayNorm","numeric"),
  	function(object, value) {
	  maNspots(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNspots", signature("marraySpots","numeric"),
  	function(object, value) {
	  maNspots(slot(object,"maLayout"))<- value
	  object
	})

setReplaceMethod("maNspots", signature("marrayTwo","numeric"),
  	function(object, value) {
	  maNspots(slot(object,"maLayout"))<- value
	  object
	})

###########################################################################
# Assignment methods for maSub slot of marrayLayout class

if( !isGeneric("maSub<-") )
      setGeneric("maSub<-", function(object, value)
               standardGeneric("maSub<-"))

setReplaceMethod("maSub", signature("marrayLayout","logical"),
             function(object, value) {
               slot(object, "maSub")<-value
               object
             })

setReplaceMethod("maSub", signature("marrayLayout","numeric"),
             function(object, value){
               slot(object, "maSub")<-maNum2Logic(maNspots(object),value)
               object
             })

setReplaceMethod("maSub", signature("marrayRaw"),
             function(object,value) {
                maSub(slot(object, "maLayout")) <- value
                object
             })

setReplaceMethod("maSub", signature("marrayNorm"),
             function(object,value){
                maSub(slot(object, "maLayout")) <- value
                object
             })


setReplaceMethod("maSub", signature("marraySpots"),
             function(object,value){
                maSub(slot(object, "maLayout")) <- value
                object
             })


setReplaceMethod("maSub", signature("marrayTwo"),
             function(object,value){
                maSub(slot(object, "maLayout")) <- value
                object
             })

###########################################################################
# Assignment methods for maPlate slot of marrayLayout class

if( !isGeneric("maPlate<-") )
      setGeneric("maPlate<-", function(object, value)
               standardGeneric("maPlate<-"))

setReplaceMethod("maPlate", "marrayLayout",
  	function(object, value) {
     	  slot(object,"maPlate")<- factor(value)
	  object
	})

setReplaceMethod("maPlate", "marrayRaw",
	function(object, value) {
	  maPlate(slot(object,"maLayout"))<- factor(value)
	  object
	})

setReplaceMethod("maPlate", "marrayNorm",
  	function(object, value) {
	  maPlate(slot(object,"maLayout"))<- factor(value)
	  object
	})

setReplaceMethod("maPlate", "marraySpots",
  	function(object, value) {
	  maPlate(slot(object,"maLayout"))<- factor(value)
	  object
	})


setReplaceMethod("maPlate", "marrayTwo",
  	function(object, value) {
	  maPlate(slot(object,"maLayout"))<- factor(value)
	  object
	})

###########################################################################
# Assignment methods for maControls slot of marrayLayout class

if( !isGeneric("maControls<-") )
      setGeneric("maControls<-", function(object, value)
               standardGeneric("maControls<-"))

setReplaceMethod("maControls", "marrayLayout",
  function(object, value) {
     slot(object,"maControls")<- factor(value)
     object
  })

setReplaceMethod("maControls", "marrayRaw",
  function(object, value) {
     maControls(slot(object,"maLayout"))<- factor(value)
     object
  })

setReplaceMethod("maControls", "marrayNorm",
  function(object, value) {
     maControls(slot(object,"maLayout")) <- factor(value)
     object
  })


setReplaceMethod("maControls", "marraySpots",
  function(object, value) {
     maControls(slot(object,"maLayout")) <- factor(value)
     object
  })

setReplaceMethod("maControls", "marrayTwo",
  function(object, value) {
     maControls(slot(object,"maLayout")) <- factor(value)
     object
  })

###########################################################################
# Assignment methods for maLayout slot of marrayRaw, marrayNorm etc. classes

if( !isGeneric("maLayout<-") )
      setGeneric("maLayout<-", function(object, value)
               standardGeneric("maLayout<-"))

setReplaceMethod("maLayout", signature("marrayRaw", "marrayLayout"),
  function(object, value) {
     slot(object,"maLayout")<- value
     object
  })

setReplaceMethod("maLayout", signature("marrayNorm", "marrayLayout"),
  function(object, value) {
     slot(object,"maLayout")<- value
     object
  })

setReplaceMethod("maLayout", signature("marraySpots", "marrayLayout"),
  function(object, value) {
     slot(object,"maLayout")<- value
     object
  })

setReplaceMethod("maLayout", signature("marrayTwo", "marrayLayout"),
  function(object, value) {
     slot(object,"maLayout")<- value
     object
  })

###########################################################################
# Assignment methods for maGnames slot of marrayRaw, marrayNorm, and
# marraySpots classes

if( !isGeneric("maGnames<-") )
      setGeneric("maGnames<-", function(object, value)
               standardGeneric("maGnames<-"))

setReplaceMethod("maGnames", signature("marrayRaw", "marrayInfo"),
  function(object, value) {
     slot(object,"maGnames")<- value
     object
  })

setReplaceMethod("maGnames", signature("marrayNorm", "marrayInfo"),
  function(object, value) {
     slot(object,"maGnames")<- value
     object
  })

setReplaceMethod("maGnames", signature("marraySpots", "marrayInfo"),
  function(object, value) {
     slot(object,"maGnames")<- value
     object
  })

###########################################################################
# Assignment methods for maTargets slot of marrayRaw, marrayNorm, and
# marrayTwo classes

if( !isGeneric("maTargets<-") )
      setGeneric("maTargets<-", function(object, value)
               standardGeneric("maTargets<-"))

setReplaceMethod("maTargets", signature("marrayRaw", "marrayInfo"),
  function(object, value) {
     slot(object,"maTargets")<- value
     object
  })

setReplaceMethod("maTargets", signature("marrayNorm", "marrayInfo"),
  function(object, value) {
     slot(object,"maTargets")<- value
     object
  })

setReplaceMethod("maTargets", signature("marrayTwo", "marrayInfo"),
  function(object, value) {
     slot(object,"maTargets")<- value
     object
  })

###########################################################################
# Assignment methods for maW slot of marrayRaw and marrayNorm classes

if( !isGeneric("maW<-") )
      setGeneric("maW<-", function(object, value)
               standardGeneric("maW<-"))

setReplaceMethod("maW", signature("marrayRaw", "matrix"),
  function(object, value) {
     slot(object,"maW")<- value
     object
  })

setReplaceMethod("maW", signature("marrayNorm", "matrix"),
  function(object, value) {
     slot(object,"maW")<- value
     object
  })

###########################################################################
# Assignment methods for maGf slot of marrayRaw class

if( !isGeneric("maGf<-") )
      setGeneric("maGf<-", function(object, value)
               standardGeneric("maGf<-"))

setReplaceMethod("maGf", signature("marrayRaw", "matrix"),
  function(object, value) {
     slot(object,"maGf")<- value
     object
  })

###########################################################################
# Assignment methods for maRf slot of marrayRaw class

if( !isGeneric("maRf<-") )
      setGeneric("maRf<-", function(object, value)
               standardGeneric("maRf<-"))

setReplaceMethod("maRf", signature("marrayRaw", "matrix"),
  function(object, value) {
     slot(object,"maRf")<- value
     object
  })

###########################################################################
# Assignment methods for maGb slot of marrayRaw class

if( !isGeneric("maGb<-") )
      setGeneric("maGb<-", function(object, value)
               standardGeneric("maGb<-"))

setReplaceMethod("maGb", signature("marrayRaw", "matrix"),
  function(object, value) {
     slot(object,"maGb")<- value
     object
  })

setReplaceMethod("maGb", signature("marrayRaw", "NULL"),
  function(object, value) {
     slot(object,"maGb")<- value
     object
  })

###########################################################################
# Assignment methods for maRb slot of marrayRaw class

if( !isGeneric("maRb<-") )
      setGeneric("maRb<-", function(object, value)
               standardGeneric("maRb<-"))

setReplaceMethod("maRb", signature("marrayRaw", "matrix"),
  function(object, value) {
     slot(object,"maRb")<- value
     object
  })

setReplaceMethod("maRb", signature("marrayRaw", "NULL"),
  function(object, value) {
     slot(object,"maRb")<- value
     object
  })

###########################################################################
# Assignment methods for maA slot of marrayNorm class

if( !isGeneric("maA<-") )
      setGeneric("maA<-", function(object, value)
               standardGeneric("maA<-"))

setReplaceMethod("maA", signature("marrayNorm", "matrix"),
  function(object, value) {
     slot(object,"maA")<- value
     object
  })
###########################################################################
# Assignment methods for maM slot of marrayNorm class

if( !isGeneric("maM<-") )
      setGeneric("maM<-", function(object, value)
               standardGeneric("maM<-"))

setReplaceMethod("maM", signature("marrayNorm", "matrix"),
  function(object, value) {
     slot(object,"maM")<- value
     object
  })

###########################################################################
# Assignment methods for maMloc slot of marrayNorm class

if( !isGeneric("maMloc<-") )
      setGeneric("maMloc<-", function(object, value)
               standardGeneric("maMloc<-"))

setReplaceMethod("maMloc", signature("marrayNorm", "matrix"),
  function(object, value) {
     slot(object,"maMloc")<- value
     object
  })

###########################################################################
# Assignment methods for maMscale slot of marrayNorm class

if( !isGeneric("maMscale<-") )
      setGeneric("maMscale<-", function(object, value)
               standardGeneric("maMscale<-"))

setReplaceMethod("maMscale", signature("marrayNorm", "matrix"),
  function(object, value) {
     slot(object,"maMscale")<- value
     object
  })

###########################################################################
# Assignment methods for maNotes slot of various classes

if( !isGeneric("maNotes<-") )
      setGeneric("maNotes<-", function(object, value)
               standardGeneric("maNotes<-"))

setReplaceMethod("maNotes", signature("marrayLayout", "character"),
  function(object, value) {
     slot(object,"maNotes")<- value
     object
  })

setReplaceMethod("maNotes", signature("marrayRaw", "character"),
  function(object, value) {
     slot(object,"maNotes")<- value
     object
  })

setReplaceMethod("maNotes", signature("marrayNorm", "character"),
  function(object, value) {
     slot(object,"maNotes")<- value
     object
  })

setReplaceMethod("maNotes", signature("marrayInfo", "character"),
  function(object, value) {
     slot(object,"maNotes")<- value
     object
  })

###########################################################################
# Assignment methods for slots of marrayTwo class

if( !isGeneric("maX<-") )
      setGeneric("maX<-", function(object, value)
               standardGeneric("maX<-"))

setReplaceMethod("maX", signature("marrayTwo", "matrix"),
  function(object, value) {
     slot(object,"maX")<- value
     object
  })

if( !isGeneric("maY<-") )
      setGeneric("maY<-", function(object, value)
               standardGeneric("maY<-"))

setReplaceMethod("maY", signature("marrayTwo", "matrix"),
  function(object, value) {
     slot(object,"maY")<- value
     object
  })

###########################################################################
# maPrint.R
#
# Print methods for microarray classes
#
###########################################################################
# marrayLayout class


setMethod("print", "marrayLayout", function(x)
{
    cat("Array layout:\t Object of class marrayLayout. \n\n")

    cat("Total number of spots:\t\t\t")
    cat(maNspots(x))
    cat("\n")

    cat("Dimensions of grid matrix:\t\t")
    cat(paste( maNgr(x), "rows by", maNgc(x), "cols"))
    cat("\n")

    cat("Dimensions of spot matrices:\t\t")
    cat(paste(maNsr(x), "rows by", maNsc(x), "cols"))
    cat("\n\n")

    if(length(maNspots(x))==1)
      nsub<-length((1:maNspots(x))[maSub(x)])
    else
      nsub<-NULL
    cat(paste("Currently working with a subset of ", nsub, " spots.\n\n", sep=""))

    cat("Control spots: \n")
    if(length(maControls(x))!=0){
      tmp <- table(maControls(x))
      cat(paste("There are  ",  length(tmp),  "types of controls : \n"))
      print(tmp[1:min(10, length(tmp))])
      if(length(tmp) > 10) cat ("...")
    }
    cat("\n\n")

    cat("Notes on layout: \n")
    cat(maNotes(x))
    cat("\n")
  })


#########################
# marrayInfo class

setMethod("print","marrayInfo", function(x)
{
  cat("Object of class marrayInfo. \n\n")

  {

    nr <- length(maLabels(x))
    ds <- dim(maInfo(x))

    if(!is.null(ds))
    {
      if(nr!=0)
      {
        tmp<-data.frame(maLabels(x),maInfo(x))
        dimnames(tmp)[[2]][1]<-"maLabels"
        dimnames(tmp)[[2]][-1]<-dimnames(maInfo(x))[[2]]
      }
      else
	tmp<-maInfo(x)
      print(tmp[1:min(nrow(tmp), 10),])
      if(nrow(tmp) > 10)  cat("... \n")
    }

    if(is.null(ds)&(nr!=0))
    {
      tmp <- as.vector(maLabels(x))
      print(tmp[1:min(length(tmp), 10)])
      if(nr > 10)  cat("... \n")
    }

   cat("\nNumber of labels: ", nr, " \n")
   cat("Dimensions of maInfo matrix: ", ds[1], " rows by ", ds[2], " columns\n")
   }
   cat("\n")

  cat("Notes: \n")
  cat(maNotes(x))

  cat("\n")

})

#########################
# marrayRaw class

setMethod("print","marrayRaw", function(x)
{
  if((length(maGf(x))==0) | (length(maRf(x))==0))
    cat("Input is empty \n")
  else
    {
      cat("Pre-normalization intensity data:\t Object of class marrayRaw. \n\n")

      cat(paste("Number of arrays: \t",ncol(maM(x)), " arrays.\n \n", sep=""))
      cat("A) Layout of spots on the array: \n")
      print(maLayout(x))

      cat("\n")

      cat("B) Samples hybridized to the array: \n")
      print(maTargets(x))
      cat("\n")

      cat("C) Summary statistics for log-ratio distribution: \n")
      results <- apply(maM(x),2, function(x){round(summary(x),2)})
      if(is.matrix(results))
        print(t(results))

      if(is.list(results)){
        x.min <- x.1q <- x.med <- x.mean <- x.3q <- x.max <- x.NA <- NULL
        x.names <- names(results)
        if(is.null(x.names)) x.names <- as.character(1:length(results))
        for(i in 1:length(results)){
          x.min <- c(x.min, results[[i]][1])
          x.1q <- c(x.1q, results[[i]][2])
          x.med <- c(x.med, results[[i]][3])
          x.mean <- c(x.mean, results[[i]][4])
          x.3q <- c(x.3q, results[[i]][5])
          x.max <- c(x.max, results[[i]][6])
          x.NA <- c(x.NA, results[[i]][7])
        }
        y <- data.frame(x.names, x.min, x.1q, x.med, x.mean, x.3q, x.max, as.vector(x.NA))
        dimnames(y)[[2]] <-
          c("   ", "Min.", "1st Qu.", "Median", "Mean", "3rd Qu.", "Max", "NA")
        print(y)
      }

      cat("\n")

      cat("D) Notes on intensity data: \n")
      cat(maNotes(x))
      cat("\n")
    }
})

#########################
# marrayNorm class

setMethod("print","marrayNorm", function(x)
{

  if((length(maA(x))==0) | (length(maM(x))==0))
    cat("Input is empty \n")
  else
  {
  cat("Normalized intensity data:\t Object of class marrayNorm.\n \n")

  cat("Call to normalization function:\n")
  print(maNormCall(x))

  cat(paste("\nNumber of arrays: \t",ncol(maM(x)), " arrays.\n", sep=""))

  cat("\nA) Layout of spots on the array: \n")
  print(maLayout(x))
  cat("\n")

  cat("B) Samples hybridized to the array: \n")
  print(maTargets(x))
  cat("\n")

  cat("C) Summary statistics for log-ratio distribution: \n")
  if(length(maM(x))!=0)
     results <- apply(maM(x),2, function(x){round(summary(x),2)})
      if(is.matrix(results))
        print(t(results))

      if(is.list(results)){
        x.min <- x.1q <- x.med <- x.mean <- x.3q <- x.max <- x.NA <- NULL
        x.names <- names(results)
        if(is.null(x.names)) x.names <- as.character(1:length(results))
        for(i in 1:length(results)){
          x.min <- c(x.min, results[[i]][1])
          x.1q <- c(x.1q, results[[i]][2])
          x.med <- c(x.med, results[[i]][3])
          x.mean <- c(x.mean, results[[i]][4])
          x.3q <- c(x.3q, results[[i]][5])
          x.max <- c(x.max, results[[i]][6])
          x.NA <- c(x.NA, results[[i]][7])
        }
        y <- data.frame(x.names, x.min, x.1q, x.med, x.mean, x.3q, x.max, as.vector(x.NA))
        dimnames(y)[[2]] <-
          c("   ", "Min.", "1st Qu.", "Median", "Mean", "3rd Qu.", "Max", "NA")
        print(y)
      }
  cat("\n")

  cat("D) Notes on intensity data: \n")
  cat(maNotes(x))
  cat("\n")
  }
})


#########################
# marrayTwo class

setMethod("print","marrayTwo", function(x)
{
  printsummary <- function(matrixX)
    {
      results <- apply(matrixX,2, function(x){round(summary(x),2)})
      if(is.matrix(results))
        print(t(results))

      if(is.list(results)){
        x.min <- x.1q <- x.med <- x.mean <- x.3q <- x.max <- x.NA <- NULL
        x.names <- names(results)
        if(is.null(x.names)) x.names <- as.character(1:length(results))
        for(i in 1:length(results)){
          x.min <- c(x.min, results[[i]][1])
          x.1q <- c(x.1q, results[[i]][2])
          x.med <- c(x.med, results[[i]][3])
          x.mean <- c(x.mean, results[[i]][4])
          x.3q <- c(x.3q, results[[i]][5])
          x.max <- c(x.max, results[[i]][6])
          x.NA <- c(x.NA, results[[i]][7])
        }
        y <- data.frame(x.names, x.min, x.1q, x.med, x.mean, x.3q, x.max, as.vector(x.NA))
        dimnames(y)[[2]] <-
          c("   ", "Min.", "1st Qu.", "Median", "Mean", "3rd Qu.", "Max", "NA")
        print(y)
      }
    }
  cat("Object of class marrayTwo. \n\n")

  cat("A) Layout of spots on the array: \n")
  print(maLayout(x))
  cat("\n")

  cat("B) Summary statistics for X: \n")
  if(length(maX(x))!=0)
    printsummary(maX(x))
  cat("\n")

  cat("C) Summary statistics for Y: \n")
  if(length(maY(x))!=0)
    printsummary(maY(x))
  cat("\n")

})

#########################
# marraySpots class

setMethod("print","marraySpots", function(x)
{
  cat("Object of class marraySpots. \n\n")
  cat("A) Layout of spots on the array: \n")
  print(maLayout(x))
  cat("\n")

  cat("B) Names hybridized to the array: \n")
  print(maGnames(x))
  cat("\n")
})


###########################################################################
###########################################################################
# maShow.R
#
# Show methods for microarray classes
#
###########################################################################


# marrayLayout class

setMethod("show", "marrayLayout", function(object)
{
  print(object)
})


#########################
# marrayInfo class

setMethod("show","marrayInfo", function(object)
{
  print(object)
})

#########################
# marrayRaw class

setMethod("show","marrayRaw", function(object)
{
  print(object)
})

#########################
# marrayNorm class

setMethod("show","marrayNorm", function(object)
{
  print(object)
})


#########################
# marrayTwo class

setMethod("show","marrayTwo", function(object)
{
  print(object)
})

#########################
# marraySpots class

setMethod("show","marraySpots", function(object)
{
  print(object)
})


###########################################################################
###########################################################################
# maSubset.R
#
# Subsetting methods for microarray classes
#
###########################################################################
# marrayInfo class


setMethod("[", "marrayInfo", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(missing(j))
   j<-TRUE
  if(missing(i))
   i<-TRUE
  if(length(maLabels(x))!=0)
    slot(newx,"maLabels")<-maLabels(x)[i]
  if(length(maInfo(x))!=0)
    slot(newx,"maInfo")<-maInfo(x)[i, j, drop=FALSE]
  return(newx)
})

#########################
# marrayLayout class

setMethod("[", "marrayLayout", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(!missing(i))
  {
    if(length(maPlate(x))!=0)
      slot(newx,"maPlate")<-maPlate(x)[i]
    if(length(maControls(x))!=0)
    slot(newx,"maControls")<-maControls(x)[i]
    if(length(maNspots(x))!=0)
    {
      sub<-rep(FALSE,maNspots(x))
      sub[maSub(x)][i]<-TRUE
      slot(newx, "maSub")<-sub
    }
  }
  return(newx)
})

#########################
# marrayRaw class

setMethod("[", "marrayRaw", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(missing(j))
   j<-TRUE
  if(missing(i))
   i<-TRUE

  if(length(maRf(x))!=0)
    slot(newx,"maRf")<-maRf(x)[i,j,drop=FALSE]
  if(length(maGf(x))!=0)
    slot(newx,"maGf")<-maGf(x)[i,j,drop=FALSE]
  if(length(maRb(x))!=0)
    slot(newx,"maRb")<-maRb(x)[i,j,drop=FALSE]
  if(length(maGb(x))!=0)
    slot(newx,"maGb")<-maGb(x)[i,j,drop=FALSE]
 if(length(maW(x))!=0)
    slot(newx,"maW")<-maW(x)[i,j,drop=FALSE]
  slot(newx,"maLayout")<-maLayout(x)[i]
  slot(newx,"maGnames")<-maGnames(x)[i]
  slot(newx,"maTargets")<-maTargets(x)[j]
  return(newx)
})


#########################
# marrayTwo class

setMethod("[", "marrayTwo", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(missing(j))
   j<-TRUE
  if(missing(i))
   i<-TRUE

  if(length(maX(x))!=0)
    slot(newx,"maX")<-maX(x)[i,j,drop=FALSE]
  if(length(maY(x))!=0)
    slot(newx,"maY")<-maY(x)[i,j,drop=FALSE]
  slot(newx,"maLayout")<-maLayout(x)[i]
  slot(newx,"maTargets")<-maTargets(x)[j]
  return(newx)
})

#########################
# marraySpots class

setMethod("[", "marraySpots", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(missing(i))
   i<-TRUE
  if(missing(j))
   j<-TRUE
  slot(newx,"maGnames")<-maGnames(x)[i, j, drop=FALSE]
  slot(newx,"maLayout")<-maLayout(x)[i]
  return(newx)
})

#########################
# marrayNorm class

setMethod("[", "marrayNorm", function(x, i, j, ..., drop=FALSE)
{
  newx<-x
  if(missing(j))
   j<-TRUE
  if(missing(i))
   i<-TRUE


  if(length(maA(x))!=0)
    slot(newx,"maA")<-maA(x)[i,j,drop=FALSE]
  if(length(maM(x))!=0)
    slot(newx,"maM")<-maM(x)[i,j,drop=FALSE]
  if(length(maMloc(x))!=0)
    slot(newx,"maMloc")<-maMloc(x)[i,j,drop=FALSE]
  if(length(maMscale(x))!=0)
    slot(newx,"maMscale")<-maMscale(x)[i,j,drop=FALSE]
  if(length(maW(x))!=0)
    slot(newx,"maW")<-maW(x)[i,j,drop=FALSE]
  slot(newx,"maLayout")<-maLayout(x)[i]
  slot(newx,"maGnames")<-maGnames(x)[i]
  slot(newx,"maTargets")<-maTargets(x)[j]
  slot(newx,"maNormCall")<-maNormCall(x)
  return(newx)
})

###########################################################################
  setMethod("rbind", "marrayInfo", function(..., deparse.level = 1)
          {
            data <- list(...)
            newx <- data[[1]]
            for(i in 2:length(data))
              {
                x <- data[[i]]
                if(length(maLabels(x))!=0)
                  slot(newx,"maLabels") <- c(maLabels(newx), maLabels(x))
                if(length(maInfo(x))!=0)
                  slot(newx,"maInfo")<-  rbind(maInfo(newx), maInfo(x))
                if(length(maNotes(x)) != 0)
                  slot(newx,"maNotes")<- paste(maNotes(newx), maNotes(x))
              }
            return(newx)
          })

  setMethod("cbind", "marrayRaw", function(..., deparse.level = 1)
            {
              data <- list(...)
              newx <- data[[1]]
              for(x in data[2:length(data)])
                {
                  if(length(maGf(x))!=0)
                    maGf(newx) <- cbind(maGf(newx), maGf(x))
                  if(length(maRf(x))!=0)
                    maRf(newx) <- cbind(maRf(newx), maRf(x))
                  if(length(maGb(x))!=0)
                    maGb(newx) <- cbind(maGb(newx), maGb(x))
                  if(length(maRb(x))!=0)
                    maRb(newx) <- cbind(maRb(newx), maRb(x))
                  if(length(maW(x))!=0)
                    maW(newx) <- cbind(maW(newx), maW(x))
                  maTargets(newx) <- rbind(maTargets(newx), maTargets(x))
                  if(length(maNotes(x)) != 0)
                    slot(newx,"maNotes")<- paste(maNotes(newx), maNotes(x))
                }
              return(newx)
            })

  setMethod("cbind", "marrayNorm", function(..., deparse.level = 1)
            {
              data <- list(...)
              newx <- data[[1]]
              for(x in data[2:length(data)])
                {
                  if(length(maM(x))!=0)
                    maM(newx) <- cbind(maM(newx), maM(x))
                  if(length(maA(x))!=0)
                    maA(newx) <- cbind(maA(newx), maA(x))
                  if(length(maGb(x))!=0)
                    maMloc(newx) <- cbind(maMloc(newx), maMloc(x))
                  if(length(maRb(x))!=0)
                    maMscale(newx) <- cbind(maMscale(newx), maMscale(x))
                  if(length(maW(x))!=0)
                    maW(newx) <- cbind(maW(newx), maW(x))
                  maTargets(newx) <- rbind(maTargets(newx), maTargets(x))
                  if(length(maNotes(x)) != 0)
                    slot(newx,"maNotes")<- paste(maNotes(newx), maNotes(x))
                }
              return(newx)
            })


##########################################################################
# maComp.R
#
# Simple computations for cDNA microarray objects
#
###########################################################################
# Convert integer vector of indices to logical vector

maNum2Logic<-function(n=length(subset), subset=TRUE)
{
  if(is.logical(subset))
    return(subset)
  if(is.numeric(subset))
  {
    which<-rep(FALSE,n)
    which[subset]<-TRUE 
    return(which)
  }
}

###########################################################################
# Produce a table of with spot coordinates and gene names

maGeneTable <- function(object)
{
  tmp <- data.frame(
  GR=maGridRow(object),
  GC=maGridCol(object),
  SR=maSpotRow(object),
  SC=maSpotCol(object),
  maInfo(maGnames(object)))
  colnames(tmp) <- c("Grid.R", "Grid.C", "Spot.R", "Spot.C", names(maInfo(maGnames(object))))
  return(tmp)
}

###########################################################################
# Generate plate IDs from dimensions of grid and spot matrices

maCompPlate<-function(x, n=384){
  totalPlate <- maNspots(x) /n
  tmp <- n / (maNgr(x) * maNgc(x))
  factor(rep(rep(1:totalPlate, rep(tmp, totalPlate)), (maNgr(x) * maNgc(x))))
}

###########################################################################
# Convert spot index to grid and spot matrix coordinates (4 coords)

maInd2Coord <- function (x, L)
{
    coord<-cbind(maGridRow(L), maGridCol(L), maSpotRow(L), maSpotCol(L))[x,]
    colnames(coord) <- c("Grid.R", "Grid.C", "Spot.R", "Spot.C")
    coord
}

# Convert grid and spot matrix coordinates (4 coords) to spot index.
# Works for subsetted arrays

maCoord2Ind <- function (x, L)
{
  ngr<-maNgr(L)
  ngc<-maNgc(L)
  nsr<-maNsr(L)
  nsc<-maNsc(L)
  n<-maNspots(L)
  ind<-(nsr * nsc)* ((x[,1] - 1) * ngc + (x[,2] - 1)) + (x[,3] - 1) *
        nsc + x[,4]
  ord<-order(ind)
  ind<-ind[ord]
  coord<-x[ord,]
  sub<-(1:n)[maSub(L)]
  ind<-intersect(ind, sub)
  ind<-order(sub)[sub %in% ind]
  ind
}

###########################################################################
# Generate grid and spot matrix coordinates from ranges of rows and
# columns for the grid and spot matrices

maCompCoord<-function(grows, gcols, srows, scols)
{
  ngr <- length(grows)
  ngc <- length(gcols)
  nsr <- length(srows)
  nsc <- length(scols)
  t1 <- rep(grows, rep(nsr * nsc * ngc, ngr))
  t2 <- rep(rep(gcols, rep(nsr * nsc, ngc)), ngr)
  t3 <- rep(rep(srows, rep(nsc, nsr)), ngc * ngr)
  t4 <- rep(scols, nsr * ngc * ngr)
  coord<-cbind(t1, t2, t3, t4)
  colnames(coord) <- c("Grid.R", "Grid.C", "Spot.R", "Spot.C")
  coord
}

# Generate spot index from ranges of rows and columns for the grid 
# and spot matrices

maCompInd<-function(grows, gcols, srows, scols, L)
{
  coord<-maCompCoord(grows, gcols, srows, scols)
  maCoord2Ind(coord, L)
}

###########################################################################
##
##  Move controlCode and maGenControls from marrayTools to marrayClasses
##  May 7, 2003
###########################################################################
controlCode <-
structure(c("Buffer", "Empty", "AT", "NC", "M200009348", "M200012700", 
"M200016219", "M200016205", "M200013499", "M200003425", "M200006376", 
"M200001318", "M200004477", "M200001732", "M200006590", "M200000829", 
"H200000553", "H200000680", "H200001719", "H200001847", "H200007830", 
"H200008181", "H200008484", "H200008489", "H200009216", "H200009498", 
"H200011103", "H200019704", "18S", "SSC", "mCNR", "NLG", "GABA", 
"pNICE", "M13", "Cot-1", "orward", "everse", "Genomic", "p133", 
"Blam", "T7/SP6", "ephrin", "Buffer", "Empty", "Negative", "Negative", 
"Positive", "Positive", "Positive", "Positive", "Positive", "Positive", 
"Positive", "Positive", "Positive", "Positive", "Positive", "Positive", 
"Positive", "Positive", "Positive", "Positive", "Positive", "Positive", 
"Positive", "Positive", "Positive", "Positive", "Positive", "Positive", 
"Positive", "Buffer", "con", "con", "con", "con", "con", "Positive", 
"Negative", "Negative", "Negative", "Negative", "Negative", "con", 
"con"), .Dim = c(43, 2), .Dimnames = list(c("1", "2", "3", "4", 
"5", "6", "7", "8", "9", "10", "11", "12", "13", "14", "15", 
"16", "17", "18", "19", "20", "21", "22", "23", "24", "25", "26", 
"27", "28", "29", "30", "31", "32", "33", "34", "35", "36", "37", 
"38", "39", "40", "41", "42", "43"), c("Pattern", "Name")))

## But Fix March 03, 2003
## Replace controlCode by controlcode (1.0.6)
## Add controlcode = controlCode (1.0.9), add defaults for gpTools and spotTools to work

maGenControls <- function(Gnames, controlcode=controlCode, id="ID")
{
  if(data.class(Gnames) == "marrayInfo")
    {
      ifelse(is.numeric(id), tmp <- id, tmp <- grep(id, colnames(maInfo(Gnames))))
      ID <- as.vector(maInfo(Gnames[,tmp])[[1]])
    }
  else
    {
      ifelse(is.numeric(id), tmp <- id, tmp <- grep(id, colnames(Gnames)))
      ID <- as.vector(Gnames[,tmp])
    }
  
  Control <- rep("probes", length(ID))
  for(i in 1:nrow(controlcode))
    {
      position <- grep(as.vector(controlcode[i,"Pattern"]), ID)
      if(length(position) > 0)
        Control[position] <- as.vector(controlcode[i, "Name"])
    }
  return(Control)
}
###########################################################################
## Date : October 11, 2002
##
## 
###########################################################################

maDotsMatch <- function(dots, defaults)
  {
    ind <- intersect(names(dots), setdiff(names(defaults), "..."))
    for(i in ind)
      defaults[[i]] <- dots[[i]]
    return(defaults[!names(defaults)=="..."])
  }
