.packageName <- "limma"
#  Following two lines are to enable limma to work with R 1.4-1.6
require("methods")
require("modreg")
#  BACKGROUND CORRECTION

backgroundCorrect <- function(RG, method="subtract") {
#	Apply background correction to microarray data
#	Gordon Smyth
#	12 April 2003.  Last modified 2 October 2003.

	method <- match.arg(method, c("none","subtract", "half", "minimum", "edwards"))
	switch(method,
	subtract={
		RG$R <- RG$R-RG$Rb
		RG$G <- RG$G-RG$Gb
	},
	half={
		RG$R <- pmax(RG$R-RG$Rb, 0.5)
		RG$G <- pmax(RG$G-RG$Gb, 0.5)
	},
	minimum={
		RG$R <- as.matrix(RG$R - RG$Rb)
		RG$G <- as.matrix(RG$G - RG$Gb)
		for (slide in 1:ncol(RG$R)) {
			i <- RG$R[,slide] < 1e-18
			if(any(i,na.rm=TRUE)) {
				m <- min(RG$R[!i,slide],na.rm=TRUE)
				RG$R[i,slide] <- m/2
			}
			i <- RG$G[,slide] < 1e-18
			if(any(i,na.rm=TRUE)) {
				m <- min(RG$G[!i,slide],na.rm=TRUE)
				RG$G[i,slide] <- m/2
			}
		}
	},
	edwards={
#		Log-linear interpolation for dull spots as in Edwards (2003).
#		The threshold values (delta) are chosen such that the number of
#		spots with (0 <= R-Rb < delta) is 100f=10% of the number spots,
#		with (R-Rb < 0) for each channel and array.
		nspots <- NROW(RG$R)
		del <- function(d,f=0.1) quantile(d,sum(d<0)*(1+f)/length(d),na.rm=TRUE)
		sub <- as.matrix(RG$R-RG$Rb)
		delta <- apply(sub, 2, del)
		RG$R <- ifelse(sub > rep(1,nspots)%o%delta, sub, delta*exp(1-(RG$Rb+delta)/RG$R))
		sub <- as.matrix(RG$G-RG$Gb)
		delta <- apply(sub, 2, del)
		RG$G <- ifelse(sub > rep(1,nspots)%o%delta, sub, delta*exp(1-(RG$Gb+delta)/RG$G))
	})
	RG$Rb <- NULL
	RG$Gb <- NULL
	new("RGList",unclass(RG))
}
#	CLASSES.R

setClass("RGList",
#  Class to hold initial read-in data
representation("list")
)

setClass("MAList",
#  Class to hold normalized, rotated data
representation("list")
)

setClass("MArrayLM",
#  Linear model fit
representation("list")
)

setClass("exprSet2",representation(
	expressions="matrix",
	weights="matrix",
	targets="data.frame",
	probes="data.frame",
	printer="list",
	notes="character"
))

printHead <- function(x) {
	what <- "other"
	if(is.vector(x)) what <- "vector"
	if(is.matrix(x) || is.data.frame(x)) what <- "TwoD"
	switch(what,
		vector={
			n <- length(x)
			if(n > 20) {
				print(x[1:5])
				cat(n-5,"more elements ...\n")
			} else
				print(x)
		},
		TwoD={
			n <- nrow(x)
			if(n > 10) {
				print(x[1:5,])
				cat(n-5,"more rows ...\n")
			} else
				print(x)
		},
		other=print(x)
	)
}

setClass("LargeDataObject")
setIs("RGList","LargeDataObject")
setIs("MAList","LargeDataObject")
setIs("MArrayLM","LargeDataObject")
setIs("exprSet2","LargeDataObject")

setMethod("show","LargeDataObject",
#  Print and show method large data objects
#  Gordon Smyth
#  May 2003
function(object) {
	cat("An object of class \"",class(object),"\"\n",sep="")
	for (what in names(object)) {
		x <- object[[what]]
		cat("$",what,"\n",sep="")
		printHead(x)
		cat("\n")
	}
	for (what in setdiff(slotNames(object),".Data")) {
		x <- slot(object,what)
		if(length(x) > 0) {
			cat("@",what,"\n",sep="")
			printHead(x)
			cat("\n")
		}
	}
})

dim.RGList <- function(x) dim(x$R)
dim.MAList <- function(x) dim(x$M)
dim.MArrayLM <- function(x) dim(x$coefficients)

as.MAList <- function(object) {
#	Convert marrayNorm object to MAList
#	Gordon Smyth
#	20 Sep 2003.  Last modified 5 Nov 2003.

	MA <- new("MAList")
	ifposlen <- function(x) if(length(x)) return(x) else return(NULL)
	MA$A <- ifposlen(object@maA)
	MA$M <- ifposlen(object@maM)
	MA$weights <- ifposlen(object@maW)
	MA$printer$ngrid.r <- ifposlen(object@maLayout@maNgr)
	MA$printer$ngrid.c <- ifposlen(object@maLayout@maNgc)
	MA$printer$nspot.r <- ifposlen(object@maLayout@maNsr)
	MA$printer$nspot.c <- ifposlen(object@maLayout@maNsc)
	MA$printer$ngrid.r <- ifposlen(object@maLayout@maNgr)
	MA$printer$ngrid.r <- ifposlen(object@maLayout@maNgr)
	MA$printer$notes <- ifposlen(object@maLayout@maNotes)
	MA$genes <- ifposlen(object@maGnames@maInfo)
	MA$genes$Labels <- ifposlen(object@maGnames@maLabels)
	attr(MA$genes,"notes") <- ifposlen(object@maGnames@maNotes)
	MA$genes$Sub <- ifposlen(object@maLayout@maSub)
	MA$genes$Plate <- ifposlen(object@maLayout@maPlate)
	MA$genes$Controls <- ifposlen(object@maLayout@maControls)
	MA$targets <- ifposlen(object@maTargets@maInfo)
	MA$targets$Labels <- ifposlen(object@maTargets@maLabels)
	MA$notes <- ifposlen(object@maNotes)
	MA$maNormCall <- ifposlen(object@maNormCall)
	MA
} 

#  CLASSIFICATION

classifyTests <- function(tstat,cor.matrix=NULL,design=NULL,contrasts=diag(ncol(design)),df=Inf,p.value=0.01) {
#	Use F-tests to classify vectors of t-test statistics into outcomes
#	Gordon Smyth
#	20 Mar 2003.  Last revised 15 September 2003.

#	Method intended for MAList objects but accept unclassed lists as well
	if(is.list(tstat)) {
		if(is.null(tstat$t)) stop("tstat cannot be extracted from object")
		if(missing(design) && !is.null(tstat$design)) design <- tstat$design
		if(missing(contrasts) && !is.null(tstat$contrasts)) contrasts <- tstat$contrasts
		if(missing(df) && !is.null(tstat$df.prior) && !is.null(tstat$df.residual)) df <- tstat$df.prior+tstat$df.residual
		tstat <- tstat$t
	}

	tstat <- as.matrix(tstat)
	ngenes <- nrow(tstat)
	ntests <- ncol(tstat)
	if(ntests == 1) {
		qT <- qt(p.value/2, df, lower.tail=FALSE)
		return( sign(tstat) * (abs(tstat) > qT) )
	}

#	cor.matrix is estimated correlation matrix of the coefficients
#	and also the estimated covariance matrix of the t-statistics
	if(!is.null(cor.matrix) && !is.null(design)) stop("Cannot specify both cor.matrix and design")
	if(!is.null(design)) {
		design <- as.matrix(design)
		R <- chol(crossprod(design))
		cor.matrix <- crossprod(backsolve(R,contrasts,transpose=TRUE))
		d <- sqrt(diag(cor.matrix))
		cor.matrix <- cor.matrix / (d %*% t(d))
	}
	if(is.null(cor.matrix)) {
		r <- ntests
		Q <- diag(r)/sqrt(r)
	} else {
		E <- eigen(cor.matrix,symmetric=TRUE)
		r <- sum(E$values/E$values[1] > 1e-8)
		Q <- matvec( E$vectors[,1:r], 1/sqrt(E$values[1:r]))/sqrt(r)
	}

	qF <- qf(p.value, r, df, lower.tail=FALSE)
	if(length(qF)==1) qF <- rep(qF,ngenes) 
	result <- matrix(0,ngenes,ntests,dimnames=dimnames(tstat))
	Fstat <- rep(NA,ngenes)
	if(is.null(colnames(tstat)) && !is.null(colnames(contrasts))) colnames(result) <- colnames(contrasts)
	for (i in 1:ngenes) {
		x <- tstat[i,]
		if(any(is.na(x)))
			result[i,] <- NA
		else
			if( (Fstat[i] <- crossprod(crossprod(Q,x))) > qF[i] ) {
				ord <- order(abs(x),decreasing=TRUE)
				result[i,ord[1]] <- sign(x[ord[1]])
				for (j in 2:ntests) {
					bigger <- ord[1:(j-1)]
					x[bigger] <- sign(x[bigger]) * abs(x[ord[j]])
					if( crossprod(crossprod(Q,x)) > qF[i] )
						result[i,ord[j]] <- sign(x[ord[j]])
					else
						break
				}
			}
	}
	structure(list(classification=result,Fstat=Fstat),class="classification")
}

classifyTestsT <- function(tstat,t1=4,t2=3) {
#	Simple classification of vectors of t-test statistics
#	Gordon Smyth
#	1 July 2003.

	if(is.list(tstat)) tstat <- tstat$t
	if(is.null(dim(tstat))) dim(tstat) <- c(1,length(tstat))
	apply(tstat,1,function(x) any(abs(x)>t1,na.rm=TRUE)) * sign(tstat)*(abs(tstat)>t2)
}

classifyTestsP <- function(tstat,df=Inf,p.value=0.05,method="holm") {
#	Simple classification of vectors of t-test statistics using adjusted p-values
#	Gordon Smyth
#	12 July 2003.

	if(is.list(tstat)) tstat <- tstat$t
	if(is.null(dim(tstat))) dim(tstat) <- c(1,length(tstat))
	ngenes <- nrow(tstat)
	P <- 2*pt(-abs(tstat),df=df)
	result <- array(0,dim(P))
	for (i in 1:ngenes) {
		P[i,] <- p.adjust(P[i,],method=method)
		result[i,] <- sign(tstat[i,])*(P[i,]<p.value)
	}
	result
}
designMatrix <- function(targets, ref) {
#	Design matrix for two-color experiments
#	'targets' is matrix or data.frame with columns Cy3 and Cy5
#	Gordon Smyth
#	25 June 2003.  Last modified 29 June 2003.

	tar <- targets[,c("Cy3","Cy5")]
	tar <- as.vector(t(as.matrix(tar)))
	lev <- unique(tar)
	treatments <- setdiff(lev,ref)
	lev <- c(ref,treatments)
	tar <- factor(tar,levels=lev)
	n <- length(tar)
	col <- factor(rep(c(1,2),length=n))
	contrasts(col) <- matrix(c(-1,1),2,1)
	X <- model.matrix(~-1+tar*col)
	keeprows <- X[,1]==0
	keepcols <- attr(X,"assign")==3
	design <- X[keeprows,keepcols,drop=FALSE]
	rownames(design) <- rownames(targets)
	colnames(design) <- treatments
	design
}

makeContrasts <- function(..., levels) {
#	Construct matrix of custom contrasts
#	Gordon Smyth
#	30 June 2003.

	if(is.factor(levels)) levels <- levels(levels)
	if(is.matrix(levels)) levels <- colnames(levels)
	levels <- make.names(levels)
	n <- length(levels)
	if(n < 1) stop("No levels to construct contrasts from")
	indicator <- function(i,n) {
		out <- rep(0,n)
		out[i] <- 1
		out
	}
	for (i in 1:n) assign(levels[i], indicator(i,n))
	e <- substitute(list(...))
	ne <- length(e)
	cm <- matrix(0,n,ne-1)
	rownames(cm) <- levels
	if(ne < 2) return(cm)
	colnames(cm) <- as.character(e)[2:ne]
	for (j in 1:(ne-1)) {
		ej <- e[[j+1]]
		if(is.character(ej)) ej <- parse(text=ej)
		cm[,j] <- eval(ej)
	}
	cm
}
#  DIFFERENTIAL EXPRESSION

eBayes <- function(fit,proportion=0.01,std.coef=NULL) {
#	Empirical Bayes statistics to select differentially expressed genes
#	Object orientated version
#	Gordon Smyth
#	4 August 2003.

	eb <- ebayes(fit=fit,proportion=proportion,std.coef=std.coef)
	fit$df.prior <- eb$df.prior
	fit$s2.prior <- eb$s2.prior
	fit$var.prior <- eb$var.prior
	fit$proportion <- proportion
	fit$s2.post <- eb$s2.post
	fit$t <- eb$t
	fit$p.value <- eb$p.value
	fit$lods <- eb$lods
	fit
}

ebayes <- function(fit,proportion=0.01,std.coef=NULL) {
#	Empirical Bayes statistics to select differentially expressed genes
#	Gordon Smyth
#	8 Sept 2002.  Last revised 14 August 2003.

	coefficients <- fit$coefficients
	stdev.unscaled <- fit$stdev.unscaled
	sigma <- fit$sigma
	df.residual <- fit$df.residual
	if(is.null(coefficients) || is.null(stdev.unscaled) || is.null(sigma) || is.null(df.residual)) stop("No data, or argument is not a valid lmFit object")
	if(all(df.residual==0)) stop("No residual degrees of freedom in linear model fits")
	if(all(!is.finite(sigma))) stop("No finite residual standard deviations")

#	Moderated t-statistic
	out <- fitFDist(sigma^2,df1=df.residual)
	out$s2.prior <- out$scale
	out$df.prior <- out$df2
	out$df2 <- out$scale <- NULL
	df.total <- df.residual + out$df.prior
	if(is.null(out$df.prior) || is.na(out$df.prior)) stop("Could not estimate prior df")
	if(out$df.prior == Inf)
		out$s2.post <- rep(out$s2.prior,length(sigma))
	else
		out$s2.post <- (ifelse(df.residual==0, 0, df.residual*sigma^2) + out$df.prior*out$s2.prior) / df.total
	out$t <- coefficients / stdev.unscaled / sqrt(out$s2.post)
	out$p.value <- 2*pt(-abs(out$t),df=df.total)

#	B-statistic
	if(is.null(std.coef)) {
		varpriorlim <- 10/out$s2.prior
		out$var.prior <- tmixture.matrix(out$t,stdev.unscaled,df.total,proportion,varpriorlim)
		out$var.prior[ is.na(out$var.prior) ] <- 1/out$s2.prior
		out$var.prior <- pmax(out$var.prior, 0.1/out$s2.prior)
	} else
		out$var.prior <- rep(std.coef^2/out$s2.prior, NCOL(out$t))
	r <- rep(1,NROW(out$t)) %o% out$var.prior
	r <- (stdev.unscaled^2+r) / stdev.unscaled^2
	t2 <- out$t^2
	if(out$df.prior > 10^6)
		kernel <- t2*(1-1/r)/2
	else
		kernel <- (1+df.total)/2*log((t2+df.total) / (t2/r+df.total))
	out$lods <- drop( log(proportion/(1-proportion))-log(r)/2+kernel )
	out
}

tmixture.matrix <- function(tstat,stdev.unscaled,df,proportion,c0lim=NULL) {
#	Estimate the prior variance of the coefficients for DE genes
#	Gordon Smyth
#	18 Nov 2002

	tstat <- as.matrix(tstat)
	stdev.unscaled <- as.matrix(stdev.unscaled)
	if(any(dim(tstat) != dim(stdev.unscaled))) stop("Dims of tstat and stdev.unscaled don't match")
	ncoef <- ncol(tstat)
	c0 <- rep(0,ncoef)
	for (j in 1:ncoef) c0[j] <- tmixture.vector(tstat[,j],stdev.unscaled[,j],df,proportion,c0lim)	
	c0
}

tmixture.vector <- function(tstat,stdev.unscaled,df,proportion,c0lim=NULL) {
#	Estimate scale factor in mixture of two t-distributions
#	tstat is assumed to follow (c0+c1)/c1*t(df) with probability proportion and t(df) otherwise
#	Gordon Smyth
#	18 Nov 2002

	if(any(is.na(tstat))) {
		sel <- !is.na(tstat)
		tstat <- tstat[sel]
		stdev.unscaled <- stdev.unscaled[sel]
		df <- df[sel]
	}
	ngenes <- length(tstat)
	ntarget <- ceiling(proportion/2*ngenes)
	if(ntarget < 1) return(NA)
	tstat <- abs(tstat)
	ttarget <- quantile(tstat,(ngenes-ntarget)/(ngenes-1))
	top <- (tstat >= ttarget)
	tstat <- tstat[top]
	c1 <- stdev.unscaled[top]^2
	df <- df[top]
	r <- ntarget-rank(tstat)+1
	p0 <- pt(-tstat,df=df)
	ptarget <- ( (r-0.5)/2/ngenes - (1-proportion)*p0 ) / proportion
	pos <- ptarget > p0
	c0 <- rep(0,ntarget)
	if(any(pos)) {
		qtarget <- qt(ptarget[pos],df=df[pos])
		c0[pos] <- c1[pos]*((tstat[pos]/qtarget)^2-1)
	}
	if(!is.null(c0lim)) c0 <- pmin(c0,c0lim)
	mean(c0)
}

fitFDist <- function(x,df1) {
#	Moment estimation of the parameters of a scaled F-distribution
#	The first degrees of freedom is given
#	Gordon Smyth
#	8 Sept 2002.  Last revised 12 Apr 2003.

#	Remove missing or infinite values and zero degrees of freedom
	o <- is.finite(x) & is.finite(df1) & (x > 0) & (df1 > 0)
	if(any(!o)) {
		x <- x[o]
		df1 <- df1[o]
	}
	n <- length(x)
	if(n==0) return(list(scale=NA,df2=NA))
	
#	Better to work on with log(F)
	z <- log(x)
	e <- z-digamma(df1/2)+log(df1/2)
	emean <- mean(e)
	evar <- mean(n/(n-1)*(e-emean)^2-trigamma(df1/2))
	if(evar > 0) {
		df2 <- 2*trigammaInverse(evar)
		s20 <- exp(emean+digamma(df2/2)-log(df2/2))
	} else {
		df2 <- Inf
		s20 <- exp(emean)
	}
	list(scale=s20,df2=df2)
}

trigammaInverse <- function(x) {
#	Solve trigamma(y) = x for y
#	Gordon Smyth
#	8 Sept 2002.  Last revised 12 April 2003.

#	Non-numeric or zero length input
	if(!is.numeric(x)) stop("Non-numeric argument to mathematical function")
	if(length(x)==0) return(numeric(0))

#	Treat out-of-range values as special cases
	omit <- is.na(x)
	if(any(omit)) {
		y <- x
		if(any(!omit)) y[!omit] <- Recall(x[!omit])
		return(y)
	}
	omit <- (x < 0)
	if(any(omit)) {
		y <- x
		y[omit] <- NaN
		warning("NaNs produced")
		if(any(!omit)) y[!omit] <- Recall(x[!omit])
		return(y)
	}
	omit <- (x > 1e7)
	if(any(omit)) {
		y <- x
		y[omit] <- 1/sqrt(x[omit])
		if(any(!omit)) y[!omit] <- Recall(x[!omit])
		return(y)
	}
	omit <- (x < 1e-6)
	if(any(omit)) {
		y <- x
		y[omit] <- 1/x[omit]
		if(any(!omit)) y[!omit] <- Recall(x[!omit])
		return(y)
	}

#	Newton's method
#	1/trigamma(y) is convex, nearly linear and strictly > y-0.5,
#	so iteration to solve 1/x = 1/trigamma is monotonically convergent
	y <- 0.5+1/x
	iter <- 0
	repeat {
		iter <- iter+1
		tri <- trigamma(y)
		dif <- tri*(1-tri/x)/tetragamma(y)
		y <- y+dif
		if(max(-dif/y) < 1e-8) break
		if(iter > 50) {
			warning("Iteration limit exceeded")
			break
		}
	}
	y
}

qqt <- function(y,df=Inf,ylim=range(y),main="Student's t Q-Q Plot",xlab="Theoretical Quantiles",ylab="Sample Quantiles",plot.it=TRUE,...)
{
#	Student's t probability plot
#	Gordon Smyth
#	3 Oct 2002

    y <- y[!is.na(y)]
    if(0 == (n <- length(y))) stop("y is empty")
    x <- qt(ppoints(n),df=df)[order(order(y))]
    if (plot.it) plot(x,y,main=main,xlab=xlab,ylab=ylab,ylim=ylim,...)
    invisible(list(x=x,y=y))
}

topTable <- function(fit,coef=1,number=10,genelist=NULL,adjust.method="holm",sort.by="B") {
#	Summary table of top genes, object-orientated version
#	Gordon Smyth
#	4 August 2003

	if(!missing(genelist)) fit$genes <- genelist
	toptable(fit=fit[c("coefficients","stdev.unscaled")],
		coef=coef,
		number=number,
		genelist=fit$genes,
		A=fit$Amean,
		eb=fit[c("t","p.value","lods")],
		adjust.method=adjust.method,
		sort.by=sort.by)
}

toptable <- function(fit,coef=1,number=10,genelist=NULL,A=NULL,eb=NULL,adjust.method="holm",sort.by="B",...) {
#	Summary table of top genes
#	Gordon Smyth
#	21 Nov 2002. Last revised 16 Oct 2003.

	if(is.null(eb)) {
		fit$coefficients <- as.matrix(fit$coef)[,coef]
		fit$stdev.unscaled <- as.matrix(fit$stdev)[,coef]
		eb <- ebayes(fit,...)
		coef <- 1
	}
	M <- as.matrix(fit$coef)[,coef]
	if(is.null(A)) {
		if(sort.by=="A") stop("Cannot sort by A-values as these have not been given")
	} else {
		if(NCOL(A)>1) A <- rowMeans(A,na.rm=TRUE)
	}
	tstat <- as.matrix(eb$t)[,coef]
	P.Value <- as.matrix(eb$p)[,coef]
	B <- as.matrix(eb$lods)[,coef]
	ord <- switch(sort.by,
		M=order(abs(M),decreasing=TRUE),
		A=order(A,decreasing=TRUE),
		P=order(P.Value,decreasing=FALSE),
		p=order(P.Value,decreasing=FALSE),
		T=order(abs(tstat),decreasing=TRUE),
		t=order(abs(tstat),decreasing=TRUE),
		B=order(B,decreasing=TRUE),order(B,decreasing=TRUE))
	top <- ord[1:number]
	i <- is.na(P.Value)
	if(any(i))
		P.Value[!i] <- p.adjust(P.Value[!i],method=adjust.method)
	else
		P.Value <- p.adjust(P.Value,method=adjust.method)
	if(is.null(genelist))
		tab <- data.frame(M=M[top])
	else if(is.null(dim(genelist)))
		tab <- data.frame(Name=I(genelist[top]),M=M[top])
	else
		tab <- data.frame(genelist[top,],M=M[top])
	if(!is.null(A)) tab <- data.frame(tab,A=A[top])
	tab <- data.frame(tab,t=tstat[top],P.Value=P.Value[top],B=B[top])
	rownames(tab) <- as.character(1:length(M))[top]
	tab
}

#  EVALUATION

bwss <- function(x,group) {
#	Within and between group sums of squares
#	Gordon Smyth
#	14 Mar 2002. Revised 13 May 2002.

	is.na.xg <- is.na(x) | is.na(group)
	if(any(is.na.xg)) {
		x <- x[!is.na.xg]
		group <- group[!is.na.xg]
	}
	if(length(x)==0) return(list(bss=NA,wss=NA,bdf=NA,wdf=NA))
	m <- tapply(x,group,mean)
	v <- tapply(x,group,var)
	n <- table(group)
	if(any(n==0)) {
		m <- m[n>0]
		v <- v[n>0]
		n <- n[n>0]
	}
	mm <- sum(n*m)/sum(n)
	wss <- sum((n-1)*v,na.rm=TRUE)
	bss <- sum(n*(m-mm)^2)
	wdf <- sum(n-1)
	bdf <- length(m)-1
	list(bss=bss,wss=wss,bdf=bdf,wdf=wdf)
}

bwss.matrix <- function(x) {
#	Sums of squares between and within columns of a matrix
#	Gordon Smyth
#	16 Mar 2002. Revised 13 May 2002.

	n <- colSums(!is.na(x))
	if(any(n==0)) {
		x <- x[,n>0,drop=FALSE]
		n <- n[n>0]
	}
	nc <- length(n)
	if(nc==0) return(list(bss=NA,wss=NA,bdf=NA,wdf=NA))
	m <- colMeans(x,na.rm=TRUE)
	v <- apply(x,2,var,na.rm=TRUE)
	mm <- sum(n*m)/sum(n)
	wss <- sum((n-1)*v,na.rm=TRUE)
	bss <- sum(n*(m-mm)^2)
	wdf <- sum(n-1)
	bdf <- nc-1
	list(bss=bss,wss=wss,bdf=bdf,wdf=wdf)
}

anova.MAList <- function(object,design=NULL,ndups=2,...) {
#	Analysis of variance, gene x array, for series of replicate arrays
#	Useful for comparing between to within gene variability
#	Gordon Smyth
#	16 March 2002.  Last modified 21 April 2003.

	M <- object$M
	if(!is.null(design)) M <- matvec(M,design)
	bwss.array <- bwss.matrix(M)
	nspots <- dim(M)[1]
	narrays <- dim(M)[2]
	ngenes <- nspots/ndups
	dim(M) <- c(ndups,ngenes*narrays)
	bwss.genearray <- bwss.matrix(M)
	dim(M) <- c(ndups,ngenes,narrays)
	M <- aperm(M,c(1,3,2))
	dim(M) <- c(ndups*narrays,ngenes)
	bwss.gene <- bwss.matrix(M)
	src <- c("Genes","Arrays","Interaction","Duplicates")
	ss <- c(bwss.gene$bss,bwss.array$bss,bwss.genearray$bss-bwss.array$bss-bwss.gene$bss,bwss.genearray$wss)
	df <- c(bwss.gene$bdf,bwss.array$bdf,bwss.genearray$bdf-bwss.array$bdf-bwss.gene$bdf,bwss.genearray$wdf)
	ms <- ss/df
	sig <- ms
	sig[1] <- (ms[1]-ms[3])/ndups/narrays
	sig[2] <- (ms[2]-ms[3])/ndups/ngenes
	sig[3] <- (ms[3]-ms[4])/ndups
	ratio <- c(sig[1]/sum(sig[2:4]),NA,NA,NA)
	table <- data.frame(df, ss, ms, sig, ratio)
	dimnames(table) <- list(src,c("Df","Sum Sq","Mean Sq","Var Comp","Ratio")) 
	structure(table,heading=c("Analysis of Variance Table\n","Between and Within Genes"),class=c("anova","data.frame"))
}
#  PRESENTATION PLOTS

heatDiagram <- function(classification,coef,primary=1,names=NULL,treatments=colnames(coef),limit=NULL,orientation="landscape",cex=1,low="green",high="red",ncolors=123,...) {
#	Heat diagram to display fold changes of genes under different conditions
#	Gordon Smyth
#	27 Oct 2002. Last revised 6 July 2003.

#	Check input
	classification <- as.matrix(classification)
	classification[is.na(classification)] <- 0
	coef <- as.matrix(coef)
	if(!identical(dim(classification),dim(coef))) stop("classification and coef must be the same size")
	nt <- ncol(classification)
	if(is.null(names)) names <- as.character(1:nrow(coef))
	names <- substring(names,1,15)
	if(is.null(treatments)) treatments <- as.character(1:nt)

#	Sort coefficients
	DE <- (abs(classification[,primary]) > 0.5)
	ng <- sum(DE)
	if(ng == 0) {
		warning("Nothing significant to plot")
		return(invisible())
	}
	classification <- classification[DE,,drop=FALSE]
	coef <- coef[DE,,drop=FALSE]
	coef[abs(classification) < 0.5] <- NA
	names <- names[DE]
	ord <- order(coef[,primary],decreasing=TRUE)

#	Truncate coefficients if limit is preset
	if(!is.null(limit))
		if(limit > 0) {
			coef[coef < -limit] <- -limit
			coef[coef > limit] <- limit
		} else
			warning("limit ignored because not positive")

#	Check colours
	if(is.character(low)) low <- col2rgb(low)/255
	if(is.character(high)) high <- col2rgb(high)/255
	col <- rgb( seq(low[1],high[1],len=ncolors), seq(low[2],high[2],len=ncolors), seq(low[3],high[3],len=ncolors) )

#	Heat plot
	coef <- coef[ord,,drop=FALSE]
	names <- names[ord]
	out <- data.frame(Name=names,coef)
	if(orientation=="portrait") {
		coef <- t(coef)
		coef <- coef[,ng:1,drop=FALSE]
	}
	old.par <- par(no.readonly = TRUE)
	on.exit(par(old.par))
	if(orientation=="portrait") {
		par(mar=cex*c(1,1,4,3))
		image(coef,col=col,xaxt="n",yaxt="n",...)
		cext <- cex*min(1,8/nt)
		mtext(paste(" ",treatments,sep=""),side=3,las=2,at=(cext-1)*0.005+(0:(nt-1))/(nt-1),cex=cext)
		cex <- cex*min(1,40/ng)
		mtext(paste(" ",names,sep=""),side=4,las=2,at=(1-cex)*0.005+((ng-1):0)/(ng-1),cex=cex)
	} else {
		par(mar=cex*c(5,6,1,1))
		image(coef,col=col,xaxt="n",yaxt="n",...)
		cext <- cex*min(1,12/nt)
		mtext(paste(treatments," ",sep=""),side=2,las=1,at=(1-cext)*0.005+(0:(nt-1))/(nt-1),cex=cext)
		cex <- cex*min(1,60/ng)
		mtext(paste(names," ",sep=""),side=1,las=2,at=(cex-1)*0.005+(0:(ng-1))/(ng-1),cex=cex)
	}
	invisible(out)
}

heatdiagram <- function(stat,coef,primary=1,names=NULL,treatments=colnames(stat),critical.primary=4,critical.other=3,limit=NULL,orientation="landscape",cex=1,low="green",high="red",ncolors=123,...) {
#	Heat diagram to display fold changes of genes under different conditions
#	Gordon Smyth
#	27 Oct 2002. Last revised 6 Feb 2003.

#	Check input
	stat <- as.matrix(stat)
	coef <- as.matrix(coef)
	if(any(dim(stat) != dim(coef))) stop("stat and coef must be the same size")
	nt <- ncol(stat)
	if(is.null(names)) names <- as.character(1:nrow(stat))
	names <- substring(names,1,15)
	if(is.null(treatments)) treatments <- as.character(1:nt)

#  Sort coefficients
	DE <- (stat[,primary] > critical.primary)
	if(any(is.na(DE))) DE[is.na(DE)] <- FALSE
	ng <- sum(DE)
	if(sum(DE) == 0) {
		warning("Nothing significant to plot")
		return(invisible())
	}
	stat <- stat[DE,,drop=FALSE]
	coef <- coef[DE,,drop=FALSE]
	if(!is.null(names)) names <- names[DE]
	if(critical.other > critical.primary) warning("critical.other greater than critical.primary")
	otherDE <- (stat > critical.other)
	otherDE[,primary] <- TRUE
	coef[!otherDE] <- NA
	ord <- order(coef[,primary],decreasing=TRUE)

#  Check colours
	if(is.character(low)) low <- col2rgb(low)/255
	if(is.character(high)) high <- col2rgb(high)/255
	col <- rgb( seq(low[1],high[1],len=ncolors), seq(low[2],high[2],len=ncolors), seq(low[3],high[3],len=ncolors) )

#  Truncate coefficients if limit is preset
	if(!is.null(limit))
		if(limit > 0) {
			coef[coef < -limit] <- -limit
			coef[coef > limit] <- limit
		} else
			warning("limit ignored because not positive")

#  Heat plot
	coef <- coef[ord,,drop=FALSE]
	names <- names[ord]
	out <- data.frame(Name=names,coef)
	if(orientation=="portrait") {
		coef <- t(coef)
		coef <- coef[,ng:1,drop=FALSE]
	}
	old.par <- par(no.readonly = TRUE)
	on.exit(par(old.par))
	if(orientation=="portrait") {
		par(mar=cex*c(1,1,4,3))
		image(coef,col=col,xaxt="n",yaxt="n",...)
		cext <- cex*min(1,8/nt)
		mtext(paste(" ",treatments,sep=""),side=3,las=2,at=(cext-1)*0.005+(0:(nt-1))/(nt-1),cex=cext)
		cex <- cex*min(1,40/ng)
		mtext(paste(" ",names,sep=""),side=4,las=2,at=(1-cex)*0.005+((ng-1):0)/(ng-1),cex=cex)
	} else {
		par(mar=cex*c(5,6,1,1))
		image(coef,col=col,xaxt="n",yaxt="n",...)
		cext <- cex*min(1,12/nt)
		mtext(paste(treatments," ",sep=""),side=2,las=1,at=(1-cext)*0.005+(0:(nt-1))/(nt-1),cex=cext)
		cex <- cex*min(1,60/ng)
		mtext(paste(names," ",sep=""),side=1,las=2,at=(cex-1)*0.005+(0:(ng-1))/(ng-1),cex=cex)
	}
	invisible(out)
}

#	INPUT.R

#	GAL FILES

readGAL <- function(galfile=NULL,path=NULL,header=TRUE,sep="\t",quote="\"",skip=NULL,as.is=TRUE,...) {
#	Read GenePix Allocation List (GAL) file
#	Gordon Smyth
#	1 Mar 2003.  Last revised 16 Sept 2003.

	if(is.null(galfile)) {
		if(is.null(path)) path <- "."
		galfile <- dir(path=path,pattern="*\\.gal$")
		nfiles <- length(galfile)
		if(nfiles == 0) stop("Cannot find GAL file")
		if(nfiles > 1) {
			galfile <- galfile[1]
			warning(paste("More than one GAL file found. Reading",galfile))
		}
	}
	if(!is.null(path)) galfile <- file.path(path,galfile)
	if(is.null(skip)) {
		chunk <- readLines(galfile,n=100)
		skip <- intersect(grep("Name",chunk), grep("ID",chunk)) - 1
		n <- length(skip)
		if(n == 0) stop("Cannot find ID and Name columns in GAL file")
		if(n > 1) stop("Multiple lines with ID and Name labels")
	}
	read.table(galfile,header=header,sep=sep,quote=quote,skip=skip,as.is=as.is,comment.char="",...)
}

splitName <- function(x, split=";", extended=TRUE) {
#	Split composite gene names into short names and annotation information
#	Gordon Smyth
#	8 May 2003.  Last modified 29 May 2003.

	s <- strsplit(x,split,extended)
	function1 <- function(x) {
		n <- length(x)
		if(n > 0)
			if(n > 2) paste(x[1:(n-1)],collapse=split) else x[1]
		else
			""
	}
	function2 <- function(x) {
		n <- length(x)
		if(n > 1) x[n] else ""
	}
	list(Name=unlist(lapply(s,function1)), Annotation=unlist(lapply(s,function2)))
}

getLayout <- function(gal) {
#	Guess print layout from a GenePix Allocation List (GAL)
#	Gordon Smyth
#	7 Apr 2003.  Last revised 8 May 2003.

	n <- nrow(gal)
	lapply(list(ngrid.r=gal$Block[n]/4,ngrid.c=4,nspot.r=gal$Row[n],nspot.c=gal$Column[n]),as.integer)
}

readTargets <- function(file="Targets.txt",sep="\t")
#	Data frame of target information
#	Gordon Smyth
#	19 Oct 2003
{
	tab <- read.table(file,header=TRUE,as.is=TRUE,sep=sep,quote="\"")
	if(all(c("Cy3","Cy5") %in% names(tab)))
		return(tab)
	else
		stop("File should contain columns: Cy3 and Cy5")
}

readSpotTypes <- function(file="SpotTypes.txt",sep="\t")
#	Read regexp for spot types
#	Gordon Smyth following idea of James Wettenhall
#	19 Oct 2003
{
	tab <- read.table(file,header=TRUE,as.is=TRUE,sep=sep,quote="\"")
	if(all(c("SpotType","ID","Name","Color") %in% names(tab)))
		return(tab)
	else
		stop("File should contain columns: SpotType, ID, Name and Color")
}

controlStatus <- function(types, genes)
#	Set status of each spot
#	Gordon Smyth
#	19 Oct 2003.  Last modified 21 Oct 2003.
{
	if(is(genes,"RGList") || is(genes,"MAList")) genes <- genes$genes
	status <- character(nrow(genes))
	for (i in 1:nrow(types)) {
		sel <- intersect(grep(types$ID[i],genes$ID),grep(types$Name[i],genes$Name))
		status[sel] <- types$SpotType[i]
	}
	status
}

#	READ IMAGE ANALYSIS FILES INTO RG LIST

read.matrix <- function(file,nrows=0,skip=0,...) {
#	Read numeric matrix with headers from file
#	Gordon Smyth
#	9 Mar 2003.

	h <- scan(file,what="character",skip=skip,nlines=1,quote="\"",quiet=TRUE,...)
	x <- matrix(scan(file,skip=skip+1,nlines=nrows,quiet=TRUE,...),byrow=TRUE,ncol=length(h))
	colnames(x) <- h
	x
}

read.maimages <- function(files,source="spot",path=NULL,ext=NULL,names=NULL,columns=NULL,wt.fun=NULL,verbose=TRUE,sep="\t",quote="\"",...) {
#	Extracts an RG list from a series of image analysis output files
#	Gordon Smyth
#	1 Nov 2002.  Last revised 27 Sep 2003.

	if(missing(files)) {
		if(missing(ext))
			stop("Must specify input files")
		else {
			extregex <- paste("\\.",ext,"$",sep="")
			files <- dir(path=ifelse(is.null(path),".",path),pattern=extregex)
			files <- sub(extregex,"",files)
		}
	}
	if(!missing(source) && !missing(columns)) stop("Cannot specify both source and columns")
	source <- match.arg(source,c("arrayvision","genepix","imagene","quantarray","smd","spot","spot.close.open"))
	if(source=="imagene") return(read.imagene(files=files,path=path,ext=ext,names=names,columns=columns,wt.fun=wt.fun,verbose=verbose,sep=sep,quote=quote,...))
	slides <- as.vector(as.character(files))
	if(!is.null(ext)) slides <- paste(slides,ext,sep=".")
	nslides <- length(slides)
	if(is.null(names)) names <- removeExt(files)

	if(is.null(columns)) columns <- switch(source,
		smd = list(Gf="CH1I_MEAN",Gb="CH1B_MEDIAN",Rf="CH2I_MEAN",Rb="CH2B_MEDIAN"),
		spot = list(Rf="Rmean",Gf="Gmean",Rb="morphR",Gb="morphG"),
		spot.close.open = list(Rf="Rmean",Gf="Gmean",Rb="morphR.close.open",Gb="morphG.close.open"),
		genepix = list(Rf="F635 Mean",Gf="F532 Mean",Rb="B635 Median",Gb="B532 Median"),
		quantarray = list(Rf="ch2 Intensity",Gf="ch1 Intensity",Rb="ch2 Background",Gb="ch1 Background")
	)

#	Read first file to get nspots
	fullname <- slides[1]
	if(!is.null(path)) fullname <- file.path(path,fullname)
	if(source=="quantarray") {
		firstfield <- scan(fullname,what="",sep="\t",flush=TRUE,quiet=TRUE,blank.lines.skip=FALSE,multi.line=FALSE)
		skip <- grep("Begin Data",firstfield)
		if(length(skip)==0) stop("Cannot find \"Begin Data\" in image output file")
		nspots <- grep("End Data",firstfield) - skip -2
		obj <- read.table(fullname,skip=skip,header=TRUE,sep=sep,quote=quote,as.is=TRUE,check.names=FALSE,comment.char="",nrows=nspots,...)
	} else if(source=="arrayvision") {
		skip <- 1
		cn <- scan(fullname,what="",sep=sep,quote=quote,skip=1,nlines=1,quiet=TRUE)
		fg <- grep("^Median Dens - RFU",cn)
		if(length(fg) != 2) stop(paste("Cannot find foreground columns in",fullname))
		bg <- grep("Bkgd",cn)
		if(length(fg) != 2) stop(paste("Cannot find background columns in",fullname))
		columns <- list(Rf=fg[1],Rb=bg[1],Gf=fg[2],Gb=bg[2])
		obj <- read.table(fullname,skip=skip,header=TRUE,sep=sep,quote=quote,as.is=TRUE,check.names=FALSE,comment.char="",...)
		nspots <- nrow(obj)
	} else {
		skip <- grep(columns$Rf,readLines(fullname,n=80)) - 1
		if(length(skip)==0)
			stop("Cannot find column heading in image output file")
		else
			skip <- skip[1]
		obj <- read.table(fullname,skip=skip,header=TRUE,sep=sep,quote=quote,as.is=TRUE,check.names=FALSE,comment.char="",...)
		nspots <- nrow(obj)
	}

#	Now read rest
	Y <- matrix(0,nspots,nslides)
	colnames(Y) <- names
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y)
	if(source=="smd") {
		anncol <- grep(columns$Gf,colnames(obj))-1
		if(anncol>0) RG$genes <- data.frame(obj[,1:anncol])
	}
	if(!is.null(wt.fun)) RG$weights <- Y
	for (i in 1:nslides) {
		if(i > 1) {
			fullname <- slides[i]
			if(!is.null(path)) fullname <- file.path(path,fullname)
			obj <- read.table(fullname,skip=skip,header=TRUE,sep=sep,as.is=TRUE,quote=quote,check.names=FALSE,comment.char="",nrows=nspots,...)
		}
		RG$R[,i] <- obj[,columns$Rf]
		RG$G[,i] <- obj[,columns$Gf]
		RG$Rb[,i] <- obj[,columns$Rb]
		RG$Gb[,i] <- obj[,columns$Gb]
		if(!is.null(wt.fun)) RG$weights[,i] <- wt.fun(obj)
		if(verbose) cat(paste("Read",fullname,"\n"))
	}
	new("RGList",RG)
}

read.imagene <- function(files,path=NULL,ext=NULL,names=NULL,columns=NULL,wt.fun=NULL,verbose=TRUE,sep="\t",quote="\"",...) {
#	Extracts an RG list from a series of Imagene analysis output files.
#	Imagene requires special treatment because red and green channel
#	intensities are in different files.
#	Gordon Smyth
#	14 Aug 2003.  Last modified 2 Oct 2003.

	files <- as.matrix(files)
	if(ncol(files) != 2) stop("Need a two column matrix of file names")
	if(!is.null(ext)) files <- array(paste(files,ext,sep="."),dim(files))
	narrays <- nrow(files)
	if(is.null(columns)) columns <- list(f="Signal Mean",b="Background Median")
	if(is.null(columns$f) || is.null(columns$b)) stop("'columns' should have components 'f' and 'b'")
	if(is.null(names)) names <- removeExt(files[,1])

#	Read header information from first file to get nspots
	fullname <- files[1,1]
	if(!is.null(path)) fullname <- file.path(path,fullname)
	headers <- readImageneHeaders(fullname)
	if(verbose) cat("Read header information\n")
	skip <- headers$Begin.Raw.Data
	printer <- headers$Field.Dimensions[c("Metarows","Metacols","Rows","Cols")]
	names(printer) <- c("ngrid.r","ngrid.c","nspot.r","nspot.c")
	if(length(printer) != 4) stop("Cannot read field dimension information")
	nspots <- prod(unlist(printer))

#	Now read data
	Y <- matrix(0,nspots,narrays)
	colnames(Y) <- names
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y,printer=printer)
	if(!is.null(wt.fun)) RG$weights <- Y
	for (i in 1:narrays) {
		fullname <- files[i,1]
		if(!is.null(path)) fullname <- file.path(path,fullname)
		if(i > 1) {
			headers <- readImageneHeaders(fullname)
			if(any(unlist(printer) != unlist(headers$Field.Dimensions[c("Metarows","Metacols","Rows","Cols")])))
				stop(paste("Field dimensions of array",i,"not same as those of first array"))
			skip <- headers$Begin.Raw.Data
		}
		obj<- read.table(fullname,skip=skip,header=TRUE,sep=sep,quote=quote,check.names=FALSE,comment.char="",nrows=nspots,...)
		if(verbose) cat(paste("Read",fullname,"\n"))
		if(i==1) RG$genes <- obj[,c("Field","Meta Row","Meta Column","Row","Column","Gene ID")]
		RG$G[,i] <- obj[,columns$f]
		RG$Gb[,i] <- obj[,columns$b]
		fullname <- files[i,2]
		if(!is.null(path)) fullname <- file.path(path,fullname)
		obj<- read.table(fullname,skip=skip,header=TRUE,sep=sep,quote=quote,check.names=FALSE,comment.char="",nrows=nspots,...)
		if(verbose) cat(paste("Read",fullname,"\n"))
		RG$R[,i] <- obj[,columns$f]
		RG$Rb[,i] <- obj[,columns$b]
		if(!is.null(wt.fun)) RG$weights[,i] <- wt.fun(obj)
	}
	new("RGList",RG)
}

readGPRHeaders <- function(file) {
#	Extracts header information from a GenePix Results (GPR) file
#	Gordon Smyth
#	4 October 2003.

	con <- file(file, "r")	
	if(substring(readLines(con,n=1),1,3) != "ATF") stop("File is not in Axon Text File (ATF) format")
	nfields <- as.numeric(strsplit(readLines(con,n=1),split="\t")[[1]])
	out <- list()
	for (i in 1:nfields[1]) {
		txt <- strsplit(sub("\"$","",sub("^\"","",readLines(con,n=1))),split="=")[[1]]
		out[[i]] <- txt[2]
		names(out)[i] <- txt[1]
	}
	close(con)
	out$Wavelengths <- strsplit(out$Wavelengths,split="\t")[[1]]
	out$ImageFiles <- strsplit(out$ImageFiles,split="\t")[[1]]
	out
}

readImageneHeaders <- function(file) {
#	Extracts header information from an Imagene analysis output file
#	Gordon Smyth
#	14 Aug 2003.

	firstfield <- scan(file,what="",sep="\t",quote="\"",nlines=60,flush=TRUE,quiet=TRUE,blank.lines.skip=FALSE,multi.line=FALSE)
	beginrawdata <- grep("Begin Raw Data",firstfield)
	txt <- scan(file,what="",sep="\t",quote="\"",nlines=beginrawdata-1,quiet=TRUE)
	out <- list(Begin.Raw.Data=beginrawdata)
	out$Version <- txt[grep("^version$",txt)+1]
	out$Date <- txt[grep("^Date$",txt)+1]
	out$Image.File <- txt[grep("^Image File$",txt)+1]
	out$Inverted <- as.logical(txt[grep("^Inverted$",txt)+1])
	out$Field.Dimensions <- list()
	out$Field.Dimensions$Field <- txt[grep("^Field$",txt)+7]
	out$Field.Dimensions$Metarows <- as.integer(txt[grep("^Metarows$",txt)+7])
	out$Field.Dimensions$Metacols <- as.integer(txt[grep("^Metacols$",txt)+7])
	out$Field.Dimensions$Rows <- as.integer(txt[grep("^Rows$",txt)+7])
	out$Field.Dimensions$Cols <- as.integer(txt[grep("^Cols$",txt)+7])
	out$Measurement.Parameters <- list()
	out$Measurement.Parameters$Signal.Low <- txt[grep("^Signal Low$",txt)+1]
	out$Measurement.Parameters$Signal.High <- txt[grep("^Signal High$",txt)+1]
	out$Measurement.Parameters$Background.Low <- txt[grep("^Background Low$",txt)+1]
	out$Measurement.Parameters$Background.High <- txt[grep("^Background High$",txt)+1]
	out$Measurement.Parameters$Background.Buffer <- txt[grep("^Background Buffer$",txt)+1]
	out$Measurement.Parameters$Background.Width <- txt[grep("^Background Width$",txt)+1]
	out
}

rg.series.spot <- function(slides,path=NULL,names.slides=names(slides),suffix="spot",wt.fun=NULL,verbose=TRUE,...) {
#	Extracts an RG list from a series of Spot image analysis files
#	Gordon Smyth
#	1 Nov 2002.  Last revised 23 Mar 2003.

	slides <- as.vector(as.character(slides))
	if(is.null(names.slides)) names.slides <- slides
	nslides <- length(slides)
	if(!is.null(suffix)) slides <- paste(slides,suffix,sep=".")

#	Read first file to get nspots
	fullname <- slides[1]
	if(!is.null(path)) fullname <- file.path(path,fullname)
	obj <- read.matrix(fullname,...)
	nspots <- nrow(obj)

#	Now read rest
	Y <- matrix(0,nspots,nslides)
	colnames(Y) <- names.slides
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y)
	if(!is.null(wt.fun)) RG$weights <- Y
	for (i in 1:nslides) {
		if(i > 1) {
			fullname <- slides[i]
			if(!is.null(path)) fullname <- file.path(path,fullname)
			obj <- read.matrix(fullname,nrows=nspots,...)
		}
		RG$R[,i] <- obj[,"Rmean"]
		RG$G[,i] <- obj[,"Gmean"]
		RG$Rb[,i] <- obj[,"morphR"]
		RG$Gb[,i] <- obj[,"morphG"]
		if(!is.null(wt.fun)) RG$weights[,i] <- wt.fun(obj)
		if(verbose && interactive()) cat(paste("Read",fullname,"\n"))
	}
	RG
}

#	READ COMPLETE IMAGE ANALYSIS OUTPUT FILES INTO DATA FRAMES

read.series <- function(slides,path=NULL,suffix="spot",...) {
#	Read in a series of image analysis files
#	Gordon Smyth
#	11 Mar 2002.  Last revised 2 Mar 2003.

	slides <- as.vector(as.character(slides))
	nslides <- length(slides)
	for (i in 1:nslides) {
		slidename <- slides[i]
		if(!is.null(suffix)) slidename <- paste(slidename,suffix,sep=".")
		fullname <- slidename
		if(!is.null(path)) fullname <- file.path(path,slidename)
		assign(slidename,read.table(fullname,header=TRUE,...),pos=1)
	}
	return(paste(as.character(nslides),"slides read"))
}

#	EXTRACT COLUMNS OF INTEREST FROM COMPLETE DATA FRAMES

m.spot <- function(spot) {
#	Extracts log-ratio M from a Spot image analysis dataframe
#	Gordon Smyth
#	18 Nov 2001.  Last revised 2 Mar 2003.

	R <- spot[,"Rmean"] - spot[,"morphR"]
	G <- spot[,"Gmean"] - spot[,"morphG"]
	log(R,2) - log(G,2)
}

a.spot <- function(spot) {
#	Extracts intensity A from a Spot image analysis dataframe
#	Gordon Smyth
#	18 Nov 2001.  Last revised 2 Mar 2003.

	R <- spot[,"Rmean"] - spot[,"morphR"]
	G <- spot[,"Gmean"] - spot[,"morphG"]
	(log(R,2) + log(G,2))/2
}

rg.spot <- function(slides,names.slides=names(slides),suffix="spot",area=FALSE) {
#	Extract RG list from a series of Spot dataframes
#	Gordon Smyth
#	17 Jan 2002. Last revised 1 Nov 2002.

	slides <- as.vector(as.character(slides))
	if(is.null(names.slides)) names.slides <- slides
	nslides <- length(slides)
	if(!is.null(suffix)) slides <- paste(slides,suffix,sep=".")
	ngenes <- dim(get(slides[1]))[1]
	Y <- matrix(0,ngenes,nslides)
	colnames(Y) <- names.slides
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y)
	if(area) RG$Area <- Y
	for (i in 1:nslides) {
		RG$R[,i] <- get(slides[i])[,"Rmean"]
		RG$G[,i] <- get(slides[i])[,"Gmean"]
		RG$Rb[,i] <- get(slides[i])[,"morphR"]
		RG$Gb[,i] <- get(slides[i])[,"morphG"]
		if(area) RG$Area[,i] <- get(slides[i])[,"area"]
	}
	RG
}

rg.quantarray <- function(slides,names.slides=names(slides),suffix="qta") {
#	Extract RG list from a series of Quantarray dataframes
#	Gordon Smyth
#	23 July 2002

	slides <- as.vector(as.character(slides))
	if(is.null(names.slides)) names.slides <- slides
	nslides <- length(slides)
	if(!is.null(suffix)) for (i in 1:nslides) slides[i] <- paste(slides[i],suffix,sep=".")
	ngenes <- dim(get(slides[1]))[1]
	Y <- matrix(0,ngenes,nslides)
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y)
	for (i in 1:nslides) {
		RG$R[,i] <- get(slides[i])$ch2.Intensity
		RG$G[,i] <- get(slides[i])$ch1.Intensity
		RG$Rb[,i] <- get(slides[i])$ch2.Background
		RG$Gb[,i] <- get(slides[i])$ch1.Background
	}
	colnames(RG$R) <- colnames(RG$G) <- colnames(RG$Rb) <- colnames(RG$Gb) <- names.slides
	RG
}

rg.genepix <- function(slides,names.slides=names(slides),suffix="gpr") {
#	Extract RG list from a series of Genepix dataframes
#	Gordon Smyth
#	23 July 2002. Last revised 13 Feb 2003.

	slides <- as.vector(as.character(slides))
	if(is.null(names.slides)) names.slides <- slides
	nslides <- length(slides)
	if(!is.null(suffix)) for (i in 1:nslides) slides[i] <- paste(slides[i],suffix,sep=".")
	ngenes <- dim(get(slides[1]))[1]
	Y <- matrix(0,ngenes,nslides)
	RG <- list(R=Y,G=Y,Rb=Y,Gb=Y)
	for (i in 1:nslides) {
		RG$R[,i] <- get(slides[i])$F635.Mean
		RG$G[,i] <- get(slides[i])$F532.Mean
		RG$Rb[,i] <- get(slides[i])$B635.Median
		RG$Gb[,i] <- get(slides[i])$B532.Median
	}
	colnames(RG$R) <- colnames(RG$G) <- colnames(RG$Rb) <- colnames(RG$Gb) <- names.slides
	RG
}

removeExt <- function(x) {
#	Remove any common extension from a vector of file names
#	Gordon Smyth
#	19 July 2002.

	n <- length(x)
	if(length(grep("\\.",x)) < n) return(x)
	ext <- sub("(.*)\\.(.*)$","\\2",x)
	if(all(ext[1] == ext))
		return(sub("(.*)\\.(.*)$","\\1",x))
	else
		return(x)
}

#	LAYOUT FUNCTIONS

spotc <- function(layout) {
#	GKS  24 Nov 2002
	rep(1:layout$nspot.c, length=prod(unlist(layout)))
}

spotr <- function(layout) {
#	GKS  24 Nov 2002
	rep(1:layout$nspot.r, times=layout$ngrid.r*layout$ngrid.c, each=layout$nspot.c)
}

gridc <- function(layout) {
#	GKS  24 Nov 2002
	rep(1:layout$ngrid.c, times=layout$ngrid.r, each=layout$nspot.c*layout$nspot.r)
}

gridr <- function(layout) {
#	GKS  24 Nov 2002
	rep(1:layout$ngrid.r, each=layout$nspot.c*layout$nspot.r*layout$ngrid.c)
}

printorder <- function(layout, ndups=1, npins=layout$ngrid.r*layout$ngrid.c, start="topleft") {
#	Identify order in which spots were printed
#	Gordon Smyth
#	2 May 2003.  Last revised 2 July 2003.

	if(is(layout,"RGList") || is(layout,"MAList")) {
		if(missing(ndups) && !is.null(layout$printer$ndups)) ndups <- layout$printer$ndups
		if(missing(npins) && !is.null(layout$printer$npins)) npins <- layout$printer$npins
		if(missing(start) && !is.null(layout$printer$start)) start <- layout$printer$start
		layout <- layout$printer
	}

#	All input values should be integers
#	DNA plates assumed to have 384 wells
	nwell.r <- as.integer(16)
	nwell.c <- as.integer(24)
	nwells <- nwell.r * nwell.c
	ngrid.r <- layout$ngrid.r
	ngrid.c <- layout$ngrid.c
	nspot.r <- layout$nspot.r
	nspot.c <- layout$nspot.c
#	Number of grid blocks
	ngrids <- ngrid.r * ngrid.c
#	Number of spots per grid
	nspots <- nspot.r * nspot.c
	ndups <- round(ndups)
	npins <- round(npins)
#	Pin columns assumed to be same an grid columns
	npin.c <- ngrid.c
	npin.r <- npins %/% npin.c
#	Number of times print head dips into each plate
	ndips <- ndups * nwells %/% npins

#	First do everything for subarray with ngrids=npins

	spot.c <- rep(1:nspot.c, times = npins * nspot.r)
	spot.r <- rep(1:nspot.r, times = npins, each = nspot.c)
	pin.c <- rep(1:npin.c, times = npin.r, each = nspots)
	pin.r <- rep(1:npin.r, each = npin.c * nspots)

#	Printorder
	start <- match.arg(start,c("topright","topleft"))
	po <- switch(start,
		topleft=nspot.c*(spot.r-1)+spot.c,
		topright=nspot.c*(spot.r-1)+nspot.c-spot.c+1
	)

#	Now repeat subarrays if fewer pins than actual grids

	if(ngrids %% npins != 0) stop("ngrids not a multiple of npins")
	m <- ngrids %/% npins
	if(m > 1) po <- rep(po,m) + nspots * rep(0:(m-1), each = npins * nspots)

#	Plates
	plate <- 1 + (po-1) %/% ndips

#	Position within plate
	platedip <- 1 + (po-1) %% ndips
	plateblock <- 1 + (platedip-1) %/% ndups
	plateblock.r <- 1 + (plateblock-1) %/% (nwell.c %/% npin.r)
	plateblock.c <- 1 + (plateblock-1) %% (nwell.c %/% npin.r)
	plate.r <- (plateblock.r-1) * npin.c + npin.c - pin.c + 1
	plate.c <- (plateblock.c-1) * npin.r + pin.r
	platedigits <- 1+floor(log(ngrids*nspots/ndups/nwells,10))
	platepos <- paste("p",formatC(plate, width=platedigits, flag = "0"), LETTERS[plate.r], formatC(plate.c, width=2, flag = "0"), sep="")

	list(printorder=po, plate=plate, plate.r=plate.r, plate.c=plate.c, plateposition=platepos)
}

#  KOOPERBERG BACKGROUND ADUSTMENT FOR GENEPIX DATA

kooperberg <- function(names, fg="mean", bg="median", a=TRUE, layout)
#       Kooperberg Bayesian background correction
#       Matt Ritchie 18 June 2003.
#       Modifications by Gordon Smyth 16 June 2003.
#       Charles Kooperberg contributed 'a' estimation functions (.getas, .varaux1, .varaux2)

        {
        choices <- c("mean","median")
        fg <- choices[pmatch(fg,choices)]
        bg <- choices[pmatch(bg,choices)]
        meanfg <- (fg=="mean")
        meanbg <- (bg=="mean")
        nslides <- length(names)
        ngenes <- layout$ngrid.c*layout$ngrid.r*layout$nspot.r*layout$nspot.c
        Y <- matrix(0,ngenes,nslides)
        RG <- list(R=Y,G=Y)
        for(i in 1:nslides)
                {
                temp <- .bayesianAdjustedFG(get(names[i]), meanfg, meanbg, a, layout)
                RG$R[,i] <- temp$R
                RG$G[,i] <- temp$G
                }
        RG
        }

.bayesianAdjustedFG <- function(slide, meanfg=TRUE, meanbg=FALSE, a, layout) {
        ngenes <- dim(slide)[1]
        Y <- rep(0, ngenes)
        RG <- list(R=Y, G=Y)
        if(meanfg) {
          Rfg <- slide[,"F635.Mean"] 
          Gfg <- slide[,"F532.Mean"] 
        } else {
          Rfg <- slide[,"F635.Median"] 
          Gfg <- slide[,"F532.Median"] 
        }
        if(meanbg) {
          Rbg <- slide[,"B635.Mean"] 
          Gbg <- slide[,"B532.Mean"] 
          } else {
          Rbg <- slide[,"B635.Median"] 
          Gbg <- slide[,"B532.Median"] 
          }
        if(a) { 
          aparams <- .getas(slide, layout)
          } else {
          aparams <- c(1,1)
          }
        Rsfg=aparams[2]*slide[,"F635.SD"]/sqrt(slide[,"F.Pixels"]) # column 20
        Rsbg=aparams[2]*slide[,"B635.SD"]/sqrt(slide[,"B.Pixels"]) # column 23
        Gsfg=aparams[1]*slide[,"F532.SD"]/sqrt(slide[,"F.Pixels"]) # column 11
        Gsbg=aparams[1]*slide[,"B532.SD"]/sqrt(slide[,"B.Pixels"]) # column 14
        for(i in 1:ngenes) { 
              RG$R[i] <- .expectedBayesianAdjustedFG(fg=Rfg[i], bg=Rbg[i], sfg=Rsfg[i], sbg=Rsbg[i])
              RG$G[i] <- .expectedBayesianAdjustedFG(fg=Gfg[i], bg=Gbg[i], sfg=Gsfg[i], sbg=Gsbg[i])
              }
        RG$R[RG$R>2^16] <- NA
        RG$G[RG$G>2^16] <- NA
        RG
      }

.expectedBayesianAdjustedFG <- function(fg, bg, sfg, sbg) {
        integrate(.numeratorBayesianAdjustedFG, ifelse((fg-bg-4*sqrt(sbg^2+sfg^2))<0, 0, fg-bg-4*sqrt(sbg^2+sfg^2)), 
                ifelse((fg-bg+4*sqrt(sfg^2+sbg^2))<0, 1000, fg-bg+4*sqrt(sfg^2+sbg^2)) , fg=fg, bg=bg, sfg=sfg, sbg=sbg, subdivisions=10000)$value/.denominatorBayesianAdjustedFG(fg, bg, sfg, sbg)
        }

.numeratorBayesianAdjustedFG <- function(ut, fg, bg, sfg, sbg)
        ut*exp(dnorm((fg-ut-bg)/sqrt(sfg^2+sbg^2), log=TRUE)+pnorm(((fg-ut)*sbg^2+bg*sfg^2)/(sbg*sfg*sqrt(sfg^2+sbg^2)), log.p=TRUE))

.denominatorBayesianAdjustedFG <- function(fg, bg, sfg, sbg)    {
        sqrt(sfg^2+sbg^2) / sbg * integrate(.normalConvolution,
        ifelse((bg-4*sbg)<0, 0, bg-4*sbg),
        bg+4*sbg, fg=fg, bg=bg, sfg=sfg,
        sbg=sbg, subdivisions=10000)$value
}

.normalConvolution <- function(v, fg, bg, sfg, sbg)
        exp(pnorm((fg-v)/sfg, log.p=TRUE)+dnorm((bg-v)/sbg, log=TRUE))

.getas <- function(x, layout)
{
    b1 <- .varaux1(x,"B532.Mean", layout) # bg G median
    b2 <- .varaux1(x,"B635.Mean", layout) # bg R median
    c1 <- x[, "B532.SD"]/sqrt(x[, "B.Pixels"])
    c2 <- x[, "B635.SD"]/sqrt(x[, "B.Pixels"])
    m1 <- lm(b1 ~ c1 - 1, weights = 1/(c1 + 1))
    m2 <- lm(b2 ~ c2 - 1, weights = 1/(c2 + 1))
    c(m1$coef,m2$coef)
}

# Function which calculates the empirical standard deviation for each spot (based on average of spot and 4 neighbours
.varaux1 <- function(x, j, layout)
{
    numblocks <- layout$ngrid.c*layout$ngrid.r
    ncols <- layout$nspot.c
    nrows <- layout$nspot.r
    uu <- .varaux2(x, 1, j, ncols, nrows)
    if(numblocks>1) {
            for(i in 2:numblocks) {
                uu <- c(uu, .varaux2(x, i, j, ncols, nrows))
               }
        }
    uu
}

# Function which averages the standard deviations
.varaux2 <- function(x, i, j, ncols, nrows)
   {
    v1 <- x[x[, 1] == i, j]
    v2 <- matrix(v1, ncol=ncols)

# mid grid spot variances
    v4a <- v2[c(-1, -nrows), c(-1, -ncols)]
    v4b <- v2[c(-1, -2), c(-1, -ncols)]
    v4c <- v2[c(-1, -nrows), c(-1, -2)]
    v4d <- v2[c(-(nrows-1), -nrows), c(-1, -ncols)]
    v4e <- v2[c(-1, -nrows), c(-(ncols-1), -ncols)]
    v4x <- cbind(as.vector(v4a), as.vector(v4b), as.vector(v4c),
                 as.vector(v4d), as.vector(v4e))
    VAR <- matrix(0, ncol=ncols, nrow=nrows)
    mid.var <- apply(v4x, 1, FUN=var)
    VAR[2:(nrows-1), 2:(ncols-1)] <- sqrt(mid.var)


# edge spot variances    
# top
    v4a <- v2[1, c(-1, -ncols)]
    v4b <- v2[1, c(-(ncols-1), -ncols)]
    v4c <- v2[2, c(-1, -ncols)]
    v4d <- v2[1, c(-1, -2)]
    v4x <- cbind(as.vector(v4a), as.vector(v4b), as.vector(v4c),
                 as.vector(v4d))
    edge <- apply(v4x, 1, FUN=var)
    VAR[1, 2:(ncols-1)] <- sqrt(edge)

# bottom
    v4a <- v2[nrows, c(-1, -ncols)]
    v4b <- v2[nrows, c(-(ncols-1), -ncols)]
    v4c <- v2[nrows-1, c(-1, -ncols)]
    v4d <- v2[nrows, c(-1, -2)]
    v4x <- cbind(as.vector(v4a), as.vector(v4b), as.vector(v4c),
                 as.vector(v4d))
    
    edge <- apply(v4x, 1, FUN=var)
    VAR[nrows, 2:(ncols-1)] <- sqrt(edge)
    
 # left
    v4a <- v2[c(-1, -nrows), 1]
    v4b <- v2[c(-(nrows-1), -nrows), 1]
    v4c <- v2[c(-1, -nrows), 2]
    v4d <- v2[c(-1, -2), 1]
    v4x <- cbind(as.vector(v4a), as.vector(v4b), as.vector(v4c),
                 as.vector(v4d))
    
    edge <- apply(v4x, 1, FUN=var)
    VAR[2:(nrows-1), 1] <- sqrt(edge)   
   
 # right
    v4a <- v2[c(-1, -nrows), ncols]
    v4b <- v2[c(-(nrows-1), -nrows), ncols]
    v4c <- v2[c(-1, -nrows), ncols-1]
    v4d <- v2[c(-1, -2), ncols]
    v4x <- cbind(as.vector(v4a), as.vector(v4b), as.vector(v4c),
                 as.vector(v4d))
    
    edge <- apply(v4x, 1, FUN=var)
    VAR[2:(nrows-1), ncols] <- sqrt(edge)   
    
 # corners
     v4x <- cbind(c(v2[1,1], v2[1,ncols], v2[nrows,1], v2[nrows, ncols]),
                  c(v2[1,2], v2[1,ncols-1], v2[nrows,2], v2[nrows, ncols-1]),
                  c(v2[2,1], v2[2,ncols], v2[nrows-1,1], v2[nrows-1, ncols]),
                  c(v2[2,2], v2[2,ncols-1], v2[nrows-1,2], v2[nrows-1, ncols-1]))
     
     corner <- apply(v4x, 1, FUN=var)
     VAR[1,1] <- sqrt(corner[1])
     VAR[1, ncols] <- sqrt(corner[2])
     VAR[nrows, 1] <- sqrt(corner[3])
     VAR[nrows,ncols] <- sqrt(corner[4])
     as.vector(VAR)
}
#  LINEAR MODELS

lmFit <- function(object,design=NULL,ndups=1,spacing=1,correlation=0.75,weights=NULL,method="ls",...) {
#	Fit linear model
#	Gordon Smyth
#	30 June 2003.  Last modified 27 October 2003.

	M <- NULL
	if(is(object,"marrayNorm")) {
#		don't use accessor function so don't have to require marrayClasses
		M <- object@maM
		if(missing(weights) && length(object@maW)) weights <- object@maW
	}
	if(is(object,"exprSet")) {
#		don't use accessor function so don't have to require Biobase
		M <- object@exprs
#		don't use weights until this is more thoroughly tested
#		if(missing(weights) && length(object@se.exprs)) weights <- 1/pmax(object@se.exprs,1e-5)^2
	}
#	Method intended for MAList objects but allow unclassed lists as well
	if(is.list(object)) {
		M <- object$M
		if(missing(design) && !is.null(object$design)) design <- object$design
		if(missing(ndups) && !is.null(object$printer$ndups)) ndups <- object$printer$ndups
		if(missing(spacing) && !is.null(object$printer$spacing)) spacing <- object$printer$spacing
		if(missing(correlation) && !is.null(object$correlation)) correlation <- object$correlation
		if(missing(weights) && !is.null(object$weights)) weights <- object$weights
	}
#	Default method
	if(is.null(M)) M <- as.matrix(object)

	if(is.null(design)) design <- matrix(1,ncol(M),1)
	design <- as.matrix(design)
	method <- match.arg(method,c("ls","robust"))
	if(method=="robust")
		fit <- rlm.series(M,design=design,ndups=ndups,spacing=spacing,weights=weights,...)
	else
		if(ndups < 2 || correlation==0)
			fit <- lm.series(M,design=design,ndups=ndups,spacing=spacing,weights=weights)
		else
			fit <- gls.series(M,design=design,ndups=ndups,spacing=spacing,correlation=correlation,weights=weights,...)
	fit$method <- method
	fit$design <- design
	if(ndups > 1) fit$correlation <- correlation
	if(is(object,"MAList")) {
		if(!is.null(object$genes)) fit$genes <- uniquegenelist(object$genes,ndups=ndups,spacing=spacing) 
		if(!is.null(object$A)) fit$Amean <- rowMeans(unwrapdups(as.matrix(object$A),ndups=ndups,spacing=spacing),na.rm=TRUE)
	}
	if(is(object,"marrayNorm")) {
		if(length(object@maGnames)) {
			fit$genes <- uniquegenelist(data.frame(Labels=object@maGnames@maLabels,object@maGnames@maInfo),ndups=ndups,spacing=spacing)
			attr(fit$genes, "Notes") <- object@maGnames@maNotes
		}
		if(length(object@maA)) fit$Amean <- rowMeans(unwrapdups(object@maA,ndups=ndups,spacing=spacing),na.rm=TRUE)
	}
	new("MArrayLM",fit)
}

unwrapdups <- function(M,ndups=2,spacing=1) {
#	Unwrap M matrix for a series of experiments so that all spots for a given gene are in one row
#	Gordon Smyth
#	18 Jan 2002. Last revised 2 Nov 2002.

	if(ndups==1) return(M)
	M <- as.matrix(M)
	nspots <- dim(M)[1]
	nslides <- dim(M)[2]
	ngroups <- nspots / ndups / spacing
	dim(M) <- c(spacing,ndups,ngroups,nslides)
	M <- aperm(M,perm=c(1,3,2,4))
	dim(M) <- c(spacing*ngroups,ndups*nslides)
	M
}

uniquegenelist <- function(genelist,ndups=2,spacing=1) {
#	Eliminate entries in genelist for duplicate spots
#	Gordon Smyth
#	2 Nov 2002.  Last revised 12 Apr 2003

	if(ndups <= 1) return(genelist)
	if(is.null(dim(genelist))) dim(genelist) <- c(length(genelist),1)
	index <- drop(unwrapdups(1:nrow(genelist),ndups=ndups,spacing=spacing)[,1])
	drop(genelist[index,])
}

lm.series <- function(M,design=NULL,ndups=1,spacing=1,weights=NULL)
{
#	Fit linear model for each gene to a series of arrays
#	Gordon Smyth
#	18 Apr 2002. Revised 30 June 2003.

	M <- as.matrix(M)
	narrays <- ncol(M)
	if(is.null(design)) design <- matrix(1,narrays,1)
	design <- as.matrix(design)
	nbeta <- ncol(design)
	if(!is.null(weights)) {
		weights <- as.matrix(weights)
		if(any(dim(weights) != dim(M))) weights <- array(weights,dim(M))
		weights[weights <= 0] <- NA
		M[!is.finite(weights)] <- NA
	}
	if(ndups>1) {
		M <- unwrapdups(M,ndups=ndups,spacing=spacing)
		design <- design %x% rep(1,ndups)
		if(!is.null(weights)) weights <- unwrapdups(weights,ndups=ndups,spacing=spacing)
	}
	ngenes <- nrow(M)
	stdev.unscaled <- beta <- matrix(NA,ngenes,nbeta,dimnames=list(NULL,colnames(design)))
	sigma <- rep(NA,ngenes)
	df.residual <- rep(0,ngenes)
	for (i in 1:ngenes) {
		y <- as.vector(M[i,])
		obs <- is.finite(y)
		if(sum(obs) > 0) {
			X <- design[obs,,drop=FALSE]
			y <- y[obs]
			if(is.null(weights))
				out <- lm.fit(X,y)
			else {
				w <- as.vector(weights[i,obs])
				out <- lm.wfit(X,y,w)
			}
			est <- !is.na(out$coef)
			beta[i,] <- out$coef
			stdev.unscaled[i,est] <- sqrt(diag(chol2inv(out$qr$qr,size=out$rank)))
			df.residual[i] <- out$df.residual
			if(df.residual[i] > 0)
				if(is.null(weights))
					sigma[i] <- sqrt(sum(out$residuals^2)/out$df.residual)
				else
					sigma[i] <- sqrt(sum(w*out$residuals^2)/out$df.residual)
		}
	}
	list(coefficients=drop(beta),stdev.unscaled=drop(stdev.unscaled),sigma=sigma,df.residual=df.residual)
}

rlm.series <- function(M,design=NULL,ndups=1,spacing=spacing,weights=NULL,...)
{
#	Robustly fit linear model for each gene to a series of arrays
#	Gordon Smyth
#	20 Mar 2002.  Last revised 19 Sept 2003.

	require(MASS) # need rlm.default
	M <- as.matrix(M)
	narrays <- ncol(M)
	if(is.null(design)) design <- matrix(1,narrays,1)
	design <- as.matrix(design)
	nbeta <- ncol(design)
	if(!is.null(weights)) {
		weights <- as.matrix(weights)
		if(any(dim(weights) != dim(M))) weights <- array(weights,dim(M))
		weights[weights <= 0] <- NA
		M[!is.finite(weights)] <- NA
	}
	if(ndups>1) {
		M <- unwrapdups(M,ndups=ndups,spacing=spacing)
		design <- design %x% rep(1,ndups)
		if(!is.null(weights)) weights <- unwrapdups(weights,ndups=ndups,spacing=spacing)
	}
	ngenes <- nrow(M)
	stdev.unscaled <- beta <- matrix(NA,ngenes,nbeta)
	sigma <- rep(NA,ngenes)
	df.residual <- rep(0,ngenes)
	for (i in 1:ngenes) {
		y <- as.vector(M[i,])
		obs <- is.finite(y)
		X <- design[obs,,drop=FALSE]
		y <- y[obs]
		if(is.null(weights))
			w <- rep(1,length(y))
		else
			w <- as.vector(weights[i,obs])
		if(length(y) > nbeta) {
			out <- rlm(x=X,y=y,weights=w,...)
			beta[i,] <- coef(out)
			stdev.unscaled[i,] <- sqrt(diag(chol2inv(out$qr$qr)))
			df.residual[i] <- length(y) - out$rank
			if(df.residual[i] > 0) sigma[i] <- out$s
		}
	}
	list(coefficients=drop(beta),stdev.unscaled=drop(stdev.unscaled),sigma=sigma,df.residual=df.residual)
}

gls.series <- function(M,design=NULL,ndups=2,spacing=1,correlation=NULL,weights=NULL,...)
{
#	Fit linear model for each gene to a series of microarrays.
#	Fit is by generalized least squares allowing for correlation between duplicate spots.
#	Gordon Smyth
#	11 May 2002.  Last revised 18 Aug 2003.

	if(ndups<2) {
		warning("No duplicates: correlation between duplicates set to zero")
		ndups <- 1
		correlation <- 0
	}
	M <- as.matrix(M)
	narrays <- ncol(M)
	if(is.null(design)) design <- matrix(1,narrays,1)
	design <- as.matrix(design)
	if(nrow(design) != narrays) stop("Number of rows of design matrix does not match number of arrays")
	if(is.null(correlation)) correlation <- dupcor.series(M,design,ndups,...)$cor
	if(!is.null(weights)) {
		weights <- as.matrix(weights)
		if(any(dim(weights) != dim(M))) weights <- array(weights,dim(M))
		M[weights < 1e-15 ] <- NA
		weights[weights < 1e-15] <- NA
	}
	nbeta <- ncol(design)
	coef.names <- colnames(design)
	M <- unwrapdups(M,ndups=ndups,spacing=spacing)
	ngenes <- nrow(M)
	if(!is.null(weights)) weights <- unwrapdups(weights,ndups=ndups,spacing=spacing)
	design <- design %x% rep(1,ndups)
	cormatrix <- diag(rep(correlation,narrays)) %x% array(1,c(ndups,ndups))
	diag(cormatrix) <- 1
	stdev.unscaled <- beta <- matrix(NA,ngenes,nbeta,dimnames=list(NULL,coef.names))
	sigma <- rep(NA,ngenes)
	df.residual <- rep(0,ngenes)
	for (i in 1:ngenes) {
		y <- drop(M[i,])
		o <- is.finite(y)
		y <- y[o]
		n <- length(y)
		if(n > 0) {
			X <- design[o,,drop=FALSE]
			V <- cormatrix[o,o]
			if(!is.null(weights)) {
				wrs <- 1/sqrt(drop(weights[i,o]))
				V <- wrs * t(wrs * t(V))
			}
			cholV <- chol(V)
			y <- backsolve(cholV,y,transpose=TRUE)
			if(all(X==0)) {
				df.residual[i] <- n
				sigma[i] <- sqrt( array(1/n,c(1,n)) %*% y^2 )
			} else {
				X <- backsolve(cholV,X,transpose=TRUE)
				out <- lm.fit(X,y)
				est <- !is.na(out$coefficients)
				beta[i,] <- out$coefficients
				stdev.unscaled[i,est] <- sqrt(diag(chol2inv(out$qr$qr,size=out$rank)))
				df.residual[i] <- out$df.residual
				if(df.residual[i] > 0)
					sigma[i] <- sqrt( array(1/out$df.residual,c(1,n)) %*% out$residuals^2 )
			}
		}
	}
	list(coefficients=drop(beta),stdev.unscaled=drop(stdev.unscaled),sigma=sigma,df.residual=df.residual,correlation=correlation)
}

duplicateCorrelation <- function(object,design=rep(1,ncol(M)),ndups=2,spacing=1,initial=0.8,trim=0.15,weights=NULL)
{
#	Estimate the correlation between duplicates given a series of arrays
#	Gordon Smyth
#	25 Apr 2002. Last revised 30 June 2003.

	if(is(object,"MAList")) {
		M <- object$M
		if(missing(design) && !is.null(object$design)) design <- object$design
		if(missing(ndups) && !is.null(object$printer$ndups)) ndups <- object$printer$ndups
		if(missing(spacing) && !is.null(object$printer$spacing)) spacing <- object$printer$spacing
		if(missing(weights) && !is.null(object$weights)) weights <- object$weights
	}

	M <- as.matrix(M)
	if(ndups<2) {
		warning("No duplicates: correlation between duplicates not estimable")
		return( list(cor=NA,cor.genes=rep(NA,nrow(M))) )
	}
	require( "nlme" ) # need gls function
	narrays <- ncol(M)
	design <- as.matrix(design)
	if(nrow(design) != narrays) stop("Number of rows of design matrix does not match number of arrays")
	if(!is.null(weights)) {
		weights <- as.matrix(weights)
		if(any(dim(weights) != dim(M))) weights <- array(weights,dim(M))
		M[weights < 1e-15 ] <- NA
		weights[weights < 1e-15] <- NA
	}
	nbeta <- ncol(design)
	M <- unwrapdups(M,ndups=ndups,spacing=spacing)
	ngenes <- nrow(M)
	if(!is.null(weights)) weights <- unwrapdups(weights,ndups=ndups,spacing=spacing)
	design <- design %x% rep(1,ndups)
	Array <- rep(1:narrays,rep(ndups,narrays))
	rho <- rep(NA,ngenes)
	for (i in 1:ngenes) {
		y <- drop(M[i,])
		if(any(!is.finite(y))) y[!is.finite(y)] <- NA
		if(any(diff(Array[is.finite(y)])==0) && sum(!is.na(y)) > nbeta+1)
		if(!is.null(weights)) {
			w <- 1/drop(weights[i,])
			rho[i] <- coef(gls(y~design-1,correlation=corCompSymm(initial,form=~1|Array,fixed=FALSE),weights=~w,na.action=na.omit,control=list(singular.ok=TRUE,returnObject=TRUE,apVar=FALSE))$modelStruct,FALSE)
		} else
			rho[i] <- coef(gls(y~design-1,correlation=corCompSymm(initial,form=~1|Array,fixed=FALSE),na.action=na.omit,control=list(singular.ok=TRUE,returnObject=TRUE,apVar=FALSE))$modelStruct,FALSE)
	}
	rhom <- tanh(mean(atanh(rho),trim=trim,na.rm=TRUE))
	list(cor=rhom,cor.genes=rho)
}


dupcor.series <- function(M,design=rep(1,ncol(M)),ndups=2,spacing=1,initial=0.8,trim=0.15,weights=NULL)
{
#	Estimate the correlation between duplicates given a series of arrays
#	Gordon Smyth
#	25 Apr 2002. Last revised 28 Jan 2003.

	M <- as.matrix(M)
	if(ndups<2) {
		warning("No duplicates: correlation between duplicates not estimable")
		return( list(cor=NA,cor.genes=rep(NA,nrow(M))) )
	}
	require( "nlme" ) # need gls function
	narrays <- ncol(M)
	if(is.vector(design)) dim(design) <- c(narrays,1)
	if(nrow(design) != narrays) stop("Number of rows of design matrix does not match number of arrays")
	if(!is.null(weights)) {
		weights <- as.matrix(weights)
		if(any(dim(weights) != dim(M))) weights <- array(weights,dim(M))
		M[weights < 1e-15 ] <- NA
		weights[weights < 1e-15] <- NA
	}
	nbeta <- ncol(design)
	M <- unwrapdups(M,ndups=ndups,spacing=spacing)
	ngenes <- nrow(M)
	if(!is.null(weights)) weights <- unwrapdups(weights,ndups=ndups,spacing=spacing)
	design <- design %x% rep(1,ndups)
	Array <- rep(1:narrays,rep(ndups,narrays))
	rho <- rep(NA,ngenes)
	for (i in 1:ngenes) {
		y <- drop(M[i,])
		if(any(!is.finite(y))) y[!is.finite(y)] <- NA
		if(any(diff(Array[is.finite(y)])==0) && sum(!is.na(y)) > nbeta+1)
		if(!is.null(weights)) {
			w <- 1/drop(weights[i,])
			rho[i] <- coef(gls(y~design-1,correlation=corCompSymm(initial,form=~1|Array,fixed=FALSE),weights=~w,na.action=na.omit,control=list(singular.ok=TRUE,returnObject=TRUE,apVar=FALSE))$modelStruct,FALSE)
		} else
			rho[i] <- coef(gls(y~design-1,correlation=corCompSymm(initial,form=~1|Array,fixed=FALSE),na.action=na.omit,control=list(singular.ok=TRUE,returnObject=TRUE,apVar=FALSE))$modelStruct,FALSE)
	}
	rhom <- tanh(mean(atanh(rho),trim=trim,na.rm=TRUE))
	list(cor=rhom,cor.genes=rho)
}

contrasts.fit <- function(fit,contrasts) {
#	Convert coefficients and std deviations in fit object to reflect contrasts of interest
#	Gordon Smyth
#	13 Oct 2002.  Last modified 4 September 2003.

	ncoef <- NCOL(fit$coefficients)
	if(nrow(contrasts)!=ncoef) stop("Number of rows of contrast matrix must match number of coefficients")
	fit$coefficients <- fit$coefficients %*% contrasts
	design <- fit$design
	if(!is.null(design) && ncoef > 1) {
		A <- crossprod( abs(design) > 1e-14 )
		orthog <- all(A[lower.tri(A)]==0) 
	}
	if(is.null(design) || ncoef==1 || orthog)
		fit$stdev.unscaled <- sqrt(fit$stdev.unscaled^2 %*% contrasts^2)
	else {
		A <- La.chol2inv(La.chol(crossprod(design)))
		s <- sqrt(diag(A))
		R <- La.chol(t(A/s)/s)
		ngenes <- NROW(fit$stdev.unscaled)
		ncont <- NCOL(contrasts)
		U <- matrix(1,ngenes,ncont,dimnames=list(rownames(fit$stdev.unscaled),colnames(contrasts)))
		for (i in 1:ngenes) {
			RUC <- R %*% t(t(contrasts)*fit$stdev.unscaled[i,])
			U[i,] <- sqrt(array(1,c(1,ncoef)) %*% RUC^2)
		}
		fit$stdev.unscaled <- U
	}
	fit$contrasts <- contrasts
	fit
}

#contrasts.fit0 <- function(fit,contrasts) {
##	Extract contrast information from oneway linear model fit
##	Gordon Smyth
##	13 Oct 2002.  Last modified 1 July 2003.
#
#	fit$coefficients <- fit$coefficients %*% contrasts
#	fit$stdev.unscaled <- sqrt(fit$stdev.unscaled^2 %*% contrasts^2)
#	fit$contrasts <- contrasts
#	fit
#}

#	LOESS FUNCTIONS

loessFit <- function(y, x, weights=NULL, span=0.3, bin=0.01/(2-is.null(weights)), iterations=4) {
#	Fast loess fit for simple x and y
#	Gordon Smyth
#	28 June 2003.  Last revised 3 September 2003.

	n <- length(y)
	if(is.null(weights)) {
		obs <- is.finite(y) & is.finite(x)
		xobs <- x[obs]
		yobs <- y[obs]
		nobs <- length(yobs)
		if(nobs==0) stop("no observed points")
		o <- order(xobs)
		oo <- order(o)
		iter <- iterations-1
		delta = bin * diff(range(xobs)) 
		smoothy <- .C("lowess", x = as.double(xobs[o]), as.double(yobs[o]), 
			nobs, as.double(span), as.integer(iter), as.double(delta), 
			y = double(nobs), double(nobs), double(nobs), PACKAGE = "base")$y[oo]
		out <- list(fitted=rep(NA,n),residuals=rep(NA,n))
		out$fitted[obs] <- smoothy
		out$residuals[obs] <- yobs-smoothy
	} else {
		obs <- is.finite(y) & is.finite(x) & is.finite(weights)
		xobs <- x[obs]
		yobs <- y[obs]
		wobs <- weights[obs]
		nobs <- length(yobs)
		if(nobs==0) stop("no observed points")
#		Suppress warning "k-d tree limited by memory"
		oldopt <- options(warning.expression=expression())
		on.exit(options(oldopt))
		fit <- .vsimpleLoess(y=yobs, x=xobs, weights=wobs, span=span, degree=1,
			cell=bin/span, iterations=iterations)
		out <- list(fitted=rep(NA,n),residuals=rep(NA,n))
		out$fitted[obs] <- fit$fitted
		out$residuals[obs] <- fit$residuals
	}
	out
}

.vsimpleLoess <- function (y, x, weights, span=0.75, degree=2, cell=0.2, iterations=1)
#	Cut-down version of simpleLoess from modreg package
#	Gordon Smyth
#	28 June 2003
{
	statistics <- "none"
	surface <- "interpolate"
	surf.stat <- paste(surface, statistics, sep = "/")
	D <- 1
	N <- NROW(x)
	if (!N || !D) stop("invalid `x'")
	if (!length(y)) stop("invalid `y'")
	x <- as.matrix(x)
	max.kd <- max(N, 200)
	robust <- rep(1, N)
	sum.drop.sqr <- 0
	sum.parametric <- 0
	nonparametric <- 1
	order.parametric <- 1
	order.drop.sqr <- 2
	if (iterations) 
		for (j in 1:iterations) {
			robust <- weights * robust
			z <- .C("loess_raw", as.double(y), as.double(x), 
				as.double(weights), as.double(robust), as.integer(D), 
				as.integer(N), as.double(span), as.integer(degree), 
				as.integer(nonparametric), as.integer(order.drop.sqr), 
				as.integer(sum.drop.sqr), as.double(span * cell), 
				as.character(surf.stat), fitted.values = double(N), 
				parameter = integer(7), a = integer(max.kd), 
				xi = double(max.kd), vert = double(2 * D), vval = double((D + 
				1) * max.kd), diagonal = double(N), trL = double(1), 
				delta1 = double(1), delta2 = double(1), as.integer(surf.stat == 
				"interpolate/exact"), PACKAGE = "modreg")
			fitted.residuals <- y - z$fitted.values
			if (j < iterations) 
				robust <- .Fortran("lowesw", as.double(fitted.residuals), 
				as.integer(N), robust = double(N), double(N), 
				PACKAGE = "modreg")$robust
		}
	list(fitted = z$fitted.values, residuals = fitted.residuals)
}

#  WITHIN ARRAY NORMALIZATION

MA.RG <- function(object, log.transform=TRUE) {
#	Convert RGList to MAList
#	Gordon Smyth
#	2 March 2003.  Last revised 5 September 2003.

	R <- object$R
	G <- object$G

#	Background correction
	if(!is.null(object$Rb)) R <- R-object$Rb
	if(!is.null(object$Gb)) G <- G-object$Gb

#	Log
	if(log.transform) { 
		R[R <= 0] <- NA
		G[G <= 0] <- NA
		R <- log(R,2)
		G <- log(G,2)
	}
	
#	Minus and Add
	object$R <- object$G <- object$Rb <- object$Gb <- NULL
	object$M <- as.matrix(R-G)
	object$A <- as.matrix((R+G)/2)
	new("MAList",unclass(object))
}

RG.MA <- function(object) {
#	Convert MAList to logged RGList
#	Gordon Smyth
#	5 September 2003.

	object$R <- object$A+object$M/2
	object$G <- object$A-object$M/2
	object$M <- NULL
	object$A <- NULL
	new("RGList",unclass(object))
}

normalizeWithinArrays <- function(object,layout=object$printer,method="printtiploess",weights=object$weights,span=0.3,iterations=4,controlspots=NULL,df=5,robust="M") {
#	Within array normalization
#	Gordon Smyth
#	2 March 2003.  Last revised 8 October 2003.

	if(!is(object,"MAList")) object <- MA.RG(object)
	choices <- c("none","median","loess","printtiploess","composite","robustspline")
	method <- match.arg(method,choices)
	if(method=="none") return(object)
	narrays <- ncol(object$M)
	if(method=="median") {
		for (j in 1:narrays) object$M[,j] <- object$M[,j] - median(object$M[,j],na.rm=TRUE)
		return(object)
	}
#	All remaining methods use regression of M-values on A-values
	switch(method,
		loess = {
			for (j in 1:narrays) {
				y <- object$M[,j]
				x <- object$A[,j]
				w <- weights[,j]
				object$M[,j] <- loessFit(y,x,w,span=span,iterations=iterations)$residuals
			}
		},
		printtiploess = {
			if(is.null(layout)) stop("Layout argument not specified")
			ngr <- layout$ngrid.r
			ngc <- layout$ngrid.c
			nspots <- layout$nspot.r * layout$nspot.c
			for (j in 1:narrays) {
				spots <- 1:nspots
				for (gridr in 1:ngr)
				for (gridc in 1:ngc) {
					y <- object$M[spots,j]
					x <- object$A[spots,j]
					w <- weights[spots,j]
					object$M[spots,j] <- loessFit(y,x,w,span=span,iterations=iterations)$residuals
					spots <- spots + nspots
				}
			}
		},
		composite = {
			if(is.null(layout)) stop("Layout argument not specified")
			ntips <- layout$ngrid.r * layout$ngrid.c
			nspots <- layout$nspot.r * layout$nspot.c
			for (j in 1:narrays) {
				y <- object$M[,j]
				x <- object$A[,j]
				w <- weights[,j]
				f <- is.finite(y) & is.finite(x) & is.finite(w)
				y[!f] <- NA
				fit <- loess(y~x,weights=w,span=span,subset=controlspots,na.action=na.exclude,degree=0,surface="direct",family="symmetric",trace.hat="approximate",iterations=iterations)
				global <- predict(fit,newdata=x)
				alpha <- (rank(x)-1) / sum(!is.na(x))
				spots <- 1:nspots
				for (tip in 1:ntips) {
					y <- object$M[spots,j]
					x <- object$A[spots,j]
					w <- weights[spots,j]
					local <- loessFit(y,x,w,span=span,iterations=iterations)$fitted
					object$M[spots,j] <- object$M[spots,j] - alpha[spots]*global[spots]-(1-alpha[spots])*local
					spots <- spots + nspots
				}
			}
		},
		robustspline = {
			if(is.null(layout)) stop("Layout argument not specified")
			for (j in 1:narrays)
				object$M[,j] <- normalizeRobustSpline(object$M[,j],object$A[,j],layout,df=df,method=robust)
		}
	)
	object
}

normalizeRobustSpline <- function(M,A,layout,df=5,method="M") {
#	Robust spline normalization
#	Gordon Smyth
#	27 April 2003.  Last revised 28 April 2003.

	require(MASS)
	require(splines)
	ngrids <- layout$ngrid.r * layout$ngrid.c
	nspots <- layout$nspot.r * layout$nspot.c
#	col <- rainbow(ngrids,end=(ngrids-2)/ngrids)

#	Global splines
	O <- is.finite(M) & is.finite(A)
	X <- matrix(NA,ngrids*nspots,df)
	X[O,] <- ns(A[O],df=df,intercept=TRUE)
	x <- X[O,]
	y <- M[O]
	s <- summary(rlm(x,y,method=method))
	beta0 <- s$coefficients[,1]
	covbeta0 <- s$cov * s$stddev^2

#	Tip-wise splines
	beta <- array(0,c(ngrids,df))
	covbeta <- array(0,c(ngrids,df,df))
	spots <- 1:nspots
	for (i in 1:ngrids) {
		o <- O[spots]
		y <- M[spots][o]
		x <- X[spots,][o,]
		s <- summary(rlm(x,y,method=method))
		beta[i,] <- s$coefficients[,1]
		covbeta[i,,] <- s$cov * s$stddev^2
		spots <- spots + nspots
	}

#	Empirical Bayes estimates
	res.cov <- cov(beta) - apply(covbeta,c(2,3),mean)
	Sigma0 <- covbeta0 * max(0, sum(diag(res.cov)) / sum(diag(covbeta0)) )
#	Sigma0 <- covbeta0 * max(0,mean(eigen(solve(covbeta0,res.cov))$values))

#	Shrunk splines
	spots <- 1:nspots
	for (i in 1:ngrids) {
		beta[i,] <- beta0 + Sigma0 %*% solve(Sigma0+covbeta[i,,],beta[i,]-beta0)
		o <- O[spots]
		x <- X[spots,][o,]
		M[spots][o] <- M[spots][o] - x %*% beta[i,]
		M[spots][!o] <- NA
		spots <- spots + nspots
	}
	M
}


#  PRINTORDER

normalizeForPrintorder <- function(object,layout,start="topleft",method="loess",separate.channels=FALSE,span=0.1,plate.size=32) {
#	Pre-normalize the foreground intensities for print order
#	Gordon Smyth
#	11 Mar 2002.  Last revised 18 June 2003.

	if(is.null(object$R) || is.null(object$G)) stop("List must contain components R and G")
	start <- match.arg(start,c("topleft","topright"))
	method <- match.arg(method,c("loess","plate"))
	po <- printorder(layout,start=start)$printorder
	nslides <- NCOL(object$R)
	for (i in 1:nslides) {
		RG <- normalizeForPrintorder.rg(R=object$R[,i],G=object$G[,i],printorder=po,method=method,separate.channels=separate.channels,span=span,plate.size=plate.size)
		object$R[,i] <- RG$R
		object$G[,i] <- RG$G
	}
	object
}

normalizeForPrintorder.rg <- function(R,G,printorder,method="loess",separate.channels=FALSE,span=0.1,plate.size=32,plot=FALSE) {
#	Pre-normalize the foreground intensities for print order, given R and G for a single array.
#	Gordon Smyth
#	8 Mar 2002.  Last revised 18 June 2003.

	if(plot) ord <- order(printorder)
	Rf <- log(R,2)
	Gf <- log(G,2)
	Rf[is.infinite(Rf)] <- NA
	Gf[is.infinite(Gf)] <- NA
	if(!separate.channels) Af <- (Rf+Gf)/2
	method <- match.arg(method,c("loess","plate"))
	if(method=="plate") {
		# Correct for plate pack (usually four 384-well plates)
		plate <- 1 + (printorder-0.5) %/% plate.size
		hubermu <- function(...) huber(...)$mu
		if(separate.channels) {
			plate.mR <- tapply(Rf,plate,hubermu)
			plate.mG <- tapply(Gf,plate,hubermu)
			mR <- mG <- Rf
			for (i in 1:max(plate)) {
				mR[plate==i] <- plate.mR[i]
				mG[plate==i] <- plate.mG[i]
			}
			if(plot) {
				plot(printorder,Rf,xlab="Print Order",ylab="Log Intensity",type="n")
				points(printorder,Rf,pch=".",col="red")
				points(printorder,Gf,pch=".",col="green")
				lines(printorder[ord],mR[ord],col="red")
				lines(printorder[ord],mG[ord],col="green")
			}
			mR <- mR - mean(mR,na.rm=TRUE)
			mG <- mG - mean(mG,na.rm=TRUE)
		} else {
			plate.m <- tapply(Af,plate,hubermu)
			m <- Af
			for (i in 1:max(plate)) m[plate==i] <- plate.m[i]
			if(plot) {
				plot(printorder,Af,xlab="Print Order",ylab="Log Intensity",pch=".")
				lines(printorder[ord],m[ord])
			}
			mR <- mG <- m - mean(m,na.rm=TRUE)
		}
	} else {
		# Smooth correction for time order
		if(separate.channels) {
			mR <- fitted(loess(Rf~printorder,span=span,degree=0,family="symmetric",trace.hat="approximate",iterations=5,surface="direct",na.action=na.exclude))
			mG <- fitted(loess(Gf~printorder,span=span,degree=0,family="symmetric",trace.hat="approximate",iterations=5,surface="direct",na.action=na.exclude))
			if(plot) {
				plot(printorder,Rf,xlab="Print Order",ylab="Log Intensity",type="n")
				points(printorder,Rf,pch=".",col="red")
				points(printorder,Gf,pch=".",col="green")
				lines(printorder[ord],mR[ord],col="red")
				lines(printorder[ord],mG[ord],col="green")
			}
			mR <- mR - mean(mR,na.rm=TRUE)
			mG <- mG - mean(mG,na.rm=TRUE)
		} else {
			m <- fitted(loess(Af~printorder,span=span,degree=0,family="symmetric",trace.hat="approximate",iterations=5,surface="direct",na.action=na.exclude))
			if(plot) {
				plot(printorder,Af,xlab="Print Order",ylab="Log Intensity",pch=".")
				lines(printorder[ord],m[ord])
			}
			mR <- mG <- m - mean(m,na.rm=TRUE)
		}
	}
	list(R=2^(Rf-mR),G=2^(Gf-mG),R.trend=mR,G.trend=mG)
}

plotPrintorder <- function(object,layout,start="topleft",slide=1,method="loess",separate.channels=FALSE,span=0.1,plate.size=32) {
#	Pre-normalize the foreground intensities for print order.
#	Gordon Smyth
#	9 Apr 2002.  Last revised 18 June 2003.

	if(is.null(object$R) || is.null(object$G)) stop("List must contain components R and G")
	G <- object$G[,slide]
	R <- object$R[,slide]
	if(length(R) != length(G)) stop("R and G must have same length")
	start <- match.arg(start,c("topleft","topright"))
	po <- printorder(layout,start=start)$printorder
	invisible(normalizeForPrintorder.rg(R=R,G=G,printorder=po,method=method,separate.channels=separate.channels,span=span,plate.size=plate.size,plot=TRUE))
}

#  BETWEEN ARRAY NORMALIZATION

normalizeBetweenArrays <- function(object, method="scale", ties=FALSE) {
#	Normalize between arrays
#	Gordon Smyth
#	12 Apri 2003.  Last revised 26 September 2003.

	choices <- c("none","scale","quantile","Aquantile")
	method <- match.arg(method,choices)
	if(is(object,"matrix")) {
		if(method=="Aquantile") stop("Aquantile normalization not applicable to matrix object")
		return(switch(method,
			none = object,
			scale = normalizeMedianDeviations(object),
			quantile = normalizeQuantiles(object, ties=ties)
		))
	}
	if(is.null(object$M) || is.null(object$A)) stop("object must be a list with M and A components")
	switch(method,
		scale = {
			object$M <- normalizeMedianDeviations(object$M)
			object$A <- normalizeMedians(object$A)
		},
		quantile = {
			narrays <- NCOL(object$M)
			Z <- normalizeQuantiles(cbind(object$A+object$M/2,object$A-object$M/2),ties=ties)
			R <- Z[,1:narrays]
			G <- Z[,narrays+(1:narrays)]
			object$M <- R-G
			object$A <- (R+G)/2
		},
		Aquantile = {
			object$A <- normalizeQuantiles(object$A,ties=ties)
		})
	object
}

normalizeQuantiles <- function(A, ties=FALSE) {
#	Normalize columns of a matrix to have the same quantiles, allowing for missing values.
#	Gordon Smyth
#	25 June 2002.  Last revised 5 June 2003.

	n <- dim(A)
	if(is.null(n)) return(A)
	if(n[2]==1) return(A)
	O <- S <- array(,n)
	if(ties) R <- O
	nobs <- rep(n[1],n[2])
	i <- (0:(n[1]-1))/(n[1]-1)
	for (j in 1:n[2]) {
		Si <- sort(A[,j], method="quick", index.return=TRUE)
		if(ties) R[,j] <- rank(A[,j])
		nobsj <- length(Si$x)
		if(nobsj < n[1]) {
			nobs[j] <- nobsj
			isna <- is.na(A[,j])
			S[,j] <- approx((0:(nobsj-1))/(nobsj-1), Si$x, i, ties="ordered")$y
			O[!isna,j] <- ((1:n[1])[!isna])[Si$ix]
		} else {
			S[,j] <- Si$x
			O[,j] <- Si$ix
		}
	}
	m <- rowMeans(S)
	for (j in 1:n[2]) {
		if(nobs[j] < n[1]) {
			isna <- is.na(A[,j])
			if(ties)
				A[!isna,j] <- approx(i, m, (R[!isna,j]-1)/(nobs[j]-1), ties="ordered")$y
			else
				A[O[!isna,j],j] <- approx(i, m, (0:(nobs[j]-1))/(nobs[j]-1), ties="ordered")$y
		} else {
			if(ties)
				A[,j] <- approx(i, m, (R[,j]-1)/(n[1]-1), ties="ordered")$y
			else
				A[O[,j],j] <- m
		}
	}
	A
}

normalizeMedianDeviations <- function(x) 
{
#	Normalize columns of a matrix to have the same median absolute value
#	Gordon Smyth
#	14 Mar 2002.  Last revised 12 Apr 2003.

	narrays <- NCOL(x)
	if(narrays==1) return(x)
	medabs <- function(x) median(abs(as.numeric(x[!is.na(x)])))
	xmat.mav <- apply(x, 2, medabs)
	denom <- (prod(xmat.mav))^(1/narrays)
	si <- xmat.mav/denom
	t(t(x)/si)
}

normalizeMedians <- function(x) 
{
#	Normalize columns of a matrix to have the same median value
#	Gordon Smyth
#	12 April 2003

	narrays <- NCOL(x)
	if(narrays==1) return(x)
	a.med <- apply(x, 2, median, na.rm=TRUE)
	a.med <- a.med / (prod(a.med))^(1/narrays)
	t(t(x)/a.med)
}

#  M-A PLOTS

plotMA <- function(MA, array=1, pch=16, status=NULL,
	values=c("gene","blank","buffer","utility","negative","calibration","ratio"),
	col=c("black","yellow","orange","pink","brown","blue","red"),
	cex=c(0.1,0.6,0.6,0.6,0.6,0.6,0.6)) {
#	MA-plot with color coding for controls
#	Gordon Smyth  7 April 2003.  Last modified 27 June 2003.
#	Revised by James Wettenhall  27 June 2003.

	x <- MA$A[,array]
	y <- MA$M[,array]
	if (is.list(pch)) 
		isListPCH <- TRUE 
	else 
		isListPCH <- FALSE
	pch <- as.list(pch)
	plot(x,y,xlab="A",ylab="M",main=colnames(MA$M)[array],type="n")
	if(is.null(status))
		points(x,y,pch=pch[[1]],cex=cex[1])
	else {
		nvalues <- length(values)
		if (length(pch) < nvalues)
			pch <- rep(pch,length=nvalues)
		col <- rep(col,nvalues)
		cex <- rep(cex,nvalues)
		for (i in 1:nvalues) {
			sel <- status==values[i]
			points(x[sel],y[sel],pch=pch[[i]],cex=cex[i],col=col[i])
		}
		if (isListPCH)
			legend(min(x,na.rm=TRUE),fill=col,max(y,na.rm=TRUE),legend=values,col=col,cex=0.9)
		else
			legend(min(x,na.rm=TRUE),pch=unlist(pch),max(y,na.rm=TRUE),legend=values,col=col,cex=0.9)
	}
	invisible()
}

#  PLOT DENSITIES

plotDensities<-function(object,log.transform=FALSE,arrays=NULL,singlechannels=NULL,groups=NULL,col=NULL)
#  Plot empirical single-channel densities
#  Natalie Thorne, 9 September 2003
#  Modified by Gordon Smyth, 8 October 2003
{
  matDensities<-function(X){
    densXY<-function(Z){
      zd<-density(Z,na.rm=TRUE)
      x<-zd$x
      y<-zd$y
      cbind(x,y)
    }
    out<-apply(X,2,densXY)
    outx<-out[(1:(nrow(out)/2)),]
    outy<-out[(((nrow(out)/2)+1):nrow(out)),]
    list(X=outx,Y=outy)
  }
    
  if(is(object, "MAList")){
    object<- RG.MA(object)
  }

  
  if( ((is.null(arrays))&(is.null(singlechannels))) ){
    arrays <- 1:(ncol(object$R))
    x <- cbind(object$R,object$G)
    if(is.null(groups)){
      groups <- c(length(arrays),length(arrays))
      if(is.null(col))
        colors <- rep(c("red","green"),groups)
      if(!is.null(col)){
        if(length(col)!=2){
          print("number of groups=2 not equal to number of col")
          colors<-"black"
        }else{
          colors<-rep(col,groups)
        }
      }
    }else{
      if(!is.null(col)){
        if(length(as.vector(table(groups)))!=length(col)){
          print("number of groups not equal to number of col")
          colors <- col
        }else{
          colors <- col[groups]
        }                         
      }else{
        print("warning no colors in col specified for the groups")
        colors <- "black"
      }
    } 
  }else{  
    if(!is.null(singlechannels)){
      if(!is.null(arrays))
        print("warning cant index using arrays AND singlechannels")    
      x<-cbind(object$R,object$G)[,singlechannels]
      if(is.null(groups)){
        groups<-c(length(intersect((1:ncol(object$R)),singlechannels)),
       length(intersect(((ncol(object$R)+1):ncol(cbind(object$G,object$R))),
                  singlechannels)))
        if(is.null(col))
          colors <-rep(c("red","green"),groups)
        if(!is.null(col)){
          if(length(col)!=2){
            print("number of groups=2 not equal to number of col")
            colors<-"black"
          }else{
            colors<-rep(col,groups)
          }
        }
      }else{
        if(!is.null(col)){
          if(length(as.vector(table(groups)))!=length(col)){
            print("number of groups not equal to number of col")
            colors <- col
          }else{
            colors <-col[groups]
          }
        }else{
          print("warning no colors in col specified for the groups")
          colors <- "black"
        }
      }
    }else{      
      if(!is.null(arrays)){
        if(!is.null(singlechannels))
          print("warning cant index using arrays AND singlechannels")
        x <- cbind(object$R[, arrays],object$G[,arrays])
        if(is.null(groups)){
          groups <- c(length(arrays),length(arrays))
          if(is.null(col))
            colors <- rep(c("red","green"),groups)
          if(!is.null(col)){
            if(length(col)!=2){
              print("number of groups=2 not equal to number of col")
              colors<-"black"
            }else{
              colors<-rep(col,groups)
            }
          }
        }else{
          if(!is.null(col)){
            if(length(as.vector(table(groups)))!=length(col)){
              print("number of groups not equal to number of col")
              colors<-"black"
            }else{
              colors<-col[groups]
            }
          }else{
            print("warning no colors in col specified for the groups")
            colors <- "black"
          }
        }
      }
    }
  }
  if(log.transform) x <- log(x,2)
  
  dens.x<-matDensities(x)
#  Commented out by GKS  8 Oct 2003
#  XLIM<-c(min(dens.x$X),max(dens.x$X))
#  YLIM<-c(min(dens.x$Y),max(dens.x$Y))
  matplot(dens.x$X,dens.x$Y, xlab = "Intensity", ylab = "Density",
          main = "RG densities",type="l",col=colors,lwd=2,lty=1)
}
#  PLOTS

imageplot <- function(z, layout=list(ngrid.r=12,ngrid.c=4,nspot.r=26,nspot.c=26), low=NULL, high=NULL, ncolors=123, zerocenter=NULL, zlim=NULL, mar=rep(1,4), ...) {
#  Image plot of spotted microarray data
#  Gordon Smyth
#  20 Nov 2001.  Last revised 18 Sept 2003.

#  Check input
	gr <- layout$ngrid.r
	gc <- layout$ngrid.c
	sr <- layout$nspot.r
	sc <- layout$nspot.c
	if(is.null(gr)||is.null(gc)||is.null(sr)||is.null(sc)) stop("Layout needs to contain components ngrid.r, ngrid.c, nspot.r and spot.c")
	if(length(z) != gr*gc*sr*sc) stop("Number of image spots does not agree with layout dimensions")

#  Check colours
	if(is.character(low)) low <- col2rgb(low)/255
	if(is.character(high)) high <- col2rgb(high)/255
	if(!is.null(low) && is.null(high)) high <- c(1,1,1) - low
	if(is.null(low) && !is.null(high)) low <- c(1,1,1) - high

#  Is zlim preset?
	if(!is.null(zlim)) {
		z <- pmax(zlim[1],z)
		z <- pmin(zlim[2],z)
	}

#  Plot differential expression from "green" to "red" or plot one variable from "white" to "blue"?
	zr <- range(z,na.rm=TRUE)
	zmax <- max(abs(zr))
	zmin <- zr[1]
	if(is.null(zerocenter)) zerocenter <- (zmin < 0)
	if(zerocenter) {
		if(is.null(low)) low <- c(0,1,0)
		if(is.null(high)) high <- c(1,0,0)
		if(is.null(zlim)) zlim <- c(-zmax,zmax)
	} else {
		if(is.null(low)) low <- c(1,1,1)
		if(is.null(high)) high <- c(0,0,1)
		if(is.null(zlim)) zlim <- c(zmin,zmax)
	}

#  Now make the plot
	col <- rgb( seq(low[1],high[1],len=ncolors), seq(low[2],high[2],len=ncolors), seq(low[3],high[3],len=ncolors) )
	dim(z) <- c(sc,sr,gc,gr)
	z <- aperm(z,perm=c(2,4,1,3))
	dim(z) <- c(gr*sr,gc*sc)
	old.par <- par(no.readonly = TRUE)
	on.exit(par(old.par))
	par(mar=mar)
	image(0:(gr*sr),0:(gc*sc),z,zlim=zlim,col=col,axes=FALSE,...)
	for (igrid in 0:gc) lines( c(0,gr*sr), rep(igrid*sc,2) )
	for (igrid in 0:gr) lines( rep(igrid*sr,2), c(0,gc*sc) )
	invisible()
}

plotPrintTipLoess <- function(MA,layout,array=1,span=0.4,...) {
#  MA-plots by print-tip group
#  Gordon Smyth
#  7 April 2003.  Last revised 27 Oct 2003.

	if(!is.null(MA$printer) && missing(layout)) layout <- MA$printer
	y <- MA$M[,array]
	x <- MA$A[,array]
	coplot(y~x|factor(gridc(layout))*factor(gridr(layout)),xlab=c("A","Tip Column"),ylab=c("M","Tip Row"),pch=".",span=span,show.given=FALSE,panel=panel.smooth)
	invisible()
}
#  QUALITY MEASURES

wtarea <- function(ideal=c(160,170))
#	Quality weights based on spot area from SPOT output
#	Gordon Smyth
#	9 March 2003.  Last revised 11 Mar 2003.

function(spot) {
	e <- range(ideal)
	x <- c(-Inf,0,e,sum(e),Inf)
	y <- c(0,0,1,1,0,0)
	approx(x,y,xout=spot[,"area"],ties="ordered")$y
}

wtflags <- function(w=0.1)
#	Quality weights based on Flags from GenePix output
#	Gordon Smyth
#	9 March 2003.  Last revised 11 June 2003.

function(gpr) {
	flagged <- (gpr[,"Flags"] < 0)
	w*flagged + !flagged
}

wtIgnore.Filter <- function(qta) {
#	Quality weights based on Ignore Filter from QuantArray output
#	Gordon Smyth
#	23 May 2003.  Last modified 27 Sep 2003.

	qta[,"Ignore Filter"]
}
#  SUBSET AND COMBINE DATA SETS

assign("[.RGList",
function(object, i, j, ...) {
#  Subsetting for RGList objects
#  Gordon Smyth
#  29 June 2003.  Last modified 5 July 2003.

	if(nargs() != 3) stop("Two subscripts required",call.=FALSE)
	if(missing(i))
		if(missing(j))
			return(object)
		else {
			object$R <- object$R[,j,drop=FALSE]
			object$G <- object$G[,j,drop=FALSE]
			object$Rb <- object$Rb[,j,drop=FALSE]
			object$Gb <- object$Gb[,j,drop=FALSE]
			object$weights <- object$weights[,j,drop=FALSE]
			object$targets <- object$targets[j,,drop=FALSE]
		}
	else
		if(missing(j)) {
			object$R <- object$R[i,,drop=FALSE]
			object$G <- object$G[i,,drop=FALSE]
			object$Rb <- object$Rb[i,,drop=FALSE]
			object$Gb <- object$Gb[i,,drop=FALSE]
			object$weights <- object$weights[i,,drop=FALSE]
			object$genes <- object$genes[i,,drop=FALSE]
		} else {
			object$R <- object$R[i,j,drop=FALSE]
			object$G <- object$G[i,j,drop=FALSE]
			object$Rb <- object$Rb[i,j,drop=FALSE]
			object$Gb <- object$Gb[i,j,drop=FALSE]
			object$weights <- object$weights[i,j,drop=FALSE]
			object$genes <- object$genes[i,,drop=FALSE]
			object$targets <- object$targets[j,,drop=FALSE]
		}
	object
})

assign("[.MAList",
function(object, i, j, ...) {
#  Subsetting for MAList objects
#  Gordon Smyth
#  29 June 2003.  Last modified 29 Oct 2003.

	if(nargs() != 3) stop("Two subscripts required",call.=FALSE)
	if(missing(i))
		if(missing(j))
			return(object)
		else {
			object$M <- object$M[,j,drop=FALSE]
			object$A <- object$A[,j,drop=FALSE]
			object$weights <- object$weights[,j,drop=FALSE]
			object$targets <- object$targets[j,,drop=FALSE]
			if(!is.null(object$design)) {
				object$design <- as.matrix(object$design)[j,,drop=FALSE]
				if(!is.fullrank(object$design)) warning("design matrix is singular",call.=FALSE)
			}
		}
	else
		if(missing(j)) {
			object$M <- object$M[i,,drop=FALSE]
			object$A <- object$A[i,,drop=FALSE]
			object$weights <- object$weights[i,,drop=FALSE]
			object$genes <- object$genes[i,,drop=FALSE]
		} else {
			object$M <- object$M[i,j,drop=FALSE]
			object$A <- object$A[i,j,drop=FALSE]
			object$weights <- object$weights[i,j,drop=FALSE]
			object$genes <- object$genes[i,,drop=FALSE]
			object$targets <- object$targets[j,,drop=FALSE]
			if(!is.null(object$design)) {
				object$design <- as.matrix(object$design)[j,,drop=FALSE]
				if(!is.fullrank(object$design)) warning("design matrix is singular",call.=FALSE)
			}
		}
	object
})

is.fullrank <- function(x) {
#	Check whether a numeric matrix has full column rank
#	Gordon Smyth
#	18 August 2003.  Last modified 29 Oct 2003.

	x <- as.matrix(x)
	e <- La.eigen(crossprod(x),symmetric=TRUE,only.values=TRUE)$values
	e[1] > 0 && abs(e[length(e)]/e[1]) > 1e-13
}

cbind.RGList <- function(..., deparse.level=1) {
#  Combine MAList objects assuming same genelists
#  Gordon Smyth
#  27 June 2003

	objects <- list(...)
	nobjects <- length(objects)
	out <- objects[[1]]
	if(nobjects > 1)
	for (i in 2:nobjects) {
		out$R <- cbind(out$R,objects[[i]]$R)
		out$G <- cbind(out$G,objects[[i]]$G)
		out$Rb <- cbind(out$Rb,objects[[i]]$Rb)
		out$Gb <- cbind(out$Gb,objects[[i]]$Gb)
		out$weights <- cbind(out$weights,objects[[i]]$weights)
		out$targets <- rbind(out$targets,objects[[i]]$targets)
	}
	out
}

cbind.MAList <- function(..., deparse.level=1) {
#  Combine MAList objects assuming same genelists
#  Gordon Smyth
#  27 June 2003

	objects <- list(...)
	nobjects <- length(objects)
	out <- objects[[1]]
	if(nobjects > 1)
	for (i in 2:nobjects) {
		out$M <- cbind(out$M,objects[[i]]$M)
		out$A <- cbind(out$A,objects[[i]]$A)
		out$weights <- cbind(out$weights,objects[[i]]$weights)
		out$targets <- rbind(out$targets,objects[[i]]$targets)
	}
	out
}

makeUnique <- function(x) {
#  Add characters to the elements of a character vector to make all values unique
#  Gordon Smyth
#  10 April 2003

	x <- as.character(x)
	tab <- table(x)
	tab <- tab[tab>1]
	lentab <- length(tab)
	if(lentab > 0) {
		u <- names(tab)
		for (i in 1:lentab) {
			n <- tab[i]
			x[x==u[i]] <- paste(x[x==u[i]],formatC(1:n,width=1+floor(log(n,10)),flag="0"),sep="")
		}
	}
	x
}

merge.RGList <- function(x,y,...) {
#  Merge RGList y into x aligning by row names
#  Gordon Smyth
#  11 April 2003

	if(!is(y,"RGList")) stop("both x and y must be RGList objects")
	genes1 <- rownames(x$R)
	if(is.null(genes1)) genes1 <- rownames(x$G)
	genes2 <- rownames(y$R)
	if(is.null(genes2)) genes2 <- rownames(y$G)
	if(is.null(genes1) || is.null(genes2)) stop("Need row names to align on") 

	fields1 <- names(x)
	fields2 <- names(y)
	if(!identical(fields1,fields2)) stop("The two RGLists have different elements")

	ord2 <- match(makeUnique(genes1), makeUnique(genes2))
	for (i in fields1) x[[i]] <- cbind(x[[i]],y[[i]][ord2,])
	x
}
#  UTILITY FUNCTIONS

matvec <- function(M,v) {
#	Multiply the columns of matrix by the elements of a vector,
#	i.e., compute M %*% diag(v)
#	Gordon Smyth
#	5 July 1999
#
	v <- as.vector(v)
	M <- as.matrix(M)
	if(length(v)!=dim(M)[2]) stop("Dimensions do not match")
	t(v * t(M))
}

vecmat <- function(v,M) {
#	Multiply the rows of matrix by the elements of a vector,
#	i.e., compute diag(v) %*% M
#	Gordon Smyth
#	5 July 1999
#
	v <- as.vector(v)
	M <- as.matrix(M)
	if(length(v)!=dim(M)[1]) stop("Dimensions do not match")
	v * M
}

isNumeric <- function(x) {
#	Test for numeric argument or data.frame with numeric columns
#	Gordon Smyth
#	12 April 2003

	is.numeric(x) || (is.data.frame(x) && length(x)>0 && all(unlist(lapply(x,is.numeric))))
}

helpMethods <- function(genericFunction) {
#	Prompt user for help topics on methods for generic function
#	Gordon Smyth
#	21 April 2003.  Last revised 28 Oct 2003.

	objectclass <- class(genericFunction)
 	if(objectclass != "standardGeneric") {
		if(objectclass == "character" && isGeneric(genericFunction))
			genericFunction <- getGeneric(genericFunction)
		else {
			cat("Not a generic function\n")
			return(invisible())
		}
	}
	functionname <- genericFunction@generic
	methodnames <- names(getMethods(genericFunction)@methods)
	nmethods <- length(methodnames)
	if(nmethods == 0) {
		cat("No available methods\n")
		return(invisible())
	}
	aliasnames <- paste(functionname,",",methodnames,"-method",sep="")
	for (i in 1:nmethods) cat(i,": ",aliasnames[i],"\n",sep="")
	cat("Type number to choose help topic: ")
	n <- as.integer(readline())
	if(n > 0 && n <= nmethods)
		eval(parse(text=paste("help(\"",aliasnames[n],"\")",sep="")))
	else {
	 	cat("No topic chosen\n")
	 	return(invisible())
	}
}
#  VENN DIAGRAM COUNTS AND PLOTS

vennCounts <- function(classification,include="both") {
#	Venn diagram counts
#	Gordon Smyth
#	4 July 2003.  Last modified 15 September 2003.

	if(is(classification,"classification")) classification <- classification$classification
	include <- match.arg(include,c("both","up","down"))
	classification <- switch(include,
		both = as.matrix(abs(classification) > 0.5),
		up = as.matrix(classification > 0.5),
		down = as.matrix(classification < -0.5)
	)
	ngenes <- nrow(classification)
	ncontrasts <- ncol(classification)
	names <- colnames(classification)
	if(is.null(names)) names <- paste("Group",1:ncontrasts)
	noutcomes <- 2^ncontrasts
	outcomes <- matrix(0,noutcomes,ncontrasts)
	colnames(outcomes) <- names
	for (j in 1:ncontrasts)
		outcomes[,j] <- rep(0:1,times=2^(j-1),each=2^(ncontrasts-j))
	counts <- rep(0,noutcomes)
	one <- rep(1,ngenes)
	for (i in 1:noutcomes)
		counts[i] <- sum(apply(classification==(one%*%outcomes[i,,drop=FALSE]),1,all),na.rm=TRUE)
	structure(cbind(outcomes,Counts=counts),class="VennCounts")
}

vennDiagram <- function(object,include="both",names,...) {
#	Plot Venn diagram
#	Gordon Smyth and James Wettenhall
#	4 July 2003.  Last modified 4 September 2003.

	if(class(object) != "VennCounts") object <- vennCounts(object,include=include)
	nsets <- ncol(object)-1
	if(nsets > 3) stop("Can't plot Venn diagram for more than 3 sets")
	if(missing(names)) names <- colnames(object)[1:nsets]
	counts <- object[,"Counts"]
	theta <- 2*pi*(1:360)/360
	xcentres <- list(0,c(-1,1),c(-1,1,0))[[nsets]]
	ycentres <- list(0,c(0,0),c(1/sqrt(3),1/sqrt(3),-2/sqrt(3)))[[nsets]]
	r <- c(1.5,1.5,1.5)[nsets]
	xtext <- list(-1.2,c(-1.2,1.2),c(-1.2,1.2,0))[[nsets]]
	ytext <- list(1.8,c(1.8,1.8),c(2.4,2.4,-3))[[nsets]]
	cex <- 1.5
	plot(x=0,y=0,type="n",xlim=c(-4,4),ylim=c(-4,4),xlab="",ylab="",axes=FALSE,...)
	for(circle in 1:nsets) {
		lines(xcentres[circle]+r*cos(theta),ycentres[circle]+r*sin(theta))
		text(xtext[circle],ytext[circle],names[circle],cex=cex)
	}
	switch(nsets,
		{
			rect(-3,-2.5,3,2.5)
			text(2.3,-2.1,counts[1],cex=cex)
			text(0,0,counts[2],cex=cex)
		}, {
			rect(-3,-2.5,3,2.5)
		    text(2.3,-2.1,counts[1],cex=cex)
			text(1.5,0.1,counts[2],cex=cex)
			text(-1.5,0.1,counts[3],cex=cex)
			text(0,0.1,counts[4],cex=cex)
		}, {
			rect(-3,-3.5,3,3.3)
			text(2.5,-3,counts[1],cex=cex)
			text(0,-1.7,counts[2],cex=cex)
			text(1.5,1,counts[3],cex=cex)
			text(.75,-.35,counts[4],cex=cex)
			text(-1.5,1,counts[5],cex=cex)
			text(-.75,-.35,counts[6],cex=cex)
			text(0,.9,counts[7],cex=cex)
			text(0,0,counts[8],cex=cex)
		}
	)
	invisible()
}

#  SCORE.R

zscoreGamma <- function(q, shape, rate = 1, scale = 1/rate) 
#  Z-score equivalents for gamma deviates
#  Gordon Smyth
#  1 October 2003
{
	z <- q
	n <- length(q)
	shape <- rep(shape,length.out=n)
	scale <- rep(scale,length.out=n)
	up <- (q > shape*scale)
	if(any(up)) z[up] <- qnorm(pgamma(q[up],shape=shape[up],scale=scale[up],lower.tail=FALSE,log.p=TRUE),,lower.tail=FALSE,log.p=TRUE)
	if(any(!up)) z[!up] <- qnorm(pgamma(q[!up],shape=shape[!up],scale=scale[!up],lower.tail=TRUE,log.p=TRUE),lower.tail=TRUE,log.p=TRUE)
	z
}

zscoreT <- function(x, df)
#  Z-score equivalents for gamma deviates
#  Gordon Smyth
#  24 August 2003
{
	z <- x
	df <- rep(df,length.out=length(x))
	pos <- x>0
	if(any(pos)) z[pos] <- qnorm(pt(x[pos],df=df[pos],lower.tail=FALSE,log.p=TRUE),lower.tail=FALSE,log.p=TRUE) 
	if(any(!pos)) z[!pos] <- qnorm(pt(x[!pos],df=df[!pos],lower.tail=TRUE,log.p=TRUE),lower.tail=TRUE,log.p=TRUE)
	z
}

