.packageName <- "graph"

validGraph<-function(object, quietly=FALSE)
{
  if (class(object) == "graphNEL") {
      objEdges<-edges(object)
      objNodes<-nodes(object)
      bad <- FALSE
      if (any(is.na(objNodes))) {
          if (!quietly ) cat("NA element in nodes.\n")
          bad <- TRUE
      }
      if (any(is.na(unlist(objEdges,use.names=FALSE)))) {
          if( !quietly )
              cat ("NA element in edges.\n")
          bad <- TRUE
      }
      ##don't think we want to force this one
#      if (length(objNodes)!=length(objEdges)) {
#          if( !quietly )
#              cat("Nodes and edges must have the same length.\n")
#          bad <- TRUE
#      }
      if (!all( names(objEdges) %in% objNodes )) {
           if( !quietly )
              cat("Edges don't have the same names as the nodes.\n")
           bad <- TRUE
       }
      if (any(duplicated(objNodes))) {
          if( !quietly )
              cat("Node names may not be duplicated\n")
          bad <- TRUE
      }
      ##check for reciprocity in undirected graphs
      ##paste to->from and from->to if any are not duplicated then
      ##the edge is not reciprocal. Note we are not going to handle
      ##multiedges well.
      if( object@edgemode == "undirected") {
          eds <- lapply(object@edgeL, function(x) x$edges)
          v1 <- sapply(eds, length)
          v2 <- sum(v1)
          tM <- paste(rep(1:length(v1), v1), unlist(eds), sep=" -> " )
          tM2 <- paste(unlist(eds), rep(1:length(v1), v1), sep=" -> " )
          tM3 <- c(tM, tM2)
          vv <- duplicated(tM3)
          badn <- which(vv == FALSE)
          badn <- badn[badn>v2]
          if( length(badn)>0 ) {
              if( !quietly ) {
                  from <- badn-v2
                  cat("The graph is undirected and\n")
                  cat("the following edges are not reciprocated\n")
                  cat(tM3[from], sep="\n")
                  cat("\n")
              }
              return(FALSE)
          }
      }
  }
  return(!bad)
}

#graph<-function(newnodes,newedges)
#{
#  new("graphNEL",nodes=as(newnodes,"vector"),
#         edgeL=as(newedges,"list"))
#}

  ## we define a virtual graph class -- to hold generic functions
  setClass("graph", representation(edgemode="character",
                                   "VIRTUAL"))

  if( !exists("edgemode", mode="function") )
      setGeneric("edgemode", function(object)
                 standardGeneric("edgemode"))
  setMethod("edgemode", "graph", function(object) object@edgemode)


  setGeneric("edgemode<-", function(object, value)
             standardGeneric("edgemode<-"))

  setReplaceMethod("edgemode", c("graph", "character"),
             function(object, value) {
                 if(length(value) != 1)
                     stop("edgemode is the wrong length")
                 if( !(value %in% c("directed", "undirected")) )
                     stop(paste("supplied mode is", value,
                                "it must be either directed or undirected"))
                 object@edgemode <- value
                 object
             })

  ## a node-edge-list graph
  ##the edgeL is a list, with edges, weights etc

  setClass("graphNEL",representation(nodes="vector",edgeL="list"),
            contains="graph")

  setMethod("initialize", "graphNEL", function(.Object, nodes=character(0),
       edgeL = vector("list",length=0), edgemode) {
       if( missing(edgemode) )
           edgemode <- "undirected"
       if( !missing(edgeL) && length(edgeL) > 1 ) {
           if(length(nodes) != length(edgeL) )
               stop("nodes and edges must align")
           nameE <- names(edgeL)
           if( !is.null(nameE) && !all( nameE %in% nodes) )
               stop("names of nodes and edges must agree")
           if( !is.null(nameE) )
               edgeL <- edgeL[nodes]
           if( is.character(edgeL[[1]]) )
               edgeL <- lapply(edgeL, function(x) match(x, nodes))
       }
       if( missing(edgeL) )
           edgeL = vector("list",length=0)
       if( missing(nodes) )
           nodes = character(0)

       .Object@nodes <- nodes
       .Object@edgeL <- edgeL
       .Object@edgemode <- edgemode
       .Object})

  setGeneric("nodes", function(object) standardGeneric("nodes"))
  setMethod("nodes", "graphNEL", function(object) object@nodes)

  setGeneric("nodes<-", function(object, value)
             standardGeneric("nodes<-"))
  setReplaceMethod("nodes", c("graphNEL", "character"),
             function(object, value) {
                 if(length(value) != length(object@nodes))
                     stop("need as many names as there are nodes")
                 if(any(duplicated(value)))
                     stop("node names must be unique")
                 object@nodes <- value
                 names(object@edgeL) <- value
                 object})

  if (!exists("edges", mode="function"))
      setGeneric("edges", function(object, which)
                 standardGeneric("edges"))

  ##the graphNEL representation stores edges as indexes into the
  ##node label vector
  setMethod("edges", c("graphNEL", "missing"), function(object, which) {
      gNodes <- object@nodes
      lapply(object@edgeL, function(x) gNodes[x$edges])})

  setMethod("edges", c("graphNEL", "character"),
            function(object, which) {
                gNodes <- nodes(object)
                lapply(object@edgeL[which], function(x) gNodes[x$edges])})


  ##we are going to need some notion of degree
  if( !isGeneric("degree") )
    setGeneric("degree", function(object, Nodes) standardGeneric("degree"))

  ##handle directed graphs by a list inDegree and outDegree
  setMethod("degree", signature(object="graph", Nodes="missing"),
      function(object)  {
          ns <- nodes(object)
          nl <- edges(object)
          deg <- sapply(nl, length)
          names(deg) <- ns
          if( object@edgemode == "undirected" )
              return(deg)
          else if( object@edgemode == "directed" ) {
              b1<- unlist(nl)
              b2<- table(b1)
              inDegree <- rep(0, length(ns))
              names(inDegree) <- ns
              inDegree[names(b2)]<-b2
              return(list(inDegree=inDegree, outDegree=deg))
          }
          stop(paste("edgemode", object@edgemode, "is not valid"))
     })

  setMethod("degree", "graph",  function(object, Nodes) {
       nl <- edges(object)
       nls <- nl[Nodes]

       deg <- sapply(nls, length)
       names(deg) <- Nodes
       if( object@edgemode == "undirected" )
           return(deg)
       else if( object@edgemode == "directed" ) {
           b1 <- unlist(nl)
           b2 <- table(b1)[Nodes]
           inDegree <- rep(0, length(nls))
           names(inDegree) <- Nodes
           inDegree[names(b2)] <- b2
           return(list(inDegree=inDegree, outDegree=deg))
       }
       stop(paste("edgemode", object@edgemode, "is not valid"))
     })

  setGeneric("edgeWeights", function(object, index)
      standardGeneric("edgeWeights"))

  setMethod("edgeWeights", "graphNEL", function(object, index) {
           if( missing(index) )
           	tlist <- object@edgeL
           else
                tlist <- object@edgeL[index]
           wts <- lapply(tlist, function(x) {
               wts <- x$weights
               if(is.null(wts)) {
                   wts <- rep(1, length(x$edges))
                   names(wts) <- x$edges
               }
               ## Always make sure that the weight vector
               ## has names attached
               if (is.null(names(wts)))
                   names(wts) <- x$edges

               wts})
            wts})


##RG: some methods for adjcency and accessibility
  setGeneric("adj", function(object, index)
	standardGeneric("adj"))

  ## do we want the indices or the nodes?
  setMethod("adj", "graphNEL", function(object, index) {
        initI <- as.character(index)
        nd <- nodes(object)
        if( is.character(index) )
          index <- match(index, nd)
        if( is.na(index) || index < 0 || index > length(nd) )
          stop(paste("selected vertex", initI, "is not in the graph"))
	edges(object)[index]})

  setGeneric("acc", function(object, index)
	standardGeneric("acc"))


##  setMethod("acc", "graph", function(object, index) {
##       visit <- function(ind) {
##          marked[ind] <<- TRUE
##          alist <- adj(object, ind)[[1]]
##          for( EDGE in alist) {
##            if( !marked[EDGE] ) {
##               visit(EDGE)
##               rval <<- c(EDGE, rval)
##            }
##          }
##        }
##        marked <- rep(FALSE, numNodes(object))
##        rval <- vector(length=0)
##        names(marked) <- nodes(object)
##        visit(index)
##        return(rval)
##   }, where = where)

##an iterative method ! yuck

  setMethod("acc", "graph", function(object, index) {
       nN <- numNodes(object)
       nNames<- nodes(object)
       marked<-rep(0, nN)
       distv <- rep(0, nN)
       names(distv) <- nNames
       distx <- 1
       names(marked) <- nNames
       nmkd <- 0
       marked[index] <- 1
       done <- FALSE
       while( !done ) {
         minds <- nNames[marked==1]
         for( node in minds) {
           avec <- adj(object, node)[[1]]
           avec <- avec[marked[avec]==0] #don't mark any already marked
           marked[avec] <- 1
           distv[avec] <- distx
         }
         marked[minds] <- 2
         distx <- distx+1
         newmk <- sum(marked==1)
         if( newmk == 0 )
           done <- TRUE
       }
       marked[index] <- 0 ##not the node itself
       return(distv[marked==2])
    })

setGeneric("DFS", function(object, node, checkConn=FALSE)
           standardGeneric("DFS"))

setMethod("DFS", c("graph", "character", "ANY"), function(object, node,
    checkConn) {
    nNames <- nodes(object)
    marked <- rep(NA, length(nNames))
    names(marked) <- nNames
    m1 <- match(node, nNames)
    if( is.na(m1) )
        stop(paste("node:", node, "is not in the graph"))

    ##this could be expensive
    if (checkConn) {
        c1 <- connComp(object)
        if(length(c1) != 1)
            stop("graph is not connected")
    }
    marked[m1] <- 0
    ##repeat until all are marked - marked has no NA's
    counter <- 1
    while( any(is.na(marked)) ) {
        fE <- boundary(nNames[!is.na(marked)], object)
        fE <- fE[sapply(fE, length) > 0]
        wh <- marked[names(fE)]
        v1 <- sort(wh)
        newN <- fE[[names(v1)[v1==max(v1)]]]
        marked[newN[1]] <- counter
        counter <- counter+1
    }
    return(marked)
})

setGeneric("edgeL", function(graph, index) standardGeneric("edgeL"))

setMethod("edgeL", "graphNEL", function(graph, index) {
    if( missing(index) )
        graph@edgeL
    else
        graph@edgeL[index]})

setGeneric("subGraph", function(snodes, graph) standardGeneric("subGraph"))

  ##the map from the labels to the integers, 1:n must be used
  ## you must renumber the edges of the subGraph
  ##from 1 to length(snodes)
  ## it is important to get the subsets of the edgeList items right
  setMethod("subGraph", c("character", "graphNEL"), function(snodes, graph) {
      nn<-nodes(graph)
      numN <- length(nn)
      if( is.character(snodes) ) {
         ma <- match(snodes, nn)
         if( any(is.na(ma)) )
             stop("error the nodes in the graph do not match the subnodes")
      }
      else {
         ma <- snodes
         if( any( snodes < 0 ) || any( snodes > numN ) )
            stop("subnodes must be between 1 and the number of nodes in G")
      }
      ##subset the edgeList elements here
      ed <- edgeL(graph)[ma]
      newed <- lapply(ed, function(x) { good <- x$edges %in% ma
                 lapply(x, function(y) y[good]) })

      ##map from the old indices - 1:nn to the new 1:length(snodes)
      t1 <- rep(NA, numN)
      t1[ma] <- 1:length(snodes)
      newed <- lapply(newed,
                  function(x) {ed <- t1[x$edges] ##new names
                               x$edges <- ed
                               lapply(x, function(y) {names(y) <- ed; y})})
      ##finally return
      new("graphNEL", nodes=nn[ma], edgeL=newed) })


  ###some other tools
   setGeneric("intersection", function(x, y) standardGeneric("intersection"))

   setMethod("intersection", c("graph", "graph"), function(x,y) {
       if( edgemode(x) != edgemode(y) )
           stop("both graphs must have the same edgemode")
       xN <- nodes(x)
       yN <- nodes(y)
       bN <- intersect(xN,yN)
       lb <- length(bN)
       if(lb != length(xN) || lb != length(yN))
           stop("graphs must have the same node set")
       yOrd <- match(xN, yN)
       xE <- edges(x)
       yE <- edges(y)[yOrd]
       rval <- vector("list", length=length(xE))
       for(i in 1:length(xE) ) {
           ans <- intersect(xE[[i]], yE[[i]])
           if( length(ans) > 0 )
               rval[[i]] <- list(edges=match(ans, xN),
                                 weights=rep(1, length(ans)))
           else
               rval[[i]] <- list(edges=numeric(0), weights=numeric(0))
       }
       names(rval) <- xN
       new("graphNEL", nodes=bN, edgeL=rval, edgemode=edgemode(x))
   })

  setGeneric("join", function(x, y) standardGeneric("join"))

  setMethod("join", c("graph", "graph"), function(x, y) {
      ex <- edgemode(x); ey <- edgemode(y);
      if( ex == ey )
          outmode <- ex
      else
          stop("cannot handle different edgemodes, yet")

      nX <- nodes(x)
      numXnodes <- length(nX)
      nY <- nodes(y)
      ## Combine the two sets of nodes, removing any duplications
      newNodes <- unique(c(nX, nY))

      eLX <- edgeL(x)
      eLY <- edgeL(y)

      newEdgeL <- eLX

      ## Can't just cat the edgeL's together like this
      ## as the node #s have all changed.
      if (length(eLY) > 0) {
          eLYnames <- names(eLY)
          for (i in 1:length(eLY)) {
              newEntry <- eLY[i]
              ## !! first need to adjust the targets on the edges
              newEdges <- newEntry[[1]]$edges
              if (length(newEdges) > 0) {
                  for (j in 1:length(newEdges)) {
                      curTo <- nY[newEdges[j]]
                      newTo <- match(curTo,newNodes)
                      if (is.na(newTo))
                          stop("Error reassigning duplicated nodes")
                      newEdges[j] <- newTo
                  }
              }
              newEntry[[1]]$edges <- newEdges

              ## now need to attach it to the list.  If this
              ## is a duplicated node, combine it with the
              ## original, otherwise add it ot the list
              if (length(newEdgeL) == 0)
                  newEdgeL <- newEntry
              else if (eLYnames[i] %in% nX) {
                  entry <- which(names(newEdgeL) == eLYnames[i])
                  if (length(entry) > 1)
                      stop("Duplicated node names in original graph")
                  curEntry <- newEdgeL[[entry]]
                  curEntry$edges <- c(curEntry$edges, newEntry[[1]]$edges)
                  curEntry$weights <- c(curEntry$weights,
                                        newEntry[[1]]$weights)
                  if (!is.null(curEntry))
                      newEdgeL[[entry]] <- curEntry
              }
              else
                  newEdgeL <- c(newEdgeL,newEntry)
          }
      }

      new("graphNEL", nodes=newNodes, edgeL=newEdgeL, edgemode=ex)
  })

  setGeneric("union", function(x, y) standardGeneric("union"))

  setMethod("union", c("graph", "graph"), function(x, y) {
      ex <- edgemode(x); ey <- edgemode(y);
      if( ex == ey )
          outmode <- ex
      else
          stop("cannot handle different edgemodes, yet")

      xN <- sort(nodes(x))
      yN <- sort(nodes(y))
      if( any(xN != yN) )
          stop("graphs must have the same nodes")
      xE <- edges(x)
      yE <- edges(y)
      rval <- vector("list", length=length(xE))
      names(rval) <- xN
      for(i in names(xE) ) {
          ans <- unique(c(xE[[i]], yE[[i]]))
          if( length(ans) > 0 )
              rval[[i]] <- list(edges = match(ans, xN), weights=rep(1,
                                                        length(ans) ))
          else
              rval[[i]] <- list(edges=numeric(0), weights=numeric(0))
      }
      names(rval) <- xN
      new("graphNEL", nodes=xN, edgeL=rval, edgemode=outmode)
  })

   setGeneric("complement", function(x) standardGeneric("complement"))

   setMethod("complement", c("graph"), function(x) {
       if( edgemode(x) != "undirected" )
           stop(paste("can't handle edgemode:", x@edgemode, "yet"))

       xN <- nodes(x)
       xE <- edges(x)
       rval <- vector("list", length=length(xE))
       names(rval) <- xN
       for( i in xN ) {
           ans <-xN[ !(xN %in% c(i, xE[[i]])) ]
           lena <- length(ans)
           if( lena > 0 )
               rval[[i]] <- list(edges=match(ans, xN),
                                 weights=rep(1, lena))
           else
               rval[[i]] <- list(edges=numeric(0), weights=numeric(0))
       }
       new("graphNEL", nodes=xN, edgeL=rval, edgemode=edgemode(x))
   })

  ##connected components
  setGeneric("connComp", function(object) standardGeneric("connComp"))

  setMethod("connComp", "graph", function(object) {
    ##if directed we do weak connectivity
    ##by transforming to the underlying undirected graph
    if( edgemode(object) == "directed")
        object = ugraph(object)
    NL <- nodes(object)
    marked <- rep(0, length(NL))
    names(marked) <- NL
    done <- FALSE
    rval <- vector("list", 1)
    cnode <- 1
    index <- 1
    nused <- numeric(0)
    while( !done ) {
	curracc <- acc(object, cnode)
        rval[[index]] <- curracc
        nused <- c(nused, cnode)
        index <- index + 1
        if( length(curracc) > 0 )
            marked[names(curracc)] <- 1
        marked[cnode] <- 1
        cnode <- match(0, marked)
        if( is.na(cnode) )
          done <- TRUE
    }
    nmR <- NL[nused]
    nc <- length(rval)
    rL <- vector("list", length=nc)
    for(i in 1:nc) rL[[i]]<-c(nmR[[i]], names(rval[[i]]))
    return(rL)
 })

  setGeneric("isConnected", function(object, ...)
             standardGeneric("isConnected"))
  setMethod("isConnected", "graph", function(object, ...) {
      if (length(connComp(object)) == 1)
          TRUE
      else
          FALSE
  })

  setGeneric("numNodes", function(object) standardGeneric("numNodes"))

  setMethod("numNodes", "graph", function(object) length(nodes(object)))

  setMethod("numNodes", "graphNEL", function(object) length(object@nodes))

  setGeneric("addNode", function(node, object)
    standardGeneric("addNode"))

  setMethod("addNode", c("character", "graphNEL"),
        function(node, object) {
            gN = nodes(object)
            already <- match(node, gN)
            if( any(!is.na(already)) )
                stop(paste(node[already], "is already a node"))
            ##add them on the end so we don't renumber
            gN = c(gN, node)
            edgeL <-  object@edgeL
            nNode <- length(node)
            nEd <- vector("list", length=nNode)
            names(nEd) <- node
            for(i in 1:nNode)
                nEd[[i]] <- list(edges=numeric(0), weights=numeric(0))
            edgeL <- c(edgeL, nEd)
            new("graphNEL", gN, edgeL, edgemode(object))
        })

  setGeneric("removeNode", function(node, object)
      standardGeneric("removeNode"))

  setMethod("removeNode", c("character", "graphNEL"),
       function(node, object) {
           ##first clear the node -- does the checking too
           nG <- clearNode(node, object)
           nN <- nodes(nG)
           wh <- match(node, nN)
           gN <- nN[-wh]
           nE <- nG@edgeL[-wh]
           ##now renumber the nodes as stored in the edgelist
           nE2 <- lapply(nE, function(el) {
               oldN <- nN[el$edges]
               el$edges <- match(oldN, gN)
               el
           })
           new("graphNEL", gN, nE2, edgemode(object))
       })

  setGeneric("clearNode", function(node, object)
  standardGeneric("clearNode"))

  setMethod("clearNode", c("character", "graphNEL"),
       function(node, object) {
           gN <- nodes(object)
           whN <- match(node, gN)
           if(any(is.na(whN)) )
               stop(paste(whN[is.na(whN)], "is not a node in the graph"))
            gE <- .dropEdges(whN, object@edgeL)
           gE[[whN]] = list(edges=numeric(0), weights=numeric(0))
           new("graphNEL", gN, gE, edgemode(object))
       })



  ##FIXME: vectorize
  setGeneric("removeEdge", function(from, to, graph)
  standardGeneric("removeEdge"))

  setMethod("removeEdge", c("character", "character", "graphNEL"),
            function(from, to, graph) {
                gN <- nodes(graph)
                wh <- match(c(from, to), gN)
                if( any(is.na(wh)) )
                    stop(paste(wh[is.na[wh]], "is not a node"))
                nE <- edges(graph, from)
                whD <- match(to, nE[[1]])
                if( is.na(whD) )
                    stop(paste("no edge from", from, "to", to))
                nEd <- graph@edgeL
                nEd[[from]]$edges <- nEd[[from]]$edges[-whD]
                nEd[[from]]$weights <- nEd[[from]]$weights[-whD]
                ##now if undirected we need to remove the other one
                if( edgemode(graph) == "undirected" ) {
                    nE <- edges(graph, to)
                    whD <- match(from, nE[[1]])
                    if( is.na(whD) )
                        stop("problem with graph edges")
                    nEd[[to]]$edges <- nEd[[to]]$edges[-whD]
                    nEd[[to]]$weights <- nEd[[to]]$weights[-whD]
                }
                new("graphNEL", gN, nEd, edgemode(graph))
            })


    setGeneric("addEdge", function(from, to, graph, weights)
        standardGeneric("addEdge"))


     ##FIXME: do  we care if the edge already exists?
     setMethod("addEdge", c("character", "character", "graphNEL", "numeric"),
         function(from, to, graph, weights) {
             gN <- nodes(graph)
             whF <- match(from, gN)
             if( any(is.na(whF)) )
                 stop(paste(from[is.na(whF)], "is not a node"))
             whT <- match(to, gN)
             if( any(is.na(whT)) )
                 stop(paste(to[is.na(whT)], "is not a node"))
             ##roll out the shorter one
             lenT <- length(to)
             lenF <- length(from)
             if( lenT > lenF ) {
                 from <- rep(from, lenT)
                 whF <- rep(whF, lenT)
             }
             if( lenF > lenT ) {
                 whT <- rep(whT, lenF)
                 to <- rep(to, lenF)
             }
             ##now the same
             lenN <- max(lenT, lenF)
             weights <- rep(weights, lenN)
             eL <- graph@edgeL
             for(i in 1:lenN) {
                 old <- eL[[from[i]]]
                 old$edges <- c(old$edges, whT[i])
                 old$weights <- c(old$weights, weights[i])
                 eL[[from[i]]] <- old
             }
             ##if undirected, then we need to go in both directions
             if( edgemode(graph) == "undirected")
                 for(i in 1:lenN) {
                     old <- eL[[to[i]]]
                     old$edges <- c(old$edges, whF[i])
                     old$weights <- c(old$weights, weights[i])
                     eL[[to[i]]] <- old
                 }

             new("graphNEL", gN, eL, edgemode(graph))
         })

   setGeneric("combineNodes", function(nodes, graph, newName)
         standardGeneric("combineNodes"))

  setMethod("combineNodes", c("character", "graphNEL", "character"),
        function(nodes, graph, newName) {
            if( length(newName) > 1 )
                stop("must have a single name")
            gN <- nodes(graph)
            whN <- match(nodes, gN)
            if( any(is.na(whN)) )
                stop(paste(from[is.na(whN)], "is not a node"))
            eL <- graph@edgeL
            outE <- eL[nodes]
            if( length(nodes) == 1 ) {
                warning("nothing to collapse")
                return(graph)
            }
            ##if undirected then we know everything
            if( edgemode(graph) == "directed" )
                inE <- inEdges(nodes, graph)
            else
                inE <- NULL
            g2 <- removeNode(nodes, graph)
            g2 <- addNode(newName, g2)
            nC <- length(nodes)
            oE <- NULL; oW <- NULL;
            ##seems very inefficient
            for(i in 1:nC) {
                oE <- c(oE, outE[[nodes[i]]]$edges)
                oW <- c(oW, outE[[nodes[i]]]$weights)
            }
            oE <- gN[oE]
            oEd <- match(nodes, oE)
            oEd <- oEd[!is.na(oEd)]
            if( length(oEd) > 0 ) {
                oE <- oE[-oEd]
                oW <- oW[-oEd]
            }
            g2 <- addEdge(newName, oE, g2, oW)
            ##if directed we need to fix up the in edges
            if( !is.null(inE) ) {
                nC <- length(inE)
                oE <- NULL; oW <- NULL
                nmE <- names(inE)
                for(i in 1:nC) {
                    oE <- c(oE, inE[[i]])
                    for( j in inE[[i]] )
                        oW <- c(oW, .edgeWeight(j, nmE[i], graph))
                }
                g2 <- addEdge(oE, newName, g2, oW)
            }
            g2})


    ##inEdges returns a list of all nodes with edges
    ##to the nodes in "node"
    setGeneric("inEdges", function(node, object)
       standardGeneric("inEdges"),)

    setMethod("inEdges", c("missing", "graphNEL"),
              function(node, object)
                  inEdges(nodes(object), object))

    setMethod("inEdges", c("character", "graphNEL"),
         function(node, object) {
             gN <- nodes(object)
             whN <- match(node, gN)
             if( any(is.na(whN)) )
                 stop(paste(from[is.na[whN]], "is not a node"))
             nN <- length(node)
             rval <- vector("list", length=nN)
             names(rval) <- node
             eL <- object@edgeL
             for(i in 1:nN) {
                 whOnes <- sapply(eL,
                   function(x) {
                       if( whN[i] %in% x$edges)
                           return(TRUE)
                       FALSE})

                 rval[[i]] <- gN[whOnes]
             }
             rval})


##print methods
  setMethod("show", "graphNEL",
  function(object)
   {
     numNull<-numNoEdges(object)
     numNodes<- numNodes(object)
     numEdge<-numEdges(object)
##     totalEdgesPossible<-numNodes*(numNodes-1)/2
##     nodeMostEdges<-mostEdges(object)
##     aveEdge<-aveNumEdges(object)
     cat("A graph with ", object@edgemode, " edges\n")
     cat("Number of Nodes = ",numNodes,"\n",sep="")
     cat("Number of Edges = ",numEdge,"\n",sep="")
##     cat("Number of Nodes with no edges = ",numNull,"\n",sep="")
##     cat("If the graph was completely linked, the number of edges would be ",format(totalEdgesPossible,big.mark=","),".\n",sep="")
##     cat("The node with the most edges is ",nodeMostEdges$id," with ", nodeMostEdges$maxLen, " edges.","\n",sep="")
##     cat("The average number of edges is ",format(aveEdge,digits=3),".\n",sep="")
   })

   .dropEdges <- function(x,z) {
       ans =  lapply(z,function(ww) {
           bad <- match(x, ww$edges)
           bad <- bad[!is.na(bad)]
           if(length(bad) > 0 )
               ans = list(edges= ww$edges[-bad],
                  weights = ww$weights[-bad])
           else
               ans = ww
       })
       ans}


    .edgeWeight <- function(from, to, graph)
    {
        gN <- nodes(graph)
        wF <- match(from, gN)
        if( is.na(wF) )
            stop(paste(from, "is not a node"))
        wT <- match(to, gN)
        if( is.na(wT) )
            stop(paste(to, "is not a node"))
        eL <- graph@edgeL[[from]]
        mt <- match(wT, eL$edges)
        if(is.na(mt) )
            stop(paste("no edge from", from, "to", to))
        eL$weights[mt]
    }









##Copyright 2002 Robert Gentleman, all rights reserved

##a class structure for graphs
##the class structure is based on GXL

    setClass("graphID")
    ##some functions to allow nodes and edges to have identifiers
    ##there are lots of problems with the integer version --
##    if( require(Ruuid, quietly=TRUE) ) {
##        setIs( "Ruuid", "graphID")
##        .haveUUID <- TRUE
##    }  else{
        setClassUnion("graphID", "integer")
        .haveUUID <- FALSE
##        warning("without Ruuid you may have problems with node or edge IDS")
##    }

    ##here we set up some global variables (that should be internal to
    ##and will be once we have namespaces
    assign("idenv", new.env(hash=TRUE))
    assign("idval", 1, env=idenv)

    if( .haveUUID ) {
        assign("startids", function(x) NULL)
        assign("newID", getuuid)
##        assign("nullgraphID", new("Ruuid"), pos="package:graph")
    } else {
        assign("startids", function(x) assign("idval", x, env=idenv))
        assign("newID", function() {
            val <- get("idval", env=idenv)
            assign("idval", val+1, env=idenv)
            return(val)
        })
        assign("nullgraphID", as.integer(-1))
        }

    ##for undirected graphs the toEdges and fromEdges lists are the same
    setClass("gNode",
           representation(nodeID="graphID",
                          nodeType="character",
                          toEdges="list",
                          fromEdges="list",
                          edgeOrder="numeric",
                          label="character" ),
           prototype = list(nodeID=as.integer(1), nodetype="unknown",
             edgeOrder=0, label=""))
    ##I think we need to separate directed from the type of the edge
    ##if directed=FALSE then the bNode and eNode are just ends, not
    ##beginning and ending nodes
    setClass("gEdge",
          representation(edgeID="graphID",
                         edgeType="character",
                         weight="numeric",
                         directed="logical",
                         bNode="graphID",    ##begin - if directed
                         eNode="graphID"),   ##end   - if directed
          prototype = list(edgeID=nullgraphID, edgeType="unknown",
          directed=FALSE, bNode=nullgraphID, eNode=nullgraphID, weight=1))

    ##setup the accessor functions

    if (is.null(getGeneric("edgeID")) && !exists("edgeID",
                                                 mode="function") )
        setGeneric("edgeID", function(object)
                   standardGeneric("edgeID"))
    setMethod("edgeID", "gEdge", function(object)
              object@edgeID)

    if (is.null(getGeneric("eNode")))
        setGeneric("eNode", function(object)
                   standardGeneric("eNode"))
    setMethod("eNode", "gEdge", function(object)
              object@eNode)
    if (is.null(getGeneric("bNode")))
        setGeneric("bNode", function(object)
                   standardGeneric("bNode"))
    setMethod("bNode", "gEdge", function(object)
              object@bNode)

   setGeneric("toEdges", function(object) standardGeneric("toEdges"))

   setMethod("toEdges", "gNode", function(object) object@toEdges)

   setGeneric("toEdges<-",
               function(object, value) standardGeneric("toEdges<-"))
   setReplaceMethod("toEdges", "gNode", function(object, value) {
      object@toEdges <- value
      object})

    setGeneric("fromEdges",
               function(object) standardGeneric("fromEdges"))
   setMethod("fromEdges", "gNode", function(object) object@fromEdges)

    setGeneric("fromEdges<-",
               function(object, value) standardGeneric("fromEdges<-"))
     setReplaceMethod("fromEdges", "gNode", function(object, value) {
      object@fromEdges <- value
      object})

      setGeneric("label", function(object) standardGeneric("label"))
   setMethod("label", "gNode", function(object) object@label)

     setGeneric("edgeOrder", function(object) standardGeneric("edgeOrder"))
   setMethod("edgeOrder", "gNode", function(object) object@edgeOrder)

    setGeneric("nodeID", function(object) standardGeneric("nodeID"))

   setMethod("nodeID", "gNode", function(object) object@nodeID)

    setGeneric("nodeType", function(object) standardGeneric("nodeType"))

   setMethod("nodeType", "gNode", function(object) object@nodeType)


#### hashtables -- very primitive start

    setClass("hashtable", representation(hashtable="environment"))
    setMethod("initialize", "hashtable", function(.Object) {
        .Object@hashtable=new.env(hash=TRUE)
        .Object})

   if( !exists("hash", mode="function") )
       setGeneric("hash", function(key, value, htable) standardGeneric("hash"))

    setMethod("hash", signature("ANY", "ANY", "hashtable"),
              function(key, value, htable) {
                  if(!is.character(key) )
                      key <- as.character(key)
                  assign(key, value, env= htable@hashtable)
              })

      if( !isGeneric("contents") && !exists("contents", mode="function") )
       setGeneric("contents", function(object)
                  standardGeneric("contents"))

    setMethod("contents", "hashtable",
              function(object) ls(env=object@hashtable))


#### define a general graph structure here
    setClass("generalGraph", representation(nodes="hashtable",
                                            edges="hashtable"),
                                            contains="graph")

    setMethod("initialize", "generalGraph", function(.Object,
             nodes=NULL, edges=NULL) {
        .Object@nodes = new("hashtable")
        .Object@edges = new("hashtable")
        browser()
         for(node in nodes )
            hash(nodeID(node), node,.Object@nodes )
        for(edge in edges )
            hash(edgeID(edge), edge,.Object@edges)
        .Object})

    ##coercion to generalGraph -- this will not be efficient
      setAs("graphNEL", "generalGraph", def=function(from) {
      nodes <- nodes(from)
      edges <- edges(from)
      eWts <- edgeWeights(from)
      nNodes <- length(nodes)
      nodeIDS <- as.list(rep(NA, nNodes))
      names(nodeIDS) <- nodes
      for(i in seq(along=nodes) ) nodeIDS[[i]] <- newID()
      nEdges <- numEdges(from)
      nodeObj <- vector("list", length=nNodes)
      edgeObj <- vector("list", length=nEdges)
      edgeCt <- 1
      tlist <- vector("list", length=nNodes) ##to edges
      flist <- vector("list", length=nNodes) ##from edges
      for( i in seq(along=nodes) ) {
          tlist[[i]] <- vector("list", 0)
          flist[[i]] <-  vector("list", 0)
      }
      if( edgemode(from) != "directed" )
          DIRECTED <- FALSE
      else
          DIRECTED <- TRUE

      for(j in seq(along=nodes) ) {
          eD <- edges[[j]]
          eW <- eWts[[j]]
          ##for undirected graphs we drop 1/2 the edges
          if( !DIRECTED ) {
              whE <- match(eD, nodes)
              eD <- eD[whE>j]
              eW <- eW[whE>j]
              whE <- whE[whE>j]
          }
          ct2 <- 1
          for( e1 in eD ) {
              eID <- newID();
              edgeObj[[edgeCt]] <-
                  new("gEdge", edgeID=eID, weight=eW[ct2], bNode =
                      nodeIDS[[j]], eNode = nodeIDS[[whE[ct2]]])
              flist[[j]] <- c(flist[[j]], eID)
              tlist[[whE[ct2]]] <- c(tlist[[whE[ct2]]], eID)
              ct2 <- ct2+1
              edgeCt <- edgeCt+1
          }
      }
      for(i in seq(along=nodes) )
          nodeObj[[i]] <- new("gNode", label=nodes[j], fromEdges=flist[[i]],
                     toEdges=tlist[[i]], nodeID=nodeIDS[[i]])

      new("generalGraph", nodes=nodeObj, edges=edgeObj)
  })

# as(gR, "generalGraph")

##copyright 2002 R. Gentleman, all rights reserved

## a simple implementation of the notions of cluster graphs
## and clustering using distance measures

## for the adjacency matrix view of undirected graphs the
## storage of the graph in lower triangular form is appealing

##  The lower triangle of the distance matrix stored by columns in a
##     single vector.  The vector has the attributes `"Size"', `"Diag"',
##     `"Upper"', `"Labels"' and `"class"' equal to `"dist"'.

## the lower triangle stored by columns
## for i < j <= n the dissimilarity between row i and j is
## x[n*(i-1)-i*(i-1)/2+j-i]
## size is the number of obs
## labels are their labels, Diag and Upper indicate how to print
##

##a function for Martin -- if it ever gets good enough.

##FIXME: should look at other matrix packages; this is just subsetting
##from a lower (or upper) triangular matrix representation -- which
##should be easier...but maybe it needs to be done in C

 "[.dist" <- function(x, i, j, drop=TRUE) {
    lend <- attr(x, "Size")
    if( missing(i) )
      if( missing(j) )
         return(x)
      else { ## return columns
         i <- 1:lend
      }
    if( missing(j) ) {
       j <- 1:lend
    }
    ## we have both -- return a matrix
    leni <- length(i)
    lenj <- length(j)
    outl <- leni*lenj
    iuse <- rep(i, length.out=outl)
    juse <- rep(j, rep(leni, lenj), length.out=outl)

    igtj <- iuse > juse
    newi <- ifelse(!igtj, iuse, juse)
    newj <- ifelse(igtj, iuse, juse)
    lend <- attr(x, "Size")
    subs <- lend*(newi-1)-newi*(newi-1)/2+newj-newi
    zeros <- newi==newj
    rval <- rep(0, length(subs))
    subs <- subs[!zeros]
    sdata <- unclass(x)[subs]
    rval[!zeros]<-sdata
    labels <- attr(x, "Labels")
    if( drop && (lenj == 1 || leni==1) ) {
	out<-rval
        if( leni == 1 )
          names(out) <- labels[j]
        else
          names(out) <- labels[i]
    }
    else {
        out <- matrix(rval, nc=lenj, nr=leni)
        dimnames(out) <- list(labels[i], labels[j])
   }
   out
 }

##rely on .initGraph having been called first to set up the
##classes and generics

 setClass("distGraph",
     representation( Dist = "dist"),
          prototype=list(edgemode="undirected"),
          contains="graph")

 setMethod("nodes", "distGraph", function(object)
      attr(object@Dist, "Labels" ))

 setGeneric("Dist", function(object) standardGeneric("Dist"))

 setMethod("Dist", "distGraph", function(object)
    object@Dist)

  setMethod("show", "distGraph", function(object) {
    cat("distGraph with ", attr(object@Dist, "Size"),
   " nodes \n", sep="")})

  setGeneric("threshold", function(object, k)
  standardGeneric("threshold"))

  setMethod("threshold", "distGraph", function(object, k) {
        nd <- object@Dist
        nd[nd > k ] <- 0
        new("distGraph", Dist=nd)
     })


  setMethod("numNodes", "distGraph", function(object)
     attr(Dist(object), "Size"))

  setMethod("adj", "distGraph", function(object, index) {
        nodenames<- nodes(object)
        if( is.character(index) )
          index <- match(index, nodenames)
        if( !is.numeric(index) )
	   stop("index not recognized")

        adjM <- object@Dist[index,]
        if( is.matrix(adjM) )
           adjL <- split(adjM, row(adjM))
        else
           adjL <- list(adjM)

        for(i in 1:length(adjL) )
          adjL[[i]] <- names(adjL[[i]])[adjL[[i]] > 0 ]
        return(adjL) })

   setMethod("edges", c("distGraph", "missing"), function(object, which) {
       nN <- nodes(object)
       eL <- lapply(nN, function(x) adj(object, x)[[1]])
       names(eL) <- nN
       return(eL) })

   setMethod("edges", c("distGraph", "character"), function(object, which) {
       nN <- nodes(object)
       wh<-match(which, nN)
       if( any(is.na(wh)) )
           stop("not all nodes are in the supplied graph")
       eL <- lapply(which, function(x) adj(object, x)[[1]])
       names(eL) <- which
       eL})

 if( !isClass("graph") )
   stop("cannot initialize clusterGraph without graph")

 setClass("clusterGraph",
     representation( clusters = "list"), contains="graph",
          prototype=list(edgemode="undirected"))

 setMethod("nodes", "clusterGraph", function(object)
    as.character(unlist(object@clusters)))

 setMethod("edges", c("clusterGraph", "missing"), function(object, which) {
     edges<-list()
     for(clust in object@clusters) {
         cc <- as.character(clust)
         for(i in seq(along=cc) )
             edges[[cc[i]]] <- cc[-i]
     }
     edges})

 setMethod("edges", c("clusterGraph", "character"), function(object, which) {
     nN <- nodes(object)
     wh <- match(which, nN)
     if( any(is.na(wh)) )
         stop("not all nodes are in the supplied graph")
     edges<-list()
     for(clust in object@clusters) {
         cc <- as.character(clust)[which]
         for(i in seq(along=cc) )
             edges[[cc[i]]] <- cc[-i]
     }
     edges})


##FIXME: this should be done from distances, but for now...)
##eg, if a distance matrix was supplied we could use that to define
##edge weights -- as that seems appropriate

 setMethod("edgeWeights", "clusterGraph", function(object, index) {
     edg <- edges(object)
     if( !missing(index) )
         edg <- edg[index]

     ans <- lapply(edg, function(x) { ans <- rep(1, length(x));
                                      names(ans) <- x; ans})
     ans})

 setMethod("subGraph", c("character", "clusterGraph"),
           function(snodes, graph) {
               cList <- graph@clusters
               cL <- lapply(cList, function(x) intersect(x, snodes))
               graph@clusters <- cL
               graph})

 setMethod("numNodes", "clusterGraph", function(object)
    sum(sapply(object@clusters, length)))

 setMethod("adj", "clusterGraph", function(object, index)
    for(cl in object@clusters) if( index %in% cl ) cl)

 ##for cluster graphs, acc and adj are the same
 setMethod("acc", "clusterGraph", function(object, index)
      for(cl in object@clusters) if( index %in% cl ) cl)

 setMethod("connComp", "clusterGraph", function(object)
      object@clusters)

  setMethod("show", "clusterGraph",
  function(object)
   {
     numNull<-numNoEdges(object)
     numNodes<- numNodes(object)
     numEdge<-numEdges(object)
     cat("A graph with ", object@edgemode, " edges\n")
     cat("Number of Nodes = ",numNodes,"\n",sep="")
     cat("Number of Edges = ",numEdge,"\n",sep="")
   })
################################################################
# function:
# aveNumEdges takes one parameter:
#   objgraph is the graph object
# aveNumEdges counts the number of edges in the graph and divides
# that by the number of nodes in the graph to give the
# average number of edges.  A double representing the average
# number of edges will be returned.
#
# created by: Elizabeth Whalen
# last updated: July 22, 2002
################################################################

aveNumEdges<-function(objgraph)
  numEdges(objgraph)/length(nodes(objgraph))

################################################################
# function:
# numEdges takes one parameter:
#   objgraph is the graph object
# numEdges counts the number of edges in the graph.  First, it
# sums up all the length of the edge list and this sum must be
# an even number because each edge is repeated twice.  To calculate
# the number of edges, the sum is divided by two.  An integer
# representing the number of edges will be returned.
#
# notes: The number of edges is divided by two because each edge
# is repeated twice, for most of our ways of representing undirected
# graphs
#
# created by: Elizabeth Whalen
# last updated: July 22, 2002
################################################################

numEdges <- function(graph)
{
  numbEdges <- length(unlist(edges(graph),use.names=FALSE))
  if( graph@edgemode == "undirected" )
      numbEdges<-numbEdges/2
  numbEdges
}

################################################################
# function:
# calcProb takes two parameters:
#   origgraph is the original graph from which the subgraph was made
#   subgraph is the subgraph made from the original graph
# calcProb calculates the probability of having the number of edges
# found in the subgraph given that it was made from origgraph.
# The hypergeometric distribution is used to calculate the
# probability (using the pdf).
#
# created by: Elizabeth Whalen
# last updated: July 22, 2002
################################################################

calcProb <- function(subgraph, origgraph)
{
  origNumNodes<-length(nodes(origgraph))
  subNumNodes<-length(nodes(subgraph))

  origNumEdges<-numEdges(origgraph)
  subNumEdges<-numEdges(subgraph)

  dyads <- (origNumNodes * (origNumNodes - 1) / 2) - origNumEdges
  sampledyads <- subNumNodes * (subNumNodes - 1) / 2

  prob<-dhyper(subNumEdges,origNumEdges,dyads,sampledyads)

  prob
}

################################################################
# function:
# calcSumProb takes two parameters:
#   origgraph is the original graph from which the subgraph was made
#   subgraph is the subgraph made from the original graph
# calcSumProb calculates the probability of having greater than or equal
# to the number of edges found in the subgraph given that it was made
# from origgraph.
# The hypergeometric distribution is used to calculate the summed
# probability (using the cdf).
#
# notes: This calculates the upper tail of the hypergeometric
# distribution.
#
# created by: Elizabeth Whalen
# last updated: July 22, 2002
################################################################

calcSumProb <- function(subgraph, origgraph)
{
  origNumNodes<-length(nodes(origgraph)) #g
  subNumNodes<-length(nodes(subgraph))   #gs

  origNumEdges<-numEdges(origgraph)      #L
  subNumEdges<-numEdges(subgraph)        #Ls

  dyads <- (origNumNodes * (origNumNodes - 1) / 2) - origNumEdges
  sampledyads <- subNumNodes * (subNumNodes - 1) / 2

  prob<-phyper(subNumEdges,origNumEdges,dyads,sampledyads,lower.tail=FALSE)

  prob
}

################################################################
# function:
# mostEdges takes one parameter:
#   objGraph is the graph object
# mostEdges finds the node that has the most edges in the graph.
# The index of the node, the node id (ex. affy id, locus
# link id, ...), and the number of edges for that node is returned
# in a list.
#
# created by: Elizabeth Whalen
# last updated: August 2, 2002
################################################################

mostEdges<-function(objGraph)
{
  oEdges<-edges(objGraph)
  lens <- sapply(oEdges, length)
  mx <- max(lens)
  return(names(oEdges)[match(mx, lens)])
}

################################################################
# function:
# numNoEdges takes one parameter:
#   objGraph is the graph object
# numNoEdges calculates the number of nodes that have an edge list
# of NULL (i.e. no edges) and returns an integer representing
# the number of NULL edge lists in the graph.
#
# created by: Elizabeth Whalen
# last updated: July 22, 2002
################################################################

numNoEdges<-function(objGraph)
{
 els <- sapply(edges(objGraph), length)
 sum(els==0)
}



################################################################
# function:
# boundary takes two parameters:
#   graph is the original graph from which the subgraph will be created
#   subgraph either the subgraph or the nodes of the subgraph
# boundary returns a list of length equal to the number of nodes in the
#   subgraph. Each element is a list of the nodes in graph
#
# created by: Elizabeth Whalen
# last updated: Feb 15, 2003, RG
################################################################

boundary<-function(subgraph, graph)
{
  if ( !is(graph, "graph") )
    stop("The second parameter must be an object of type graph.")

  if( is(subgraph, "graph") )
      snodes <- nodes(subgraph)
  else if( is.character(subgraph) )
      snodes <- subgraph
  else
      stop("wrong type of first argument")

  if( any( !(snodes %in% nodes(graph)) ) )
      stop("the supplied subgraph is not a subgraph of the supplied graph")

  subE <- edges(graph)[snodes]

  lapply(subE, function(x) x[!(x %in% snodes)] )
}


##check to see if any edges are duplicated, as we often don't have
##good ways to deal with that
duplicatedEdges <- function(graph) {
    if( !is(graph, "graphNEL") )
        stop("only graphNEL supported for now")

    for(e in graph@edgeL)
        if( any(duplicated(e$edges)) )
            return(TRUE)
    return(FALSE)
}

##ugraph: take a directed graph and return the underlying undirected graph
ugraph <- function(graph)
{
    if( edgemode(graph) == "undirected")
        return(graph)
    if( !is(graph, "graphNEL") )
        stop("only graphNEL supported for now")

    if( duplicatedEdges(graph) )
        stop("there are duplicated edges, cannot handle multigraphs")

    eL <- graph@edgeL
    nN <- nodes(graph)
    ##just in case they are not in the same order!
    eL <- eL[nN]
    for( i in 1:length(eL) ) {
        cNode <- nN[i]
        e <- eL[[i]]
        if( length(e$edges) > 0 ) {
            wh <- nN[e$edges]
            for(j in 1:length(wh) ) {
                eX <- eL[[wh[j]]]$edges
                ##the current node is i so check for it
                if( i %in% eX)
                    next
                eL[[wh[j]]]$edges <- c(i, eX)
                eL[[wh[j]]]$weights <- c(e$weights[j],
                                         eL[[wh[j]]]$weights)
            }
        }
    }
    edgemode(graph) <- "undirected"
    graph@edgeL <- eL
    return(graph)
}


 setGeneric("edgeMatrix",
            function(object, duplicates=FALSE) standardGeneric("edgeMatrix"))

 setMethod("edgeMatrix", c("graphNEL", "ANY"),
           function(object, duplicates) {
                   ## Return a 2 row numeric matrix (from, to, weight)
               ed <- object@edgeL
               ##reorder to the same order as nodes
               ed <- ed[nodes(object)]
               nN <- length(ed)
               elem <- sapply(ed, function(x) length(x$edges))
               from <- rep(1:nN, elem)
               to <- unlist(lapply(ed, function(x) x$edges))
               ans <- rbind(from, to)
               ##we duplicate edges in undirected graphNEL
               ##so here we remove them
               if( edgemode(object) == "undirected"  && !duplicates) {
                   t1 <- apply(ans, 2, function(x) {paste(sort(x),
                                                           collapse="+")})
                   ans <- ans[ ,!duplicated(t1)]
               }
               ans
           })


  setMethod("edgeMatrix", c("clusterGraph", "ANY"),
            function(object, duplicates) {
                cls<-object@clusters
                nd <- nodes(object)
                ans <- numeric(0)
                for(cl in cls) {
                    idx <- match(cl, nd)
                    nn <- length(idx)
                    v1 <- rep(idx[-nn], (nn-1):1)
                    v2 <- numeric(0)
                    for( i in 2:nn)
                        v2 <- c(v2, i:nn)
                    v2 <- idx[v2]
                    ta <- rbind(v1, v2)
                    if( is.matrix(ans) )
                        ans <- cbind(ans, rbind(v1, v2))
                    else
                        ans <- rbind(v1, v2)
                }
                dimnames(ans) <- list(c("from", "to"), NULL)
                ans
            })


edgeWeightVector <- function (g,...)
{
    m <- edgeMatrix(g,...)
    w <- edgeWeights(g)
    n <- apply(m,2,function(x)paste(x,collapse=
       ifelse(edgemode(g)=="directed","->","--"),sep=""))
    o <- rep(NA, ncol(m))
    for (i in 1:ncol(m)) o[i] <- edgeWeights(g, m[1, i])[[1]][as.character(m[2,
        i])]
    names(o) <- n
    o
}
#
# GXL support
#
## need methods on connections
     setClass("file")
     setClass("connection")
     setIs("file","connection")
## fromGXL returns the graphNEL object only, and it may
##  need to return more properties (7 mar 03)
     setGeneric("fromGXL",function(con)standardGeneric("fromGXL"))
     setMethod("fromGXL", "connection", function(con)
       {
       require("XML") || stop("XML package needed")
       xmlEventParse(paste(readLines(con),collapse=""),NELhandler(),asText=TRUE)$asGraphNEL()
       })
## dumpGXL returns an R list with all? properties
     setGeneric("dumpGXL",function(con)standardGeneric("dumpGXL"))
     setMethod("dumpGXL", "connection", function(con)
       {
       require(XML)
       xmlEventParse(paste(readLines(con),collapse=""),NELhandler(),asText=TRUE)$dump()
       })
## validate against the dtd
     setGeneric("validateGXL",function(con)standardGeneric("validateGXL"))
     setMethod("validateGXL", "connection", function(con)
       {
       require(XML)
# maybe need a try here, xmlTreeParse dumps the whole stream when it hits an error
       tmp <- xmlTreeParse(paste(readLines(con),collapse=""),asText=TRUE,
              validate=TRUE)
       })
#
#  exporting
#
    setGeneric("toGXL", function(object)standardGeneric("toGXL"))
    setMethod("toGXL", "graphNEL", function(object)
       gxlTreeNEL(object))


gxlTreeNEL <- function(gnel) {
 require(XML)
 nds <- nodes(gnel)
 eds <- lapply(edges(gnel),unique)
 enms <- names(eds)
 out <- xmlTree("gxl", #dtd="http://www.gupro.de/GXL/gxl-1.0.1.dtd",
   namespaces=c(gxl="http://www.w3.org/1999/xlink"))
#<!DOCTYPE gxl SYSTEM "http://www.gupro.de/GXL/gxl-1.0.1.dtd">
#<gxl xmlns:xlink="http://www.w3.org/1999/xlink">
 out$addTag("gxl",close=FALSE)
 out$addTag("graph", attrs=c(id="graphNEL", edgemode=
          as.character(edgemode(gnel))), close=FALSE)
 for (i in 1:length(nds))
   {
   out$addTag("node", attrs=c(id=nds[i]), close=FALSE)
   out$closeTag()
   }
 ued <- 0
 for (i in 1:length(eds))
   {
   if (length(eds[[i]])>0) for (j in 1:length(eds[[i]]))
     {
     ued <- ued + 1
     etag <- paste("e",ued,sep="")
     out$addTag("edge", attrs=c(id=etag,from=enms[i],
        to=eds[[i]][j]), close=FALSE)
     out$closeTag()
     }
   }
 out$closeTag() # graph
 out$closeTag() # gxl
 out
}


NELhandler <- function () 
{
#
# handler for xmlEventParse for the node and edge
# components of GXL 1.0.1
#  REVISED 15 Aug 03 to recognize and return edgemode
#  REVISED 1 apr 03 so that weights come into graphNEL if
#   present as weights child of edges in GXL
#
#
# local store
#
    graphID <- NULL
    curNode <- NULL
    curEdge <- NULL
    curAttr <- NULL
    inNode <- FALSE
    inEdge <- FALSE
    inAttr <- FALSE
    nodeL <- list()
    edgeL <- list()
    edgemode <- NULL
#
# handler elements: start elements are cased for
#  graph, node, attr, or edge
# text is limited in the simple example to the attr tag,
#   which lives under a node or an edge
#
    startElement = function(x, atts, ...) {
        if (x == "graph") 
            {
            graphID <<- atts["id"]
            edgemode <<- atts["edgemode"]
            }
        else if (x == "node") {
            inNode <<- TRUE
            nodeL[[atts["id"]]] <<- list()
            curNode <<- atts["id"]
        }
        else if (x == "attr") {
            inAttr <<- TRUE
            curAttr <<- atts["name"]
        }
        else if (x == "edge") {
            inNode <<- FALSE
            inEdge <<- TRUE
#
# you should get rid of the dependence on atts["id"] to allow edgeids=FALSE GXL to succeed
# consider an automatic edge labeler
#
            edgeL[[atts["id"]]] <<- list()
            edgeL[[atts["id"]]][["span"]] <<- c(atts["from"], 
                atts["to"])
            curEdge <<- atts["id"]
        }
    }
    text = function(x, atts, ...) {
        if (inNode & inAttr & nchar(x) > 0) 
            nodeL[[curNode]][[curAttr]] <<- x
        else if (inEdge & inAttr & nchar(x) > 0) 
            edgeL[[curEdge]][[curAttr]] <<- c(edgeL[[curEdge]][[curAttr]], 
                x)
    }
    endElement = function(x, ...) {
        if (x == "attr") 
            inAttr <<- FALSE
        else if (x == "node") 
            inNode <<- FALSE
        else if (x == "edge") 
            inEdge <<- FALSE
    }
    dump = function() {
        list(graphID = graphID, nodeL = nodeL, edgeL = edgeL, edgemode=edgemode)
    }
    asGraphNEL = function() {
#
# just handles weights children of edges when present, needs to handle
# arbitrary children of edges and nodes
#
        require(graph)
        ns <- names(nodeL)
        if (length(edgeL) == 0)
        return(new("graphNEL", nodes = ns, edgemode = edgemode))
        src <- sapply(edgeL, function(x) x$span["from"])
        dest <- sapply(edgeL, function(x) x$span["to"])
        wts <- sapply(edgeL, function(x) x[["weights"]])
        IGNWTS <- FALSE
        if (any(sapply(wts,function(x)is.null(x)))) IGNWTS <- TRUE
        if (!IGNWTS) sw <- split(wts, src)
        edl <- split(dest, src)
        nne <- names(edl)
        nl <- list()
        if (!IGNWTS) for (i in 1:length(edl)) nl[[nne[i]]] <- list(edges = match(edl[[i]], 
            ns), weights = as.numeric(sw[[i]]))
        else for (i in 1:length(edl)) nl[[nne[i]]] <- list(edges = match(edl[[i]], 
            ns))
	chkENconsis <- match(ns, names(nl))
        if (any(inn <- is.na(chkENconsis)))
		{
		badinds <- (1:length(chkENconsis))[inn==TRUE]
		for (i in 1:length(badinds))
			nl[[ ns[ badinds[i] ] ]] <- character(0)
		}
        g <- new("graphNEL", nodes = ns, edgeL = nl, edgemode = edgemode)
        if (!validGraph(g)) stop("GXL did not define a valid graph package graphNEL object.\nMost likely there is a failure of reciprocity for edges in\nan undirected graph.  If there is a node for edge from A to B\nin an undirected graphNEL, there must also be an edge from B to A.")
	g
    }
    list(startElement = startElement, endElement = endElement, 
        text = text, dump = dump, asGraphNEL = asGraphNEL)
}
##Copyright R. Gentleman, 2002, all rights reserved

##code for generating random graphs

##two different methods of generating random graphs on a given set of
##nodes
##M1: randomly generate edges from the choose(|V|, 2) possible edges
##M2: given V and a set of shared attributes M, select for each node
##    of V a subset of M. Then compute edges according to whether
##    nodes share common elements of M


##M2: sample "properties" from M with prob p

randomGraph <- function(V, M, p, weights = TRUE)
{
    if( any(duplicated(V)) )
       stop("node names must be unique")

    lenM <- length(M)
    lenV <- length(V)
    nSel <- lapply(V, function(x)
                 M[sample(c(TRUE,FALSE), lenM, TRUE, prob=c(p, 1-p))])
    lens <- sapply(nSel, length)
    objs <- unlist(nSel)
    wh <- rep(1:lenV, lens)
    rval <- vector("list",lenV)
    names(rval) <- V
    tmp <- split(wh, objs)
    for( vec in tmp )
      for(i in vec)
       for(j in vec)
          if( i != j ) {
              pos <- match(j, rval[[i]]$edges)
              if(is.na(pos) ) {
                  rval[[i]]$edges <- c(rval[[i]]$edges, j)
                  ln <-length(rval[[i]]$edges)
                  rval[[i]]$weights <- c(rval[[i]]$weights, 1)
                  names(rval[[i]]$weights)[ln] <- j
              }
              else
                  rval[[i]]$weights[[pos]] <- rval[[i]]$weights[[pos]]+1
          }
      new("graphNEL", nodes = V, edgeL=rval)
}

##  gg<-randomGraph(letters[1:10], 1:4, .3)




##generate edges at random according to some probability
##there are choose(numN, 2) possible edges; to make life simple,
##(but less efficient), we first make up a matrix of all possible node edge
##combinations, then select entries and finally make the graph

randomEGraph <- function(V, p)
{
  numN <- length(V)
  numE <- choose(numN, 2)
  inds <- 1:numN
  fromN <- rep(inds[-numN], (numN-1):1)
  s<- numeric(0)
  for(i in 2:numN) s<-c(s, i:numN)
  ## tmat is a 2 column matrix, the first column is the from node, the second
  ## the to node
  tmat <- cbind(fromN, s)
  wh<- sample(c(TRUE, FALSE), numE, TRUE, p=c(p,1-p))
  tmat <- tmat[wh,]
  numE <- sum(wh)
  rval <- vector("list", length=numN)
  for( i in 1:numE ) {
      ##first put in from -> to
      rval[[tmat[i,1]]]$edges <- c(rval[[tmat[i,1]]]$edges, tmat[i,2])
      ln <- length(rval[[tmat[i,1]]]$edges)
      rval[[tmat[i,1]]]$weights <- c(rval[[tmat[i,1]]]$weights, 1)
      names(rval[[tmat[i,1]]]$weights)[ln] <- tmat[i,2]
      ##since undirected, put in to -> from
      rval[[tmat[i,2]]]$edges <- c(rval[[tmat[i,2]]]$edges, tmat[i,1])
      ln <- length(rval[[tmat[i,2]]]$edges)
      rval[[tmat[i,2]]]$weights <- c(rval[[tmat[i,2]]]$weights, 1)
      names(rval[[tmat[i,2]]]$weights)[ln] <- tmat[i,1]
  }
  names(rval) <- V
  new("graphNEL", nodes = V, edgeL = rval)
}

## g2 <- randomEGraph(letters[1:10], .2)



