.packageName <- "annaffy"
.aaf.raw <- function(probeids, chip, type) {

    if (! do.call("require", list(chip)))
        stop(paste("Data library couldn't be loaded:", chip))

    environment <- paste(chip, type, sep="")

    do.call("multiget", list(probeids, as.name(environment)))
}

.aaf.character <- function(probeids, chip, type, class) {

    anns <- .aaf.raw(probeids, chip, type)
    result <- vector("list", length(probeids))
    attrs <- list(class = class)
    for(i in 1:length(probeids)) {
        ann <- anns[[i]]
        if( is.na(ann[1]) )
            ann <- character(0)
        attributes(ann) <- attrs
        result[[i]] <- ann
    }
    class(result) <- "aafList"
    
    return(result)
}

.aaf.integer <- function(probeids, chip, type, class) {

    anns <- .aaf.raw(probeids, chip, type)
    result <- vector("list", length(probeids))
    attrs <- list(class = class)
    for(i in 1:length(probeids)) {
        ann <- as.integer(anns[[i]])
        if( is.na(ann[1]) )
            ann <- integer(0)
        attributes(ann) <- attrs
        result[[i]] <- ann
    }
    class(result) <- "aafList"
    
    return(result)
}

.aaf.goterm <- function (num) {

    if (!nchar(num))
        return(list())
    
    if (exists(num, GOBPID2TERM))
        return(list(name = get(num, GOBPID2TERM), type = "Biological Process"))
    
    if (exists(num, GOCCID2TERM))
        return(list(name = get(num, GOCCID2TERM), type = "Cellular Component"))
    
    if (exists(num, GOMFID2TERM))
        return(list(name = get(num, GOMFID2TERM), type = "Molecular Function"))
    
    return(list())
}

aaf.handler <- function (probeids, chip, name)
{
# This function keeps track of all the types of annotation data that can
# currently be handled. If called with no arguments, it returns a list of the
# names of annotation data that can be fetched. Otherwise, it dispatches the
# request to the appropriate handler function.

    if (missing(probeids))
    c("Probe", "Symbol", "Description", "Function", "Chromosome",
      "Chromosome Location", "GenBank", "LocusLink", "Cytoband",
      "UniGene", "PubMed", "Gene Ontology", "Pathway")
    else
        switch(name,
               Probe = aafProbe(probeids),
               Symbol = aafSymbol(probeids, chip),
               Description = aafDescription(probeids, chip),
               Function = aafFunction(probeids, chip),
               Chromosome = aafChromosome(probeids, chip),
               "Chromosome Location" = aafChromLoc(probeids, chip),
               GenBank = aafGenBank(probeids, chip),
               LocusLink = aafLocusLink(probeids, chip),
               Cytoband = aafCytoband(probeids, chip),
               UniGene = aafUniGene(probeids, chip),
               PubMed = aafPubMed(probeids, chip),
               "Gene Ontology" = aafGO(probeids, chip),
               Pathway = aafPathway(probeids, chip))
}

## Set generic methods

if( !isGeneric("getText") )
    setGeneric("getText", function(object) standardGeneric("getText"))

if( !isGeneric("getURL") )
    setGeneric("getURL", function(object) standardGeneric("getURL"))

if( !isGeneric("getHTML") )
    setGeneric("getHTML", function(object) standardGeneric("getHTML"))

if( !isGeneric("getTD") )
    setGeneric("getTD", function(object) standardGeneric("getTD"))

if( !isGeneric("getCSS") )
    setGeneric("getCSS", function(object) standardGeneric("getCSS"))

## Define methods for vector class

setMethod("getText", "ANY", function(object) {

    if( !length(object) )
        return("")
    return(paste(object, collapse = ", "))
})

setMethod("getURL", "ANY", function(object) {

    return(character(0))
})

setMethod("getHTML", "ANY", function(object) {
    
    if( is.double(object) )
        object <- signif(object, getOption("sigfigs"))
    if( !nchar(text <- getText(object)) )
        return("")
    if( length(url <- getURL(object)) )
        return(paste(paste("<a href=\"", url, "\">", text, "</a>", sep = ""), collapse = " "))
    else
        return(text)
})

setMethod("getTD", "ANY", function(object) {
    
    html <- getHTML(object)
    if (!nchar(html))
       html <- "&nbsp;"
    
    return(paste("<td class=\"", class(object), "\">", html, "</td>", sep = ""))       
})

setMethod("getCSS", "ANY", function(object) {
    
    return(character(0))       
})

## Define class aafList

setClass("aafList", "list", prototype = list())

setMethod("getText", "aafList", function(object) {
    
    if( !length(object) )
        return(character(0))
    result <- character(length(object))
    for(i in 1:length(object))
        result[i] <- getText(object[[i]])
    return(result)
})

setMethod("getURL", "aafList", function(object) {

    result <- character(0)
    for (i in 1:length(object))
        result <- c(result, getURL(object[[i]]))
    return(result)
})

setMethod("getHTML", "aafList", function(object) {
    
    if( !length(object) )
        return(character(0))
    result <- character(length(object))
    for(i in 1:length(object))
        result[i] <- getHTML(object[[i]])
    return(result)
})

setMethod("getTD", "aafList", function(object) {
    
    if( !length(object) )
        return(character(0))
    result <- character(length(object))
    for(i in 1:length(object))
        result[i] <- getTD(object[[i]])
    return(result)
})

setMethod("getCSS", "aafList", function(object) {
    
    return(getCSS(object[[1]]))
})

setMethod("[", "aafList", function(x, i, j, ..., drop = F) {

    result <- x@.Data[i]
    class(result) <- class(x)
    return(result)
})

setMethod("show", "aafList", function(object) {

    frame <- parent.frame()
    if( exists("showHistory", frame) )
        history <- get("showHistory", frame)
    else
        history <- integer(0)
    
    cat("An object of class \"", class(object), "\"\n", sep = "")
    if( length(object) )
        for(i in 1:length(object)) {
            showHistory <- c(history, i)
            cat("[[", paste(showHistory, collapse = "]][["), "]]\n", sep = "")
            show(object[[i]])
            cat("\n")
        }
    else
        cat("list()\n", sep = "")
})

## Define class aafProbe

setClass("aafProbe", "character", prototype = character(0))

aafProbe <- function(probeids) {
    
    probes <- as.list(probeids)
    for(i in 1:length(probes))
        class(probes[[i]]) <- "aafProbe"
    
    return(new("aafList", probes))
}

setMethod("getURL", "aafProbe", function(object) {
    
    url <- "https://www.affymetrix.com/LinkServlet?&probeset="
    
    if( !length(object) )
        return(character(0))
    return(paste(url, object, sep = ""))
})

## Define class aafSymbol

setClass("aafSymbol", "character", prototype = character(0))

aafSymbol <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "SYMBOL", "aafSymbol"))
}

## Define class aafDescription

setClass("aafDescription", "character", prototype = character(0))

aafDescription <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "GENENAME", "aafDescription"))
}

## Define class aafFunction

setClass("aafFunction", "character", prototype = character(0))

aafFunction <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "SUMFUNC", "aafFunction"))
}

## Define class aafChromosome

setClass("aafChromosome", "character", prototype = character(0))

aafChromosome <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "CHR", "aafChromosome"))
}

## Define class aafChromLoc

setClass("aafChromLoc", "integer", prototype = integer(0))

aafChromLoc <- function(probeids, chip) {
    
    return(.aaf.integer(probeids, chip, "CHRLOC", "aafChromLoc"))
}

## Define class aafGenBank

setClass("aafGenBank", "character", prototype = character(0))

aafGenBank <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "ACCNUM", "aafGenBank"))
}

setMethod("getURL", "aafGenBank", function(object) {

    url <- "http://www.ncbi.nlm.nih.gov/entrez/query.fcgi?cmd=search&db=nucleotide&term="
    urlsuffix <- "%5BACCN%5D&doptcmdl=GenBank"
    
    if( !length(object) )
        return(character(0))
    return(paste(url, object, urlsuffix, sep = ""))
})

## Define class aafLocusLink

setClass("aafLocusLink", "integer", prototype = integer(0))

aafLocusLink <- function(probeids, chip) {
    
    return(.aaf.integer(probeids, chip, "LOCUSID", "aafLocusLink"))
}

setMethod("getURL", "aafLocusLink", function(object) {
    
    url <- "http://www.ncbi.nlm.nih.gov/LocusLink/LocRpt.cgi?l="
    
    if( !length(object) )
        return(character(0))
    return(paste(url, object, sep = ""))
})

## Define class aafCytoband

setClass("aafCytoband", representation(band = "character",
                                       genbank = "character"),
         prototype = list(band = character(0),
                          genbank = character(0)))

aafCytoband <- function(probeids, chip) {
    
    band <- .aaf.raw(probeids, chip, "MAP")
    genbank <- .aaf.raw(probeids, chip, "ACCNUM")
    result <- vector("list", length(probeids))
    navals <- is.na(band)
    result[which(navals)] <- list(new("aafCytoband"))
    result[which(!navals)] <- list(list())
    for(i in which(!navals))
        attributes(result[[i]]) <- list(band = band[[i]], genbank = genbank[[i]], class = "aafCytoband")
    class(result) <- "aafList"
    
    return(result)
}

setMethod("getText", "aafCytoband", function(object) {

    if( !length(object@band) )
        return("")
    return(object@band)
})

setMethod("getURL", "aafCytoband", function(object) {

    url <- "http://www.ncbi.nlm.nih.gov/mapview/map_search.cgi?direct=on&query="
    urlsuffix <- "%5BACCN%5D"
    
    if( !length(object@band) )
        return(character(0))
    return(paste(url, object@genbank, urlsuffix, sep = ""))
})

setMethod("show", "aafCytoband", function(object) {
    
    cat("An object of class \"aafCytoband\"\n")
    cat("@band    ", object@band, "\n", sep = "\"")
    cat("@genbank ", object@genbank, "\n", sep = "\"")
})

## Define class aafUniGene

setClass("aafUniGene", "character", prototype = character(0))

aafUniGene <- function(probeids, chip) {
    
    return(.aaf.character(probeids, chip, "UNIGENE", "aafUniGene"))
}

setMethod("getURL", "aafUniGene", function(object) {
    
    url <- "http://www.ncbi.nlm.nih.gov/UniGene/clust.cgi?ORG="
    urlinter <- "&CID="
    
    if( !length(object) )
        return(character(0))
    return(paste(url, sub("[.]", urlinter, object), sep = ""))
})

## Define class aafPubMed

setClass("aafPubMed", "integer", prototype = integer(0))

aafPubMed <- function(probeids, chip) {
    
    return(.aaf.integer(probeids, chip, "PMID", "aafPubMed"))
}

setMethod("getURL", "aafPubMed", function(object) {
    
    url <- "http://www.ncbi.nih.gov/entrez/query.fcgi?tool=bioconductor&cmd=Retrieve&db=PubMed&list_uids="
    
    if( !length(object) )
        return(character(0))
    return(paste(url, paste(object, collapse = "%2c"), sep = ""))
})

setMethod("getHTML", "aafPubMed", function(object) {
    
    if( !length(object) )
        return("")
    return(paste("<a href=\"", getURL(object), "\">", length(object), "</a>", sep = ""))
})

setMethod("getCSS", "aafPubMed", function(object) {
    
    return("td.aafPubMed { text-align: center }")       
})

## Define class aafGO

setClass("aafGO", "aafList", prototype = list())

aafGO <- function(probeids, chip) {
    
    gos <- .aaf.raw(probeids, chip, "GO")
    results <- vector("list", length(probeids))
    attrs <- list(class = "aafGO")
    for(i in 1:length(probeids)) {
        go <- gos[[i]]
        results[[i]] <- list()
        if( !is.na(go[1]) ) {
            for(j in 1:length(go)) {
                nametype <- .aaf.goterm(go[j])
                if( length(nametype) ) {
                    result <- list()
                    attributes(result) <- list(id = go[j], name = nametype$name, type = nametype$type, evid = names(go)[j], class = "aafGOItem")
                    results[[i]] <- c(results[[i]], list(result))
                }
            }
        }
        attributes(results[[i]]) <- attrs
    }
    class(results) <- "aafList"
    
    return(results)
}

setMethod("getText", "aafGO", function(object) {

    result = callNextMethod()
    return(paste(result, collapse = ", "))
})

setMethod("getURL", "aafGO", function(object) {

    url <- "http://godatabase.org/cgi-bin/go.cgi?open_0="
    
    if( !length(object) )
        return(character(0))
    url <- paste(url, object[[1]]@id, sep = "")
    for(i in 2:length(object))
        url <- paste(url, object[[i]]@id, sep = "&open_0=")
    return(url)
})

setMethod("getHTML", "aafGO", function(object) {

    result = callNextMethod()
    return(paste(result, collapse = " "))
})

setMethod("getTD", "aafGO", function(object) {
    
    html <- getHTML(object)
    if (!nchar(html))
       html <- "&nbsp;"
    
    return(paste("<td class=\"", class(object), "\">", html, "</td>", sep = ""))       
})

setMethod("getCSS", "aafGO", function(object) {
    
    return("p.aafGOItem { margin-top: 1px; margin-bottom: 1px; padding-left: 10px; text-indent: -10px }")       
})

## Define class aafGOItem

setClass("aafGOItem", representation(id = "character",
                                     name = "character",
                                     type = "character",
                                     evid = "character"),
         prototype = list(id = character(0),
                          name = character(0),
                          type = character(0),
                          evid = character(0)))

setMethod("getText", "aafGOItem", function(object) {

    if( !length(object@id) )
        return("")
    return(paste(object@id, ": ", object@name, sep = ""))
})

setMethod("getURL", "aafGOItem", function(object) {

    url <- "http://godatabase.org/cgi-bin/go.cgi?open_0="
    
    if( !length(object@id) )
        return(character(0))
    return(paste(url, object@id, sep = ""))
})

setMethod("getHTML", "aafGOItem", function(object) {
    
    if( !length(object@id) )
        return("")
    return(paste("<p class=\"aafGOItem\"><a href=\"", getURL(object), "\" title=\"", object@type, " (", object@evid, ")\">", object@name, "</a></p>", sep = ""))
})

setMethod("show", "aafGOItem", function(object) {
    
    cat("An object of class \"aafGOItem\"\n")
    cat("@id   ", object@id, "\n", sep = "\"")
    cat("@name ", object@name, "\n", sep = "\"")
    cat("@type ", object@type, "\n", sep = "\"")
    cat("@evid ", object@evid, "\n", sep = "\"")
})

## Define class aafPathway

setClass("aafPathway", "aafList", prototype = list())

aafPathway <- function(probeids, chip) {
    
    pathways <- .aaf.raw(probeids, chip, "PATH")
    enzymes <- .aaf.raw(probeids, chip, "ENZYME")
    results <- vector("list", length(probeids))
    attrs <- list(class = "aafPathway")
    for(i in 1:length(probeids)) {
        pathway <- pathways[[i]]
        if( is.na(pathway[1]) ) {
            results[[i]] <- list()
        }
        else {
            name <- multiget(pathway, KEGGPATHID2NAME)
            enzyme <- enzymes[[i]][1]
            if( is.na(enzyme) )
                enzyme <- character(0)
            result <- vector("list", length(pathway))
            for(j in 1:length(pathway)) {
                result[[j]] <- list()
                attributes(result[[j]]) <- list(id = pathway[j], name = name[[j]], enzyme = enzyme, class = "aafPathwayItem")
            }
            results[[i]] <- result
        }
        attributes(results[[i]]) <- attrs
    }
    class(results) <- "aafList"
    
    return(results)
}

setMethod("getText", "aafPathway", function(object) {

    result = callNextMethod()
    return(paste(result, collapse = ", "))
})

setMethod("getHTML", "aafPathway", function(object) {

    result = callNextMethod()
    return(paste(result, collapse = " "))
})

setMethod("getTD", "aafPathway", function(object) {
    
    html <- getHTML(object)
    if (!nchar(html))
       html <- "&nbsp;"
    
    return(paste("<td class=\"", class(object), "\">", html, "</td>", sep = ""))       
})

setMethod("getCSS", "aafPathway", function(object) {
    
    return("p.aafPathwayItem { margin-top: 1px; margin-bottom: 1px; padding-left: 10px; text-indent: -10px }")       
})

## Define class aafPathwayItem

setClass("aafPathwayItem", representation(id = "character",
                                          name = "character",
                                          enzyme = "character"),
         prototype = list(id = character(0),
                          name = character(0),
                          enzyme = character(0)))

setMethod("getText", "aafPathwayItem", function(object) {

    if( !length(object@id) )
        return("")
    return(paste(object@id, ": ", object@name, sep = ""))
})

setMethod("getURL", "aafPathwayItem", function(object) {
    
    url <- "http://www.genome.ad.jp/dbget-bin/show_pathway?MAP"
    urlnoenzyme <- "http://www.genome.ad.jp/kegg/pathway/hsa/hsa"
    
    if( !length(object@id) )
        return(character(0))
    if( length(object@enzyme) )
        return(paste(url, object@id, "+", object@enzyme, sep = ""))
    return(paste(urlnoenzyme, object@id, ".html", sep = ""))
})

setMethod("getHTML", "aafPathwayItem", function(object) {
    
    if( !length(object@id) )
        return("")
    return(paste("<p class=\"aafPathwayItem\"><a href=\"", getURL(object), "\">", object@name, "</a></p>", sep = ""))
})

setMethod("show", "aafPathwayItem", function(object) {
    
    cat("An object of class \"aafGOItem\"\n")
    cat("@id     ", object@id, "\n", sep = "\"")
    cat("@name   ", object@name, "\n", sep = "\"")
    cat("@enzyme ", object@enzyme, "\n", sep = "\"")
})
.aaf.envbuild <- function() {

    subenv <- function (oldenv, names) {
    
        newenv <- new.env(TRUE, NULL);
        for (name in names) {
            value <- do.call("get", list(name, as.name(oldenv), inherits=F))
            do.call("assign", list(name, value, newenv))
        }
    
        return(newenv);
    }
    
    saveannaffy <- function (probeids, chip, type) {
    
        oldenvname <- paste(chip, type, sep="")
        newenvname <- paste("annaffy", type, sep = "")
        do.call("assign", list(newenvname, subenv(oldenvname, probeids)))
        do.call("save", list(newenvname, file = paste(newenvname, ".rda", sep = "")))
    }
    
    require(hgu95av2)
    
    probeids <- c("32547_at", "720_at", "40142_at", "38903_at", "34834_at", 
                  "37049_g_at", "2021_s_at", "38375_at", "41770_at",
                  "32354_at", "35256_at", "33201_at", "35214_at", "36387_at",
                  "40685_at", "1974_s_at", "32815_at", "39496_s_at",
                  "40806_at", "570_at", "35205_at", "34111_s_at", "34262_at",
                  "36912_at", "39001_at", "38412_at", "40160_at", "33603_at",
                  "40864_at", "41178_at", "37644_s_at", "40944_at",
                  "31693_f_at", "39440_f_at", "1099_s_at", "41157_at",
                  "417_at", "33742_f_at", "39600_at", "234_s_at", "35174_i_at",
                  "33511_at", "36097_at", "831_at", "131_at", "31537_at",
                  "37031_at", "34594_at", "38324_at", "34200_at", "40372_at",
                  "37238_s_at", "41394_at", "40112_at", "39687_at", "36640_at",
                  "768_at", "31654_at", "41316_s_at", "556_s_at", "31901_at",
                  "33759_at", "32954_at", "34745_at", "39651_at", "40651_s_at",
                  "33332_at", "35315_at", "31848_at", "35728_at", "1435_f_at",
                  "36662_at", "35168_f_at", "36962_at", "32733_at", "1027_at",
                  "198_g_at", "33725_at", "41095_at", "34826_at", "584_s_at",
                  "41018_at", "163_at", "36519_at", "154_at", "37307_at",
                  "1144_at", "34364_at", "40614_at", "32443_at", "33048_at",
                  "37628_at", "34195_at", "35441_at", "32988_at", "40357_at",
                  "33114_f_at", "1165_at", "35612_at", "37822_at", "33098_at",
                  "40675_r_at", "39501_f_at", "35492_at", "36848_r_at",
                  "36379_at", "34051_at", "1214_s_at", "34920_at", "262_at",
                  "35933_f_at", "40218_at", "33715_r_at", "36267_at",
                  "33012_at", "1495_at", "1221_at", "35815_at", "31622_f_at",
                  "41161_at", "31521_f_at", "33573_at", "40186_at", "34815_at",
                  "33973_at", "38283_at", "32692_at", "32498_at", "35353_at",
                  "40217_s_at", "33186_i_at", "37808_at", "39997_at",
                  "38225_at", "32230_at", "652_g_at", "38981_at", "31469_s_at",
                  "2090_i_at", "39722_at", "35857_at", "36508_at", "39271_at",
                  "36593_at", "38481_at", "908_at", "41531_at",
                  "AFFX-HUMGAPDH/M33197_M_at", "35494_at", "35854_at",
                  "37279_at", "40125_at", "35154_at", "34438_at", "36238_at",
                  "1899_s_at", "39073_at", "36568_at", "39619_at", "256_s_at",
                  "33806_at", "40563_at", "31687_f_at", "31552_at", "39919_at",
                  "37598_at", "500_at", "1914_at", "35596_at", "610_at",
                  "32073_at", "34840_at", "209_at", "40990_at", "35209_at",
                  "39350_at", "40873_at", "39005_s_at", "39421_at", "1470_at",
                  "37799_at", "32544_s_at", "37903_at", "35087_at", "151_s_at",
                  "41757_at", "38318_at", "40771_at", "36116_at", "880_at",
                  "508_at", "32879_at", "40401_at", "35488_at", "1096_g_at",
                  "33884_s_at", "36113_s_at", "32010_at", "40600_at",
                  "41205_at", "40323_at", "35574_i_at", "36790_at", "34695_at",
                  "37572_at", "37338_at", "40466_at", "914_g_at", "35977_at",
                  "971_s_at", "1986_at", "1724_at", "40016_g_at", "35447_s_at",
                  "32701_at", "38587_at", "1487_at", "37830_at", "33251_at",
                  "702_f_at", "37734_at", "38023_at", "40605_at", "37602_at",
                  "38018_g_at", "34597_at", "31504_at", "38413_at", "1506_at",
                  "1676_s_at", "40336_at", "1088_at", "37942_at", "41329_at",
                  "41237_at", "36156_at", "36499_at", "33145_at", "35858_at",
                  "33304_at", "32525_r_at", "32155_at", "32768_at", "40189_at",
                  "35772_at", "38207_at", "33720_at", "38573_at", "38149_at",
                  "1065_at")

        
    types <- c("ACCNUM", "CHRLOC", "CHR", "ENZYME", "GENENAME", "GO", "LOCUSID", 
               "MAP", "PATH", "PMID", "SUMFUNC", "SYMBOL", "UNIGENE")
    
    for(type in types)
        saveannaffy(probeids, "hgu95av2", type)
}
aafSearchText <- function(chip, colnames, text, logic = "OR") {
	
	if (! do.call("require", list(chip)))
        stop(paste("Data library couldn't be loaded:", chip))

    environment <- paste(chip, "SYMBOL", sep="")

    probeids <- do.call("ls", list(as.name(environment)))
    
    ann <- aafTableAnn(probeids, chip, colnames)
    
    matches <- NULL
    for (col in colnames) {
    	coltext <- getText(ann[[col]])
    	for (tex in text) {
    	    match <- grep(tex, coltext, ignore.case = TRUE)
    	    if (logic == "OR" || is.null(matches))
    	        matches <- union(matches, match)
    	    else
    	        matches <- intersect(matches, match)
    	}
    }
    matches <- sort(matches)
    
    return(probeids[matches])
}

aafSearchGO <- function(chip, ids, descendents = TRUE, logic = "OR") {
	
	if (! do.call("require", list(chip)))
        stop(paste("Data library couldn't be loaded:", chip))

    if (descendents)
    	environment <- paste(chip, "GO2ALLPROBES", sep="")
    else
    	environment <- paste(chip, "GO2PROBE", sep="")

    probeids <- NULL
    for (id in ids) {
		if (is.numeric(id) || length(grep("^[0-9]+$", id)))
			id <- sprintf("GO:%07i", as.integer(id))
		
		if (! do.call("exists", list(id, as.name(environment))))
			next
        probes <- do.call("get", list(id, as.name(environment)))
        if (length(probes) == 1 && is.na(probes))
            next
        if (logic == "OR" || is.null(probeids))
            probeids <- union(probeids, probes)
        else
            probeids <- intersect(probeids, probes)
    }
    
    if (is.null(probeids) || length(probeids) == 1 && is.na(probeids))
        return(character())
    
    return(probeids)
}

## Define class aafIntensity

setClass("aafIntensity", "numeric", prototype = numeric(0))

setMethod("getTD", "aafIntensity", function(object) {
    
    min <- getOption("minIntensity")
    max <- getOption("maxIntensity")
    color <- 100*(1-(object-min)/(max-min))
    style <- paste("background-color: rgb(", signif(color,3), "%, 100%, ", signif(color,3), "%)", sep = "")
    
    return(paste("<td class=\"", class(object), "\" style=\"", style, "\">", getHTML(object), "</td>", sep = ""))       
})

## Define class aafSigned

setClass("aafSigned", "numeric", prototype = numeric(0))

setMethod("getTD", "aafSigned", function(object) {
    
    if (object > 0)
        class <- "aafSignedPos"
    else if (object < 0)
        class <- "aafSignedNeg"
    else
        class <- "aafSignedZero"
    
    return(paste("<td class=\"", class, "\">", getHTML(object), "</td>", sep = ""))       
})

setMethod("getCSS", "aafSigned", function(object) {
    
    return(c("td.aafSignedPos { background-color: #ccf }",
             "td.aafSignedNeg { background-color: #fcc }"))
})

## Set generic methods

if( !isGeneric("probeids") )
    setGeneric("probeids", function(object) standardGeneric("probeids"))

if( !isGeneric("probeids<-") )
    setGeneric("probeids<-", function(object, value) standardGeneric("probeids<-"))

if( !isGeneric("colnames") )
    setGeneric("colnames", function(x, do.NULL = TRUE, prefix = "col") standardGeneric("colnames"))

if( !isGeneric("colnames<-") )
    setGeneric("colnames<-", function(x, value) standardGeneric("colnames<-"))

if( !isGeneric("saveHTML") )
    setGeneric("saveHTML", function(object, ...) standardGeneric("saveHTML"))

if( !isGeneric("saveText") )
    setGeneric("saveText", function(object, ...) standardGeneric("saveText"))

## Define class aafTable

setClass("aafTable", representation(probeids = "character",
                                    table = "list"),
         prototype = list(probeids = character(0),
                          table = list()))

## Constructors for aafTable
                          
aafTable <- function(..., items = list(...), colnames = names(items), 
                     probeids = character(0), signed = FALSE) {
    
    len <- length(items[[1]])
    
    if (is.null(colnames))
        stop("Column names must be provided")
    
    if (sum(duplicated(colnames)))
        stop("All column names must be unique")
    
    if (length(items) != length(colnames))
        stop("There must be the same number of column names as columns")
        
    for (col in colnames)
        if (!nchar(col))
            stop("Blank column names not allowed")
    
    for (item in items)
        if (length(item) != len)
            stop("All columns must be of equal length")
    
    table <- vector("list", length(items))
    for (col in 1:length(items)) {
        if (class(items[[col]]) == "aafList")
            table[[col]] <- items[[col]]
        else
            table[[col]] <- new("aafList", as.list(items[[col]]))
        if (signed)
            for (row in 1:len)
                class(table[[col]][[row]]) <- "aafSigned"
    }
    names(table) <- colnames
    
    return(new("aafTable", probeids = probeids, table = table))
}

aafTableFrame <- function(frame, colnames = names(frame), 
                          probeids = row.names(frame), signed = FALSE) {

    len <- dim(frame)[1]
    
    if (sum(duplicated(colnames)))
        stop("All column names must be unique")
    
    for (col in colnames)
        if (!nchar(col))
            stop("Blank column names not allowed")
    
    table <- vector("list", dim(frame)[2])
    for (col in 1:dim(frame)[2]) {
        table[[col]] <- new("aafList", as.list(frame[,col]))
        if (signed)
            for (row in 1:len)
                class(table[[col]][[row]]) <- "aafSigned"
    }
    names(table) <- colnames
    
    return(new("aafTable", probeids = probeids, table = table))
}

aafTableAnn <- function(probeids, chip, colnames = aaf.handler(),
                        widget = FALSE) {
    
    colnames <- intersect(colnames, aaf.handler())
    if (widget)
        colnames <- selectorWidget(aaf.handler(), colnames, ordernsel = TRUE,
                                   title = "Select Annotation Data Columns")
    
    table <- vector("list", length(colnames))
    for (i in 1:length(colnames)) {
        table[[i]] = aaf.handler(probeids, chip, colnames[i])
    }
    names(table) = colnames
    
    return(new("aafTable", probeids = probeids, table = table))
}

aafTableInt <- function(exprSet, colnames = sampleNames(exprSet), 
                        probeids = geneNames(exprSet)) {
    
    range <- match(probeids, geneNames(exprSet))
    expr <- exprs(exprSet)
    table <- vector("list", dim(expr)[2])
    for (col in 1:length(table)) {
        table[[col]] <- as.list(as.double(expr[range,col]))
        class(table[[col]]) <- "aafList"
        for (row in 1:length(range))
            class(table[[col]][[row]]) <- "aafIntensity"
    }
    names(table) = colnames
    
    return(new("aafTable", probeids = probeids, table = table))
}

## Methods for aafTable

setMethod("probeids", "aafTable", function(object) {

    return(object@probeids)
})

setReplaceMethod("probeids", "aafTable", function(object, value) {

    if (!length(value))
        value <- character(0)
    else if (length(value) != length(object@table[[1]]))
        stop("Wrong number of probe ids")
    
    if (sum(!nchar(value)))     stop("Blank probe ids not allowed")
    if (sum(is.na(value)))      stop("NA probe ids not allowed")
    if (sum(duplicated(value))) stop("All probe ids must be unique")
    
    object@probeids <- value
    return(object)
})

setMethod("colnames", c("aafTable","missing","missing"), function(x) {

    return(names(x@table))
})

setReplaceMethod("colnames", "aafTable", function(x, value) {

    if (length(value) != length(x@table))
        stop("Wrong number of column names")
    
    if (sum(!nchar(value)))     stop("Blank column names not allowed")
    if (sum(is.na(value)))      stop("NA column names not allowed")
    if (sum(duplicated(value))) stop("All column names must be unique")
    
    names(x@table) <- value
    return(x)
})

dim.aafTable <- function(x) {

    return(c(length(x@table[[1]]), length(x@table)))
}

merge.aafTable <- function(x, y, suffixes = c(".x",".y"), ...) {
    
    if (!length(y@probeids)) {
        if (length(x@table[[1]]) != length(y@table[[1]]))
            stop("The tables must have the same number of rows")
        probeids = x@probeids
        xrange = 1:length(x@table[[1]])
        yrange = xrange
    } else if (!length(x@probeids)) {
        if (length(x@table[[1]]) != length(y@table[[1]]))
            stop("The tables must have the same number of rows")
        probeids = y@probeids
        yrange = 1:length(y@table[[1]])
        xrange = yrange
    } else {
        probeids = intersect(x@probeids, y@probeids)
        if (!length(probeids))
            stop("The tables do not share any common probe ids")
        xrange = match(probeids, x@probeids)
        yrange = match(probeids, y@probeids)
    }
    
    xnames <- names(x@table)
    ynames <- names(y@table)
    common <- intersect(xnames, ynames)
    xmatch <- match(common, xnames)
    ymatch <- match(common, ynames)
    xnames[xmatch] <- paste(xnames[xmatch], suffixes[1], sep = "")
    ynames[ymatch] <- paste(ynames[ymatch], suffixes[2], sep = "")
    names(x@table) <- xnames
    names(y@table) <- ynames
    
    for (col in 1:length(x@table))
        x@table[[col]] <- x@table[[col]][xrange]
    for (col in 1:length(y@table))
        y@table[[col]] <- y@table[[col]][yrange]
    
    return(new("aafTable", probeids = probeids, table = c(x@table, y@table)))
}

rbind.aafTable <- function(..., deparse.level = 1)  {
    
    tables <- list(...)
    cols <- colnames(tables[[1]])
    noprobeids <- length(probeids(tables[[1]])) == 0
    probeids <- character(0)
    table <- vector("list", dim(tables[[1]])[2])
    names(table) <- cols
    
    for (tab in tables) {
        if (!(length(cols) == length(colnames(tab)) && !sum(!(cols == colnames(tab)))))
            stop("The column names must be the same in all tables")
        if (noprobeids && length(probeids(tab)))
            stop("Tables cannot have both defined and undefined probe ids")
    }
    
    for (tab in tables) {
        if (!noprobeids)
            probeids <- c(probeids, probeids(tab))
        for (i in 1:length(cols))
            table[[i]] <- c(table[[i]], tab[[i]])
    }
    for (i in 1:length(cols))
        class(table[[i]]) <- "aafList"
    
    return(return(new("aafTable", probeids = probeids, table = table)))
}

setMethod("[", "aafTable", function(x, i, j, ..., drop = FALSE) {

    if (missing(i)) i <- 1:dim(x)[1]
    if (missing(j)) j <- 1:dim(x)[2]
    if (is.character(i)) i <- match(i, probeids(x))
    if (is.character(j)) j <- match(j, colnames(x))
    
    if (drop && length(j) == 1) {
        if (length(i) == 1)
            return(x@table[[j]][[i]])
        return(x@table[[j]][i])
    }
    
    table <- vector("list", length(j))
    for (col in 1:length(table))
        table[[col]] <- x@table[[j[col]]][i]
    names(table) <- names(x@table)[j]
    
    return(new("aafTable", probeids = x@probeids[i], table = table))
})

setMethod("[[", "aafTable", function(x, i, j, ...) {

    result <- x@table
    for (ik in i)
        result <- result[[ik]]
    
    return(result)
})

"$.aafTable" <- function(x, val) {

    return(x@table[[as.character(val)]])
}

setMethod("saveHTML", "aafTable", function(object, filename, 
                                           title = "Bioconductor Affymetrix Probe Listing",
                                           colnames = names(object@table), 
                                           range = 1:dim(object)[1],
                                           open = FALSE, widget = FALSE) {

    colnames <- intersect(colnames, names(object@table))
    if (widget)
        colnames <- selectorWidget(aaf.handler(), colnames, ordernsel = TRUE,
                                   title = "Select Columns to Save")
    
    if (is.character(range))
        range <- match(range, probeids(object))
    
    maxIntensity = 0.0;
    minIntensity = 100.0;
    for(col in colnames) {
        if (class(object@table[[col]][[1]]) == "aafIntensity") {
            maxIntensity = max(as.numeric(object@table[[col]]), maxIntensity, na.rm=TRUE)
            minIntensity = min(as.numeric(object@table[[col]]), minIntensity, na.rm=TRUE)
        }
    }
    options(maxIntensity = maxIntensity)
    options(minIntensity = minIntensity)
    
    css <- character(0)
    th <- ""
    rows <- vector("character", length(range))
    
    for(col in colnames) {
        css <- c(css, getCSS(object@table[[col]]))
        th <- paste(th, "<th>", col, "</th>\n", sep = "")
        rows <- paste(rows, getTD(object@table[[col]][range]), "\n", sep = "")
    }
    
    outfile <- file(filename, "w")
    cat("<html>", "<head>", "<title>", title, "</title>", 
        "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\">", 
        "<style type=\"text/css\">", unique(css), "</style>",
        "<script language=\"JavaScript\">", "</script>", "</head>", 
        "<body bgcolor=\"#FFFFFF\">", 
        "<h1 align=\"center\">", title, "</h1>", 
        "<table border=\"2\">", file = outfile, sep = "\n")
        
    cat("<tr>\n", th, "</tr>\n", file = outfile, sep = "")
    for (i in 1:length(range))
        cat("<tr>\n", rows[i], "</tr>\n", file = outfile, sep = "")
    
    cat("</table>", paste("<p>", length(range), " Genes</p>", sep = ""), 
        "</body>", "</html>", file = outfile, sep = "\n")
    close(outfile)
    
    if (open)
        browseURL(filename)
})

setMethod("saveText", "aafTable", function(object, filename, 
                                           header = TRUE,
                                           colnames = names(object@table), 
                                           range = 1:dim(object)[1],
                                           widget = FALSE) {

    colnames <- intersect(colnames, names(object@table))
    if (widget)
        colnames <- selectorWidget(aaf.handler(), colnames, ordernsel = TRUE,
                                   title = "Select Columns to Save")
    
    if (is.character(range))
        range <- match(range, probeids(object))
    
    rows <- vector("character", length(range))
    sep <- ""
    
    head <- paste(colnames, collapse = "\t")
    for(col in colnames) {
        rows <- paste(rows, getText(object@table[[col]][range]), sep = sep)
        sep <- "\t"
    }
    
    outfile <- file(filename, "w")
    
    if (header)
        cat(head, "\n", file = outfile, sep = "")
    cat(rows, file = outfile, sep = "\n")
    
    close(outfile)
})
selectorWidget <- function(options, selected = character(0), title = "Selector",
                           ordersel = FALSE, ordernsel = FALSE, 
                           height = max(10,min(20,length(options)))) {
    
    require(tcltk) || stop("requires the package tcltk")
    
    move <- function(from, to, order) {
        while (length(index <- as.integer(tkcurselection(from)))) {
            item <- tclvalue(tkget(from, index[1]))
            tkinsert(to, "end", item)
            tkdelete(from, index[1])
        }
        tkselection.clear(from, 0, "end")
        if (!missing(order)) {
            newto <- as.character(tkget(to, 0, "end"))
            newto <- intersect(order, newto)
            tkdelete(to, 0, "end")
            tkinsert(to, "end", newto)
        }
    }
    
    ok <- function() {
        selected <<- as.character(tkget(selList, 0, "end"))
        tkdestroy(tt)
    }
    
    tt <- tktoplevel()
    tkwm.title(tt, title)
    tkwm.resizable(tt, 0, 0)
    
    selectorFrame <- tkframe(tt, bd=10)
    
    selFrame <- tkframe(selectorFrame)
    selList <- tklistbox(selFrame, selectmode="extended", height=height,
                         yscrollcommand=function(...) tkset(selScroll, ...))
    selScroll <- tkscrollbar(selFrame, takefocus=0, 
                             command=function(...) tkyview(selList, ...))
    tkgrid(selLabel <- tklabel(selFrame, text="Selected"), columnspan=2)
    tkgrid(selList, selScroll, sticky="ns")
    
    nselFrame <- tkframe(selectorFrame)
    nselList <- tklistbox(nselFrame, selectmode="extended", height=height,
                           yscrollcommand=function(...) tkset(nselScroll, ...))
    nselScroll <- tkscrollbar(nselFrame, takefocus=0, 
                               command=function(...) tkyview(nselList, ...))
    tkgrid(nselLabel <- tklabel(nselFrame, text="Not Selected"), columnspan=2)
    tkgrid(nselList, nselScroll, sticky="ns")
    
    if (ordersel)
        addcmd <- function() move(nselList, selList, options)
    else
        addcmd <- function() move(nselList, selList)
    if (ordernsel)
        removecmd <- function() move(selList, nselList, options)
    else
        removecmd <- function() move(selList, nselList)
    
    moveFrame <- tkframe(selectorFrame)
    addBut <- tkbutton(moveFrame, text="<-", command=addcmd)
    removeBut <- tkbutton(moveFrame, text="->", command=removecmd)
    tkgrid(addBut, pady=5)
    tkgrid(removeBut, padx=10, pady=5)
    
    if (length(selected))
        tkinsert(selList, "end", selected)
    if (length(setdiff(options, selected)))
        tkinsert(nselList, "end", setdiff(options, selected))
    
    tkgrid(selFrame, moveFrame, nselFrame)
    
    buttonFrame <- tkframe(tt)
    cancelBut <- tkbutton(buttonFrame, text="Cancel", command=function() tkdestroy(tt))
    okBut <- tkbutton(buttonFrame, text="OK", default="active", command=ok)
    tkbind(tt, "<Key-Return>", ok)
    tkgrid(cancelBut, okBut, padx=10, pady=10)
    
    tkgrid(selectorFrame)
    tkgrid(buttonFrame, sticky="se")
    
    tkwait.window(tt)
    
    return(selected)
}
.First.lib <- function (libname, pkgname)
{
    require(Biobase)
    require(GO)
    require(KEGG)

    path = .path.package(pkgname)
    where <- as.environment(match(paste("package:", pkgname, sep = ""),search()))
    dataPath = file.path(path, "data")
    rdas <- list.files(path = dataPath, pattern = "*.rda")
    rdas <- gsub("(^.*)\.rda", "\\1", rdas)
    for (i in rdas)
        load(file.path(path, "data", paste(i, ".rda", sep = "")), envir = where)
    options(sigfigs=6)
}
