.packageName <- "affyPLM"
###########################################################################
##
## file: LESN.R  (originally this file was called AdHoc.R, but it was
##                formally renamed on Mar 21, 2003)
##
## Copyright (C) 2003     Ben Bolstad
##
## "AdHoc" background correction routines - The LESN routines
##
## Feb 14, 2003 - give reasonable defaults to functions
## Feb 27, 2003 - clean out two commented out functions
## Mar 21, 2003 - Rename file LESN.R to keep consistant naming across methods
## Oct 7, 2003 - fix problem with shifting not proper function call
##
###########################################################################

bg.correct.shift <- function(object,baseline=0.25){

  objsize <- dim(pm(object))
  
  pm(object) <- matrix(.C("R_shift_down",as.double(as.vector(pm(object))),as.double(baseline),as.integer(objsize[1]),as.integer(objsize[2]))[[1]],objsize[1],objsize[2])
  object
}


bg.correct.stretch <- function(object,baseline=0.25,type=c("linear","exponential","loglinear","logexponential","loggaussian"),theta){
  
  objsize <- dim(pm(object))
  
  if (type == "linear"){
    ty <- 1
  } else if (type == "exponential"){
    ty <- 2
  } else if (type == "loglinear"){
    ty <- 3
  } else if (type=="logexponential"){
    ty <- 4
  } else {
    ty <- 5
    
  }
  
  pm(object) <- matrix(.C("R_stretch_down",as.double(as.vector(pm(object))),as.double(baseline),as.integer(objsize[1]),as.integer(objsize[2]),as.integer(ty),as.double(theta))[[1]],objsize[1],objsize[2])
  object
}




bg.correct.LESN <- function(object,method = 2,baseline = 0.25, theta=4){
  if (method == 2){
    bg.correct.stretch(object, baseline, type="loggaussian",theta=2*theta^2) 
  } else if (method == 1){
    bg.correct.stretch(object, baseline, type="logexponential",theta) 
  } else {
    bg.correct.shift(object,baseline)
  }
}

###########################################################
##
## file: PLMset.R
##
## Copyright (C) 2003    Ben Bolstad
##
## created by: B. M. Bolstad <bolstad@stat.berkeley.edu>
## created on: Jan 14, 2003
##
##
## aim: define and implement the PLMset object class and
##      its methods
##
## The PLMset object should hold Probe Level Model fits
## in particular we will concentrate on fits by the
## robust linear model methodology.
##
## Will use some of the ideas from the exprSet class
## from Biobase.
##
## the PLMset object has slots to hold probe and chip coefficients
## their standard errors, and model weights, along with the
## usual phenoData, description, annotation and notes fields.
## the exprs and se slot of the parent exprSet will be used for
## storing the constant coef and its se. ie if you fit the model
##
## pm'_ij(k) = mu_(k) + probe_i(k) + chip_j(k) + chipcovariates_j(k) + \epsilon_ij
##
##  then mu(k) would be stored in the exprs slot, its standard error in the se slot
##  probe_i(k) is stored in the probe.coef slot, with its standard error in its respective slot
##  chip_j(k) and chipcovariates_j(k) would be stored in chip.coefs (and ses in se.chip.coefs)
## 
##
## Modification History
##
## Jan 14, 2003 - Initial version, weights, coef accessors
## Jan 16, 2003 - added slots for probe coefs and there standard errors. some people
##                might find these useful. Made PLMset extend exprSet. This
##                saves us having to redefine accessors to phenoData,
##                description, annotataion, notes.
## Feb 15, 2003 - Add in a "show" method. Add in an coefs.probe accessor function
## Apr 29, 2003 - Add a replacement function for se
## Sep 25, 2003 - Port to R-1.8
## Oct 7, 2003 - update getCDFInfo to reflect changes in affy package.
## Oct 14, 2003 - cdfName accessor.
###########################################################

#creating the PLMset object

setClass("PLMset",
           representation(probe.coefs="matrix",
                          se.probe.coefs="matrix",
                          chip.coefs = "matrix",
                          se.chip.coefs = "matrix",
                          cdfName="character",
                          nrow="numeric",
                          ncol="numeric",
                          model.description="character",
                          model.call = "call",
                          weights="matrix"),
                         # phenoData="phenoData",
                         # description="characterORMIAME",
                         # annotation="character",
                         # notes="character"
           prototype=list(
             probe.coefs=matrix(nr=0,nc=0),
             se.probe.coefs=matrix(nr=0,nc=0),
             chip.coefs=matrix(nr=0,nc=0),
             se.chip.coefs=matrix(nr=0,nc=0),
             model.description="",
             weights=matrix(nr=0,nc=0),
             description=new("MIAME"),
             annotation="",
             cdfName="",
             nrow=0, ncol=0,
             notes=""),contains="exprSet")


  #now some accessors.
  if (is.null(getGeneric("cdfName")))
     setGeneric("cdfName", function(object)
             standardGeneric("cdfName"))

  setMethod("cdfName", "PLMset", function(object)
          object@cdfName)

  #access weights
  setMethod("weights",signature(object="PLMset"),
            function(object) object@weights)



  if (!isGeneric("weights<-"))
    setGeneric("weights<-",function(object,value)
               standardGeneric("weights<-"))

  
  #replace weights
  setReplaceMethod("weights",signature(object="PLMset"),
                   function(object,value){
                     object@weights <- value
                     object
                   })



  #access parameter estimates (chip level coefficients)

  if (!isGeneric("coefs"))
    setGeneric("coefs",function(object)
               standardGeneric("coefs"))
  
  setMethod("coefs",signature(object="PLMset"),
            function(object) object@chip.coefs)

  if (!isGeneric("coefs<-"))
    setGeneric("coefs<-",function(object,value)
               standardGeneric("coefs<-"))


  #replace coefs (chip level coefficients)
  setReplaceMethod("coefs",signature(object="PLMset"),
                   function(object,value){
                     object@chip.coefs <- value
                     object
                   })


  #access the probe level coefficents
  if (!isGeneric("coefs.probe"))
    setGeneric("coefs.probe",function(object)
               standardGeneric("coefs.probe"))

  setMethod("coefs.probe",signature(object="PLMset"),
            function(object) object@probe.coefs)
  
  
  
  if (!isGeneric("se"))
    setGeneric("se",function(object)
               standardGeneric("se"))
  
  setMethod("se",signature(object="PLMset"),
            function(object) object@se.chip.coefs)

  if (!isGeneric("se.probe"))
    setGeneric("se.probe",function(object)
               standardGeneric("se.probe"))
  
  setMethod("se.probe",signature(object="PLMset"),
            function(object) object@se.probe.coefs)

  if (!isGeneric("se<-"))
    setGeneric("se<-",function(object,value)
               standardGeneric("se<-"))


  #replace coefs (chip level coefficients)
  setReplaceMethod("se",signature(object="PLMset"),
                   function(object,value){
                     object@se.chip.coefs <- value
                     object
                   })  
  
  

  ## indexProbes, similar to that used in the AffyBatch class
  ## use the cdfenv to get what we need.
  
if( !isGeneric("indexProbes") )
    setGeneric("indexProbes", function(object, which, ...)
               standardGeneric("indexProbes"))

  setMethod("indexProbes", signature("PLMset", which="character"),
            function(object, which=c("pm", "mm","both"),
                     genenames=NULL, xy=FALSE) {
              
              which <- match.arg(which)
              
              i.probes <- match(which, c("pm", "mm", "both"))
              ## i.probes will know if "[,1]" or "[,2]"
              ## if both then [,c(1,2)]
              if(i.probes==3) i.probes=c(1,2)
              
              envir <- getCdfInfo(object)
              
              if(is.null(genenames)) 
                genenames <- ls(envir)
              
              ## shorter code, using the features of multiget
              ## (eventually more readable too)
              ## note: genenames could be confusing (the same gene can be
              ## found in several affyid (ex: the 3' and 5' controls)
              
              ans <-  multiget(genenames, pos, envir, iffail=NA)

              ## this kind of thing could be included in 'multiget' as
              ## and extra feature. A function could be specified to
              ## process what is 'multiget' on the fly
              for (i in seq(along=ans)) {


                #this line needs to be changed for R 1.7.0
                if ( is.na(ans[[i]][1]) )
                  next
                
                ##as.vector cause it might be a matrix if both
                tmp <- as.vector(ans[[i]][, i.probes])
                
                
                if (xy) {
                  warning("flag 'xy' is deprecated")
                  x <- tmp %% nrow(object)
                  x[x == 0] <- nrow(object)
                  y <- tmp %/% nrow(object) + 1
                  tmp <- cbind(x, y)
                }
                
                ans[[i]] <- tmp
              }
              
              return(ans)
            })
  

    
#  if( !isGeneric("image.weights") )
#    setGeneric("image.weights", function(x)
#               standardGeneric("image.weights"), where=where)

    
  setMethod("image",signature(x="PLMset"),
            function(x,which=0,...){
              pm.index <- unique(unlist(indexProbes(x, "pm")))
              rows <- x@nrow
              cols <- x@ncol
              pm.x.locs <- pm.index%%rows
              pm.x.locs[pm.x.locs == 0] <- rows
              pm.y.locs <- pm.index%/%rows + 1
              xycoor <- matrix(cbind(pm.x.locs,pm.y.locs),ncol=2)
              xycoor2 <- matrix(cbind(pm.x.locs,pm.y.locs+1),ncol=2)

              if (which == 0){
                which <- 1:dim(x@weights)[2]
              }
              for (i in which){
                weightmatrix <- matrix(nrow=rows,ncol=cols)
                weightmatrix[xycoor]<- x@weights[,i]
                weightmatrix[xycoor2]<- x@weights[,i]
                #this line flips the matrix around so it is correct
                weightmatrix <- as.matrix(rev(as.data.frame(weightmatrix)))
                image(weightmatrix,col=terrain.colors(12),xaxt='n', yaxt='n',main=sampleNames(x)[i])
                #title(sampleNames(x)[i])
              }
            }
            )
  
  setMethod("boxplot",signature(x="PLMset"),
            function(x,...){
              grp.rma.se1.median <- apply(se(x), 1, median)
              grp.rma.rel.se1.mtx <- sweep(se(x),1,grp.rma.se1.median,FUN='/')
              boxplot(data.frame(grp.rma.rel.se1.mtx),...)
            }
            )



  setMethod("show", "PLMset",
             function(object) {

              cat("Probe level linear model (PLMset) object\n")
              cat("size of arrays=", object@nrow, "x", object@ncol,"\n",sep="")

              ## Location from cdf env
              try( cdf.env <- getCdfInfo(object) )
              if (! inherits(cdf.env, "try-error")) {
                num.ids <- length(ls(env=cdf.env))
              } else {
                warning("missing cdf environment !")
                num.ids <- "???"
              }

              cat("cdf=", object@cdfName,
                  " (", num.ids, " probeset ids)\n",
                  sep="")
              cat("number of samples=",dim(object@weights)[2],"\n",sep="")
              cat("number of probesets=", num.ids, "\n",sep="")
              cat("number of chip level parameters for each probeset=",dim(object@chip.coefs)[2],"\n")
              cat("annotation=",object@annotation,"\n",sep="")
              cat("notes=",object@notes,"\n",sep="")
              cat("The model fit for each probeset was",object@model.description,"\n")
            })

  if (!isGeneric("coefs.const"))
    setGeneric("coefs.const",function(object)
               standardGeneric("coefs.const"))
  
  setMethod("coefs.const","PLMset",
            function(object){
              exprs(object)
            })


  if (!isGeneric("se.const"))
    setGeneric("se.const",function(object)
               standardGeneric("se.const"))
  
  setMethod("se.const","PLMset",
            function(object){
              se.exprs(object)
            })

# A summary method, to be cleaned up better at a later date.
# 
#  setMethod("summary","PLMset",
#          function(object,which=NULL){#
#
#              if (is.null(which)){
#                which <- rownames(object@chip.coefs)
#              }
#              cur.const.coef <-  NULL
#              cur.const.se <- NULL
#              for (probeset.names in which){
#                if (dim( exprs(object)) == 0){
#                  cur.const.coef <- exprs(object)[rownames(exprs(object)) == probeset.names]
#                  cur.const.se <-  se.exprs(object)[rownames(exprs(object)) == probeset.names]
#                }
#                inds <- grep(probeset.names,rownames(object@probe.coefs))
#                cur.probe.coef <- object@probe.coefs[inds,]
#                cur.se.probe.coef <- object@se.probe.coefs[inds,]
#                cur.chip.coef <- object@chip.coefs[grep(probeset.names,rownames(object@chip.coefs)),]
#                cur.chip.se <- object@se.chip.coefs[grep(probeset.names,rownames(object@se.chip.coefs)),]#
#
#                print(cbind(c(cur.const.coef, cur.chip.coef,cur.probe.coef),c(cur.const.se, cur.chip.se,cur.se.probe.coef)))                
#              }
#            },where=where)

  if (!isGeneric("Mbox"))
    setGeneric("Mbox",function(object,...)
               standardGeneric("Mbox"))
  

  
  setMethod("Mbox",signature("PLMset"),
            function(object,...){
              medianchip <- apply(coefs(object), 1, median)
              M <- sweep(coefs(object),1,medianchip,FUN='-')
              boxplot(data.frame(M),...)
            })






###########################################################
##
## file: fitPLM.R
##
## Copyright (C) 2003   Ben Bolstad
##
## created by: B. M. Bolstad <bolstad@stat.berkeley.edu>
## created on: Jan 15, 2003
##
## Last modified: Feb 22, 2003
##
## method for fitting robust linear models.
## interface to C code which does the heavy lifting
##
##
## by default model should be written in the following terms
##
## PM  : should appear on left hand side
## ~ : equivalent to  =
## -1 : remove the intercept term
## probes : include terms for probes in the model. a variable of type factor
## samples : defaults to be a factor, one for each chip or array
## other names for chip level factors or covariates. By default we will check the
## phenoData object to see if the named covariate or factor exists in the
## phenoData slot of the provided affybatch, then the parent enivronment is checked
## if the named variable does not exist there then procedure exits.
##
## an example model fitting terms for probes and samples
##
##    pm ~ -1 + probes + samples
##
## another example model
##
##    pm ~ -1 + probes + trt.group
##
## where trt.group is a factor in the phenoData object.
##
## A valid model should always include some chip effects
##
##
## Feb 1, 2003 - cache rownames(PM(object)) since this is slow operation
## Feb 15 - set model description to the function call: THIS was undone.
##         made it possible to fit models with an intercept term.
## Feb 17 -  start testing and fixing up column naming when model fit with no
##         intercept term, also make sure that se.exprs is set if there is
##         an intercept in the fitted model. re-order some parts of the function
##         work towards being able to fit models with no slope or probe-effects.
## Feb 22 - Add ability to fit models where parameters use a sum to zero constraint.
##          A similar mechanism to the variables.type parameter will be used.
##          note that R uses the convention for endpoint constraint that first
##          item is constrained to 0 and contr.sum that the last item is left unshown
##          we will try to follow this convention, except in the case constraints on
##          the probe coefficient.
##          Remember that the probe coefficient is handled by c code.
## Mar 22 - Add in ability to use LESN background correction.
## Mar 23 - Add ability to use MAS style background in fit PLM
## Jun 4-8 - Ability to specify other psi functions.
## Jul 22 - No longer need to specify psi.k, if left unspecified (ie null)
##          we set it to the default parameter. These defaults are the following:
## ** Huber - k = 1.345
## ** Fair - k = 1.3998
## ** Cauchy - k=2.3849 
## ** Welsch - k = 2.9846
## ** Tukey Biweight - k = 4.6851
## ** Andrews Sine - K = 1.339
##          the other methods have no tuning parameter so set to 1.
## Jul 27 - Clean up parameters passed to function
## Sep 12 - Fix up applying constraint when variables are defined in
##          parent enivronment rather than in pData slot.
## Oct 7  - Some codetools warnings fixed.
##
###########################################################

fitPLM <- function(object,model=PM ~ -1 + probes + samples,variable.type=c(default="factor"),constraint.type=c(default="contr.treatment"),background=TRUE,normalize=TRUE, background.method = "RMA.2",normalize.method = "quantile",se.type=4,psi.type="Huber",psi.k=NULL,background.param=list(),normalize.param=list()){

  
  get.background.code <- function(name) {
    background.names <- c("RMA.1", "RMA.2", "IdealMM","MAS","MASIM","LESN2","LESN1","LESN0")
    if (!is.element(name, background.names)) {
      stop(paste(name, "is not a valid summary method. Please use one of:",
                 "RMA.1", "RMA.2", "IdealMM","LESN2","LESN1","LESN0"))
    }
    code <- c(1, 2, 3, 4, 5, 6, 7, 8)[name == background.names]
    code
  }
  
  get.normalization.code <- function(name) {
    normalization.names <- c("quantile","quantile.probeset","scaling")
    if (!is.element(name, normalization.names)) {
      stop(paste(name, "is not a valid summary method. Please use one of:",
                 "quantile","quantile.probeset","scaling"))
    }
    code <- c(1,2,3)[name == normalization.names]
    code
  }

  get.psi.code <- function(name){
    psi.names <- c("Huber","fair","Cauchy","Geman-Mclure","Welsch","Tukey","Andrews")
     if (!is.element(name, psi.names)) {
      stop(paste(name, "is not a valid Psi type. Please use one of:",
                 "Huber","fair","Cauchy","Geman-Mclure","Welsch","Tukey","Andrews"))
    }
    code <- c(0:6)[name == psi.names]
    code
  }
  
  convert.LESN.param <- function(param.list){
    defaults <- c(0.25,4)
    if (!is.null(param.list$baseline)){
      defaults[1] <- param.list$baseline
    }
    if (!is.null(param.list$theta)){
      defaults[2] <- param.list$theta
    }
    defaults
  }

  get.default.psi.k <- function(name){
    psi.code <- get.psi.code(name)
    ## ** Huber - k = 1.345
    ## ** Fair - k = 1.3998
    ## ** Cauchy - k=2.3849 
    ## ** Welsch - k = 2.9846
    ## ** Tukey Biweight - k = 4.6851
    ## ** Andrews Sine - K = 1.339
    if (psi.code == 0){
      psi.k <- 1.345
    } else if (psi.code == 1){
      psi.k <- 1.3998
    } else if (psi.code == 2){
      psi.k <- 2.3849
    } else if (psi.code == 4){
      psi.k <- 2.9846
    } else if (psi.code == 5){
       psi.k <- 4.6851
    } else if (psi.code == 6){
      psi.k <- 1.339
    } else {
      psi.k <- 1
    }
    psi.k
  }




  
  
  if (class(object) != "AffyBatch"){
    stop(paste("argument is",class(object),"fitPLM requires AffyBatch"))
  }

#  oldconstraints <- options()$contrasts
#  if (constraint.type != "contr.treatment"){
#    stop("only endpoint constraint currently implemented")
#  }
  
#  options(contrasts=c(constraint.type,"contr.poly"))

  
  model.terms <- terms(model)
  mt.variables <- attr(model.terms,"variables")
  
  if ((mt.variables[[2]] != "PM") & (mt.variables[[2]] != "pm")){
    stop(paste("Response term in model should be 'PM' or 'pm'"))
  }
  
  mt.intercept <- attr(model.terms,"intercept")
  mt.termlabels <- attr(model.terms,"term.labels")

  
  length.parameters <- length(mt.termlabels)
  
  if (length.parameters < 1){
    stop("Insufficent parameters supplied in model")
  }

  # check to see if there is other chip level parameters and that they are valid
  mt.termlabels.abbrev <- mt.termlabels[mt.termlabels != "samples"]
  mt.termlabels.abbrev <- mt.termlabels.abbrev[mt.termlabels.abbrev != "probes"]
  
  has.probeeffects <- is.element("probes",mt.termlabels)
  has.chipeffects <- is.element("samples",mt.termlabels)
  


  
  #if (!has.probeeffects){
  #  stop("Fitting a model without probe-effects currently not supported")
  #}

  #check to see if anyone dumb enough to specify variable types for the "probes" and "samples" parameters

  if (!is.null(variable.type$probes)){
    cat("A variable type has been specified for probes parameter, this will be ignored. Assuming variable type is factor\n")
  }
  if (!is.null(variable.type$samples)){
    cat("A variable type has been specified for chips parameter, this will be ignored. Assuming variable type is factor\n")
  }

  #now go through the non default parameters seeing which type they should be treated as.

  #cat(mt.termlabels.abbrev,"\n")
  
  vt <- NULL

  #check to see that everything is either "factor" or "covariate"
  #figure out what the default type is 
  if (is.na(variable.type["default"])){
    cat("No default type given. Assuming default variable type is factor\n")
    vt.default <- "factor"
  } else {
    if (is.element(variable.type["default"],c("factor","covariate"))){
      vt.default <- variable.type["default"]
    } else {
      stop("Incorrect default variable.type given")
    }
  } 
    
  if (sum(!is.element(variable.type,c("factor","covariate")))){
    stop("An incorrect variable type provided")
  }

  
  if (length(mt.termlabels.abbrev) >=1){
    
    vt <- rep(vt.default,length(mt.termlabels.abbrev))
    names(vt) <- mt.termlabels.abbrev
    
    vt[names(variable.type)] <- variable.type    
    #has.vt <- is.element(mt.termlabels.abbrev,names(variable.type))    
  }
  #cat(has.vt,"\n");
  #print(vt)

  #figure out what the default constraint type is 

  if (is.na(constraint.type["default"])){
    cat("No default type given. Assuming default constraint type is contr.treatment\n")
    ct.default <- "contr.treatment"
  } else {
    if (is.element(constraint.type["default"],c("contr.sum","contr.treatment"))){
      ct.default <- constraint.type["default"]
    } else {
      stop("Incorrect default constraint.type given")
    }
  } 

  # check to see if there are any constraints on for the "probes" and "samples" parameters. Constraints on the first chip
  # level parameter will be ignored unless, the default constraint on probes will always "contr.sum", however
  # will allow the user to shoot themselves in the foot by setting it to something else (ie "contr.treatment") and there may be times when
  # this is useful.
  

  if (is.na(constraint.type["probes"])){
    ct.probe <- "contr.sum"
  } else {
    ct.probe <- constraint.type["probes"]
 }	

  if (is.na(constraint.type["samples"])){
    ct.samples <- ct.default
  } else {
    ct.samples <- constraint.type["samples"]
  }
 

  # Constraint.types given for each variable.
  # note that if user puts a constraint on a covariate variable it will be ignored.

  if (length(mt.termlabels.abbrev) >=1){
    
    ct <- rep(ct.default,length(mt.termlabels.abbrev))
    names(ct) <- mt.termlabels.abbrev
    
    ct[names(constraint.type)] <- constraint.type 
  #  print(ct)
  }	
  #print(ct.probe)
  #print(ct.samples)
  
  if (mt.intercept == 0){
    if (has.probeeffects){
      if (ct.probe == "contr.sum"){
        model.type <- 0
      } else {
        model.type <- 10
      }
    } else {
      model.type <- 20
    }
  } else {
    #stop("models with intercept terms currently not supported")
    if (has.probeeffects){
      if (ct.probe == "contr.sum"){
        model.type <- 1
      } else {
        model.type <- 11
      }
    } else {
      model.type <- 21
    }
  }
	
  
  if (!has.chipeffects & (length(mt.termlabels.abbrev) < 1)){
    stop("Model does not have enough chip level parameters")
  }

  nsamples <- dim(pm(object))[2]

  # start making the chip.covariate matrix
  # when no intercept, we need no contraint on the chipeffects parameter
  #print var.type
  #print(nsamples)
  chip.param.names <- NULL

  
  if (has.chipeffects){
    our.samples <- 1:nsamples
    if (!mt.intercept){
      chip.effects <- model.matrix(~ -1 + as.factor(our.samples))[,1:nsamples]
      chip.param.names <- sampleNames(object)
    } else {
      chip.effects <- model.matrix( ~ C(as.factor(our.samples),ct.samples))[,2:nsamples]
      if (ct.samples == "contr.treatment"){
        chip.param.names <- sampleNames(object)[2:nsamples]
      } else {
        chip.param.names <- sampleNames(object)[1:(nsamples-1)]
      }
    }
  } else {
    chip.effects <- NULL    
  }


  
  if (length(mt.termlabels.abbrev) >=1){

    in.pheno <- is.element(mt.termlabels.abbrev, names(pData(object)))
    in.parent.frame <- is.element(mt.termlabels.abbrev,ls(parent.frame()))

    #cat(mt.termlabels.abbrev)
    #cat(in.pheno,in.parent.frame,"\n")

    
    if (sum(in.pheno | in.parent.frame) != length(mt.termlabels.abbrev)){
      stop("Specified parameter does not exist in phenoData or parent frame")
    }

    chipeffect.names <-  NULL
    
    if (has.chipeffects){
      # chip.effects will handle intercept, use constraints on treatments
      # if chip.effects then the matrix will be singular, but we will go
      # through with the forming the chip.covariates matrix anyway.
      #options(contrasts=c("contr.treatment","contr.poly"))

      stop("Can not fit a model with an effect for every chip and additional parameters")
      
      #for (trt in mt.termlabels.abbrev){
      #   if (is.element(trt,  names(pData(object)))){
      #     trt.values <- pData(object)[,names(pData(object))==trt]
      #     if (vt[names(vt)==trt] == "covariate"){
      #       trt.effect <- model.matrix(~ -1 + trt.values)
      #     } else {
      #       trt.effect <- model.matrix(~ as.factor(trt.values))[,-1]
      #     }           
      #   } else {
      #     trt.values <- eval(as.name(trt))
      #     if (length(trt.values) != nsamples){
      #       stop("Model parameter in parent environment is incorrect")
      #     }
      #     trt.effect <- model.matrix(~ as.factor(trt.values))[,-1]
      #   }
      #   chip.effects <- cbind(chip.effects,trt.effect)
      #   if (vt[names(vt)==trt] == "covariate"){
      #     chipeffect.names <- c(chipeffect.names,trt)
      #   } else {
      #     for (levs in levels(as.factor(trt.values))[-1]){
      #       chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
      #     }
      #   }
      # }
    } else {
      # no chipeffect, first factor treatment will be unconstrained if no intercept
      #
      #
      
      #options(contrasts=c("contr.treatment","contr.poly"))
      #print(ct)
      first.factor <- FALSE
      for (trt in mt.termlabels.abbrev){
                                        #     print(vt[names(vt)==trt])
        if (is.element(trt,  names(pData(object)))){
          trt.values <- pData(object)[,names(pData(object))==trt]
          if (vt[names(vt)==trt] == "covariate"){
                                        #         print("is covariate")
            trt.effect <- model.matrix(~ -1 + trt.values)
          } else {
            if (!first.factor){
                                        #first.factor <- TRUE
              if (!mt.intercept){
                trt.effect <- model.matrix(~ -1+ as.factor(trt.values))
              } else {
                trt.effect <- model.matrix(~ C(as.factor(trt.values),ct[trt]))[,-1]
              }
             } else {
              
               trt.effect <- model.matrix(~  C(as.factor(trt.values),ct[trt]))[,-1]
             }
          }           
        } else {
          trt.values <- eval(as.name(trt))
          if (length(trt.values) != nsamples){
            stop("Model parameter in parent environment is incorrect")
          }
          if (vt[names(vt)==trt] == "covariate"){
            trt.effect <- model.matrix(~ -1 + trt.values)
          } else {
            if (!first.factor){
               #first.factor <- TRUE
              if (!mt.intercept){
                trt.effect <- model.matrix(~ -1+ as.factor(trt.values))
              } else {
                trt.effect <- model.matrix(~  C(as.factor(trt.values),ct[trt]))[,-1]
              }
            } else {
              trt.effect <- model.matrix(~ C(as.factor(trt.values),ct[trt]))[,-1]
            }
          }  
        }
        chip.effects <- cbind(chip.effects,trt.effect)
        
        if (vt[names(vt)==trt] == "covariate"){
          chipeffect.names <- c(chipeffect.names,trt)
        } else {
          if (!first.factor){
            first.factor <- TRUE
            if (!mt.intercept){
              for (levs in levels(as.factor(trt.values))){
                chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
              }
            } else {
              if (ct[trt] == "contr.treatment"){
                for (levs in levels(as.factor(trt.values))[-1]){
                  chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
                }
              } else {
                for (levs in levels(as.factor(trt.values))[-length(levels(as.factor(trt.values)))]){
                  chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
                }
              }
            }
          } else {
            if (ct[trt] == "contr.treatment"){
                for (levs in levels(as.factor(trt.values))[-1]){
                  chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
                }
              } else {                
                for (levs in levels(as.factor(trt.values))[-length(levels(as.factor(trt.values)))]){
                  chipeffect.names <- c(chipeffect.names,paste(trt,"_",levs,sep=""))
                }
            }
          }
        }
      }
    }
  }


  #print(chip.effects)
  #colnames(chip.effects) <- chipeffect.names
  chip.covariates <- chip.effects


  #print(chipeffect.names)
  #print(chip.effects)
  #print(colnames(chip.covariates))

  
  #check that chip covariates are not singular
  
  if (qr(chip.covariates)$rank < ncol(chip.covariates)){
    stop("chiplevel effects is singular: singular fits are not implemented in fitPLM")
  }
  
  #cat("Method is ",model.type,"\n")
  # now add other variables onto chip.covariates matrix

##  rows <- length(probeNames(object))
##  cols <- length(object)
  ngenes <- length(geneNames(object))
  
  # background correction for RMA type backgrounds
  bg.dens <- function(x){density(x,kernel="epanechnikov",n=2^14)}


  # to avoid having to pass location information to the c code, we will just call the R code method
  if (is.element(background.method,c("MAS","MASIM")) & background){
    cat("Background Correcting\n")
    object <- bg.correct.mas(object)
  }
    
  LESN.param <-list(baseline=0.25,theta=4)
  LESN.param <- convert.LESN.param(LESN.param)

  b.param <- list(densfun =  body(bg.dens), rho = new.env(),lesnparam=LESN.param)
  b.param[names(background.param)] <- background.param
  
  n.param <- list(scaling.baseline=-4,scaling.trim=0.0,use.median=FALSE,use.log2=TRUE)
  n.param[names(normalize.param)] <- normalize.param

    
  if (is.null(psi.k)){
    psi.k <- get.default.psi.k(psi.type)
  }
    
  
  # lets do the actual model fitting
  fit.results <- .Call("R_rlmPLMset_c",pm(object),mm(object),probeNames(object),
        ngenes, normalize, background,
        get.background.code(background.method), get.normalization.code(normalize.method),model.type, se.type,chip.covariates, get.psi.code(psi.type),psi.k,b.param,n.param)
  

  
 #put names on matrices and return finished object

  #chip.param.names <- NULL

  if (has.chipeffects){
    #chip.param.names <- c(chip.param.names,sampleNames(object))    
  } else {
    chip.param.names <- c(chip.param.names,chipeffect.names)
  }

  #options(contrasts=oldconstraints)


  #cache probenames rather than calling rownames(pm(object))
  probenames <- rownames(pm(object))
  #cat(chip.param.names,"\n")
  #colnames(fit.results[[1]]) <- sampleNames(object)
  colnames(fit.results[[1]]) <- chip.param.names
  colnames(fit.results[[3]]) <- sampleNames(object)
  rownames(fit.results[[3]]) <- probenames
  colnames(fit.results[[2]]) <- "ProbeEffects"
  rownames(fit.results[[2]]) <- probenames
  colnames(fit.results[[5]]) <- "SEProbeEffects"
  rownames(fit.results[[5]]) <- probenames
  colnames(fit.results[[4]]) <- chip.param.names
  if (mt.intercept){
      rownames(fit.results[[6]]) <- rownames(fit.results[[1]])
      rownames(fit.results[[7]]) <- rownames(fit.results[[1]])
       if (!has.probeeffects){
         fit.results[[2]] <- matrix(0,0,0)
         fit.results[[5]] <- matrix(0,0,0)
       }
  } else {
    if (has.probeeffects){
      fit.results[[6]] <- matrix(0,0,0)
      fit.results[[7]] <- matrix(0,0,0)
    } else {
      fit.results[[2]] <- matrix(0,0,0)
      fit.results[[5]] <- matrix(0,0,0)
      fit.results[[6]] <- matrix(0,0,0)
      fit.results[[7]] <- matrix(0,0,0)
    }
  }
  phenodata <- phenoData(object)
  annotation <- annotation(object)
  description <- description(object)
  notes <- notes(object)
  
  
  new("PLMset",chip.coefs=fit.results[[1]],probe.coefs= fit.results[[2]],weights=fit.results[[3]],se.chip.coefs=fit.results[[4]],se.probe.coefs=fit.results[[5]],exprs=fit.results[[6]],se.exprs=fit.results[[7]],phenoData = phenodata, annotation = annotation, description = description, notes = notes,cdfName=object@cdfName,nrow=object@nrow,ncol=object@ncol)
}

##################################################
##
## file: normalize.exprSet.R
##
## aim: normalization routines as applied to
##      exprSets (ie normalize expression values)
##
## Methods implemented
## quantiles  (quantile normalization)
## loess  (cyclic loess)
## contrasts (contrast loess)
## qspline (quantile spline method)
## invariantset (Similar to method used in dChip)
##
## Most of this is just wrappers around the approrpiate
## routine from the affy package, just adapted to deal
## with exprSet
##
##
## created: Aug 22, 2003
## written by: B. M. Bolstad <bolstad@stat.berkeley.edu>
##
## History
## Aug 22, 2003 - Initial version
## Aug 23, 2003 - added nomalize.exprSet.scaling
## Aug 25, 2003 - Added parameters to control whether
##                a log/antilog transformation should be applied
##                this is most useful with RMA expression estimate
##                that is usually given in log2 scale
##                note that while normalization is carried
##                out in transformed scale, returned values will be in
##                original scale
##
##################################################

normalize.exprSet.quantiles <- function(eset,transfn=c("none","log","antilog")){
  transfn <- match.arg(transfn)
  
  col.names <- colnames(exprs(eset))
  row.names <- rownames(exprs(eset))
  if (transfn == "none"){
    exprs(eset) <- normalize.quantiles(exprs(eset))
  } else if (transfn == "antilog"){
    exprs(eset) <- log2(normalize.quantiles(2^exprs(eset)))
  } else {
    exprs(eset) <- 2^(normalize.quantiles(log2(exprs(eset))))
  }
  colnames(exprs(eset)) <- col.names
  rownames(exprs(eset)) <- row.names
  return(eset)
}


normalize.exprSet.loess <- function(eset,transfn=c("none","log","antilog"),...){
  transfn <- match.arg(transfn)

  if (transfn == "none"){
    exprs(eset) <- normalize.loess(exprs(eset),...)
  } else if (transfn == "antilog"){
    exprs(eset) <- log2(normalize.loess(2^exprs(eset),...))
  } else {
    exprs(eset) <- 2^(normalize.loess(log2(exprs(eset)),...))
  }
  return(eset)
}


normalize.exprSet.contrasts <- function(eset, span = 2/3, choose.subset = TRUE, subset.size = 5000, verbose = TRUE, family = "symmetric",transfn=c("none","log","antilog")){

  transfn <- match.arg(transfn)
  
  col.names <- colnames(exprs(eset))
  row.names <- rownames(exprs(eset))
  alldata <- exprs(eset)

  if (transfn=="antilog"){
    alldata <- 2^(alldata)
  }
  if (transfn=="log"){
    alldata <- log2(alldata)
  }
  
  if (choose.subset)
    subset1 <- maffy.subset(alldata, verbose = verbose, subset.size = subset.size)$subset
  else subset1 <- sample(1:dim(alldata)[1], subset.size)
  aux <- maffy.normalize(alldata, subset = subset1, verbose = verbose,
                         span = span, family = family)

  if (transfn=="antilog"){
    alldata <- log2(alldata)
  }

  if (transfn=="log"){
    alldata <- 2^(alldata)
  }
  
  exprs(eset) <- aux
  colnames(exprs(eset)) <- col.names
  rownames(exprs(eset)) <- row.names
  return(eset)
}

normalize.exprSet.qspline <- function(eset,transfn=c("none","log","antilog"),...){

  transfn <- match.arg(transfn)
  col.names <- colnames(exprs(eset))
  row.names <- rownames(exprs(eset))
  if (transfn == "none"){
    exprs(eset) <- normalize.qspline(exprs(eset),...)
  } else if (transfn == "antilog"){
     exprs(eset) <- log2(normalize.qspline(2^exprs(eset),...))
  } else {
    exprs(eset) <- 2^(normalize.qspline(log2(exprs(eset)),...))
  }
  colnames(exprs(eset)) <- col.names
  rownames(exprs(eset)) <- row.names
  return(eset)
}


normalize.exprSet.invariantset <- function(eset,prd.td = c(0.003, 0.007), verbose = FALSE,transfn=c("none","log","antilog")){
  
  transfn <- match.arg(transfn)

  require(modreg, quietly = TRUE)
  nc <- length(sampleNames(eset))
  m  <- vector("numeric", length = nc)


  alldata <- exprs(eset)
  if (transfn == "log"){
    alldata <- log2(alldata)
  }

  if (transfn == "antilog"){
    alldata <- 2^(alldata)
  }
   
  for (i in 1:nc) m[i] <- mean(alldata[, i])
  refindex <- trunc(median(rank(m)))
  if (verbose)
    cat("Data from", sampleNames(eset)[refindex], "used as baseline.\n")
  
  for (i in (1:nc)[-refindex]) {
    if (verbose)
      cat("normalizing array", sampleNames(eset)[i], "...")
    tmp <- normalize.invariantset(alldata[, i],
                                  alldata[, refindex], prd.td)
    tmp <- as.numeric(approx(tmp$n.curve$y, tmp$n.curve$x,
                             xout = alldata[, i], rule = 2)$y)
    alldata[, i] <- tmp
    if (verbose)
      cat("done.\n")
  }

  if (transfn == "log"){
    alldata <- 2^(alldata)
  }

  if (transfn == "antilog"){
    alldata <- log2(alldata)
  }

  exprs(eset) <- alldata
  return(eset)
}


normalize.exprSet.scaling <- function(eset,trim=0.02,baseline=-1,transfn=c("none","log","antilog")){

  transfn <- match.arg(transfn)
    
  col.names <- colnames(exprs(eset))
  row.names <- rownames(exprs(eset))
  if (transfn == "none"){
    exprs(eset) <- normalize.scaling(exprs(eset),trim,baseline)
  } else if (transfn =="antilog"){
    exprs(eset) <- log2(normalize.scaling(2^exprs(eset)))
  } else {
    exprs(eset) <- 2^(normalize.scaling(log2(exprs(eset))))
  }
  colnames(exprs(eset)) <- col.names
  rownames(exprs(eset)) <- row.names
  return(eset)
}


#############################################################
##
## file: normalize.scaling.R
##
## aim: scaling normalization functions
##
## Written by B. M. Bolstad <bolstad@stat.berkeley.edu>
##
## History
## Aug 23, 2003 - Initial version
##              - Added type argument to 
##
#############################################################


normalize.scaling <- function(X,trim=0.02,baseline=-1){
  .Call("R_normalize_scaling",X,trim,baseline)
}


normalize.AffyBatch.scaling <- function(abatch,type=c("together","pmonly","mmonly","separate"),trim=0.02,baseline=-1){

  type <- match.arg(type)

  if (type == "pmonly"){
    Index <- unlist(indexProbes(abatch,"pm"))
  } else if (type == "mmonly"){
    Index <- unlist(indexProbes(abatch,"mm"))
  } else if (type == "together"){
    Index <- unlist(indexProbes(abatch,"both"))
  } else if (type == "separate"){
    abatch <- normalize.AffyBatch.scaling(abatch,type="pmonly",trim=trim,baseline=baseline)
    Index <- unlist(indexProbes(abatch,"mm"))
  }
  
  
  col.names <- colnames(exprs(abatch))
  exprs(abatch)[Index,] <- normalize.scaling(exprs(abatch)[Index,],trim,baseline)
  colnames(exprs(abatch)) <- col.names
  return(abatch)
}


normalize.AffyBatch.quantiles.probeset <- function(abatch,type=c("separate","pmonly","mmonly","together"),use.median=FALSE,use.log=TRUE) {

  type <- match.arg(type)
   
  
  rows <- length(probeNames(abatch))
  cols <- length(abatch)
  
  ngenes <- length(geneNames(abatch))

  if ((type == "pmonly")|(type == "separate")){
    pms <- unlist(pmindex(abatch))
    noNA <- apply(intensity(abatch)[pms,],1,function(x) all(!is.na(x)))
    pms <- pms[noNA]
    intensity(abatch)[pms,] <- .C("qnorm_probeset_R",as.double(intensity(abatch)[pms, ]),as.integer(rows), as.integer(cols),as.integer(ngenes),as.character(probeNames(abatch)),as.integer(use.median),as.integer(use.log))[[1]]
  }

  if ((type == "mmonly")|(type == "separate")){
    mms <- unlist(mmindex(abatch))
    noNA <- apply(intensity(abatch)[mms, , drop = FALSE],
                  1, function(x) all(!is.na(x)))
    mms <- mms[noNA]
    intensity(abatch)[mms, ] <- .C("qnorm_probeset_R",as.double(intensity(abatch)[mms,]),as.integer(rows), as.integer(cols),as.integer(ngenes),as.character(probeNames(abatch)),as.integer(use.median),as.integer(use.log))[[1]]
  }
  
  if (type == "together"){
    cat("together not current supported in quantiles.probeset\n")
  }
  

  ##this is MIAME we need to decide how to do this properly.
  ##for (i in 1:length(abatch)) {
  ##  history(abatch)[[i]]$name <- "normalized by quantiles"
  ##}
 
  return(abatch)
}


####################################################################
#
# threestep - threestep interface to c code
#
# Copyright (C) 2003    Ben Bolstad
#
# function by B. M. Bolstad <bolstad@stat.berkeley.edu>
#
# based on rma.R from the affy package
# the three step method implemented in c code
#
# this code serves as interface to the c code.
#
#
# note this function does not leave the supplied
# AffyBatch unchanged if you select DESTRUCTIVE=TRUE. this is 
# for memory purposes but can be quite
# dangerous if you are not careful. Use destructive=FALSE if this is
# deemed likely to be a problem. NOTE DESTRUCTIVE item removed for now
#
# Feb 6, 2003 - additional summary methods added
# Mar 22, 2003 - methods so that can use LESN backgrounds
# Mar 24, 2003 - add in ability to use MAS style background
# Jul 23, 2003 - standard errors from three step methods
# Jul 24, 2003 - added scaling as an option
# Jul 26, 2003 - introduced normalization options parameter
#                converted background parameters in same manner
# Jul 27, 2003 - cleaned up parameter list
# Oct 7, 2003 - comment out 
#
#####################################################################

threestep <- function(object,subset=NULL, verbose=TRUE,normalize=TRUE,background=TRUE,background.method="RMA.2",normalize.method="quantile",summary.method="median.polish",background.param = list(),normalize.param=list()){

  get.background.code <- function(name) {
    background.names <- c("RMA.1", "RMA.2", "IdealMM","MAS","MASIM","LESN2","LESN1","LESN0","SB")
    if (!is.element(name, background.names)) {
      stop(paste(name, "is not a valid summary method. Please use one of:",
                 "RMA.1", "RMA.2", "IdealMM","LESN2","LESN1","LESN0","SB"))
    }
    code <- c(1, 2, 3, 4, 5, 6, 7, 8,9)[name == background.names]
    code
  }
    
  get.normalization.code <- function(name){
    normalization.names <- c("quantile","quantile.probeset","scaling")

    if (!is.element(name,normalization.names)){
      stop(paste(name,"is not a valid summary method. Please use one of:","quantile","quantile.probeset","scaling"))
    }
    code <- c(1,2,3)[name == normalization.names]
    code
  }


  get.summary.code <- function(name){
    summary.names <- c("median.polish","tukey.biweight","average.log","rlm","log.average","log.median","median.log","log.2nd.largest","lm")

    if (!is.element(name,summary.names)){
      stop(paste(name,"is not a valid summary method. Please use one of:","median.polish","tukey.biweight","average.log","rlm","log.average","log.median","median.log","log.2nd.largest","lm"))
    }
    code <- c(1,2,3,4,5,6,7,8,9)[name ==summary.names]
    code
  }

  convert.LESN.param <- function(param.list){
    defaults <- c(0.25,4)
    if (!is.null(param.list$baseline)){
      defaults[1] <- param.list$baseline
    }
    if (!is.null(param.list$theta)){
      defaults[2] <- param.list$theta
    }
    defaults
  }


  #rows <- length(probeNames(object))
  #cols <- length(object)
 
  ngenes <- length(geneNames(object))
  
  # background correction for RMA type backgrounds
  bg.dens <- function(x){density(x,kernel="epanechnikov",n=2^14)}


  # to avoid having to pass location information to the c code, we will just call the R code method
  if (is.element(background.method,c("MAS","MASIM")) & background){
    cat("Background Correcting\n")
    object <- bg.correct.mas(object)
  }
    
  LESN.param <-list(baseline=0.25,theta=4)
  LESN.param <- convert.LESN.param(LESN.param)

  b.param <- list(densfun =  body(bg.dens), rho = new.env(),lesnparam=LESN.param)
  b.param[names(background.param)] <- background.param
  
  n.param <- list(scaling.baseline=-4,scaling.trim=0.0,use.median=FALSE,use.log2=TRUE)
  n.param[names(normalize.param)] <- normalize.param

  results <- .Call("R_threestep_c",pm(object), mm(object), probeNames(object), ngenes, normalize, background, get.background.code(background.method), get.normalization.code(normalize.method), get.summary.code(summary.method),b.param,n.param) #, PACKAGE="AffyExtensions") 
  
  colnames(results[[1]]) <- sampleNames(object)
  colnames(results[[2]]) <- sampleNames(object)
  #se.exprs <- array(NA, dim(exprs)) # to be fixed later, besides which don't believe much in nominal se's with medianpolish
  
  phenodata <- phenoData(object)
  annotation <- annotation(object)
  description <- description(object) 
  notes <- notes(object)
  
  new("exprSet", exprs = results[[1]], se.exprs = results[[2]], phenoData = phenodata, 
       annotation = annotation, description = description, notes = notes)
}


############################################################
##
## file: init_fns.R
##
## Copyright (C) 2003   Ben Bolstad
##
## aim: implemement initialization functions for AffyExtensions
##
## Created by: B. M. Bolstad <bolstad@stat.berkeley.edu>
##
##
##
## Aug 22, 2003 - Added a initialization function registering
##                the a normalize method for exprSet objects
## Aug 23, 2003 - make sure to make scaling available via normalize
## Sep 25, 2003 - Port to R-1.8
## Oct 7, 2003 - Remove unused variable
##
############################################################


.initNormfunctions <- function(where){
  all.affy <- ls(where)
  start <- nchar("normalize.exprSet.")
  assign("normalize.exprSet.methods",
         substr(all.affy[grep("normalize\.exprSet\.*", all.affy)], start+1, 100),
         envir=as.environment(where))

  setMethod("normalize", signature(object="exprSet"),
            function(object, method=getOption("BioC")$affy$normalize.method, ...) {
              method <- match.arg(method, normalize.exprSet.methods)
              if (is.na(method))
                stop("unknown method")
              method <- paste("normalize.exprSet", method, sep=".")
              object <- do.call(method, alist(object, ...))
              return(object)
            })
 
  
}






.First.lib <- function(libname, pkgname) {


  .initNormfunctions(match(paste("package:",pkgname,sep=""),search()))
    
  #if (length(search()) > length(s)) {
  #  detach("package:AffyExtensions")
  #  library(AffyExtensions,warn.conflicts=FALSE,verbose=FALSE)
  #} 	

  
  library.dynam("affyPLM",pkgname,libname,now=FALSE)
  
  current.normmethods <- get("normalize.AffyBatch.methods",envir=as.environment("package:affy"))
  
  assign("normalize.AffyBatch.methods",
         c(current.normmethods,"quantiles.probeset","scaling"),
         envir=as.environment(match("package:affy", search())))
}
