\name{SAGELyzer}
\alias{SAGELyzer}
\alias{getSAGESQL}
\alias{getTotalRNum}
\alias{getKNN}
\alias{noChunkKNN}
\alias{chunkKNN}
\alias{findNeighborTags}
\alias{getColNames}
\title{Function to filter out the k nearest neighbors for a given tag}
\description{
  This function finds the k nearest neighbors for a given SAGE tag based
  on the expression of SAGE tags across selected SAGE labraries. The
  calculations are based on data stored in a table in a databse.
}
\usage{
SAGELyzer(dbArgs, targetSAGE, libs = "*", normalize = "min", tagColName
= "tag", k = 500, dist = "euclidean", trans = "sqrt")
getSAGESQL(dbArgs, conn, targetSAGE, libs, tagColName, chunk = FALSE,
cursor = "sageRows", ignorZeros = TRUE, what = c("map", "counts",
"info"))
getTotalRNum(dbArgs, conn, tagColName, what = "counts")
getKNN(dbArgs, targetSAGE, libs, tagColName, normalize, k, dist, trans,
max = 50000)
noChunkKNN(dbArgs, conn, targetSAGE, libs, tagColName, normalize, k,
dist, trans)
chunkKNN(dbArgs, conn, targetSAGE, libs, tagColName, normalize, k, dist,
trans, rowNum, max = 50000)
findNeighborTags(targetRow, data, k, NF, dist, trans)
 getColNames(dbArgs, conn, what = "counts")
}

\arguments{
  \item{dbArgs}{\code{dbArgs} a list containing arguments needed to make
    connection to a database and queries against a table. The elements
    include a DSN under Windows and database name, user name, password,
    and host under Unix plus the names for three tables that will be
    used by SAGElyzer}
  \item{targetSAGE}{\code{targetSAGE} a character string for the SAGE
    tag whose neighbors will be sought}
  \item{libs}{\code{libs} a vector of character strings for column names
    of database table where SAGE library data are stored}
  \item{normalize}{\code{normalize} a character string for the means to
    perfrom data normalization. Can be either "min", "max", or "none"}
  \item{tagColName}{\code{tagColName} a character string for the column
    name of a database table where SAGE tags are stored}
  \item{k}{\code{k} an integer for the number of nearest neighbors to be
    sought}
  \item{dist}{\code{dist} a character string corresponding to an
    existing R object for calculating distances between two data sets}
  \item{trans}{\code{trans} a character string corresponding to an
    existing R object that will be used to transform the data}
  \item{conn}{\code{conn} a connection to a database}
  \item{chunk}{\code{chunk} a boolean indicating whether data will be
    processed in chunks to avoid running out space}
  \item{ignorZeros}{\code{ignorZeros} a boolean indicating whether data
    rows with all 0s will be ignored}
  \item{what}{\code{what} a character string for the type of database
    table to use for getting data. Have to be either "map", "counts", or
    "info"}
  \item{max}{\code{max} an integer for the maximum number of data rows
    in a chunk to be processed}
  \item{rowNum}{\code{rowNum} an integer for row number}
  \item{NF}{\code{NF} a vector of numerical data that will be used as
    normalization factor for SAGE counts}
  \item{targetRow}{\code{targetRow} a vector of character strings
    containing data for the target SAGE tag}
  \item{data}{\code{data} a matrix containing SAGE counts across
    selected libraries}
  \item{cursor}{\code{cursor} a character string for the name of a
    cursor to reterive data in chunks from a database table}
}
\details{
  Two database tables (default names "sagecounts" and "sageinfo" have to
  exist (tables can be created using other function in this
  package). One table (sagecounts) contains counts for SAGE tags for
  libraries and the other (sageinfo) contains mappings between column
  names used in "sagecounts" to store data for a given SAGE library.

  Functions in this package are normally called by interactive
  interfaces that are invoked when the package is loaded.
}
\value{
  \code{\link{SAGELyzer}} returns a named vector with SAGE tags being
  the names and the corresponding calculated distances to a given tag
  being the values.

  \code{\link{getSAGESQL}} returns a character string for a SQL
  statement to use to query a database.

  \code{\link{getTotalRNum}} returns an integer for the total row number
  of a database table.
}
\references\url{www.sagenet.org}{}
\author{Jianhua Zhang}
\note{This function is part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}

\seealso{\code{\link{SAGE4Unix}}}
\examples{
# No example is given as the code requires data with existing tables
}
\keyword{manip}

\eof
\name{SAGEMapper}
\alias{SAGEMapper}
\alias{doTag2UG}
\alias{doUG2Tag}
\alias{env2File}
\alias{getMapFileName}
\title{Annotating SAGE tags using data from public databases}
\description{
  Functions that provide data annotation using public databases and
  package AnnBuilder}.
}
\usage{
SAGEMapper(tag2UG = TRUE, tagUrl =
"ftp://ftp.ncbi.nih.gov/pub/sage/map/Hs/NlaIII/SAGEmap_tag_ug-rel_Hs.zip",
organism = "Hs", fromWeb = TRUE)
doTag2UG(fileName)
doUG2Tag(fileName, sep = "\t", header = FALSE)
getMapFileName()
}

\arguments{
  \item{tag2UG}{A boolean set to be TRUE if the mapping will be between
    SAGE tags and UniGene ids or FALSE is the mapping will be between
    UniGene ids and SAGE tags}
  \item{tagUrl}{A character string for the url where mapping information
    can be downloaded}
  \item{fileName}{A character string for the name of the file where the
    mapping will be stored}
  \item{sep}{\code{sep} a character string for the separator used in the
    source file}
  \item{header}{\code{header} a boolean indicating whether the source
    file has a header line}
  \item{organism}{\code{organism} a character string for the organism of
    concern (e. g. Hs for human)}
  \item{fromWeb}{\code{fromWeb} a boolean indicating whether the source
    data should be downloaded from the web or read from a directory
    locally} 
}
\details{
  \code{\link{SAGEMapper}} reads mapping data from NCBI
  (\url{ftp://ftp.ncbi.nih.gov/pub/sage/map/Hs/NlaIII/SAGEmap_tag_ug-rel_Hs.zip}) and produces a text file containing the mappings between SAGE tags and UniGene ids or UniGene ids and SAGE tags. The default url was valid for human genes at the time of development but needs to be updated when needed.

  \code{\link{doTag2UG}}, \code{\link{doUG2Tag}}, and
  \code{\link{env2File}} are called by \code{\link{SAGEMapper}} to
  perfome the required functions
}
\value{
  \item{doTag2UG}{Returns an R environment object containing mappings
    between SAGE tags and UniGene ids}
  \item{doUG2Tag}{Returns an R environment object containing mappings
    between UniGene ids SAGE tags}
}
\references{The help files for package AnnBuilder provides explations on
how to annotate data using AnnBuilder} 
\author{J. Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide bioinformatics functionalities through R}

\seealso{\code{\link{SAGELyzer}}}

\examples{
# The following code takes a while to run and is thus inactivated
\dontrun{SAGEMapper("theMap", "", TRUE,
"ftp://ftp.ncbi.nih.gov/pub/sage/map/Hs/NlaIII/SAGEmap_tag_ug-rel_Hs.zip")}
}
\keyword{manip}

\eof
\name{SAGEToolTips}
\alias{SAGEToolTips}
\docType{data}
\title{A list that maps SAGE task or procedure names to tips}
\description{
  This binary data is for the purpose of providing tooltips for
  SAGElyzer
}
\usage{data(SAGEToolTips)}
\format{
  The format is:
  \item{Manage Data}{Get and map SAGE data and wrtite to a database}
  \item{knn}{Performs knn on a selected tag and presents the results} 
  \item{Run knn}{Run knn based on the target tag and selected SAGE libraries} 
  \item{Get counts}{Get counts for tags neighboring the target tag
    acroos selected libraries}  
  \item{Map SAGE}{Map tags that are neighbors of the target tag to
    UniGene id and link to UniGene web site for annotation}  
  \item{Get GEO SAGE}{Get SAGE libraries from GEO web site}
  \item{Integrate SAGE}{Put data from SAGE libraries to a database}
  \item{Map SAGE}{Download and write mappings between SAGE tags and
    UniGene ids to a database}   
  \item{Set arguments}{Set the arguments for knn}
}

\examples{
data(SAGEToolTips)
SAGEToolTips
}
\keyword{datasets}

\eof
\name{SAGEWidget}
\alias{SAGEWidget}
\alias{getTasks}
\alias{getDMProc}
\alias{getKNNProc}
\alias{butsInTBox}
\alias{getTaskTips}
\alias{KNNArgsWidget}
\title{A function that is the main interface for SAGElyzer}
\description{
  This function serves as the main interface for SAGElyzer, which
  contains buttons for making a connection to a database and invokes all
  the tasks and procedures to take to complete a task.
}
\usage{
SAGEWidget()
getTasks()
getDMProc(base, TBox, status)
getKNNProc(base, TBox, status)
butsInTBox(base, TBox, status, butList, clear = FALSE)
getTaskTips(task)
KNNArgsWidget()
}
\arguments{
  \item{base}{\code{base} a tkwin object that can be a parent of other
    widgets (e.g. window, frame)}
  \item{TBox}{\code{TBox} a tkwin object that can be used as a status bar}
  \item{status}{\code{status} a character string for the status of a process}
  \item{butList}{\code{butList} a list of chracter strings for function
    names. The name of the list will be used to create buttons baring
    the same name and values will be called when corresponding buttons
    are pressed} 
  \item{clear}{\code{clear} a boolean indicating if a status bar will be
    cleard before updating}
  \item{task}{\code{task} a character string for the name of a task of
    interest} 
}
\details{
  Each task may involve several procedures that require user inputs for
  arguments. Blanks need to be filled. Default values are provided
  whereever it is possible. Defaults are advised to be used if a user
  are not sure about what to enter for inputs.

  \code{\link{SAGEWidget}} calls the other functions listed in this man
  page.
}
\value{
  This function returns invisible().
}
\references{\url{http://www.ncbi.nlm.nih.gov/SAGE/}}
\author{Jianhua Zhang}
\note{This function is part of the Bioconductor poject at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}

\examples{
if(interactive()){
    SAGEWidget()
}
}
\keyword{misc}

\eof
\name{con4Win}
\alias{con4Win}
\alias{con4Unix}
\alias{makeConnection}
\alias{executeQuery}
\alias{query4Unix}
\alias{closeConn}
\alias{tableExists}
\title{Functions for database connection and manipulation}
\description{
  These functions make connections to or query against a database.
}
\usage{
con4Win(args)
con4Unix(args)
makeConnection(args)
executeQuery(sqlStat, conn, noReturn = FALSE)
query4Unix(sqlStat, conn, noReturn = FALSE)
closeConn(conn)
tableExists(conn, tableName)
}

\arguments{
  \item{args}{\code{args} a list of arguments that will be used for
    database connection and query}
  \item{sqlStat}{\code{sqlStat} a character string for the SQL statement
    to be sent to the database server}
  \item{conn}{\code{conn} a connection object}
  \item{noReturn}{\code{noReturn} a boolean to indicate whether a query
    sent to the database server will return any value}
  \item{tableName}{\code{tableName} a character string for the name of a
    database table}
}
\details{
  \code{\link{con4Win}} makes a connection to a database for windows.

  \code{\link{con4Unix}} makes a connection to a database for unix.

  \code{\link{makeConnection}} direct the effort of making a database
  connection depending on the platform.

  \code{\link{executeQuery}} executes a SQL query statement against a
    database.
  \code{\link{query4Unix}} executes a SQL query statement against a
    database under unix.

  \code{\link{closeConn}} closes a connection to a database.

  \code{\link{tableExists}} checks to see if a given table exists in the
  database. 
}
\value{
  \code{\link{con4Win}} returns an ODBC connection object.

  \code{\link{con4Unix}} returns an Rdbi connection object.

  \code{\link{makeConnection}} returns a connection object.

  \code{\link{executeQuery}} returns the values for a query.

  \code{\link{query4Unix}} returns the results of a query for unix.

  \code{\link{tableExists}} returns TRUE if a given table exists and
  FALSE otherwise.
}
\author{Jianhua Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}

\seealso{\code{\link{SAGELyzer}}, \code{\link{mergeSAGE}}}
\examples{
# No examples are given as database support will be required
}
\keyword{misc}

\eof
\name{getGEOSAGE}
\alias{getGEOSAGE}
\alias{getFileNames}
\alias{getSampleId}
\alias{parseSAGE}

\title{Automatically downloads SAGE libraries from NCBI}
\description{
  Given an organism name (e.g. human) and a correct url,
  \code{\link{getGEOSAGE}} downloads SAGE libraries and stores them in a
  specified directory
}
\usage{
getGEOSAGE(organism = "human", targetDir = "", quiet = TRUE, url =
"http://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?")
getFileNames(organism, url)
getSampleId(url)
}

\arguments{
  \item{organism}{A character string for the name of the organism of
    interests}
  \item{targetDir}{A character string for the directory where the
    downloaded SAGE libraries will be stored}
  \item{quiet}{A boolean indicating whether the status message from
    \code{\link{download.file}} will be supressed}
  \item{url}{A character string for part of the url from which SAGE libraries
    will be downloaded}
}
\details{
  \code{\link{getGEOSAGE}} downloads SAGE libraries from NCBI's GEO site
  and stores them in a specified directory. The url passed is the
  location where the cgi resides and will be appended the correct
  parameters that specifies the content and format of the data file to
  be downloaded.

  The system relies on GPL numbers that differ among organisms to find
  the correct platform sample ids for files belonging to a given
  organism. The platform sample ids will then be used to fetch the
  desired annotation files. \code{\link{getFileNames}} gets the correct
  GPL number and \code{\link{getSampleId}} gets the platform sample
  ids. \code{\link{parseSAGE}} parses the downloaded file and stores the
  data to a specified place.   
}
\value{
  \item{getFileNames}{Returns a vector of GPL numbers}
  \item{getSampleId}{Returns a vector of platform sample ids}
}
\references{\url{http://www.ncbi.nlm.nih.gov/geo/query/}}
\author{J. Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide bioinformatics functionalities through R}

 ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{mergeSAGE}}}

\examples{
# Since downloading and parsing SAGE libraries are time consuming, the
# example code is inactivated.
\dontrun{getGEOSAGE(organism = "human", targetDir = "", quiet = TRUE, url =
"http://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?")}  
}
\keyword{manip}

\eof
\name{getNormFactor}
\alias{getNormFactor}
\alias{queryInfoDB}
\title{Functions that get normalization factors for SAGE libraries}
\description{
  SAGE libraries vary in the total number of tags so that counts need to
  be normalized across libraries. These functions get the normalization
  factors that are stored in a database table. 
}
\usage{
getNormFactor(normalize = c("min", "max", "none"), libs)
queryInfoDB(libCol = "libname",
                        infoCol = c("filename", "minfactor", "maxfactor"))
}
\arguments{
  \item{normalize}{\code{normalize} a character string for the means of
    normalization. Have to be either "min", "max", or "none"}
  \item{libs}{\code{libs} a vector of character strings for the names of
    SAGE libraries to be normalized}
  \item{libCol}{\code{libCol} a character string for the name of the
    column in a database table where names of SAGE libraries are stored}
  \item{infoCol}{\code{infoCol} a vector of character strings for the
    names of database columns where SAGE library information is kept}
}
\details{
  The normalization factor is calculated by dividing the total number of
  tags for a given library by the maximum or minimum value across the
  library.

  \code{\link{getNormFactor}} returns the normalization factors for a
  given set of SAGE libraries.

  \code{\link{queryInfoDB}} queries a database table containing
  information about SAGE libries to get the normalization factor for
  SAGE libraries.
}
\value{
  Both \code{\link{getNormFactor}} and  \code{\link{queryInfoDB}} return
  a data frame containing normalization factors for a set of SAGE libraries.
}
\references{\url{http://www.ncbi.nlm.nih.gov/SAGE/} }
\author{Jianhua Zhang }
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}

\seealso{\code{\link{SAGELyzer}}}
\examples{
# No example is given as database support is required
}
\keyword{misc}

\eof
\name{getTargetRow}
\alias{getTargetRow}
\title{Function that retrieves data from selected SAGE libraries for a
  given SAGE tag}
\description{
  Given a SAGE tag, this function queries an existing table in a
  database and retrieves data across all the selected SAGE libraies for
  that SAGE tag
}
\usage{
getTargetRow(dbArgs, conn, libs, tagColName, targetSAGE, what = "counts")
}
\arguments{
  \item{dbArgs}{\code{dbArgs} a list containing arguments needed to make
    connection to a database and queries against a table. The elements
    include a DSN under Windows and database name, user name, password,
    and host under Unix plus the names for three tables that will be
    used by functions of SAGElyzer}
  \item{conn}{\code{conn} a connection to a database}
  \item{libs}{\code{libs} a vector of character strings for column names
    of database table where SAGE library data are stored}
  \item{tagColName}{\code{tagColName} a character string for the column
    name of a database table where SAGE tags are stored}
  \item{targetSAGE}{\code{targetSAGE} a character string for the SAGE
    tag whose counts across SAGE libraries will be retrieved}
  \item{what}{\code{what} a character string that is either "counts",
    "info", or "map" to indicating the what database table to use}
}
\details{
  This function is called by \code{\link{SAGELyzer}} for the calculation
  of nearest neighbors for a given SAGE tag. It may not have much other
  practical use. 
}
\value{
  \code{\link{getTargetRow}} returns a vector containing the data
  retrieved 
}
\author{Jianhua Zhang}
\note{This function is part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}

\examples{
  # No example is give as the function needs a database support
}
\keyword{manip}

\eof
\name{mergeSAGE}
\alias{mergeSAGE}
\alias{getLibInfo}
\alias{calNormFact}
\alias{getLibNNum}
\alias{getUniqTags}
\alias{writeSAGE4Win}
\alias{writeSAGE4Unix}
\alias{mapFile2Tag}
\alias{writeSAGECounts}
\alias{writeSAGE2DB}
\alias{getColSQL}
\title{Functions to merge SAGE libraries based on unique SAGE tags}
\description{
  These functions merge individual SAGE libraries based on unique SAGE
  tags and write the merged data into a file and a table in a database
  with the unique SAGE tags as one column and counts from all the
  libraries as the others.  
}
\usage{
mergeSAGE(libNames, isDir = TRUE,  skip = 1, pattern = ".sage")
getLibInfo <- function(fileNames}
calNormFact(normalize = c("min", "max"), libNNum)
getLibNNum(fileNames)
getUniqTags(fileNames, skip = 1)
writeSAGE4Win(fileNames, uniqTags, infoData, pace = 500)
mapFile2Tag(fileNames, tags, skip, n)
writeSAGECounts(fileNames, uniqTags, skip, ncol = 2)
writeSAGE2DB(dbArgs, colNames, keys, numCols, fileName, what =
c("counts", "map", "info"), charNum = 20, type = "int4")
getColSQL(colNames, charNum, keys, numCols, type)
writeSAGE4Unix <- function(countData, infoData)
}

\arguments{
  \item{libNames}{\code{libNames} - a vector of character strings for
    the name of the SAGE libraries to be merged. \code{libNames} can be
    the name of the directory containing SAGE libraries to be merged}
  \item{isDir}{\code{isDir} - a boolean that is TRUE if libNames is the
    name for the directory that contains SAGE libraries to be merged}
  \item{skip}{\code{skip} - an integer for the number of lines to be
    skiped when the libraries are merged}
  \item{pattern}{\code{pattern} - a character string for the pattern to
    be used to get the file SAGE data files from the directory when
    \code{libNames} is for a directory. Only files that match the
    pattern will be merged} 
}

\details{
  Each SAGE library typically contains two columns with the first one
  being SAGE tags and the second one being their
  counts. \code{\link{mergeSAGE}} merges library files based on the
  tags. Tags that are missing from a given library but exist in other
  will be assigned 0s for the library. 

  \code{\link{mergeSAGE}} will generate two files. One contains the
  merged data and the other contains four columns with the first one
  being the column names of the database table to store the SAGE counts,
  the second one being the original SAGE library names, the third being
  the normalization factor that will be used to normalize counts based
  on the library with the smallest number of tags, and the forth being
  the factor based on the library with the largest number of tag.

  \code{\link{getLibInfo}} creates the file that contains the
  information about the data file.

  \code{\link{calNormFact}} calculates the normalization factor.

}
\value{
  \code{\link{mergeSAGE}} returns a list containing two file names
  \item{data}{a character string for the name of the file containing the
    merged data} 
  \item{info}{a character string for the name of the file containing
    information about the merged data}

  \code{\link{getLibInfo}} returns a matrix with four columns.
}
\references{\url{http://www.ncbi.nlm.nih.gov/geo}} 
\author{Jianhua Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}
 
\seealso{\code{\link{SAGELyzer}}}
\examples{
path <- tempdir()
# Create two libraries
lib1 <- cbind(paste("tag", 1:10, sep = ""), 1:10)
lib2 <- cbind(paste("tag", 5:9, sep = ""), 15:19)
write.table(lib1, file = file.path(path, "lib1.sage"), sep = "\t",
row.names = FALSE, col.names = FALSE)
write.table(lib2, file = file.path(path, "lib2.sage"), sep = "\t",
row.names = FALSE, col.names = FALSE) 
libNNum <- getLibNNum(c(file.path(path, "lib1.sage"),
file.path(path, "lib2.sage")))
normFact <- calNormFact("min", libNNum)
uniqTag <- getUniqTags(c(file.path(path, "lib1.sage"),
file.path(path, "lib2.sage")), skip = 0)
}

\keyword{manip}

\eof
\name{mergeSAGEWidget}
\alias{mergeSAGEWidget}
\alias{GEOSAGEWidget}
\alias{mapSAGEWidget}
\alias{SAGE4Unix}
\title{Widgets that provide the interface}
\description{
  These widgets are specific to the package and may be of litter use
  otherwise.
}
\usage{
mergeSAGEWidget()
GEOSAGEWidget()
mapSAGEWidget()
SAGE4Unix()
}

\details{
  \code{\link{mergeSAGEWidget}} provides an interface for users to input
  values for arguments for the name sage libraries, is the name a directory
  name, and the type of separator used.

  \code{\link{GEOSAGEWidget}} provides an interface for users to input
  values for arguments for the organism of concern, a directory name for
  storing data, and the url where GEO data can be downloaded.

  \code{\link{mapSAGEWidget}} provides an interface for users to input
  values for arguments that are need to map SAGE tags to UniGene ids.

  \code{\link{SAGE4Unix}} is the interface to call various functions of
  SAGElyzer. 
}
\value{
  All the widgets except \code{\link{SAGE4Unix}} return a list
  containing values for input argument.
  
}
\author{Jianhua Zhang}
\note{These functions are part of the Bioconductor porject at
  Dana-Farber Cancer Institute to provide Bioinformatics fucntionalies
  through R}

\examples{
# No example is given
}
\keyword{interface}



\eof
\name{querySAGE}
\alias{querySAGE}
\alias{getTableNames}
\alias{getColumnNames}
\title{Functions that provide an interface to allow users to query a SAGE
  library database table }
\description{
  These functions provides an interface for inputing query parameters
  for querying a table in a given database. Interface between R and
  the underlaying database management system is through Rdbi.
}
\usage{
querySAGE(args, dbObj = PgSQL())
getTableNames(args, dbObj)
getColumnNames(tableName, args, dbObj)
}

\arguments{
  \item{args}{\code{args} a list containing the arguments presented as
    name and value pairs. Valid element names include "dbname", "user",
    "password", "host", "hostaddr", and "port" }
  \item{dbObj}{\code{dbObj} a binding object for a given dbms
    (e. g. PgSQL() for postgreSQL)}
  \item{tableName}{\code{tableName} a character string for the name of a
    database table}
}
\details{
  \code{\link{getTableNames}} and \code{\link{getColumnNames}} get the
  names of selected database columns.
}
\value{
  \code{\link{getTableNames}} returns a vector of character strings for
  database table names.

  \code{\link{getColumnNames}} returns a vector of character strings for
  column names of a given database table.
}
\author{Jianhua Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}
\seealso{\code{\link{SAGE4Unix}}}
\examples{
# No example is provided as support of a database is required
}
\keyword{interface}


\eof
\name{selectLibs}
\alias{selectLibs}
\alias{getDBArgs}
\alias{getUnixDBArgs}
\alias{getBinding}
\alias{getWinDBArgs}
\alias{getTag}
\alias{getDefaults}
\alias{getSLyzerArgs}
\alias{writeSLyzerArgs}
\alias{modSLyzerArgs}
\alias{setSLyzerArgs}
\alias{runSLyzer}
\alias{writeSAGEKNN}
\alias{getSAGEKNN}
\alias{getLibCounts}
\alias{mapLib2File}
\alias{linkTag2UG}
\alias{remapTagNUG}
\alias{SAGEFromGEO}
\alias{procSAGE}
\alias{mapSAGE2UG}
\alias{SAGELyzerArgs}
\alias{showDBError}
\alias{setKNNArgs}
\alias{findNG4Tag}
\alias{ENV}
\title{Functions to support the widgets of SAGElyzer}
\description{
  SAGElyzer has several widgets that takes inputs from users before
  performing certain tasks. Functions described in this help
  documentation provides supports for the widgets. They may have limited
  reuse values for other purposes.
}
\usage{
selectLibs()
getDBArgs(what = c("map", "counts", "info"))
getUnixDBArgs(binding = "pg")
getBinding(binding = c("pg"))
getWinDBArgs(what = c("map", "counts", "info"))
getTag()
getDefaults()
getSLyzerArgs(argName = "SAGELyzerArgs")
writeSLyzerArgs(args,argName = "SAGELyzerArgs")
modSLyzerArgs(argName, value)
setSLyzerArgs()
runSLyzer()
writeSAGEKNN(knn, targetSAGE)
getSAGEKNN()
getLibCounts()
mapLib2File()
linkTag2UG()
remapTagNUG(mappings)
SAGEFromGEO()
procSAGE()
mapSAGE2UG()
showDBError()
setKNNArgs()
findNG4Tag()
}

\arguments{
  \item{what}{\code{what} - a character string that can be "map",
    "counts", or "info" to indicate the database table to be accessed}
  \item{binding}{\code{binding} - a character string for the name of the
    database binding object to be used. "pg" for PostgreSQL is the only
    one that is currently supported}
  \item{args}{\code{args} a list of arguments that will be stored in a
    predefined environment for later usr}
  \item{argName}{\code{argName} a character string for the name of an
    element in the stored list of arguments}
  \item{value}{\code{value} a value to be associated with an element of
    the stored list of arguments}
  \item{knn}{\code{knn} a named vector with the names being the tags
    that are closely related to a given target tag and values being the
    calculated distances}
  \item{targetSAGE}{\code{targetSAGE} a character string for the target
    tag selectd}
  \item{mappings}{\code{mappings} a matrix with mappings between SAGE
    tags and UniGene ids}
}

\details{
  \code{selectLibs} allows users to select a set of SAGE libraries from an
  interactive interface and store the data in an pre-defined environment
  for later use.

  \code{getDBArgs} invokes an interactive interface for users to input the
  arguments for making connections to an existing database. User inputs
  are stored in a predefined environment for later use. The function
  calls \code{\link{getUnixDBArgs}} or \code{\link{getWinDBArgs}}
  depending on the type of the operating system.

  \code{getUnixDBArgs} invokes the interface for getting arguments for
  making a connection to a database under Unix. 

  \code{getWinDBArgs} invokes an interactive interface for getting arguments
  for making a connection to a database under windows through an ODBC
  bridge.

  \code{getTag} invokes an interfactive interface for users to input the
  target SAGE tag base on which tags with similar expression pattern will be
  sought.

  \code{getDefaults} invokes an interactive interface for users to reset
  the default values  arguments for \code{\link{SAGELyzer}}.

  \code{getSLyzerArgs} returns the argument list stored in a predefined
  environment. 

  \code{writeSLyzerArgs} overwrites the argument list stored in a
  predefined environment.

  \code{modSLyzerArgs} modifies the value of a given element of the
  argument list that is stroed in a predefined environment.

  \code{setSLyzerArgs} initializes the values of the argument list and
  store teh list in a predefined environment for later use.

  \code{runSLyzer} invokes the function \code{\link{SAGELyzer}} with correct
  arguments to calculate the nearest neighbor for a given targe tag.
  
  \code{writeSAGEKNN} writes the calculated distances for tags that are
  closely related to the target tag to a predefined environment.

  \code{getSAGEKNN} gets the value for the nearest neighbor of a target
  tag that have been stored in a predefined environment.
  
  \code{getLibCounts} invokes the interface to show the actual counts of
  the nearest neighbors of a target tag.

  \code{mapLib2File} invokes the interface that displays the mapping
  between the column name of the database table for counts and the
  original file name that was used to populate the column.

  \code{linkTag2UG} invokes an HTML page with links of the tags that are
  closely related to the target tag to UniGene data page for each of the
  tags.

  \code{remapTagNUG} returns a matrix with each SAGE tag mapped to a
  single UniGene id from a matrix where a tag may be mapped to multiple
  UniGene ids separated by a ";"
  
  \code{SAGEFromGEO} invokes the interactive interface to get SAGE
  libraries from the NCBI web site.
  
  \code{procSAGE} process SAGE libraries and write the merged data to a
  database table.

  \code{mapSAGE2UG} invokes an interactive interface for taking inputs
  from users in order to map SAGE tags to UniGene ids and write the
  mappings to a database table.
}
\value{

  \code{getUnixDBArgs} returns a list containing
  \item{dbname}{\code{dbname} a character string for the name of the
    database} 
  \item{user}{\code{user} a character string for the name of the user}
  \item{password}{\code{password} a character string for a password}
  \item{host}{\code{host} a character string for the domain name or IP
    address of the host for the database}
  \item{counts}{\code{counts} a character string for the name of the
    databsae table storing SGAE count data}
  \item{info}{\code{info} a character string for the name of the
    database table storing mapping information and normalization data}
  \item{map}{\code{map} a character string for the name of the database
    table storing the mappings between tag and UniGene ids}

  \code{getWinDBArgs} returns a list containing
  \item{DSN}{\code{DSN} a character string for the name of data source
    name}
   \item{counts}{\code{counts} a character string for the name of the
    databsae table storing SGAE count data}
  \item{info}{\code{info} a character string for the name of the
    database table storing mapping information and normalization data}
  \item{map}{\code{map} a character string for the name of the database
    table storing the mappings between tag and UniGene ids}

  \code{getSLyzerArgs} returns a list of arguments that is stroed in a
  predefined environment.

  \code{getSAGEKNN} returns a named vector with tag names and their
  distances to a target tag.

  \code{remapTagNUG} returns a matrix with two columns.
}
\references{\url{http://www.ncbi.nlm.nih.gov/geo}}
\author{Jianhua Zhang}
\note{The functions are part of the Bioconductor project at Dana-Farber
  Cancer Institute to provide Bioinformatics functionalities through R}
 
\seealso{\code{\link{SAGELyzer}}, \code{\link{mergeSAGE}}}
\examples{
## No examples are provided as the functions may require the support of
## database or human interference
}
\keyword{misc}

\eof
