
#include "StdAfx.h"
#include <math.h>

#include "SpaceShip.h"
#include "GameEngine.h"
#include "GameStatus.h"
#include "GlobalSettings.h"
#include "Asteroid.h"
#include "Calculator.h"
#include "PositionCalculator.h"
#include "Shot.h"

#include "TargetSystem.h"


TargetSystem::TargetSystem(SpaceShip* agent)
: mAgent(agent), mLastShotTime(0),  mAsteroidCount(0), mAgentShotCount(0), mPrecisionFactor(0.90),
  mShotRepetitionCounter(0), mDontWaitForAsteroids(false), mMultibleShotPresicionDiv(1.5)
{
}

TargetSystem::~TargetSystem(void)
{
}

void TargetSystem::takeAimAndShot(const int& time, KeysPacket* keys)
{
	GameStatus* gameStatus = GameEngine::Instance()->getGameStatus();
	mAsteroidCount = gameStatus->getAsteroidsCount();
	MovingEntity* nextTarget = 0;

	// Feststellen wieviele Sche vom Raumschiff vorhanden sind
	mAgentShotCount = 0;
	for (list<Shot*>::iterator iterShot = gameStatus->getShotList()->begin(); iterShot != gameStatus->getShotList()->end(); iterShot++)
	{
		if ((*iterShot)->getSender() == Shot::SpaceShip)
		{
			mAgentShotCount++;
		}
	}

	if (mAgentShotCount >= 3) { mShotRepetitionCounter = 0; }

	if ((mShotRepetitionCounter > 0) && !mDontWaitForAsteroids)
	{
		shot(time, keys, 0);
		return;
	}

	mDontWaitForAsteroids = false;
	
	// Asteroiden sortieren. Je nach Position und Flugbahn werden sie in eine der verschiedenen
	// Listen gepackt
	sortAsteroids();

	// Das Schiff aufs nchste Ziel neu ausrichten und eventuell schieen
	if (((nextTarget = aimShipAtNextTarget(keys)) != 0) &&
		(nextTarget->getAlreadyUnderFire() == false))
	{
		shot(time, keys, nextTarget);
	}

	else if (mDirectShots.size() > 0)
	{	// Asteroiden vorhanden, die durch einen direkten Schu getroffen werden knnen.
		for (list<Asteroid*>::iterator iter = mDirectShots.begin(); iter != mDirectShots.end(); iter++)
		{
			// TODO: berprfen, ob dieses if ntig ist
			if (!(*iter)->getAlreadyUnderFire())
			{
				shot(time, keys, *iter);
			}
		}
	}

	// Falls Asteroiden vorhanden sind, die durch eine kurze Wartezeit abgeschossen werden knnen,
	// werden eventuelle Steuerkommandos wieder gelscht
	if (((mDirectShotsAfterWaiting.size() > 0) || (mShotRepetitionCounter > 0)) && !mDontWaitForAsteroids)
	{
		keys->clearNavigationCommands();
	}
}

void TargetSystem::shot(const int& time, KeysPacket* keys, MovingEntity* target)
{
	if (((time - mLastShotTime) > 1) && (mAgentShotCount < 4))
	{	// Nur schieen, wenn der Befehl auch wieder verarbeitet werden kann
		if (target == 0)
		{	// Wegen Schuwiederholung
 			keys->fire(true);
			mLastShotTime = time;
			mShotRepetitionCounter--;
		}
		else
		{	// Neues Ziel
			GameStatus* gameStatus = GameEngine::Instance()->getGameStatus();

 			keys->fire(true);
			mLastShotTime = time;

			int targetId = target->getId();

			MovingEntity* realObeject = gameStatus->getObjectById(targetId);
			if (realObeject != 0)
			{
				realObeject->setAlreadyUnderFire(true);
				realObeject->fireAndForget(static_cast<int>(target->getRendezvousTime()));
				realObeject->mShotCalculationMethod = target->mShotCalculationMethod;
			}

			gameStatus->addTarget(targetId);

			// Fr Mehrfachsche
			mShotRepetitionCounter = static_cast<int>(target->getRadius() / 10) - 1;
		}
	}

	if (mAgentShotCount >= 3)
	{
		mShotRepetitionCounter = 0;
	}
}

void TargetSystem::sortAsteroids()
{
	list<Asteroid*>* allAsteroids = GameEngine::Instance()->getGameStatus()->getAsteroidList();

	mCrashAsteroids.clear();
	mRestAsteroids.clear();
	mDirectShots.clear();
	mDirectShotsAfterWaiting.clear();

	AnalyseResult analyseResult = NoResult;

	int counter = 0;

	for (list<Asteroid*>::iterator iter = allAsteroids->begin(); iter != allAsteroids->end(); iter++)
	{	
		if (((*iter)->getAlreadyUnderFire() == true) || ((*iter)->getVelocityCalcCounter() < 4))
		{	// Asteroiden, die schon beschoen wurden, oder gerade erst enststanden sind,
			// werden erstmal nicht weiter betrachtet
			continue;
		}

		// Neues Objekt von dem Asteroiden erzeugen. Auf diesem werden alle Berechnungen gemacht
		mAsteroids[counter] = *(*iter); 

		analyseResult = analyseMovingEntity(mAsteroids + counter);

		switch(analyseResult)
		{
		case(DirectShotPossible): mDirectShots.push_back(mAsteroids + counter); break;
		case(CollisionHeading): mCrashAsteroids.push_back(mAsteroids + counter); break;
		case(DirectShotAfterWaiting): mDirectShotsAfterWaiting.push_back(mAsteroids + counter);  break;
		default: mRestAsteroids.push_back(mAsteroids + counter); break;
		}

		counter++;
	}
}


TargetSystem::AnalyseResult TargetSystem::analyseMovingEntity(MovingEntity* entity)
{
	AnalyseResult analyseResult = NoResult;
	double rotEntity = Calculator::calculateRotationInRad(entity->getVelocity());

	SpaceShip localSpaceShip(true);
	localSpaceShip = *mAgent;

	// lokales Raumschiff in den lokalen Raum des Objekts verfrachten 
	PositionCalculator::tansferEntityToLocalSpace(entity->getPosition(), rotEntity, &localSpaceShip);

	// Auf Crash und direkten Schu analysieren
	if ((analyseResult = analyseForDirectShot(&localSpaceShip, entity)) == NoResult)
	{	
		analyseResult = analyseForCrash(&localSpaceShip, entity);

		// Objekt in den lokalen Raum des Raumschiffs verschieben. Die spteren Berechnungen sttzen
		// sich dann alle auf den lokalen Raum
		double rotSpaceShip = Calculator::calculateRotationInRad(mAgent->getHeading());
		PositionCalculator::tansferEntityToLocalSpace(mAgent->getPosition(), rotSpaceShip, entity);
	}

	return analyseResult;	
}


TargetSystem::AnalyseResult TargetSystem::analyseForCrash(SpaceShip* spaceship, MovingEntity* entity)
{
	if (spaceship->getPosition().getX() < 0.0f) 
	{	// Raumschiff hinter dem Objekt
		return NoResult; 
	}

	// die 1.2 sollen einen kleinen Sicherheitsabstand geben
	double expandedRadius = entity->getRadius() + 1.2f * spaceship->getRadius();
	if (expandedRadius < abs(spaceship->getPosition().getY()))
	{	
		return NoResult;
	}

	// Zeitpunkt des Zusammenstoes berechnen
	entity->setRendezvousTime((spaceship->getPosition().getLength() - expandedRadius) / entity->getSpeed());

	return CollisionHeading;
}

TargetSystem::AnalyseResult TargetSystem::analyseForDirectShot(SpaceShip* spaceship, MovingEntity* entity)
{
	AnalyseResult analyseResult = NoResult;
	GlobalSettings* globalSettings = GlobalSettings::Instance();
	double shotSpeed = globalSettings->shotSpeedSpaceShip;

	double precisionFactor = mPrecisionFactor;
	if (multipleShotPossible(entity))
	{
		precisionFactor /= mMultibleShotPresicionDiv;
	}

	double rotSpaceShip = Calculator::calculateRotationInRad(spaceship->getScreenHeading());

	// Diese Vars stehen fr die Position und die Geschwindigkeit eines Schusses
	double px = spaceship->getPosition().getX();
	double py = spaceship->getPosition().getY();
	double vx = shotSpeed * cos(rotSpaceShip);
	double vy = shotSpeed * sin(rotSpaceShip);

	double c = 10000.0f;
	if (vx != 0)
	{
		c = py - vy / vx * px;
	}

	if ((((px > 0) && (vx < 0)) || ((px < 0) && (vx > 0))) &&
		(abs(py) < entity->getRadius()) && (abs(c) < (precisionFactor * entity->getRadius())))
	{	// Das Raumschiff liegt fast auf der X-Achse und ist in Richtung Objekt gerichtet
		// Rendezvous Zeit berechnen
		double timeToReachEntity = abs(px) / shotSpeed;

		if (timeToReachEntity < globalSettings->shotLifeTime)
		{
			analyseResult = DirectShotPossible;
			entity->setRendezvousTime(timeToReachEntity);
			entity->mShotCalculationMethod = MovingEntity::CalculationMethodDirectShot_1;
		}
	}

	else if (((py > 0) && (vy < 0)) || ((py < 0 ) && (vy > 0)))
	{	
		// X-Wert fr Y-Nulldurchgang berechen 
		double x0_Shot = px - py * vx / vy;

		if ((x0_Shot + (precisionFactor * entity->getRadius())) < 0)
		{	// Schu fliegt hinter dem Objekt vorbei
			analyseResult = NoResult;
		}
		else
		{	// Zeitpunkt berechnen zu dem der Schu die X-Achse schneidet
			double t0_Shot = 0.0;
			t0_Shot = sqrt((py * py * (1 + (vx * vx / (vy * vy)))) / (vx * vx + vy * vy));

			if (t0_Shot < globalSettings->shotLifeTime)
			{	// Schu ist genug am Leben
				// Berechenen wo das Objekt zu dem Zeitpunkt ist 
				double x_Object_t0 = entity->getSpeed() * t0_Shot;

				// Test, ob der Schu und das Objekt zusammentreffen
				double diffX = x0_Shot - x_Object_t0;
				double hitCondition = precisionFactor * entity->getRadius();

				if (abs(diffX) < hitCondition)
				{	// Collision -> Schu
					analyseResult = DirectShotPossible;
					entity->setRendezvousTime(t0_Shot);
					entity->mShotCalculationMethod = MovingEntity::CalculationMethodDirectShot_2;
				}
				else if ((diffX - hitCondition) > 0)
				{	// Objekt trifft spter ein -> das Objekt kann durch Warten getroffen werden
					entity->setRendezvousTime((diffX - hitCondition) / entity->getSpeed());
					if (entity->getRendezvousTime() < 4.0)
					{
						analyseResult = DirectShotAfterWaiting;
					}
				}
			}
		}
	}

	return analyseResult;
}


MovingEntity* TargetSystem::aimShipAtNextTarget(KeysPacket* keys)
{
	// Diese Methode richtet das Schiff auf neue Ziele aus. Dabei werden zuerst die Asteroiden 
	// ausgewhlt, die auf das Schiff zu fliegen. Danach die, die durch Drehung ereichbar sind
	MovingEntity* nextTarget = 0;
	bool shotPossible = false;

	GameStatus *gameStatus = GameEngine::Instance()->getGameStatus();

	if (mCrashAsteroids.size() > 0)
	{
		nextTarget = selectNextTargetFromCrashAsteroids(keys, shotPossible);
	}
	
	if (nextTarget == 0) 
	{	// Unter Umstnden Ufo jagen !!!
		nextTarget = huntSaucer(keys, shotPossible);
	}

	// Bis hier wurden Objekte behandelt, die eine direkte Bedrohung sein knnen.
	// Jetzt werden die restlichen Asteroiden untersucht
	if (nextTarget == 0)
	{
		nextTarget = selectNextTargetFromRestAsteroids(keys, shotPossible);
	}

	if (shotPossible)
	{
		return nextTarget;
	}
	else
	{
		return 0;
	}
}


MovingEntity* TargetSystem::selectNextTargetFromCrashAsteroids(KeysPacket* keys, bool& shot)
{
	Asteroid* nextTarget = 0;
	std::list<Asteroid*>::iterator iter;

	double timeToReachShip = 0.0, mintimeToReachShip = 100000000.0;

	for (iter = mCrashAsteroids.begin(); iter != mCrashAsteroids.end(); iter++)
	{	
		// Erst ein Test, wann der Asteroid auf das Schiff trift und wie lange das Schiff 
		// bentigt, um sich auf den Asteroiden auszurichten. 
		timeToReachShip = (*iter)->getRendezvousTime();
		
		if (timeToReachShip > 135)
		{	// Asteroid braucht lnger als eine komplette Umdrehung um das Schiff zu erreichen. Dieser
			// wird wie alle anderen behandelt
			mRestAsteroids.push_back(*iter);
			continue;
		}

		if (timeToReachShip < 45)
		{
			mDontWaitForAsteroids = true;
		}

		// Es wird der Asteroid ausgewhlt, der als erstes das Schiff ereichen wrde
		if (timeToReachShip < mintimeToReachShip)
		{
			mintimeToReachShip = timeToReachShip;
			nextTarget = *iter;
		}
	}

	if (nextTarget != 0)
	{	// Schiff auf Asteroid ausrichten
		turnShipAtTarget(nextTarget, keys, shot, nextTarget->getPosition().getLength() / GlobalSettings::Instance()->shotSpeedSpaceShip);
		nextTarget->mShotCalculationMethod = MovingEntity::CalculationMethodCrashAsteroid;
	}

	return nextTarget;
}

MovingEntity* TargetSystem::selectNextTargetFromRestAsteroids(KeysPacket* keys, bool& shot)
{
	MovingEntity *nextTargetLeft = 0, *nextTargetRight = 0, *nextTarget = 0;
	std::list<Asteroid*>::iterator iter;

	Vector2D predictedPosition;

	int leftAsteroids = 0, rightAsteroids = 0;
	double minCosDiffAngleLeft = -1.0, minCosDiffAngleRight = -1.0, currCosDiffAngle = 0.0;
	double rendezvousTime = 0.0, minRendezvousTimeLeft = 0.0, minRendezvousTimeRight = 0.0, minRendezvousTime = 0.0;
	double sumLeftRendezvousTimes = 1000000.0, sumRightRendezvousTimes = 1000000.0;
	double turnAroundTime = 0.0;
	double sumLeftTurnAroundTimes = 1000000.0, sumRightTurnAroundTimes = 1000000.0;

	for (iter = mRestAsteroids.begin(); iter != mRestAsteroids.end(); iter++)
	{
		// Posistion berechnen, wann ein Schu den Asteroid treffen knnte
		(*iter)->setPosition(predictPosition(*iter, rendezvousTime, turnAroundTime));

		if ((mAsteroidCount > 0) && 
			((rendezvousTime - turnAroundTime) > GlobalSettings::Instance()->shotLifeTime))
		{	// Asteroid zu weit weg
			continue;
		}

		// Winkel zwischen Asteroid und Ausrichtung berechnen
		currCosDiffAngle = Vector2D(1.0, 0.0).dotProduct((*iter)->getPosition());

		// Test, ob der Asteroid links oder rechts liegt
		if (Calculator::isVector2LeftFromVector1(Vector2D(1.0, 0.0), (*iter)->getPosition()) > 0)
		{	// Links
			if (leftAsteroids == 0) 
			{ 
				sumLeftRendezvousTimes = rendezvousTime; 
				sumLeftTurnAroundTimes = turnAroundTime; 
			}
			else 
			{ 
				sumLeftRendezvousTimes += rendezvousTime; 
				sumLeftTurnAroundTimes += turnAroundTime; 
			}

			leftAsteroids++;

			if (currCosDiffAngle > minCosDiffAngleLeft)
			{
				minCosDiffAngleLeft = currCosDiffAngle;
				minRendezvousTimeLeft = rendezvousTime;
				nextTargetLeft = *iter;
			}
		}
		else
		{	// Rechts
			if (rightAsteroids == 0) 
			{ 
				sumRightRendezvousTimes = rendezvousTime; 
				sumRightTurnAroundTimes = turnAroundTime; 
			}
			else 
			{ 
				sumRightRendezvousTimes += rendezvousTime; 
				sumRightTurnAroundTimes += turnAroundTime; 
			}

			rightAsteroids++;

			if (currCosDiffAngle > minCosDiffAngleRight)
			{
				minCosDiffAngleRight = currCosDiffAngle;
				minRendezvousTimeRight = rendezvousTime;
				nextTargetRight = *iter;
			}
		}
	}

	//////////////////////////////////////////////////////////////////////
	// Auswahl, ob wir uns links oder rechts drehen
	if (leftAsteroids > 0)
	{
		sumLeftRendezvousTimes /= (leftAsteroids * leftAsteroids);
		sumLeftTurnAroundTimes /= (leftAsteroids * leftAsteroids);
	}
	if (rightAsteroids > 0)
	{
		sumRightRendezvousTimes /= (rightAsteroids * rightAsteroids);
		sumRightTurnAroundTimes /= (rightAsteroids * rightAsteroids);
	}

//	if (sumLeftRendezvousTimes < sumRightRendezvousTimes)
	if (sumLeftTurnAroundTimes < sumRightTurnAroundTimes)
	{
		nextTarget = nextTargetLeft;
		currCosDiffAngle = minCosDiffAngleLeft;
		minRendezvousTime = minRendezvousTimeLeft;
	}
	else
	{
		nextTarget = nextTargetRight;
		currCosDiffAngle = minCosDiffAngleRight;
		minRendezvousTime = minRendezvousTimeRight;
	}

	//////////////////////////////////////////////////////////////////////
	// Nchstes Ziel anvisieren
	if (nextTarget != 0)
	{
		turnShipAtTarget(nextTarget, keys, shot, minRendezvousTime);
		nextTarget->mShotCalculationMethod = MovingEntity::CalculationMethodRestAsteroid;
	}

	return nextTarget;
}


double TargetSystem::calculateTurnAroundTime(const Vector2D& targetPos) const
{
	double dot = Vector2D(1.0, 0.0).dotProduct(targetPos);
	return acos(dot) / 0.07363; // 0.07363 entsprechen 4.21875
}

MovingEntity* TargetSystem::huntSaucer(KeysPacket* keys, bool& shot)
{
	double rendezvousTime = 0.0, turnAroundTime = 0.0;
	GameStatus* gameStatus = GameEngine::Instance()->getGameStatus();

	if (!gameStatus->getSaucer()->getIsPresent() || gameStatus->getSaucer()->getAlreadyUnderFire())
	{
		return 0;
	}
	
	mSaucer = *(gameStatus->getSaucer());

	// Ufo in den lokalen Raum des Raumschiffs versetzen
	double rotSpaceShip = Calculator::calculateRotationInRad(mAgent->getHeading());

	PositionCalculator::tansferEntityToLocalSpace(mAgent->getPosition(), rotSpaceShip, &mSaucer);

	// Posistion berechnen, wann ein Schu das Ufo treffen knnte
	mSaucer.setPosition(predictPosition(&mSaucer, rendezvousTime, turnAroundTime));

	if ((mAsteroidCount > 0) && ((rendezvousTime - turnAroundTime) > GlobalSettings::Instance()->shotLifeTime))
	{	// Ufo zu weit weg
		return 0;
	}

	turnShipAtTarget(&mSaucer, keys, shot, rendezvousTime);

	return &mSaucer;
}

void TargetSystem::turnShipAtTarget(MovingEntity* target, KeysPacket* keys, bool& shot, const double& rendezvousTime)
{
	// Test, ob wir schieen knnen
	double precisionFactor = mPrecisionFactor;
	bool multipleShot = multipleShotPossible(target);

	if (multipleShot)
	{
		precisionFactor /= mMultibleShotPresicionDiv;
	}

	if (abs(target->getPosition().getY()) < (precisionFactor * target->getRadius()))
	{	
		shot = true;
		target->setRendezvousTime(rendezvousTime);
	}

	// Falls wir mehrfach schieen knnen, drehen wir nicht weiter 
	if (multipleShot && shot) { return; }

	/////////////////////////////////////////////////////////////////
	// Falls nicht geschoen wurde testen wir, ob es besser ist zu drehen oder zu warten
	MovingEntity localTarget(true);
	localTarget = *target;

	localTarget.predictPosition(1);

	double cosWaiting = Vector2D(1.0, 0.0).dotProduct(localTarget.getPosition());

	SpaceShip localAgent(true);
	localAgent = *mAgent;

	double cosTurn;
	bool targetIsLeft = false;
	if (Calculator::isVector2LeftFromVector1(Vector2D(1.0, 0.0), localTarget.getPosition()) > 0)
	{
		targetIsLeft = true;
		cosTurn = Vector2D(0.9973, 0.0736).dotProduct(localTarget.getPosition());
	}
	else
	{
		cosTurn = Vector2D(0.9973, -0.0736).dotProduct(localTarget.getPosition());
	}

	if (cosWaiting < cosTurn)
	{
		/////////////////////////////////////////////////////////////////
		if (Calculator::isVector2LeftFromVector1(Vector2D(1.0, 0.0), target->getPosition()) > 0)
		{
			keys->left(true);
		}
		else
		{
			keys->right(true);
		}
	}
}


Vector2D TargetSystem::predictPosition(MovingEntity* entity, double& rendezvousTime, double& turnAroundTime) const
{
	Vector2D toObject, predictedPosition, prevPredictedPos;

	int counter = 1, maxIterationCount = 20;

	// 1. Iteration. Muss auerhalb der Schleife gemacht werden, da fr prevPredictedPos die
	// derzeitige Postion des Objekts verwendet wird
	predictedPosition = predictPosition(entity->getPosition(), entity->getPosition(), entity->getVelocity(), rendezvousTime, turnAroundTime);

	do 
	{	// Alle anderen Iterationen in der Schleife machen
		prevPredictedPos = predictedPosition;
		predictedPosition = predictPosition(entity->getPosition(), prevPredictedPos, entity->getVelocity(), rendezvousTime, turnAroundTime);
		counter++;
	}
	while ((counter < maxIterationCount) && 
		   (!predictedPosition.approximatelyEqual(prevPredictedPos)));

	// Position noch umbrechen
	if (predictedPosition.getLengthSq() > GlobalSettings::Instance()->shotDistanceSpaceShipSq)
		return PositionCalculator::wrapAroundLocalCoordinates(predictedPosition);
	else
		return predictedPosition;
}

Vector2D TargetSystem::predictPosition(const Vector2D& currentPosition, const Vector2D& prevPredictedPosition, const Vector2D velocity, double& rendezvousTime, double& turnAroundTime) const
{
	double lookAheadTime = 0.0;

	lookAheadTime = prevPredictedPosition.getLength() / GlobalSettings::Instance()->shotSpeedSpaceShip;

	turnAroundTime = calculateTurnAroundTime(prevPredictedPosition);
	lookAheadTime += turnAroundTime;
	rendezvousTime = lookAheadTime;

	return (currentPosition + velocity * lookAheadTime);
}

bool TargetSystem::multipleShotPossible(MovingEntity* entity) const
{
	return ((entity->getRadius() > 15) && (mAgentShotCount < 2));
}
