// player.h: Beispielspieler fr Asteroids
// Harald Bgeholz / c't
#if defined(WINDOWS)
#define ADDRESS DWORD
#else
#define SOCKET int
#define ADDRESS in_addr_t
// 3 Includes fr sockaddr_in
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#include <assert.h>
#include <math.h>
#include "turn.h"
#include "Vector2d.h"

static const int MAX_ASTEROIDS = 100;
static const int MAX_SHOTS = 10;

static const int LOOKAHEAD = 120; // Soviele Frames aus der Kristallkugel lesen

double normalizeDX(double dx);
double normalizeDY(double dy);



class MovingObject 
{
public:
	double t;
	Vector r;
	Vector v;

	virtual void Show(char *tag=0) const{
		if (tag) fprintf(stdout, "%s ", tag);
		fprintf(stdout, "t=%f pos(%f, %f) vel(%f, %f)\n",
						 t, r.x, r.y,       v.x, v.y);
	}

	// past is our past 
	virtual void CalculateSpeed(const MovingObject& past) {
		v = (r - past.r) / (t - past.t);
	}
	// ship should have it's absolute speed set
	virtual void CalculateRelative(const MovingObject& ship) {
		assert(0 == "no relative pos/speed in Object");
	}
	// we project ourselfs into the future
	virtual void CalculateProjection(MovingObject& future, double t_abs) const {
		future = *this;
		future.t = t_abs;
		future.r = r + v * (t_abs - t);
		future.v = v;
	}
	virtual double rsquare() const {return 0;}
	virtual double vsquare() const {return v.square();}
	virtual double squareDist(const Vector& rhs, Vector* dist=0) const{
		Vector d = (r - rhs).asteroids_norm();
		if (dist) *dist=d;
		return d.square();
	}
	virtual double squareDist(const MovingObject& rhs, Vector* dist=0) const {
		return squareDist(rhs.r, dist);
	}
	virtual bool collidesWith(const MovingObject& other, double narrow = 1.0) const {
		return narrow * (rsquare() + other.rsquare()) > squareDist(other);
	}
};

class Asteroid : public MovingObject
{
public:
	int type; // 1 ... 4, uere Form
	int sf;   // scale factor: 0 = gro, 15 = mittel, 14 = klein

	virtual void Show(char *tag=0) const {
		if (0 == tag) tag = "Asteroid";
		MovingObject::Show(tag);
	}

	void set(int x, int y, int type, int sf, double t);
	double rsquare() const {
		switch (sf)
		{	// ungefhrer Radius zum Quadrat
			case 0:  // groer Asteroid
				return 40*40;
			case 15: // mittlerer Asteroid
				return 20*20;
			case 14: // kleiner Asteroid
				return 8*8;
		}
		return 0;
	}
};

class Shot : public MovingObject
{
public:
	// g == global
	int gScale;
	// p == pause
	int pOpcode;
	// l == local
	int lOpcode;

	virtual void Show(char *tag=0) const {
		if (0 == tag) tag = "Shot";
		MovingObject::Show(tag);
	}


	void set(double X, double Y) {
		r.x = X;
		r.y = Y;
		v.x = v.y = 0;
	}

};

class Ship : public MovingObject 
{
public:
	double dx;        // Blickrichtung des Schiffes
	double dy;

	virtual void Show(char *tag=0) const {
		if (0 == tag) tag = "Ship";
		MovingObject::Show(tag);
	}


	double rsquare() const {return 27*27;}
};

class Saucer : public MovingObject
{
public:
	int size;    // Gre: 15 = gro, 14 = klein

	virtual void Show(char *tag=0) const {
		if (0 == tag) tag = "Saucer";
		MovingObject::Show(tag);
	}


	double rsquare() const {
		switch (size)
		{	// Abstand um den ungefhren Radius des UFOs korrigieren
			case 15: // groes UFO
				return 20*12;
			case 14: // kleines UFO
				return 10*6;
		}
		return 0;
	}
};

class GameStatus
{
public:
	bool ship_present;  // Schiff sichtbar
	Ship ship;
	bool saucer_present;// UFO sichtbar
	Saucer saucer;
	int nasteroids; // Anzahl Asteroiden
	Asteroid asteroids[MAX_ASTEROIDS];
	int nshots;     // Anzahl Schsse
	Shot shots[MAX_SHOTS];
	void clear(void);

	void Show(char *tag = 0) const {
		fprintf(stdout, "Begin GameStatus %s\n", tag ? tag : "");
		if (ship_present) {
			ship.Show();
			for(int i=0; i<nshots; i++) 
				if (shots[i].collidesWith(ship))
					fprintf(stdout, "Shot %d hits ship.\n", i);
		}
		if (saucer_present) {
			saucer.Show();
			for(int i=0; i<nshots; i++) 
				if (shots[i].collidesWith(saucer))
					fprintf(stdout, "Shot %d hits saucer.\n", i);
		}
		for(int i=0; i<nasteroids; i++) {
			asteroids[i].Show();
			for(int j=0; j<nshots; j++) 
				if (shots[j].collidesWith(asteroids[i]))
					fprintf(stdout, "Shot %d hits asteroid %d.\n", j, i);
		}
		for(int i=0; i<nshots; i++) shots[i].Show();
		fprintf(stdout, "End GameStatus %s\n", tag ? tag : "");
	};

	// Ist der Schu von uns? (Oder vom Ufo?)
	bool OurShot(int index, bool haveSpeeds, 
		bool RetvalIfUnsure = false /* Pessimistisch... */ ) {
		assert(index >= 0 && index < nshots);
		if (index > 3) return false; // Ufo-Schsse sind immer hinten im Array.
		if (nshots > 4 && index < 4) return true; // Unsere immer vorne.
		Shot& shot = shots[index];
		// Eingehend?
		if (haveSpeeds) {
			//double outx = shot.rx * shot.vrx;
			//double outy = shot.ry * shot.vry;
			//if (outx > 0.0 || outy > 0.0) return true;
			
			if ( Vector::cos(ship.r - shot.r, ship.v) > 0.0) return true;
		}
		// Letzter Versuch: Beim Entstehen liegen unsere Schsse im Schiff.
		// Falls ein Ufo-Schu soweit kommt, ist es eh' zu spt.
		//double rx = normalizeDX( ship.x - shot.x );
		//double ry = normalizeDY( ship.y - shot.y );
		//double dist = rx*rx+ry*ry;
		//if (dist < 20*20) return true;
		if ((ship.r - shot.r).asteroids_norm().square() < 20 * 20) return true;

		return RetvalIfUnsure;
	}

	// past is our past 
	bool CalculateSpeed(const GameStatus& past) {
		if (nshots != past.nshots) return false;
		if (nasteroids != past.nasteroids) return false;
		if (ship_present != past.ship_present) return false;
		if (saucer_present != past.saucer_present) return false;
		for(int i=0; i< nasteroids; i++) 
			if (asteroids[i].sf != past.asteroids[i].sf
				|| asteroids[i].type != past.asteroids[i].type)
					return false;
		

		if (ship_present) ship.CalculateSpeed(past.ship);
		if (saucer_present) saucer.CalculateSpeed(past.saucer);
		for(int i=0; i<nshots; i++) 
			shots[i].CalculateSpeed(past.shots[i]);
		for(int i=0; i<nasteroids; i++) 
			asteroids[i].CalculateSpeed(past.asteroids[i]);
		return true;
	}


	// we project ourselfs into the future
	void CalculateProjection(GameStatus& future, double t_abs) const {
		future = *this;
		if (ship_present) {
			future.ship_present = true;
			future.ship.t = t_abs;
			future.ship.r = ship.r + ship.v * (t_abs - ship.t);
			future.ship.v = ship.v;
		}
		if (saucer_present) {
			future.saucer_present = true;
			saucer.CalculateProjection(future.saucer, t_abs);
		}
		future.nshots = nshots;
		for(int i=0; i< future.nshots; i++) {
			future.shots[i] = shots[i];
			shots[i].CalculateProjection(future.shots[i], t_abs);
		}
		future.nasteroids = nasteroids;
		for(int i=0; i< future.nasteroids; i++) {
			future.asteroids[i] = asteroids[i];
			asteroids[i].CalculateProjection(future.asteroids[i], t_abs);
		}
	}

	const MovingObject* shipCollides(int& count, double narrow = 1.0) {
		return shipCollidesFromReference(*this, count, narrow);
	}

	const MovingObject* shipCollidesFromReference(const GameStatus& ref, int& count, double narrow = 1.0) {
		count = 0;
		if (!ship_present) return 0;
		const MovingObject *R = 0;
		for(int i=0; i<nshots; i++) 
			if (!OurShot(i, true, true)) { 
				if (ship.collidesWith(shots[i], narrow)) {
					if (!R) R = ref.shots+i;
					assert(ref.nshots>i);
					count++;
				}
			}
		if (saucer_present && ship.collidesWith(saucer, narrow)) {
			if (!R) R = &ref.saucer;
			assert(ref.saucer_present);
			count++;
		}
		for(int i=0; i<nasteroids; i++) 
			if (ship.collidesWith(asteroids[i], narrow)) {
				if (!R) R = ref.asteroids+i;
				assert(ref.nasteroids > i);
				count++;
			}
		return R;
	}

	bool shotHits(const Shot& S, double narrow = 1.0) {
		bool c = saucer_present && S.collidesWith(saucer, narrow);
		if (c) return true;
		for(int i=0; i<nasteroids; i++) c |= S.collidesWith(asteroids[i], narrow);
		return c;
	}

};

#pragma pack(1)
struct FramePacket
{
	char vectorram[1024];
	char frameno;  // wird bei jedem Frame inkrementiert
	char ping;     // Der Server schickt das letzte empfangene ping-Byte zurck
};

class KeysPacket
{
public:
	static const char KEY_HYPERSPACE = 1;
	static const char KEY_FIRE = 2;
	static const char KEY_THRUST = 4;
	static const char KEY_RIGHT = 8;
	static const char KEY_LEFT = 0x10;
private:
	char signature[6];
	char keys;
public:
	char ping;     // wird vom Server bei nchster Gelegenheit zurckgeschickt. Fr Latenzmessung.

	KeysPacket(void);
	void clear(void);         // alle Tasten loslassen
	void hyperspace(bool b);  // Hyperspace drcken (true) oder loslassen (false)
	void fire(bool b);        // Feuerknopf drcken (true) oder loslassen (false)
	void thrust(bool b);      // Beschleunigen ...
	void right(bool b);       // rechts drehen ...
	void left(bool b);        // links drehen
	char getKeys() {return keys;}
};
#pragma pack()


class Player
{
public:
	Player(SOCKET sd, ADDRESS server_ip) : sd(sd), server_ip(server_ip) {};
	void OriginalStrategy(int t, const GameStatus& gameToUse, KeysPacket& keys);
	void BetterStrategy(int t, const GameStatus& gameToUse, KeysPacket& keys, bool haveSpeeds);
    void RunComplex(void);
    void RunSimple(void);
    void RunMeasureShots(void);
	void InterpretScreen(FramePacket &packet, GameStatus& game, double t);
	void ReceivePacket(FramePacket &packet);
	void SendPacket(KeysPacket &packet);
	void SendName();

	void GotoCenter(const GameStatus& game, KeysPacket& keys, bool haveSpeeds) {
		if (!game.ship_present) return;
		Vector ra = (game.ship.r - Vector(524.0, 524.0) ).asteroids_norm();

		if (ra.square() < 350*350) {
			SlowDownShip(game, keys, haveSpeeds);
			return;
		}

		fprintf(stdout, "GotoCenter\n");

		// normierte Ausrichtung des Schiffes
		Vector dn(game.ship.dx, game.ship.dy);
		dn.normalize();

		if (dn % ra > 0)
			keys.right(true);
		else
			keys.left(true);
		
		double cos_phi = Vector::cos(ra, dn);

		if (cos_phi < -0.9 && game.ship.vsquare() < 9) keys.thrust(true);
	}

	void SlowDownShip(const GameStatus& game, KeysPacket& keys, bool haveSpeeds) {
		if (!haveSpeeds || !game.ship_present) return; 
		fprintf(stdout, "SlowDownShip\n");
		// Berechne normierte absolute Geschwindigkeit
		Vector vn = game.ship.v.normalized();
		Vector dn = Vector(game.ship.dx, game.ship.dy).normalized();

		double cos_phi = Vector::cos(vn, dn);

		if (cos_phi < -0.98 && game.ship.v.square() > 2 ) keys.thrust(true);

		if (dn % vn > 0)
			keys.right(true);
		else
			keys.left(true);

	}

	bool AttackSaucer(int t, int angle_byte, const GameStatus& game, KeysPacket& keys, bool haveSpeeds, int latency) 
	{
		fprintf(stdout, "AttackSaucer\n");
		assert(game.saucer_present);
		double dist = (game.saucer.r - game.ship.r).square();
		bool straight = false;
		double num_turns, travel_time;

		if ( game.nasteroids < 5 && dist > 400*400 && game.ship.vsquare() < 9) {
			double time = AttackObjectWithoutThrust3(angle_byte, game.saucer, game, keys, haveSpeeds, latency, true, &num_turns, &travel_time);
			straight = (num_turns<=1.0);
			keys.thrust( straight );
		} else {
			double time = AttackObjectWithoutThrust3(angle_byte, game.saucer, game, keys, haveSpeeds, latency, true, &num_turns, &travel_time);
			straight = (num_turns<=1.0);
		}
		return straight;
	}

	bool AttackObject(int angle_byte, const MovingObject& threat, const GameStatus& game, KeysPacket& keys, bool haveSpeeds, int latency, bool aiming = true) 
	{
		fprintf(stdout, "AttackObject\n");
		double time = AttackObjectWithoutThrust3(angle_byte, threat, game, keys, haveSpeeds, latency, aiming);
		double dist = (threat.r - game.ship.r).asteroids_norm().square();
		if (dist > 400*400 && game.ship.vsquare() < 9)
			keys.thrust(true);
		return time < 60;
	}

	double AttackObjectWithoutThrust3(
		int angle_byte, const MovingObject& threat, const GameStatus& game, KeysPacket& keys, bool haveSpeeds, int latency,
		bool aiming = true, double *ret_num_turns = 0, double *ret_pure_travel_time = 0) 
	{
		if(!haveSpeeds) {
			// simpel
			Vector d = threat.r - game.ship.r;
			if (game.ship.dx * d.y - game.ship.dy * d.x > 0)
				keys.left(true);
			else
				keys.right(true);
			if (ret_num_turns) *ret_num_turns = 1e6;
			if (ret_pure_travel_time) *ret_pure_travel_time = 1e6;
			return 2e6;
		}


		// fprintf(stdout, "AttackObjectWithoutThrust\n");
		Vector target = (threat.r - game.ship.r).asteroids_norm();
		double dist = target.square();

		// normierte Ausrichtung des Schiffes
		double angle_r = radian_from_angle_byte(angle_byte);
		double angle_d = r2d(angle_r);
		double dx_n = cos(angle_r);
		double dy_n = sin(angle_r);
		Vector dn(cos(angle_r), sin(angle_r));
		Vector cdn = dn;

		// Meine Version von "Vorhalten".
		// c steht fr "corrected", na mal sehen
		Vector ctarget = target;
		Vector ctarget_n = ctarget.normalized();
		double pure_travel_time = sqrt(dist) / 8.0; // Schsse fliegen mit etwa 8 Pixel pro Frame.
		double travel_time = pure_travel_time;
		double last_travel_time = 1000;
		double num_turns = 0;
		double last_num_turns = 100;
		//fprintf(stdout, "%f %f %f\n", crx, cry, travel_time);
		if (haveSpeeds && aiming) 
		for (int i = 0; i<30; i++) {
			ctarget = (threat.r + threat.v * (travel_time+latency) - game.ship.r - 19.0 * cdn).asteroids_norm();
			ctarget_n = ctarget.normalized();
			dist = ctarget.square();
			// Traveltime ist die Zeit zum Drehen plus die Zeit, die der Schu
			// unterwegs ist.
			last_travel_time = pure_travel_time;
			pure_travel_time = sqrt(dist) / 8.0;
			travel_time = pure_travel_time + num_turns;
			// fprintf(stdout, "%f %f %f %f\n", ctarget.x, ctarget.y, travel_time, num_turns);

			// Winkel, unter dem das Ziel erscheint
			double angle_threat_r = atan2(ctarget.y, ctarget.x);
			double angle_threat_d = r2d(angle_threat_r);
			
			double delta_d = min( fabs(angle_threat_d - angle_d),
				min( fabs(angle_threat_d + 360 - angle_d),
				min( fabs(angle_d - angle_threat_d),
				fabs(angle_d + 360 - angle_threat_d)
				)));

			last_num_turns = num_turns;
			num_turns = fabs(delta_d / STEP);
			// Ausrichtung des Schiffes nach Drehung neu berechnen
			cdn = ctarget_n;
		}

		// Ist die Lsung konvergiert?
		if (fabs(num_turns - last_num_turns) > 0.0001) {
			// nein!
			if (ret_num_turns) *ret_num_turns = 1e6;
			if (ret_pure_travel_time) *ret_pure_travel_time = 1e6;
			return 2e6;
		}

		if ((dn % ctarget_n) > 0) { 
			keys.left(true); // fprintf(stdout, "ATTACK LLLEFT %p\n", &threat);
		} else if ((dn % ctarget_n) < 0) {
			keys.right(true); // fprintf(stdout, "ATTACK RRIGHT %p\n", &threat);
		} else {keys.left(false);keys.right(false);}

		keys.fire(	fabs(dn % ctarget_n) < 0.5 * sqrt(threat.rsquare())/sqrt(dist)  
					&&
					pure_travel_time < 70.0
					);

		//keys.thrust( dist > 400*400 && fabs(num_turns) < 2 && haveSpeeds && game.ship.vsquare() < 2*2);

		if (ret_num_turns) *ret_num_turns = num_turns;
		if (ret_pure_travel_time) *ret_pure_travel_time = pure_travel_time;
		return travel_time;
	}



	bool AttackFutureAsteroid(int angle_byte, const GameStatus *future, KeysPacket& keys, bool haveSpeeds, int latency, bool aiming = true)
	{
		fprintf(stdout, "AttackFutureAsteroid\n");
		double min_dist = 1.0e6;
		int min_i = 0;
		int min_f = 0;
		double dist;
		int l = LOOKAHEAD;
		for(int f=l-1; f>=0; f--)
		for(int i=0; i<future[f].nasteroids; i++) {
			Vector d = (future[f].asteroids[i].r - future[f].ship.r).asteroids_norm();
			dist = d.square() - future[f].asteroids[i].rsquare();
			if (dist <= min_dist) {
				min_dist = dist;
				min_i = i;
				min_f = f;
			}
		}
		return AttackObject(angle_byte, future[0].asteroids[min_i], future[0], keys, haveSpeeds, latency, aiming);
	}

	bool AttackNearestAsteroid(int angle_byte, const GameStatus& game, KeysPacket& keys, bool haveSpeeds, int latency, bool aiming = true)
	{
		static int last_i = -1;
		static double last_dist = 1.0e6;
		fprintf(stdout, "AttackNearestAsteroid\n");
		double min_dist = 1.0e6;
		int min_i = 0;
		double dist;
		for(int i=0; i<game.nasteroids; i++) {
			Vector d = (game.asteroids[i].r - game.ship.r).asteroids_norm();
			dist = d.square() - game.asteroids[i].rsquare();
			if (dist<min_dist) {
				min_dist = dist;
				min_i = i;
			}
			if (i == last_i) last_dist = dist;
		}

		/*if (last_i < game.nasteroids && -1 != last_i && 1.6 * min_dist > last_dist) {
			min_i = last_i;
			fprintf(stdout, "AttackNearestAsteroid last_i chosen \n", min_i);
		} else */{
			last_i = min_i;
			last_dist = min_dist;
		}
		fprintf(stdout, "AttackNearestAsteroid chosen = %d\n", min_i);
		AttackObjectWithoutThrust3(angle_byte, game.asteroids[min_i], game, keys, haveSpeeds, latency, aiming);
		return (keys.getKeys() & keys.KEY_FIRE) != 0;
	}

	double AttackCheapestObject(int angle_byte, const GameStatus& game, KeysPacket& keys, bool haveSpeeds, int latency, bool aiming = true)
	{
		fprintf(stdout, "AttackCheapestObject\n");
		static int last_i = -2;
		static double last_time = 1e7;

		double min_time = 1e7;
		int min_i = 0;
		const MovingObject *min_o = 0, *o = 0; 

		for(int i=-1; i<game.nasteroids; i++) {
			if (-1 == i && !game.saucer_present) continue;
			if (-1 == i) o = &game.saucer; else o = game.asteroids+i;
			double num_turns, pure_travel_time;
			double time = AttackObjectWithoutThrust3(angle_byte, *o, game, keys, haveSpeeds, latency, true, &num_turns, &pure_travel_time);
			time = pure_travel_time*pure_travel_time+num_turns;
			if (-1 == i) time /= 2.0;
			if (time < min_time) {
				min_time = time;
				min_i = i;
				min_o = o;
			}
		}

		if (!haveSpeeds) last_time = 1e7; // delete sticky object, because it maybe died away

		if (last_i >= 0 && last_i < game.nasteroids && last_time * 3 < min_time * 4) {
			fprintf(stdout, "AttackCheapestObject sticky\n");
			min_i = last_i;
			min_o = game.asteroids+min_i;
		} else {
			last_i = min_i;
			last_time = min_time;
		}

		keys.clear();

		fprintf(stdout, "AttackCheapestObject chosen i = %d\n", min_i);
		if (min_o == 0) {keys.clear();return 1000000;}
		return AttackObjectWithoutThrust3(angle_byte, *min_o, game, keys, haveSpeeds, latency, aiming);
	}

	void EscapeFromObject(const MovingObject& threat, const GameStatus& game, KeysPacket& keys, bool haveSpeeds) 
	{
		fprintf(stdout, "EscapeFromObject\n");

		if (!haveSpeeds) {
			Vector d = (threat.r - game.ship.r).asteroids_norm();
			if (game.ship.dx * d.y - game.ship.dy * d.x > 0)
				keys.right(true);
			else
				keys.left(true);
			keys.thrust(true);
			return;
		}

		// Diese vier Quadranten mu man sich dann mal aufmalen...
		Vector threat_vrel = threat.v - game.ship.v;
		Vector ship_look(game.ship.dx, game.ship.dy);
		double dot   = ship_look * threat_vrel;
		double cross = ship_look % threat_vrel;

		if (cross > 0) {
			if (dot>0) keys.right(true);
			else keys.left(true);
		} else {
			if (dot>0) keys.left(true);
			else keys.right(true);
		}

		keys.thrust(true);
	}

	static void InitializeShotFromShip(int angle_byte, const Ship& ship, Shot& shot) {
		double angle_r = radian_from_angle_byte(angle_byte);
		double dx_n = cos( angle_r );
		double dy_n = sin( angle_r );
	
		shot.t = ship.t;
		shot.r = ship.r + Vector(dx_n, dy_n) * 19;
		shot.v = ship.v + Vector(dx_n, dy_n) * 8.0;
		if (shot.v.x > 111.0/8.0) shot.v.x = 111.0/8.0;
		if (shot.v.x <-111.0/8.0) shot.v.x =-111.0/8.0;
		if (shot.v.y > 111.0/8.0) shot.v.y = 111.0/8.0;
		if (shot.v.y <-111.0/8.0) shot.v.y =-111.0/8.0;
	}

private:
	SOCKET sd;
	ADDRESS server_ip;
};
