package gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableRowSorter;

import util.Statistics;

public class HighscoreTable extends JPanel {

	private static final long serialVersionUID = 1L;
	
	
	private ArrayList<Object[]> highscores;
	
	private JTable stats;
	private DefaultTableColumnModel columnModel;
	private StatisticsTableModel tableModel;
	private TableRowSorter<StatisticsTableModel> sorter;
	
	
	public HighscoreTable() {
		loadHighscores();
		initModels();

		stats = new JTable(tableModel, columnModel);
		stats.setRowSorter(sorter);
		stats.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		stats.setShowGrid(false);
		stats.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

		setPreferredSize(new Dimension(500, 200));
		setLayout(new BorderLayout());
		setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Highscores"));
		JScrollPane sp = new JScrollPane(stats);
		sp.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
		add(BorderLayout.CENTER, sp);
		
		updateTable();
	}
	
	
	private void loadHighscores() {
		try {
			ObjectInputStream in = new ObjectInputStream(new FileInputStream("highscores.dat"));
			highscores = (ArrayList<Object[]>) in.readObject();
			in.close();
		} catch(Exception e) {
			highscores = new ArrayList<Object[]>();
		}
	}
	
	
	public void saveHighscores() {
		try {
			ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream("highscores.dat"));
			out.writeObject(highscores);
			out.close();
		} catch(Exception e) {
			e.printStackTrace();
		}
	}


	private void initModels() {
		columnModel = new DefaultTableColumnModel();
		int col = 0;
		
		TableColumn nr = new TableColumn(col++, 200);
		nr.setHeaderValue("Name");
		columnModel.addColumn(nr);
		
		TableColumn score = new TableColumn(col++, 75);
		score.setHeaderValue("Punkte");
		columnModel.addColumn(score);
		
		TableColumn scorePM = new TableColumn(col++, 75);
		scorePM.setHeaderValue("Punkte/min");
		columnModel.addColumn(scorePM);
		
		TableColumn wave = new TableColumn(col++, 75);
		wave.setHeaderValue("Wave");
		columnModel.addColumn(wave);
		
		TableColumn lifes = new TableColumn(col++, 75);
		lifes.setHeaderValue("Leben");
		columnModel.addColumn(lifes);
		
		TableColumn deaths = new TableColumn(col++, 75);
		deaths.setHeaderValue("Tode");
		columnModel.addColumn(deaths);
		
		TableColumn jumps = new TableColumn(col++, 75);
		jumps.setHeaderValue("Sprnge");
		columnModel.addColumn(jumps);
		
		TableColumn kills = new TableColumn(col++, 75);
		kills.setHeaderValue("Abschsse");
		columnModel.addColumn(kills);
		
		TableColumn shots = new TableColumn(col++, 75);
		shots.setHeaderValue("Schsse");
		columnModel.addColumn(shots);
		
		TableColumn hitRate = new TableColumn(col++, 75);
		hitRate.setHeaderValue("Trefferrate");
		columnModel.addColumn(hitRate);
		
		TableColumn droppedFrames = new TableColumn(col++, 75);
		droppedFrames.setHeaderValue("Dropped Frames");
		columnModel.addColumn(droppedFrames);
		
		TableColumn syncLost = new TableColumn(col++, 75);
		syncLost.setHeaderValue("Winkelsync verloren");
		columnModel.addColumn(syncLost);
		
		TableColumn lostPackets = new TableColumn(col++, 75);
		lostPackets.setHeaderValue("Verlorene Pakete");
		columnModel.addColumn(lostPackets);

		tableModel = new StatisticsTableModel(columnModel.getColumnCount());
		tableModel.setRowCount(1);
		sorter = new TableRowSorter<StatisticsTableModel>(tableModel);
	}
	
	
	public void setHighscore() {
		Object[] data = new Object[13];
		int i=0;
		data[i++] = getDumpName();
		data[i++] = Statistics.get(Statistics.KEY_SCORE);
		data[i++] = Statistics.get(Statistics.KEY_SCORE_PER_MIN);
		data[i++] = Statistics.get(Statistics.KEY_WAVE);
		data[i++] = Statistics.get(Statistics.KEY_LIFES);
		data[i++] = Statistics.get(Statistics.KEY_DEATHS);
		data[i++] = Statistics.get(Statistics.KEY_HYPERJUMPS);
		data[i++] = Statistics.get(Statistics.KEY_KILLS);
		data[i++] = Statistics.get(Statistics.KEY_SHOTS);
		data[i++] = Statistics.get(Statistics.KEY_HIT_RATE);
		data[i++] = Statistics.get(Statistics.KEY_DROPPED_FRAMES);
		data[i++] = Statistics.get(Statistics.KEY_SYNC_LOST_CNT);
		data[i++] = Statistics.get(Statistics.KEY_LOST_PACKETS);
		highscores.add(data);
		updateTable();
	}
	
	
	private String getDumpName() {
		// Dump File Namen zusammenbauen
		/*GregorianCalendar cal = new GregorianCalendar();
		cal.add(Calendar.MINUTE, -5);
		String name = String.format("asteroids-%4d-%02d-%02d-%02d-%02d-%02d.gz",
				cal.get(Calendar.YEAR),
				cal.get(Calendar.MONTH)+1,
				cal.get(Calendar.DAY_OF_MONTH),
				cal.get(Calendar.HOUR_OF_DAY),
				cal.get(Calendar.MINUTE),
				cal.get(Calendar.SECOND));*/
		
		File f = new File("." + File.separatorChar + "mame");
		File[] l = f.listFiles();
		Arrays.sort(l, new Comparator<File>() {
			@Override
			public int compare(File o1, File o2) {
				long t1 = o1.lastModified();
				long t2 = o2.lastModified();
				if(t1 > t2) {
					return -1;
				} else if(t1 == t2) {
					return 0;
				} else {
					return 1;
				}
			}
		});
		return l[0].getName();
	}


	private void updateTable() {
		int cnt = highscores.size();
		tableModel.setRowCount(cnt+1);
		float[] avg = new float[12];
		for(int i=0; i<avg.length; i++) avg[i] = 0;
		
		for(int i=0; i<cnt; i++) {
			Object[] data = highscores.get(i);
			int j=0;
			for(int n=0; n<avg.length; n++) avg[n] += (Float)data[n+1];
			tableModel.setValueAt(data[j], i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SCORE, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SCORE_PER_MIN, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_WAVE, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_LIFES, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_DEATHS, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_HYPERJUMPS, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_KILLS, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SHOTS, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_HIT_RATE, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_DROPPED_FRAMES, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SYNC_LOST_CNT, data[j]), i, j++);
			tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_LOST_PACKETS, data[j]), i, j++);
		}
		
		int j = 0;
		tableModel.setValueAt("Schnitt", cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SCORE, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SCORE_PER_MIN, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_WAVE, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_LIFES, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_DEATHS, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_HYPERJUMPS, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_KILLS, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SHOTS, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_HIT_RATE, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_DROPPED_FRAMES, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_SYNC_LOST_CNT, avg[j-1]/cnt), cnt, j++);
		tableModel.setValueAt(Statistics.getFormattedValue(Statistics.KEY_LOST_PACKETS, avg[j-1]/cnt), cnt, j++);
	}
}
