package gui;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.Timer;

import util.MameFrameListener;
import util.MameRobot;
import util.Statistics;
import core.GameInfo;
import core.MameClient;
import core.bot.AsteroidBot;

public class AsteroidBotWnd extends JFrame implements ActionListener, MameFrameListener {

	private static final long serialVersionUID = 1L;
	
	
	private AsteroidScreen screen;
	private String host;
	private MameClient mame;
	private AsteroidBot bot;
	
	private HighscoreTable highscores;
	private StatisticsTable stats;
	private JCheckBox batchMode;
	
	private Timer busyTimer;
	private int busyTime;
	
	
	public AsteroidBotWnd(String host, boolean autoConnect) {
		this.host = host;
		
		initGUI();
		initFrame();
		
		if(autoConnect) {
			connectMame();
		}
	}

	
	private void initFrame() {
		setTitle("Asteroid Bot ["+host+" - not connected]");
		pack();
		Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
		setLocation((d.width-getWidth())/2, (d.height-getHeight())/2);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				exit();
			}
		});
		
		setVisible(true);
	}


	private void initGUI() {
		Container con = getContentPane();
		GridBagLayout gb = new GridBagLayout();
		GridBagConstraints c = new GridBagConstraints();
		con.setLayout(gb);
		
		c.insets = new Insets(2, 2, 2, 2);
		c.fill = GridBagConstraints.BOTH;
		c.anchor = GridBagConstraints.WEST;
		c.gridwidth = 1;
		c.gridheight = 1;
		c.weightx = 1;
		c.weighty = 0;
		highscores = new HighscoreTable();
		gb.setConstraints(highscores, c);
		con.add(highscores);

		c.weightx = 0;
		c.weighty = 0;
		c.gridwidth = GridBagConstraints.REMAINDER;
		c.gridheight = 1;
		JPanel ctrlPanel = new JPanel();
		ctrlPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Einstellungen"));
		BoxLayout ctrlLayout = new BoxLayout(ctrlPanel, BoxLayout.Y_AXIS);
		ctrlPanel.setLayout(ctrlLayout);
		gb.setConstraints(ctrlPanel, c);
		con.add(ctrlPanel);
		
		String text = "MAME verbinden";
		JButton start = new JButton(text);
		start.setActionCommand("start");
		start.addActionListener(this);
		ctrlPanel.add(start);
		
		text = "MAME trennen";
		JButton stop = new JButton(text);
		stop.setActionCommand("stop");
		stop.addActionListener(this);
		ctrlPanel.add(stop);
		
		JButton replay = new JButton("Replay abspielen");
		replay.setActionCommand("replay");
		replay.addActionListener(this);
		ctrlPanel.add(replay);

		batchMode = new JCheckBox("Batch Mode", true);
		ctrlPanel.add(batchMode);
		
		JCheckBox drawInfo = new JCheckBox("Objekt Infos anzeigen", false);
		drawInfo.setActionCommand("drawInfo");
		drawInfo.addActionListener(this);
		ctrlPanel.add(drawInfo);

		c.weighty = 1;
		Component glue = Box.createGlue();
		gb.setConstraints(glue, c);
		con.add(glue);

		c.weightx = 1;
		c.weighty = 1;
		c.gridheight = 1;
		c.gridwidth = 1;
		screen = new AsteroidScreen();
		gb.setConstraints(screen, c);
		con.add(screen);
		
		c.weightx = 0;
		c.gridwidth = GridBagConstraints.REMAINDER;
		stats = new StatisticsTable();
		gb.setConstraints(stats, c);
		con.add(stats);
	}
	
	
	private void exit() {
		//Ship.saveDirectionTable();
		highscores.saveHighscores();
		screen.requestStop();
		disconnectMame();
		MameRobot.exitMame();
		System.exit(0);
	}
	
	
	public void setServerBusyTime(int secs) {
		if(busyTimer != null) busyTimer.stop();
		
		ActionListener a = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTitle(String.format("Asteroid Bot [%s - busy %02d:%02d]", host, busyTime/60, busyTime%60));
				if(busyTime-- == 0) {
					busyTimer.stop();
					connectMame();
				}
			}
		};
		busyTime = secs+3;
		busyTimer = new Timer(1000, a);
		busyTimer.start();
		disconnectMame();
	}
	
	
	public void gameOver() {
		setTitle("Asteroid Bot ["+host+" - not connected]");
		highscores.setHighscore();
		// Spiel zu Ende - neu starten
		if(batchMode.isSelected()) {
			Runnable r = new Runnable() {
				public void run() {
					try {
						Thread.sleep(1000);
					} catch(InterruptedException e) {
						e.printStackTrace();
					}
					Statistics.reset();
					connectMame();
				}
			};
			Thread t = new Thread(r);
			t.start();
		}
	}
	
	
	private void readDump(String file) {
		disconnectMame();
		mame = new MameClient(this);
		mame.readDumpFile(file);
		mame.addAsteroidFrameListener(this);
		mame.addAsteroidFrameListener(screen);
		setTitle("Asteroid Bot [replay]");
	}
	
	
	private void connectMame() {
		mame = new MameClient(this);
		mame.connectToServer(host, "Mackes Bot");
		bot = new AsteroidBot(mame);
		mame.addAsteroidFrameListener(this);
		mame.addAsteroidFrameListener(screen);
		setTitle("Asteroid Bot ["+host+" - connected]");
	}
	
	
	private void disconnectMame() {
		if(bot != null) bot.requestStop();
		if(mame != null) {
			mame.removeAsteroidFrameListener(this);
			mame.requestStop();
			try {
				mame.join(100);
			} catch(InterruptedException e) {
				e.printStackTrace();
			}
		}
		setTitle("Asteroid Bot ["+host+" - not connected]");
	}


	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();
		if(cmd.equals("start")) {
			connectMame();
			
		} else if(cmd.equals("stop")) {
			disconnectMame();
			
		} else if(cmd.equals("drawInfo")) {
			screen.setDrawInfo(!screen.isDrawInfo());
			
		} else if(cmd.equals("replay")) {
			JFileChooser chooser = new JFileChooser(".");
			int retVal = chooser.showOpenDialog(this);
			if(retVal == JFileChooser.APPROVE_OPTION) {
				batchMode.setSelected(false);
				File f = chooser.getSelectedFile();
				readDump(f.getAbsolutePath());
			}
		}
	}


	public void mameFrameReceived(GameInfo objects) {
		float score = Statistics.getFloat(Statistics.KEY_SCORE);
		float lifes = Statistics.getFloat(Statistics.KEY_LIFES);
		int time = objects.getGameTime() / 60;
		
		Statistics.putFloat(Statistics.KEY_SCORE_PER_MIN, Math.round(60 * (float)score / (float)time));
		Statistics.putFloat(Statistics.KEY_DEATHS, 3+(int)score/10000-(int)lifes);

		stats.updateTable();
		if(objects.getGameTime() >= 18000 && batchMode.isSelected()) {
			highscores.setHighscore();
		}
	}


	public static void main(String[] args) {
		String host = "localhost";
		boolean autoConnect = true;
		
		if(args.length > 0) {
			host = args[0];
		}
		if(args.length > 1 && args[1].equals("-no_auto_con")) {
			autoConnect = false;
		}
		new AsteroidBotWnd(host, autoConnect);
	}
}
